#!/bin/sh
#
# Test / warn for broken systemd-resolved that mistakenly can
# drop DNSSEC # records from the reply resulting in libreswan
# resolving failures (even for non-DNSSEC domains!)

# No known fixed version as of version 246
FIXED_VERSION=500

# Is systemd-resolved active ?
/usr/bin/systemctl is-active systemd-resolved > /dev/null || exit 0
# Is /etc/resolv.conf pointing to systemd-resolved ?
grep "nameserver 127.0.0.53" /etc/resolv.conf > /dev/null || exit 0
# Is the installed version of systemd broken ?
VERSION=$(resolvectl --version |grep systemd | awk '{ print $2}')
test ${VERSION} -ge ${FIXED_VERSION} && exit 0

# A broken systemd-resolved was found which will break libreswan's use
# of the libunbound binary.
# https://github.com/systemd/systemd/issues/4621"
# https://github.com/systemd/systemd/issues/12317
# https://github.com/systemd/systemd/issues/17272
# https://bugzilla.redhat.com/show_bug.cgi?id=1879028


echo "------------------------------------------------------------------------"
echo "You are using version ${VERSION} of systemd-resolved which breaks DNS security!"
echo "------------------------------------------------------------------------"
echo "This setup breaks any application (including libreswan) that uses a"
echo "modern DNS library for additional DNS security and privacy features."
echo "This bug was reported in 2016 and remains unfixed to date:"
echo "  https://github.com/systemd/systemd/issues/4621"
echo ""
echo "To resolve this issue, you can (in order of preference):"
echo "- Upgrade systemd to version ${FIXED_VERSION}"
echo "- Reconfigure NetworkManager to not list systemd-resolved in /etc/resolv.conf"
echo "- Reconfigure /etc/resolv.conf so that NetworkManager won't configure"
echo "  systemd-resolved again by Removing or changing the line with"
echo "      # Generated by NetworkManager"
echo "  and then changing the nameserver entry for 127.0.0.53 to point to a real"
echo "  DNS server (eg 8.8.8.8 or 1.1.1.1 or 9.9.9.9 or your own)"
echo "- Degrade libreswan's DNS security by adding the following line to"
echo "  /etc/ipsec.conf in the 'global setup' section:"
echo "      dnssec-enable=no"

exit 1
