/**
 * @file test_lyb.c
 * @author Michal Vasko <mvasko@cesnet.cz>
 * @brief Cmocka tests for LYB binary data format.
 *
 * Copyright (c) 2020 CESNET, z.s.p.o.
 *
 * This source code is licensed under BSD 3-Clause License (the "License").
 * You may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://opensource.org/licenses/BSD-3-Clause
 */
#define _UTEST_MAIN_
#include "utests.h"

#include "hash_table.h"
#include "libyang.h"

#define CHECK_PARSE_LYD(INPUT, OUT_NODE) \
                CHECK_PARSE_LYD_PARAM(INPUT, LYD_XML, LYD_PARSE_ONLY | LYD_PARSE_STRICT, 0, LY_SUCCESS, OUT_NODE)

#define CHECK_LYD_STRING(MODEL, TEXT) \
                CHECK_LYD_STRING_PARAM(MODEL, TEXT, LYD_XML, LYD_PRINT_WITHSIBLINGS | LYD_PRINT_SHRINK)

static void
check_print_parse(void **state, const char *data_xml)
{
    struct lyd_node *tree_1;
    struct lyd_node *tree_2;
    char *lyb_out;

    CHECK_PARSE_LYD(data_xml, tree_1);
    assert_int_equal(lyd_print_mem(&lyb_out, tree_1, LYD_LYB, LYD_PRINT_WITHSIBLINGS), 0);
    assert_int_equal(LY_SUCCESS, lyd_parse_data_mem(UTEST_LYCTX, lyb_out, LYD_LYB, LYD_PARSE_ONLY | LYD_PARSE_STRICT,
            0, &tree_2));
    assert_non_null(tree_2);
    CHECK_LYD(tree_1, tree_2);

    free(lyb_out);
    lyd_free_all(tree_1);
    lyd_free_all(tree_2);
}

static int
setup(void **state)
{
    UTEST_SETUP;
    assert_int_equal(LY_SUCCESS, ly_ctx_set_searchdir(UTEST_LYCTX, TESTS_DIR_MODULES_YANG));

    return 0;
}

static void
tests_leaflist(void **state)
{
    const char *mod;
    const char *data_xml;

    mod =
            "module mod { namespace \"urn:test-leaflist\"; prefix m;"
            "  container cont {"
            "    presence \"\";"
            "    leaf-list ll {"
            "      type uint8;"
            "    }"
            "  }"
            "}";
    UTEST_ADD_MODULE(mod, LYS_IN_YANG, NULL, NULL);

    data_xml =
            "<cont xmlns=\"urn:test-leaflist\">\n"
            "</cont>\n";
    check_print_parse(state, data_xml);

    data_xml =
            "<cont xmlns=\"urn:test-leaflist\">\n"
            "  <ll>1</ll>\n"
            "</cont>\n";
    check_print_parse(state, data_xml);

    data_xml =
            "<cont xmlns=\"urn:test-leaflist\">\n"
            "  <ll>1</ll>\n"
            "  <ll>2</ll>\n"
            "</cont>\n";
    check_print_parse(state, data_xml);
}

static void
tests_list(void **state)
{
    const char *mod;
    const char *data_xml;

    mod =
            "module mod { namespace \"urn:test-list\"; prefix m;"
            "  container cont {"
            "    presence \"\";"
            "    list lst {"
            "      key \"lf\";"
            "      leaf lf {"
            "        type uint8;"
            "      }"
            "    }"
            "  }"
            "}";
    UTEST_ADD_MODULE(mod, LYS_IN_YANG, NULL, NULL);

    data_xml =
            "<cont xmlns=\"urn:test-list\">\n"
            "  <lst>"
            "    <lf>1</lf>"
            "  </lst>"
            "</cont>\n";
    check_print_parse(state, data_xml);

    data_xml =
            "<cont xmlns=\"urn:test-list\">\n"
            "  <lst>"
            "    <lf>1</lf>"
            "    <lf>2</lf>"
            "  </lst>"
            "</cont>\n";
    check_print_parse(state, data_xml);
}

static void
tests_any(void **state)
{
    const char *mod;
    const char *data_xml;

    mod =
            "module mod { namespace \"urn:test-any\"; prefix m;"
            "  container cont {"
            "    presence \"\";"
            "    anyxml anxml;\n"
            "  }"
            "}";
    UTEST_ADD_MODULE(mod, LYS_IN_YANG, NULL, NULL);

    data_xml =
            "<cont xmlns=\"urn:test-any\">\n"
            "</cont>\n";
    check_print_parse(state, data_xml);

    data_xml =
            "<cont xmlns=\"urn:test-any\">\n"
            "  <anxml><node>value</node></anxml>\n"
            "</cont>\n";
    check_print_parse(state, data_xml);

    data_xml =
            "<cont xmlns=\"urn:test-any\">\n"
            "  <anxml><node1>value</node1></anxml>\n"
            "  <anxml><node2>value</node2></anxml>\n"
            "</cont>\n";
    check_print_parse(state, data_xml);
}

static void
test_ietf_interfaces(void **state)
{
    const char *data_xml =
            "<interfaces xmlns=\"urn:ietf:params:xml:ns:yang:ietf-interfaces\">\n"
            "    <interface>\n"
            "        <name>eth0</name>\n"
            "        <description>Ethernet 0</description>\n"
            "        <type xmlns:ianaift=\"urn:ietf:params:xml:ns:yang:iana-if-type\">ianaift:ethernetCsmacd</type>\n"
            "        <enabled>true</enabled>\n"
            "        <ipv4 xmlns=\"urn:ietf:params:xml:ns:yang:ietf-ip\">\n"
            "            <enabled>true</enabled>\n"
            "            <mtu>1500</mtu>\n"
            "            <address>\n"
            "                <ip>192.168.2.100</ip>\n"
            "                <prefix-length>24</prefix-length>\n"
            "            </address>\n"
            "        </ipv4>\n"
            "    </interface>\n"
            "    <interface>\n"
            "        <name>eth1</name>\n"
            "        <description>Ethernet 1</description>\n"
            "        <type xmlns:ianaift=\"urn:ietf:params:xml:ns:yang:iana-if-type\">ianaift:ethernetCsmacd</type>\n"
            "        <enabled>true</enabled>\n"
            "        <ipv4 xmlns=\"urn:ietf:params:xml:ns:yang:ietf-ip\">\n"
            "            <enabled>true</enabled>\n"
            "            <mtu>1500</mtu>\n"
            "            <address>\n"
            "                <ip>10.10.1.5</ip>\n"
            "                <prefix-length>16</prefix-length>\n"
            "            </address>\n"
            "        </ipv4>\n"
            "    </interface>\n"
            "    <interface>\n"
            "        <name>gigaeth0</name>\n"
            "        <description>GigabitEthernet 0</description>\n"
            "        <type xmlns:ianaift=\"urn:ietf:params:xml:ns:yang:iana-if-type\">ianaift:ethernetCsmacd</type>\n"
            "        <enabled>false</enabled>\n"
            "    </interface>\n"
            "</interfaces>\n";

    assert_non_null(ly_ctx_load_module(UTEST_LYCTX, "ietf-ip", NULL, NULL));
    assert_non_null(ly_ctx_load_module(UTEST_LYCTX, "iana-if-type", NULL, NULL));

    check_print_parse(state, data_xml);
}

static void
test_origin(void **state)
{
    const char *origin_yang =
            "module test-origin {"
            "   namespace \"urn:test-origin\";"
            "   prefix to;"
            "   import ietf-origin {"
            "       prefix or;"
            "   }"
            ""
            "   container cont {"
            "       leaf leaf1 {"
            "           type string;"
            "       }"
            "       leaf leaf2 {"
            "           type string;"
            "       }"
            "       leaf leaf3 {"
            "           type uint8;"
            "       }"
            "   }"
            "}";
    const char *data_xml =
            "<cont xmlns=\"urn:test-origin\">\n"
            "  <leaf1 xmlns:or=\"urn:ietf:params:xml:ns:yang:ietf-origin\" or:origin=\"or:default\">value1</leaf1>\n"
            "  <leaf2>value2</leaf2>\n"
            "  <leaf3 xmlns:or=\"urn:ietf:params:xml:ns:yang:ietf-origin\" or:origin=\"or:system\">125</leaf3>\n"
            "</cont>\n";

    UTEST_ADD_MODULE(origin_yang, LYS_IN_YANG, NULL, NULL);
    assert_int_equal(LY_SUCCESS, lys_set_implemented(ly_ctx_get_module_latest(UTEST_LYCTX, "ietf-origin"), NULL));

    check_print_parse(state, data_xml);
}

static void
test_statements(void **state)
{
    const char *links_yang =
            "module links {\n"
            "    yang-version 1.1;\n"
            "    namespace \"urn:module2\";\n"
            "    prefix mod2;\n"
            "\n"
            "    identity just-another-identity;\n"
            "\n"
            "    leaf one-leaf {\n"
            "        type string;\n"
            "    }\n"
            "\n"
            "    list list-for-augment {\n"
            "        key keyleaf;\n"
            "\n"
            "        leaf keyleaf {\n"
            "            type string;\n"
            "        }\n"
            "\n"
            "        leaf just-leaf {\n"
            "            type int32;\n"
            "        }\n"
            "    }\n"
            "\n"
            "    leaf rleaf {\n"
            "        type string;\n"
            "    }\n"
            "\n"
            "    leaf-list llist {\n"
            "        type string;\n"
            "        min-elements 0;\n"
            "        max-elements 100;\n"
            "        ordered-by user;\n"
            "    }\n"
            "\n"
            "    grouping rgroup {\n"
            "        leaf rg1 {\n"
            "            type string;\n"
            "        }\n"
            "\n"
            "        leaf rg2 {\n"
            "            type string;\n"
            "        }\n"
            "    }\n"
            "}\n";

    const char *statements_yang =
            "module statements {\n"
            "    namespace \"urn:module\";\n"
            "    prefix mod;\n"
            "    yang-version 1.1;\n"
            "\n"
            "    import links {\n"
            "        prefix mod2;\n"
            "    }\n"
            "\n"
            "    identity random-identity {\n"
            "        base \"mod2:just-another-identity\";\n"
            "        base \"another-identity\";\n"
            "    }\n"
            "\n"
            "    identity another-identity {\n"
            "        base \"mod2:just-another-identity\";\n"
            "    }\n"
            "\n"
            "    typedef percent {\n"
            "        type uint8 {\n"
            "            range \"0 .. 100\";\n"
            "        }\n"
            "        units percent;\n"
            "    }\n"
            "\n"
            "    container ice-cream-shop {\n"
            "        container employees {\n"
            "            list employee {\n"
            "                config true;\n"
            "                key id;\n"
            "                unique name;\n"
            "                min-elements 0;\n"
            "                max-elements 100;\n"
            "\n"
            "                leaf id {\n"
            "                    type uint64;\n"
            "                    mandatory true;\n"
            "                }\n"
            "\n"
            "                leaf name {\n"
            "                    type string;\n"
            "                }\n"
            "\n"
            "                leaf age {\n"
            "                    type uint32;\n"
            "                }\n"
            "            }\n"
            "        }\n"
            "    }\n"
            "\n"
            "    container random {\n"
            "        choice switch {\n"
            "            case a {\n"
            "                leaf aleaf {\n"
            "                    type string;\n"
            "                    default aaa;\n"
            "                }\n"
            "            }\n"
            "\n"
            "            case c {\n"
            "                leaf cleaf {\n"
            "                    type string;\n"
            "                }\n"
            "            }\n"
            "        }\n"
            "\n"
            "        anyxml xml-data;\n"
            "        anydata any-data;\n"
            "        leaf-list leaflist {\n"
            "            type string;\n"
            "            min-elements 0;\n"
            "            max-elements 20;\n"
            "            ordered-by system;\n"
            "        }\n"
            "\n"
            "        grouping group {\n"
            "            leaf g1 {\n"
            "                mandatory false;\n"
            "                type percent;\n"
            "            }\n"
            "\n"
            "            leaf g2 {\n"
            "                type string;\n"
            "            }\n"
            "        }\n"
            "\n"
            "        uses group;\n"
            "        uses mod2:rgroup;\n"
            "\n"
            "        leaf lref {\n"
            "            type leafref {\n"
            "                path \"/mod2:one-leaf\";\n"
            "            }\n"
            "        }\n"
            "\n"
            "        leaf iref {\n"
            "            type identityref {\n"
            "                base \"mod2:just-another-identity\";\n"
            "            }\n"
            "        }\n"
            "    }\n"
            "\n"
            "    notification notif;\n"
            "\n"
            "    augment \"/random\" {\n"
            "        leaf aug-leaf {\n"
            "            type string;\n"
            "        }\n"
            "    }\n"
            "}\n";

    const char *data_xml =
            "<one-leaf xmlns=\"urn:module2\">reference leaf</one-leaf>\n"
            "<ice-cream-shop xmlns=\"urn:module\">\n"
            "  <employees>\n"
            "    <employee>\n"
            "      <id>0</id>\n"
            "      <name>John Doe</name>\n"
            "      <age>28</age>\n"
            "    </employee>\n"
            "    <employee>\n"
            "      <id>1</id>\n"
            "      <name>Dohn Joe</name>\n"
            "      <age>20</age>\n"
            "    </employee>\n"
            "  </employees>\n"
            "</ice-cream-shop>\n"
            "<random xmlns=\"urn:module\">\n"
            "  <aleaf>string</aleaf>\n"
            "  <xml-data><anyxml>data</anyxml></xml-data>\n"
            "  <any-data><notif/></any-data>\n"
            "  <leaflist>l0</leaflist>\n"
            "  <leaflist>l1</leaflist>\n"
            "  <leaflist>l2</leaflist>\n"
            "  <g1>40</g1>\n"
            "  <g2>string</g2>\n"
            "  <aug-leaf>string</aug-leaf>\n"
            "  <rg1>string</rg1>\n"
            "  <rg2>string</rg2>\n"
            "  <lref>reference leaf</lref>\n"
            "  <iref>random-identity</iref>\n"
            "</random>\n";

    UTEST_ADD_MODULE(links_yang, LYS_IN_YANG, NULL, NULL);
    UTEST_ADD_MODULE(statements_yang, LYS_IN_YANG, NULL, NULL);

    check_print_parse(state, data_xml);
}

static void
test_opaq(void **state)
{
    const char *nc_feats[] = {"writable-running", NULL};
    const char *data_xml =
            "<edit-config xmlns=\"urn:ietf:params:xml:ns:netconf:base:1.0\">\n"
            "  <target>\n"
            "    <running/>\n"
            "  </target>\n"
            "  <config>\n"
            "    <top xmlns=\"urn:ed\">\n"
            "      <first>TestFirst</first>\n"
            "    </top>\n"
            "  </config>\n"
            "</edit-config>\n";
    struct ly_in *in;
    struct lyd_node *tree_1;
    struct lyd_node *tree_2;
    char *xml_out; /* tree_2 */
    LY_ERR rc;

    assert_non_null(ly_ctx_load_module(UTEST_LYCTX, "ietf-netconf", NULL, nc_feats));

    ly_in_new_memory(data_xml, &in);
    rc = lyd_parse_op(UTEST_LYCTX, NULL, in, LYD_XML, LYD_TYPE_RPC_YANG, &tree_1, NULL);
    ly_in_free(in, 0);
    assert_int_equal(rc, LY_SUCCESS);

    assert_int_equal(lyd_print_mem(&xml_out, tree_1, LYD_LYB, LYD_PRINT_WITHSIBLINGS), 0);

    ly_in_new_memory(xml_out, &in);
    rc = lyd_parse_op(UTEST_LYCTX, NULL, in, LYD_LYB, LYD_TYPE_RPC_YANG, &tree_2, NULL);
    ly_in_free(in, 0);
    assert_int_equal(rc, LY_SUCCESS);

    /* compare models */
    CHECK_LYD(tree_1, tree_2);

    /* clean */
    free(xml_out);
    lyd_free_all(tree_1);
    lyd_free_all(tree_2);
}

static void
test_collisions(void **state)
{
    char *counters_yang, *data_xml;

    counters_yang = malloc(32768);
    strcpy(counters_yang,
            "module counters {\n"
            "    namespace \"urn:counters\";\n"
            "    prefix c;\n"
            "\n"
            "    container stats {\n");
    strcat(counters_yang,
            "        leaf counter1 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter2 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter3 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter4 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter5 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter6 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter7 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter8 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter9 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter10 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter11 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter12 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter13 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter14 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter15 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter16 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter17 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter18 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter19 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter20 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter21 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter22 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter23 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter24 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter25 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter26 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter27 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter28 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter29 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter30 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter31 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter32 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter33 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter34 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter35 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter36 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter37 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter38 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter39 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter40 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter41 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter42 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter43 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter44 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter45 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter46 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter47 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter48 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter49 {\n"
            "            type uint64;\n"
            "        }\n");
    strcat(counters_yang,
            "        leaf counter50 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter51 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter52 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter53 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter54 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter55 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter56 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter57 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter58 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter59 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter60 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter61 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter62 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter63 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter64 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter65 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter66 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter67 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter68 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter69 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter70 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter71 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter72 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter73 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter74 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter75 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter76 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter77 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter78 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter79 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter80 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter81 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter82 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter83 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter84 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter85 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter86 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter87 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter88 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter89 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter90 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter91 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter92 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter93 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter94 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter95 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter96 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter97 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter98 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter99 {\n"
            "            type uint64;\n"
            "        }\n");
    strcat(counters_yang,
            "        leaf counter100 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter101 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter102 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter103 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter104 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter105 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter106 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter107 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter108 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter109 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter110 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter111 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter112 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter113 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter114 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter115 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter116 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter117 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter118 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter119 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter120 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter121 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter122 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter123 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter124 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter125 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter126 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter127 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter128 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter129 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter130 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter131 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter132 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter133 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter134 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter135 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter136 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter137 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter138 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter139 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter140 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter141 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter142 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter143 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter144 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter145 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter146 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter147 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter148 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter149 {\n"
            "            type uint64;\n"
            "        }\n");
    strcat(counters_yang,
            "        leaf counter150 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter151 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter152 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter153 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter154 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter155 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter156 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter157 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter158 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter159 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter160 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter161 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter162 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter163 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter164 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter165 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter166 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter167 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter168 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter169 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter170 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter171 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter172 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter173 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter174 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter175 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter176 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter177 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter178 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter179 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter180 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter181 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter182 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter183 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter184 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter185 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter186 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter187 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter188 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter189 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter190 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter191 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter192 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter193 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter194 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter195 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter196 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter197 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter198 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter199 {\n"
            "            type uint64;\n"
            "        }\n");
    strcat(counters_yang,
            "        leaf counter200 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter201 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter202 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter203 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter204 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter205 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter206 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter207 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter208 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter209 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter210 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter211 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter212 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter213 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter214 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter215 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter216 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter217 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter218 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter219 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter220 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter221 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter222 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter223 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter224 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter225 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter226 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter227 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter228 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter229 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter230 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter231 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter232 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter233 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter234 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter235 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter236 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter237 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter238 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter239 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter240 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter241 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter242 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter243 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter244 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter245 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter246 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter247 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter248 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter249 {\n"
            "            type uint64;\n"
            "        }\n");
    strcat(counters_yang,
            "        leaf counter250 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter251 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter252 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter253 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter254 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter255 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter256 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter257 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter258 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter259 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter260 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter261 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter262 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter263 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter264 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter265 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter266 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter267 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter268 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter269 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter270 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter271 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter272 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter273 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter274 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter275 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter276 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter277 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter278 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter279 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter280 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter281 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter282 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter283 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter284 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter285 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter286 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter287 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter288 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter289 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter290 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter291 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter292 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter293 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter294 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter295 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter296 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter297 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter298 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter299 {\n"
            "            type uint64;\n"
            "        }\n");
    strcat(counters_yang,
            "        leaf counter300 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter301 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter302 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter303 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter304 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter305 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter306 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter307 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter308 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter309 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter310 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter311 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter312 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter313 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter314 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter315 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter316 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter317 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter318 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter319 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter320 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter321 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter322 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter323 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter324 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter325 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter326 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter327 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter328 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter329 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter330 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter331 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter332 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter333 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter334 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter335 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter336 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter337 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter338 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter339 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter340 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter341 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter342 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter343 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter344 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter345 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter346 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter347 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter348 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter349 {\n"
            "            type uint64;\n"
            "        }\n");
    strcat(counters_yang,
            "        leaf counter350 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter351 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter352 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter353 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter354 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter355 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter356 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter357 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter358 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter359 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter360 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter361 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter362 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter363 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter364 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter365 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter366 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter367 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter368 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter369 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter370 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter371 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter372 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter373 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter374 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter375 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter376 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter377 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter378 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter379 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter380 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter381 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter382 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter383 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter384 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter385 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter386 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter387 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter388 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter389 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter390 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter391 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter392 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter393 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter394 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter395 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter396 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter397 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter398 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter399 {\n"
            "            type uint64;\n"
            "        }\n");
    strcat(counters_yang,
            "        leaf counter400 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter401 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter402 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter403 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter404 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter405 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter406 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter407 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter408 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter409 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter410 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter411 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter412 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter413 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter414 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter415 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter416 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter417 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter418 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter419 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter420 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter421 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter422 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter423 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter424 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter425 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter426 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter427 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter428 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter429 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter430 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter431 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter432 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter433 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter434 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter435 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter436 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter437 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter438 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter439 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter440 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter441 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter442 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter443 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter444 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter445 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter446 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter447 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter448 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter449 {\n"
            "            type uint64;\n"
            "        }\n");
    strcat(counters_yang,
            "        leaf counter450 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter451 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter452 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter453 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter454 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter455 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter456 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter457 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter458 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter459 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter460 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter461 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter462 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter463 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter464 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter465 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter466 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter467 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter468 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter469 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter470 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter471 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter472 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter473 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter474 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter475 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter476 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter477 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter478 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter479 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter480 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter481 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter482 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter483 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter484 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter485 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter486 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter487 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter488 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter489 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter490 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter491 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter492 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter493 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter494 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter495 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter496 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter497 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter498 {\n"
            "            type uint64;\n"
            "        }\n"
            "        leaf counter499 {\n"
            "            type uint64;\n"
            "        }\n"
            "    }\n"
            "}\n");

    data_xml = malloc(16384);
    strcpy(data_xml,
            "<stats xmlns=\"urn:counters\">\n");
    strcat(data_xml,
            "    <counter1>1</counter1>\n"
            "    <counter2>2</counter2>\n"
            "    <counter3>3</counter3>\n"
            "    <counter4>4</counter4>\n"
            "    <counter5>5</counter5>\n"
            "    <counter6>6</counter6>\n"
            "    <counter7>7</counter7>\n"
            "    <counter8>8</counter8>\n"
            "    <counter9>9</counter9>\n"
            "    <counter10>10</counter10>\n"
            "    <counter11>11</counter11>\n"
            "    <counter12>12</counter12>\n"
            "    <counter13>13</counter13>\n"
            "    <counter14>14</counter14>\n"
            "    <counter15>15</counter15>\n"
            "    <counter16>16</counter16>\n"
            "    <counter17>17</counter17>\n"
            "    <counter18>18</counter18>\n"
            "    <counter19>19</counter19>\n"
            "    <counter20>20</counter20>\n"
            "    <counter21>21</counter21>\n"
            "    <counter22>22</counter22>\n"
            "    <counter23>23</counter23>\n"
            "    <counter24>24</counter24>\n"
            "    <counter25>25</counter25>\n"
            "    <counter26>26</counter26>\n"
            "    <counter27>27</counter27>\n"
            "    <counter28>28</counter28>\n"
            "    <counter29>29</counter29>\n"
            "    <counter30>30</counter30>\n"
            "    <counter31>31</counter31>\n"
            "    <counter32>32</counter32>\n"
            "    <counter33>33</counter33>\n"
            "    <counter34>34</counter34>\n"
            "    <counter35>35</counter35>\n"
            "    <counter36>36</counter36>\n"
            "    <counter37>37</counter37>\n"
            "    <counter38>38</counter38>\n"
            "    <counter39>39</counter39>\n"
            "    <counter40>40</counter40>\n"
            "    <counter41>41</counter41>\n"
            "    <counter42>42</counter42>\n"
            "    <counter43>43</counter43>\n"
            "    <counter44>44</counter44>\n"
            "    <counter45>45</counter45>\n"
            "    <counter46>46</counter46>\n"
            "    <counter47>47</counter47>\n"
            "    <counter48>48</counter48>\n"
            "    <counter49>49</counter49>\n"
            "    <counter50>50</counter50>\n"
            "    <counter51>51</counter51>\n"
            "    <counter52>52</counter52>\n"
            "    <counter53>53</counter53>\n"
            "    <counter54>54</counter54>\n"
            "    <counter55>55</counter55>\n"
            "    <counter56>56</counter56>\n"
            "    <counter57>57</counter57>\n"
            "    <counter58>58</counter58>\n"
            "    <counter59>59</counter59>\n"
            "    <counter60>60</counter60>\n"
            "    <counter61>61</counter61>\n"
            "    <counter62>62</counter62>\n"
            "    <counter63>63</counter63>\n"
            "    <counter64>64</counter64>\n"
            "    <counter65>65</counter65>\n"
            "    <counter66>66</counter66>\n"
            "    <counter67>67</counter67>\n"
            "    <counter68>68</counter68>\n"
            "    <counter69>69</counter69>\n"
            "    <counter70>70</counter70>\n"
            "    <counter71>71</counter71>\n"
            "    <counter72>72</counter72>\n"
            "    <counter73>73</counter73>\n"
            "    <counter74>74</counter74>\n"
            "    <counter75>75</counter75>\n"
            "    <counter76>76</counter76>\n"
            "    <counter77>77</counter77>\n"
            "    <counter78>78</counter78>\n"
            "    <counter79>79</counter79>\n"
            "    <counter80>80</counter80>\n"
            "    <counter81>81</counter81>\n"
            "    <counter82>82</counter82>\n"
            "    <counter83>83</counter83>\n"
            "    <counter84>84</counter84>\n"
            "    <counter85>85</counter85>\n"
            "    <counter86>86</counter86>\n"
            "    <counter87>87</counter87>\n"
            "    <counter88>88</counter88>\n"
            "    <counter89>89</counter89>\n"
            "    <counter90>90</counter90>\n"
            "    <counter91>91</counter91>\n"
            "    <counter92>92</counter92>\n"
            "    <counter93>93</counter93>\n"
            "    <counter94>94</counter94>\n"
            "    <counter95>95</counter95>\n"
            "    <counter96>96</counter96>\n"
            "    <counter97>97</counter97>\n"
            "    <counter98>98</counter98>\n"
            "    <counter99>99</counter99>\n");
    strcat(data_xml,
            "    <counter100>100</counter100>\n"
            "    <counter101>101</counter101>\n"
            "    <counter102>102</counter102>\n"
            "    <counter103>103</counter103>\n"
            "    <counter104>104</counter104>\n"
            "    <counter105>105</counter105>\n"
            "    <counter106>106</counter106>\n"
            "    <counter107>107</counter107>\n"
            "    <counter108>108</counter108>\n"
            "    <counter109>109</counter109>\n"
            "    <counter110>110</counter110>\n"
            "    <counter111>111</counter111>\n"
            "    <counter112>112</counter112>\n"
            "    <counter113>113</counter113>\n"
            "    <counter114>114</counter114>\n"
            "    <counter115>115</counter115>\n"
            "    <counter116>116</counter116>\n"
            "    <counter117>117</counter117>\n"
            "    <counter118>118</counter118>\n"
            "    <counter119>119</counter119>\n"
            "    <counter120>120</counter120>\n"
            "    <counter121>121</counter121>\n"
            "    <counter122>122</counter122>\n"
            "    <counter123>123</counter123>\n"
            "    <counter124>124</counter124>\n"
            "    <counter125>125</counter125>\n"
            "    <counter126>126</counter126>\n"
            "    <counter127>127</counter127>\n"
            "    <counter128>128</counter128>\n"
            "    <counter129>129</counter129>\n"
            "    <counter130>130</counter130>\n"
            "    <counter131>131</counter131>\n"
            "    <counter132>132</counter132>\n"
            "    <counter133>133</counter133>\n"
            "    <counter134>134</counter134>\n"
            "    <counter135>135</counter135>\n"
            "    <counter136>136</counter136>\n"
            "    <counter137>137</counter137>\n"
            "    <counter138>138</counter138>\n"
            "    <counter139>139</counter139>\n"
            "    <counter140>140</counter140>\n"
            "    <counter141>141</counter141>\n"
            "    <counter142>142</counter142>\n"
            "    <counter143>143</counter143>\n"
            "    <counter144>144</counter144>\n"
            "    <counter145>145</counter145>\n"
            "    <counter146>146</counter146>\n"
            "    <counter147>147</counter147>\n"
            "    <counter148>148</counter148>\n"
            "    <counter149>149</counter149>\n"
            "    <counter150>150</counter150>\n"
            "    <counter151>151</counter151>\n"
            "    <counter152>152</counter152>\n"
            "    <counter153>153</counter153>\n"
            "    <counter154>154</counter154>\n"
            "    <counter155>155</counter155>\n"
            "    <counter156>156</counter156>\n"
            "    <counter157>157</counter157>\n"
            "    <counter158>158</counter158>\n"
            "    <counter159>159</counter159>\n"
            "    <counter160>160</counter160>\n"
            "    <counter161>161</counter161>\n"
            "    <counter162>162</counter162>\n"
            "    <counter163>163</counter163>\n"
            "    <counter164>164</counter164>\n"
            "    <counter165>165</counter165>\n"
            "    <counter166>166</counter166>\n"
            "    <counter167>167</counter167>\n"
            "    <counter168>168</counter168>\n"
            "    <counter169>169</counter169>\n"
            "    <counter170>170</counter170>\n"
            "    <counter171>171</counter171>\n"
            "    <counter172>172</counter172>\n"
            "    <counter173>173</counter173>\n"
            "    <counter174>174</counter174>\n"
            "    <counter175>175</counter175>\n"
            "    <counter176>176</counter176>\n"
            "    <counter177>177</counter177>\n"
            "    <counter178>178</counter178>\n"
            "    <counter179>179</counter179>\n"
            "    <counter180>180</counter180>\n"
            "    <counter181>181</counter181>\n"
            "    <counter182>182</counter182>\n"
            "    <counter183>183</counter183>\n"
            "    <counter184>184</counter184>\n"
            "    <counter185>185</counter185>\n"
            "    <counter186>186</counter186>\n"
            "    <counter187>187</counter187>\n"
            "    <counter188>188</counter188>\n"
            "    <counter189>189</counter189>\n"
            "    <counter190>190</counter190>\n"
            "    <counter191>191</counter191>\n"
            "    <counter192>192</counter192>\n"
            "    <counter193>193</counter193>\n"
            "    <counter194>194</counter194>\n"
            "    <counter195>195</counter195>\n"
            "    <counter196>196</counter196>\n"
            "    <counter197>197</counter197>\n"
            "    <counter198>198</counter198>\n"
            "    <counter199>199</counter199>\n");
    strcat(data_xml,
            "    <counter200>200</counter200>\n"
            "    <counter201>201</counter201>\n"
            "    <counter202>202</counter202>\n"
            "    <counter203>203</counter203>\n"
            "    <counter204>204</counter204>\n"
            "    <counter205>205</counter205>\n"
            "    <counter206>206</counter206>\n"
            "    <counter207>207</counter207>\n"
            "    <counter208>208</counter208>\n"
            "    <counter209>209</counter209>\n"
            "    <counter210>210</counter210>\n"
            "    <counter211>211</counter211>\n"
            "    <counter212>212</counter212>\n"
            "    <counter213>213</counter213>\n"
            "    <counter214>214</counter214>\n"
            "    <counter215>215</counter215>\n"
            "    <counter216>216</counter216>\n"
            "    <counter217>217</counter217>\n"
            "    <counter218>218</counter218>\n"
            "    <counter219>219</counter219>\n"
            "    <counter220>220</counter220>\n"
            "    <counter221>221</counter221>\n"
            "    <counter222>222</counter222>\n"
            "    <counter223>223</counter223>\n"
            "    <counter224>224</counter224>\n"
            "    <counter225>225</counter225>\n"
            "    <counter226>226</counter226>\n"
            "    <counter227>227</counter227>\n"
            "    <counter228>228</counter228>\n"
            "    <counter229>229</counter229>\n"
            "    <counter230>230</counter230>\n"
            "    <counter231>231</counter231>\n"
            "    <counter232>232</counter232>\n"
            "    <counter233>233</counter233>\n"
            "    <counter234>234</counter234>\n"
            "    <counter235>235</counter235>\n"
            "    <counter236>236</counter236>\n"
            "    <counter237>237</counter237>\n"
            "    <counter238>238</counter238>\n"
            "    <counter239>239</counter239>\n"
            "    <counter240>240</counter240>\n"
            "    <counter241>241</counter241>\n"
            "    <counter242>242</counter242>\n"
            "    <counter243>243</counter243>\n"
            "    <counter244>244</counter244>\n"
            "    <counter245>245</counter245>\n"
            "    <counter246>246</counter246>\n"
            "    <counter247>247</counter247>\n"
            "    <counter248>248</counter248>\n"
            "    <counter249>249</counter249>\n"
            "    <counter250>250</counter250>\n"
            "    <counter251>251</counter251>\n"
            "    <counter252>252</counter252>\n"
            "    <counter253>253</counter253>\n"
            "    <counter254>254</counter254>\n"
            "    <counter255>255</counter255>\n"
            "    <counter256>256</counter256>\n"
            "    <counter257>257</counter257>\n"
            "    <counter258>258</counter258>\n"
            "    <counter259>259</counter259>\n"
            "    <counter260>260</counter260>\n"
            "    <counter261>261</counter261>\n"
            "    <counter262>262</counter262>\n"
            "    <counter263>263</counter263>\n"
            "    <counter264>264</counter264>\n"
            "    <counter265>265</counter265>\n"
            "    <counter266>266</counter266>\n"
            "    <counter267>267</counter267>\n"
            "    <counter268>268</counter268>\n"
            "    <counter269>269</counter269>\n"
            "    <counter270>270</counter270>\n"
            "    <counter271>271</counter271>\n"
            "    <counter272>272</counter272>\n"
            "    <counter273>273</counter273>\n"
            "    <counter274>274</counter274>\n"
            "    <counter275>275</counter275>\n"
            "    <counter276>276</counter276>\n"
            "    <counter277>277</counter277>\n"
            "    <counter278>278</counter278>\n"
            "    <counter279>279</counter279>\n"
            "    <counter280>280</counter280>\n"
            "    <counter281>281</counter281>\n"
            "    <counter282>282</counter282>\n"
            "    <counter283>283</counter283>\n"
            "    <counter284>284</counter284>\n"
            "    <counter285>285</counter285>\n"
            "    <counter286>286</counter286>\n"
            "    <counter287>287</counter287>\n"
            "    <counter288>288</counter288>\n"
            "    <counter289>289</counter289>\n"
            "    <counter290>290</counter290>\n"
            "    <counter291>291</counter291>\n"
            "    <counter292>292</counter292>\n"
            "    <counter293>293</counter293>\n"
            "    <counter294>294</counter294>\n"
            "    <counter295>295</counter295>\n"
            "    <counter296>296</counter296>\n"
            "    <counter297>297</counter297>\n"
            "    <counter298>298</counter298>\n"
            "    <counter299>299</counter299>\n");
    strcat(data_xml,
            "    <counter300>300</counter300>\n"
            "    <counter301>301</counter301>\n"
            "    <counter302>302</counter302>\n"
            "    <counter303>303</counter303>\n"
            "    <counter304>304</counter304>\n"
            "    <counter305>305</counter305>\n"
            "    <counter306>306</counter306>\n"
            "    <counter307>307</counter307>\n"
            "    <counter308>308</counter308>\n"
            "    <counter309>309</counter309>\n"
            "    <counter310>310</counter310>\n"
            "    <counter311>311</counter311>\n"
            "    <counter312>312</counter312>\n"
            "    <counter313>313</counter313>\n"
            "    <counter314>314</counter314>\n"
            "    <counter315>315</counter315>\n"
            "    <counter316>316</counter316>\n"
            "    <counter317>317</counter317>\n"
            "    <counter318>318</counter318>\n"
            "    <counter319>319</counter319>\n"
            "    <counter320>320</counter320>\n"
            "    <counter321>321</counter321>\n"
            "    <counter322>322</counter322>\n"
            "    <counter323>323</counter323>\n"
            "    <counter324>324</counter324>\n"
            "    <counter325>325</counter325>\n"
            "    <counter326>326</counter326>\n"
            "    <counter327>327</counter327>\n"
            "    <counter328>328</counter328>\n"
            "    <counter329>329</counter329>\n"
            "    <counter330>330</counter330>\n"
            "    <counter331>331</counter331>\n"
            "    <counter332>332</counter332>\n"
            "    <counter333>333</counter333>\n"
            "    <counter334>334</counter334>\n"
            "    <counter335>335</counter335>\n"
            "    <counter336>336</counter336>\n"
            "    <counter337>337</counter337>\n"
            "    <counter338>338</counter338>\n"
            "    <counter339>339</counter339>\n"
            "    <counter340>340</counter340>\n"
            "    <counter341>341</counter341>\n"
            "    <counter342>342</counter342>\n"
            "    <counter343>343</counter343>\n"
            "    <counter344>344</counter344>\n"
            "    <counter345>345</counter345>\n"
            "    <counter346>346</counter346>\n"
            "    <counter347>347</counter347>\n"
            "    <counter348>348</counter348>\n"
            "    <counter349>349</counter349>\n"
            "    <counter350>350</counter350>\n"
            "    <counter351>351</counter351>\n"
            "    <counter352>352</counter352>\n"
            "    <counter353>353</counter353>\n"
            "    <counter354>354</counter354>\n"
            "    <counter355>355</counter355>\n"
            "    <counter356>356</counter356>\n"
            "    <counter357>357</counter357>\n"
            "    <counter358>358</counter358>\n"
            "    <counter359>359</counter359>\n"
            "    <counter360>360</counter360>\n"
            "    <counter361>361</counter361>\n"
            "    <counter362>362</counter362>\n"
            "    <counter363>363</counter363>\n"
            "    <counter364>364</counter364>\n"
            "    <counter365>365</counter365>\n"
            "    <counter366>366</counter366>\n"
            "    <counter367>367</counter367>\n"
            "    <counter368>368</counter368>\n"
            "    <counter369>369</counter369>\n"
            "    <counter370>370</counter370>\n"
            "    <counter371>371</counter371>\n"
            "    <counter372>372</counter372>\n"
            "    <counter373>373</counter373>\n"
            "    <counter374>374</counter374>\n"
            "    <counter375>375</counter375>\n"
            "    <counter376>376</counter376>\n"
            "    <counter377>377</counter377>\n"
            "    <counter378>378</counter378>\n"
            "    <counter379>379</counter379>\n"
            "    <counter380>380</counter380>\n"
            "    <counter381>381</counter381>\n"
            "    <counter382>382</counter382>\n"
            "    <counter383>383</counter383>\n"
            "    <counter384>384</counter384>\n"
            "    <counter385>385</counter385>\n"
            "    <counter386>386</counter386>\n"
            "    <counter387>387</counter387>\n"
            "    <counter388>388</counter388>\n"
            "    <counter389>389</counter389>\n"
            "    <counter390>390</counter390>\n"
            "    <counter391>391</counter391>\n"
            "    <counter392>392</counter392>\n"
            "    <counter393>393</counter393>\n"
            "    <counter394>394</counter394>\n"
            "    <counter395>395</counter395>\n"
            "    <counter396>396</counter396>\n"
            "    <counter397>397</counter397>\n"
            "    <counter398>398</counter398>\n"
            "    <counter399>399</counter399>\n");
    strcat(data_xml,
            "    <counter400>400</counter400>\n"
            "    <counter401>401</counter401>\n"
            "    <counter402>402</counter402>\n"
            "    <counter403>403</counter403>\n"
            "    <counter404>404</counter404>\n"
            "    <counter405>405</counter405>\n"
            "    <counter406>406</counter406>\n"
            "    <counter407>407</counter407>\n"
            "    <counter408>408</counter408>\n"
            "    <counter409>409</counter409>\n"
            "    <counter410>410</counter410>\n"
            "    <counter411>411</counter411>\n"
            "    <counter412>412</counter412>\n"
            "    <counter413>413</counter413>\n"
            "    <counter414>414</counter414>\n"
            "    <counter415>415</counter415>\n"
            "    <counter416>416</counter416>\n"
            "    <counter417>417</counter417>\n"
            "    <counter418>418</counter418>\n"
            "    <counter419>419</counter419>\n"
            "    <counter420>420</counter420>\n"
            "    <counter421>421</counter421>\n"
            "    <counter422>422</counter422>\n"
            "    <counter423>423</counter423>\n"
            "    <counter424>424</counter424>\n"
            "    <counter425>425</counter425>\n"
            "    <counter426>426</counter426>\n"
            "    <counter427>427</counter427>\n"
            "    <counter428>428</counter428>\n"
            "    <counter429>429</counter429>\n"
            "    <counter430>430</counter430>\n"
            "    <counter431>431</counter431>\n"
            "    <counter432>432</counter432>\n"
            "    <counter433>433</counter433>\n"
            "    <counter434>434</counter434>\n"
            "    <counter435>435</counter435>\n"
            "    <counter436>436</counter436>\n"
            "    <counter437>437</counter437>\n"
            "    <counter438>438</counter438>\n"
            "    <counter439>439</counter439>\n"
            "    <counter440>440</counter440>\n"
            "    <counter441>441</counter441>\n"
            "    <counter442>442</counter442>\n"
            "    <counter443>443</counter443>\n"
            "    <counter444>444</counter444>\n"
            "    <counter445>445</counter445>\n"
            "    <counter446>446</counter446>\n"
            "    <counter447>447</counter447>\n"
            "    <counter448>448</counter448>\n"
            "    <counter449>449</counter449>\n"
            "    <counter450>450</counter450>\n"
            "    <counter451>451</counter451>\n"
            "    <counter452>452</counter452>\n"
            "    <counter453>453</counter453>\n"
            "    <counter454>454</counter454>\n"
            "    <counter455>455</counter455>\n"
            "    <counter456>456</counter456>\n"
            "    <counter457>457</counter457>\n"
            "    <counter458>458</counter458>\n"
            "    <counter459>459</counter459>\n"
            "    <counter460>460</counter460>\n"
            "    <counter461>461</counter461>\n"
            "    <counter462>462</counter462>\n"
            "    <counter463>463</counter463>\n"
            "    <counter464>464</counter464>\n"
            "    <counter465>465</counter465>\n"
            "    <counter466>466</counter466>\n"
            "    <counter467>467</counter467>\n"
            "    <counter468>468</counter468>\n"
            "    <counter469>469</counter469>\n"
            "    <counter470>470</counter470>\n"
            "    <counter471>471</counter471>\n"
            "    <counter472>472</counter472>\n"
            "    <counter473>473</counter473>\n"
            "    <counter474>474</counter474>\n"
            "    <counter475>475</counter475>\n"
            "    <counter476>476</counter476>\n"
            "    <counter477>477</counter477>\n"
            "    <counter478>478</counter478>\n"
            "    <counter479>479</counter479>\n"
            "    <counter480>480</counter480>\n"
            "    <counter481>481</counter481>\n"
            "    <counter482>482</counter482>\n"
            "    <counter483>483</counter483>\n"
            "    <counter484>484</counter484>\n"
            "    <counter485>485</counter485>\n"
            "    <counter486>486</counter486>\n"
            "    <counter487>487</counter487>\n"
            "    <counter488>488</counter488>\n"
            "    <counter489>489</counter489>\n"
            "    <counter490>490</counter490>\n"
            "    <counter491>491</counter491>\n"
            "    <counter492>492</counter492>\n"
            "    <counter493>493</counter493>\n"
            "    <counter494>494</counter494>\n"
            "    <counter495>495</counter495>\n"
            "    <counter496>496</counter496>\n"
            "    <counter497>497</counter497>\n"
            "    <counter498>498</counter498>\n"
            "    <counter499>499</counter499>\n"
            "</stats>\n");

    UTEST_ADD_MODULE(counters_yang, LYS_IN_YANG, NULL, NULL);

    check_print_parse(state, data_xml);

    free(counters_yang);
    free(data_xml);
}

#if 0

static void
test_types(void **state)
{
    struct state *st = (*state);
    int ret;

    ly_ctx_set_searchdir(st->ctx, TESTS_DIR "/data/files");
    assert_non_null(ly_ctx_load_module(st->ctx, "types", NULL));

    st->dt1 = lyd_parse_path(st->ctx, TESTS_DIR "/data/files/types.xml", LYD_XML, LYD_OPT_CONFIG);
    assert_ptr_not_equal(st->dt1, NULL);

    ret = lyd_print_mem(&st->mem, st->dt1, LYD_LYB, LYP_WITHSIBLINGS);
    assert_int_equal(ret, 0);

    st->dt2 = lyd_parse_mem(st->ctx, st->mem, LYD_LYB, LYD_OPT_CONFIG | LYD_OPT_STRICT);
    assert_ptr_not_equal(st->dt2, NULL);

    check_data_tree(st->dt1, st->dt2);
}

static void
test_annotations(void **state)
{
    struct state *st = (*state);
    int ret;

    ly_ctx_set_searchdir(st->ctx, TESTS_DIR "/data/files");
    assert_non_null(ly_ctx_load_module(st->ctx, "annotations", NULL));

    st->dt1 = lyd_parse_path(st->ctx, TESTS_DIR "/data/files/annotations.xml", LYD_XML, LYD_OPT_CONFIG);
    assert_ptr_not_equal(st->dt1, NULL);

    ret = lyd_print_mem(&st->mem, st->dt1, LYD_LYB, LYP_WITHSIBLINGS);
    assert_int_equal(ret, 0);

    st->dt2 = lyd_parse_mem(st->ctx, st->mem, LYD_LYB, LYD_OPT_CONFIG | LYD_OPT_STRICT);
    assert_ptr_not_equal(st->dt2, NULL);

    check_data_tree(st->dt1, st->dt2);
}

static void
test_similar_annot_names(void **state)
{
    struct state *st = (*state);
    int ret;

    ly_ctx_set_searchdir(st->ctx, TESTS_DIR "/data/files");
    assert_non_null(ly_ctx_load_module(st->ctx, "annotations", NULL));

    st->dt1 = lyd_parse_path(st->ctx, TESTS_DIR "/data/files/similar-annot-names.xml", LYD_XML, LYD_OPT_CONFIG);
    assert_ptr_not_equal(st->dt1, NULL);

    ret = lyd_print_mem(&st->mem, st->dt1, LYD_LYB, LYP_WITHSIBLINGS);
    assert_int_equal(ret, 0);

    st->dt2 = lyd_parse_mem(st->ctx, st->mem, LYD_LYB, LYD_OPT_CONFIG | LYD_OPT_STRICT);
    assert_ptr_not_equal(st->dt2, NULL);

    check_data_tree(st->dt1, st->dt2);
}

static void
test_many_child_annot(void **state)
{
    struct state *st = (*state);
    int ret;

    ly_ctx_set_searchdir(st->ctx, TESTS_DIR "/data/files");
    assert_non_null(ly_ctx_load_module(st->ctx, "annotations", NULL));

    st->dt1 = lyd_parse_path(st->ctx, TESTS_DIR "/data/files/many-childs-annot.xml", LYD_XML, LYD_OPT_CONFIG);
    assert_ptr_not_equal(st->dt1, NULL);

    ret = lyd_print_mem(&st->mem, st->dt1, LYD_LYB, LYP_WITHSIBLINGS);
    assert_int_equal(ret, 0);

    st->dt2 = lyd_parse_mem(st->ctx, st->mem, LYD_LYB, LYD_OPT_CONFIG | LYD_OPT_STRICT);
    assert_ptr_not_equal(st->dt2, NULL);

    check_data_tree(st->dt1, st->dt2);
}

static void
test_union(void **state)
{
    struct state *st = (*state);
    int ret;

    ly_ctx_set_searchdir(st->ctx, TESTS_DIR "/data/files");
    assert_non_null(ly_ctx_load_module(st->ctx, "union", NULL));

    st->dt1 = lyd_parse_path(st->ctx, TESTS_DIR "/data/files/union.xml", LYD_XML, LYD_OPT_CONFIG);
    assert_ptr_not_equal(st->dt1, NULL);

    ret = lyd_print_mem(&st->mem, st->dt1, LYD_LYB, LYP_WITHSIBLINGS);
    assert_int_equal(ret, 0);

    st->dt2 = lyd_parse_mem(st->ctx, st->mem, LYD_LYB, LYD_OPT_CONFIG | LYD_OPT_STRICT);
    assert_ptr_not_equal(st->dt2, NULL);

    check_data_tree(st->dt1, st->dt2);
}

static void
test_union2(void **state)
{
    struct state *st = (*state);
    int ret;

    ly_ctx_set_searchdir(st->ctx, TESTS_DIR "/data/files");
    assert_non_null(ly_ctx_load_module(st->ctx, "statements", NULL));

    st->dt1 = lyd_parse_path(st->ctx, TESTS_DIR "/data/files/union2.xml", LYD_XML, LYD_OPT_CONFIG);
    assert_ptr_not_equal(st->dt1, NULL);

    ret = lyd_print_mem(&st->mem, st->dt1, LYD_LYB, LYP_WITHSIBLINGS);
    assert_int_equal(ret, 0);

    st->dt2 = lyd_parse_mem(st->ctx, st->mem, LYD_LYB, LYD_OPT_CONFIG | LYD_OPT_STRICT);
    assert_ptr_not_equal(st->dt2, NULL);

    check_data_tree(st->dt1, st->dt2);
}

static void
test_collisions(void **state)
{
    struct state *st = (*state);
    int ret;

    ly_ctx_set_searchdir(st->ctx, TESTS_DIR "/data/files");
    assert_non_null(ly_ctx_load_module(st->ctx, "annotations", NULL));

    st->dt1 = lyd_parse_path(st->ctx, TESTS_DIR "/data/files/collisions.xml", LYD_XML, LYD_OPT_CONFIG);
    assert_ptr_not_equal(st->dt1, NULL);

    ret = lyd_print_mem(&st->mem, st->dt1, LYD_LYB, LYP_WITHSIBLINGS);
    assert_int_equal(ret, 0);

    st->dt2 = lyd_parse_mem(st->ctx, st->mem, LYD_LYB, LYD_OPT_CONFIG | LYD_OPT_STRICT);
    assert_ptr_not_equal(st->dt2, NULL);

    check_data_tree(st->dt1, st->dt2);
}

static void
test_anydata(void **state)
{
    struct state *st = (*state);
    const struct lys_module *mod;
    int ret;
    const char *test_anydata =
            "module test-anydata {"
            "   namespace \"urn:test-anydata\";"
            "   prefix ya;"
            ""
            "   container cont {"
            "       anydata ntf;"
            "   }"
            "}";

    assert_non_null(ly_ctx_load_module(st->ctx, "ietf-netconf-notifications", NULL));

    st->dt1 = lyd_parse_path(st->ctx, TESTS_DIR "/data/files/ietf-netconf-notifications.json", LYD_JSON, LYD_OPT_NOTIF | LYD_OPT_TRUSTED, NULL);
    assert_ptr_not_equal(st->dt1, NULL);

    / *get notification in LYB format to set as anydata content * /
    ret = lyd_print_mem(&st->mem, st->dt1, LYD_LYB, LYP_WITHSIBLINGS);
    assert_int_equal(ret, 0);

    lyd_free_withsiblings(st->dt1);
    st->dt1 = NULL;

    / *now comes the real test, test anydata * /
    mod = lys_parse_mem(st->ctx, test_anydata, LYS_YANG);
    assert_non_null(mod);

    st->dt1 = lyd_new(NULL, mod, "cont");
    assert_non_null(st->dt1);

    assert_non_null(lyd_new_anydata(st->dt1, NULL, "ntf", st->mem, LYD_ANYDATA_LYBD));
    st->mem = NULL;

    ret = lyd_print_mem(&st->mem, st->dt1, LYD_LYB, LYP_WITHSIBLINGS);
    assert_int_equal(ret, 0);

    ret = lyd_validate(&st->dt1, LYD_OPT_CONFIG, NULL);
    assert_int_equal(ret, 0);

    st->dt2 = lyd_parse_mem(st->ctx, st->mem, LYD_LYB, LYD_OPT_CONFIG | LYD_OPT_STRICT);
    assert_ptr_not_equal(st->dt2, NULL);

    check_data_tree(st->dt1, st->dt2);

    /* and also test the embedded notification itself */
    free(st->mem);
    ret = lyd_lyb_data_length(((struct lyd_node_anydata *)st->dt1->child)->value.mem);
    st->mem = malloc(ret);
    memcpy(st->mem, ((struct lyd_node_anydata *)st->dt1->child)->value.mem, ret);

    lyd_free_withsiblings(st->dt2);
    st->dt2 = lyd_parse_mem(st->ctx, st->mem, LYD_LYB, LYD_OPT_NOTIF | LYD_OPT_STRICT | LYD_OPT_NOEXTDEPS, NULL);
    assert_ptr_not_equal(st->dt2, NULL);

    /* parse the JSON again for this comparison */
    lyd_free_withsiblings(st->dt1);
    st->dt1 = lyd_parse_path(st->ctx, TESTS_DIR "/data/files/ietf-netconf-notifications.json", LYD_JSON, LYD_OPT_NOTIF | LYD_OPT_TRUSTED, NULL);
    assert_ptr_not_equal(st->dt1, NULL);

    check_data_tree(st->dt1, st->dt2);
}

static void
test_submodule_feature(void **state)
{
    struct state *st = (*state);
    const struct lys_module *mod;
    int ret;

    ly_ctx_set_searchdir(st->ctx, TESTS_DIR "/data/files");
    mod = ly_ctx_load_module(st->ctx, "feature-submodule-main", NULL);
    assert_non_null(mod);
    assert_int_equal(lys_features_enable(mod, "test-submodule-feature"), 0);

    st->dt1 = lyd_parse_path(st->ctx, TESTS_DIR "/data/files/test-submodule-feature.json", LYD_JSON, LYD_OPT_CONFIG);
    assert_ptr_not_equal(st->dt1, NULL);

    ret = lyd_print_mem(&st->mem, st->dt1, LYD_LYB, LYP_WITHSIBLINGS);
    assert_int_equal(ret, 0);

    st->dt2 = lyd_parse_mem(st->ctx, st->mem, LYD_LYB, LYD_OPT_CONFIG | LYD_OPT_STRICT);
    assert_ptr_not_equal(st->dt2, NULL);

    check_data_tree(st->dt1, st->dt2);
}

static void
test_coliding_augments(void **state)
{
    struct state *st = (*state);
    int ret;

    ly_ctx_set_searchdir(st->ctx, TESTS_DIR "/data/files");
    assert_non_null(ly_ctx_load_module(st->ctx, "augment-target", NULL));
    assert_non_null(ly_ctx_load_module(st->ctx, "augment0", NULL));
    assert_non_null(ly_ctx_load_module(st->ctx, "augment1", NULL));

    st->dt1 = lyd_parse_path(st->ctx, TESTS_DIR "/data/files/augment.xml", LYD_XML, LYD_OPT_CONFIG);
    assert_ptr_not_equal(st->dt1, NULL);

    ret = lyd_print_mem(&st->mem, st->dt1, LYD_LYB, LYP_WITHSIBLINGS);
    assert_int_equal(ret, 0);

    st->dt2 = lyd_parse_mem(st->ctx, st->mem, LYD_LYB, LYD_OPT_CONFIG | LYD_OPT_STRICT);
    assert_ptr_not_equal(st->dt2, NULL);

    check_data_tree(st->dt1, st->dt2);
}

static void
test_leafrefs(void **state)
{
    struct state *st = (*state);
    int ret;

    ly_ctx_set_searchdir(st->ctx, TESTS_DIR "/data/files");
    assert_non_null(ly_ctx_load_module(st->ctx, "leafrefs2", NULL));

    st->dt1 = lyd_parse_path(st->ctx, TESTS_DIR "/data/files/leafrefs2.json", LYD_JSON, LYD_OPT_CONFIG | LYD_OPT_STRICT);
    assert_ptr_not_equal(st->dt1, NULL);

    ret = lyd_print_mem(&st->mem, st->dt1, LYD_LYB, LYP_WITHSIBLINGS);
    assert_int_equal(ret, 0);

    st->dt2 = lyd_parse_mem(st->ctx, st->mem, LYD_LYB, LYD_OPT_CONFIG | LYD_OPT_STRICT);
    assert_ptr_not_equal(st->dt2, NULL);

    check_data_tree(st->dt1, st->dt2);
}

#endif

int
main(void)
{
    const struct CMUnitTest tests[] = {
        UTEST(tests_leaflist),
        UTEST(tests_list),
        UTEST(tests_any),
        UTEST(test_ietf_interfaces, setup),
        UTEST(test_origin, setup),
        UTEST(test_statements, setup),
        UTEST(test_opaq, setup),
        UTEST(test_collisions, setup),
#if 0
        cmocka_unit_test_setup_teardown(test_types, setup_f, teardown_f),
        cmocka_unit_test_setup_teardown(test_annotations, setup_f, teardown_f),
        cmocka_unit_test_setup_teardown(test_similar_annot_names, setup_f, teardown_f),
        cmocka_unit_test_setup_teardown(test_many_child_annot, setup_f, teardown_f),
        cmocka_unit_test_setup_teardown(test_union, setup_f, teardown_f),
        cmocka_unit_test_setup_teardown(test_union2, setup_f, teardown_f),
        cmocka_unit_test_setup_teardown(test_collisions, setup_f, teardown_f),
        cmocka_unit_test_setup_teardown(test_anydata, setup_f, teardown_f),
        cmocka_unit_test_setup_teardown(test_submodule_feature, setup_f, teardown_f),
        cmocka_unit_test_setup_teardown(test_coliding_augments, setup_f, teardown_f),
        cmocka_unit_test_setup_teardown(test_leafrefs, setup_f, teardown_f),
#endif
    };

    return cmocka_run_group_tests(tests, NULL, NULL);
}
