/*
    libzint - the open source barcode library
    Copyright (C) 2019 - 2020 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/* vim: set ts=4 sw=4 et : */

#include "testcommon.h"

static void test_options(int index, int debug) {

    testStart("");

    int ret;
    struct item {
        int symbology;
        int option_1;
        int option_2;
        int option_3;
        unsigned char *data;
        int ret_encode;
        int ret_vector;

        int expected_option_1;
        int expected_option_2;
        int expected_rows;
        int expected_width;
        int compare_previous;
    };
    // s/\/\*[ 0-9]*\*\//\=printf("\/*%3d*\/", line(".") - line("'<"))
    struct item data[] = {
        /*  0*/ { BARCODE_PDF417, -1, -1, -1, "12345", 0, 0, 2, 2, 6, 103, -1 }, // ECC auto-set to 2, cols auto-set to 2
        /*  1*/ { BARCODE_PDF417, -1, -1, 928, "12345", 0, 0, 2, 2, 6, 103, 0 }, // Option 3 ignored
        /*  2*/ { BARCODE_PDF417, -1, -1, 300, "12345", 0, 0, 2, 2, 6, 103, 0 }, // Option 3 ignored
        /*  3*/ { BARCODE_PDF417, 3, -1, -1, "12345", 0, 0, 3, 3, 7, 120, -1 }, // ECC 3, cols auto-set to 3
        /*  4*/ { BARCODE_PDF417, 3, 2, -1, "12345", 0, 0, 3, 2, 10, 103, -1 }, // ECC 3, cols 2
        /*  5*/ { BARCODE_PDF417, 8, 2, -1, "12345", ZINT_ERROR_TOO_LONG, -1, 8, 3, 0, 0, -1 }, // ECC 8, cols 2, fails
        /*  6*/ { BARCODE_PDF417, 7, 2, -1, "12345", 0, 0, 7, 3, 87, 120, -1 }, // ECC 7, cols 2 auto-upped to 3 (no warning, unlike MICROPDF417)
        /*  7*/ { BARCODE_PDF417, -1, 10, -1, "12345", 0, 0, 2, 10, 3, 239, -1 }, // ECC auto-set to 2, cols 10
        /*  8*/ { BARCODE_PDF417, 9, -1, -1, "12345", ZINT_WARN_INVALID_OPTION, 0, 2, 2, 6, 103, -1 }, // Invalid ECC, auto-set
        /*  9*/ { BARCODE_PDF417, -1, 31, -1, "12345", ZINT_WARN_INVALID_OPTION, 0, 2, 2, 6, 103, 0 }, // Invalid cols, auto-set
        /* 10*/ { BARCODE_PDF417, -1, 1, -1, "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHI", ZINT_ERROR_TOO_LONG, -1, 4, 2, 0, 0, -1 }, // Cols 1 too small
        /* 11*/ { BARCODE_MICROPDF417, -1, 5, -1, "12345", ZINT_WARN_INVALID_OPTION, 0, -1, 1, 11, 38, -1 }, // Invalid cols, auto-set to 1
        /* 12*/ { BARCODE_MICROPDF417, -1, 1, -1, "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLM", ZINT_WARN_INVALID_OPTION, 0, -1, 2, 17, 55, -1 }, // Cols 1 too small, auto-upped to 2 with warning
    };
    int data_size = sizeof(data) / sizeof(struct item);

    struct zint_symbol previous_symbol;

    for (int i = 0; i < data_size; i++) {

        if (index != -1 && i != index) continue;

        struct zint_symbol *symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        int length = testUtilSetSymbol(symbol, data[i].symbology, -1 /*input_mode*/, -1, data[i].option_1, data[i].option_2, data[i].option_3, -1 /*output_options*/, data[i].data, -1, debug);

        ret = ZBarcode_Encode(symbol, data[i].data, length);
        assert_equal(ret, data[i].ret_encode, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret_encode, symbol->errtxt);

		assert_equal(symbol->option_1, data[i].expected_option_1, "i:%d symbol->option_1 %d != %d (%d) (%s)\n", i, symbol->option_1, data[i].expected_option_1, data[i].option_1, symbol->errtxt);
		assert_equal(symbol->option_2, data[i].expected_option_2, "i:%d symbol->option_2 %d != %d (%d) (%s)\n", i, symbol->option_2, data[i].expected_option_2, data[i].option_2, symbol->errtxt);
		if (data[i].option_3 != -1) {
			assert_equal(symbol->option_3, data[i].option_3, "i:%d symbol->option_3 %d != %d\n", i, symbol->option_3, data[i].option_3); // Unchanged
		} else {
			assert_zero(symbol->option_3, "i:%d symbol->option_3 %d != 0\n", i, symbol->option_3);
		}

		assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d (%s)\n", i, symbol->rows, data[i].expected_rows, symbol->errtxt);
		assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d (%s)\n", i, symbol->width, data[i].expected_width, symbol->errtxt);

        if (index == -1 && data[i].compare_previous != -1) {
            ret = testUtilSymbolCmp(symbol, &previous_symbol);
            assert_equal(!ret, !data[i].compare_previous, "i:%d testUtilSymbolCmp !ret %d != %d\n", i, ret, data[i].compare_previous);
        }
        memcpy(&previous_symbol, symbol, sizeof(previous_symbol));

        if (data[i].ret_vector != -1) {
            ret = ZBarcode_Buffer_Vector(symbol, 0);
            assert_equal(ret, data[i].ret_vector, "i:%d ZBarcode_Buffer_Vector ret %d != %d\n", i, ret, data[i].ret_vector);
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_reader_init(int index, int generate, int debug) {

    testStart("");

    int ret;
    struct item {
        int symbology;
        int input_mode;
        int output_options;
        unsigned char *data;
        int ret;
        int expected_rows;
        int expected_width;
        char *expected;
        char *comment;
    };
    struct item data[] = {
        /*  0*/ { BARCODE_PDF417, UNICODE_MODE, READER_INIT, "A", 0, 6, 103, "(12) 4 921 900 29 60 257 719 198 75 123 199 98", "Outputs Test Alpha flag 900" },
        /*  1*/ { BARCODE_MICROPDF417, UNICODE_MODE, READER_INIT, "A", 0, 11, 38, "(11) 921 900 29 900 179 499 922 262 777 478 300", "Outputs Test Alpha flag 900" },
    };
    int data_size = ARRAY_SIZE(data);

    char escaped[1024];

    for (int i = 0; i < data_size; i++) {

        if (index != -1 && i != index) continue;

        struct zint_symbol *symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->debug = ZINT_DEBUG_TEST; // Needed to get codeword dump in errtxt

        int length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, -1 /*eci*/, -1 /*option_1*/, -1 /*option_2*/, -1, data[i].output_options, data[i].data, -1, debug);

        ret = ZBarcode_Encode(symbol, data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (generate) {
            printf("        /*%3d*/ { %s, %s, %s, \"%s\", %s, %d, %d, \"%s\", \"%s\" },\n",
                    i, testUtilBarcodeName(data[i].symbology), testUtilInputModeName(data[i].input_mode), testUtilOutputOptionsName(data[i].output_options),
                    testUtilEscape(data[i].data, length, escaped, sizeof(escaped)),
                    testUtilErrorName(data[i].ret), symbol->rows, symbol->width, symbol->errtxt, data[i].comment);
        } else {
            if (ret < 5) {
                assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d (%s)\n", i, symbol->rows, data[i].expected_rows, data[i].data);
                assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d (%s)\n", i, symbol->width, data[i].expected_width, data[i].data);
                assert_zero(strcmp(symbol->errtxt, data[i].expected), "i:%d strcmp(%s, %s) != 0\n", i, symbol->errtxt, data[i].expected);
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_input(int index, int generate, int debug) {

    testStart("");

    int ret;
    struct item {
        int symbology;
        int input_mode;
        int eci;
        unsigned char *data;
        int ret;
        int expected_eci;
        int expected_rows;
        int expected_width;
        char *expected;
        char *comment;
    };
    // é U+00E9 (\351, 233), UTF-8 C3A9
    // β U+03B2 in ISO 8859-7 Greek (but not other ISO 8859 or Win page), in Shift JIS 0x83C0, UTF-8 CEB2
    struct item data[] = {
        /*  0*/ { BARCODE_PDF417, UNICODE_MODE, -1, "é", 0, 0, 6, 103, "(12) 4 913 233 900 398 878 279 350 217 295 231 77", "" },
        /*  1*/ { BARCODE_PDF417, UNICODE_MODE, 3, "é", 0, 3, 7, 103, "(14) 6 927 3 913 233 900 162 81 551 529 607 384 164 108", "" },
        /*  2*/ { BARCODE_PDF417, UNICODE_MODE, 26, "é", 0, 26, 7, 103, "(14) 6 927 26 901 195 169 574 701 519 908 84 241 360 642", "" },
        /*  3*/ { BARCODE_PDF417, UNICODE_MODE, 9, "β", 0, 9, 7, 103, "(14) 6 927 9 913 226 900 487 92 418 278 838 500 576 84", "" },
        /*  4*/ { BARCODE_PDF417, UNICODE_MODE, -1, "β", ZINT_WARN_USES_ECI, 9, 7, 103, "Warning (14) 6 927 9 913 226 900 487 92 418 278 838 500 576 84", "" },
        /*  5*/ { BARCODE_PDF417, UNICODE_MODE, 3, "β", ZINT_ERROR_INVALID_DATA, 3, 0, 0, "Error 204: Invalid characters in input data", "" },
        /*  6*/ { BARCODE_PDF417, UNICODE_MODE, 899, "A", 0, 899, 7, 103, "(14) 6 927 899 900 29 900 727 69 915 482 371 771 641 35", "" },
        /*  7*/ { BARCODE_PDF417, UNICODE_MODE, 900, "A", 0, 900, 7, 103, "(14) 6 926 0 0 900 29 56 795 921 763 468 267 410 129", "" },
        /*  8*/ { BARCODE_PDF417, UNICODE_MODE, 810899, "A", 0, 810899, 7, 103, "(14) 6 926 899 899 900 29 847 901 749 718 89 792 660 273", "" },
        /*  9*/ { BARCODE_PDF417, UNICODE_MODE, 810900, "A", 0, 810900, 7, 103, "(14) 6 925 0 900 29 900 652 613 857 390 38 450 415 899", "" },
        /* 10*/ { BARCODE_PDF417, UNICODE_MODE, 811799, "A", 0, 811799, 7, 103, "(14) 6 925 899 900 29 900 456 300 328 160 510 753 157 159", "" },
        /* 11*/ { BARCODE_PDF417, UNICODE_MODE, 811800, "A", ZINT_ERROR_INVALID_OPTION, 811800, 0, 0, "Error 472: Invalid ECI", "" },
        /* 12*/ { BARCODE_MICROPDF417, UNICODE_MODE, -1, "é", 0, 0, 11, 38, "(11) 913 233 900 900 805 609 847 211 598 4 603", "" },
        /* 13*/ { BARCODE_MICROPDF417, UNICODE_MODE, 3, "é", 0, 3, 11, 38, "(11) 927 3 913 233 803 477 85 249 824 813 830", "" },
        /* 14*/ { BARCODE_MICROPDF417, UNICODE_MODE, 26, "é", 0, 26, 6, 82, "(18) 927 26 901 195 169 900 288 96 509 365 709 784 713 403 219 81 851 866", "" },
        /* 15*/ { BARCODE_MICROPDF417, UNICODE_MODE, 9, "β", 0, 9, 11, 38, "(11) 927 9 913 226 23 103 74 194 394 667 324", "" },
        /* 16*/ { BARCODE_MICROPDF417, UNICODE_MODE, -1, "β", ZINT_WARN_USES_ECI, 9, 11, 38, "Warning (11) 927 9 913 226 23 103 74 194 394 667 324", "" },
        /* 17*/ { BARCODE_MICROPDF417, UNICODE_MODE, 3, "β", ZINT_ERROR_INVALID_DATA, 3, 0, 0, "Error 204: Invalid characters in input data", "" },
        /* 18*/ { BARCODE_MICROPDF417, UNICODE_MODE, 899, "A", 0, 899, 11, 38, "(11) 927 899 900 29 533 437 884 3 617 241 747", "" },
        /* 19*/ { BARCODE_MICROPDF417, UNICODE_MODE, 900, "A", 0, 900, 6, 82, "(18) 926 0 0 900 29 900 913 543 414 141 214 886 461 1 419 422 54 495", "" },
        /* 20*/ { BARCODE_MICROPDF417, UNICODE_MODE, 810899, "A", 0, 810899, 6, 82, "(18) 926 899 899 900 29 900 351 555 241 509 787 583 3 326 41 628 534 151", "" },
        /* 21*/ { BARCODE_MICROPDF417, UNICODE_MODE, 810900, "A", 0, 810900, 11, 38, "(11) 925 0 900 29 233 533 43 483 708 659 704", "" },
        /* 22*/ { BARCODE_MICROPDF417, UNICODE_MODE, 811800, "A", ZINT_ERROR_INVALID_OPTION, 811800, 0, 0, "Error 473: Invalid ECI", "" },
        /* 23*/ { BARCODE_HIBC_PDF, UNICODE_MODE, -1, ",", ZINT_ERROR_INVALID_DATA, 0, 0, 0, "Error 203: Invalid characters in data", "" },
        /* 24*/ { BARCODE_HIBC_MICPDF, UNICODE_MODE, -1, ",", ZINT_ERROR_INVALID_DATA, 0, 0, 0, "Error 203: Invalid characters in data", "" },
    };
    int data_size = ARRAY_SIZE(data);

    char escaped[1024];

    for (int i = 0; i < data_size; i++) {

        if (index != -1 && i != index) continue;

        struct zint_symbol *symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->debug = ZINT_DEBUG_TEST; // Needed to get codeword dump in errtxt

        int length = testUtilSetSymbol(symbol, data[i].symbology, data[i].input_mode, data[i].eci, -1 /*option_1*/, -1, -1, -1 /*output_options*/, data[i].data, -1, debug);

        ret = ZBarcode_Encode(symbol, data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (generate) {
            printf("        /*%3d*/ { %s, %s, %d, \"%s\", %s, %d, %d, %d, \"%s\", \"%s\" },\n",
                    i, testUtilBarcodeName(data[i].symbology), testUtilInputModeName(data[i].input_mode), data[i].eci,
                    testUtilEscape(data[i].data, length, escaped, sizeof(escaped)), testUtilErrorName(data[i].ret),
                    symbol->eci, symbol->rows, symbol->width, symbol->errtxt, data[i].comment);
        } else {
            if (ret < 5) {
                assert_equal(symbol->eci, data[i].expected_eci, "i:%d symbol->eci %d != %d (%s)\n", i, symbol->eci, data[i].expected_eci, data[i].data);
                assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d (%s)\n", i, symbol->rows, data[i].expected_rows, data[i].data);
                assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d (%s)\n", i, symbol->width, data[i].expected_width, data[i].data);
                assert_zero(strcmp(symbol->errtxt, data[i].expected), "i:%d strcmp(%s, %s) != 0\n", i, symbol->errtxt, data[i].expected);
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

static void test_encode(int index, int generate, int debug) {

    testStart("");

    int do_bwipp = (debug & ZINT_DEBUG_TEST_BWIPP) && testUtilHaveGhostscript(); // Only do BWIPP test if asked, too slow otherwise

    int ret;
    struct item {
        int symbology;
        int input_mode;
        int option_1;
        int option_2;
        unsigned char *data;
        int ret;

        int expected_rows;
        int expected_width;
        int bwipp_cmp;
        char *comment;
        char *expected;
    };
    struct item data[] = {
        /*  0*/ { BARCODE_PDF417, UNICODE_MODE, 1, 2, "PDF417 Symbology Standard", 0, 10, 103, 0, "ISO 15438:2015 Figure 1, same, BWIPP uses different encodation, same codeword count",
                    "1111111101010100011101010011100000111010110011110001110111011001100011110101011110000111111101000101001"
                    "1111111101010100011111010100110000110100001110001001111010001010000011111010100110000111111101000101001"
                    "1111111101010100011101010111111000101100110111100001110111111000101011010100111110000111111101000101001"
                    "1111111101010100010101111101111100100000100100000101000101000100000010101111001111000111111101000101001"
                    "1111111101010100011010111000100000111100100000101001001000011111011011010111000100000111111101000101001"
                    "1111111101010100011110101111010000100111100001010001100111110010010011110101111001000111111101000101001"
                    "1111111101010100010100111001110000101110001110100001111001101000111011010011101111000111111101000101001"
                    "1111111101010100011010111111011110111101011001100001010011111101110011010111111011110111111101000101001"
                    "1111111101010100011010011011111100110000101001111101101111100010001010100110011111000111111101000101001"
                    "1111111101010100010100011000001100100010111101111001100011100011001011010001100011100111111101000101001"
                },
        /*  1*/ { BARCODE_PDF417, UNICODE_MODE, 1, 2, "PDF417", 0, 5, 103, 1, "ISO 15438:2015 Annex Q example for generating ECC",
                    "1111111101010100011110101011110000110101000110000001110111011001100011110101011110000111111101000101001"
                    "1111111101010100011111101010011100110100001110001001111010001010000011111101010111000111111101000101001"
                    "1111111101010100011101010111111000101100110011110001100011111001001011101010011111100111111101000101001"
                    "1111111101010100010101111001111000101011101110000001100001101000100010101111001111000111111101000101001"
                    "1111111101010100011101011100011000100001101011111101111110110001011011101011100110000111111101000101001"
                },
        /*  2*/ { BARCODE_PDF417, UNICODE_MODE, 0, 1, "ABCDEFGHIJKLMNOPQRSTUVWXYZ ", 0, 17, 86, 1, "Text Compaction Alpha",
                    "11111111010101000111110101001111101101011001110000011101010111000000111111101000101001"
                    "11111111010101000111111010101110001111110101011100011110101000100000111111101000101001"
                    "11111111010101000110101011111000001010011001111100011101010111111000111111101000101001"
                    "11111111010101000111010010111000001101000001011000011010111101111100111111101000101001"
                    "11111111010101000111010111001100001010000111110011011110101110001110111111101000101001"
                    "11111111010101000111110101110000101011001110011111011110101111010000111111101000101001"
                    "11111111010101000101001110001110001100100001101110010100111011100000111111101000101001"
                    "11111111010101000101011111000011001101100111100100010101111110001110111111101000101001"
                    "11111111010101000101001101111100001001100001000111011010011011111100111111101000101001"
                    "11111111010101000110100011101111001001100111100111010100011000110000111111101000101001"
                    "11111111010101000111010011101100001001100101111110011110100111001110111111101000101001"
                    "11111111010101000110100010011111001000001010111100010100010001111000111111101000101001"
                    "11111111010101000110100000101100001000110011001000010100001100001100111111101000101001"
                    "11111111010101000111101000101000001110001100111010011111010001001100111111101000101001"
                    "11111111010101000111101000011110101100011110010011011101000011111010111111101000101001"
                    "11111111010101000110010110111000001100011000100001011100101000111000111111101000101001"
                    "11111111010101000101000111100100001110000101100010010100011110000100111111101000101001"
                },
        /*  3*/ { BARCODE_PDF417, UNICODE_MODE, 1, 1, "abcdefghijklmnopqrstuvwxyz ", 0, 19, 86, 1, "Text Compaction Lower",
                    "11111111010101000110101000110000001101011001110000011101010111000000111111101000101001"
                    "11111111010101000111110101001100001100000101110010011111010100011000111111101000101001"
                    "11111111010101000110101011111000001111101011110110011010100111110000111111101000101001"
                    "11111111010101000111101001011110001010001110111000011010111101111100111111101000101001"
                    "11111111010101000110101110001000001010001111000001011010111000010000111111101000101001"
                    "11111111010101000111110101110000101110010011111001011110101111001000111111101000101001"
                    "11111111010101000110100111000111101011001111000111010100111011100000111111101000101001"
                    "11111111010101000110101111110111101111101100100100011110100101000000111111101000101001"
                    "11111111010101000101001101111100001011100100000011010100110011111000111111101000101001"
                    "11111111010101000101000111001110001100010001001100010100011000110000111111101000101001"
                    "11111111010101000110100111001000001110001001110011011010011100010000111111101000101001"
                    "11111111010101000110100010011111001101111000100110010100010000111100111111101000101001"
                    "11111111010101000111010000010111001000110111101110010100001100001100111111101000101001"
                    "11111111010101000111111010001011101100011011110010011110100010001000111111101000101001"
                    "11111111010101000111101000011110101000001100101110011010000010111110111111101000101001"
                    "11111111010101000111001011011110001000010010000001011100101000111000111111101000101001"
                    "11111111010101000101000111100010001111010000101000010100011110000010111111101000101001"
                    "11111111010101000111111001011101101010000001001111010010111001111110111111101000101001"
                    "11111111010101000111011010000110001000100111001110011110110100111000111111101000101001"
                },
        /*  4*/ { BARCODE_PDF417, UNICODE_MODE, 1, 4, "0123456&\015\011,:#-.$/+%*=^ 789", 0, 5, 137, 1, "Text Compaction Mixed",
                    "11111111010101000111101010111100001110101100111100010000110111001100110101111001111101010001110111000011101010011100000111111101000101001"
                    "11111111010101000111111010100111001010001111000001011101101111001100110110011110010001110010000011010011111101010111000111111101000101001"
                    "11111111010101000110101001111100001100111010000111011011110010110000100000101011110001101111101010000011101010011111100111111101000101001"
                    "11111111010101000101011110011110001000010000100001010010011000011000110010000100110001000011000110010010101111101111100111111101000101001"
                    "11111111010101000111010111000110001001111001001111010000101111101100100011110010111101001111110110111011101011100110000111111101000101001"
                },
        /*  5*/ { BARCODE_PDF417, UNICODE_MODE, 3, 2, ";<>@[\\]_'~!\015\011,:\012-.$/\"|*()?{", 0, 16, 103, 1, "Text Compaction Punctuation",
                    "1111111101010100011111010100111110111010110011110001000111011100100011110101011110000111111101000101001"
                    "1111111101010100011111010100001100111111010101110001101011111101111011110101000100000111111101000101001"
                    "1111111101010100011101010111111000101000001000111101011011001111000011010100001111100111111101000101001"
                    "1111111101010100011101001011100000110000110010100001100100001101110010101111001111000111111101000101001"
                    "1111111101010100011101011100000110110110011110010001110010000011010011110101110001110111111101000101001"
                    "1111111101010100011110101111010000110011101000011101101111001011000011101011111001000111111101000101001"
                    "1111111101010100010100111000111000110001101000100001000110011001000011010011101111000111111101000101001"
                    "1111111101010100011110100100100000111000110011101001110000010111011010101111110001110111111101000101001"
                    "1111111101010100011010011011111100110011111010100001001111000010010011111101001110110111111101000101001"
                    "1111111101010100011010001110111100110101000110000001100011010010000011010001100011100111111101000101001"
                    "1111111101010100011101001110000110110100111110011101001000011110100011110100111001110111111101000101001"
                    "1111111101010100010100010001111000110010011011111101000101000001111010100011001111100111111101000101001"
                    "1111111101010100011010000010110000110001011001110001100100010011000010100001100000110111111101000101001"
                    "1111111101010100011110100010000010110000010001110101111010011000110011111010001001100111111101000101001"
                    "1111111101010100011101000011111010111111010001101001011000010011100010010101111000000111111101000101001"
                    "1111111101010100011001011011100000110011001100001101100100101100000011110010100011110111111101000101001"
                },
        /*  6*/ { BARCODE_PDF417, UNICODE_MODE, 2, 3, "12345678901234", 0, 5, 120, 1, "Numeric Compaction",
                    "111111110101010001111010101111000011101010001110000100111101111010001001011100001110011111010101111100111111101000101001"
                    "111111110101010001111110101000111011010000001110010111111011010011001111010100000010011111101010111000111111101000101001"
                    "111111110101010001010100111100000010111000110011100101110011000011101110001111110101011101010001111110111111101000101001"
                    "111111110101010001010111100111100010001100001100010100001100011101101110101100111100011010111100111110111111101000101001"
                    "111111110101010001110101110000110011000000101110010110001001110000101011001000111111011101011100110000111111101000101001"
               },
        /*  7*/ { BARCODE_PDF417, UNICODE_MODE, 1, 4, "\177\177\177\177\177\177\177\177\177\177\177", 0, 4, 137, 1, "Byte Compaction",
                    "11111111010101000111101010111100001101011011100000010000010000100010111001001100111101000010100001000011101010011100000111111101000101001"
                    "11111111010101000111110101001100001110010000111011010100111110000110111101001100001101111101000100011011111101010111000111111101000101001"
                    "11111111010101000110101001111100001010000001011110010100000010111100101000000101111001010000001011110011010100111110000111111101000101001"
                    "11111111010101000101011110011110001010001000001000011011000010100000111000110001001101100111000110010010101111101111100111111101000101001"
               },
        /*  8*/ { BARCODE_PDF417, UNICODE_MODE, 1, 4, "\177\177\177\177\177\177\177\177\177\177\177\177", 0, 4, 137, 1, "Byte Compaction, mod 6 == 0 (924 emitted)",
                    "11111111010101000111101010111100001101011011100000011000111000110100111001001100111101000010100001000011101010011100000111111101000101001"
                    "11111111010101000111110101001100001110010000111011010100111110000110111101001100001101111001010010000011111101010111000111111101000101001"
                    "11111111010101000110101001111100001001110000100110010011000100001110101000011001111101101000101111100011010100111110000111111101000101001"
                    "11111111010101000101011110011110001101000100011000010011000111001100110001100001000101110100010111000010101111101111100111111101000101001"
                },
        /*  9*/ { BARCODE_PDF417, UNICODE_MODE, -1, 5, "ABCDEF1234567890123\177\177\177\177VWXYZ", 0, 6, 154, 1, "Text, Numeric, Byte, Text",
                    "1111111101010100011110101011110000110101110111100001111010101111000010100111001110000110100000101100001001111011110100011110101001111000111111101000101001"
                    "1111111101010100011110101000010000111101011001100001010011110000100011111100011101010110000010111000101111001011011000011111101010111000111111101000101001"
                    "1111111101010100011101010011111100110011111101100101010000001011110010100000010111100101000000101111001010000001011110010101000011110000111111101000101001"
                    "1111111101010100010101111001111000100001100011001001000110111101110011110111101101100110111100111000101000011000110010011111010111111010111111101000101001"
                    "1111111101010100011010111000001000101111110101100001011111101011000011001011111001110111100100100100001011111101011000011101011100110000111111101000101001"
                    "1111111101010100011111010111100110110111110110011001101001011111000010101110011111100100100001000111101011000000101110011110101111101100111111101000101001"
                },
        /* 10*/ { BARCODE_PDF417COMP, UNICODE_MODE, 1, 2, "PDF417 APK", 0, 6, 69, 0, "ISO 15438:2015 Figure G.1, same, BWIPP uses different encodation, same codeword count",
                    "111111110101010001111010101111000011010100001100000111011101100110001"
                    "111111110101010001111010100010000011010000111000100111101000101000001"
                    "111111110101010001110101011111100010110011011110000100111110011000101"
                    "111111110101010001010111100111100011001100100000010100001100011001001"
                    "111111110101010001111010111000111011011000001111010110010011101000001"
                    "111111110101010001111010111101000011110100111101000110010010011111001"
                },
        /* 11*/ { BARCODE_PDF417COMP, UNICODE_MODE, 4, 4, "ABCDEFG", 0, 10, 103, 1, "",
                    "1111111101010100011101010011100000110101000011000001111010101111000010100111001110000110100000101100001"
                    "1111111101010100011110101000000100110100000011100101011111101011000010111111010110000101111110101100001"
                    "1111111101010100011010100111110000101111001100011001000001111010100010011111001100100111001011111001001"
                    "1111111101010100010101111101111100100001011000110001100011100110010011011011100111000101111000111100101"
                    "1111111101010100011101011110111000111110011010000101110000010110010011100100011001000100011111001011101"
                    "1111111101010100011110101111001000111111010111011001110111111010100010100000010011110100111100110001101"
                    "1111111101010100010100111001110000100001101101000001101100110000110011011000110011000101000000101000001"
                    "1111111101010100011110100100010000111101001000010001111101011011100011100100001100100101001111100011001"
                    "1111111101010100010100110011111000100110000110111101100111000010111010010001011110000110011111010001001"
                    "1111111101010100010100011000001100110001101010000001100011000110011011001001101110000111110111110101001"
                },
        /* 12*/ { BARCODE_HIBC_PDF, UNICODE_MODE, -1, 3, "H123ABC01234567890D", 0, 8, 120, 0, "BWIPP uses different encodation, same codeword count but zint half-pad shorter",
                    "111111110101010001111101010111110011101011001111000100000100010010001110001110100010011111010101111100111111101000101001"
                    "111111110101010001111110101000111011110000010001010110101111110111101111100011101101011110101001000000111111101000101001"
                    "111111110101010001010100111100000011111010111101100100001111000101001100101000011111011101010001111110111111101000101001"
                    "111111110101010001101011110011111010000100000101000110001110110010001100100101100000011010111100111110111111101000101001"
                    "111111110101010001110101110000110010111111010110000110100111100110001011111101011000011110101110011100111111101000101001"
                    "111111110101010001111101011110110011000111110010010110010100111110001000001001001111011101011111010000111111101000101001"
                    "111111110101010001110100111011111010100110001100000110100011100111101111010010111100011101001110111110111111101000101001"
                    "111111110101010001111101001011000011100001001100100111010000011001001111011000110100010101111110111000111111101000101001"
                },
        /* 13*/ { BARCODE_HIBC_PDF, UNICODE_MODE, 1, 3, "A123BJC5D6E71", 0, 6, 120, 1, "BWIPP example",
                    "111111110101010001111010101111000011110101101111100100000100010010001000011011100110011111010101111100111111101000101001"
                    "111111110101010001111010100010000011110000010001010110101111110111101111000001000101011111101010111000111111101000101001"
                    "111111110101010001010100111100000010110001100011110101111110111101101000111100011011010101000111100000111111101000101001"
                    "111111110101010001010111100111100011100011101001000100001101111011101110001110100010011010111100111110111111101000101001"
                    "111111110101010001111010111000111011010111110011100110100000011100101111110101000111011101011100110000111111101000101001"
                    "111111110101010001111101011110110010011100110011100100011110110011001011001011100000011110101111000100111111101000101001"
                },
        /* 14*/ { BARCODE_MICROPDF417, UNICODE_MODE, -1, 1, "ABCDEFGHIJKLMNOPQRSTUV", 0, 20, 38, 1, "ISO 24728:2006 Figure 1 1st 1x20, same",
                    "11110101001000011000110010011110101001"
                    "11100101001111110101011100011100101001"
                    "11101101001010011001111100011101101001"
                    "11101001001101000001011000011101001001"
                    "11101001101010000111110011011101001101"
                    "11101011101011001110011111011101011101"
                    "11101011001100100001101110011101011001"
                    "11101010001101100111100100011101010001"
                    "11001010001001100001000111011001010001"
                    "11001011001001100111100111011001011001"
                    "11001011101001100101111110011001011101"
                    "11001001101000001010111100011001001101"
                    "11001101101111001100100111011001101101"
                    "11101101101011100111111010011101101101"
                    "11100101101110011110100111011100101101"
                    "11000101101110001010011100011000101101"
                    "11000101001100010111100110011000101001"
                    "11001101001000011001000111011001101001"
                    "11011101001111011111011010011011101001"
                    "11011001001100010001110100011011001001"
                },
        /* 15*/ { BARCODE_MICROPDF417, UNICODE_MODE, -1, 2, "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCD", 0, 20, 55, 1, "ISO 24728:2006 Figure 1 2nd 2x20, same",
                    "1111010100100001100011001001111010101111000011110101001"
                    "1110010100110101111110111101111101000100110011100101001"
                    "1110110100101101100111100001011001110011111011101101001"
                    "1110100100110010000110111001110001011001111011101001001"
                    "1110100110111001000001101001001101011111100011101001101"
                    "1110101110110111100101100001000001010111100011101011101"
                    "1110101100100011001100100001100111001110011011101011001"
                    "1110101000111111010101110001101011111101111011101010001"
                    "1100101000101000001000111101011011001111000011001010001"
                    "1100101100110010001001100001100100001101110011001011001"
                    "1100101110110110011110010001110010000011010011001011101"
                    "1100100110110011101000011101101111001011000011001001101"
                    "1100110110110001101000100001000110011001000011001101101"
                    "1110110110111000110011101001111110101011100011101101101"
                    "1110010110101001100111110001000110010011100011100101101"
                    "1100010110110110011011000001100110100010000011000101101"
                    "1100010100111111001010011101110000100110100011000101001"
                    "1100110100111001111100101101111110010011010011001101001"
                    "1101110100111010110011110001000001001101100011011101001"
                    "1101100100111100110110100001001001111001000011011001001"
                },
        /* 16*/ { BARCODE_MICROPDF417, UNICODE_MODE, -1, 3, "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMN", 0, 20, 82, 1, "ISO 24728:2006 Figure 1 3rd 3x20",
                    "1100100010100001100011001001011110010111101010111100001010011100111000011100101101"
                    "1110100010111110100010011001011110110101000011111001101001011110010000011000101101"
                    "1110110010111100010111101001001110110110111011001111001001100001000111011000101001"
                    "1100110010100110011110011101001110100111001100001001101100011010001000011001101001"
                    "1101110010110000101111011001001100100111000110011101001111110101011100011011101001"
                    "1101111010101001100111110001001100110101000001000111101011011001111000011011001001"
                    "1100111010110010001001100001001000110110010000110111001110001011001111011011001101"
                    "1110111010111001000001101001001000010100110101111110001001100101111110011011011101"
                    "1110011010100000101011110001011000010110111110101000001101111100010001011011011001"
                    "1111011010111101010111100001011100010101001110011100001101000001011000011011010001"
                    "1111001010101000011111001101011100110100101111001000001110110111100110011010010001"
                    "1110001010110111011001111001011100100101111100110010001110011111010110011010110001"
                    "1100001010111100111101010001011101100111100111100101001111101110100111011010111001"
                    "1100011010111001101110010001001101100101001111000001001111010001100110011010111101"
                    "1100010010110010001111110101000101100110011100001011101111001000111101011010011101"
                    "1110010010100110001100000101000101000110111101110001001101100110110000011010011001"
                    "1111010010100111010001111101001101000100110111110100001111001110111010011010001001"
                    "1111010110101100100000011101011101000110001101000111101000000100101111011010001101"
                    "1111010100101111011110100001011001000111110011010111101011110111110110011010000101"
                    "1110010100110010001111011001011001100111000010111011001110001011100110011011000101"
                },
        /* 17*/ { BARCODE_MICROPDF417, UNICODE_MODE, -1, 4, "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZAB", 0, 20, 99, 1, "ISO 24728:2006 Figure 1 4th 4x20, same",
                    "110010001010000110001100100111101010111100001011110010101001110011100001101000001011000011100101101"
                    "111010001010100001111100110100101111001000001011110110111011011110011001101100111100100011000101101"
                    "111011001010011000010001110110011101000011101001110110110111100101100001000001010111100011000101001"
                    "110011001010001100110010000110011100111001101001110100111101010111100001010011100111000011001101001"
                    "110111001011111010001001100101000011111001101001100100100101111001000001110110111100110011011101001"
                    "110111101011011101100111100100110000100011101001100110110011101000011101101111001011000011011001001"
                    "110011101011000110100010000100011001100100001001000110110011100111001101111010101111000011011001101"
                    "111011101011010111111011110111110100010011001001000010101000011111001101001011110010000011011011101"
                    "111001101011110001011110100110111011001111001011000010100110000100011101100111010000111011011011001"
                    "111101101011100110000100110110001101000100001011100010100011001100100001100111001110011011011010001"
                    "111100101011111101010111000110101111110111101011100110111110100010011001010000111110011011010010001"
                    "111000101010110011100111110111100010111101001011100100110111011001111001001100001000111011010110001"
                    "110000101010011001111001110111001100001001101011101100110001101000100001000110011001000011010111001"
                    "110001101011100011001110100111111010101110001001101100111101001110111001011111011001111011010111101"
                    "110001001011110101111101100101111001110011101000101100101110110111000001000010000101111011010011101"
                    "111001001010000010100000100111000111010010001000101000110010000010110001101100010000001011010011001"
                    "111101001011110111001110100110000010011100101001101000111101100110001001011110100001111011010001001"
                    "111101011010000001010011110100110110001111001011101000110000111100101101111010011110010011010001101"
                    "111101010011100011101010000110001011101111001011001000111110111101011001100101110111100011010000101"
                    "111001010010001000001111010111100010100001001011001100100111101101111101001110100111110011011000101"
                },
        /* 18*/ { BARCODE_MICROPDF417, UNICODE_MODE, -1, 1, "123456789012345", 0, 14, 38, 1, "Number Compaction",
                    "11101110101011111101001100011101110101"
                    "11100110101110101011111100011100110101"
                    "11110110101000001011001100011110110101"
                    "11110010101111100100110111011110010101"
                    "11100010101000111110110010011100010101"
                    "11000010101010000010100000011000010101"
                    "11000110101110001100111010011000110101"
                    "11000100101000100001001111011000100101"
                    "11100100101011110011110010011100100101"
                    "11110100101110011000011101011110100101"
                    "11110101101000101000001111011110101101"
                    "11110101001111001010011110011110101001"
                    "11100101001101011110000110011100101001"
                    "11101101001101000111111001011101101001"
                },
        /* 19*/ { BARCODE_MICROPDF417, UNICODE_MODE, -1, 2, "\177\177\177", 0, 8, 55, 1, "Byte Compaction",
                    "1100100010100000100001000101010000010010000011001000101"
                    "1110100010111110100010001101111101000100011011101000101"
                    "1110110010110001111100100101100011111001001011101100101"
                    "1100110010100001100011001001000011000110010011001100101"
                    "1101110010111001000111011001011011111101100011011100101"
                    "1101111010111010011111101101001110010011000011011110101"
                    "1100111010111001111001100101000001001101100011001110101"
                    "1110111010111000101111011101110001000011010011101110101"
                },
        /* 20*/ { BARCODE_MICROPDF417, UNICODE_MODE, -1, 2, "\177\177\177\177\177\177", 0, 8, 55, 1, "Byte Compaction, mod 6 == 0 (924 emitted)",
                    "1100100010110001110001101001110010011001111011001000101"
                    "1110100010100010001111010001110010000111011011101000101"
                    "1110110010101000011001111101101000101111100011101100101"
                    "1100110010100001100011001001000011000110010011001100101"
                    "1101110010101110001001111101101000111000001011011100101"
                    "1101111010100011001101111001010010000111100011011110101"
                    "1100111010100100010000100001110111101100001011001110101"
                    "1110111010111110011010100001101100001111010011101110101"
                },
        /* 21*/ { BARCODE_MICROPDF417, UNICODE_MODE, -1, 3, "ABCDEFG\177\177\177", 0, 8, 82, 1, "Text & Byte Compaction",
                    "1100111010100001100011001001000010110111101010111100001010011100111000011001110101"
                    "1110111010111110100010011001000010010110100000011100101101111110101110011101110101"
                    "1110011010101000000101111001000011010101000000101111001010000001011110011100110101"
                    "1111011010100001100011001001000111010100100001010000001111001110100011011110110101"
                    "1111001010111000001001100101000110010111010011000010001110100110000100011110010101"
                    "1110001010101111010001000001000100010101100010000011101000000111001011011100010101"
                    "1100001010111110111010111001001100010110011100011000101101100001100110011000010101"
                    "1100011010110100011100001001001110010110110000111101001100011011110010011000110101"
                },
        /* 22*/ { BARCODE_MICROPDF417, UNICODE_MODE, -1, 4, "\177\177\177abcdefgh1234567890123", 0, 8, 99, 1, "Byte & Text & Numeric Compaction",
                    "110011101010000010000100010101000001001000001000010110101000001001000001010000010010000011001110101"
                    "111011101010111111010110000110000010111001001000010010111101011100111001110100111001100011101110101"
                    "111001101011111001011110110101100110011110001000011010100001111000101001111110101100010011100110101"
                    "111101101011010000100001100100000100110110001000111010100000101101100001001001110011100011110110101"
                    "111100101010111111010110000101111110101100001000110010101111101011100001110011000111010011110010101"
                    "111000101010111111011110110110111111000110101000100010100111000000101101001111101101000011100010101"
                    "110000101011000011010000100100000101101100001001100010101110111110111001111001110010110011000010101"
                    "110001101011101110111100010100100011110100001001110010100000101111000101111001010010000011000110101"
                },
        /* 23*/ { BARCODE_HIBC_MICPDF, UNICODE_MODE, -1, 4, "H123ABC01234567890D", 0, 8, 99, 0, "BWIPP uses different encodation, same codeword count but zint full-pad shorter",
                    "110011101010000110001100100100000100010010001000010110111000111010001001000001001100011011001110101"
                    "111011101011010111111011110111110001110110101000010010111101011100111001011111101001100011101110101"
                    "111001101011001010000111110100011110101000001000011010100111110001101001011011000111100011100110101"
                    "111101101010000110001100100101000010001000001000111010100001100011001001000011000110010011110110101"
                    "111100101010111111010110000101111110101100001000110010111001001100001001111110001101011011110010101"
                    "111000101010001100101110000110010010011111001000100010100111100001101101111110101100010011100010101"
                    "110000101010110110001000000111000101100111101001100010110111101110000101100010101100000011000010101"
                    "110001101011110110000011010111100100001101101001110010101101011111100001111001000110011011000110101"
                },
        /* 24*/ { BARCODE_HIBC_MICPDF, UNICODE_MODE, -1, 1, "/EAH783", 0, 17, 38, 1, "HIBC Provider Applications Standard (PAS) example",
                    "11001101001100011111001001011001101001"
                    "11011101001000001000100100011011101001"
                    "11011001001000100011110100011011001001"
                    "11011001101111010000111101011011001101"
                    "11011011101101100100010000011011011101"
                    "11011011001111001010000100011011011001"
                    "11011010001010000110011111011011010001"
                    "11010010001101001100001110011010010001"
                    "11010110001011111101011000011010110001"
                    "11010111001100011111001001011010111001"
                    "11010111101100110001000010011010111101"
                    "11010011101100001000111001011010011101"
                    "11010011001011110111000111011010011001"
                    "11010001001110001100100011011010001001"
                    "11010001101110010000110100011010001101"
                    "11010000101101100100001111011010000101"
                    "11011000101110111000100010011011000101"
                },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    char escaped[1024];
    char bwipp_buf[8192];
    char bwipp_msg[1024];

    for (int i = 0; i < data_size; i++) {

        if (index != -1 && i != index) continue;

        struct zint_symbol *symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        symbol->input_mode = data[i].input_mode;
        if (data[i].option_1 != -1) {
            symbol->option_1 = data[i].option_1;
        }
        if (data[i].option_2 != -1) {
            symbol->option_2 = data[i].option_2;
        }
        symbol->debug |= debug;

        int length = strlen(data[i].data);

        ret = ZBarcode_Encode(symbol, data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        if (generate) {
            printf("        /*%3d*/ { %s, %s, %d, %d, \"%s\", %s, %d, %d, %d, \"%s\",\n",
                    i, testUtilBarcodeName(data[i].symbology), testUtilInputModeName(data[i].input_mode), data[i].option_1, data[i].option_2,
                    testUtilEscape(data[i].data, length, escaped, sizeof(escaped)), testUtilErrorName(data[i].ret),
                    symbol->rows, symbol->width, data[i].bwipp_cmp, data[i].comment);
            testUtilModulesDump(symbol, "                    ", "\n");
            printf("                },\n");
        } else {
            if (ret < 5) {
                assert_equal(symbol->rows, data[i].expected_rows, "i:%d symbol->rows %d != %d (%s)\n", i, symbol->rows, data[i].expected_rows, data[i].data);
                assert_equal(symbol->width, data[i].expected_width, "i:%d symbol->width %d != %d (%s)\n", i, symbol->width, data[i].expected_width, data[i].data);

                if (ret == 0) {
                    int width, row;
                    ret = testUtilModulesCmp(symbol, data[i].expected, &width, &row);
                    assert_zero(ret, "i:%d testUtilModulesCmp ret %d != 0 width %d row %d (%s)\n", i, ret, width, row, data[i].data);
                }

                if (do_bwipp && testUtilCanBwipp(symbol->symbology, data[i].option_1, data[i].option_2, -1, debug)) {
                    if (!data[i].bwipp_cmp) {
                        if (debug & ZINT_DEBUG_TEST_PRINT) printf("%d: %s not BWIPP compatible (%s)\n", i, testUtilBarcodeName(symbol->symbology), data[i].comment);
                    } else {
                        ret = testUtilBwipp(symbol, data[i].option_1, data[i].option_2, -1, data[i].data, length, NULL, bwipp_buf, sizeof(bwipp_buf));
                        assert_zero(ret, "i:%d %s testUtilBwipp ret %d != 0\n", i, testUtilBarcodeName(data[i].symbology), ret);

                        ret = testUtilBwippCmp(symbol, bwipp_msg, bwipp_buf, data[i].expected);
                        assert_zero(ret, "i:%d %s testUtilBwippCmp %d != 0 %s\n  actual: %s\nexpected: %s\n",
                                       i, testUtilBarcodeName(data[i].symbology), ret, bwipp_msg, bwipp_buf, data[i].expected);
                    }
                }
            }
        }

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

// #181 Nico Gunkel OSS-Fuzz
static void test_fuzz(int index, int debug) {

    testStart("");

    int ret;
    struct item {
        int symbology;
        unsigned char *data;
        int length;
        int option_1;
        int ret;
    };
    struct item data[] = {
        /* 0*/ { BARCODE_PDF417,
                    "\060\075\204\060\204\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075"
                    "\204\060\075\060\103\204\060\214\060\204\060\075\060\031\060\073\060\025\060\075\060\204\060\103\204\060\075\060\204\060\000\075\060\226\060\100\204\060\204\060"
                    "\204\060\075\204\060\120\214\060\204\060\074\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\000\060\204\060\120\214\060\204\060\074\204"
                    "\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\204\060\204\060\126\060\075\060\204\060\177\060\103\204\060\204\060\377\060\262\060"
                    "\000\075\060\226\060\100\060\073\060\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\041\060\110\060\160\060\075\060\075\204\060"
                    "\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\073\060\074\060\075"
                    "\060\204\060\103\204\060\075\060\204\060\204\060\075\204\060\120\214\060\204\060\074\204\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041"
                    "\060\000\060\000\060\200\060\204\060\214\060\204\060\075\060\141\060\000\060\204\060\120\214\060\204\060\075\060\204\060\075\204\060\204\060\204\075\060\075\060"
                    "\204\060\075\060\075\204\060\075\060\103\204\060\204\060\372\120\060\124\060\000\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060"
                    "\075\060\073\060\075\060\204\060\103\204\060\075\060\204\060\204\060\122\060\000\060\075\060\000\076\060\100\000\060\075\060\103\204\060\214\060\204\060\075\060"
                    "\200\060\204\075\060\075\060\204\060\000\075\060\226\060\100\204\060\204\060\075\204\060\204\060\204\075\060\075\060\204\060\134\060\075\204\060\040\060\103\204"
                    "\060\372\120\060\124\060\004\060\103\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\000\060\113\060\377\060\235\060\075\060"
                    "\204\060\103\204\060\204\060\075\060\204\060\204\060\075\204\060\075\214\060\204\060\074\204\060\204\060\075\204\060\075\060\103\211\060\214\060\204\060\075\060"
                    "\041\060\204\060\204\060\120\060\075\060\204\060\003\060\103\204\060\204\060\377\060\350\060\223\060\000\075\060\226\060\103\204\060\204\060\204\120\060\075\060"
                    "\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120"
                    "\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\031\060\155\060\000\075\060\226\060\100\204\060\204\060"
                    "\204\060\075\204\060\120\214\060\204\060\074\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\000\060\204\060\120\214\060\204\060\074\204"
                    "\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\204\060\204\060\126\060\075\060\204\060\177\060\103\204\060\204\060\377\060\262\060"
                    "\000\075\060\226\060\100\204\060\204\060\204\075\060\073\060\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\075\060\110\060\160"
                    "\060\075\060\075\204\060\075\060\103\204\060\204\060\372\120\060\124\060\000\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075"
                    "\060\073\060\074\060\075\060\204\060\103\204\060\075\060\204\060\204\060\075\204\060\075\214\060\204\060\074\204\060\204\060\075\204\060\075\060\103\214\060\214"
                    "\060\204\060\075\060\041\060\000\060\000\060\200\060\204\060\214\060\204\060\075\060\141\060\000\060\204\060\075\214\060\204\060\075\060\204\060\075\204\060\204"
                    "\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103"
                    "\204\060\214\060\204\060\075\060\073\060\075\060\204\060\103\204\060\075\060\204\060\204\060\122\060\000\060\075\060\000\076\060\100\000\060\004\060\103\204\060"
                    "\204\060\003\060\204\075\060\120\214\060\204\060\004\060\103\204\060\204\060\003\060\211\074\060\120\060\124\060\351\060\120\060\075\060\351\060\072\375\060\204\060",
                    1001, -1, ZINT_ERROR_TOO_LONG }, // Original OSS-Fuzz triggering data
        /* 1*/ { BARCODE_PDF417COMP,
                    "\060\075\204\060\204\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075"
                    "\204\060\075\060\103\204\060\214\060\204\060\075\060\031\060\073\060\025\060\075\060\204\060\103\204\060\075\060\204\060\000\075\060\226\060\100\204\060\204\060"
                    "\204\060\075\204\060\120\214\060\204\060\074\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\000\060\204\060\120\214\060\204\060\074\204"
                    "\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\204\060\204\060\126\060\075\060\204\060\177\060\103\204\060\204\060\377\060\262\060"
                    "\000\075\060\226\060\100\060\073\060\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\041\060\110\060\160\060\075\060\075\204\060"
                    "\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\073\060\074\060\075"
                    "\060\204\060\103\204\060\075\060\204\060\204\060\075\204\060\120\214\060\204\060\074\204\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041"
                    "\060\000\060\000\060\200\060\204\060\214\060\204\060\075\060\141\060\000\060\204\060\120\214\060\204\060\075\060\204\060\075\204\060\204\060\204\075\060\075\060"
                    "\204\060\075\060\075\204\060\075\060\103\204\060\204\060\372\120\060\124\060\000\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060"
                    "\075\060\073\060\075\060\204\060\103\204\060\075\060\204\060\204\060\122\060\000\060\075\060\000\076\060\100\000\060\075\060\103\204\060\214\060\204\060\075\060"
                    "\200\060\204\075\060\075\060\204\060\000\075\060\226\060\100\204\060\204\060\075\204\060\204\060\204\075\060\075\060\204\060\134\060\075\204\060\040\060\103\204"
                    "\060\372\120\060\124\060\004\060\103\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\000\060\113\060\377\060\235\060\075\060"
                    "\204\060\103\204\060\204\060\075\060\204\060\204\060\075\204\060\075\214\060\204\060\074\204\060\204\060\075\204\060\075\060\103\211\060\214\060\204\060\075\060"
                    "\041\060\204\060\204\060\120\060\075\060\204\060\003\060\103\204\060\204\060\377\060\350\060\223\060\000\075\060\226\060\103\204\060\204\060\204\120\060\075\060"
                    "\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120"
                    "\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\031\060\155\060\000\075\060\226\060\100\204\060\204\060"
                    "\204\060\075\204\060\120\214\060\204\060\074\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\000\060\204\060\120\214\060\204\060\074\204"
                    "\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\204\060\204\060\126\060\075\060\204\060\177\060\103\204\060\204\060\377\060\262\060"
                    "\000\075\060\226\060\100\204\060\204\060\204\075\060\073\060\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\075\060\110\060\160"
                    "\060\075\060\075\204\060\075\060\103\204\060\204\060\372\120\060\124\060\000\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075"
                    "\060\073\060\074\060\075\060\204\060\103\204\060\075\060\204\060\204\060\075\204\060\075\214\060\204\060\074\204\060\204\060\075\204\060\075\060\103\214\060\214"
                    "\060\204\060\075\060\041\060\000\060\000\060\200\060\204\060\214\060\204\060\075\060\141\060\000\060\204\060\075\214\060\204\060\075\060\204\060\075\204\060\204"
                    "\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103"
                    "\204\060\214\060\204\060\075\060\073\060\075\060\204\060\103\204\060\075\060\204\060\204\060\122\060\000\060\075\060\000\076\060\100\000\060\004\060\103\204\060"
                    "\204\060\003\060\204\075\060\120\214\060\204\060\004\060\103\204\060\204\060\003\060\211\074\060\120\060\124\060\351\060\120\060\075\060\351\060\072\375\060\204\060",
                    1001, -1, ZINT_ERROR_TOO_LONG },
        /* 2*/ { BARCODE_MICROPDF417,
                    "\060\075\204\060\204\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075"
                    "\204\060\075\060\103\204\060\214\060\204\060\075\060\031\060\073\060\025\060\075\060\204\060\103\204\060\075\060\204\060\000\075\060\226\060\100\204\060\204\060"
                    "\204\060\075\204\060\120\214\060\204\060\074\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\000\060\204\060\120\214\060\204\060\074\204"
                    "\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\204\060\204\060\126\060\075\060\204\060\177\060\103\204\060\204\060\377\060\262\060"
                    "\000\075\060\226\060\100\060\073\060\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\041\060\110\060\160\060\075\060\075\204\060"
                    "\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\073\060\074\060\075"
                    "\060\204\060\103\204\060\075\060\204\060\204\060\075\204\060\120\214\060\204\060\074\204\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041"
                    "\060\000\060\000\060\200\060\204\060\214\060\204\060\075\060\141\060\000\060\204\060\120\214\060\204\060\075\060\204\060\075\204\060\204\060\204\075\060\075\060"
                    "\204\060\075\060\075\204\060\075\060\103\204\060\204\060\372\120\060\124\060\000\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060"
                    "\075\060\073\060\075\060\204\060\103\204\060\075\060\204\060\204\060\122\060\000\060\075\060\000\076\060\100\000\060\075\060\103\204\060\214\060\204\060\075\060"
                    "\200\060\204\075\060\075\060\204\060\000\075\060\226\060\100\204\060\204\060\075\204\060\204\060\204\075\060\075\060\204\060\134\060\075\204\060\040\060\103\204"
                    "\060\372\120\060\124\060\004\060\103\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\000\060\113\060\377\060\235\060\075\060"
                    "\204\060\103\204\060\204\060\075\060\204\060\204\060\075\204\060\075\214\060\204\060\074\204\060\204\060\075\204\060\075\060\103\211\060\214\060\204\060\075\060"
                    "\041\060\204\060\204\060\120\060\075\060\204\060\003\060\103\204\060\204\060\377\060\350\060\223\060\000\075\060\226\060\103\204\060\204\060\204\120\060\075\060"
                    "\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120"
                    "\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\031\060\155\060\000\075\060\226\060\100\204\060\204\060"
                    "\204\060\075\204\060\120\214\060\204\060\074\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\000\060\204\060\120\214\060\204\060\074\204"
                    "\060\377\060\075\204\060\075\060\103\204\060\214\060\204\060\075\060\041\060\204\060\204\060\126\060\075\060\204\060\177\060\103\204\060\204\060\377\060\262\060"
                    "\000\075\060\226\060\100\204\060\204\060\204\075\060\073\060\204\060\000\075\060\226\060\100\060\103\204\060\204\060\075\204\060\204\060\204\075\060\110\060\160"
                    "\060\075\060\075\204\060\075\060\103\204\060\204\060\372\120\060\124\060\000\060\204\060\074\204\060\204\060\075\204\060\075\060\103\204\060\214\060\204\060\075"
                    "\060\073\060\074\060\075\060\204\060\103\204\060\075\060\204\060\204\060\075\204\060\075\214\060\204\060\074\204\060\204\060\075\204\060\075\060\103\214\060\214"
                    "\060\204\060\075\060\041\060\000\060\000\060\200\060\204\060\214\060\204\060\075\060\141\060\000\060\204\060\075\214\060\204\060\075\060\204\060\075\204\060\204"
                    "\060\204\041\060\075\060\204\060\075\060\075\204\060\075\060\103\204\060\204\060\003\120\060\075\060\004\060\204\060\074\204\060\204\060\075\204\060\075\060\103"
                    "\204\060\214\060\204\060\075\060\073\060\075\060\204\060\103\204\060\075\060\204\060\204\060\122\060\000\060\075\060\000\076\060\100\000\060\004\060\103\204\060"
                    "\204\060\003\060\204\075\060\120\214\060\204\060\004\060\103\204\060\204\060\003\060\211\074\060\120\060\124\060\351\060\120\060\075\060\351\060\072\375\060\204\060",
                    1001, -1, ZINT_ERROR_TOO_LONG },
        /* 3*/ { BARCODE_PDF417,
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "1234567890",
                    2710, 0, 0 }, // Max numerics with ECC 0
        /* 4*/ { BARCODE_PDF417,
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "12345678901",
                    2711, 0, ZINT_ERROR_TOO_LONG },
        /* 5*/ { BARCODE_PDF417,
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678",
                    2528, -1, 0 }, // Max numerics with ECC 5
        /* 6*/ { BARCODE_PDF417,
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEF",
                    1852, 0, 0 }, // Max text with ECC 0 Note: before default Text mode change the max was 1850 as given in ISO 15438:2015
        /* 7*/ { BARCODE_PDF417,
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFGHIJKLMNOPQRSTUVWXYZ"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZABCDEFG",
                    1853, 0, ZINT_ERROR_TOO_LONG },
        /* 8*/ { BARCODE_PDF417,
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240",
                    1110, 0, 0 }, // Max bytes with ECC 0 should be 1108 according to ISO 15438:2015 TODO: investigate
        /* 9*/ { BARCODE_PDF417,
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240"
                    "\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240\240",
                    1111, 0, ZINT_ERROR_TOO_LONG }, // See above
        /*10*/ { BARCODE_MICROPDF417,
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456",
                    366, -1, 0 }, // Max numerics
        /*11*/ { BARCODE_MICROPDF417,
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "1234567890123456789012345678901234567890123456789012345678901234567",
                    367, -1, ZINT_ERROR_TOO_LONG },
    };
    int data_size = sizeof(data) / sizeof(struct item);

    for (int i = 0; i < data_size; i++) {

        if (index != -1 && i != index) continue;

        struct zint_symbol *symbol = ZBarcode_Create();
        assert_nonnull(symbol, "Symbol not created\n");

        symbol->symbology = data[i].symbology;
        if (data[i].option_1 != -1) {
            symbol->option_1 = data[i].option_1;
        }
        symbol->debug |= debug;

        int length = data[i].length;
        if (length == -1) {
            length = strlen(data[i].data);
        }

        ret = ZBarcode_Encode(symbol, data[i].data, length);
        assert_equal(ret, data[i].ret, "i:%d ZBarcode_Encode ret %d != %d (%s)\n", i, ret, data[i].ret, symbol->errtxt);

        ZBarcode_Delete(symbol);
    }

    testFinish();
}

int main(int argc, char *argv[]) {

    testFunction funcs[] = { /* name, func, has_index, has_generate, has_debug */
        { "test_options", test_options, 1, 0, 1 },
        { "test_reader_init", test_reader_init, 1, 1, 1 },
        { "test_input", test_input, 1, 1, 1 },
        { "test_encode", test_encode, 1, 1, 1 },
        { "test_fuzz", test_fuzz, 1, 0, 1 },
    };

    testRun(argc, argv, funcs, ARRAY_SIZE(funcs));

    testReport();

    return 0;
}
