/**********************************************************************
 * Jhove - JSTOR/Harvard Object Validation Environment
 * Copyright 2003-4 by JSTOR and the President and Fellows of Harvard College
 **********************************************************************/

package edu.harvard.hul.ois.jhove;

import java.io.*;
import java.util.*;

/**
 * Public interface for Jhove format-specific plug-in modules.
 * 
 * All format modules must implement the Module interface; 
 * usually the best way to do this will be to subclass ModuleBase.
 * Modules must be declared in the configuration file and present
 * in the Classpath to be recognized by Jhove.  
 *
 * @see  ModuleBase
 */
public interface Module
{
    public static final int
	MAXIMUM_VERBOSITY = 1,
	MINIMUM_VERBOSITY = 2;

    /**
     * Per-instantiation initialization.
     *
     * @param init    Initialization parameter.  This is typically obtained
     *                from the configuration file.
     */
    public void init (String init)
	throws Exception;

    /**
     * Sets list of default parameters.
     * 
     * @param   params     A List whose elements are Strings.
     *                     May be empty.
     */
    public void setDefaultParams (List<String> params);


    /**
     *  Applies the default parameters.
     */
    public void applyDefaultParams ()
        throws Exception;
    
    /** Reset parameter settings.
     *  Returns to a default state without any parameters.
     */
    public void resetParams ()
        throws Exception;

    /**
     * Per-action initialization.
     * May be called multiple times.
     *
     * @param param   Initialization parameter.
     */
    public void param (String param)
	throws Exception;

    /**
     *  Pass the associated App object to this Module.
     *  The App makes various services available.
     */
    public void setApp (App app);

    /**
     *  Pass the JHOVE engine object to this Module.
     */
    public void setBase (JhoveBase je);

    /**
     *  Set the degree of verbosity desired from the module.  
     *
     *  @param  verbosity  The requested verbosity value.  
     */
    public void setVerbosity (int verbosity);

    /**
     *  Return the name of this Module.
     */
    public String getName ();
    
    /**
     *   Return the release identifier
     */    
    public String getRelease ();

    /**
     *  Return the last modification date of the Module, as a
     *  Java Date object
     */
    public Date getDate ();

    /**
     *  Return the vendor information
     */
    public Agent  getVendor ();

    /**
     *   Return the copyright information string
     */
    public String getRights ();

    /**
     *   Return the Module note
     */
    public String getNote ();

    /**
     *   Return the array of format names supported by this Module
     */
    public String [] getFormat ();

    /**
     *   Return the array of MIME type strings for formats supported
     *   by this Module
     */
    public String [] getMimeType ();

    /**
     *  Return details as to the specific format versions or 
     *  variants that are supported by this Module
     */
    public String getCoverage ();

    /**
     *   Return the List of Signatures recognized by this Module
     */
    public List<Signature> getSignature ();

    /**
     *  Returns a list of <code>Document</code> objects (one for each 
     *  specification document of the format).  The specification
     *  list is generated by the Module, and specifications cannot
     *  be added by callers.
     *
     *  @see Document
     */
    public List<Document> getSpecification ();

    /**
     *   Return the string describing well-formedness criteria
     */
    public String getWellFormedNote ();

    /**
     *   Returns the RepInfo note
     */
    public String getRepInfoNote ();

    /**
     *   Returns the string describing validity criteria
     */
    public String getValidityNote ();

    /**
     *  Returns the random access flag (true if the module operates
     *  on random access files, false if it operates on streams)
     */
    public boolean isRandomAccess ();
    
    /**
     *  Returns <code>true</code> if the module supports a given
     *  named feature, and <code>false</code> if the feature is
     *  unsupported or unknown.
     */
    public boolean hasFeature (String feature);

    /**
     *  Returns the full list of features. 
     */
    public List<String> getFeatures ();
    
    /**
     *  Returns the list of default parameters. 
     */
    public List<String> getDefaultParams ();



    /**
     *   Parse the content of a stream digital object and store the
     *   results in RepInfo.
     *   A given Module will normally implement only one of the two
     *   parse methods, leaving the other as a stub.
     *
     *   @param stream    An InputStream, positioned at its beginning,
     *                    which is generated from the object to be parsed.
     *                    If multiple calls to <code>parse</code> are made 
     *                    on the basis of a nonzero value being returned,
     *                    a new InputStream must be provided each time.
     * 
     *   @param info      A fresh (on the first call) RepInfo object 
     *                    which will be modified
     *                    to reflect the results of the parsing
     *                    If multiple calls to <code>parse</code> are made 
     *                    on the basis of a nonzero value being returned, 
     *                    the same RepInfo object should be passed with each
     *                    call.
     *
     *   @param parseIndex  Must be 0 in first call to <code>parse</code>.  If
     *                    <code>parse</code> returns a nonzero value, it must be
     *                    called again with <code>parseIndex</code> 
     *                    equal to that return value.
     */
    public int parse (InputStream stream, RepInfo info, int parseIndex)
    throws IOException;

    /**
     *   Parse the content of a random access digital object and store the
     *   results in RepInfo.
     *   A given Module will normally implement only one of the two
     *   parse methods, leaving the other as a stub.
     *
     *   @param file      A RandomAccessFile, positioned at its beginning,
     *                    which is generated from the object to be parsed
     *   @param info      A fresh RepInfo object which will be modified
     *                    to reflect the results of the parsing
     */
    public void parse (RandomAccessFile file, RepInfo info)
        throws IOException;

    /**
     *  Check if the digital object conforms to this Module's
     *  internal signature information.
     *
     *   @param file      A File object for the object being parsed
     *   @param stream    An InputStream, positioned at its beginning,
     *                    which is generated from the object to be parsed
     *   @param info      A fresh RepInfo object which will be modified
     *                    to reflect the results of the test
     */
    public void checkSignatures (File file,
                InputStream stream, 
                RepInfo info)
        throws IOException;

    /**
     *  Check if the digital object conforms to this Module's
     *  internal signature information.
     *
     *   @param file      A File object for the object being parsed
     *   @param raf       A RandomAccessFile, positioned at its beginning,
     *                    which is generated from the object to be parsed
     *   @param info      A fresh RepInfo object which will be modified
     *                    to reflect the results of the test
     */
    public void checkSignatures (File file, 
            RandomAccessFile raf, 
            RepInfo info)
        throws IOException;

	/**
	 *  Generates information about this Module.
	 *  The format of the output depends on the OutputHandler.
	 */
    public void show (OutputHandler handler);
}
