// ==++==
//
//   Copyright (c) Microsoft Corporation.  All rights reserved.
//
// ==--==
/*============================================================
**
** Class:  Single
**
**
** Purpose: A wrapper class for the primitive type float.
**
**
===========================================================*/
namespace System {

    using System.Globalization;
    using System;
///#if GENERICS_WORK
///    using System.Numerics;
///#endif
    using System.Runtime.InteropServices;
    using System.Runtime.CompilerServices;
    using System.Runtime.ConstrainedExecution;
    using System.Diagnostics.Contracts;

[Serializable]
[System.Runtime.InteropServices.StructLayout(LayoutKind.Sequential)]
[System.Runtime.InteropServices.ComVisible(true)]
#if GENERICS_WORK
    public struct Single : IComparable, IFormattable, IConvertible
        , IComparable<Single>, IEquatable<Single> 
///     , IArithmetic<Single>
#else
    public struct Single : IComparable, IFormattable, IConvertible
#endif
    {
        internal float m_value;

        //
        // Public constants
        //
        public const float MinValue = (float)-3.40282346638528859e+38;
        public const float Epsilon = (float)1.4e-45;
        public const float MaxValue = (float)3.40282346638528859e+38;
        public const float PositiveInfinity = (float)1.0 / (float)0.0;
        public const float NegativeInfinity = (float)-1.0 / (float)0.0;
        public const float NaN = (float)0.0 / (float)0.0;

        [Pure]
        [System.Security.SecuritySafeCritical]  // auto-generated
        public unsafe static bool IsInfinity(float f) {
            return (*(int*)(&f) & 0x7FFFFFFF) == 0x7F800000;
        }

        [Pure]
        [System.Security.SecuritySafeCritical]  // auto-generated
        public unsafe static bool IsPositiveInfinity(float f) {
            return *(int*)(&f) == 0x7F800000;
        }

        [Pure]
        [System.Security.SecuritySafeCritical]  // auto-generated
        public unsafe static bool IsNegativeInfinity(float f) {
            return *(int*)(&f) == unchecked((int)0xFF800000);
        }

        [Pure]
        [ReliabilityContract(Consistency.WillNotCorruptState, Cer.Success)]
        [System.Security.SecuritySafeCritical]
        public unsafe static bool IsNaN(float f) {
            return (*(int*)(&f) & 0x7FFFFFFF) > 0x7F800000;
        }

        // Compares this object to another object, returning an integer that
        // indicates the relationship.
        // Returns a value less than zero if this  object
        // null is considered to be less than any instance.
        // If object is not of type Single, this method throws an ArgumentException.
        //
        public int CompareTo(Object value) {
            if (value == null) {
                return 1;
            }
            if (value is Single) {
                float f = (float)value;
                if (m_value < f) return -1;
                if (m_value > f) return 1;
                if (m_value == f) return 0;

                // At least one of the values is NaN.
                if (IsNaN(m_value))
                    return (IsNaN(f) ? 0 : -1);
                else // f is NaN.
                    return 1;
            }
            throw new ArgumentException (Environment.GetResourceString("Arg_MustBeSingle"));
        }


        public int CompareTo(Single value) {
            if (m_value < value) return -1;
            if (m_value > value) return 1;
            if (m_value == value) return 0;

            // At least one of the values is NaN.
            if (IsNaN(m_value))
                return (IsNaN(value) ? 0 : -1);
            else // f is NaN.
                return 1;
        }

        public static bool operator ==(Single left, Single right) {
            return left == right;
        }

        public static bool operator !=(Single left, Single right) {
            return left != right;
        }

        public static bool operator <(Single left, Single right) {
            return left < right;
        }

        public static bool operator >(Single left, Single right) {
            return left > right;
        }

        public static bool operator <=(Single left, Single right) {
            return left <= right;
        }

        public static bool operator >=(Single left, Single right) {
            return left >= right;
        }

        public override bool Equals(Object obj) {
            if (!(obj is Single)) {
                return false;
            }
            float temp = ((Single)obj).m_value;
            if (temp == m_value) {
                return true;
            }

            return IsNaN(temp) && IsNaN(m_value);
        }

        public bool Equals(Single obj)
        {
            if (obj == m_value) {
                return true;
            }

            return IsNaN(obj) && IsNaN(m_value);
        }

        [System.Security.SecuritySafeCritical]  // auto-generated
        public unsafe override int GetHashCode() {
            float f = m_value;
            if (f == 0) {
                // Ensure that 0 and -0 have the same hash code
                return 0;
            }
            int v = *(int*)(&f);
            return v;
        }

        [System.Security.SecuritySafeCritical]  // auto-generated
        public override String ToString() {
            Contract.Ensures(Contract.Result<String>() != null);
            return Number.FormatSingle(m_value, null, NumberFormatInfo.CurrentInfo);
        }

        [System.Security.SecuritySafeCritical]  // auto-generated
        public String ToString(IFormatProvider provider) {
            Contract.Ensures(Contract.Result<String>() != null);
            return Number.FormatSingle(m_value, null, NumberFormatInfo.GetInstance(provider));
        }

        [System.Security.SecuritySafeCritical]  // auto-generated
        public String ToString(String format) {
            Contract.Ensures(Contract.Result<String>() != null);
            return Number.FormatSingle(m_value, format, NumberFormatInfo.CurrentInfo);
        }

        [System.Security.SecuritySafeCritical]  // auto-generated
        public String ToString(String format, IFormatProvider provider) {
            Contract.Ensures(Contract.Result<String>() != null);
            return Number.FormatSingle(m_value, format, NumberFormatInfo.GetInstance(provider));
        }

        // Parses a float from a String in the given style.  If
        // a NumberFormatInfo isn't specified, the current culture's
        // NumberFormatInfo is assumed.
        //
        // This method will not throw an OverflowException, but will return
        // PositiveInfinity or NegativeInfinity for a number that is too
        // large or too small.
        //
        public static float Parse(String s) {
            return Parse(s, NumberStyles.Float | NumberStyles.AllowThousands, NumberFormatInfo.CurrentInfo);
        }

        public static float Parse(String s, NumberStyles style) {
            NumberFormatInfo.ValidateParseStyleFloatingPoint(style);
            return Parse(s, style, NumberFormatInfo.CurrentInfo);
        }

        public static float Parse(String s, IFormatProvider provider) {
            return Parse(s, NumberStyles.Float | NumberStyles.AllowThousands, NumberFormatInfo.GetInstance(provider));
        }

        public static float Parse(String s, NumberStyles style, IFormatProvider provider) {
            NumberFormatInfo.ValidateParseStyleFloatingPoint(style);
            return Parse(s, style, NumberFormatInfo.GetInstance(provider));
        }
        
        private static float Parse(String s, NumberStyles style, NumberFormatInfo info) {
            return Number.ParseSingle(s, style, info);
        }

        public static Boolean TryParse(String s, out Single result) {
            return TryParse(s, NumberStyles.Float | NumberStyles.AllowThousands, NumberFormatInfo.CurrentInfo, out result);
        }

        public static Boolean TryParse(String s, NumberStyles style, IFormatProvider provider, out Single result) {
            NumberFormatInfo.ValidateParseStyleFloatingPoint(style);
            return TryParse(s, style, NumberFormatInfo.GetInstance(provider), out result);
        }
        
        private static Boolean TryParse(String s, NumberStyles style, NumberFormatInfo info, out Single result) {
            if (s == null) {
                result = 0;
                return false;
            }
            bool success = Number.TryParseSingle(s, style, info, out result);
            if (!success) {
                String sTrim = s.Trim();
                if (sTrim.Equals(info.PositiveInfinitySymbol)) {
                    result = PositiveInfinity;
                } else if (sTrim.Equals(info.NegativeInfinitySymbol)) {
                    result = NegativeInfinity;
                } else if (sTrim.Equals(info.NaNSymbol)) {
                    result = NaN;
                } else
                    return false; // We really failed
            }
            return true;

        }

        //
        // IConvertible implementation
        //

        public TypeCode GetTypeCode() {
            return TypeCode.Single;
        }


        /// <internalonly/>
        bool IConvertible.ToBoolean(IFormatProvider provider) {
            return Convert.ToBoolean(m_value);
        }

        /// <internalonly/>
        char IConvertible.ToChar(IFormatProvider provider) {
            throw new InvalidCastException(Environment.GetResourceString("InvalidCast_FromTo", "Single", "Char"));
        }

        /// <internalonly/>
        sbyte IConvertible.ToSByte(IFormatProvider provider) {
            return Convert.ToSByte(m_value);
        }

        /// <internalonly/>
        byte IConvertible.ToByte(IFormatProvider provider) {
            return Convert.ToByte(m_value);
        }

        /// <internalonly/>
        short IConvertible.ToInt16(IFormatProvider provider) {
            return Convert.ToInt16(m_value);
        }

        /// <internalonly/>
        ushort IConvertible.ToUInt16(IFormatProvider provider) {
            return Convert.ToUInt16(m_value);
        }

        /// <internalonly/>
        int IConvertible.ToInt32(IFormatProvider provider) {
            return Convert.ToInt32(m_value);
        }

        /// <internalonly/>
        uint IConvertible.ToUInt32(IFormatProvider provider) {
            return Convert.ToUInt32(m_value);
        }

        /// <internalonly/>
        long IConvertible.ToInt64(IFormatProvider provider) {
            return Convert.ToInt64(m_value);
        }

        /// <internalonly/>
        ulong IConvertible.ToUInt64(IFormatProvider provider) {
            return Convert.ToUInt64(m_value);
        }

        /// <internalonly/>
        float IConvertible.ToSingle(IFormatProvider provider) {
            return m_value;
        }

        /// <internalonly/>
        double IConvertible.ToDouble(IFormatProvider provider) {
            return Convert.ToDouble(m_value);
        }

        /// <internalonly/>
        Decimal IConvertible.ToDecimal(IFormatProvider provider) {
            return Convert.ToDecimal(m_value);
        }

        /// <internalonly/>
        DateTime IConvertible.ToDateTime(IFormatProvider provider) {
            throw new InvalidCastException(Environment.GetResourceString("InvalidCast_FromTo", "Single", "DateTime"));
        }

        /// <internalonly/>
        Object IConvertible.ToType(Type type, IFormatProvider provider) {
            return Convert.DefaultToType((IConvertible)this, type, provider);
        }
///#if GENERICS_WORK
///        //
///        // IArithmetic<Single> implementation
///        //
///
///        /// <internalonly/>
///        Single IArithmetic<Single>.AbsoluteValue(out bool overflowed) {
///            Single abs = (m_value < 0 ? -m_value : m_value);
///            overflowed = IsInfinity(abs) || IsNaN(abs);
///            return abs;
///        }
///
///        /// <internalonly/>
///        Single IArithmetic<Single>.Negate(out bool overflowed) {
///            Single neg= -m_value;
///            overflowed = IsInfinity(neg) || IsNaN(neg);
///            return neg;
///        }
///
///        /// <internalonly/>
///        Single IArithmetic<Single>.Sign(out bool overflowed) {
///            overflowed = IsNaN(m_value);
///            if (overflowed) {
///                return m_value;
///            }
///            return (m_value >= 0 ? (m_value == 0 ? 0 : 1) : -1);
///        }
///
///        /// <internalonly/>
///        Single IArithmetic<Single>.Add(Single addend, out bool overflowed) {
///            Single s = m_value + addend;
///            overflowed = IsInfinity(s) || IsNaN(s);
///            return s;
///        }
///
///        /// <internalonly/>
///        Single IArithmetic<Single>.Subtract(Single subtrahend, out bool overflowed) {
///            Single s = m_value - subtrahend;
///            overflowed = IsInfinity(s) || IsNaN(s);
///            return s;
///        }
///
///        /// <internalonly/>
///        Single IArithmetic<Single>.Multiply(Single multiplier, out bool overflowed) {
///            Single s = m_value * multiplier;
///            overflowed = IsInfinity(s) || IsNaN(s);
///            return s;
///        }
///
///
///        /// <internalonly/>
///        Single IArithmetic<Single>.Divide(Single divisor, out bool overflowed) {
///            Single s = m_value / divisor;
///            overflowed = IsInfinity(s) || IsNaN(s);
///            return s;
///        }
///
///        /// <internalonly/>
///        Single IArithmetic<Single>.DivideRemainder(Single divisor, out Single remainder, out bool overflowed) {
///            remainder = m_value % divisor;
///            Single s = m_value / divisor;
///            overflowed = IsInfinity(s) || IsInfinity(remainder) || IsNaN(s) || IsNaN(remainder);
///            return s;
///        }
///
///        /// <internalonly/>
///        Single IArithmetic<Single>.Remainder(Single divisor, out bool overflowed) {
///            Single s = m_value % divisor;
///            overflowed = IsInfinity(s) || IsNaN(s);
///            return s;
///        }
///
///        /// <internalonly/>
///        ArithmeticDescriptor<Single> IArithmetic<Single>.GetDescriptor() {
///            if (s_descriptor == null) {
///                s_descriptor = new SingleArithmeticDescriptor( ArithmeticCapabilities.One
///                                                             | ArithmeticCapabilities.Zero
///                                                             | ArithmeticCapabilities.MaxValue
///                                                             | ArithmeticCapabilities.MinValue
///                                                             | ArithmeticCapabilities.PositiveInfinity
///                                                             | ArithmeticCapabilities.NegativeInfinity);
///            }
///            return s_descriptor;
///        }
///
///        private static SingleArithmeticDescriptor s_descriptor;
/// 
///        class SingleArithmeticDescriptor : ArithmeticDescriptor<Single> {
///            public SingleArithmeticDescriptor(ArithmeticCapabilities capabilities) : base(capabilities) {}
///            public override Single One {
///                get {
///                    return (Single) 1;
///                }
///            }
///
///            public override Single Zero {
///                get {
///                    return (Single) 0;
///                }
///            }
///
///            public override Single MinValue {
///                get {
///                    return Single.MinValue;
///                }
///            }
///
///            public override Single MaxValue {
///                get {
///                    return Single.MaxValue;
///                }
///            }
///
///            public override Single PositiveInfinity {
///                get {
///                    return Single.PositiveInfinity;
///                }
///            }
///
///            public override Single NegativeInfinity {
///                get {
///                    return Single.NegativeInfinity;
///                }
///            }
///
///        }
///#endif // #if GENERICS_WORK
    }
}
