/*	Profile

PIRL CVS ID: Profile.java,v 1.35 2012/04/16 06:04:11 castalia Exp

Copyright (C) 2008-2012	 Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Conductor.Maestro;

import	PIRL.Configuration.Configuration;
import	PIRL.Configuration.Configuration_Exception;
import	PIRL.Messenger.Message;
import	PIRL.PVL.Parameter;
import	PIRL.PVL.Value;
import	PIRL.PVL.Parser;
import	PIRL.PVL.PVL_Exception;
import	PIRL.Conductor.String_Vector_Comparator;

import 	java.io.OutputStream;
import 	java.io.FileOutputStream;
import 	java.io.File;
import	java.io.IOException;
import	java.util.Collections;
import	java.util.List;
import	java.util.Vector;
import	java.util.Date;


/** A <i>Profile</i> contains Theater and Conductor definitions.
<p>
	A Profile completely specifies the Conductor instances running, or
	to be run, on a set of Theaters. A Profile may take the form of a
	{@link Configuration} that can be {@link #Read(String) read} from
	or {@link #Write(String) written} to and file.
<h4>
    Theater definitions
</h4><p>
<p>
	A Theater definition is a list of one to three entries:
<p>
<dl>
<dt>Theater location (String)
<dd>The location of a Theater in <i>hostname</i>[<b>:</b><i>port</i>]
	format. The <i>hostname</i> specifies the system where the
	Stage_Manager with which to communicate is expected to be running.
	This name may be a full or short hostname or an IP address. The
	<i>port</i> specifies the network port to be used to communicate with
	the Stage_Manager. The port number and the colon delimiter may be
	ommitted in user supplied Profile Configuration files, in which case
	the current default Theater port number will be used.
<p>
	The Theater location is a required entry in the definition. It must
	always be the first entry in the definition.
<p>
<dt>Conductor name (String)
<dd>The name of a Conductor definition in the Profile. At least one of
	the named Conductor definition instances are expected to be run at
	the Theater location.
<p>
	The Conductor name is not required in the definition. It may be the
	second or third entry in the definition. A Theater definition witout
	a Conductor name is a "discovery" definition: A connection is to be
	made to the Theater and whatever Conductors are running there are to
	be discovered. However, a Theater definition must not have a
	Conductor name if it occurs within a Conductor definition, in which
	case the Conductor name is implicitly the name of the Conductor
	definition.
<p>
<dt>Count (Integer)
<dd>The number of named Conductor instances to be run at the Theater
    location. 
<p>
	The Count is not required in the definition, but must not be present
	if no Conductor name is present, unless the Theater definition occurs
	within a Conductor definition. It may be the second or third entry in
	the definition. The default Count is 1.
</dl>
<p>
	A Profile may contain zero or more Theater definitions. In a Profile
	Configuration the Theater definitions are always contained in an
	Assignment Parameter named "Theaters" with an Array Value that is a
	list of one or more Theater locations, or a list of one or more 
	Theater definition Array Values. <b<N.B.</b>: a single multi-entry
	Theater definition must be contained in in an Array to distinguish it
	from a list of Theater locations; this results in doubled parentheses
	(or curly braces) around the single definition.
<p>
	Each Theater definition in the Profile is guaranteed to have a unique
	combination of Theater location and Conductor name values. In a user
	supplied Profile Configuration file, however, a Theater location and
	Conductor name combination may occur in more than one Theater
	definition. The Count of each duplicate location/name combination is
	added to the Count of the initial definition. That is, Theater
	definitions read from Configuration files are accumulated into a
	consolidated list of unique definitions.
<p>
	If a Conductor Theater definition with only a location entry is
	duplicated, the duplicates are ignored. However, a definition with
	the same location but that also contains a Conductor name causes the
	location-only definition to be ignored.
<p>
	A Theaters parameter in a user supplied Profile Configuration file
	may occur at the top level of the file or within Conductor
	definitions. A Theater definition within a Conductor definition must
	not contain a Conductor name since this is implicitly the name of the
	containing Conductor definition.
<p>
	Example Profile Configuration Theater definitions:
<p><code><pre>
	# Single Theater location.
	Theaters = host
	Theaters = host:1234
	Theaters = (host)

	# Multiple Theater locations.
	Theaters = (host_1, host_2, host_3, host_3:1234)

	# Single Theater definition.
	Theaters = ((host, name, 1))
	Theaters = ((host, name))

	# Multiple Theater definitions.
	Theaters = ((host_1, name), (host_2, name, 2), (host_3, name_3, 3))
	Theaters =
		(
		host_1:1234,
		(host_2:1234, name),
		(host_3:1234, name_3, 3)
		)
</pre></code><p>
<h4>
    Conductor definitions
</h4><p>
	A Conductor definition is an Aggregate (Group or Object) Parameter
	that specifies the command line arguments used to instantiate a
	Conductor at a Theater location. The name of the Aggregate is the
	Conductor name used in a Theater definition. All Parameter Aggregates
	in a Profile Configuration are taken to be Conductor definitions. The
	parameters contained in a Conductor definition are all Assignments
	with String values (except the Theaters parameter) and are all
	optional:
<p>
<dl>
<dt>Pipeline
<dd>The name of the Pipeline to be managed by the Conductor.
<p>
	If this parameter is not present it will be provided with the value
	being the Conductor definition name.
<dt>Configuration
<dd>The source of the Conductor Configuration file to be used by the
	Conductor. This may be a filesystem pathname or a URL. <b>N.B.</b>: A
	pathname is for the filesystem on the Theater host. A relative
	pathname is relative to the working directory of the Stage_Manager
	used to run the Conductor. In general, absolute pathnames may be
	preferrable to avoid the uncertainty of relative pathnames.
<p>
	If this parameter is not present the instantiated Conductor will use
	its default Configuration source, which is expected to be the
	"Conductor.conf" pathname.
<p>
<dt>Server
<dd>The name of the Configuration Server that specifies the name of the
	Configuration parameters group containing database server access
	information to be used by the Conductor.
<p>
	If this parameter is not present the instantiated Conductor will use
	the first entry in the Configuration Server parameter list.
<p>
<dt>Catalog
<dd>The name of the Database Catalog on the selected database server
	that contains the pipeline Sources and Procedures definitions.
<p>
	If this parameter is not present the instantiated Conductor will use
	the Catalog prefix of the Pipeline (catalog.pipeline) if present;
	otherwise it will use the value of the "Catalog" name from the
	Configuration Server parameter group, or the default "Catalog"
	parameter in the Configuration.
<p>
<dt>Theaters
<dd>A Theater definition list. <b.B.B.</b>: The definitions in this list
	must not contain a Conductor name since this is implicitly the
	name of the Conductor definition.
</dl>
<p>
	A Profile may contain zero or more Conductor definitions. It is not
	necessary that all Conductor definitions be referenced by a Conductor
	name in a Theater definition. Any Conductor name in a Theater definition
	for which a Conductor definition has not been provided will be satisfied
	by a generic Conductor definition as if an emtpy group with the
	Conductor name had been provided.
<p>
	All Conductor definitions will be unique; Conductor definitions with
	the same definition parameters must have different names. <b>N.B.</b>:
	A Profile Configuration file should contain only one Conductor definition
	Group (or Object) having a given name because when the Configuration is
	ingested Groups with the same name will have their parameter contents
	coalesced into a single Group; duplicate parameters names that are
	encountered will result in an exception being thrown.
<p>
	Example Profile Configuration Conductor definitions:
<p><code><pre>
	# A generic Conductor definition.
	Group = name
	End_Group

	# Simple Conductor definitions.
	Group = name
		Pipeline = pipeline
	End_Group

	Group = name_1
		Pipeline = pipeline
		Configuration = config_pathname_1
	End_Group

	Group = name_2
		Pipeline = pipeline_2
		Configuration = config_pathname_2
	End_Group

	# Complete Conductor definition.
	Group = name
		Pipeline = pipeline
		Configuration = /an/absolute/pathname
		Server = server_name
		Catalog = catalog_name
	End_Group

	# Conductor definition with Theater definitions.
	Group = pipeline
		Theaters = ((host_1), (host2, 2), (host_3, 3))
		Configuration = http://config.host/pathname/to/file
	End_Group
</pre></code><p>
<p>
	A Profile Configuration may be managed as separate files by taking
	advantage of the "@Include" capability of a Configuration. For
	example, sets of "standard" Conductor definitions may be maintained
	in files separate from various files that contain Theater
	definitions and "@Include" one or more of the standard Conductor
	definition sets:
<p><code><pre>
	# Theater definitions.
	Theaters =
		(
		# Using standard Conductor definitions.
		(host_1, standard_1),
		(host_2, standard_2),

		# Using special processing Conductor definitions.
		(host_1, special_1),
		(host_2, special_2)

		# Custom Conductor definitions.
		(host_1, custom_1),

		# Local Conductor definition.
		(host_2, local, 3)
		)

	# Standard Conductor definitions.
	&#064;Include = /Conductor/definitions/repository/Standard.defs

	# Special processing Conductor definitions.
	&#064;Include = /Conductor/definitions/repository/Special_Processing.defs

	# Custom Conductor definitions.
	&#064;Include = Custom_Conductor.defs

	# Local Conductor definition.
	Group = local
		Configuration = /an/absolute/pathname
	End_Group
</pre></code><p>
	In this example the included files may have many Conductor
	definitions, most of which are not used in any particular Theaters
	Theater definitions list.
<p>
	"&#064;Include" may also be used to provide common parts of various
	Coductor definitions by putting it inside the appropriate
	Conductor definitions.
<p>
	Profiles are used by the {@link Kapellmeister} application.
<p>
	@author		Michael Wendell and Bradford Castalia, UA/HiROC
	@version	1.35 
*/
public class Profile
{
/**	Class identification name with source code version and date.
*/
public static final String 
	ID = "PIRL.Conductor.Maestro.Profile (1.35 2012/04/16 06:04:11)";


/**	The name of a Profile {@link #Configuration() Configuration}.
*/
public static final String
	PROFILE_NAME					= "Profile";

/**	The Profile Configuration parameter name for a list of Theater
	definitions.
*/
public static final String
	THEATERS_PARAMETER_NAME			= "Theaters";

/**	The size of a complete Theater definition Vector.
*/
public static final int
	THEATER_DEFINITION_SIZE			= 3;

/**	The index of the Theater location String in a theater definition Vector.
*/
public static final int
	THEATER_LOCATION_INDEX			= 0;

/**	The index of the Conductor name String or Conductor_Definition
	in a theater definition Vector.
*/
public static final int
	CONDUCTOR_NAME_INDEX			= 1;

/**	The index of the Conductor count Integer in a theater definition Vector.
*/
public static final int
	CONDUCTOR_COUNT_INDEX			= 2;

/**	Theater Conductor name entry when no Conductor_Definition is provided.
*/
public static final String 
	NO_CONDUCTOR_DEFINITION			= null;

/**	The name of the Conductor definition parameter that specifies the
	pipeline to be processed.
*/
public static final String
	PIPELINE_PARAMETER_NAME
		= Conductor_Definition.PIPELINE_PARAMETER_NAME;

private Vector<Vector<Object>>
	Theater_Definitions				= new Vector<Vector<Object>> ();

private Vector<Conductor_Definition>
	Conductor_Definitions			= new Vector<Conductor_Definition> ();

/**	Undigested parameters after a Configuration source has been read
	and ingested.
*/
private Configuration 
	Source_Parameters				= null;

private static final String 
	NL								= System.getProperty ("line.separator");


// Debug control.
private static final int
	DEBUG_OFF						= 0,
	DEBUG_CONSTRUCTOR				= 1 << 1,
	DEBUG_READ						= 1 << 2,
	DEBUG_ADD_CONDUCTOR_DEFINITION	= 1 << 3,
	DEBUG_ALL						= -1,

	DEBUG							= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/** Construct a Profile from a source Configuration file.
<p>
	Any parameters remaining after the source has been {@link #Read(String)
	read} will be held in the {@link #Unprocessed_Source_Parameters()
	unprocessed source Configuration}.
<p>	
	@param	source	A Configuration source String. This may be a file
		pathname or a URL.
	@throws	Configuration_Exception	If there was a problem with the source
		Configuration syntax.
*/
public Profile 
	(
	String	source
	)
	throws 
		Configuration_Exception,
		PVL_Exception
{
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">>> Profile: " + source);
Read (source);
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("<<< Profile");
}

/** Construct a Profile from Theater and Conductor definitions.
<p>
	@param	theater_definitions	A List of Theater definitions.
	@param	conductor_definitions	A List of Conductor_Definitions.
	@throws	IllegalArgumentException	If there was a problem creating the 
		Profile.
	@see	#Add(List, List)
*/	  
public Profile 
	(
	List<List<Object>>			theater_definitions, 
	List<Conductor_Definition>	conductor_definitions
	)
	throws IllegalArgumentException
{
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">>> Profile:" + NL
		+"    theater_definitions - " + theater_definitions + NL
		+"    conductor_definitions -" + NL
		+ conductor_definitions);
Add (theater_definitions, conductor_definitions);
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("<<< Profile");
}

/**	Construct an empty Profile.
*/	  
public Profile () 
{
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">-< Profile");
}

/*==============================================================================
	Configuration
*/
/**	Read Theater and Conductor definitions from a Configuration source file.
<p>
	Theater and Conductor definitions found in the Configuration are
	added to the definitions currently in this Profile.
<p>
	@param	source	A Configuration source String. This may be a file
		pathname or a URL.
	@return	This Profile.
	@throws	Configuration_Exception	If the source can not be found or parsed
		as valid PVL or the contents do not conform to valid Profile
		syntax.
	@see	#Read(Configuration)
*/
public Profile Read
	(
	String	source
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_READ) != 0)
	System.out.println
		(">>> Profile.Read: " + source);
Source_Parameters = new Configuration ();
try
	{
	Source_Parameters.Defaults (false);
	Source_Parameters.Configure (source);
	}
catch (Configuration_Exception exception)
	{
	throw new Configuration_Exception (ID + NL
		+ "Could not read a Profile from the source -" + NL
		+ source + NL
		+ exception.getMessage ());
	}

return Read (Source_Parameters);
}

/**	Read Theater and Conductor definitions from a Configuration.
<p>
	Theater and Conductor definitions found in the Configuration are
	added to the definitions currently in this Profile. Only {@link
	#THEATERS_PARAMETER_NAME} Assignment parameters are used as Theater
	definitions. All Aggregate parameters are taken to be Conductor
	definitions.
<p>
	<b>N.B.</b>: The Theater and Conductor definitions found in the
	Configuration are removed from the Configuration leaving only the
	{@link #Unprocessed_Source_Parameters() unprocessed source parameters}
	in the original Configuration.
<p>
	@param	configuration	A Configuration. If null nothing is done.
	@return	This Profile.
	@throws	Configuration_Exception	If the Configuration Theater or
		Conductor definition contents do not conform to valid Profile
		syntax.
*/
public Profile Read
	(
	Configuration	configuration
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_READ) != 0)
	System.out.println
		(">>> Profile.Read: Configuration -" + NL
		+ configuration);
if ((Source_Parameters = configuration) == null)
	{
	if ((DEBUG & DEBUG_READ) != 0)
		System.out.println
			("<<< Profile.Read: Configuration");
	return this;
	}

//	Conductor Groups.
if ((DEBUG & DEBUG_READ) != 0)
	System.out.println
		("    Profile.Read: Finding Conductor groups");
Parameter 
	parameter;
while ((parameter = Source_Parameters.Find (Parameter.AGGREGATE)) != null)
	{
	if ((DEBUG & DEBUG_READ) != 0)
		System.out.println
			("    Profile.Read: Conductor definition -" + NL
			+ parameter.Description ());
	Parameter
		invalid = parameter.Find (Parameter.AGGREGATE);
	if (invalid != null)
		throw new Configuration_Exception (ID + NL
			+ "Invalid Profile syntax in the Configuration source -" + NL
			+ configuration.Source () + NL
			+ "The \"" + parameter.Name () + "\" Conductor definition "
				+ "contains the \"" + invalid.Name () 
				+ "\" Aggregate parameter.");
 	else
		{
		try {Add_Conductor_Definition (parameter);}
		catch (PVL_Exception exception)
			{
			throw new Configuration_Exception
				(exception.getMessage () + NL
				+ "Problem reading the Configuration source -" + NL
				+ configuration.Source ());
			}
		}
	Source_Parameters.Remove (parameter);
	}

//	Theater definitions.
if ((DEBUG & DEBUG_READ) != 0)
	System.out.println
		("    Profile.Read: Finding Theaters");
if ((parameter = Source_Parameters.Find
			(Parameter.Absolute_Pathname (THEATERS_PARAMETER_NAME),
				Parameter.ASSIGNMENT))
		!= null)
	{
	//	Theaters Value.
	if ((DEBUG & DEBUG_READ) != 0)
		System.out.println
			("    Profile.Read: Theater definition -" + NL
			+ parameter.Description ());
	try {Add_Theater_Definitions (parameter.Value ());}
	catch (PVL_Exception exception)
		{
		throw new Configuration_Exception
			(exception.getMessage () + NL
			+ "Problem reading the Configuration source -" + NL
			+ configuration.Source ());
		}
	Source_Parameters.Remove (parameter);
	}

if (Source_Parameters.List_Size () == 0)
	Source_Parameters = null;
if ((DEBUG & DEBUG_READ) != 0)
	System.out.println
		("    Profile.Read: Remaining parameters - "
			+ ((Source_Parameters == null) ? "null" : (NL
			+ Source_Parameters.Description ())) + NL
		+ NL
		+"                  Profile -" + NL
		+ toString () + NL
		+"<<< Profile.Read");
return this;
}

/**	Get unprocessed Configuration parameters.
<p>
	@return	A Configuration containing any parameters remaining after a
		Configuration source was {@link #Read(Configuration) read}. This
		will be null if there were no unprocessed parameters.
*/
public Configuration Unprocessed_Source_Parameters ()
{return Source_Parameters;}

/**	Write the Profile to a {@link #Configuration() Configuration} file.
<p>
	If the Profile is empty nothing is written.
<p>
	@param	pathname	The file pathname String. If null or empty
		nothing is done.
	@return	This Profile.
	@throws	Configuration_Exception	If there was a problem creating a
		Configuration for this Profile.
	@throws	IOException	If there was a problem writing the file.
	@see	#Write(OutputStream)
*/
public Profile Write
	(
	String	pathname
	)
	throws 
		Configuration_Exception,
		IOException
{
if (pathname == null ||
	pathname.length () == 0)
	return this;

OutputStream
	output_stream = null;
try {output_stream = new FileOutputStream (pathname);}
catch (IOException exception)
	{
	throw new IOException (ID + NL
		+ "Could not write the Profile to -" + NL
		+ new File (pathname).getAbsolutePath () + NL
		+ exception.getMessage ());
	}

try {Write (output_stream);}
catch (Configuration_Exception exception)
	{
	throw new Configuration_Exception
		(exception.getMessage () + NL
		+ "File pathname: " + new File (pathname).getAbsolutePath ());
	}
catch (IOException exception)
	{
	throw new IOException
		(exception.getMessage () + NL
		+ "File pathname: " + new File (pathname).getAbsolutePath ());
	}
return this;
}

/**	Write the Profile to an OutputStream.
<p>
	A {@link #Configuration() configuration} is created from the current
	Profile contents and this is {@link Configuration#Write(OutputStream)
	written} to the OutputStream.
<p>
	If the Profile is empty nothing is written.
<p>
	@param	output_stream	An OutputStream where the Profile Configuration
		is to be written. If null nothing is done.
	@return	This Profile.
	@throws	Configuration_Exception	If there was a problem creating a
		Configuration for this Profile.
	@throws	IOException	If there was a problem writing the file.
*/
public Profile Write
	(
	OutputStream	output_stream
	)
	throws 
		Configuration_Exception,
		IOException
{
if (output_stream == null)
	return this;

Configuration
	configuration = null;
try {configuration = Configuration ();}
catch (Configuration_Exception exception)
	{
	throw new Configuration_Exception (ID + NL
		+ exception.getMessage () + NL
		+ "Problem creating the Profile Configuration to write.");
	}

if (configuration.List_Size () != 0)
	{
	try
		{
    	configuration.Name (Parser.CONTAINER_NAME);
		((Parameter)configuration.List ().get (0))
    		.Comments
				("Profile of Theater and Conductor definitions." + NL
				+ NL
				+ Stage_Manager.DATE_FORMAT.format (new Date ()) + NL
				+ "Generated by:" + NL
				+ ID);
		configuration.Write (output_stream);
		}
	catch (PVL_Exception exception)
		{
		throw new Configuration_Exception (ID + NL
			+ exception.getMessage () + NL
			+ "Problem writing the Profile.");
		}
	catch (IOException exception)
		{
		throw new IOException (ID + NL
			+ exception.getMessage () + NL
			+ "Problem writing the Profile.");
		}
	}
return this;
}

/**	Get the Profile as a Configuration.
<p>
	@return	A Configuration containing the Profile definition parameters.
		The {@link Configuration#Name() Name} of the Configuration will
		be the {@link #PROFILE_NAME}.
	@throws	Configuration_Exception	If there was a problem creating the
		Configuration
*/
public Configuration Configuration ()
	throws 
		Configuration_Exception
{
int
	array_type = Value.Default_Array_Type (Value.SEQUENCE);
Configuration
	configuration = new Configuration ((Parameter)null);

if (Theater_Definitions.size () > 0)
	{
	//	Get a Theater definitions list sorted on the location.
	Vector<Vector<Object>>
		definitions = Theater_Definitions ();
	Collections.sort (definitions,
		new String_Vector_Comparator (THEATER_LOCATION_INDEX));
	try {configuration.Set (THEATERS_PARAMETER_NAME, definitions);}
	catch (Configuration_Exception exception)
		{
		Value.Default_Array_Type (array_type);
		throw new Configuration_Exception (ID + NL
			+ exception.getMessage () + NL
			+ "Could not create the Profile Configuration.");
		}
	}

if (Conductor_Definitions.size () > 0)
	{
	int
		index = -1,
		size = Conductor_Definitions.size ();
	while (++index < size)
		{
		Conductor_Definition
			conductor_definition = Conductor_Definitions.get (index);
		if (conductor_definition != null)
			{
			try {configuration.Add
					(new Conductor_Definition (conductor_definition));}
			catch (PVL_Exception exception)
				{
				Value.Default_Array_Type (array_type);
				throw new Configuration_Exception (ID + NL
					+ "Invalid Conductor definition for a Configuration -" + NL
					+ conductor_definition + NL
					+ exception.getMessage ());
				}
			}
		}
	}

Value.Default_Array_Type (array_type);
configuration.Name (PROFILE_NAME);
return configuration;
}

/**	Get a PVL description of the Profile.
<p>
	The current contents of the Profile are converted to a {@link
	#Configuration() Configuration} from which a PVL {@link
	Configuration#Description() description} is obtained.
<p>
	@return	A PVL String description of the Profile. This will be null
		if the {@link #Configuration() Configuration} description
		could not be generated.
*/
public String toString ()
{
try {return Configuration ().Description ();}
catch (Exception exception) {}
return null;
}

/*==============================================================================
	Manipulators
*/
/**	Adds Theater and Conductor definitions to the Profile.
<p>
	@param	theater_definitions	A List of Theater definitions.
	@param	conductor_definitions	A List of Conductor_Definitions.
	@throws	IllegalArgumentException	If the arguments are not one
		List of valid Theater definitions and one List of valid
		Conductor_Definitions.
*/
public Profile Add
	(
	List<List<Object>>			theater_definitions,
	List<Conductor_Definition>	conductor_definitions
	)
	throws IllegalArgumentException 
{
//	Add the conductor definitions first for use by the theater definitions.
Add_Conductor_Definitions (conductor_definitions);
return Add_Theater_Definitions (theater_definitions);
}

/**	Add a Conductor_Definition associated with a Theater Location.
<p>
	The Theater definition will have as a Conductor name the
	{@link Conductor_Table_Model#Conductor_Name(Message) Conductor name}
	obtained from the Conductor definition, and it will have a Conductor
	count of 1.
<p>
	@param	theater_location	A Theater location String. If null or
		empty nothing is done.
	@param	conductor_definition	A Conductor definition Message. If null
		nothing is done.
	@return	This Profile.
	@throws	IllegalArgumentException	If the Conductor definition
		conflicts with an existing definition.
	@see	#Add_Theater_Definition(String)
	@see	#Add_Conductor_Definition(Message)
*/
public Profile Add
	(
	String	theater_location,
	Message	conductor_definition
	)
	throws IllegalArgumentException
{
if (theater_location == null ||
	theater_location.length () == 0 ||
	conductor_definition == null)
	return this;

Add_Conductor_Definition (conductor_definition);

Vector<Object>
	theater_definition = new Vector<Object> (THEATER_DEFINITION_SIZE);
theater_definition.add (theater_location);
theater_definition.add
	(Conductor_Table_Model.Conductor_Name (conductor_definition));
theater_definition.add (new Integer (1));
Add_a_Theater_Definition (theater_definition);

return this;
}

/*------------------------------------------------------------------------------
	Theaters
*/
/**	Add Theater definitions from a Value.
<p>
	@param	theater_definitions	A Value containing Theater definitions.
*/
private void Add_Theater_Definitions
	(
	Value	theater_definitions
	) 
	throws PVL_Exception
{
if (theater_definitions == null)
	return;

if (! theater_definitions.Is_Array ())
	//	A single Value is an Array of one Value.
	theater_definitions.Type (Value.ARRAY);

Vector<Object>
	definition;
Value 
	theater_definition,
	entry_one,
	entry_two;
String
	name;
int 
	count = 0,
	index = -1,
	size = theater_definitions.Array_Size ();
while (++index < size)
	{
	theater_definition = theater_definitions.Get (index);
	if (! theater_definition.Is_Array ())
		//	A single Value is an Array of one Value.
		theater_definition.Type (Value.ARRAY);

	switch (theater_definition.Array_Size ())
		{
		case 0:	//	No entries.
		continue;

		case 1:	//	Location only.
		name = NO_CONDUCTOR_DEFINITION;
		count = 0;
		break;

		case 2:	//	Location and Name.
		entry_one = Check_Theater_Definition_Entry
			(theater_definition, CONDUCTOR_NAME_INDEX);
		if (entry_one.Is_Numeric ())
			throw new PVL_Exception (ID, PVL_Exception.ILLEGAL_SYNTAX,
 				"Invalid Theater definition to add -" + NL
				+ theater_definition.Description () + NL
 				+ "The second of two entries must not be numeric.");

		name = entry_one.String_Data ();
		count = 1;
		break;

		case 3:	//	Location, Name, Count or Location, Count, Name.
		entry_one = Check_Theater_Definition_Entry
			(theater_definition, CONDUCTOR_NAME_INDEX);
		entry_two = Check_Theater_Definition_Entry
			(theater_definition, CONDUCTOR_COUNT_INDEX);

		if (entry_one.Is_Numeric () &&
			entry_two.Is_Numeric ())
			throw new PVL_Exception (ID, PVL_Exception.ILLEGAL_SYNTAX,
 				"Invalid Theater definition to add -" + NL
				+ theater_definition.Description () + NL
 				+ "Only one of the second and third entries may be numeric");
		else
		if (! entry_one.Is_Numeric () &&
			! entry_two.Is_Numeric ())
			throw new PVL_Exception (ID, PVL_Exception.ILLEGAL_SYNTAX,
 				"Invalid Theater definition to add -" + NL
				+ theater_definition.Description () + NL
				+ "Either the second or third entry must be numeric.");
		else
		if (entry_one.Is_Numeric () &&
			! entry_two.Is_Numeric ())
			{
			name = entry_two.String_Data ();
			count = (int)entry_one.long_Data ();
			}
		else
			{
			name = entry_one.String_Data ();
			count = (int)entry_two.long_Data ();
			}

		break;

		default:
			throw new PVL_Exception (ID, PVL_Exception.ILLEGAL_SYNTAX,
 				"Invalid Theater definition to add -" + NL
				+ theater_definition.Description () + NL
 				+ "There are more than the maximum of "
					+ THEATER_DEFINITION_SIZE + " entries.");
		}

	definition = new Vector<Object> (THEATER_DEFINITION_SIZE);
	entry_one = Check_Theater_Definition_Entry
		(theater_definition, THEATER_LOCATION_INDEX);
	definition.add (entry_one.String_Data ());

	if (name == null ||
		name.length () == 0 ||
		count <= 0)
		{
		name = NO_CONDUCTOR_DEFINITION;
		count = 0;
		}
	definition.add (name);
	definition.add (count);

	Add_a_Theater_Definition (definition);
	}
}


private static Value Check_Theater_Definition_Entry
	(
	Value	theater_definition,
	int		entry
	)
	throws PVL_Exception
{
Value
	value = theater_definition.Get (entry);
if (value.Is_Array ())
	throw new PVL_Exception (ID, PVL_Exception.ILLEGAL_SYNTAX,
 		"Invalid Theater definition to add -" + NL
		+ theater_definition.Description () + NL
 		+ "Entry " + entry + " must not be an Array.");
return value;
}

/**	Add a list of Theater definitions to the Profile.
<p>
	@param	theater_definitions	A List containing Theater definitions. If
		null or empty nothing is done.
	@return	This Profile.
	@throws	IllegalArgumentException	If any Theater definition in the
		theater_definitions is not {@link
		#Validate_Theater_Definition(List) valid}.
	@see	#Add_Theater_Definition(List)
*/
public Profile Add_Theater_Definitions
	(
	List<List<Object>>	theater_definitions
	) 
	throws IllegalArgumentException
{
if (theater_definitions == null ||
	theater_definitions.size () == 0)
	return this;;

try {Validate_Theater_Definitions (theater_definitions);}
catch (IllegalArgumentException exception)
	{
	throw new IllegalArgumentException
		(exception.getMessage () + NL
		+ "The Theater definitions list was not added.");
	}
int
	index = -1,
	size = theater_definitions.size ();
while (++index < size)
	Add_a_Theater_Definition
		(new Vector<Object> (theater_definitions.get (index)));
return this;
}

/**	Add a Theater definition to the Profile.
<p>
	The theater definition is first {@link
	#Validate_Theater_Definition(List) validated}. It is then copied and
	added to the Profile.
<p>
	@param	theater_definition	A List containing a Theater definition.
		If null or empty nothing is done.
	@return This Profile.
	@throws	IllegalArgumentException	If the theater_definition is not
		valid.
*/
public Profile Add_Theater_Definition
	(
	List<Object>	 theater_definition
	)
	throws IllegalArgumentException
{
if (theater_definition == null ||
	theater_definition.size () == 0)
	return this;

try {Validate_Theater_Definition (theater_definition);}
catch (IllegalArgumentException exception)
	{
	throw new IllegalArgumentException
		(exception.getMessage () + NL
		+ "The Theater definition was not added.");
	}
Add_a_Theater_Definition (new Vector<Object> (theater_definition));
return this;
}

/**	Add a general Theater definition.
<p>
	A general Theater definition has a Theater location but no Conductor
	name (this will be null) or count (this will be zero).
<p>
	Adding a general Theater definition for a location that already has
	a Theater definition in the Profile will have no effect; a general
	definition does not override a specific definition.
<p>
	@param	theater_location	A Theater location String. If this is null
		or empty nothing will be done.
	@return	This Profile.
*/
public Profile Add_Theater_Definition
	(
	String	theater_location
	)
{
if (theater_location != null &&
	theater_location.length () != 0)
	{
	Vector<Object>
		definition = new Vector<Object> (THEATER_DEFINITION_SIZE);
	definition.add (theater_location);
	definition.add (null);
	definition.add (new Integer (0));

	Add_a_Theater_Definition (definition);
	}
return this;
}


private void Add_a_Theater_Definition
	(
	//	The definition must be valid.
	Vector<Object>	definition
	)
{
String
	conductor_name = (String)definition.get (CONDUCTOR_NAME_INDEX);

//	Set the location to the standard format.
String
	theater_location =
		Theater.Full_Location ((String)definition.get (THEATER_LOCATION_INDEX));
definition.set (THEATER_LOCATION_INDEX, theater_location);

//	Check for a existing Theater definition that matches.
Vector<Object>
	theater_definition = Theater_Definition (theater_location, conductor_name);
if (theater_definition != null &&
	conductor_name != null)
	{
	//	Matching non-null Conductor names; increment the Conductor count.
	Integer
		count = (Integer)theater_definition.get (CONDUCTOR_COUNT_INDEX);
	theater_definition.set (CONDUCTOR_COUNT_INDEX,
		count += (Integer)definition.get (CONDUCTOR_COUNT_INDEX));
	}
else
if (theater_definition == null)
	{
	//	Non-matching.
	if (! Contains_Theater_Definition (theater_location))
		//	New definition.
		Theater_Definitions.add (definition);
	else
	if (conductor_name != null)
		{
		if ((theater_definition = Theater_Definition (theater_location, null))
				!= null)
			{
			//	Make the general definition specific.
			theater_definition.set (CONDUCTOR_NAME_INDEX, conductor_name);
			theater_definition.set (CONDUCTOR_COUNT_INDEX,
				definition.get (CONDUCTOR_COUNT_INDEX));
			}
		else
			//	New definition.
			Theater_Definitions.add (definition);
		}
	//	A specific definition will not be made general.
	}
//	Matching null Conductor names ignored.
}

/**	Validate a Theater definition.
<p>
	A valid Theater definition Vector contains three entries:
<dl>
<dt>Theater Location
<dd>The Theater location String is in the form:
<p>
	<i>host</i>[<b>:</b><i>port</i>]
<p>
	The <i>host</i> may be a hostname, short or fully qualified, or IP
	address. The optional <i>port</i> is the system port number used to
	connect to the Stage_Manager.

<dt>Conductor Name
<dd>The Conductor name String is the name of a {@link
	#Conductor_Definition(String) Conductor_Definition} in the Profile.
	This may be null to indicate that the Conductors on the Theater are
	to be used as-is ("discovered").

<dt>Conductor Count
<dd>The Conductor count Integer specifies the number of Conductors as
	defined by the Conductor_Defintiion having the Conductor name that
	are to be run on the Theater.
</dl>
	The Theater location must always be the first entry in the definition.
	The Conductor name and count need not be in that order. <b>N.B.</b>:
	If the Conductor name and count are not in that order their order in
	the Vector will be swapped.
<p>
	@param	theater_definition	A List containing a Theater definition.
	@throws	IllegalArgumentException	If the theater_definition is invalid.
*/
public static void Validate_Theater_Definition
	(
	List<Object>	theater_definition
	)
	throws IllegalArgumentException
{
if (theater_definition == null ||
	theater_definition.size () != THEATER_DEFINITION_SIZE)
	throw new IllegalArgumentException (ID + NL
		+ "Invalid Theater definition -" + NL
		+ theater_definition + NL
		+ "A Theater definition must contain three entries.");

if (! (theater_definition.get (THEATER_LOCATION_INDEX) instanceof String))
	throw new IllegalArgumentException (ID + NL
		+ "Invalid Theater definition -" + NL
		+ theater_definition + NL
		+ "The first entry of a Theater definition must be a String.");

Object
	entry_one = theater_definition.get (CONDUCTOR_NAME_INDEX),
	entry_two = theater_definition.get (CONDUCTOR_COUNT_INDEX);

if ((entry_one instanceof String ||
	 entry_one == null ) &&
	entry_two instanceof Integer)
	return;

if (entry_one instanceof Integer &&
	(entry_two instanceof String ||
	 entry_two == null))
	{
	//	Swap name and count entries.
	theater_definition.set (CONDUCTOR_NAME_INDEX, entry_two);
	theater_definition.set (CONDUCTOR_COUNT_INDEX, entry_one);
	return;
	}
throw new IllegalArgumentException (ID + NL
	+ "Invalid Theater definition -" + NL
	+ theater_definition + NL
	+ "The second and third entries of a Theater definition" + NL
	+ "must be a String and Integer (in either order).");
}

/**	Validate a Theater definitions list.
<p>
	@param	theater_definitions	The List that is to be validated. If null
		or empty nothing is validated.
	@throws	IllegalArgumentException	If a theater_definitions entry
		is invalid.
	@see	#Validate_Theater_Definition(List)
*/
public static void Validate_Theater_Definitions
	(
	List<List<Object>>	theater_definitions
	)
	throws IllegalArgumentException
{
if (theater_definitions == null ||
	theater_definitions.isEmpty ())
	return;

int
	index = -1,
	size = theater_definitions.size ();
try {while (++index < size)
		Validate_Theater_Definition (theater_definitions.get (index));}
catch (IllegalArgumentException exception)
	{
	throw new IllegalArgumentException
		(exception.getMessage () + NL
		+ "at entry " + index + " of a theaters definitions list.");
	}
}

/**	Get a Theater definition at its Profile index.
<p>
	@param	index	A Profile Theater definition index.
	@return	A reference to the Theater definition Vector in this
		Profile; the definition is not copied. This will be null if the
		index is invalid.
*/
public Vector<Object> Theater_Definition
	(
	int		index
	)
{
if (index < 0 ||
	index >= Theater_Definitions.size ())
	return null;
return Theater_Definitions.get (index);
}

/**	Get the Profile index of the next Theater definition for a Theater
	location starting after a given index.
<p>
	Starting at the Profile index immediately following the specified
	index, the index of the next Theater definition for the specified
	location is returned.
<p>
	@param	theater_location	A Theater location String.
	@param	index	The Profile index after which the search for the
		next Theater definition for the the location will begin.
		If less than zero the search begins with the first definition.
	@return	The Profile index of the next Theater definition that
		contains a matching location, or -1 if no match is found or the
		theater location is null or the empty String.
	@see	#Theater_Definition(int)
*/
public int Next_Theater_Index
	(
	String	theater_location,
	int		index
	)
{
if (theater_location != null &&
	theater_location.length () != 0)
	{
	theater_location = Theater.Full_Location (theater_location);
	if (index < -1)
		index = -1;
	int
		size = Theater_Definitions.size ();
	while (++index < size)
		if (theater_location.equals (Theater_Definitions.get (index)
				.get (THEATER_LOCATION_INDEX)))
			return index;
	}
return -1;
}

/**	Get the list of Theater definitions.
<p>
	@return	A Vector containing a copy of the Theater definitions list
		in this Profile.
*/
public Vector<Vector<Object>> Theater_Definitions ()
{
int
	index = -1,
	size = Theater_Definitions.size ();
Vector<Vector<Object>>
	theater_definitions = new Vector<Vector<Object>> (size);
Vector<Object>
	this_definition,
	that_definition;
/*
	Note that all the theater definition objects are final
	so the object references can simply be copied from
	this current definition to the that new definition
	without having to construct clones of the objects.
*/
while (++index < size)
	{
	this_definition = Theater_Definitions.get (index);
	that_definition = new Vector<Object> (THEATER_DEFINITION_SIZE);
	that_definition.add (this_definition.get (THEATER_LOCATION_INDEX));
	that_definition.add (this_definition.get (CONDUCTOR_NAME_INDEX));
	that_definition.add (this_definition.get (CONDUCTOR_COUNT_INDEX));
	theater_definitions.add (that_definition);
	}
return theater_definitions;
}

/**	Get the Theater definition for a Theater location and Conductor name.
<p>
	@param	theater_location	A Theater location String. If null, null
		is returned.
	@param	conductor_name		A Conductor name. If null, null is
		returned.
	@return	A Theater definition Vector. <b>N.B.</b>: This is a reference
		to the definition in the Profile; copy first if it is to be
		modified. This will be null if the matching Theater definition
		could not be found.
*/
public Vector<Object> Theater_Definition
	(
	String	theater_location,
	String	conductor_name
	)
{
int
	index = Theater_Definition_Index
		(Theater_Definitions, theater_location, conductor_name);
if (index >= 0)
	return Theater_Definitions.get (index);
return null;
}

/**	Get the canonical Theater definitions list.
<p>
	<b>N.B.</b>: The {@link #CONDUCTOR_NAME_INDEX} entry in the
	definition Vector will be the full Conductor_Definition, not just its
	name. This will be null if the definition for the Theater does not
	specify any Conductors.
<p>
	@return	A Vector containing of all the Theater definitions in the
		Profile. <b>N.B.</b>: The Conductor_Definitions in the Theater
		definition Vectors will be references to the objects in
		the Profile, not copies.
*/
public Vector<Vector<Object>> Canonical_Theater_Definitions ()
{
Vector<Vector<Object>>
	theater_definitions = new Vector<Vector<Object>> ();
int
	index = -1,
	size = Theater_Definitions.size ();
while (++index < size)
	{
	Vector<Object>
		theater_definition = Theater_Definitions.get (index);
	Vector<Object>
		definition = new Vector<Object> ();

	//	Theater location.
	definition.add (theater_definition.get (THEATER_LOCATION_INDEX));

	//	Conductor definition.
	String
		conductor_name = (String)theater_definition.get (CONDUCTOR_NAME_INDEX);
	if (conductor_name == null ||
		conductor_name.equals (NO_CONDUCTOR_DEFINITION))
		definition.add (null);
	else
		{
		Conductor_Definition
			conductor_definition = Conductor_Definition (conductor_name);
		if (conductor_definition != null)
			definition.add (conductor_definition);
		else
			definition.add (Generic_Conductor_Definition (conductor_name));
		}

	//	Conductor count.
	definition.add (theater_definition.get (CONDUCTOR_COUNT_INDEX));

	theater_definitions.add (definition);
	}
return theater_definitions;
}

/**	Test if any Theater definition for a location is present in this Profile.
<p>
	@param	theater_location	A Theater location String. If null or empty
		false is returned.
	@return	true if at least one Theater defifinition for the theater_location
		is present in this Profile; false otherwise.
*/
public boolean Contains_Theater_Definition
	(
	String	theater_location
	)
{
if (theater_location != null &&
	theater_location.length () != 0)
	{
	int
		index = Theater_Definitions.size ();
	while (--index >= 0)
		if (theater_location.equals (Theater_Definitions.get (index)
				.get (THEATER_LOCATION_INDEX)))
			return true;
	}
return false;
}

/**	Get the index of a Theater definition for a Theater location and
	Conductor name in a Vector of Theater definitions.
<p>
	@param	theater_definitions	A Vector of Theater definitions. If null
		or empty -1 is returned. <b>N.B.</b>: This must be a
		{@link #Validate_Theater_Definitions(List) valid Theater definitions
		list}.
	@param	theater_location	A Theater location String. If null or
		empty -1 is returned.
	@param	conductor_name		A Conductor name String. May be null.
	@return	The index of the Theater definition in the theater_definitions.
		This will be -1 if a Theater definition with the theater_location
		and conductor_name could not be found.
*/
public static int Theater_Definition_Index
	(
	List	theater_definitions,
	String	theater_location,
	String	conductor_name
	)
{
if (theater_definitions != null &&
	theater_definitions.size () != 0 &&
	theater_location != null &&
	theater_location.length () != 0)
	{
	theater_location = Theater.Full_Location (theater_location);
	int
		index = theater_definitions.size ();
	while (--index >= 0)
		{
		List 
			theater_definition = (List)theater_definitions.get (index);
		if (theater_location.equals
				((String)theater_definition.get (THEATER_LOCATION_INDEX)))
			{
			String 
				name = (String)theater_definition.get (CONDUCTOR_NAME_INDEX);
			if ((conductor_name != null &&
			  	 conductor_name.equals (name)) ||
				(conductor_name == null &&
			  	 name == null))
				return index;
			}
		}
	}
return -1;
}

/**	Get the number of Theater definitions in the Profile.
<p>
	@return	The number of Theater definitions in the Profile.
*/
public int Total_Theater_Definitions ()
{return Theater_Definitions.size ();}

/**	Remove a Theater definition from the Profile for a Theater location
	and Conductor name.
<p>
	@param	theater_location	A Theater location String. If null or
		empty null will be returned.
	@param	conductor_name	A Conductor name String. May be null.
	@return	The Theater definition Vector that was removed from the Profile.
		This will be null if a matching definition could not be found.
*/
public Vector<Object> Remove_Theater_Definition 
	(
	String	theater_location,
	String	conductor_name
	) 
{return Remove_Theater_Definition
	(Theater_Definition_Index
		(Theater_Definitions, theater_location, conductor_name));}

/**	Removes a Theater definition at a {@link #Theater_Definitions()} index.
<p>
	@param	index	The index of a Theater definition in this Profile.
	@return	The Theater definition Vector that was removed from the Profile.
		This will be null if index is not valid.
*/
public Vector<Object> Remove_Theater_Definition 
	(
	int		index
	) 
{
if (index >= 0 &&
	index < Theater_Definitions.size ())
	return Theater_Definitions.remove (index);
return null;
}

/**	Clear the Theater definitions list.
<p>
	@return	This Profile.
*/
public Profile Clear_Theater_Definitions () 
{Theater_Definitions.clear (); return this;}

/*------------------------------------------------------------------------------
	Conductors
*/
/**	Add Conductor_Definitions to the Profile.
<p>
	@param	conductor_definitions	A List of Conductor_Definitions.
	@return	This Profile.
	@throws	IllegalArgumentException	If a definition conflicts with an
		existing definition of the same name.
	@see	#Add_Conductor_Definition(Message)
*/
public Profile Add_Conductor_Definitions
	(
	List<Conductor_Definition>	conductor_definitions
	)
	throws IllegalArgumentException 
{
if (conductor_definitions == null ||
	conductor_definitions.size () == 0)
	return this;

int
	index = -1,
	size = conductor_definitions.size ();
while (++index < size)
	Add_Conductor_Definition (conductor_definitions.get (index));

return this;
}

/**	Adds a Conductor definition from a Message.
<p>
	@param	message	A Message containing a Conductor definition to be
		added to this Profile.
	@return	This Profile.
	@throws	IllegalArgumentException	If the Message does not contain a
		valid Conductor definition or contains a conflicting definition
		with a definition of the same name already in the Profile.
*/
public Profile Add_Conductor_Definition
	(
	Message		message
	)
	throws IllegalArgumentException 
{
if ((DEBUG & DEBUG_ADD_CONDUCTOR_DEFINITION) != 0)
	System.out.println
		(">>> Add_Conductor_Definition (Message):" + NL
		+ message);
if (message == null ||
	message.List_Size () == 0)
	{
	if ((DEBUG & DEBUG_ADD_CONDUCTOR_DEFINITION) != 0)
		System.out.println
			("<<< Add_Conductor_Definition (Message)");
	return this;
	}

try
	{
	Conductor_Definition
		definition = new Conductor_Definition (message),
		current_definition = Conductor_Definition (definition.Name ());
	if (current_definition == null)
		{
		Conductor_Definitions.add (definition);
		if ((DEBUG & DEBUG_ADD_CONDUCTOR_DEFINITION) != 0)
			System.out.println
				("    New definition added");
		}
	else if (! current_definition.Matches (definition))
		throw new IllegalArgumentException (ID + NL
			+ "The Conductor definition to be added -" + NL
			+ definition + NL
			+ "- has the same name but different definition values" + NL
			+ "as a Conductor definition already in the Profile -" + NL
			+ current_definition);
	}
catch (PVL_Exception exception)
	{
	throw new IllegalArgumentException (ID + NL
		+ "The Conductor definition to be added could not be copied -" + NL
		+ exception.getMessage () + NL
		+ message);
	}
if ((DEBUG & DEBUG_ADD_CONDUCTOR_DEFINITION) != 0)
	System.out.println
		("    Conductor_Definitions -" + NL
		+ Conductor_Definitions + NL
		+"<<< Add_Conductor_Definition (Message)");
return this;
}

/**	Add a Conductor definition assembled from a Parameter.
<p>
	@param	conductor_definition	An Aggregate Parameter containing
		the {@link Conductor_Definition} parameters.
*/
private void Add_Conductor_Definition
	(
	//	Guaranteed to be an Aggregate
	Parameter conductor_definition
	) 
	throws PVL_Exception
{
if ((DEBUG & DEBUG_ADD_CONDUCTOR_DEFINITION) != 0)
	System.out.println
		(">>> Add_Conductor_Definition (Parameter):" + NL
		+ conductor_definition.Description ());
//	Scratch copy
Message 
	definition = new Message (conductor_definition);
if ((DEBUG & DEBUG_ADD_CONDUCTOR_DEFINITION) != 0)
	System.out.println
		("    Scratch Message -" + NL
		+ definition);

//	Name:
if (conductor_definition.Name ().length () == 0)
	throw new PVL_Exception (ID, PVL_Exception.ILLEGAL_SYNTAX,
		"The Conductor definition to be added has an empty name -" + NL
		+ conductor_definition.Description ());
definition.Name (conductor_definition.Name ());

//	Pipeline:
if (definition.Get (PIPELINE_PARAMETER_NAME) == null)
	{
	definition.Set (PIPELINE_PARAMETER_NAME, definition.Name (), 0);
	if ((DEBUG & DEBUG_ADD_CONDUCTOR_DEFINITION) != 0)
		System.out.println
			("    " + PIPELINE_PARAMETER_NAME + " set to " + definition.Name ()
				+ " from Name");
	}

//	Theaters.
//	The Parameter is from a Configuration so there can be only one Theaters.
Value
	theaters = definition.Value_of (THEATERS_PARAMETER_NAME);
if (theaters != null)
	{
	if (! theaters.Is_Array ())
		//	Single Value Array
		theaters.Type (Value.ARRAY);

	if ((DEBUG & DEBUG_ADD_CONDUCTOR_DEFINITION) != 0)
		System.out.println
			("    " + THEATERS_PARAMETER_NAME + " = "
				+ theaters.Description ());
	int
		index = theaters.Array_Size ();
	while (--index >= 0)
		{
		Value 
			theater = (Value)theaters.Get (index);
		if (! theater.Is_Array ())
			//	Single Value Array
			theater.Type (Value.ARRAY);

		if (theater.Array_Size () == 0)
			{
			theaters.Remove (theater);
			continue;
			}
				
		theater.Add (new Value (definition.Name ()));
		}

	//	Remove the Theaters from the Conductor_Definition.
	definition.Remove (THEATERS_PARAMETER_NAME);
	
	Add_Theater_Definitions (theaters);
	}

Add_Conductor_Definition (definition);
if ((DEBUG & DEBUG_ADD_CONDUCTOR_DEFINITION) != 0)
	System.out.println
		("<<< Add_Conductor_Definition (Parameter)");
}

/** Test if this Profile contains a Conductor_Definition matching a Message.
<p>
	Each Conductor_Definition contained in this Profile is compared
	against the contents of the Message for a {@link
	Conductor_Definition#Matches(Message) matching definition}. <b>N.B.</b>:
	The name of the Message does not need to match the name of a
	Conductor_Definition, only the contents must logically match.
<p>
	@param	message	A Message. If null false is returned.
	@return	true if this Profile contains a Conductor_Definition that
		logically matches the Message contents; false if no match is
		found.
*/
public boolean Contains_Conductor_Definition
	(
	Message		message
	)
{
if (message != null)
	{
	int
		index = Conductor_Definitions.size ();
	while (--index >= 0)
		if (Conductor_Definitions.get (index).Matches (message))
			return true;
	}
return false;
}

/**	Get the list of Conductor_Definitions.
<p>
	@return	A Vector of Conductor_Definitions. This is a copy of the
		Conductor definitions list contained in this Profile.
*/
public Vector<Conductor_Definition> Conductor_Definitions ()
{
int
	index = -1,
	size = Conductor_Definitions.size ();
Vector<Conductor_Definition>
	conductor_definitions = new Vector<Conductor_Definition> (size);
while (++index < size)
	{
	try {conductor_definitions.add (new Conductor_Definition
			(Conductor_Definitions.get (index)));}
	catch (PVL_Exception exception)
		{/* Shouldn't happen. Valid Conductor_Definition.*/}
	}

Vector<String>
	already_added = new Vector<String> ();
String
	name;
index = -1;
size = Theater_Definitions.size ();
while (++index < size)
	{
	name = (String)Theater_Definitions.get (index).get (CONDUCTOR_NAME_INDEX);
	if (name != null &&
		! name.equals (NO_CONDUCTOR_DEFINITION) &&
		Conductor_Definition_Index (Conductor_Definitions, name) < 0 &&
		! already_added.contains (name))
		{
		already_added.add (name);
		conductor_definitions.add (Generic_Conductor_Definition (name));
		}
	}
return conductor_definitions;
}

/**	Get a Conductor_Definition index from a list of Conductor_Definition
	objects using its name.
<p>
	@param	conductor_definitions	A List of Conductor_Definition objects.
	@param	conductor_name	The Conductor_Definition name.
	@return	The index of the definition in the Conductor_Definitions Vector.
*/
public static int Conductor_Definition_Index
	(
	List<Conductor_Definition>		conductor_definitions,
	String							conductor_name
	)
{
if (conductor_definitions != null &&
	conductor_definitions.size () != 0 &&
	conductor_name != null &&
	conductor_name.length () != 0)
	{
	int
		index = conductor_definitions.size ();
	while (--index >= 0)
		if (conductor_name.equals
				(conductor_definitions.get (index).Name ()))
			return index;
	}
return -1;
}

/**	Get the Conductor_Definition for a Conductor name.
<p>
	@param	conductor_name	A Conductor_Definition name String. If null,
		null is returned.
	@return	The Conductor_Definition in the Profile (not a copy) having
		the conductor_name. This will be null if a Conductor_Definition
		with the conductor_name could not be found.
*/
public Conductor_Definition Conductor_Definition
	(
	String	conductor_name
	)
{
int
	index = Conductor_Definition_Index (Conductor_Definitions, conductor_name);
if (index >= 0)
	return Conductor_Definitions.get (index);
return null;
}

/**	Get the number of Conductor_Definitions in the Profile.
<p>
	@return	The number of Conductor_Definitions in the Profile.
*/
public int Total_Conductor_Definitions ()
{return Conductor_Definitions.size ();}

/**	Remove a Conductor_Definition by name.
<p>
	@param	conductor_name	A Conductor name.
	@return	The Conductor_Definition that was removed. This will be null
		if a Conductor_Definition with the specified name could not be
		found.
*/
public Conductor_Definition Remove_Conductor_Definition 
	(
	String	conductor_name
	) 
{
int
	index = Conductor_Definition_Index (Conductor_Definitions, conductor_name);
if (index >= 0)
	return Conductor_Definitions.remove (index);
return null;
}

/**	Removes a definition from the Conductor_Definitions Vector based on the index.
<p>
*/
public void Remove_Conductor_Definition 
	(
	int		index
	) 
{
if (Conductor_Definitions.size () >= index)
	Conductor_Definitions.remove (index);
}

/**	Clears the Conductor_Definitions Vector.
<p>
	@return	This Profile.
*/
public Profile Clear_Conductor_Definitions () 
{Conductor_Definitions.clear (); return this;}

/**  Returns a generic Conductor_Definition for a given name.
<p>
	@param	conductor_name	The name for the generic Conductor_Definition.
	@return	A Conductor_Definition.
*/
private Conductor_Definition Generic_Conductor_Definition
	(
	String	conductor_name
	)
{
if (conductor_name != null &&
	conductor_name.length () != 0)
	return new Conductor_Definition (conductor_name);
return null;
}


}
