# Copyright 2011 by Wibowo Arindrarto (w.arindrarto@gmail.com)
# This code is part of the Biopython distribution and governed by its
# license.  Please see the LICENSE file that should have been included
# as part of this package.

import unittest

from os.path import join, basename

from Bio import SeqIO
from Bio._py3k import _as_bytes

test_data = {
'data_empty': {
              'path': ['Abi', 'empty.ab1'],
              'seq': 'NNNNN',
              'qual': [0, 0, 0, 0, 0],
              'sample': '226041_C-ME-19_pCAGseqF',
              'sample_well': 'C10',
              'machine_model': '3730',
              'run_start': '2009-12-12 09:56:53',
              'run_finish': '2009-12-12 11:44:49',
              'abif_raw_keys': set(['AEPt1', 'AEPt2', 'APFN2', 'APXV1',
                                    'APrN1', 'APrV1', 'APrX1', 'ARTN1',
                                    'ASPF1', 'ASPt1', 'ASPt2', 'B1Pt1',
                                    'B1Pt2', 'BCTS1', 'BufT1', 'CTID1',
                                    'CTNM1', 'CTOw1', 'CTTL1', 'CpEP1',
                                    'DATA1', 'DATA2', 'DATA3', 'DATA4',
                                    'DATA5', 'DATA6', 'DATA7', 'DATA8',
                                    'DCHT1', 'DSam1', 'DySN1', 'Dye#1',
                                    'DyeN1', 'DyeN2', 'DyeN3', 'DyeN4',
                                    'DyeW1', 'DyeW2', 'DyeW3', 'DyeW4',
                                    'EPVt1', 'EVNT1', 'EVNT2', 'EVNT3',
                                    'EVNT4', 'FTab1', 'FVoc1', 'FWO_1',
                                    'Feat1', 'GTyp1', 'HCFG1', 'HCFG2',
                                    'HCFG3', 'HCFG4', 'InSc1', 'InVt1',
                                    'LANE1', 'LIMS1', 'LNTD1', 'LsrP1',
                                    'MCHN1', 'MODF1', 'MODL1', 'NAVG1',
                                    'NLNE1', 'NOIS1', 'PBAS1', 'PBAS2',
                                    'PCON1', 'PCON2', 'PDMF1', 'PDMF2',
                                    'PLOC1', 'PLOC2', 'PSZE1', 'PTYP1',
                                    'RGOw1', 'RMXV1', 'RMdN1', 'RMdX1',
                                    'RPrN1', 'RPrV1', 'RUND1', 'RUND2',
                                    'RUND3', 'RUND4', 'RUNT1', 'RUNT2',
                                    'RUNT3', 'RUNT4', 'Rate1', 'RevC1',
                                    'RunN1', 'S/N%1', 'SCAN1', 'SMED1',
                                    'SMLt1', 'SMPL1', 'SPAC1', 'SPAC2',
                                    'SPAC3', 'SVER1', 'SVER2', 'SVER3',
                                    'Scal1', 'Scan1', 'TUBE1', 'Tmpr1',
                                    'User1', 'phAR1', 'phCH1', 'phDY1',
                                    'phQL1', 'phTR1', 'phTR2', 'AUDT1',
                                    'PXLB1', 'RMdV1', 'DATA9', 'RGNm1',
                                    'DATA11', 'DATA10', 'DATA12']),
              },
'data_3730': {
             'path': ['Abi', '3730.ab1'],
             'seq':
'GGGCGAGCKYYAYATTTTGGCAAGAATTGAGCTCTATGGCCACAACCATGGTGAGCAAGGGCGAGGAGGATAACATGGCCATCATCAAGGAGTTCATGCGCTTCAAGGTGCACATGGAGGGCTCCGTGAACGGCCACGAGTTCGAGATCGAGGGCGAGGGCGAGGGCCGCCCCTACGAGGGCACCCAGACCGCCAAGCTGAAGGTGACCAAGGGTGGCCCCCTGCCCTTCGCCTGGGACATCCTGTCCCCTCAGTTCATGTACGGCTCCAAGGCCTACGTGAAGCACCCCGCCGACATCCCCGACTACTTGAAGCTGTCCTTCCCCGAGGGCTTCAAGTGGGAGCGCGTGATGAACTTCGAGGACGGCGGCGTGGTGACCGTGACCCAGGACTCCTCCCTGCAGGACGGCGAGTTCATCTACAAGGTGAAGCTGCGCGGCACCAACTTCCCCTCCGACGGCCCCGTAATGCAGAAGAAGACCATGGGCTGGGAGGCCTCCTCCGAGCGGATGTACCCCGAGGACGGCGCCCTGAAGGGCGAGATCAAGCAGAGGCTGAAGCTGAAGGACGGCGGCCACTACGACGCTGAGGTCAAGACCACCTACAAGGCCAAGAAGCCCGTGCAGCTGCCCGGCGCCTACAACGTCAACATCAAGTTGGACATCACCTCCCACAACGAGGACTACACCATCGTGGAACAGTACGAACGCGCCGAGGGCCGCCACTCCACCGGCGGCATGGACGAGCTGTACAAGGGCGGCAGCGGCATGGTGAGCAAGGGCGAGGAGCTGTTCACCGGGGTGGTGCCCATCCTGGTCGAGCTGGACGGCGACGTAAACGGCCACAAGTTCAGCGTGTCCGGCGAGGGCGAGGGCGATGCCACCTACGGCAAGCTGACCCTGAAGTTCATCTGCACCACCGGCAAGCTGCCCGTGCCCTGGCCCACCCTCGTGACCACCCTGACCTACGGCGTGCAGTGCTTCAGCCGCTACCCCGACCACATGAAGCAGCACGACTTCTTCAAGTCCGCCATGCCCGAAGGCTACGTCCAGGAGCGCACCATCTTCTTCAAGGACGACGGCAACTACAARACCCGCGCCGAGGTGAARTTCGAGGGCGACACCCTGGTGAACCGCATCGAGCTGAAAGGGGCAYCGCACCTTTC',
             'qual':
[20, 3, 4, 4, 4, 6, 4, 4, 0, 0, 0, 6, 0, 10, 20, 26, 22, 17, 21, 31, 29, 32, 28, 18, 23, 17, 19, 35, 36, 50, 39, 50, 50, 50, 50, 50, 25, 35, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 35, 39, 33, 20, 35, 31, 50, 50, 50, 50, 50, 50, 50, 50, 50, 31, 50, 35, 31, 23, 28, 31, 21, 43, 39, 35, 24, 30, 26, 35, 31, 50, 50, 50, 50, 50, 50, 50, 50, 50, 39, 31, 24, 39, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 31, 31, 43, 43, 50, 50, 50, 50, 50, 31, 31, 31, 31, 50, 50, 50, 50, 50, 50, 50, 50, 31, 31, 35, 50, 50, 50, 50, 31, 36, 55, 55, 55, 55, 36, 55, 55, 55, 55, 55, 36, 55, 55, 55, 55, 55, 36, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 40, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 36, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 40, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 43, 43, 50, 43, 43, 50, 43, 43, 50, 43, 43, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 43, 43, 50, 43, 43, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 28, 28, 35, 28, 28, 35, 28, 28, 35, 28, 28, 35, 28, 28, 35, 28, 21, 28, 35, 28, 28, 35, 35, 35, 35, 35, 37, 38, 21, 28, 35, 28, 28, 35, 35, 35, 35, 35, 35, 35, 36, 36, 21, 39, 35, 35, 35, 39, 35, 37, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 28, 28, 35, 35, 28, 28, 35, 35, 35, 36, 36, 22, 39, 35, 35, 35, 35, 35, 35, 37, 38, 28, 35, 21, 36, 36, 37, 35, 35, 20, 39, 39, 35, 35, 35, 35, 37, 38, 28, 35, 37, 34, 35, 24, 24, 27, 25, 20, 24, 37, 35, 27, 21, 20, 21, 27, 17, 20, 24, 32, 26, 20, 12, 20, 10, 20, 24, 25, 23, 20, 32, 24, 24, 23, 20, 24, 23, 18, 34, 34, 34, 22, 26, 24, 24, 18, 22, 22, 23, 25, 20, 12, 20, 24, 23, 24, 23, 22, 20, 20, 0, 20, 24, 23, 20, 8, 10, 4, 20, 20, 3, 7, 19, 20, 4, 4, 7, 7, 0, 7, 11, 18, 8, 3, 23, 23, 20, 11, 4, 20, 18, 12, 20, 20, 20, 4, 20, 4, 2, 3, 21, 21, 21, 21, 10, 15, 14, 15, 19, 2, 4, 3, 6, 11, 3, 4, 6, 21, 16, 20, 11, 1, 4, 12, 0, 15, 8, 1, 3, 3, 12, 1, 11, 20, 4],
             'sample': '226032_C-ME-18_pCAGseqF',
             'sample_well': 'B9',
             'dye': 'Z-BigDyeV3',
             'polymer': 'POP7                            ',
             'machine_model': '3730',
             'run_start': '2009-12-12 09:56:53',
             'run_finish': '2009-12-12 11:44:49',
             'abif_raw_keys': set(['AEPt1', 'AEPt2', 'APFN2', 'APXV1', 'APrN1',
                                   'APrV1', 'APrX1', 'ARTN1', 'ASPF1', 'ASPt1',
                                   'ASPt2', 'B1Pt1', 'B1Pt2', 'BCTS1', 'BufT1',
                                   'CTID1', 'CTNM1', 'CTOw1', 'CTTL1', 'CpEP1',
                                   'DATA1', 'DATA2', 'DATA3', 'DATA4', 'DATA5',
                                   'DATA6', 'DATA7', 'DATA8', 'DCHT1', 'DSam1',
                                   'DySN1', 'Dye#1', 'DyeN1', 'DyeN2', 'DyeN3',
                                   'DyeN4', 'DyeW1', 'DyeW2', 'DyeW3', 'DyeW4',
                                   'EPVt1', 'EVNT1', 'EVNT2', 'EVNT3', 'EVNT4',
                                   'FTab1', 'FVoc1', 'FWO_1', 'Feat1', 'GTyp1',
                                   'HCFG1', 'HCFG2', 'HCFG3', 'HCFG4', 'InSc1',
                                   'InVt1', 'LANE1', 'LIMS1', 'LNTD1', 'LsrP1',
                                   'MCHN1', 'MODF1', 'MODL1', 'NAVG1', 'NLNE1',
                                   'NOIS1', 'PBAS1', 'PBAS2', 'PCON1', 'PCON2',
                                   'PDMF1', 'PDMF2', 'PLOC1', 'PLOC2', 'PSZE1',
                                   'PTYP1', 'RGOw1', 'RMXV1', 'RMdN1', 'RMdX1',
                                   'RPrN1', 'RPrV1', 'RUND1', 'RUND2', 'RUND3',
                                   'RUND4', 'RUNT1', 'RUNT2', 'RUNT3', 'RUNT4',
                                   'Rate1', 'RevC1', 'RunN1', 'S/N%1', 'SCAN1',
                                   'SMED1', 'SMLt1', 'SMPL1', 'SPAC1', 'SPAC2',
                                   'SPAC3', 'SVER1', 'SVER2', 'SVER3', 'Scal1',
                                   'Scan1', 'TUBE1', 'Tmpr1', 'User1', 'phAR1',
                                   'phCH1', 'phDY1', 'phQL1', 'phTR1', 'phTR2',
                                   'AUDT1', 'PXLB1', 'RMdV1', 'DATA9', 'RGNm1',
                                   'DATA11', 'DATA10', 'DATA12']),
            },
'data_3100': {
             'path': ['Abi', '3100.ab1'],
             'seq':
'CAAGATTGCATTCATGATCTACGATTACTAGCGATTCCAGCTTCATATAGTCGAGTTGCAGACTACAATCCGAACTGAGAACAACTTTATGGGATTTGCTTGACCTCGCGGTTTCGCTGCCCTTTGTATTGTCCATTGTAGCACGTGTGTAGCCCAAATCATAAGGGGCATGATGATTTGACGTCATCCCCACCTTCCTCCGGTTTGTCACCGGCAGTCAACTTAGAGTGCCCAACTTAATGATGGCAACTAAGCTTAAGGGTTGCGCTCGTTGCGGGACTTAACCCAACATCTCACGACACGAGCTGACGACAACCATGCACCACCTGTCACTCTGTCCCCCGAAGGGGAAAACTCTATCTCTAGAGGAGTCAGAGGATGTCAAGATTTGGTAAGGTTCTTCGCGTTGCTTCGAATTAAACCACATGCTCCACCGCTTGTGCGGGTCCCCGTCAATTCCTTTGAGTTTCAACCTTGCGGTCGTACTCCCCAGGCGGAGTGCTTAATGCGTTAGCTGCAGCACTAAGGGGCGGAAACCCCCTAACACTTAGCACTCATCGTTTACGGCGTGGACTACCAGGGTATCTAATCCTGTTTGATCCCCACGCTTTCGCACATCAGCGTCAGTTACAGACCAGAAAGTCGCCTTCGCCACTGGTGTTCCTCCATATCTCTGCGCATTTCACCGCTACACATGGAAATTCCACTTTCCTCTTCTGCACTCAAGTTTTCCAGATTTCGATGAACCTTCAACGATGGAGGCCCGTGGCTTTTCACCATTCAAGGAACCTTTTA',
             'qual':
[5, 3, 4, 4, 4, 5, 9, 4, 4, 4, 5, 4, 4, 4, 4, 4, 6, 13, 23, 20, 15, 17, 10, 9, 13, 23, 18, 14, 19, 34, 15, 24, 34, 18, 30, 35, 41, 34, 22, 39, 53, 58, 35, 46, 24, 42, 46, 46, 54, 42, 53, 58, 52, 41, 22, 62, 52, 62, 52, 62, 41, 38, 16, 42, 62, 56, 62, 62, 62, 62, 62, 62, 54, 56, 39, 62, 49, 35, 19, 44, 44, 42, 62, 62, 62, 62, 54, 62, 62, 62, 62, 49, 49, 43, 39, 62, 62, 62, 62, 62, 62, 62, 59, 43, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 49, 59, 62, 62, 59, 59, 62, 62, 62, 62, 62, 59, 62, 62, 62, 62, 62, 62, 59, 59, 62, 62, 62, 62, 62, 59, 62, 62, 59, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 59, 59, 62, 59, 59, 62, 62, 62, 59, 59, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 50, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 59, 59, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 54, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 59, 59, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 59, 59, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 49, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 42, 31, 31, 26, 18, 24, 33, 56, 56, 62, 56, 62, 47, 39, 62, 62, 62, 62, 62, 62, 49, 47, 62, 62, 62, 49, 62, 62, 62, 62, 49, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 58, 58, 58, 58, 58, 58, 58, 58, 58, 54, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 54, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 53, 58, 58, 58, 51, 49, 58, 58, 58, 58, 50, 58, 58, 58, 58, 58, 58, 51, 32, 58, 58, 42, 58, 51, 58, 49, 39, 58, 58, 51, 58, 53, 58, 50, 58, 58, 58, 53, 54, 53, 58, 54, 53, 58, 53, 58, 32, 58, 44, 46, 50, 58, 54, 58, 58, 49, 41, 58, 58, 58, 54, 54, 54, 51, 52, 53, 51, 54, 54, 54, 38, 52, 53, 47, 54, 54, 53, 54, 54, 54, 52, 54, 42, 47, 35, 42, 54, 45, 54, 53, 54, 23, 34, 24, 51, 52, 33, 46, 44, 33, 49, 51, 45, 53, 32, 52, 53, 52, 51, 36, 52, 53, 52, 52, 37, 53, 52, 40, 27, 24, 46, 16, 33, 45, 51, 43, 51, 51, 46, 46, 45, 42, 31, 43, 41, 43, 35, 43, 43, 42, 31, 46, 46, 32, 46, 35, 42, 46, 42, 31, 31, 43, 42, 35, 21, 42, 35, 28, 46, 47, 37, 41, 29, 39, 31, 32, 39, 31, 41, 43, 34, 43, 36, 27, 31, 27, 35, 36, 24, 44, 33, 31, 32, 38, 35, 31, 32, 29, 31, 31, 31, 38, 38, 36, 31, 34, 30, 21, 28, 32, 32, 47, 31, 33, 30, 23, 22, 25, 25, 26, 30, 30, 21, 28, 31, 21, 14, 5, 3, 16, 9, 18, 27, 8, 19, 4, 5, 13, 47, 27, 14, 21, 9, 17, 21, 17, 27, 14, 23, 23, 14, 15, 28, 15, 12, 8, 5, 3, 17, 28, 16, 13, 17, 11, 2, 2, 3, 29, 7, 3, 1, 3, 3, 8, 7, 7, 24, 12, 3, 4, 15, 3, 4, 7, 5, 1, 2, 2, 8, 24, 6, 4, 11, 23, 9, 6, 4, 3, 5, 3, 5, 17, 17, 22, 11, 4, 5, 8, 16, 10, 6, 11, 6, 11, 5, 5, 27, 18, 7, 15, 16, 30, 21, 14, 7],
             'sample': '16S_S2_1387R',
             'sample_well': 'H3',
             'dye': 'Z-BigDyeV3',
             'polymer': 'POP7                            ',
             'machine_model': '3100',
             'run_start': '2010-01-27 09:52:45',
             'run_finish': '2010-01-27 10:41:07',
             'abif_raw_keys': set(['AEPt1', 'AEPt2', 'APFN2', 'APXV1', 'APrN1',
                                   'APrV1', 'APrX1', 'ARTN1', 'ASPF1', 'ASPt1',
                                   'ASPt2', 'B1Pt1', 'B1Pt2', 'BCTS1', 'CTID1',
                                   'CTNM1', 'CTOw1', 'CTTL1', 'CpEP1', 'DATA1',
                                   'DATA2', 'DATA3', 'DATA4', 'DATA5', 'DATA6',
                                   'DATA7', 'DATA8', 'DCHT1', 'DSam1', 'DySN1',
                                   'Dye#1', 'DyeN1', 'DyeN2', 'DyeN3', 'DyeN4',
                                   'DyeW1', 'DyeW2', 'DyeW3', 'DyeW4', 'EPVt1',
                                   'EVNT1', 'EVNT2', 'EVNT3', 'EVNT4', 'FTab1',
                                   'FVoc1', 'FWO_1', 'Feat1', 'GTyp1', 'HCFG1',
                                   'HCFG2', 'HCFG3', 'HCFG4', 'InSc1', 'InVt1',
                                   'LANE1', 'LAST1', 'LIMS1', 'LNTD1', 'LsrP1',
                                   'MCHN1', 'MODF1', 'MODL1', 'NAVG1', 'NLNE1',
                                   'NOIS1', 'OfSc1', 'P1AM1', 'P1RL1', 'P1WD1',
                                   'P2AM1', 'P2BA1', 'P2RL1', 'PBAS1', 'PBAS2',
                                   'PCON1', 'PCON2', 'PDMF1', 'PDMF2', 'PLOC1',
                                   'PLOC2', 'PSZE1', 'PTYP1', 'RGOw1', 'RMXV1',
                                   'RMdN1', 'RMdX1', 'RPrN1', 'RPrV1', 'RUND1',
                                   'RUND2', 'RUND3', 'RUND4', 'RUNT1', 'RUNT2',
                                   'RUNT3', 'RUNT4', 'Rate1', 'RunN1', 'S/N%1',
                                   'Satd1', 'SCAN1', 'SMED1', 'SMLt1', 'SMPL1',
                                   'SPAC1', 'SPAC2', 'SPAC3', 'SVER1', 'SVER2',
                                   'SVER3', 'Scal1', 'Scan1', 'TUBE1', 'Tmpr1',
                                   'User1', 'phAR1', 'phCH1', 'phDY1', 'phQL1',
                                   'phTR1', 'phTR2', 'AUDT1', 'PXLB1', 'RMdV1',
                                   'DATA9', 'RGNm1', 'DATA11', 'DATA10',
                                   'DATA12']),
            },
'data_310': {
            'path': ['Abi', '310.ab1'],
            'seq':
'TGATNTTNACNNTTTTGAANCANTGAGTTAATAGCAATNCTTTACNAATAAGAATATACACTTTCTGCTTAGGGATGATAATTGGCAGGCAAGTGAATCCCTGAGCGTGNATTTGATAATGACCTAAATAATGGATGGGGTTTTAATTCCCAGACCTTCCCCTTTTTAANNGGNGGATTANTGGGGGNNNAACNNGGGGGGCCCTTNCCNAAGGGGGAAAAAATTTNAAACCCCCCNAGGNNGGGNAAAAAAAAATTTCCAAATTNCCGGGGTNNCCCCCAANTTTTTNCCGCNGGGAAAANNNNCCCCCCCNGGGNCCCCCCCCNNAAAAAAAAAAAAAAAAACCCCCCCCCCNTTGGGGNGGTNTNCNCCCCCNNANAANNGGGGGNNAAAAAAAAAGGCCCCCCCCAAAAAAAACCCNCNTTCTNNCNNNNNGNNCNGNNCCCCCNNCCNTNTNGGGGGGGGGGGNGGAAAAAAAACCCCTTTNTGNNNANANNAACCCNCTCNTNTTTTTTTTTTTANGNNNNCNNNNCAAAAAAAAANCNCCCCCNNCNNNCNNNCNCCCCNNNNTNAAAANANNAANNNNTTTTTTTNGGGGGGGTGNGCGNCCCNNANCNNNNNNNNGCGNGGNCNCCNNCCCNCNANAAANNNTNTTTTTTTTTTTTTTTNTNNTCNNCCCNNNCCCCNNCCCCCCCCCCCCCNCCNCNNNNNGGGGNNNCGGNNCNNNNNNNCCNTNCTNNANATNCCNTTNNNNNNNNGNNNNNNNNACNNNNNTNNTNNNCNNNNNNNNNNNNNNCNNNNNNCNNCCCNNCANNNNNNNCNNNNNNNNNNNNNNNNNNNNNTCNCTNCNCNCCCCNCCCNNNNNNNG',
            'qual':
[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            'sample': 'D11F',
            'sample_well': 'C5',
            'dye': None,
            'polymer': None,
            'machine_model': '310 ',
            'run_start': '2009-02-19 01:19:30',
            'run_finish': '2009-02-19 04:04:15',
            'abif_raw_keys': set(['AEPt1', 'AEPt2', 'APXV1', 'APrN1', 'APrV1',
                                  'APrX1', 'ASPF1', 'ASPt1', 'ASPt2', 'B1Pt1',
                                  'B1Pt2', 'BCTS1', 'CMNT1', 'CpEP1', 'DATA1',
                                  'DATA2', 'DATA3', 'DATA4', 'DATA5', 'DATA6',
                                  'DATA7', 'DATA8', 'Dye#1', 'DyeN1', 'DyeN2',
                                  'DyeN3', 'DyeN4', 'EPVt1', 'EVNT1', 'EVNT2',
                                  'EVNT3', 'EVNT4', 'FTab1', 'FVoc1', 'FWO_1',
                                  'Feat1', 'InSc1', 'InVt1', 'LANE1', 'LNTD1',
                                  'LsrP1', 'MCHN1', 'MODF1', 'MODL1', 'PBAS1',
                                  'PBAS2', 'PCON1', 'PCON2', 'PDMF1', 'PDMF2',
                                  'PLOC1', 'PLOC2', 'RUND1', 'RUND2', 'RUND3',
                                  'RUND4', 'RUNT1', 'RUNT2', 'RUNT3', 'RUNT4',
                                  'Rate1', 'RevC1', 'S/N%1', 'SMED1', 'SMLt1',
                                  'SMPL1', 'SPAC1', 'SPAC2', 'SPAC3', 'SVER1',
                                  'SVER2', 'SVER3', 'Scan1', 'ScSt1', 'TUBE1',
                                  'Tmpr1', 'User1', 'phAR1', 'phCH1', 'phDY1',
                                  'phQL1', 'phTR1', 'phTR2', 'DyeS2', 'DyeS3',
                                  'DyeS1', 'DyeS4', 'DATA11', 'DATA10',
                                  'DATA12', 'CCDF4', 'CCDF1', 'CCDF3', 'CCDF2',
                                  'THUM2', 'THUM1', 'MTXF2', 'MTRX1', 'SpNm1',
                                  'MTRD101', 'MTRD103', 'MTRD102', 'MTRX101',
                                  'MTRX103', 'MTRX102', 'MTRD1', 'MTXF1',
                                  'OFFS1', 'DyeZ1', 'DyeZ3', 'DyeZ2', 'DyeZ4',
                                  'DATA9']),
           },
'test_fsa': {
             'path': ['Abi', 'test.fsa'],
             'seq': '',
             'sample': 'ae524aa33cc011d9990900087493610c',
             'sample_well': 'E1',
             'dye': 'F',
             'polymer': 'POP7                            ',
             'machine_model': '3100',
             'run_start': '2004-11-22 11:58:35',
             'run_finish': '2004-11-22 12:42:08',
             'abif_raw_keys': set(['RGNm1', 'MCHN1', 'DyeW1', 'Dye#1',
                                   'InVt1', 'EVNT4', 'HCFG1', 'RunN1',
                                   'PXLB1', 'DyeB1', 'DCHT1', 'DyeN3',
                                   'DyeW2', 'DyeW3', 'DyeN2', 'SVER3',
                                   'MODL1', 'SpNm1', 'HCFG2', 'RMdX1',
                                   'LANE1', 'DATA5', 'EVNT2', 'DATA7',
                                   'SVER4', 'RMdV1', 'RUND1', 'HCFG3',
                                   'RUNT1', 'RUNT2', 'DyeB4', 'User1',
                                   'RGOw1', 'DyeW4', 'DATA1', 'CTID1',
                                   'DyeB3', 'DATA2', 'HCFG4', 'SMLt1',
                                   'EVNT3', 'CpEP1', 'EVNT1', 'Tmpr1',
                                   'Satd1', 'GTyp1', 'DySN1', 'Scan1',
                                   'CTTL1', 'RUNT3', 'RPrV1', 'RUND4',
                                   'LIMS1', 'DATA4', 'DATA6', 'DyeB2',
                                   'DyeN4', 'NLNE1', 'InSc1', 'OfSc1',
                                   'RMdN1', 'DATA3', 'MODF1', 'NAVG1',
                                   'Rate1', 'SMED1', 'RUNT4', 'LNTD1',
                                   'SVER1', 'RUND2', 'DSam1', 'CTNM1',
                                   'Scal1', 'SCAN1', 'TUBE1', 'DyeN1',
                                   'CTOw1', 'RUND3', 'RMXV1', 'EPVt1',
                                   'LsrP1', 'RPrN1', 'DATA8']),
            },
}

test_data_fake = {
'fake': {
              'path': ['Abi', 'fake.ab1'],
             }
}


def open_files(test_array):
    for trace in test_array:
        test_array[trace]['handle'] = open(join(*test_array[trace]['path']), 'rb')


def open_files_wrong_mode(test_array):
    for trace in test_array:
        test_array[trace]['handle'] = open(join(*test_array[trace]['path']))


def close_files(test_array):
    for trace in test_array:
        test_array[trace]['handle'].close()


class TestAbi(unittest.TestCase):

    def setUp(self):
        open_files(test_data)

    def tearDown(self):
        close_files(test_data)

    def test_file_type(self):
        """Test if filetype is ABIF."""
        for trace in test_data:
            self.assertEqual(test_data[trace]['handle'].read(4), _as_bytes('ABIF'))

    def test_seqrecord(self):
        """Test if the extracted seqrecords data are equal to expected values."""
        for trace in test_data:
            record = SeqIO.read(test_data[trace]['handle'], 'abi')
            if trace != 'test_fsa':
                self.assertEqual(basename(test_data[trace]["path"][-1]).replace('.ab1', ''), record.name)
                self.assertEqual(test_data[trace]['seq'], str(record.seq))
                self.assertEqual(test_data[trace]['qual'], record.letter_annotations['phred_quality'])
            self.assertEqual(test_data[trace]['sample'], record.id)
            self.assertEqual(test_data[trace]['sample_well'], record.annotations['sample_well'])
            self.assertEqual(test_data[trace]['machine_model'], record.annotations['machine_model'])
            self.assertEqual(test_data[trace]['run_start'], record.annotations['run_start'])
            self.assertEqual(test_data[trace]['run_finish'], record.annotations['run_finish'])
            self.assertEqual(test_data[trace]['abif_raw_keys'], set(record.annotations['abif_raw']))

    def test_trim(self):
        """Test if trim works."""
        for trace in test_data:
            record = SeqIO.read(test_data[trace]['handle'], 'abi-trim')
            if trace != 'data_empty' and trace != 'test_fsa':
                self.assertNotEqual(str(record.seq), test_data[trace]['seq'])
                self.assertIn(str(record.seq), test_data[trace]['seq'])
            else:
                self.assertEqual(str(record.seq), test_data[trace]['seq'])

    def test_raw(self):
        """Test access to raw ABIF tags."""
        record = SeqIO.read("Abi/A6_1-DB3.ab1", "abi")
        self.assertEqual(set(record.annotations),
                         set(["polymer", "run_finish", "sample_well", "run_start",
                              "machine_model", "dye", "abif_raw"]))

        self.assertEqual(set(record.annotations["abif_raw"]),
                         set(["RUND2", "RUND1", "DySN1", "SMPL1", "GTyp1",
                              "PCON2", "RUNT2", "PBAS2", "RUNT1", "MODL1",
                              "TUBE1", "RMdX1", "ASPt1", "ASPt2", "PCON1",
                              "DyeN4", "DyeN1", "DyeN3", "DyeN2", "APrX1",
                              "SVER3", "SVER2", "SVER1", "LsrP1", "RPrV1",
                              "ARTN1", "Dye#1", "LAST1", "P1AM1", "MCHN1",
                              "DCHT1", "P1RL1", "DyeW4", "DyeW2", "RGOw1",
                              "DyeW1", "EVNT4", "ASPF1", "EVNT2", "EVNT3",
                              "PDMF2", "PDMF1", "BCTS1", "phCH1", "HCFG3",
                              "HCFG2", "HCFG1", "B1Pt1", "B1Pt2", "HCFG4",
                              "phQL1", "LANE1", "EPVt1", "Scal1", "PTYP1",
                              "PLOC2", "PLOC1", "P2RL1", "CTNM1", "RMdN1",
                              "FWO_1", "APXV1", "DATA8", "SMLt1", "DATA4",
                              "CpEP1", "DATA6", "DATA1", "DATA3", "DATA2",
                              "phAR1", "RUNT4", "RUNT3", "SMED1", "FTab1",
                              "phDY1", "FVoc1", "APFN2", "RMXV1", "InVt1",
                              "DSam1", "CMNT1", "phTR2", "phTR1", "CTOw1",
                              "Rate1", "NOIS1", "Feat1", "Scan1", "InSc1",
                              "RUND4", "RUND3", "CTID1", "NAVG1", "P2BA1",
                              "P2AM1", "SCAN1", "BufT1", "APrV1", "LIMS1",
                              "LNTD1", "P1WD1", "Tmpr1", "MODF1", "DyeW3",
                              "SPAC2", "SPAC3", "SPAC1", "APrN1", "PSZE1",
                              "RunN1", "DATA5", "DATA7", "RPrN1", "EVNT1",
                              "AEPt2", "AEPt1", "User1", "NLNE1", "PBAS1",
                              "S/N%1", "CTTL1", "AUDT1", "PXLB1", "RMdV1",
                              "DATA9", "RGNm1", "DATA11", "DATA10", "DATA12"
                              ]))

        self.assertEqual(record.annotations["abif_raw"]['AEPt1'], 16627)
        self.assertEqual(record.annotations["abif_raw"]['AEPt2'], 16627)
        self.assertEqual(record.annotations["abif_raw"]['APFN2'], 'KBanalysis')
        self.assertEqual(record.annotations["abif_raw"]['APXV1'], '2')
        self.assertEqual(record.annotations["abif_raw"]['APrN1'], 'KBanalysis')
        self.assertEqual(record.annotations["abif_raw"]['APrV1'], '3')
        self.assertEqual(record.annotations["abif_raw"]['APrX1'], """<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
<AnalysisProtocolContainer doAnalysis="true" name="KBanalysis" guid="DummyGUID" version="2"><AnalysisProtocol version="3" doAnalysis="true" guid="DummyGUID" name="Basecaller"><AnalysisSettings version="3" name="BasecallerSettings"><Parameter name="BasecallerName"><value>KB.bcp</value></Parameter><Parameter name="Description"><value>automatically performs KB analysis</value></Parameter><Parameter name="MobilityFileName"><value>KB_3730_POP7_BDTv3.mob</value></Parameter><Parameter name="MatrixFileName"><value></value></Parameter><Parameter name="WriteSeqFile"><value>true</value></Parameter><Parameter name="WriteClearRange"><value>false</value></Parameter><Parameter name="WriteSCFFile"><value>false</value></Parameter><Parameter name="WritePHREDFile"><value>true</value></Parameter><Parameter name="SeqFileFormat"><value>2</value></Parameter><Parameter name="ProcessedDataProfile"><value>0.0</value></Parameter><Parameter name="NCallQvThreshold"><value>10</value></Parameter></AnalysisSettings><AnalysisSettings version="3" name="MobilitySettings"><Parameter name="MobilityFileName"><value>KB_3730_POP7_BDTv3.mob</value></Parameter><Parameter name="MobilityFileContents"><value>QUJJRgBldGRpcgAAAAED/wAcAAAACAAAAdwAAACXAAAAAP//////////////////////////////\r
////////////////////////////////////////////////////////////////////////////\r
//////////////////8WS0JfMzczMF9QT1A3X0JEVHYzLmJjY0NNTlQAAAABABIAAQAAAAEAAAAB\r
AAAAAAAAAABDTkFNAAAAAQASAAEAAAAXAAAAFwAAAIAAAAAARFlFXwAAAAEABAACAAAAAQAAAAIA\r
AwAAAAAAAEZXT18AAAABAAIAAQAAAAQAAAAER0FUQwAAAABTSEZDAAAAAQAEAAIAAAABAAAAAgAA\r
AAAAAAAAU0hGRAAAAAEABAACAAAAAQAAAAIAAAAAAAAAAFNIRk4AAAABAAQAAgAAAAEAAAACAAAA\r
AAAAAABTSEZTAAAAAQAEAAIAAAABAAAAAgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\r
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\r
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\r
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\r
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\r
</value></Parameter></AnalysisSettings><AnalysisSettings version="3" name="BasecallerParamsSettings"><Parameter name="StopPoint"><value>0</value></Parameter><Parameter name="AverageSpacing"><value>0.0</value></Parameter><Parameter name="StartPoint"><value>0</value></Parameter><Parameter name="BaseOneLocation"><value>0</value></Parameter></AnalysisSettings><AnalysisSettings version="3" name="EndBaseSettings"><Parameter name="StopAtPCRStop"><value>false</value></Parameter><Parameter name="StopAfterNumNs"><value>false</value></Parameter><Parameter name="MaxNumNs"><value>20</value></Parameter><Parameter name="StopAfterNumNsInXBases"><value>false</value></Parameter><Parameter name="NumNsPerXBaseInterval"><value>5</value></Parameter><Parameter name="XBaseInterval"><value>10</value></Parameter><Parameter name="StopAfterNumBases"><value>false</value></Parameter><Parameter name="MaxBasesToCall"><value>800</value></Parameter></AnalysisSettings><AnalysisSettings version="3" name="MixedBasesSettings"><Parameter name="CallMixedBases"><value>false</value></Parameter><Parameter name="SecondaryPeakThreshold"><value>25.0</value></Parameter></AnalysisSettings><AnalysisSettings version="3" name="ClearRangeSettings"><Parameter name="UseConfidenceRange"><value>false</value></Parameter><Parameter name="UseClearRangeThreshold"><value>false</value></Parameter><Parameter name="UseIdentificationOfNcalls"><value>false</value></Parameter><Parameter name="UseReferenceTrimming"><value>false</value></Parameter><Parameter name="FirstBase"><value>20</value></Parameter><Parameter name="UseLastBase"><value>true</value></Parameter><Parameter name="LastBase"><value>550</value></Parameter><Parameter name="BasesToTrimFrom3PrimeEnd"><value>20</value></Parameter><Parameter name="MaxLowQualityBases"><value>4</value></Parameter><Parameter name="CvThreshold"><value>20</value></Parameter><Parameter name="LowQualityWindow"><value>20</value></Parameter><Parameter name="MaxAmbiguities"><value>4</value></Parameter><Parameter name="AmbiguityWindow"><value>20</value></Parameter><Parameter name="UseM13Masking"><value>false</value></Parameter></AnalysisSettings><AnalysisSettings version="3" name="BCPSettings"><Parameter name="BCPFileName"><value>KB.bcp</value></Parameter><Parameter name="BCPFileContents"><value># 5Prime Basecaller Paramters</value></Parameter></AnalysisSettings></AnalysisProtocol></AnalysisProtocolContainer>

""")
        self.assertEqual(record.annotations["abif_raw"]['ARTN1'], 0)
        self.assertEqual(record.annotations["abif_raw"]['ASPF1'], 1)
        self.assertEqual(record.annotations["abif_raw"]['ASPt1'], 2430)
        self.assertEqual(record.annotations["abif_raw"]['ASPt2'], 2430)
        self.assertEqual(record.annotations["abif_raw"]['B1Pt1'], 2429)
        self.assertEqual(record.annotations["abif_raw"]['B1Pt2'], 2429)
        self.assertEqual(record.annotations["abif_raw"]['BCTS1'], '2014-06-04 02:05:21 -04:00')
        self.assertEqual(record.annotations["abif_raw"]['BufT1'], (-27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27, -27))
        self.assertEqual(record.annotations["abif_raw"]['CMNT1'], '1565127-C12-brodigan, thomas-10-264763806-TB01')
        self.assertEqual(record.annotations["abif_raw"]['CTID1'], '1565127')
        self.assertEqual(record.annotations["abif_raw"]['CTNM1'], '1565127')
        self.assertEqual(record.annotations["abif_raw"]['CTOw1'], 'genewiz')
        self.assertEqual(record.annotations["abif_raw"]['CTTL1'], 'Comment:')
        self.assertEqual(record.annotations["abif_raw"]['CpEP1'], '\x01')
        self.assertEqual(record.annotations["abif_raw"]['DATA1'], (10, -1, -8, -8, -9, 0, 0, -4, 7, 8, 4, -8, -3, 3, -4, -8, -15, -7, -4, -3, -7, 0, -7, 8, -15, 1, -17, -2, 0, 2, -4, 11, -10, 6, -14, -2, -9, -2, -10, -9, -5, -5, 1, -11, -17, -8, -10, -9, 7, -12, -17, -18, -8, 1, 10, 0, -22, -10, -13, -8, -2, -12, -14, 2, -11, -23, -10, -10, -10, -4, 0, -14, -11, -5, -15, -9, -3, -10, -6, -15, 9, -9, -6, -10, -6, -13, 0, -3, -5, -7, -5, -6, -15, 0, -18, -9, -5, -2, -7, -3, -1, -8, -6, 0, -3, -3, -8, -17, -8, -8, -9, 1, -7, -15, -20, 8, -10, -3, -4, -4, -2, -3, -2, -7, -11, -7, -1, -4, 0, 0, -13, -12, 9, -1, -4, 1, 0, -5, -9, 11, -8, -1, 6, 8, -10, 6, 2, 0, 3, 1, -3, -12, -10, 6, -12, 0, -5, 0, -9, 6, 1, -6, 6, 0, 5, 2, -2, 7, -8, -2, 7, 4, 9, 11, 6, 16, 34, 46, 28, 23, 15, 8, 11, 24, 19, 26, 16, 13, 15, 17, 17, 10, 13, 19, 10, -3, 15, 6, 7, 3, 12, 16, 22, 10, 18, 16, 9, 11, 16, 16, 12, 12, 2, 10, 6, 9, 9, 0, 5, 14, -2, 15, 11, 11, 8, 1, 21, 4, 17, 5, 12, 4, 5, 14, 20, 13, 4, 13, 19, 17, 16, 16, 3, 12, 8, 18, 11, 22, 0, 7, 1, -5, 10, 5, 0, 11, 12, 14, 9, 5, 1, 4, 19, 17, 12, 20, 16, 28, 9, 19, 12, 13, 15, 17, 4, 8, 12, 15, 15, 15, 17, 17, 2, 7, 4, 22, 6, 16, 15, 14, 27, 24, 6, 22, 27, 18, 4, 13, 16, 9, 4, 9, 23, 14, 13, 18, 9, 6, 5, 8, 13, 8, 9, 19, 8, 11, 16, 7, 19, 13, 14, 7, 15, 21, 18, 7, 19, 16, 13, 21, 18, 14, 16, 5, 20, 14, 24, 9, 26, 16, 16, 9, 21, 24, 15, 14, 18, 23, 30, 23, 33, 31, 37, 43, 39, 49, 47, 55, 42, 59, 62, 51, 58, 54, 44, 47, 41, 45, 47, 40, 49, 40, 26, 25, 26, 23, 42, 39, 15, 20, 26, 23, 14, 16, 12, 16, 34, 10, 13, 14, 2, 8, 20, 8, 17, 20, 4, 27, 17, 29, 10, 11, 17, 9, 12, 15, 20, 13, 14, 2, 14, 14, 20, 11, 15, 22, 16, 9, 8, 13, 13, 13, 12, 14, 12, 5, 2, 9, 15, 20, 13, 11, 6, 16, 1, 10, 18, 17, 16, 23, 6, 4, 9, 18, 21, 3, 16, 13, 35, 15, 23, 12, 28, 22, 19, 17, 15, -2, 16, 11, 20, 17, 25, 17, 13, 33, 5, 22, 17, 16, 20, 24, 15, 19, 17, 24, 11, 14, 26, 12, 14, 20, 18, 5, 21, 16, 12, 13, 14, 28, 17, 15, 14, 16, 18, 16, 20, 13, 27, 17, 20, 11, 13, 21, 9, 0, 15, 22, 10, 13, 13, -3, 13, 23, 11, 21, 16, 11, 20, 14, 12, 13, 11, 10, 10, 8, 14, 4, 15, 9, 14, 6, 17, 15, 17, 9, 25, 7, 10, 13, 10, 16, 11, 19, 9, 26, 14, 6, 13, 8, 22, 11, 8, 18, 4, 16, 8, 14, 17, 16, 13, 4, 5, 18, 1, 25, 18, 12, 7, 28, 15, 16, 22, 0, -5, 3, 3, 11, 6, 12, 4, 19, 0, 7, 13, 20, 17, 10, 5, 9, 8, 7, 11, 10, 6, 11, 5, 10, 15, 5, 5, 13, 21, 12, 20, 3, 13, 2, 4, 6, 17, 7, 10, 0, 2, 23, 14, 7, 23, 10, 13, -2, 7, 12, 12, 15, -2, 0, 8, 17, 6, 10, 12, 20, 15, 10, 5, 0, 13, 21, 13, 18, 16, 12, 14, 14, 6, 23, 23, 19, 20, 2, 20, 19, 20, 8, 14, 8, 12, 27, 6, 10, 7, 15, 6, 7, 15, 6, 13, 7, 0, 17, 19, 0, 0, 4, 14, 0, 10, 9, 15, 11, 2, -3, 15, 10, 5, 4, 11, 12, 6, 15, 1, 11, 8, 10, -7, 0, 5, 3, 8, -2, 10, 5, 5, 0, -5, 0, 0, 2, 4, 2, 3, -5, -3, 9, 3, -2, -1, -2, -5, -2, 1, 4, -7, -3, 5, 1, -12, -3, 5, -9, -3, 1, 7, 2, 0, 4, 9, -2, -6, -3, 4, -14, 0, -9, -2, -4, 0, 1, -9, -5, 1, 12, -6, 10, -7, -1, 6, 0, -3, 9, -13, 9, 2, 6, 4, -5, -4, 1, -5, 0, 3, -1, -8, 5, -7, 3, 3, -4, 10, 15, 3, 7, -3, 7, 4, 4, 1, 14, 9, 23, 4, 19, 17, 21, 23, 25, 20, 31, 27, 34, 47, 44, 57, 71, 64, 84, 78, 91, 105, 117, 128, 140, 151, 142, 154, 150, 124, 111, 97, 86, 81, 71, 56, 51, 40, 48, 42, 39, 25, 25, 22, 21, 20, 28, 10, 23, 16, 21, 4, 22, 6, 17, 10, 12, 11, 12, 15, 18, 23, 21, 17, 21, 13, 7, 10, 18, 5, 18, 5, 6, 23, 19, 11, 33, 6, 24, 32, 24, 32, 23, 21, 27, 35, 36, 24, 31, 19, 28, 10, 34, 18, 17, 25, 13, 25, 25, 27, 22, 18, 21, 11, 11, 23, 19, 35, 19, 25, 25, 25, 16, 37, 16, 13, 26, 23, 13, 6, 16, 5, 13, 22, 10, 33, 1, 16, 25, 10, 14, 23, 32, 25, 13, 22, 17, 9, 21, 15, 14, 9, 28, 20, 22, 15, 12, 24, 25, 30, 24, 17, 23, 11, 17, 10, 15, 6, 12, 15, 21, 23, 11, 12, 16, 3, 21, 33, 12, -1, 13, 11, 12, 5, 0, 12, 11, 15, 8, 6, 0, 11, 10, 11, 21, 7, 8, 10, 18, 8, 13, 18, 5, 12, 9, 1, 4, 6, 7, 16, 11, 16, 18, 9, 12, 27, 18, 4, 9, 4, 13, 3, 13, 9, 7, 7, 13, -5, 7, 16, 17, 17, 13, 11, 1, 6, 14, 3, 2, 21, 13, 14, 21, 11, 11, 13, 3, 3, 2, 15, 2, 9, -8, 17, 10, 7, 4, -6, 0, 4, 11, 8, 8, 20, -2, 10, 0, 0, 10, 11, 5, 8, -3, 4, 8, 4, 9, 7, 1, 25, 12, 13, 3, 11, 0, 0, 5, 12, 7, 0, 4, 2, 14, 17, 9, 3, 4, 10, 9, 14, 13, 8, 12, 2, 4, -2, 0, 7, 11, 1, 9, 2, 10, 0, 19, 24, 19, 15, 22, 12, 15, 21, 27, 13, 7, 6, 13, 7, 17, 19, 11, 14, 18, 28, 21, 14, 13, 16, 16, 32, 25, 25, 26, 24, 7, 18, 25, 34, 25, 28, 35, 29, 19, 30, 27, 21, 23, 22, 0, 23, 22, 10, 19, 19, 26, 14, 17, 29, 15, 43, 15, 21, 23, 18, 33, 23, 32, 17, 36, 37, 16, 29, 21, 8, 20, 28, 14, 28, 23, 12, 20, 20, 16, 29, 24, 19, 19, 16, 24, 33, 22, 20, 14, 23, 33, 21, 28, 20, 23, 13, 33, 27, 26, 36, 21, 38, 18, 18, 20, 23, 16, 17, 26, 13, 23, 19, 19, 22, 17, 12, 24, 23, 29, 14, 20, 14, 12, 27, 15, 9, 20, 21, 19, 14, 14, 8, 5, 0, 31, 8, 9, 13, 26, 19, 23, 18, 16, 14, 22, 20, 15, 5, 11, 14, 10, 14, 16, 6, 7, 16, 12, 17, 1, 9, 22, 9, 13, 16, 18, 25, 14, 13, 9, 5, 7, 25, 7, 22, 14, 10, 23, 20, 19, 10, 5, 1, 12, 21, 1, 9, 8, 18, 16, 4, 14, -5, 15, 4, 15, -3, 0, 6, 12, 10, 11, 16, -4, 12, 11, 31, 4, 7, 10, 14, 18, 14, 14, 10, 13, 16, 17, 9, 15, 22, 19, 24, -5, 18, 5, 4, 7, 15, 7, 13, 19, 21, 8, 3, 25, 7, 7, 4, 3, -1, 10, 19, 2, 10, 15, 10, 9, 0, 2, 11, 9, 0, 16, 0, 9, 11, 5, 6, -6, 5, 8, 5, -6, -1, 7, 4, 0, 5, 17, 13, 7, 11, 16, 7, 3, 17, 9, 3, 4, 16, 27, 10, 15, 21, 10, 1, 24, 8, 10, 7, 7, 11, 3, 20, 13, 26, 15, 28, 16, 17, 23, 4, 16, -2, 8, 14, 21, 3, 20, 18, 23, 16, 19, 15, 13, 15, 14, 3, 0, 5, 8, 3, 8, 8, 10, 7, -4, 9, 9, 8, 13, 9, 2, 9, 6, 11, 4, 9, 5, 4, 18, 7, -6, -4, 8, 12, -1, -6, 12, 9, 14, 0, -1, 13, 9, -1, 11, 15, 6, 8, 5, 9, 15, 2, 2, 13, 18, 6, 6, -1, 11, 4, 8, 14, 22, 13, 0, 1, 15, 4, 4, 3, 8, 11, 11, 1, 5, 18, 15, 3, 6, 18, -2, 5, 6, 9, 8, 3, 16, 17, 0, 6, 17, 2, 14, 13, 14, 21, 9, 17, 15, 12, 11, 6, 16, 22, 2, 18, 11, 18, 30, 20, 15, 15, 26, 14, 26, 21, 18, 13, 24, 24, 18, 21, 25, 27, 18, 26, 22, 28, 10, 22, 9, 7, -2, 12, 4, 10, -1, 16, 12, 6, 5, 7, -1, 7, 0, 10, 9, 17, 16, 19, 10, 12, 9, 11, 16, -6, 7, 9, 17, 8, 6, 7, 13, 12, -2, 7, 6, 5, 2, 9, 4, 8, 0, 16, 0, 8, 12, 14, 15, 1, 0, 15, 7, 6, 12, -1, 16, 11, 16, 1, -3, 10, 4, 5, 3, 9, 0, 6, 15, 8, 2, 12, 18, 8, 6, 5, 9, 3, 10, -1, 8, 1, 6, -5, 14, 11, 1, -1, 7, 10, 7, 7, 13, 25, 16, 14, 17, 23, 10, 18, 20, 16, 12, 22, 21, 12, 20, 14, 20, 19, 28, 28, 16, 21, 16, 22, 11, 9, 6, 24, 19, 20, 7, 15, 10, 15, 15, 11, 9, 18, 22, 24, 17, 12, 19, 17, -2, 19, 6, 6, 30, 26, 25, 16, 21, 18, 19, 10, 22, 31, 18, 8, 18, 9, 11, 7, 4, 10, 11, 12, 9, 19, 6, 4, 4, 9, 8, 11, 9, 10, 14, 6, 11, 19, 0, -2, 20, -1, 10, 7, 18, 10, 11, 20, 7, 14, 8, 28, 4, 12, 5, 0, 15, 13, 10, 19, 8, 22, 13, 30, 33, 32, 47, 59, 46, 38, 19, 21, 21, 22, 16, 21, 24, 13, 11, 5, 19, 27, 19, 12, 27, 25, 28, 12, 14, 15, 10, 26, 20, 14, 6, 8, 3, 17, 21, 13, 6, 8, 2, 16, 13, 18, 24, 7, 14, 25, 5, 17, 13, 21, 2, 19, 15, 6, 27, 20, 9, 8, 12, 15, 5, 5, 15, 22, 4, 15, 13, 14, 5, 10, 17, 13, 15, 11, 19, 4, 5, 13, 7, 19, 12, 11, 0, 15, 10, 3, 23, 12, 10, 8, 9, 6, 13, 4, 15, 3, 4, 10, 8, 12, 2, 18, 20, 6, 17, 6, 7, 7, 0, 13, 10, 17, 0, 9, 11, 4, 12, 3, 4, 9, 10, 14, 19, 16, 8, 2, 0, -4, 12, 11, 11, 14, 2, 18, 4, 11, 1, 9, 6, 11, 5, 11, 13, 2, 8, 4, 3, 0, 9, 12, 10, 8, 0, 0, 6, 12, 4, 11, 17, 18, 14, 7, 4, 16, 7, -12, 9, 18, 15, 32, 10, 1, 7, 11, 5, 8, 14, 7, 11, 10, 13, 6, 10, 1, 18, 12, 15, -1, 19, 14, 5, 7, 7, 1, 19, 4, 0, 8, 0, 3, 1, -1, 11, 16, 9, 2, 3, 8, 15, 13, 12, 11, 5, 3, 6, 9, 10, -3, 5, 8, 1, 8, 1, 4, 12, 11, -10, 15, -2, 0, 6, 5, 9, 2, -3, 1, 11, -2, 0, -3, 10, -7, 4, 9, 0, -6, 10, 9, -7, -4, -9, -3, 3, 8, 0, 0, 0, -13, 0, 7, -3, 8, 11, -5, 2, -10, 4, -8, 3, 7, 1, 2, 11, 2, 0, 10, 6, 11, -3, 5, 7, 11, 12, 7, 14, 6, 0, 3, 5, 18, 18, 23, 16, 6, 14, 33, 24, 23, 17, 32, 36, 29, 31, 31, 37, 24, 12, 29, 37, 36, 32, 31, 15, 16, 31, 31, 9, 21, 19, 17, 27, 19, 21, 27, 16, 20, 13, 18, 8, 1, 19, 6, 15, 21, 25, 14, 9, 26, 12, 7, 6, 11, -1, 4, 18, 8, 10, 12, 14, 10, 11, 4, 19, 9, 16, 13, 4, 9, 27, 15, 15, 8, 10, 12, 20, 7, 11, 6, 12, 14, 24, 1, 5, 18, 14, 15, 3, 8, 15, 11, 11, 17, 13, 6, 12, 7, 14, 5, 15, 8, 10, 13, 1, 9, 6, 17, 17, 15, 19, 7, 24, 4, 11, 8, 10, 8, 14, 31, 24, 19, 9, 4, 9, 15, 14, 1, 12, 22, 25, 16, 22, 21, 25, 20, 9, 15, 14, 11, 18, 17, 7, 11, 22, 6, -6, 14, 5, 11, 27, 6, 21, 25, 12, 15, 7, 11, 6, 9, 11, 13, 2, 19, 11, 9, 20, 10, 8, 1, 13, 16, 18, 14, 14, 12, 15, 11, 16, 20, 7, 2, -1, 9, 19, -1, 12, 3, 13, 8, 9, 11, 16, 17, 17, 13, 15, 12, 5, 21, 25, 16, 15, 16, 16, 2, 13, 5, 5, 20, 11, 5, 21, 18, 4, 9, 11, 27, 19, 20, 6, 6, 17, 22, 9, 3, 23, 16, 25, 12, 22, 16, 27, 18, 17, 30, 23, 20, 28, 22, 7, 15, 24, 22, 34, 20, 22, 23, 36, 35, 30, 40, 34, 43, 37, 57, 51, 55, 40, 40, 57, 68, 81, 74, 71, 76, 80, 80, 72, 70, 71, 74, 97, 69, 87, 93, 86, 88, 98, 76, 84, 72, 66, 79, 82, 83, 94, 98, 85, 82, 90, 77, 81, 67, 75, 77, 81, 103, 65, 75, 79, 80, 77, 70, 70, 71, 86, 75, 72, 66, 65, 73, 74, 83, 70, 61, 73, 66, 67, 51, 68, 56, 54, 70, 72, 42, 50, 71, 58, 57, 55, 68, 62, 61, 48, 57, 60, 63, 69, 63, 59, 67, 53, 62, 68, 49, 59, 74, 69, 60, 72, 58, 63, 61, 57, 49, 69, 58, 63, 57, 42, 61, 53, 60, 56, 62, 87, 81, 68, 62, 66, 64, 74, 60, 72, 67, 86, 79, 97, 86, 85, 84, 100, 82, 88, 77, 88, 76, 72, 79, 88, 86, 82, 82, 82, 92, 96, 108, 97, 102, 111, 102, 118, 147, 151, 183, 213, 244, 283, 327, 384, 469, 552, 687, 800, 924, 1023, 1180, 1368, 1445, 1656, 1752, 1856, 1994, 2072, 2115, 2186, 2125, 2132, 2027, 1922, 1784, 1712, 1561, 1458, 1279, 1076, 953, 882, 814, 680, 621, 503, 492, 473, 432, 445, 432, 450, 473, 545, 512, 568, 606, 645, 656, 681, 744, 789, 830, 785, 789, 805, 794, 771, 750, 770, 738, 724, 729, 693, 655, 627, 552, 527, 471, 451, 372, 373, 324, 289, 283, 289, 293, 297, 325, 364, 393, 412, 433, 490, 590, 623, 659, 730, 720, 746, 779, 798, 785, 754, 729, 650, 599, 527, 486, 413, 378, 345, 300, 263, 239, 227, 254, 255, 267, 283, 315, 328, 386, 476, 495, 528, 614, 655, 671, 712, 751, 752, 750, 753, 731, 639, 612, 534, 453, 415, 377, 312, 290, 222, 238, 205, 198, 213, 220, 227, 253, 275, 323, 384, 449, 497, 526, 591, 654, 741, 772, 839, 886, 863, 877, 799, 810, 718, 668, 628, 562, 493, 448, 405, 366, 391, 393, 395, 444, 494, 542, 644, 695, 778, 800, 882, 923, 897, 874, 813, 762, 718, 667, 550, 518, 422, 355, 298, 265, 238, 235, 174, 193, 199, 190, 205, 236, 261, 321, 368, 415, 470, 555, 674, 719, 794, 958, 1016, 1108, 1068, 1053, 1082, 1060, 995, 851, 859, 701, 568, 484, 445, 361, 304, 240, 186, 197, 144, 140, 117, 130, 119, 125, 85, 109, 100, 90, 86, 100, 110, 100, 82, 86, 90, 101, 108, 108, 100, 107, 96, 111, 114, 132, 138, 124, 131, 157, 159, 153, 159, 158, 207, 213, 228, 241, 304, 357, 403, 461, 446, 573, 639, 675, 697, 691, 708, 676, 627, 543, 511, 441, 377, 306, 259, 248, 219, 166, 138, 150, 140, 143, 146, 150, 134, 147, 142, 151, 159, 164, 187, 189, 200, 220, 232, 257, 316, 384, 397, 479, 550, 633, 709, 774, 877, 907, 923, 922, 963, 911, 865, 721, 680, 564, 486, 392, 310, 242, 219, 178, 157, 139, 135, 110, 106, 138, 118, 101, 114, 114, 131, 128, 130, 132, 118, 121, 135, 131, 129, 129, 145, 141, 140, 155, 135, 160, 184, 223, 296, 332, 463, 562, 710, 822, 952, 1168, 1255, 1467, 1542, 1617, 1567, 1498, 1361, 1191, 1067, 963, 748, 596, 517, 364, 274, 217, 171, 152, 137, 130, 110, 100, 107, 117, 86, 119, 101, 116, 95, 111, 131, 106, 131, 125, 136, 137, 142, 108, 116, 123, 117, 110, 87, 85, 101, 104, 100, 104, 82, 90, 76, 67, 95, 84, 77, 108, 82, 92, 105, 106, 104, 120, 117, 123, 131, 137, 128, 114, 113, 108, 81, 81, 81, 81, 78, 73, 72, 74, 66, 59, 80, 73, 88, 82, 94, 94, 96, 100, 113, 104, 104, 96, 88, 93, 103, 90, 91, 90, 82, 69, 96, 114, 109, 110, 117, 116, 132, 127, 137, 133, 133, 131, 127, 115, 118, 105, 109, 96, 102, 123, 110, 110, 118, 118, 126, 144, 127, 122, 123, 127, 137, 117, 128, 144, 128, 118, 137, 138, 133, 133, 127, 129, 152, 153, 153, 139, 145, 161, 156, 159, 156, 164, 150, 141, 155, 124, 138, 130, 124, 119, 139, 111, 104, 111, 136, 146, 149, 168, 227, 232, 311, 358, 415, 447, 530, 582, 608, 621, 586, 610, 602, 528, 461, 392, 324, 291, 243, 223, 150, 153, 152, 147, 141, 145, 154, 159, 182, 161, 177, 166, 169, 159, 159, 129, 129, 135, 117, 147, 122, 128, 122, 125, 147, 127, 127, 109, 111, 109, 108, 118, 121, 123, 126, 144, 173, 222, 276, 384, 472, 648, 714, 964, 1200, 1365, 1519, 1665, 1753, 1814, 1726, 1640, 1473, 1314, 1069, 843, 729, 595, 450, 335, 298, 224, 182, 144, 142, 114, 115, 126, 123, 122, 123, 119, 112, 135, 107, 117, 111, 105, 121, 102, 110, 106, 101, 109, 119, 106, 100, 111, 114, 99, 116, 101, 111, 103, 81, 98, 101, 111, 113, 131, 130, 135, 133, 147, 152, 140, 141, 129, 143, 118, 115, 85, 97, 99, 83, 81, 75, 74, 80, 85, 73, 89, 84, 80, 64, 58, 78, 68, 59, 68, 60, 63, 98, 92, 74, 82, 75, 66, 74, 93, 73, 78, 65, 74, 61, 73, 93, 80, 90, 92, 108, 98, 115, 101, 106, 82, 93, 88, 72, 72, 77, 67, 63, 76, 80, 89, 82, 87, 100, 101, 115, 107, 134, 120, 112, 111, 124, 111, 111, 84, 90, 80, 75, 57, 57, 47, 60, 58, 69, 67, 75, 81, 73, 69, 69, 98, 83, 84, 94, 116, 105, 130, 128, 131, 140, 149, 142, 149, 141, 156, 148, 137, 120, 111, 88, 94, 85, 85, 99, 89, 77, 81, 110, 98, 100, 109, 113, 120, 120, 114, 111, 118, 118, 96, 127, 130, 128, 125, 115, 103, 112, 112, 113, 105, 117, 127, 111, 101, 125, 143, 164, 196, 216, 278, 348, 445, 618, 800, 977, 1133, 1309, 1356, 1481, 1476, 1501, 1407, 1299, 1118, 988, 774, 638, 460, 381, 260, 207, 168, 157, 124, 116, 89, 94, 103, 75, 101, 124, 110, 117, 119, 110, 116, 121, 109, 115, 123, 122, 115, 115, 76, 90, 95, 92, 96, 80, 91, 75, 84, 90, 71, 63, 79, 74, 73, 77, 77, 85, 82, 79, 80, 80, 83, 85, 72, 93, 88, 95, 79, 102, 104, 99, 98, 100, 93, 109, 85, 83, 57, 70, 78, 65, 95, 87, 84, 71, 87, 70, 63, 72, 68, 61, 50, 76, 71, 55, 62, 73, 63, 67, 80, 68, 64, 70, 82, 68, 56, 70, 58, 76, 74, 72, 76, 79, 84, 92, 88, 94, 102, 98, 106, 90, 105, 92, 90, 85, 94, 110, 120, 155, 171, 226, 277, 340, 436, 613, 815, 972, 1177, 1268, 1399, 1473, 1536, 1525, 1354, 1238, 1085, 904, 736, 566, 422, 326, 242, 169, 121, 120, 96, 101, 103, 106, 88, 94, 82, 85, 93, 93, 60, 73, 71, 67, 61, 69, 59, 69, 68, 83, 79, 76, 61, 70, 69, 76, 80, 85, 114, 105, 116, 116, 137, 169, 223, 280, 377, 529, 639, 826, 1097, 1282, 1408, 1610, 1756, 1771, 1716, 1643, 1413, 1204, 988, 845, 630, 473, 377, 313, 226, 199, 158, 152, 128, 120, 105, 129, 123, 150, 114, 130, 116, 132, 128, 115, 124, 131, 111, 106, 114, 141, 136, 132, 116, 123, 124, 122, 116, 117, 124, 137, 121, 127, 110, 130, 135, 149, 166, 154, 165, 164, 151, 152, 155, 152, 136, 113, 127, 130, 119, 124, 134, 134, 145, 135, 126, 125, 119, 127, 111, 126, 113, 112, 104, 108, 112, 105, 96, 104, 90, 115, 76, 105, 86, 101, 107, 101, 95, 89, 94, 86, 114, 100, 101, 97, 104, 95, 113, 84, 97, 98, 75, 87, 72, 106, 89, 99, 139, 168, 247, 282, 357, 466, 571, 686, 757, 861, 896, 829, 812, 756, 626, 569, 442, 357, 356, 312, 324, 401, 479, 603, 722, 744, 954, 995, 995, 1020, 882, 865, 743, 553, 488, 385, 308, 241, 162, 141, 128, 120, 106, 115, 127, 128, 136, 108, 128, 124, 119, 117, 116, 105, 96, 110, 92, 99, 98, 83, 80, 95, 80, 79, 83, 83, 96, 83, 91, 84, 89, 79, 87, 100, 78, 102, 127, 112, 131, 144, 190, 240, 268, 376, 495, 593, 714, 785, 838, 832, 888, 799, 668, 641, 543, 437, 359, 289, 244, 207, 163, 179, 199, 146, 156, 159, 167, 164, 181, 205, 231, 302, 361, 477, 591, 797, 942, 1076, 1255, 1304, 1321, 1336, 1315, 1160, 1063, 871, 721, 617, 467, 406, 301, 275, 266, 231, 199, 205, 222, 207, 213, 189, 214, 258, 366, 443, 587, 793, 1032, 1359, 1820, 2173, 2470, 2740, 2899, 2909, 2785, 2394, 2035, 1715, 1339, 1042, 862, 755, 683, 722, 793, 888, 1017, 1233, 1311, 1328, 1264, 1218, 1125, 977, 860, 635, 512, 382, 264, 221, 178, 128, 108, 97, 95, 92, 92, 98, 100, 116, 114, 111, 113, 119, 150, 134, 115, 107, 94, 104, 94, 106, 90, 98, 98, 89, 105, 125, 118, 115, 129, 119, 117, 109, 142, 129, 121, 88, 92, 89, 88, 88, 81, 69, 71, 90, 78, 75, 88, 97, 90, 91, 86, 102, 93, 88, 88, 99, 82, 80, 85, 84, 88, 102, 103, 82, 90, 69, 75, 68, 62, 59, 60, 66, 50, 50, 61, 49, 54, 54, 42, 58, 54, 53, 70, 70, 85, 64, 76, 76, 69, 81, 88, 85, 57, 72, 78, 60, 63, 68, 57, 54, 45, 55, 49, 50, 58, 80, 61, 69, 87, 88, 81, 89, 103, 101, 114, 123, 102, 106, 125, 119, 112, 105, 106, 94, 72, 73, 62, 80, 55, 67, 76, 91, 101, 113, 131, 121, 147, 130, 140, 143, 127, 115, 105, 95, 77, 71, 87, 76, 80, 67, 86, 72, 61, 63, 71, 69, 62, 55, 55, 47, 44, 51, 42, 57, 65, 75, 95, 111, 113, 120, 123, 119, 126, 110, 98, 85, 78, 56, 55, 65, 52, 38, 45, 64, 53, 46, 40, 78, 80, 74, 98, 116, 185, 254, 331, 440, 612, 749, 918, 1137, 1210, 1245, 1382, 1262, 1174, 1037, 851, 723, 573, 432, 286, 203, 165, 158, 107, 117, 129, 114, 127, 127, 116, 133, 128, 115, 102, 94, 91, 75, 73, 64, 65, 62, 60, 88, 65, 64, 66, 98, 92, 76, 79, 102, 100, 93, 106, 113, 103, 114, 99, 96, 95, 93, 90, 73, 62, 65, 60, 33, 58, 55, 59, 49, 53, 57, 39, 80, 68, 48, 55, 47, 74, 73, 44, 60, 50, 49, 39, 46, 46, 53, 48, 63, 50, 59, 32, 32, 39, 58, 61, 58, 59, 66, 69, 72, 83, 101, 122, 108, 116, 124, 156, 179, 174, 232, 283, 346, 409, 519, 548, 624, 664, 627, 576, 508, 447, 391, 299, 225, 198, 138, 122, 99, 71, 66, 79, 73, 66, 69, 57, 68, 48, 56, 48, 66, 50, 50, 52, 58, 80, 86, 99, 108, 132, 136, 170, 172, 189, 229, 219, 261, 308, 368, 476, 625, 928, 1183, 1522, 1840, 1973, 2217, 2388, 2355, 2277, 2076, 1786, 1349, 1034, 886, 652, 461, 320, 226, 183, 146, 131, 101, 119, 111, 91, 103, 107, 106, 103, 98, 125, 112, 109, 122, 131, 114, 106, 113, 109, 110, 118, 92, 98, 95, 81, 91, 84, 95, 79, 75, 86, 82, 88, 106, 112, 87, 68, 101, 110, 111, 120, 179, 253, 354, 493, 675, 905, 1114, 1395, 1659, 1716, 1843, 1915, 1764, 1635, 1345, 1116, 786, 618, 431, 312, 226, 174, 132, 116, 93, 84, 92, 89, 71, 102, 70, 86, 76, 66, 64, 75, 67, 46, 56, 47, 37, 44, 52, 48, 41, 35, 40, 48, 44, 47, 54, 42, 63, 65, 66, 80, 74, 74, 89, 73, 63, 72, 83, 73, 80, 89, 96, 98, 95, 102, 94, 117, 90, 94, 105, 80, 79, 78, 72, 76, 60, 70, 77, 54, 71, 58, 77, 72, 93, 99, 93, 85, 78, 89, 101, 84, 97, 92, 96, 87, 87, 98, 102, 96, 131, 131, 131, 147, 147, 149, 165, 131, 131, 109, 106, 106, 91, 78, 74, 82, 67, 72, 79, 91, 92, 82, 88, 83, 92, 96, 88, 86, 91, 87, 77, 97, 88, 95, 95, 116, 117, 160, 169, 242, 313, 458, 688, 920, 1052, 1281, 1499, 1634, 1572, 1477, 1372, 1228, 999, 832, 659, 498, 379, 268, 227, 189, 159, 146, 107, 112, 92, 78, 71, 49, 62, 40, 53, 40, 45, 47, 60, 35, 62, 41, 63, 30, 45, 63, 65, 52, 76, 79, 86, 131, 127, 193, 304, 367, 475, 628, 770, 793, 879, 889, 928, 821, 719, 662, 484, 384, 312, 233, 195, 179, 148, 120, 121, 136, 104, 95, 89, 81, 57, 60, 63, 53, 52, 77, 72, 93, 83, 98, 100, 116, 114, 115, 103, 91, 97, 113, 79, 107, 83, 93, 79, 90, 81, 76, 90, 76, 76, 85, 84, 90, 98, 125, 99, 106, 92, 117, 107, 97, 68, 67, 74, 65, 60, 75, 70, 77, 85, 113, 96, 118, 111, 130, 100, 125, 107, 118, 115, 128, 110, 113, 107, 108, 105, 74, 95, 81, 68, 83, 95, 66, 72, 57, 57, 47, 50, 26, 51, 44, 44, 31, 39, 25, 47, 41, 40, 57, 37, 73, 65, 77, 77, 88, 85, 92, 97, 101, 133, 100, 107, 103, 107, 104, 114, 92, 101, 86, 78, 87, 83, 77, 86, 103, 108, 88, 84, 93, 80, 90, 118, 105, 99, 92, 97, 93, 102, 107, 90, 95, 90, 90, 108, 139, 125, 149, 152, 161, 166, 158, 144, 138, 131, 115, 117, 115, 102, 99, 114, 121, 103, 126, 115, 132, 135, 121, 125, 133, 136, 137, 147, 137, 159, 189, 252, 365, 506, 765, 951, 1286, 1639, 2059, 2287, 2343, 2363, 2356, 2193, 1896, 1504, 1200, 851, 615, 431, 310, 234, 175, 164, 136, 123, 113, 104, 127, 123, 103, 97, 101, 85, 107, 103, 91, 96, 82, 77, 98, 81, 78, 82, 88, 80, 80, 69, 77, 85, 85, 82, 80, 80, 81, 83, 96, 82, 89, 100, 97, 100, 82, 95, 73, 58, 70, 68, 55, 59, 61, 53, 58, 45, 73, 66, 88, 87, 80, 80, 86, 87, 82, 78, 97, 99, 69, 79, 82, 89, 103, 124, 97, 135, 138, 119, 126, 133, 196, 219, 313, 445, 679, 905, 1057, 1495, 1766, 2051, 2380, 2299, 2379, 2213, 1942, 1688, 1397, 1085, 788, 603, 487, 445, 430, 533, 663, 803, 941, 1136, 1181, 1358, 1300, 1231, 1106, 866, 744, 618, 474, 357, 255, 199, 131, 125, 122, 98, 95, 105, 106, 97, 98, 118, 87, 118, 125, 111, 114, 105, 87, 93, 98, 85, 69, 88, 79, 82, 76, 83, 65, 78, 69, 65, 51, 81, 85, 67, 79, 68, 87, 75, 86, 80, 77, 83, 78, 96, 82, 107, 99, 116, 98, 109, 98, 94, 68, 64, 71, 69, 70, 69, 59, 71, 66, 98, 89, 92, 95, 111, 108, 105, 113, 97, 78, 89, 118, 111, 171, 188, 249, 342, 512, 716, 945, 1242, 1532, 1880, 2029, 2076, 2107, 1981, 1762, 1411, 1172, 910, 692, 528, 398, 224, 193, 133, 105, 101, 87, 68, 93, 98, 73, 94, 73, 69, 87, 58, 72, 55, 58, 47, 51, 40, 39, 39, 45, 34, 51, 47, 51, 44, 39, 55, 55, 69, 61, 39, 56, 58, 52, 65, 65, 71, 62, 71, 58, 61, 57, 60, 39, 40, 49, 29, 36, 49, 51, 37, 40, 33, 43, 63, 49, 68, 67, 89, 86, 95, 109, 102, 100, 93, 82, 89, 72, 91, 64, 88, 103, 120, 150, 223, 297, 388, 496, 624, 740, 827, 930, 959, 861, 773, 699, 576, 444, 379, 282, 204, 159, 124, 107, 106, 130, 99, 103, 82, 71, 75, 65, 83, 74, 65, 72, 54, 58, 55, 59, 39, 59, 37, 44, 41, 38, 39, 30, 35, 41, 54, 49, 53, 40, 62, 69, 72, 59, 75, 79, 84, 66, 86, 67, 71, 56, 53, 45, 58, 50, 46, 44, 50, 48, 45, 48, 37, 47, 49, 50, 62, 56, 52, 60, 77, 61, 68, 45, 48, 66, 50, 52, 51, 39, 52, 66, 60, 59, 67, 57, 65, 69, 59, 66, 83, 81, 63, 74, 89, 79, 61, 78, 73, 61, 79, 69, 101, 110, 154, 205, 280, 386, 500, 611, 772, 882, 924, 950, 923, 877, 819, 712, 597, 449, 362, 300, 229, 179, 139, 115, 100, 97, 88, 109, 95, 93, 86, 84, 93, 81, 67, 66, 73, 73, 64, 40, 46, 43, 48, 47, 36, 30, 59, 56, 49, 44, 49, 53, 37, 52, 38, 41, 49, 40, 47, 53, 57, 83, 88, 83, 83, 93, 78, 84, 73, 80, 75, 78, 95, 113, 102, 92, 104, 119, 107, 119, 94, 112, 91, 87, 86, 71, 68, 80, 74, 87, 75, 87, 92, 99, 89, 76, 106, 96, 97, 118, 138, 132, 165, 195, 234, 327, 397, 496, 567, 691, 756, 794, 843, 818, 733, 621, 512, 416, 338, 271, 218, 194, 171, 146, 135, 116, 125, 107, 103, 106, 90, 87, 89, 117, 111, 123, 152, 199, 257, 365, 490, 600, 790, 982, 1235, 1333, 1419, 1456, 1267, 1177, 1074, 854, 665, 548, 392, 300, 191, 195, 142, 115, 131, 100, 88, 92, 85, 70, 81, 99, 98, 101, 92, 82, 75, 86, 99, 64, 51, 51, 57, 49, 48, 46, 37, 56, 58, 52, 43, 66, 74, 84, 79, 63, 75, 65, 57, 54, 59, 55, 44, 50, 25, 45, 47, 23, 58, 64, 57, 80, 73, 88, 84, 85, 84, 85, 83, 86, 61, 67, 72, 48, 48, 56, 59, 58, 40, 63, 44, 50, 56, 65, 79, 76, 53, 76, 61, 69, 66, 70, 66, 90, 69, 55, 48, 54, 65, 50, 46, 38, 50, 58, 47, 52, 54, 73, 68, 62, 65, 65, 61, 69, 41, 58, 70, 47, 70, 56, 68, 71, 67, 52, 62, 55, 80, 65, 73, 75, 66, 83, 84, 88, 85, 68, 94, 110, 99, 124, 123, 197, 275, 383, 454, 621, 807, 944, 1036, 1225, 1228, 1209, 1065, 950, 860, 622, 488, 412, 329, 242, 195, 126, 123, 118, 92, 90, 78, 61, 67, 49, 62, 64, 67, 52, 50, 43, 73, 84, 74, 84, 88, 113, 106, 106, 114, 111, 128, 136, 115, 133, 143, 147, 206, 221, 314, 427, 514, 656, 762, 855, 885, 941, 922, 867, 713, 602, 466, 361, 275, 199, 180, 109, 81, 80, 65, 57, 63, 59, 54, 57, 60, 64, 72, 76, 75, 82, 95, 67, 89, 70, 68, 75, 63, 51, 45, 47, 68, 57, 60, 85, 62, 100, 117, 179, 262, 316, 460, 634, 789, 966, 1158, 1213, 1340, 1387, 1302, 1131, 983, 792, 645, 500, 345, 244, 183, 143, 121, 105, 91, 111, 86, 113, 99, 105, 105, 97, 105, 78, 66, 87, 71, 52, 63, 60, 89, 94, 89, 135, 135, 115, 136, 125, 124, 123, 86, 93, 81, 67, 48, 65, 78, 36, 37, 63, 47, 62, 64, 82, 80, 81, 82, 82, 88, 79, 86, 68, 71, 68, 59, 55, 49, 62, 65, 65, 50, 76, 71, 77, 98, 109, 139, 204, 283, 321, 419, 505, 619, 672, 728, 701, 683, 647, 508, 458, 370, 271, 230, 142, 108, 98, 75, 95, 72, 82, 83, 79, 49, 47, 59, 46, 71, 68, 43, 51, 48, 40, 62, 59, 69, 82, 93, 98, 147, 128, 125, 153, 149, 146, 109, 80, 71, 75, 70, 65, 57, 74, 65, 70, 77, 87, 100, 104, 107, 105, 106, 104, 82, 84, 69, 94, 64, 91, 84, 75, 108, 122, 139, 144, 122, 143, 158, 164, 179, 220, 273, 347, 389, 508, 661, 731, 784, 850, 782, 748, 658, 553, 471, 401, 313, 243, 174, 143, 106, 77, 61, 50, 37, 29, 37, 34, 36, 40, 44, 33, 29, 34, 44, 37, 48, 27, 39, 28, 27, 40, 34, 47, 42, 48, 50, 56, 54, 52, 51, 39, 43, 43, 59, 32, 30, 32, 45, 47, 59, 94, 68, 88, 112, 126, 112, 134, 125, 120, 93, 88, 79, 61, 68, 53, 56, 62, 56, 60, 64, 69, 79, 72, 75, 52, 45, 38, 45, 54, 39, 22, 35, 35, 33, 41, 33, 52, 34, 42, 59, 67, 69, 71, 60, 82, 62, 61, 57, 31, 42, 35, 44, 32, 46, 54, 65, 66, 64, 77, 77, 60, 73, 71, 68, 55, 36, 56, 51, 48, 43, 44, 45, 56, 44, 60, 51, 45, 59, 47, 52, 70, 72, 55, 78, 69, 78, 83, 78, 82, 58, 60, 70, 67, 55, 66, 40, 58, 33, 66, 58, 75, 56, 64, 53, 70, 70, 71, 65, 78, 77, 59, 65, 76, 70, 56, 74, 75, 70, 60, 68, 61, 69, 73, 55, 82, 80, 85, 75, 74, 83, 84, 90, 87, 105, 93, 81, 90, 87, 111, 87, 95, 84, 89, 73, 74, 73, 74, 53, 66, 61, 77, 73, 99, 82, 118, 125, 131, 126, 126, 141, 136, 149, 115, 130, 127, 120, 98, 101, 104, 88, 94, 91, 65, 91, 108, 122, 160, 210, 295, 348, 457, 565, 658, 680, 744, 786, 796, 718, 600, 510, 447, 386, 292, 291, 279, 339, 393, 463, 637, 791, 960, 1170, 1340, 1381, 1408, 1402, 1292, 1109, 935, 746, 532, 417, 283, 200, 154, 126, 110, 75, 72, 64, 68, 76, 67, 75, 75, 93, 95, 109, 99, 99, 118, 110, 118, 123, 115, 93, 76, 76, 70, 63, 60, 58, 40, 25, 40, 38, 47, 65, 78, 69, 68, 90, 93, 106, 112, 129, 119, 119, 110, 100, 99, 61, 72, 59, 52, 43, 21, 43, 30, 43, 22, 23, 25, 35, 68, 55, 51, 62, 45, 46, 54, 42, 60, 49, 46, 51, 71, 53, 47, 63, 53, 67, 66, 73, 44, 51, 52, 58, 57, 61, 70, 65, 54, 48, 65, 64, 49, 47, 50, 68, 44, 66, 82, 86, 84, 96, 107, 112, 105, 91, 90, 88, 85, 70, 70, 63, 76, 64, 70, 74, 67, 86, 97, 87, 100, 106, 84, 101, 93, 91, 88, 59, 62, 71, 61, 81, 66, 61, 76, 60, 77, 62, 71, 71, 65, 82, 59, 65, 54, 87, 84, 85, 99, 85, 95, 99, 121, 103, 107, 109, 113, 102, 101, 107, 67, 86, 56, 69, 84, 79, 85, 101, 100, 130, 116, 119, 140, 155, 145, 149, 129, 125, 142, 121, 106, 115, 123, 108, 121, 123, 109, 104, 122, 89, 93, 77, 71, 84, 84, 80, 82, 106, 96, 97, 134, 155, 170, 167, 166, 192, 180, 181, 165, 151, 197, 199, 255, 335, 442, 608, 852, 1183, 1500, 1819, 2006, 2323, 2570, 2592, 2490, 2268, 1948, 1617, 1281, 985, 730, 504, 382, 294, 212, 191, 176, 185, 186, 190, 183, 203, 214, 226, 221, 185, 206, 203, 221, 271, 361, 502, 688, 835, 1070, 1298, 1412, 1513, 1545, 1589, 1514, 1394, 1150, 932, 699, 541, 384, 337, 259, 166, 151, 121, 112, 112, 114, 100, 97, 85, 112, 90, 68, 72, 76, 82, 95, 90, 80, 91, 77, 74, 79, 78, 63, 84, 71, 73, 58, 65, 68, 66, 57, 67, 82, 93, 97, 133, 136, 156, 137, 184, 173, 192, 189, 168, 151, 131, 124, 113, 129, 151, 211, 278, 338, 470, 547, 662, 718, 816, 813, 825, 773, 642, 580, 463, 355, 295, 243, 188, 125, 100, 80, 87, 83, 60, 66, 59, 52, 57, 51, 37, 46, 36, 42, 68, 69, 81, 92, 84, 94, 105, 110, 120, 135, 132, 141, 110, 119, 93, 99, 100, 96, 98, 80, 77, 77, 109, 83, 102, 96, 101, 107, 110, 122, 108, 115, 128, 95, 90, 89, 101, 98, 95, 115, 115, 96, 112, 104, 113, 104, 77, 76, 49, 45, 61, 59, 61, 65, 67, 73, 60, 70, 72, 70, 53, 61, 71, 75, 69, 67, 59, 50, 78, 63, 63, 68, 69, 72, 51, 63, 92, 85, 67, 75, 61, 66, 69, 77, 62, 68, 104, 146, 213, 280, 343, 435, 555, 609, 660, 574, 635, 578, 508, 448, 338, 260, 208, 148, 144, 111, 103, 104, 94, 112, 97, 107, 96, 83, 70, 79, 73, 43, 55, 53, 36, 23, 33, 34, 25, 48, 44, 41, 46, 39, 54, 37, 24, 36, 39, 47, 46, 61, 59, 67, 68, 86, 93, 96, 110, 102, 113, 95, 86, 101, 86, 78, 72, 64, 44, 55, 61, 40, 48, 42, 53, 50, 55, 59, 60, 35, 57, 57, 54, 40, 45, 56, 73, 57, 45, 55, 73, 72, 83, 85, 58, 90, 56, 91, 60, 61, 61, 59, 49, 38, 33, 46, 31, 29, 39, 44, 66, 77, 77, 69, 75, 89, 84, 97, 101, 93, 100, 117, 114, 176, 241, 310, 438, 525, 764, 991, 1134, 1351, 1510, 1518, 1612, 1523, 1329, 1172, 937, 743, 609, 461, 320, 264, 171, 147, 118, 108, 105, 93, 99, 99, 87, 101, 106, 103, 120, 139, 81, 88, 91, 76, 96, 71, 67, 91, 66, 80, 59, 71, 55, 63, 42, 70, 77, 65, 75, 59, 59, 65, 62, 63, 64, 78, 60, 41, 59, 44, 55, 63, 78, 68, 71, 87, 81, 105, 113, 117, 128, 117, 151, 123, 122, 108, 93, 104, 99, 141, 99, 166, 268, 335, 464, 554, 677, 826, 920, 1009, 1054, 1022, 918, 909, 748, 571, 473, 383, 299, 219, 203, 158, 137, 126, 120, 129, 132, 117, 126, 131, 110, 128, 124, 111, 163, 186, 253, 368, 490, 658, 788, 1084, 1358, 1485, 1847, 1928, 1935, 2008, 1951, 1664, 1355, 1150, 863, 689, 506, 361, 261, 183, 168, 126, 109, 93, 74, 85, 100, 82, 83, 94, 88, 88, 86, 103, 96, 106, 122, 122, 117, 133, 131, 131, 122, 121, 105, 100, 86, 71, 65, 55, 39, 51, 46, 75, 62, 68, 79, 82, 84, 95, 85, 88, 80, 73, 80, 63, 67, 42, 48, 54, 35, 24, 34, 37, 30, 39, 36, 38, 38, 39, 38, 56, 62, 61, 51, 53, 71, 85, 84, 81, 114, 101, 106, 118, 132, 105, 138, 163, 148, 154, 163, 188, 177, 191, 211, 272, 347, 478, 578, 778, 930, 1173, 1407, 1553, 1594, 1632, 1525, 1331, 1270, 981, 773, 609, 513, 361, 265, 200, 172, 126, 90, 83, 92, 92, 73, 72, 55, 55, 65, 56, 61, 48, 58, 53, 64, 72, 83, 98, 121, 101, 111, 117, 127, 128, 124, 113, 115, 103, 76, 64, 55, 63, 44, 55, 54, 53, 58, 63, 85, 85, 107, 122, 107, 114, 127, 119, 124, 97, 81, 89, 64, 74, 73, 61, 46, 44, 26, 53, 39, 26, 16, 50, 33, 28, 22, 34, 47, 32, 40, 37, 38, 34, 32, 33, 36, 32, 36, 44, 42, 49, 37, 47, 21, 55, 51, 38, 49, 47, 46, 55, 59, 58, 53, 63, 69, 67, 68, 43, 43, 48, 33, 45, 53, 45, 36, 19, 39, 36, 28, 27, 29, 17, 24, 22, 35, 50, 33, 43, 31, 28, 41, 49, 42, 67, 65, 71, 89, 83, 90, 94, 105, 89, 86, 50, 57, 60, 64, 44, 49, 79, 68, 61, 71, 70, 72, 80, 91, 63, 79, 89, 95, 146, 177, 238, 328, 391, 488, 590, 656, 744, 839, 791, 753, 707, 571, 520, 420, 357, 246, 192, 163, 96, 107, 84, 48, 61, 34, 54, 35, 30, 49, 35, 40, 36, 40, 25, 32, 35, 51, 39, 56, 48, 35, 59, 62, 56, 30, 45, 64, 76, 75, 68, 58, 97, 115, 143, 188, 233, 284, 342, 452, 511, 565, 636, 718, 727, 612, 591, 496, 427, 343, 249, 206, 147, 104, 86, 79, 79, 72, 80, 77, 69, 83, 73, 103, 77, 73, 81, 76, 74, 69, 88, 70, 84, 90, 96, 90, 97, 97, 103, 102, 102, 96, 89, 82, 77, 49, 50, 26, 51, 47, 32, 33, 10, 43, 34, 40, 49, 41, 61, 48, 44, 45, 53, 36, 48, 45, 64, 50, 61, 66, 62, 49, 73, 54, 56, 61, 71, 46, 43, 33, 43, 50, 47, 36, 46, 56, 61, 54, 47, 55, 67, 66, 73, 86, 78, 67, 67, 72, 84, 84, 87, 68, 60, 68, 62, 77, 96, 87, 60, 76, 58, 67, 71, 63, 68, 67, 68, 64, 82, 83, 102, 84, 94, 126, 124, 128, 140, 140, 122, 112, 119, 122, 94, 87, 78, 64, 68, 100, 96, 82, 105, 114, 132, 151, 142, 167, 169, 156, 153, 149, 162, 154, 176, 200, 207, 282, 348, 473, 621, 717, 852, 1001, 1074, 1006, 1026, 993, 854, 801, 650, 607, 511, 405, 394, 406, 513, 548, 709, 902, 1074, 1216, 1398, 1520, 1445, 1406, 1355, 1249, 1027, 876, 665, 495, 361, 299, 238, 186, 175, 152, 127, 149, 151, 132, 137, 126, 114, 112, 105, 91, 53, 63, 66, 54, 38, 49, 56, 48, 73, 43, 40, 59, 52, 61, 69, 69, 57, 65, 59, 74, 72, 68, 79, 90, 93, 86, 98, 80, 95, 91, 92, 72, 80, 52, 58, 52, 65, 54, 51, 45, 70, 61, 68, 75, 69, 83, 86, 81, 68, 76, 93, 75, 72, 60, 88, 88, 98, 92, 114, 114, 113, 121, 112, 119, 109, 135, 156, 177, 199, 254, 313, 437, 619, 884, 1074, 1331, 1496, 1836, 2138, 2228, 2326, 2077, 1921, 1647, 1411, 1134, 918, 684, 493, 401, 271, 216, 190, 163, 151, 115, 106, 99, 101, 95, 84, 59, 65, 74, 64, 60, 56, 56, 63, 61, 83, 90, 90, 78, 91, 62, 86, 84, 68, 62, 75, 61, 73, 78, 69, 65, 65, 82, 71, 72, 91, 75, 87, 84, 74, 59, 68, 59, 64, 64, 53, 64, 57, 52, 60, 58, 43, 60, 51, 66, 55, 62, 74, 61, 70, 57, 65, 58, 63, 52, 61, 67, 62, 63, 70, 38, 68, 71, 67, 60, 52, 56, 51, 39, 57, 48, 50, 74, 55, 88, 63, 54, 66, 58, 55, 86, 68, 83, 82, 59, 67, 97, 123, 141, 185, 231, 328, 427, 487, 711, 784, 887, 938, 973, 910, 869, 802, 731, 606, 528, 416, 320, 251, 199, 155, 92, 109, 115, 101, 82, 92, 82, 86, 80, 80, 83, 81, 105, 111, 121, 144, 183, 206, 308, 374, 478, 590, 664, 745, 787, 754, 848, 796, 707, 611, 516, 428, 327, 318, 236, 199, 176, 169, 134, 120, 94, 105, 95, 66, 51, 51, 41, 75, 55, 47, 29, 52, 41, 24, 19, 33, 35, 55, 39, 47, 37, 50, 52, 52, 43, 51, 64, 52, 46, 61, 59, 65, 58, 36, 41, 53, 64, 58, 64, 68, 66, 63, 58, 63, 56, 80, 64, 51, 51, 44, 39, 44, 53, 62, 34, 50, 55, 47, 64, 77, 75, 58, 69, 66, 81, 84, 74, 83, 97, 70, 89, 94, 97, 76, 70, 79, 90, 91, 93, 81, 88, 97, 123, 118, 124, 129, 130, 150, 121, 127, 138, 117, 117, 100, 104, 119, 117, 138, 184, 186, 220, 261, 366, 435, 574, 668, 823, 969, 1052, 1041, 1135, 1072, 960, 886, 771, 663, 567, 485, 466, 405, 513, 606, 701, 905, 1151, 1266, 1507, 1601, 1712, 1675, 1574, 1427, 1417, 1204, 983, 832, 620, 409, 322, 272, 229, 216, 169, 138, 135, 140, 144, 120, 152, 139, 127, 157, 169, 213, 242, 335, 434, 643, 751, 998, 1222, 1482, 1599, 1642, 1827, 1798, 1688, 1553, 1369, 1110, 823, 708, 587, 453, 314, 235, 161, 151, 123, 111, 101, 99, 106, 117, 98, 116, 99, 98, 92, 95, 77, 76, 57, 59, 84, 71, 60, 69, 95, 95, 67, 65, 60, 94, 81, 71, 106, 114, 125, 156, 179, 223, 345, 352, 392, 500, 585, 667, 691, 732, 753, 712, 631, 550, 482, 413, 333, 294, 206, 169, 140, 125, 113, 99, 83, 77, 52, 53, 65, 86, 80, 79, 73, 66, 92, 90, 68, 95, 78, 83, 84, 81, 75, 96, 101, 103, 82, 104, 124, 108, 114, 119, 104, 131, 128, 111, 106, 102, 112, 104, 104, 137, 108, 108, 114, 134, 125, 135, 145, 199, 240, 307, 387, 521, 688, 931, 1093, 1257, 1535, 1840, 1979, 2165, 2172, 2082, 1877, 1629, 1348, 1141, 844, 661, 514, 346, 244, 176, 140, 142, 139, 110, 99, 96, 92, 96, 86, 101, 109, 94, 90, 89, 79, 75, 94, 90, 86, 73, 82, 50, 71, 68, 72, 56, 66, 85, 68, 75, 72, 110, 135, 170, 243, 358, 437, 545, 652, 751, 831, 863, 917, 897, 905, 827, 694, 595, 510, 433, 336, 245, 198, 150, 129, 91, 109, 98, 67, 65, 86, 85, 79, 61, 54, 77, 80, 64, 58, 56, 69, 51, 58, 41, 35, 28, 55, 51, 42, 55, 33, 35, 60, 30, 38, 27, 15, 15, 16, 16, 24, 26, 11, 19, 28, 38, 45, 43, 38, 53, 56, 56, 85, 79, 80, 91, 66, 68, 71, 65, 55, 39, 45, 66, 53, 65, 52, 65, 60, 62, 71, 47, 63, 70, 82, 67, 78, 77, 46, 69, 52, 70, 67, 69, 53, 66, 85, 71, 75, 90, 82, 79, 69, 58, 84, 51, 81, 69, 50, 64, 70, 48, 50, 73, 45, 45, 47, 47, 39, 40, 47, 32, 43, 52, 47, 67, 71, 76, 76, 76, 88, 83, 88, 98, 103, 108, 77, 67, 88, 91, 74, 85, 86, 86, 79, 80, 77, 80, 89, 82, 114, 129, 190, 254, 348, 441, 518, 600, 716, 819, 884, 806, 876, 846, 805, 734, 694, 597, 477, 481, 495, 530, 668, 814, 961, 1218, 1416, 1726, 1797, 2004, 1961, 1947, 1750, 1748, 1521, 1222, 1016, 789, 559, 440, 357, 245, 198, 150, 124, 113, 107, 75, 84, 95, 78, 60, 71, 79, 61, 63, 65, 67, 52, 68, 62, 58, 69, 64, 84, 75, 65, 76, 82, 69, 81, 71, 64, 111, 90, 108, 173, 232, 236, 326, 410, 492, 461, 579, 606, 698, 640, 569, 522, 461, 445, 337, 286, 247, 183, 136, 116, 106, 101, 96, 96, 85, 93, 89, 83, 91, 83, 80, 61, 91, 86, 85, 63, 67, 51, 58, 67, 59, 72, 100, 90, 85, 111, 111, 119, 113, 126, 125, 124, 129, 171, 210, 244, 337, 411, 494, 585, 684, 796, 935, 1079, 1077, 976, 994, 964, 893, 728, 632, 535, 492, 439, 518, 634, 704, 874, 1105, 1376, 1511, 1709, 1941, 1926, 1929, 1759, 1663, 1635, 1350, 1150, 884, 645, 504, 374, 285, 245, 202, 192, 169, 209, 207, 207, 202, 214, 211, 214, 236, 247, 282, 345, 461, 573, 656, 895, 1129, 1297, 1409, 1645, 1796, 1807, 1775, 1687, 1491, 1306, 1183, 965, 745, 569, 428, 329, 219, 191, 127, 118, 130, 110, 93, 92, 93, 106, 76, 70, 76, 50, 66, 54, 70, 75, 37, 42, 47, 39, 36, 48, 37, 51, 39, 50, 64, 42, 49, 54, 84, 73, 57, 39, 47, 65, 66, 62, 74, 68, 64, 57, 65, 71, 59, 84, 85, 102, 109, 106, 156, 183, 247, 339, 402, 523, 611, 706, 728, 869, 851, 911, 881, 824, 738, 665, 560, 494, 398, 316, 250, 205, 172, 176, 169, 162, 191, 198, 162, 176, 178, 148, 161, 166, 165, 170, 235, 309, 395, 496, 612, 829, 1011, 1253, 1448, 1584, 1829, 1806, 1858, 1733, 1569, 1382, 1117, 911, 847, 611, 450, 331, 255, 212, 177, 136, 117, 81, 89, 63, 61, 54, 64, 37, 69, 42, 49, 58, 55, 62, 70, 101, 109, 106, 136, 126, 137, 148, 145, 118, 131, 96, 97, 92, 56, 68, 57, 53, 49, 55, 44, 56, 76, 89, 85, 96, 85, 90, 96, 82, 77, 72, 69, 59, 59, 63, 51, 68, 69, 65, 84, 73, 82, 89, 77, 87, 95, 85, 56, 49, 56, 56, 41, 51, 83, 69, 71, 75, 94, 118, 96, 113, 117, 135, 141, 176, 192, 261, 375, 484, 556, 797, 962, 1301, 1555, 1780, 2090, 2150, 2112, 2116, 1952, 1972, 1710, 1465, 1273, 1040, 780, 548, 407, 314, 243, 185, 164, 133, 95, 114, 121, 120, 121, 130, 93, 107, 94, 108, 82, 74, 94, 75, 62, 57, 65, 65, 56, 62, 61, 60, 65, 58, 45, 44, 56, 48, 51, 37, 48, 52, 44, 54, 58, 34, 44, 61, 45, 69, 71, 69, 71, 95, 82, 75, 71, 105, 137, 169, 206, 252, 311, 441, 479, 521, 547, 607, 684, 686, 640, 596, 512, 460, 371, 311, 254, 187, 171, 152, 124, 111, 91, 96, 67, 75, 73, 75, 50, 71, 69, 60, 72, 68, 51, 53, 59, 52, 53, 51, 45, 65, 47, 62, 62, 55, 48, 51, 52, 55, 55, 67, 55, 53, 46, 49, 38, 52, 37, 35, 35, 48, 40, 37, 45, 24, 33, 44, 34, 31, 31, 57, 44, 62, 50, 49, 41, 35, 44, 22, 36, 18, 17, 19, 35, 16, 17, 26, 19, 32, 33, 20, 25, 30, 36, 40, 24, 42, 45, 39, 54, 81, 59, 62, 72, 85, 54, 70, 46, 58, 56, 48, 51, 28, 45, 46, 52, 39, 53, 32, 45, 34, 41, 36, 55, 39, 51, 38, 41, 44, 33, 43, 43, 38, 49, 47, 54, 49, 49, 65, 69, 56, 68, 72, 58, 69, 66, 78, 82, 78, 80, 81, 65, 72, 72, 63, 63, 51, 68, 61, 60, 71, 67, 67, 59, 40, 75, 66, 60, 69, 62, 59, 51, 33, 24, 47, 24, 32, 14, 2, 9, 17, 3, 29, 14, -2, 27, 7, 18, 23, 17, 20, 37, 41, 28, 34, 56, 34, 38, 27, 31, 39, 41, 48, 53, 52, 29, 37, 22, 24, 28, 27, 45, 36, 32, 35, 42, 22, 19, 29, 31, 28, 24, 30, 42, 33, 31, 35, 42, 47, 50, 49, 56, 47, 46, 77, 58, 58, 40, 41, 32, 25, 15, 29, 28, 41, 49, 32, 46, 37, 45, 47, 50, 36, 54, 49, 44, 32, 18, 34, 27, 41, 17, 29, 14, 39, 40, 40, 34, 46, 54, 45, 50, 50, 51, 50, 57, 54, 58, 48, 63, 67, 61, 50, 37, 25, 21, 32, 30, 41, 29, 57, 44, 38, 48, 43, 53, 59, 43, 50, 41, 40, 23, 23, 22, 40, 49, 22, 58, 51, 58, 38, 41, 56, 40, 42, 34, 43, 53, 53, 40, 57, 68, 40, 69, 57, 64, 50, 81, 92, 81, 109, 90, 89, 97, 118, 90, 94, 107, 81, 94, 81, 80, 89, 75, 60, 60, 53, 55, 60, 71, 84, 71, 68, 85, 79, 94, 109, 121, 149, 174, 213, 283, 325, 391, 448, 489, 617, 673, 689, 731, 737, 688, 618, 536, 493, 392, 329, 294, 311, 295, 334, 375, 481, 544, 608, 760, 869, 889, 917, 921, 924, 824, 779, 746, 592, 453, 354, 274, 259, 186, 123, 121, 115, 104, 87, 90, 87, 75, 71, 49, 60, 67, 61, 52, 57, 62, 65, 48, 65, 43, 49, 43, 51, 46, 55, 33, 34, 65, 76, 80, 79, 115, 150, 194, 248, 303, 431, 481, 585, 667, 797, 865, 935, 994, 1002, 826, 838, 772, 649, 579, 499, 386, 311, 250, 174, 144, 126, 106, 76, 59, 81, 59, 52, 48, 30, 41, 35, 35, 38, 46, 30, 33, 19, 23, 20, 36, 37, 18, 32, 44, 35, 33, 39, 43, 46, 56, 54, 54, 40, 41, 42, 60, 57, 68, 81, 62, 89, 80, 73, 78, 107, 92, 74, 60, 73, 68, 88, 88, 77, 81, 63, 81, 52, 56, 51, 53, 47, 38, 45, 53, 47, 45, 40, 55, 69, 50, 51, 39, 38, 35, 46, 55, 42, 49, 68, 45, 42, 39, 51, 26, 30, 19, 32, 21, 29, 32, 26, 32, 26, 34, 39, 26, 42, 50, 47, 49, 71, 66, 93, 74, 86, 81, 82, 95, 96, 86, 86, 97, 98, 80, 80, 74, 79, 76, 72, 84, 75, 63, 62, 70, 69, 68, 61, 79, 59, 71, 83, 70, 56, 64, 71, 69, 60, 59, 76, 93, 101, 114, 154, 189, 257, 335, 413, 469, 543, 622, 752, 743, 762, 793, 735, 664, 625, 545, 457, 330, 267, 208, 183, 160, 112, 95, 104, 94, 94, 83, 97, 109, 95, 101, 100, 91, 95, 96, 92, 86, 63, 75, 73, 76, 53, 50, 46, 49, 46, 44, 43, 46, 40, 63, 51, 44, 34, 40, 58, 42, 48, 42, 48, 37, 43, 68, 51, 15, 44, 48, 49, 38, 43, 22, 47, 18, 15, 26, 26, 40, 44, 29, 35, 48, 49, 36, 31, 59, 49, 49, 50, 54, 64, 53, 62, 61, 52, 57, 45, 41, 44, 42, 49, 57, 57, 51, 49, 32, 47, 52, 49, 38, 73, 62, 62, 103, 93, 80, 82, 100, 111, 116, 114, 126, 120, 131, 173, 191, 225, 283, 306, 396, 517, 769, 1012, 1166, 1516, 1727, 1896, 2047, 2153, 2192, 2176, 2035, 1839, 1652, 1471, 1210, 943, 807, 682, 515, 382, 360, 256, 216, 200, 181, 146, 162, 127, 100, 100, 106, 82, 71, 87, 74, 57, 59, 73, 80, 65, 59, 72, 61, 54, 53, 49, 54, 49, 38, 36, 44, 51, 53, 53, 60, 58, 63, 81, 79, 81, 91, 69, 91, 83, 60, 63, 43, 33, 47, 32, 31, 25, 23, 22, 27, 23, 32, 30, 19, 14, 15, 26, 35, 48, 51, 57, 37, 61, 62, 52, 69, 58, 51, 57, 50, 55, 25, 24, 42, 36, 38, 44, 44, 43, 55, 52, 44, 57, 50, 56, 67, 71, 56, 79, 70, 65, 69, 69, 75, 61, 66, 84, 77, 83, 82, 109, 155, 186, 237, 267, 362, 454, 564, 627, 680, 723, 796, 790, 795, 711, 622, 576, 549, 429, 344, 299, 217, 184, 143, 110, 99, 97, 90, 93, 78, 104, 78, 87, 96, 108, 125, 139, 184, 210, 295, 334, 390, 496, 510, 579, 690, 779, 795, 804, 794, 781, 676, 630, 572, 479, 404, 314, 274, 186, 166, 141, 116, 99, 110, 112, 98, 94, 99, 96, 92, 90, 92, 89, 63, 72, 56, 44, 47, 36, 46, 38, 43, 34, 24, 44, 35, 36, 19, 40, 42, 36, 42, 35, 27, 19, 21, 23, 16, 23, 10, 15, 18, 39, 24, 17, 21, 21, 30, 27, 30, 18, 32, 30, 22, 20, 25, 31, 25, 31, 31, 15, 30, 19, 40, 44, 44, 36, 60, 47, 55, 53, 52, 59, 58, 57, 74, 83, 95, 91, 93, 112, 108, 94, 112, 110, 96, 101, 102, 85, 87, 81, 88, 67, 53, 64, 46, 57, 70, 62, 60, 65, 69, 83, 82, 104, 140, 147, 198, 220, 265, 308, 360, 447, 511, 592, 615, 627, 672, 678, 609, 585, 518, 413, 427, 348, 301, 274, 221, 163, 138, 123, 109, 89, 100, 85, 74, 84, 63, 51, 64, 49, 53, 38, 54, 41, 41, 28, 34, 29, 20, 28, 30, 36, 41, 53, 44, 65, 60, 51, 57, 76, 84, 79, 96, 72, 99, 71, 98, 115, 111, 99, 108, 110, 110, 116, 126, 141, 137, 145, 200, 235, 283, 299, 429, 536, 666, 738, 812, 984, 1015, 1094, 1082, 1082, 1000, 960, 810, 740, 637, 487, 381, 320, 257, 215, 166, 132, 101, 95, 84, 89, 83, 53, 36, 52, 57, 63, 61, 43, 37, 40, 27, 38, 22, 57, 53, 45, 64, 53, 47, 57, 57, 54, 53, 66, 48, 56, 68, 84, 47, 78, 60, 59, 66, 69, 70, 66, 84, 107, 94, 101, 88, 106, 119, 119, 138, 158, 207, 274, 275, 296, 395, 445, 479, 494, 563, 590, 536, 493, 448, 360, 338, 273, 210, 161, 149, 122, 123, 90, 73, 65, 81, 68, 73, 70, 63, 84, 91, 94, 104, 98, 86, 79, 103, 82, 56, 65, 53, 65, 58, 35, 29, 50, 23, 22, 21, 25, 39, 47, 37, 47, 24, 38, 38, 29, 37, 28, 45, 39, 42, 45, 35, 22, 27, 30, 22, 34, 36, 33, 14, 29, 31, 18, 23, 34, 27, 40, 27, 45, 39, 50, 41, 39, 37, 23, 34, 35, 29, 32, 33, 23, 16, 25, 23, 36, 36, 37, 37, 26, 29, 39, 60, 53, 49, 53, 45, 30, 36, 54, 37, 68, 64, 47, 60, 65, 69, 67, 58, 74, 88, 91, 103, 111, 142, 160, 178, 243, 293, 359, 476, 502, 629, 689, 758, 778, 776, 867, 776, 742, 617, 603, 497, 405, 320, 269, 197, 138, 103, 100, 100, 78, 40, 31, 38, 29, 48, 45, 29, 26, 48, 51, 51, 51, 56, 58, 72, 57, 44, 62, 48, 39, 22, 43, 32, 22, 24, 34, 35, 36, 34, 38, 33, 31, 24, 53, 40, 60, 57, 60, 88, 101, 97, 87, 94, 101, 112, 107, 115, 93, 106, 93, 90, 69, 57, 58, 60, 53, 58, 59, 63, 64, 55, 58, 58, 79, 71, 68, 58, 63, 53, 68, 57, 40, 34, 23, 43, 33, 33, 35, 30, 25, 41, 27, 36, 30, 41, 35, 28, 27, 37, 35, 39, 30, 40, 36, 32, 32, 35, 46, 43, 28, 44, 31, 33, 37, 34, 48, 56, 37, 31, 46, 41, 37, 33, 48, 46, 29, 43, 43, 19, 38, 39, 25, 26, 20, 10, 26, 22, 35, 21, 38, 32, 46, 49, 66, 46, 43, 36, 46, 48, 62, 58, 75, 76, 81, 96, 81, 84, 88, 85, 81, 89, 71, 55, 59, 68, 59, 49, 51, 44, 55, 72, 51, 50, 63, 47, 48, 63, 62, 53, 37, 44, 53, 57, 56, 37, 36, 65, 55, 45, 42, 70, 69, 88, 73, 81, 87, 83, 66, 96, 118, 123, 138, 242, 308, 402, 552, 670, 837, 959, 1105, 1237, 1370, 1399, 1417, 1436, 1270, 1202, 1118, 981, 763, 640, 513, 422, 337, 261, 226, 195, 151, 108, 89, 102, 90, 79, 47, 56, 55, 44, 69, 50, 46, 51, 42, 42, 56, 36, 52, 50, 53, 45, 64, 72, 83, 62, 59, 77, 58, 55, 59, 50, 59, 48, 49, 44, 41, 39, 49, 46, 52, 55, 73, 43, 62, 48, 66, 65, 61, 66, 55, 50, 52, 35, 29, 60, 39, 40, 51, 45, 25, 35, 73, 62, 67, 81, 90, 118, 133, 193, 212, 265, 337, 406, 393, 396, 451, 455, 476, 475, 439, 407, 341, 302, 242, 216, 176, 141, 97, 82, 77, 69, 80, 62, 83, 60, 69, 89, 97, 119, 104, 102, 99, 116, 104, 94, 87, 75, 68, 65, 55, 67, 60, 48, 46, 39, 46, 53, 65, 68, 67, 66, 85, 87, 78, 74, 75, 83, 84, 93, 91, 88, 114, 120, 124, 151, 146, 163, 178, 176, 207, 249, 312, 358, 456, 505, 704, 845, 1000, 1067, 1366, 1368, 1706, 1723, 1675, 1649, 1598, 1499, 1364, 1096, 898, 751, 652, 502, 431, 312, 259, 204, 160, 121, 88, 80, 72, 76, 83, 68, 69, 71, 75, 67, 85, 61, 73, 60, 77, 63, 58, 69, 33, 53, 52, 56, 51, 51, 56, 51, 27, 65, 51, 49, 65, 55, 53, 61, 60, 44, 35, 63, 47, 49, 56, 63, 75, 80, 75, 87, 112, 109, 164, 207, 230, 267, 351, 424, 520, 596, 573, 636, 755, 753, 747, 714, 671, 621, 560, 445, 423, 347, 281, 222, 166, 160, 101, 80, 78, 49, 62, 57, 49, 64, 53, 69, 59, 62, 50, 46, 53, 53, 54, 57, 64, 66, 65, 79, 72, 69, 78, 86, 83, 81, 85, 90, 95, 82, 94, 79, 83, 77, 75, 79, 81, 89, 90, 98, 99, 98, 139, 136, 132, 155, 164, 234, 293, 347, 412, 603, 713, 929, 1103, 1234, 1468, 1677, 1759, 1643, 1643, 1688, 1775, 1526, 1439, 1319, 1040, 887, 742, 619, 498, 381, 291, 256, 204, 185, 145, 144, 158, 171, 151, 129, 136, 147, 199, 252, 235, 315, 401, 487, 529, 733, 828, 970, 1089, 1110, 1139, 1179, 1254, 1199, 1124, 1016, 860, 775, 636, 518, 445, 330, 241, 202, 177, 171, 112, 102, 82, 83, 85, 100, 83, 67, 66, 81, 75, 73, 75, 58, 42, 56, 75, 48, 57, 62, 54, 50, 48, 61, 58, 59, 62, 51, 59, 53, 53, 51, 48, 61, 62, 49, 46, 58, 51, 54, 43, 39, 43, 56, 55, 59, 63, 73, 70, 66, 71, 58, 40, 46, 56, 39, 45, 34, 51, 48, 30, 43, 50, 48, 50, 53, 39, 51, 37, 62, 60, 45, 59, 72, 41, 63, 49, 47, 54, 76, 64, 89, 73, 114, 115, 133, 178, 205, 247, 257, 322, 334, 409, 430, 476, 522, 513, 532, 478, 489, 490, 460, 420, 384, 324, 310, 315, 280, 378, 452, 496, 592, 609, 669, 698, 769, 850, 754, 794, 820, 729, 619, 584, 482, 442, 342, 273, 215, 202, 146, 144, 110, 116, 78, 76, 45, 65, 69, 55, 44, 50, 42, 24, 19, 34, 36, 22, 31, 32, 36, 32, 57, 28, 29, 44, 32, 27, 32, 36, 36, 24, 28, 42, 39, 28, 22, 23, 22, 16, 23, 17, 28, 10, 9, 21, 45, 31, 39, 35, 23, 34, 34, 56, 43, 41, 53, 62, 64, 79, 64, 64, 56, 62, 61, 84, 86, 90, 139, 202, 264, 296, 330, 436, 488, 523, 629, 694, 700, 686, 667, 625, 598, 551, 503, 404, 381, 303, 260, 195, 146, 101, 81, 67, 69, 51, 58, 66, 62, 38, 39, 28, 36, 54, 30, 20, 18, 8, 34, 34, 11, 8, 9, 34, 15, 27, 17, 15, 25, 14, 14, 15, -1, 34, 33, 9, 29, 18, 17, 13, 28, 2, 12, 18, 20, 46, 27, 34, 31, 29, 24, 19, 36, 34, 34, 45, 41, 37, 36, 52, 37, 26, 31, 41, 32, 25, 36, 39, 46, 15, 30, 28, 26, 28, 18, 43, 28, 27, 57, 52, 37, 32, 43, 27, 43, 35, 41, 20, 35, 45, 44, 45, 29, 38, 37, 37, 57, 54, 58, 65, 78, 62, 38, 61, 63, 68, 83, 55, 72, 59, 73, 84, 69, 64, 68, 82, 74, 82, 95, 127, 136, 129, 174, 224, 265, 344, 454, 535, 731, 813, 1031, 1082, 1146, 1321, 1377, 1520, 1385, 1312, 1306, 1212, 1038, 913, 781, 635, 557, 409, 305, 281, 222, 171, 150, 120, 103, 97, 95, 103, 100, 72, 58, 88, 75, 78, 32, 56, 52, 47, 49, 94, 82, 81, 86, 101, 112, 99, 139, 130, 133, 135, 151, 156, 167, 191, 193, 225, 302, 363, 386, 453, 506, 522, 547, 592, 589, 588, 575, 559, 458, 447, 396, 335, 295, 236, 215, 185, 149, 142, 115, 110, 97, 115, 108, 111, 117, 118, 133, 150, 159, 193, 266, 302, 390, 471, 619, 738, 805, 922, 1022, 1117, 1190, 1175, 1197, 1126, 1009, 961, 836, 743, 656, 558, 455, 347, 271, 222, 196, 149, 113, 94, 65, 59, 54, 56, 62, 50, 65, 54, 32, 47, 34, 46, 55, 28, 37, 32, 43, 52, 44, 32, 36, 44, 37, 44, 39, 45, 34, 38, 55, 65, 57, 57, 60, 82, 84, 110, 100, 107, 134, 111, 123, 129, 132, 171, 191, 184, 230, 288, 348, 467, 562, 637, 770, 913, 1061, 1153, 1204, 1240, 1279, 1381, 1354, 1206, 1081, 966, 877, 733, 572, 500, 427, 360, 286, 218, 160, 168, 122, 111, 89, 79, 62, 75, 60, 64, 34, 34, 19, 53, 46, 36, 52, 63, 61, 60, 50, 73, 75, 73, 69, 70, 67, 58, 54, 37, 43, 58, 54, 61, 48, 65, 52, 66, 84, 78, 94, 101, 84, 69, 78, 82, 71, 82, 69, 44, 52, 41, 54, 37, 40, 41, 23, 29, 40, 18, 32, 21, 26, 8, 23, 47, 36, 23, 34, 27, 15, 34, 23, 43, 43, 43, 38, 54, 52, 67, 76, 78, 72, 83, 86, 112, 97, 139, 160, 180, 232, 242, 319, 369, 392, 472, 461, 498, 477, 513, 477, 485, 430, 415, 327, 318, 280, 229, 178, 152, 130, 126, 101, 99, 86, 90, 61, 80, 61, 66, 55, 51, 37, 42, 38, 27, 17, 39, 34, 19, 52, 44, 34, 27, 19, 27, 35, 34, 35, 30, 21, 19, 42, 32, 43, 23, 17, 21, 34, 19, 32, 12, 18, 12, -1, 19, 9, 18, 16, 20, 13, 24, 26, 37, 21, 25, 17, 22, 25, 16, 13, 17, 25, 38, 28, 30, 35, 32, 18, 37, 43, 23, 34, 34, 21, 29, 33, 15, 13, 27, 19, 34, 48, 30, 31, 40, 7, 36, 22, 24, 26, 17, 45, 27, 40, 29, 33, 34, 39, 29, 38, 41, 26, 6, 27, 28, 28, 26, 46, 29, 46, 39, 36, 59, 33, 51, 31, 38, 42, 16, 14, 22, 32, 23, 17, 35, 19, 38, 30, 35, 34, 44, 24, 44, 35, 50, 39, 53, 47, 40, 47, 59, 47, 40, 38, 62, 32, 49, 52, 47, 36, 48, 23, 55, 51, 54, 38, 35, 37, 58, 35, 45, 49, 35, 46, 31, 26, 36, 40, 38, 21, 18, 20, 14, 29, 25, 28, 22, 21, 17, 38, 19, 24, 23, 21, 17, 24, 17, 35, 27, 15, 21, 19, 16, 15, 31, 13, 14, 40, 20, 41, 35, 29, 46, 35, 37, 35, 37, 34, 54, 39, 31, 39, 40, 45, 15, 10, 23, 24, 30, 12, 1, 18, 33, 7, 11, 3, 12, 15, 18, 15, 7, 20, 23, 49, 43, 31, 62, 50, 47, 48, 32, 35, 56, 46, 44, 33, 41, 49, 36, 35, 44, 41, 42, 48, 28, 32, 38, 45, 49, 34, 24, 42, 35, 39, 31, 34, 37, 31, 21, 41, 32, 23, 25, 29, 15, 19, 44, 15, 28, 30, 24, 33, 20, 21, 21, 20, 11, 13, 25, 29, 36, 27, 40, 42, 36, 49, 54, 67, 66, 63, 71, 85, 129, 154, 170, 241, 296, 349, 489, 550, 631, 772, 845, 887, 1075, 1129, 1103, 1154, 1172, 1135, 969, 881, 798, 728, 578, 477, 365, 334, 253, 187, 160, 146, 82, 80, 60, 59, 66, 41, 39, 64, 45, 51, 37, 29, 41, 63, 67, 45, 42, 42, 67, 46, 61, 42, 47, 55, 50, 66, 75, 27, 40, 37, 25, 46, 19, 15, 20, 40, 51, 29, 52, 50, 35, 50, 41, 54, 56, 22, 39, 55, 33, 38, 22, 26, 34, 51, 50, 32, 26, 32, 23, 14, 43, 58, 66, 56, 71, 47, 55, 61, 66, 60, 87, 78, 71, 60, 65, 67, 63, 72, 75, 76, 82, 98, 102, 135, 173, 160, 224, 295, 346, 401, 437, 527, 530, 601, 610, 577, 535, 543, 509, 477, 431, 364, 314, 237, 197, 148, 123, 105, 92, 72, 82, 81, 76, 78, 81, 68, 58, 62, 44, 46, 42, 43, 35, 39, 47, 36, 42, 41, 37, 41, 40, 49, 36, 44, 51, 50, 17, 37, 48, 23, 39, 42, 21, 38, 16, 12, 26, 21, 14, 23, 14, 8, 15, 22, 12, 13, 12, 4, 14, 18, 15, 5, 12, 17, 21, 11, 15, 13, 37, 10, 18, 14, 28, 29, 17, 17, 25, 34, 37, 34, 34, 32, 54, 42, 36, 48, 38, 54, 43, 40, 43, 43, 37, 26, 42, 33, 38, 40, 54, 33, 33, 34, 23, 38, 27, 59, 43, 30, 47, 29, 34, 41, 34, 34, 29, 37, 36, 36, 26, 26, 37, 36, 39, 41, 60, 62, 69, 86, 88, 101, 118, 116, 154, 150, 192, 220, 235, 296, 319, 366, 371, 395, 447, 482, 561, 499, 543, 535, 540, 489, 453, 440, 456, 428, 532, 519, 628, 705, 863, 841, 890, 966, 1147, 1132, 1162, 1026, 1020, 1027, 937, 770, 692, 623, 562, 487, 376, 263, 213, 189, 154, 112, 98, 67, 75, 75, 66, 65, 59, 42, 33, 28, 39, 49, 42, 35, 52, 31, 50, 61, 57, 72, 71, 66, 78, 73, 86, 61, 52, 41, 49, 50, 59, 33, 46, 20, 24, 38, 21, 25, 25, 22, 38, 13, 14, 18, 16, 21, 12, 17, 1, 24, 4, 20, 25, 41, 15, 30, 70, 41, 36, 46, 49, 41, 51, 55, 53, 55, 33, 20, 37, 27, 24, 28, 35, 40, 44, 36, 54, 39, 44, 40, 55, 40, 57, 53, 38, 33, 44, 47, 48, 27, 36, 36, 20, 11, 46, 39, 13, 16, 18, 26, 12, 19, 1, 18, 30, 27, 30, 35, 45, 43, 40, 26, 30, 50, 52, 64, 89, 89, 108, 130, 166, 196, 227, 236, 323, 359, 401, 454, 498, 500, 531, 505, 505, 461, 462, 450, 374, 341, 266, 231, 175, 173, 143, 121, 99, 83, 79, 56, 37, 45, 38, 39, 46, 41, 36, 35, 35, 33, 38, 33, 34, 44, 49, 64, 75, 59, 75, 71, 93, 123, 128, 154, 196, 252, 269, 306, 403, 456, 601, 658, 680, 729, 832, 920, 902, 879, 828, 835, 773, 801, 692, 560, 502, 431, 363, 303, 256, 193, 154, 123, 99, 110, 84, 90, 97, 58, 45, 48, 36, 55, 26, 21, 53, 43, 34, 49, 36, 66, 56, 45, 41, 37, 41, 57, 50, 58, 35, 44, 47, 72, 66, 65, 55, 79, 79, 75, 102, 68, 89, 104, 87, 88, 95, 109, 78, 83, 86, 76, 56, 52, 56, 50, 58, 35, 36, 27, 39, 34, 34, 27, 37, 41, 22, 35, 34, 24, 18, 16, 42, 32, 30, 20, 27, 19, 45, 44, 15, 38, 54, 47, 24, 30, 34, 50, 38, 36, 33, 29, 27, 21, 19, 30, 35, 28, 36, 30, 26, 41, 42, 27, 44, 27, 46, 47, 41, 44, 46, 60, 56, 48, 60, 66, 61, 49, 58, 62, 85, 93, 96, 109, 130, 161, 249, 299, 350, 366, 398, 494, 587, 566, 660, 640, 630, 663, 658, 615, 615, 596, 536, 517, 493, 522, 552, 614, 649, 705, 796, 939, 982, 1001, 988, 988, 1054, 966, 943, 843, 799, 770, 663, 675, 713, 669, 600, 589, 700, 797, 781, 843, 832, 873, 866, 808, 798, 752, 689, 664, 533, 483, 421, 352, 267, 198, 211, 151, 126, 109, 71, 78, 59, 38, 46, 33, 47, 44, 40, 57, 41, 41, 42, 40, 37, 54, 68, 73, 60, 44, 68, 82, 75, 93, 124, 167, 192, 187, 289, 334, 380, 458, 600, 697, 798, 893, 916, 1075, 1099, 1140, 1100, 1118, 1027, 904, 787, 746, 696, 589, 495, 427, 335, 276, 253, 176, 164, 143, 126, 96, 115, 84, 68, 58, 71, 75, 66, 68, 64, 74, 68, 55, 60, 68, 49, 54, 69, 57, 52, 62, 43, 54, 46, 35, 38, 48, 41, 31, 33, 36, 16, 30, 11, 16, 27, 30, 22, 16, 22, 33, 30, 16, 27, 28, 22, 23, 43, 37, 27, 31, 54, 39, 42, 62, 60, 44, 43, 54, 68, 91, 99, 156, 177, 235, 287, 355, 403, 482, 568, 685, 675, 685, 769, 727, 772, 793, 714, 643, 594, 497, 468, 386, 328, 290, 222, 179, 153, 129, 106, 96, 60, 83, 56, 57, 49, 34, 55, 36, 37, 41, 37, 22, 22, 31, 28, 31, 29, 21, 25, 31, 33, 36, 25, 22, 24, 25, 22, 22, 24, 20, 20, 37, 12, 34, 20, 12, 14, 30, 21, 13, 14, 12, 17, 7, 24, 9, 6, 32, 30, 38, 23, 15, 18, 5, 17, 17, 7, 15, 23, 18, 21, 24, 22, 37, 20, 19, 13, -1, 13, 0, 14, 37, 19, 19, 9, 25, 17, 36, 17, 22, 38, 30, 39, 40, 56, 40, 55, 41, 61, 66, 54, 48, 48, 57, 51, 28, 23, 29, 31, 31, 32, 22, 19, 20, 11, 23, 14, 38, 22, 18, 28, 16, 34, 28, 29, 13, 6, 0, 22, 36, 24, 11, 25, 18, 25, 38, 42, 39, 22, 29, 31, 46, 42, 48, 34, 33, 38, 42, 38, 47, 46, 47, 36, 47, 49, 51, 54, 56, 92, 85, 98, 88, 84, 120, 174, 185, 233, 264, 344, 379, 411, 428, 492, 504, 575, 592, 543, 484, 486, 449, 465, 442, 417, 382, 448, 416, 390, 433, 480, 509, 518, 571, 586, 621, 621, 609, 549, 629, 638, 570, 551, 549, 551, 468, 485, 514, 554, 566, 621, 617, 648, 688, 734, 736, 703, 709, 661, 660, 644, 538, 478, 391, 373, 326, 282, 233, 146, 132, 137, 90, 84, 65, 58, 58, 62, 50, 51, 65, 44, 32, 31, 38, 33, 42, 50, 42, 40, 45, 53, 52, 77, 79, 74, 67, 72, 88, 68, 82, 96, 63, 78, 73, 68, 73, 56, 59, 65, 52, 35, 53, 38, 53, 49, 41, 36, 42, 33, 34, 34, 22, 29, 15, 20, 24, 17, 23, 8, 0, -3, 16, 22, 21, 8, 30, 7, 6, 11, -3, 14, 12, 21, 1, 25, 0, 28, -2, 22, 19, 18, 2, 12, 12, 25, 29, 10, 8, 3, 7, 9, 26, 16, 15, 13, 21, 20, 13, 16, 14, 16, 32, 23, 4, 15, 8, 22, 21, 21, 13, 3, 16, 27, 13, 17, 5, 13, -2, 13, 0, 12, 11, 7, 11, 8, 7, 19, 10, 15, 5, 17, 16, 16, 13, 20, 7, -3, 15, 5, 4, 10, 10, 8, 5, 2, 15, 1, 9, 15, -1, -1, 9, 2, 3, 9, 17, 21, 30, 30, 40, 27, 27, 36, 45, 45, 31, 29, 40, 54, 64, 71, 63, 87, 94, 107, 95, 126, 176, 204, 260, 277, 315, 347, 376, 410, 375, 429, 432, 430, 420, 367, 322, 324, 260, 225, 198, 213, 141, 124, 130, 100, 78, 94, 66, 69, 65, 67, 79, 77, 77, 73, 94, 125, 146, 182, 199, 208, 255, 301, 323, 321, 376, 333, 393, 390, 370, 349, 322, 306, 247, 217, 204, 169, 155, 108, 95, 98, 76, 87, 75, 59, 74, 62, 46, 43, 56, 65, 61, 43, 38, 39, 38, 45, 34, 29, 29, 34, 20, 17, 13, 35, 37, 25, 16, 33, 30, 19, 47, 36, 34, 35, 24, 51, 41, 32, 37, 23, 36, 25, 22, 37, 34, 36, 32, 43, 44, 42, 42, 51, 66, 73, 61, 48, 84, 79, 91, 94, 103, 133, 145, 167, 214, 247, 266, 265, 313, 420, 453, 459, 520, 560, 626, 643, 610, 672, 683, 647, 635, 638, 579, 628, 641, 706, 743, 802, 847, 947, 1008, 1029, 1049, 1063, 1216, 1169, 1189, 1118, 1007, 976, 875, 854, 755, 591, 531, 425, 393, 328, 259, 212, 159, 133, 125, 107, 94, 95, 84, 70, 62, 78, 61, 55, 51, 52, 43, 59, 44, 60, 49, 31, 42, 56, 67, 69, 57, 42, 47, 70, 49, 69, 50, 72, 66, 72, 74, 86, 80, 108, 90, 126, 132, 109, 143, 174, 201, 278, 288, 374, 434, 483, 594, 655, 762, 930, 987, 1021, 1095, 1106, 1050, 1113, 1091, 1040, 938, 850, 749, 724, 643, 543, 435, 353, 307, 278, 214, 181, 158, 153, 134, 116, 90, 93, 76, 69, 78, 70, 69, 61, 67, 69, 61, 68, 76, 64, 75, 57, 71, 92, 83, 92, 83, 97, 130, 147, 168, 201, 233, 282, 357, 389, 487, 439, 503, 600, 602, 548, 515, 618, 600, 536, 504, 460, 424, 341, 340, 296, 216, 206, 166, 122, 111, 84, 63, 60, 54, 35, 30, 30, 29, 39, 42, 41, 35, 40, 40, 53, 40, 52, 53, 53, 28, 42, 65, 73, 80, 88, 106, 99, 122, 125, 172, 208, 228, 266, 322, 310, 358, 367, 417, 415, 467, 450, 405, 411, 391, 395, 333, 312, 273, 221, 169, 165, 126, 98, 99, 79, 59, 65, 36, 30, 44, 33, 43, 30, 21, 49, 19, 25, 16, 38, 23, 26, 21, 24, 40, 48, 46, 30, 25, 48, 54, 53, 70, 66, 74, 77, 73, 72, 75, 68, 65, 63, 86, 71, 67, 84, 60, 56, 64, 38, 36, 42, 55, 51, 38, 44, 27, 44, 54, 64, 91, 70, 57, 74, 76, 108, 125, 133, 144, 203, 218, 259, 306, 402, 495, 534, 590, 660, 640, 730, 762, 845, 801, 774, 718, 669, 689, 649, 582, 481, 382, 353, 276, 277, 240, 173, 137, 121, 125, 87, 81, 83, 89, 70, 43, 71, 66, 77, 60, 56, 59, 55, 53, 59, 53, 63, 62, 60, 58, 69, 48, 53, 54, 49, 44, 32, 36, 55, 56, 54, 67, 64, 62, 53, 66, 80, 107, 97, 89, 111, 153, 162, 177, 208, 243, 291, 352, 354, 441, 432, 492, 485, 492, 487, 542, 575, 616, 549, 529, 555, 505, 539, 588, 626, 664, 659, 710, 744, 784, 800, 902, 907, 922, 972, 854, 853, 839, 820, 773, 644, 603, 540, 430, 395, 302, 251, 198, 185, 164, 113, 98, 85, 68, 64, 42, 21, 31, 35, 50, 43, 34, 46, 44, 34, 37, 38, 55, 69, 44, 60, 56, 61, 41, 39, 27, 40, 47, 46, 66, 36, 71, 50, 37, 39, 23, 39, 37, 25, 34, 40, 46, 49, 46, 46, 54, 72, 53, 73, 70, 80, 78, 105, 108, 124, 160, 183, 227, 237, 316, 355, 430, 456, 563, 645, 708, 854, 994, 1025, 1052, 1121, 1255, 1215, 1267, 1280, 1202, 1088, 1042, 900, 810, 702, 636, 551, 464, 393, 305, 285, 218, 162, 124, 126, 115, 101, 58, 64, 67, 42, 46, 37, 21, 12, 15, 32, 38, 13, 6, 22, 29, 15, 32, 32, 36, 23, 19, 4, 12, 11, 0, 14, 31, 33, 11, 24, 14, 19, 26, 22, 9, 13, 11, 13, 41, 36, 30, 29, 46, 28, 48, 50, 50, 63, 48, 70, 49, 52, 78, 71, 83, 82, 126, 149, 138, 194, 260, 286, 352, 398, 428, 514, 602, 668, 756, 902, 930, 912, 993, 982, 939, 896, 829, 787, 698, 642, 541, 448, 426, 340, 307, 260, 194, 160, 125, 87, 85, 66, 63, 40, 48, 48, 43, 23, 28, 24, 21, 27, 29, 14, 28, 16, 22, 3, 3, 14, 0, 26, 16, 16, 15, 7, 16, -3, 13, 12, 36, 35, 38, 22, 35, 31, 48, 49, 56, 77, 88, 106, 121, 160, 161, 173, 217, 245, 273, 281, 343, 374, 390, 386, 377, 359, 384, 343, 332, 310, 289, 291, 249, 223, 173, 162, 165, 172, 133, 134, 117, 124, 126, 126, 128, 142, 171, 164, 155, 185, 240, 243, 288, 326, 333, 348, 353, 385, 394, 441, 422, 401, 363, 344, 335, 300, 322, 250, 233, 226, 189, 140, 118, 126, 131, 96, 92, 99, 100, 106, 108, 125, 144, 168, 197, 234, 264, 277, 327, 393, 457, 490, 534, 535, 538, 513, 558, 553, 491, 485, 505, 453, 410, 334, 317, 300, 255, 188, 152, 143, 117, 128, 87, 92, 62, 89, 96, 97, 81, 87, 84, 78, 84, 74, 96, 59, 61, 67, 72, 57, 53, 48, 40, 52, 53, 50, 64, 72, 60, 50, 58, 42, 45, 49, 40, 50, 37, 32, 28, 29, 34, 41, 26, 17, 24, 31, 21, 32, 44, 35, 39, 39, 49, 53, 45, 41, 64, 53, 56, 100, 110, 142, 172, 187, 203, 232, 254, 392, 357, 411, 483, 505, 545, 559, 541, 547, 608, 559, 523, 477, 461, 430, 435, 429, 444, 476, 451, 480, 457, 544, 596, 593, 617, 623, 588, 580, 500, 542, 534, 450, 395, 324, 310, 250, 195, 194, 170, 143, 118, 81, 73, 62, 64, 39, 54, 37, 41, 6, 9, 22, 20, 18, 16, 11, 14, 12, 14, 4, 1, 14, -2, -4, 9, 8, 7, 11, 13, 2, 23, 17, 32, 27, 53, 39, 47, 41, 53, 57, 58, 49, 62, 65, 62, 52, 50, 43, 52, 29, 43, 52, 23, 15, 10, 6, 7, 10, 14, 16, 38, 20, 20, 52, 31, 42, 46, 57, 65, 70, 76, 86, 91, 101, 105, 115, 105, 121, 103, 152, 159, 148, 201, 187, 212, 259, 254, 257, 301, 321, 339, 344, 355, 304, 311, 345, 328, 274, 266, 228, 244, 182, 161, 138, 148, 105, 107, 81, 72, 61, 55, 44, 38, 25, 35, 44, 43, 33, 34, 39, 38, 26, 45, 56, 60, 64, 72, 48, 62, 75, 110, 146, 141, 190, 226, 232, 320, 324, 370, 479, 478, 555, 645, 659, 700, 740, 769, 815, 756, 788, 729, 779, 659, 579, 551, 494, 430, 357, 305, 259, 254, 221, 184, 160, 163, 129, 108, 100, 67, 74, 86, 77, 63, 65, 39, 51, 52, 50, 36, 24, 29, 36, 42, 63, 58, 63, 68, 86, 103, 128, 138, 152, 191, 215, 245, 249, 261, 303, 302, 277, 279, 339, 315, 275, 278, 261, 259, 233, 191, 157, 115, 104, 102, 97, 54, 34, 40, 59, 47, 49, 41, 39, 43, 32, 15, 6, 17, 24, 12, 8, 6, 22, 9, 27, 17, 17, 26, 32, 25, 23, 19, 36, 47, 58, 62, 49, 52, 77, 80, 64, 71, 58, 78, 78, 64, 61, 57, 65, 63, 46, 51, 45, 50, 34, 49, 34, 43, 50, 39, 27, 56, 45, 34, 54, 36, 50, 56, 67, 52, 66, 58, 68, 76, 66, 51, 76, 78, 94, 130, 148, 180, 207, 258, 300, 343, 399, 523, 557, 736, 827, 875, 989, 1033, 1037, 1105, 1096, 1152, 1141, 1138, 1053, 975, 876, 867, 787, 639, 564, 461, 406, 356, 304, 238, 183, 145, 135, 117, 103, 73, 70, 56, 55, 27, 43, 38, 43, 45, 19, 36, 36, 36, 38, 31, 34, 27, 32, 33, 29, 14, 27, 26, 25, 6, 11, 15, 18, 13, 17, 8, 10, 11, 38, 16, 7, 20, 33, 19, 16, 25, 5, 42, 26, 31, 35, 19, 29, 28, 26, 38, 35, 55, 49, 28, 37, 39, 40, 31, 32, 50, 32, 46, 40, 50, 65, 64, 63, 74, 99, 102, 118, 159, 179, 195, 199, 253, 282, 291, 365, 422, 436, 497, 547, 610, 561, 600, 589, 675, 675, 618, 633, 626, 652, 626, 659, 650, 654, 674, 692, 714, 846, 816, 862, 962, 918, 909, 854, 905, 845, 823, 770, 768, 662, 662, 639, 625, 615, 675, 604, 523, 593, 623, 601, 576, 578, 550, 511, 518, 428, 335, 358, 312, 294, 270, 203, 170, 133, 109, 91, 72, 66, 71, 57, 66, 44, 41, 36, 38, 32, 52, 37, 32, 46, 34, 28, 36, 32, 33, 36, 30, 20, 37, 22, 28, 10, 39, 29, 29, 33, 24, 29, 29, 29, 37, 50, 46, 48, 41, 77, 52, 56, 65, 58, 69, 68, 49, 66, 63, 65, 68, 34, 44, 42, 49, 31, 32, 36, 45, 45, 42, 48, 34, 25, 36, 29, 32, 43, 32, 35, 38, 64, 35, 21, 29, 33, 42, 45, 20, 17, 21, 19, 30, 30, 18, 27, 25, 29, 29, 25, 27, 22, 31, 24, 39, 28, 38, 51, 36, 41, 52, 58, 68, 83, 60, 52, 59, 51, 65, 47, 37, 15, 29, 23, 20, 26, 20, 33, 22, 20, 39, 30, 35, 46, 44, 37, 37, 69, 48, 55, 61, 67, 64, 61, 86, 58, 60, 75, 88, 70, 87, 81, 107, 128, 112, 136, 139, 166, 193, 207, 234, 284, 322, 352, 372, 339, 345, 391, 428, 386, 367, 325, 336, 287, 293, 277, 232, 225, 178, 141, 123, 106, 102, 72, 70, 67, 48, 40, 48, 24, 37, 21, 25, 26, 12, 23, 28, 14, 15, 30, 44, 15, 19, 21, 48, 74, 76, 110, 84, 107, 125, 178, 200, 207, 240, 267, 257, 256, 275, 360, 358, 327, 329, 367, 338, 343, 316, 269, 234, 187, 176, 159, 125, 105, 79, 81, 70, 62, 56, 37, 44, 36, 51, 50, 48, 43, 48, 43, 48, 32, 37, 26, 27, 38, 36, 62, 58, 67, 87, 64, 100, 114, 117, 171, 185, 179, 262, 234, 320, 351, 391, 419, 382, 408, 438, 457, 451, 446, 453, 412, 390, 323, 325, 277, 236, 228, 189, 161, 119, 110, 103, 66, 82, 61, 72, 79, 47, 42, 33, 28, 48, 49, 24, 41, 33, 26, 52, 50, 52, 52, 48, 78, 73, 90, 103, 111, 120, 169, 180, 169, 202, 229, 291, 304, 318, 362, 429, 402, 393, 438, 530, 515, 534, 549, 537, 537, 576, 660, 703, 724, 784, 821, 841, 874, 875, 921, 972, 944, 949, 925, 876, 763, 733, 678, 568, 540, 505, 469, 345, 332, 281, 233, 207, 207, 158, 133, 109, 129, 131, 146, 152, 179, 179, 195, 231, 262, 251, 256, 245, 310, 349, 330, 363, 316, 295, 284, 303, 255, 273, 260, 237, 187, 154, 150, 169, 113, 103, 112, 77, 96, 91, 79, 87, 77, 74, 78, 50, 58, 60, 59, 53, 55, 65, 48, 71, 62, 71, 102, 83, 125, 137, 151, 157, 190, 224, 259, 304, 302, 312, 350, 381, 440, 463, 530, 535, 553, 563, 551, 555, 597, 670, 662, 757, 834, 824, 983, 986, 1076, 930, 1134, 1106, 1090, 1200, 1064, 975, 1029, 974, 895, 798, 734, 687, 572, 531, 461, 358, 308, 252, 195, 164, 158, 119, 70, 82, 64, 49, 68, 50, 46, 33, 32, 31, 31, 34, 28, 26, 24, 25, 27, 26, 13, 18, 15, 5, 3, 13, 18, 19, 25, 33, 24, 22, 26, 22, 19, 42, 25, 33, 16, 39, 33, 30, 47, 50, 50, 47, 57, 82, 86, 84, 106, 110, 122, 119, 153, 204, 163, 246, 285, 299, 293, 331, 407, 428, 457, 451, 519, 512, 493, 558, 499, 514, 581, 597, 627, 634, 657, 731, 786, 805, 831, 867, 899, 912, 847, 964, 936, 889, 861, 821, 743, 669, 636, 563, 525, 479, 438, 359, 323, 272, 252, 220, 219, 203, 157, 199, 224, 245, 289, 334, 313, 372, 495, 523, 527, 574, 655, 667, 717, 745, 691, 755, 736, 676, 724, 645, 588, 511, 434, 434, 380, 334, 305, 321, 225, 211, 219, 167, 168, 147, 158, 151, 164, 177, 166, 209, 202, 202, 216, 266, 290, 309, 291, 320, 370, 363, 353, 356, 342, 316, 316, 293, 279, 251, 257, 221, 198, 175, 147, 123, 102, 100, 76, 52, 46, 41, 33, 25, 25, 27, 30, 37, 31, 10, 8, 8, 8, 20, 40, 22, 17, 23, 21, 35, 13, 28, 21, 6, 1, 7, 22, 3, 14, 18, 23, 19, 12, 14, 15, 30, 23, 17, 29, 40, 51, 48, 40, 54, 57, 38, 48, 79, 52, 35, 53, 61, 35, 54, 47, 39, 22, 30, 40, 46, 28, 27, 26, 20, 34, 38, 28, 34, 25, 21, 20, 14, 27, 14, 23, 10, 2, 19, 21, 7, 10, -10, 6, -7, -3, 0, -7, -8, 10, -4, 2, 8, 1, 6, -1, -2, 10, 17, 5, -10, -1, 9, 1, 3, -5, -8, 4, -14, 10, 12, 8, 7, 2, -6, 4, 0, -2, 11, -9, 12, 14, 9, 7, 16, 4, 9, 0, 10, 5, 6, 16, 7, 4, 21, 19, 15, 18, 15, 3, 10, 13, 5, 5, 12, -3, 18, 0, 0, 0, 10, 10, -11, -12, -13, -9, 7, -5, -8, -11, -11, -7, 7, -3, 11, -9, -14, -10, -13, -9, -18, 5, -2, -14, -13, -14, -15, -11, -6, -2, 3, 4, -2, 0, 8, 1, 3, 7, -2, -5, 3, 1, -6, 6, 7, 12, -3, -1, 24, 14, 19, 33, 22, 20, 33, 39, 40, 38, 56, 64, 67, 71, 94, 105, 110, 161, 166, 171, 206, 215, 237, 248, 298, 296, 265, 284, 284, 251, 250, 244, 208, 240, 201, 209, 174, 185, 118, 112, 87, 93, 61, 68, 61, 53, 37, 35, 31, 46, 20, 25, 23, 20, 2, 7, 39, 31, 29, 30, 8, 27, 19, 33, 32, 29, 43, 37, 27, 20, 36, 26, 31, 36, 32, 28, 8, 13, 25, 31, 32, 28, 6, 12, 9, 3, 10, 3, 27, 20, 17, 12, 21, 25, 8, 12, 18, 19, 21, 22, 19, 25, 39, 31, 35, 42, 29, 35, 35, 31, 39, 24, 12, 30, 11, 11, 9, 17, 0, 6, 3, 6, 19, -1, 5, -1, -1, -4, -4, -5, -14, 0, 1, 8, 0, 7, 0, 3, 6, 9, 10, 6, 3, 12, 12, 22, 15, 15, 26, 17, 32, -1, 3, 10, 32, 11, 16, 7, 7, 9, 12, 9, 2, 4, 5, 18, 10, 16, 4, 10, 13, 3, 0, 1, 4, 15, 5, 3, 10, 10, 7, 15, 16, 11, 2, 5, 27, 3, 5, 3, -5, 7, 2, 16, 13, 5, 16, 13, 14, -6, 7, 1, 9, 0, 4, 3, 2, 1, 8, 2, 10, 10, 12, 17, 20, 23, 9, 15, 6, 14, 1, 26, 17, 22, 34, 13, 16, 26, 34, 28, 6, 14, 17, 13, 23, 18, 28, 27, 37, 27, 28, 43, 43, 29, 37, 29, 46, 53, 45, 51, 52, 46, 58, 58, 57, 50, 62, 60, 73, 88, 63, 82, 87, 79, 121, 176, 186, 225, 268, 289, 327, 303, 360, 438, 516, 601, 640, 605, 708, 814, 920, 946, 931, 897, 913, 844, 811, 747, 682, 612, 534, 493, 515, 476, 414, 377, 299, 282, 268, 246, 224, 226, 229, 211, 244, 231, 287, 313, 374, 391, 397, 460, 472, 528, 561, 542, 595, 576, 649, 608, 591, 548, 523, 458, 483, 435, 349, 313, 268, 214, 216, 191, 140, 111, 102, 84, 83, 75, 62, 81, 74, 82, 43, 53, 39, 51, 65, 41, 43, 49, 31, 43, 39, 62, 44, 69, 66, 58, 42, 68, 72, 74, 62, 70, 64, 66, 85, 89, 106, 104, 128, 126, 137, 150, 165, 210, 201, 200, 216, 256, 269, 292, 259, 245, 285, 290, 244, 216, 205, 233, 207, 182, 181, 149, 132, 128, 125, 117, 98, 90, 105, 94, 65, 66, 70, 83, 64, 59, 42, 65, 43, 52, 45, 35, 39, 46, 60, 39, 42, 43, 46, 43, 54, 52, 41, 38, 46, 56, 52, 30, 35, 25, 27, 26, 37, 27, 15, 31, 30, 21, 17, 14, 7, 16, 28, 8, 10, 7, 15, 8, 8, 13, 7, 3, 2, 13, 13, 4, 23, 10, 7, 14, -3, 1, 15, 4, 8, -1, 8, 23, 8, 18, 3, -5, -15, -4, 10, 4, 9, 12, 11, 13, 6, 10, 0, 18, 18, -2, 13, 19, 17, 32, 29, 21, 32, 26, 34, 20, 23, 37, 48, 39, 38, 47, 56, 54, 50, 84, 70, 104, 121, 69, 72, 74, 81, 97, 108, 121, 127, 143, 195, 215, 260, 295, 353, 418, 444, 484, 510, 605, 616, 646, 664, 713, 774, 688, 770, 828, 828, 781, 689, 687, 619, 575, 483, 435, 411, 359, 314, 329, 248, 218, 182, 181, 144, 127, 107, 106, 81, 68, 63, 64, 56, 55, 42, 56, 64, 52, 59, 34, 37, 27, 42, 53, 62, 45, 56, 47, 49, 45, 58, 53, 71, 74, 58, 49, 107, 110, 107, 122, 160, 179, 193, 208, 219, 246, 305, 371, 330, 363, 388, 405, 364, 373, 368, 316, 317, 338, 339, 317, 293, 272, 230, 226, 206, 170, 155, 143, 121, 137, 140, 114, 145, 146, 150, 158, 203, 211, 228, 257, 275, 312, 331, 335, 332, 365, 383, 364, 347, 328, 389, 362, 335, 306, 288, 271, 256, 252, 242, 228, 187, 187, 221, 212, 246, 277, 288, 331, 382, 408, 455, 515, 541, 623, 635, 690, 827, 782, 867, 896, 973, 922, 899, 928, 967, 918, 937, 852, 778, 676, 704, 777, 728, 727, 644, 696, 642, 653, 557, 527, 564, 545, 496, 421, 449, 418, 362, 328, 268, 255, 259, 210, 171, 169, 160, 107, 98, 83, 80, 68, 59, 56, 34, 26, 43, 18, 24, 61, 39, 27, 44, 41, 33, 43, 41, 39, 44, 38, 44, 62, 60, 67, 85, 110, 113, 93, 141, 142, 212, 237, 234, 298, 400, 389, 404, 524, 569, 605, 708, 777, 817, 827, 848, 847, 796, 873, 876, 843, 829, 731, 722, 658, 644, 539, 484, 427, 383, 343, 304, 253, 209, 165, 144, 106, 106, 102, 76, 84, 73, 58, 57, 45, 54, 46, 53, 84, 107, 95, 104, 131, 142, 171, 162, 168, 191, 285, 293, 317, 353, 407, 410, 460, 524, 545, 592, 544, 538, 586, 614, 556, 573, 565, 471, 408, 383, 398, 378, 321, 267, 256, 221, 193, 137, 120, 117, 113, 92, 67, 57, 66, 52, 54, 38, 26, 35, 41, 47, 58, 35, 48, 73, 59, 83, 74, 54, 86, 120, 152, 131, 144, 168, 185, 232, 226, 208, 225, 246, 237, 204, 237, 230, 216, 208, 197, 191, 159, 165, 149, 121, 88, 85, 86, 60, 52, 63, 50, 39, 23, 31, 27, 26, 42, 38, 44, 51, 46, 47, 26, 38, 17, 36, 41, 28, 34, 44, 16, 26, 33, 30, 31, 18, 16, 29, 22, 24, 3, 13, 11, 5, 24, 17, 10, 10, -2, 3, 4, 7, 7, 1, 13, 4, 4, 3, 12, -1, 3, 36, 26, 13, 19, -1, -4, 2, 4, 10, 2, 8, 21, -5, -9, 0, 10, -10, 16, 10, -6, 18, 7, 5, 4, -2, -12, 4, 4, 2, 15, 0, -4, -8, -10, -3, 7, 9, 0, 4, -24, -8, 0, -4, -8, -9, 4, -8, -9, 6, -10, 5, -1, -5, -13, 10, -12, -10, 6, -2, -1, 10, 7, -4, 0, 4, 6, -10, 2, 4, -8, -3, 8, 6, -12, 3, -1, 4, 11, 21, 22, 11, 4, 17, 3, 6, 0, 10, 5, 5, 22, 14, 15, -6, 16, 10, 21, 17, 15, 12, 4, 9, 14, 19, 3, 13, 3, 22, 8, 12, 24, 17, 20, 8, -5, 15, 10, 34, 6, 15, 7, 13, 37, 36, 45, 58, 94, 77, 79, 112, 93, 125, 141, 157, 187, 161, 196, 251, 262, 279, 258, 279, 271, 274, 273, 276, 292, 305, 260, 256, 220, 234, 198, 169, 127, 123, 102, 102, 93, 56, 46, 32, 45, 30, 24, 16, 17, 13, -2, 13, 7, 16, -3, -8, 0, 1, 10, 4, 13, -1, -6, 2, 3, 7, -5, -16, -3, -2, 0, -21, -6, -4, 0, 4, -12, 1, 2, -8, -9, -1, -5, -21, -7, -5, -11, -19, 0, 0, 4, 4, -10, 0, 4, 3, -4, -1, -1, -6, 6, 14, -1, 3, 6, 17, 1, 6, 5, -3, 3, 19, -8, -7, 3, 0, 0, 1, 0, 2, 0, 0, 5, 2, 11, 3, 9, 17, 9, -4, 10, -3, 6, 0, 9, 0, 10, 22, 7, 15, 2, 6, 1, -3, -3, -4, -11, 22, 10, 7, 8, 9, 4, 16, 25, 25, 0, 12, 23, 22, 23, 25, 24, 34, 27, 26, 30, 34, 31, 43, 42, 53, 60, 55, 87, 107, 104, 140, 150, 183, 222, 197, 242, 258, 249, 271, 297, 360, 375, 362, 340, 344, 375, 381, 358, 321, 333, 304, 294, 262, 217, 220, 165, 139, 124, 117, 91, 105, 101, 62, 73, 58, 40, 35, 31, 27, 27, 11, 4, -5, 6, 2, 13, 9, 16, 0, 9, 4, 4, 9, 2, 6, 14, 19, 11, 8, 7, 27, 14, 13, 10, 9, 20, 10, 13, 15, 24, 30, 27, 19, 33, 12, 21, 24, 27, 17, 16, 21, 17, 23, 10, 26, 32, 19, 19, 27, 36, 27, 33, 25, 37, 43, 79, 77, 94, 84, 96, 116, 135, 148, 175, 215, 198, 217, 241, 253, 250, 278, 275, 300, 253, 251, 274, 265, 237, 197, 217, 173, 162, 150, 124, 139, 126, 84, 87, 81, 59, 44, 45, 50, 60, 57, 18, 24, 31, 29, 27, 20, 19, 12, 16, 5, 9, 19, 17, 7, 7, 11, 11, 3, -9, 0, 21, 11, 17, 9, 15, 18, 13, 31, 20, 14, 41, 29, 41, 49, 52, 56, 50, 68, 97, 107, 85, 104, 117, 173, 182, 205, 202, 216, 247, 328, 308, 290, 386, 462, 513, 511, 552, 574, 541, 505, 520, 535, 504, 471, 440, 430, 378, 318, 294, 279, 244, 214, 188, 182, 146, 125, 118, 84, 83, 71, 52, 25, 21, 13, 13, 25, 19, 7, 16, 19, 14, 10, 20, 15, 8, 10, 20, 29, 30, 36, 37, 35, 36, 24, 42, 44, 48, 59, 55, 99, 115, 156, 134, 176, 174, 197, 217, 254, 254, 265, 312, 341, 338, 381, 362, 350, 373, 373, 345, 308, 319, 300, 321, 280, 248, 266, 247, 220, 195, 192, 186, 219, 226, 239, 238, 266, 291, 301, 344, 352, 341, 385, 451, 510, 518, 486, 478, 553, 516, 565, 522, 435, 453, 499, 458, 413, 380, 332, 293, 289, 232, 232, 186, 189, 148, 125, 118, 110, 81, 81, 74, 58, 48, 48, 49, 27, 24, 36, 20, 32, 16, 29, 30, 13, 17, 31, 9, 7, 15, 22, 10, 16, 20, 8, 7, 3, -2, -5, 6, 12, -3, -5, -21, 3, -2, -3, -5, -14, 7, -11, -22, -22, 3, 6, 0, -1, -4, 5, 4, 21, 7, 14, 31, 15, 42, 32, 55, 46, 37, 40, 72, 64, 89, 82, 118, 138, 133, 149, 193, 179, 180, 166, 182, 193, 186, 200, 185, 165, 179, 159, 166, 159, 136, 101, 83, 94, 96, 83, 77, 60, 69, 63, 50, 44, 23, 15, 35, 34, 34, 40, 15, 9, 26, 21, 26, 19, 36, 29, 22, 32, 19, 30, 31, 31, 33, 36, 43, 69, 67, 78, 86, 117, 101, 108, 142, 142, 165, 210, 210, 225, 251, 278, 302, 332, 362, 344, 381, 397, 405, 406, 434, 458, 426, 398, 372, 350, 347, 296, 278, 252, 232, 197, 190, 157, 133, 134, 105, 101, 85, 95, 67, 49, 43, 58, 48, 43, 70, 58, 65, 59, 74, 97, 83, 89, 121, 187, 166, 179, 208, 201, 229, 251, 300, 293, 286, 319, 306, 302, 355, 335, 326, 293, 273, 294, 283, 235, 222, 205, 190, 191, 142, 145, 120, 107, 77, 80, 50, 48, 42, 19, 29, 38, 28, 22, 14, 28, 12, 13, 12, 17, 9, -1, 0, 15, 1, 14, 6, 9, 15, 23, 25, 34, 34, 50, 55, 57, 65, 68, 68, 106, 130, 146, 140, 195, 165, 186, 218, 194, 258, 255, 260, 270, 299, 319, 273, 274, 257, 254, 269, 239, 217, 186, 164, 166, 151, 125, 120, 102, 90, 71, 75, 56, 43, 36, 52, 37, 20, 24, 24, 17, 23, 36, 7, 22, 26, 15, 7, -3, -2, 0, 2, 3, 2, 0, -6, -13, -5, -3, -5, -18, -10, -2, -3, -2, -5, -7, -17, -14, -19, -8, -11, -11, -14, -11, 5, 0, -3, 22, 8, 0, 10, 1, 10, 11, 4, 13, 5, 20, 26, 19, 23, 24, 16, 0, 0, 2, 12, 12, 21, 36, 17, 21, 1, 25, 10, 11, 9, 4, 3, 9, -2, -5, 8, 14, 22, 9, 16, 8, 0, 2, 8, 13, 1, 18, 24, -11, 12, 2, 9, -5, -2, 0, -7, 15, 1, 3, -8, -2, -2, -8, -2, -2, 10, 7, 0, 7, 4, 9, 10, -3, 8, 17, 10, 11, 8, -1, 16, 1, 2, 13, 8, 10, 0, 8, 18, 6, 4, 6, 13, 16, 25, 17, 17, 30, 21, 15, 10, 22, 31, 29, 51, 66, 63, 88, 67, 99, 119, 122, 150, 178, 198, 210, 237, 232, 339, 348, 313, 374, 352, 398, 452, 467, 432, 469, 537, 520, 445, 446, 509, 531, 565, 553, 527, 551, 479, 498, 504, 493, 512, 507, 441, 456, 421, 407, 364, 365, 346, 299, 278, 227, 189, 163, 171, 147, 104, 106, 105, 72, 66, 53, 49, 50, 29, 23, 21, 15, -2, 16, 4, -4, 8, 5, 3, 16, 14, 27, 40, 13, 14, 27, 50, 34))
        self.assertEqual(record.annotations["abif_raw"]['DATA2'], (-22, -2, 6, 0, -3, -11, -10, -9, -17, -11, -1, 3, -21, -2, 17, -4, 6, 3, 2, 4, -1, -24, -14, -20, -4, -10, 12, -6, 5, 5, 0, -5, 2, -8, -3, -9, 8, -6, 1, 6, -5, 0, 4, -1, 4, 0, 2, -4, -14, -3, -10, -16, -11, -22, -14, -12, 0, -18, -11, 6, -5, 2, -2, -5, -8, 7, -8, -3, -5, -8, -2, 15, 6, -14, -5, -8, 3, 1, -17, -11, -25, -4, -12, 1, -16, -4, 3, -12, -5, -16, -8, -8, 1, -9, 14, -16, -10, -22, 5, 2, -9, -8, -4, -10, 8, 0, -1, 5, 0, 7, 1, -7, -6, 2, 14, -12, 4, -4, -12, 0, -4, -6, -12, 3, 4, -9, 4, 0, 0, -5, 12, 0, -13, -4, -10, -3, -4, 3, 8, -3, 16, 2, 0, -7, 10, 0, 2, 0, 13, -5, 7, 9, 4, -23, 6, -1, 8, 3, 1, -2, -2, 13, 4, 8, 2, 1, 11, 8, 13, 22, 6, 24, -2, 2, 19, -2, 29, 42, 34, 13, 22, 14, 20, 1, 15, 6, 16, 12, 3, 16, -5, 20, 18, 26, 8, 18, 7, 16, 10, 10, 10, 9, 22, 20, 8, 14, 21, 18, 0, 21, 23, 4, 14, 7, 17, 19, 21, 31, 10, 6, 23, 20, 6, 5, 17, 19, 17, 15, -4, 24, 2, 23, 2, 4, 12, 3, 4, 0, 16, 15, 12, 10, 22, 10, 4, 7, 15, 13, 40, 14, 18, 19, 8, 22, 24, 9, 26, 15, 26, 19, 17, 23, 29, 13, 13, 13, 4, 2, 26, 9, 8, 7, 14, 18, 22, 18, 27, 13, 16, 36, 11, 20, 28, 36, 17, 24, 27, 9, 12, 5, 19, 16, 20, 13, -2, 15, 16, 34, 16, 8, 25, 33, 20, 7, 9, 10, 26, 27, 30, 4, 10, 17, 14, 3, 17, 22, 8, 7, 11, 3, 18, 27, 31, 25, 24, 18, 10, 21, 15, 16, 25, 23, 12, 28, 6, 28, 18, 27, 12, 27, 13, 15, 24, 4, 21, 26, 12, 25, 14, 14, 20, 18, 21, 20, 28, 19, 15, 14, 24, 4, 10, 18, 16, 21, 25, 12, 15, 12, 9, 23, 14, 22, 39, 25, 13, 24, 4, 13, 21, 37, 12, 0, 36, 2, 33, 15, 6, 29, 35, 28, 21, 25, 18, 18, 34, 2, 23, 9, 6, 9, 31, 20, 18, 29, 14, 26, 13, 16, 26, 26, 23, 21, 16, 28, 17, 21, 16, 25, 26, 29, 28, 25, 16, 11, 25, 25, 31, 34, 15, 22, 32, 23, 15, 15, 44, 16, 18, 22, 22, 31, 25, 31, 30, -1, 31, 27, 32, 20, 15, 27, 22, 31, 6, 18, 17, 17, 22, 40, 32, 24, 16, 16, 16, 12, 14, 6, 38, 6, 26, 18, 20, 10, 13, 3, 26, 20, 18, 19, 12, 29, 32, 16, 16, 35, 15, 29, 21, 22, 30, 10, 14, 25, 21, 21, 14, 23, 22, 21, 27, 30, 29, 16, 36, 17, 15, 39, 20, 28, 41, 13, 28, 29, 14, 11, 15, 12, 15, 23, 19, 19, 20, 33, 22, 22, 26, 33, 15, 17, 19, 25, 28, 28, 18, 26, 13, 24, 24, 33, 27, 21, 32, 5, 25, 15, 18, 26, 21, 22, 26, 25, 15, 10, 20, 14, 21, 19, 22, 27, 20, 20, 14, 26, 39, 24, 23, 3, 15, 23, 22, 16, 5, 10, 6, 17, 40, 19, 29, 28, 40, 26, 42, 16, 31, 32, 15, 10, 4, 7, 12, 15, 19, 14, 18, 12, 15, 22, 16, 26, 22, 33, 17, 20, 21, 19, 9, 27, 9, 18, 9, 26, 23, 16, 23, 25, 28, -5, 18, 18, 13, 26, 18, 47, 16, 24, 14, 7, 38, 40, 17, 8, 16, 19, -1, 6, 13, 9, 12, 25, 9, 3, 14, 16, 17, 35, 10, 20, 23, 18, 6, 22, 16, 22, 16, 10, 12, 25, 11, 29, 17, 14, 26, 22, 30, 10, 12, 23, 13, 18, 14, 7, 16, 15, 9, 20, 26, 20, 20, 19, 14, 21, 14, 23, 32, 23, 14, -4, 34, 32, 13, 13, -3, 1, 24, -5, 19, 15, 32, 15, 6, 17, 12, 17, 15, 16, 12, -5, 22, 4, 24, 26, 24, 28, 2, 15, 8, 0, 10, 17, 10, 15, 21, 23, 26, 5, 3, 6, 16, 20, 5, 17, 13, 10, 13, 3, 9, 5, 19, 5, 13, 23, 0, 9, 12, 27, 0, 12, 12, -8, 13, 18, 22, 19, 19, 12, 20, -8, 24, 17, 0, 16, 19, -3, 23, -3, 10, -5, 7, 15, 17, 11, 19, 16, 16, 11, 22, 17, 13, 19, 17, 0, 24, 1, 23, 4, 13, 9, 20, 17, 23, 8, 25, 23, 35, 35, 39, 36, 40, 45, 42, 41, 53, 47, 46, 55, 70, 73, 90, 79, 114, 123, 141, 149, 162, 174, 190, 225, 212, 183, 173, 167, 135, 133, 98, 92, 86, 80, 70, 53, 44, 48, 44, 42, 48, 40, 41, 37, 43, 38, 28, 34, 37, 26, 32, 22, 23, 32, 25, 31, 36, 23, 27, 25, 40, 29, 30, 46, 36, 42, 25, 32, 38, 45, 26, 48, 45, 30, 50, 38, 51, 41, 42, 50, 52, 44, 31, 37, 45, 49, 41, 41, 55, 31, 37, 35, 32, 51, 53, 43, 45, 24, 34, 33, 41, 36, 46, 37, 25, 46, 19, 38, 38, 41, 35, 39, 27, 34, 29, 28, 35, 33, 29, 39, 25, 47, 18, 47, 35, 29, 24, 44, 33, 36, 13, 33, 26, 38, 42, 33, 22, 30, 36, 37, 32, 24, 48, 45, 44, 14, 40, 33, 33, 31, 26, 38, 34, 23, 39, 28, 32, 30, 30, 24, 39, 33, 40, 31, 20, 35, 40, 40, 34, 26, 37, 32, 36, 27, 25, 28, 29, 50, 25, 17, 24, 21, 28, 12, 14, 20, 39, 33, 18, 34, 23, 19, 29, 31, 31, 34, 13, 25, 25, 25, 31, 39, 15, 35, 40, 28, 38, 32, 39, 28, 24, 30, 32, 26, 40, 31, 18, 29, 35, 23, 18, 40, 22, 12, 33, 27, 17, 23, 13, 10, 25, 25, 30, 33, 24, 22, 27, 43, 33, 24, 18, 20, 17, 26, 43, 26, 32, 39, 23, 26, 13, 35, 30, 25, 13, 20, 17, 19, 27, 26, 24, 14, 34, -1, 8, 18, 11, 23, 30, 36, 38, 43, 51, 16, 23, 29, 26, 26, 32, 21, 33, 29, 24, 33, 23, 21, 17, 25, 19, 24, 41, 32, 44, 43, 33, 28, 27, 33, 41, 31, 40, 19, 31, 48, 28, 35, 50, 45, 38, 39, 37, 44, 54, 36, 55, 28, 49, 45, 54, 33, 38, 40, 62, 56, 47, 47, 42, 32, 34, 23, 45, 56, 44, 38, 49, 53, 51, 51, 47, 63, 40, 47, 26, 52, 46, 60, 35, 45, 46, 47, 49, 33, 54, 54, 33, 52, 40, 54, 51, 52, 45, 44, 54, 53, 42, 40, 51, 58, 41, 46, 44, 39, 65, 50, 43, 56, 63, 61, 54, 57, 46, 53, 45, 46, 55, 35, 35, 51, 63, 60, 47, 42, 41, 41, 46, 36, 52, 39, 49, 49, 30, 36, 39, 43, 32, 52, 34, 54, 44, 52, 52, 37, 46, 38, 40, 50, 48, 28, 39, 40, 49, 41, 33, 48, 46, 40, 42, 39, 43, 42, 38, 36, 39, 42, 47, 29, 47, 43, 39, 31, 44, 27, 52, 37, 45, 27, 40, 36, 51, 36, 44, 38, 30, 27, 52, 42, 28, 28, 43, 51, 38, 32, 47, 37, 54, 52, 31, 25, 43, 42, 51, 55, 51, 40, 44, 37, 33, 28, 43, 36, 44, 41, 42, 30, 22, 39, 27, 37, 45, 24, 29, 27, 42, 25, 43, 41, 42, 41, 38, 40, 35, 39, 46, 53, 21, 37, 40, 44, 24, 40, 39, 41, 43, 30, 31, 48, 38, 25, 34, 48, 37, 35, 22, 50, 34, 49, 42, 32, 35, 49, 29, 32, 52, 25, 40, 19, 47, 47, 43, 38, 46, 25, 17, 45, 46, 24, 39, 43, 31, 55, 36, 51, 37, 23, 34, 28, 29, 50, 24, 34, 44, 21, 8, 42, 34, 29, 24, 31, 45, 35, 29, 26, 38, 34, 58, 45, 30, 34, 48, 49, 38, 15, 38, 29, 40, 36, 44, 15, 41, 36, 48, 36, 36, 35, 35, 35, 16, 36, 26, 30, 24, 22, 25, 24, 34, 38, 28, 33, 45, 33, 15, 22, 38, 25, 37, 33, 30, 47, 40, 39, 41, 39, 38, 31, 34, 21, 28, 38, 27, 45, 36, 27, 42, 35, 25, 26, 30, 40, 30, 28, 34, 19, 47, 30, 35, 33, 22, 18, 41, 21, 43, 35, 51, 52, 25, 27, 38, 35, 25, 36, 29, 55, 31, 23, 40, 36, 19, 30, 49, 24, 45, 21, 33, 45, 18, 20, 38, 49, 36, 17, 32, 23, 36, 27, 37, 37, 35, 35, 21, 34, 26, 36, 24, 38, 42, 36, 21, 27, 33, 48, 18, 45, 52, 17, 20, 34, 35, 47, 19, 44, 31, 36, 41, 29, 53, 33, 24, 51, 22, 32, 33, 33, 41, 41, 36, 44, 33, 44, 37, 37, 57, 27, 36, 37, 34, 27, 37, 32, 17, 40, 33, 19, 31, 33, 34, 48, 41, 64, 33, 29, 36, 31, 35, 25, 26, 27, 44, 34, 29, 41, 26, 17, 20, 44, 36, 37, 41, 26, 48, 32, 34, 6, 23, 36, 48, 20, 33, 32, 30, 21, 41, 50, 32, 27, 45, 45, 26, 43, 27, 21, 24, 27, 29, 31, 27, 57, 33, 32, 41, 30, 40, 18, 30, 24, 33, 22, 28, 42, 45, 36, 26, 31, 31, 10, 34, 20, 22, 32, 26, 21, 30, 28, 35, 31, 42, 36, 39, 37, 27, 49, 38, 44, 28, 45, 43, 37, 39, 27, 36, 32, 26, 36, 42, 33, 38, 38, 46, 34, 34, 36, 38, 43, 34, 29, 35, 52, 39, 39, 29, 47, 46, 38, 44, 49, 35, 55, 41, 50, 49, 56, 48, 53, 36, 32, 49, 40, 35, 34, 47, 53, 24, 52, 46, 28, 25, 33, 45, 30, 48, 39, 43, 48, 19, 57, 34, 39, 26, 51, 45, 47, 42, 38, 44, 31, 25, 38, 42, 40, 44, 43, 38, 54, 45, 42, 41, 36, 24, 44, 42, 16, 36, 41, 45, 27, 46, 43, 26, 48, 33, 39, 22, 30, 36, 45, 55, 30, 33, 41, 17, 38, 56, 35, 37, 49, 57, 53, 65, 61, 58, 56, 59, 39, 39, 51, 49, 51, 47, 48, 53, 37, 33, 30, 26, 39, 45, 38, 39, 44, 43, 54, 36, 57, 55, 34, 28, 47, 40, 30, 43, 42, 34, 43, 39, 40, 33, 38, 55, 45, 23, 30, 49, 63, 23, 54, 36, 46, 51, 23, 21, 47, 48, 35, 32, 27, 50, 46, 48, 49, 31, 36, 36, 44, 36, 37, 26, 43, 45, 38, 53, 50, 34, 39, 32, 38, 29, 33, 35, 31, 44, 21, 25, 31, 38, 35, 34, 34, 44, 19, 37, 34, 34, 33, 32, 44, 37, 35, 42, 30, 35, 35, 48, 52, 41, 37, 22, 40, 40, 38, 52, 41, 47, 51, 39, 38, 28, 50, 53, 34, 36, 30, 35, 29, 28, 36, 19, 47, 32, 36, 28, 30, 24, 41, 26, 47, 46, 31, 45, 27, 38, 44, 44, 37, 25, 36, 40, 34, 56, 41, 29, 25, 28, 32, 19, 30, 39, 36, 28, 21, 53, 30, 25, 39, 27, 38, 34, 40, 30, 25, 34, 47, 33, 42, 46, 42, 48, 36, 50, 20, 37, 38, 29, 28, 43, 36, 25, 37, 32, 21, 41, 44, 35, 27, 41, 28, 42, 24, 17, 34, 47, 30, 34, 17, 17, 19, 17, 28, 32, 31, 18, 17, 20, 29, 29, 34, 22, 20, 30, 27, 26, 34, 14, 30, 21, 14, 37, 33, 31, 31, 28, 26, 14, 42, 26, 26, 34, 10, 25, 25, 39, 21, 19, 40, 31, 22, 28, 21, 43, 32, 15, 16, 23, 20, 22, 28, 26, 35, 23, 23, 26, 29, 48, 25, 42, 25, 22, 20, 31, 32, 15, 16, 3, 41, 28, 7, 33, 26, 47, 36, 44, 47, 38, 30, 19, 28, 50, 57, 50, 41, 12, 48, 47, 43, 41, 60, 59, 51, 61, 47, 60, 66, 54, 52, 58, 51, 56, 56, 55, 41, 43, 61, 45, 33, 38, 43, 44, 42, 52, 45, 40, 52, 47, 40, 46, 57, 43, 48, 60, 34, 47, 50, 49, 53, 49, 46, 35, 49, 56, 38, 41, 40, 54, 39, 35, 51, 56, 36, 45, 24, 32, 47, 41, 38, 40, 32, 50, 45, 49, 29, 48, 37, 35, 36, 39, 41, 39, 47, 21, 44, 19, 42, 25, 25, 48, 28, 21, 18, 36, 36, 29, 26, 39, 40, 42, 28, 20, 37, 31, 44, 24, 26, 24, 42, 56, 24, 42, 36, 41, 44, 32, 44, 31, 23, 22, 41, 41, 34, 33, 46, 49, 39, 20, 24, 35, 25, 26, 31, 26, 32, 28, 47, 47, 33, 32, 40, 34, 33, 41, 43, 41, 38, 37, 18, 48, 25, 35, 17, 36, 41, 38, 38, 25, 26, 24, 45, 38, 34, 27, 21, 44, 41, 35, 33, 30, 20, 36, 25, 30, 27, 35, 26, 18, 29, 30, 35, 28, 31, 45, 39, 40, 30, 36, 29, 39, 29, 25, 31, 35, 43, 31, 30, 27, 14, 38, 34, 49, 43, 44, 36, 44, 55, 36, 32, 39, 42, 25, 33, 37, 43, 42, 35, 17, 29, 46, 41, 24, 35, 39, 24, 43, 31, 42, 29, 33, 20, 48, 45, 33, 50, 29, 27, 32, 25, 33, 31, 35, 43, 28, 28, 21, 34, 18, 24, 33, 45, 38, 39, 37, 33, 45, 38, 28, 48, 53, 37, 32, 27, 44, 41, 47, 43, 36, 56, 45, 51, 49, 43, 52, 42, 33, 57, 62, 64, 67, 39, 56, 59, 66, 49, 53, 71, 72, 59, 72, 72, 66, 83, 84, 79, 67, 71, 87, 94, 96, 70, 82, 73, 74, 71, 58, 71, 80, 79, 79, 65, 68, 78, 82, 61, 67, 83, 89, 65, 70, 52, 79, 63, 73, 90, 62, 78, 81, 84, 86, 89, 86, 70, 61, 64, 60, 79, 59, 64, 70, 90, 59, 66, 73, 72, 70, 53, 53, 71, 75, 72, 54, 56, 63, 60, 74, 71, 69, 73, 71, 110, 63, 76, 78, 75, 68, 93, 91, 79, 79, 77, 73, 84, 83, 91, 81, 81, 74, 78, 80, 65, 74, 78, 95, 93, 77, 100, 97, 89, 88, 84, 96, 83, 87, 60, 101, 79, 83, 76, 77, 77, 74, 61, 73, 88, 81, 95, 78, 94, 69, 78, 72, 94, 85, 96, 98, 86, 85, 75, 108, 98, 92, 79, 93, 105, 112, 110, 103, 108, 119, 154, 146, 209, 212, 193, 207, 237, 251, 274, 345, 361, 417, 452, 447, 521, 560, 559, 614, 679, 703, 762, 726, 704, 740, 744, 726, 687, 617, 586, 520, 464, 432, 405, 354, 339, 295, 286, 256, 234, 193, 214, 183, 177, 184, 165, 125, 153, 163, 167, 164, 130, 162, 142, 144, 124, 122, 143, 142, 127, 119, 134, 125, 113, 129, 90, 149, 116, 121, 116, 114, 141, 124, 115, 134, 136, 159, 142, 149, 165, 188, 217, 187, 224, 250, 335, 355, 450, 499, 534, 658, 751, 796, 802, 948, 936, 989, 1000, 981, 914, 873, 813, 764, 698, 619, 515, 418, 381, 304, 290, 253, 226, 201, 181, 174, 153, 176, 148, 148, 125, 123, 168, 135, 144, 172, 152, 155, 168, 172, 166, 151, 182, 127, 142, 147, 117, 166, 117, 136, 141, 150, 119, 115, 154, 116, 125, 118, 123, 124, 96, 111, 95, 98, 103, 112, 116, 114, 78, 117, 88, 119, 110, 107, 105, 114, 126, 109, 113, 128, 148, 139, 172, 250, 240, 283, 353, 425, 463, 573, 628, 685, 773, 847, 874, 942, 995, 881, 865, 810, 767, 679, 595, 491, 430, 374, 307, 271, 231, 195, 158, 140, 156, 145, 139, 132, 140, 126, 136, 158, 158, 141, 132, 128, 104, 133, 120, 105, 110, 127, 125, 111, 107, 101, 116, 115, 130, 111, 159, 148, 186, 187, 267, 290, 282, 386, 394, 448, 561, 617, 619, 675, 652, 687, 715, 698, 582, 628, 519, 395, 355, 286, 293, 283, 208, 206, 145, 143, 126, 139, 142, 121, 145, 127, 131, 124, 118, 111, 107, 126, 117, 102, 101, 85, 99, 100, 98, 119, 85, 98, 88, 83, 116, 113, 107, 85, 99, 109, 106, 82, 88, 88, 98, 104, 131, 98, 108, 111, 110, 125, 132, 132, 103, 110, 124, 137, 158, 178, 208, 273, 361, 377, 392, 479, 588, 718, 833, 870, 1001, 1091, 1233, 1225, 1282, 1220, 1221, 1107, 939, 819, 783, 629, 537, 434, 346, 279, 221, 183, 179, 160, 152, 129, 154, 134, 144, 129, 188, 188, 162, 169, 135, 145, 139, 137, 103, 111, 126, 104, 117, 90, 113, 99, 94, 109, 138, 150, 139, 94, 131, 115, 138, 110, 95, 103, 124, 143, 149, 128, 168, 178, 216, 246, 284, 361, 407, 477, 494, 553, 639, 706, 755, 754, 746, 742, 705, 604, 573, 539, 440, 389, 363, 307, 283, 243, 282, 271, 284, 303, 405, 447, 568, 654, 774, 821, 892, 962, 974, 989, 941, 885, 774, 645, 595, 537, 393, 373, 325, 273, 196, 173, 163, 150, 121, 123, 128, 115, 96, 113, 134, 114, 131, 141, 147, 148, 115, 116, 134, 141, 149, 127, 154, 195, 170, 179, 165, 183, 180, 181, 152, 172, 194, 159, 183, 165, 138, 163, 137, 124, 135, 111, 124, 118, 92, 108, 99, 97, 76, 97, 86, 68, 75, 88, 96, 118, 86, 135, 171, 219, 212, 272, 311, 406, 480, 557, 598, 652, 585, 651, 651, 566, 498, 476, 409, 340, 294, 282, 214, 254, 233, 277, 321, 416, 430, 482, 522, 623, 671, 716, 704, 743, 736, 672, 583, 582, 448, 397, 305, 263, 242, 241, 201, 223, 186, 207, 184, 213, 206, 265, 264, 342, 455, 619, 685, 784, 946, 1144, 1361, 1415, 1501, 1643, 1653, 1601, 1533, 1321, 1109, 934, 827, 604, 488, 363, 255, 225, 184, 162, 129, 113, 111, 78, 103, 102, 66, 79, 88, 73, 89, 104, 100, 85, 87, 102, 96, 98, 112, 129, 93, 106, 133, 87, 114, 101, 102, 100, 89, 98, 102, 116, 120, 87, 100, 104, 90, 111, 97, 82, 94, 98, 83, 117, 97, 115, 110, 123, 114, 138, 123, 134, 141, 140, 133, 153, 126, 124, 114, 120, 103, 78, 107, 101, 121, 109, 102, 104, 112, 60, 87, 104, 91, 107, 97, 80, 73, 91, 58, 70, 89, 79, 90, 81, 83, 70, 84, 90, 77, 102, 122, 87, 90, 87, 96, 79, 93, 98, 80, 81, 107, 93, 100, 79, 91, 99, 92, 56, 80, 98, 101, 107, 77, 87, 88, 99, 76, 105, 84, 92, 77, 98, 93, 110, 146, 115, 140, 168, 170, 148, 183, 246, 306, 394, 434, 572, 660, 693, 735, 801, 836, 774, 770, 670, 650, 529, 440, 349, 299, 223, 193, 156, 101, 104, 111, 105, 98, 90, 106, 83, 117, 104, 147, 140, 138, 156, 209, 212, 275, 315, 367, 445, 510, 616, 759, 762, 806, 887, 858, 787, 751, 670, 567, 489, 374, 290, 257, 224, 192, 157, 113, 89, 115, 101, 74, 107, 80, 69, 74, 94, 41, 54, 51, 56, 53, 111, 58, 80, 59, 80, 104, 60, 48, 74, 82, 67, 51, 56, 59, 84, 73, 91, 100, 47, 62, 103, 72, 98, 87, 88, 86, 53, 81, 38, 40, 38, 22, 61, 77, 95, 69, 86, 72, 65, 93, 110, 110, 116, 91, 84, 70, 111, 110, 100, 124, 69, 79, 101, 79, 74, 73, 58, 74, 102, 77, 56, 59, 83, 114, 73, 97, 99, 88, 109, 107, 93, 95, 94, 94, 114, 110, 129, 130, 129, 137, 97, 114, 112, 102, 96, 105, 103, 77, 89, 76, 77, 73, 72, 77, 94, 104, 107, 83, 106, 134, 117, 133, 125, 134, 119, 126, 119, 121, 114, 92, 97, 62, 71, 100, 82, 91, 72, 80, 78, 69, 98, 82, 111, 76, 111, 101, 108, 116, 108, 94, 105, 126, 105, 108, 116, 131, 107, 105, 116, 116, 130, 125, 140, 141, 152, 122, 135, 133, 158, 124, 119, 133, 129, 116, 126, 96, 91, 117, 130, 94, 114, 98, 131, 121, 130, 117, 108, 106, 113, 127, 123, 151, 159, 158, 187, 235, 280, 337, 429, 539, 742, 897, 1168, 1390, 1617, 1639, 1925, 1854, 1853, 1724, 1499, 1263, 1144, 862, 639, 522, 378, 321, 244, 198, 173, 178, 127, 136, 119, 169, 124, 147, 144, 135, 153, 142, 149, 173, 140, 162, 144, 158, 172, 142, 169, 160, 179, 158, 174, 161, 163, 169, 163, 172, 163, 203, 184, 198, 224, 252, 320, 380, 493, 566, 709, 861, 1142, 1246, 1322, 1394, 1395, 1384, 1331, 1116, 1010, 829, 675, 594, 437, 368, 283, 246, 249, 166, 181, 173, 196, 168, 175, 167, 186, 170, 166, 146, 148, 166, 165, 192, 181, 190, 186, 167, 174, 160, 135, 154, 172, 188, 168, 159, 161, 181, 200, 212, 228, 250, 297, 284, 396, 487, 599, 688, 843, 990, 1081, 1117, 1150, 1099, 1041, 962, 876, 740, 575, 405, 372, 301, 245, 196, 187, 159, 125, 125, 114, 102, 111, 105, 139, 81, 90, 70, 103, 101, 86, 83, 117, 118, 100, 114, 105, 151, 110, 114, 126, 130, 112, 152, 129, 140, 141, 116, 144, 129, 114, 115, 166, 126, 115, 153, 145, 158, 141, 134, 127, 148, 124, 105, 121, 126, 144, 122, 160, 161, 157, 158, 200, 154, 246, 281, 420, 555, 699, 938, 1087, 1463, 1747, 2059, 2182, 2141, 2203, 2041, 1974, 1691, 1391, 1124, 848, 666, 473, 342, 264, 175, 177, 141, 129, 109, 82, 95, 126, 100, 91, 96, 121, 125, 164, 158, 160, 181, 149, 160, 131, 140, 159, 128, 117, 146, 139, 99, 138, 119, 122, 122, 114, 132, 135, 128, 106, 130, 158, 173, 161, 224, 306, 345, 476, 631, 718, 880, 1100, 1130, 1202, 1186, 1037, 1000, 933, 762, 706, 567, 395, 319, 271, 175, 147, 138, 118, 112, 116, 82, 100, 116, 94, 101, 147, 116, 108, 92, 117, 149, 120, 142, 149, 135, 117, 151, 138, 148, 140, 152, 159, 184, 162, 135, 139, 146, 133, 120, 113, 134, 91, 125, 136, 108, 122, 123, 142, 129, 142, 151, 133, 133, 140, 149, 144, 115, 130, 148, 165, 122, 130, 121, 119, 142, 89, 136, 109, 136, 115, 115, 130, 152, 129, 162, 195, 200, 282, 362, 448, 624, 754, 957, 1119, 1155, 1323, 1323, 1323, 1192, 1066, 954, 770, 573, 467, 374, 257, 242, 176, 171, 163, 136, 116, 130, 169, 232, 284, 437, 541, 639, 861, 1009, 1163, 1292, 1349, 1358, 1291, 1031, 999, 736, 642, 449, 316, 266, 208, 174, 171, 133, 142, 118, 127, 107, 109, 66, 40, 55, 83, 87, 94, 63, 59, 82, 88, 109, 86, 101, 105, 113, 98, 86, 94, 78, 73, 58, 66, 77, 86, 94, 101, 116, 113, 132, 149, 143, 146, 129, 205, 217, 283, 374, 550, 687, 838, 961, 1071, 1173, 1240, 1121, 1064, 997, 868, 766, 666, 651, 677, 709, 851, 1002, 1076, 1237, 1213, 1305, 1168, 1099, 923, 833, 586, 472, 361, 289, 197, 175, 134, 126, 113, 121, 130, 121, 115, 97, 109, 112, 89, 96, 54, 72, 82, 103, 81, 73, 84, 93, 96, 72, 63, 81, 58, 49, 44, 52, 55, 58, 67, 87, 100, 115, 131, 124, 134, 113, 138, 128, 115, 71, 103, 81, 102, 99, 93, 77, 114, 118, 88, 97, 111, 101, 92, 111, 80, 87, 94, 96, 61, 63, 85, 92, 94, 83, 79, 135, 127, 114, 130, 87, 110, 135, 118, 113, 142, 149, 211, 268, 363, 544, 648, 881, 1013, 1241, 1448, 1572, 1601, 1579, 1428, 1201, 1056, 787, 621, 466, 358, 264, 253, 199, 171, 173, 160, 173, 220, 232, 266, 311, 443, 587, 702, 780, 906, 1058, 1064, 1091, 1062, 966, 794, 635, 511, 405, 347, 265, 228, 229, 181, 183, 170, 133, 124, 121, 122, 106, 96, 77, 91, 85, 91, 76, 88, 57, 106, 117, 79, 116, 111, 138, 128, 131, 162, 167, 247, 314, 418, 503, 652, 830, 989, 1046, 1237, 1239, 1186, 1061, 952, 778, 700, 501, 372, 288, 214, 160, 133, 119, 109, 124, 116, 109, 97, 116, 112, 124, 148, 120, 112, 97, 114, 84, 99, 102, 89, 109, 101, 101, 105, 103, 102, 97, 106, 126, 86, 116, 87, 80, 106, 94, 87, 66, 85, 73, 84, 82, 93, 61, 51, 66, 83, 52, 32, 48, 39, 29, 68, 83, 77, 41, 86, 68, 109, 95, 91, 75, 75, 92, 75, 103, 76, 72, 85, 117, 103, 112, 105, 117, 122, 136, 131, 138, 146, 124, 121, 145, 147, 134, 96, 139, 110, 119, 96, 109, 96, 74, 87, 96, 74, 55, 66, 58, 60, 79, 81, 85, 111, 75, 107, 92, 103, 85, 74, 93, 88, 83, 83, 86, 88, 70, 89, 98, 111, 106, 103, 114, 112, 81, 103, 114, 104, 122, 148, 131, 163, 221, 290, 395, 498, 620, 866, 1100, 1304, 1422, 1592, 1529, 1458, 1364, 1204, 992, 859, 689, 532, 390, 274, 244, 188, 162, 143, 156, 168, 172, 115, 180, 234, 254, 332, 508, 625, 713, 937, 1069, 1193, 1256, 1257, 1103, 1105, 940, 792, 590, 446, 359, 245, 228, 157, 132, 161, 120, 117, 148, 134, 137, 147, 185, 244, 312, 427, 558, 740, 926, 1060, 1079, 1251, 1198, 1138, 982, 860, 751, 657, 511, 305, 228, 140, 132, 110, 126, 107, 74, 98, 84, 93, 114, 100, 132, 134, 113, 86, 98, 114, 89, 92, 89, 105, 101, 93, 117, 77, 69, 88, 93, 84, 69, 70, 70, 74, 107, 85, 90, 81, 107, 84, 92, 66, 73, 58, 57, 69, 44, 77, 59, 78, 52, 61, 80, 78, 84, 50, 79, 89, 88, 63, 104, 80, 132, 123, 131, 151, 191, 235, 344, 425, 568, 841, 947, 1047, 1208, 1325, 1519, 1383, 1222, 1212, 990, 841, 750, 705, 618, 643, 795, 806, 951, 1133, 1121, 1229, 1103, 1050, 957, 769, 618, 532, 403, 319, 232, 164, 169, 140, 134, 136, 130, 93, 82, 104, 84, 89, 97, 81, 92, 87, 99, 113, 137, 131, 149, 142, 146, 137, 147, 142, 148, 153, 183, 223, 303, 402, 488, 647, 803, 924, 984, 1112, 1139, 1061, 957, 820, 692, 510, 406, 350, 227, 166, 160, 125, 130, 104, 126, 113, 116, 94, 106, 164, 189, 240, 321, 448, 580, 779, 869, 1105, 1220, 1283, 1372, 1283, 1259, 939, 851, 628, 609, 497, 492, 545, 586, 629, 727, 918, 969, 1044, 930, 1002, 910, 784, 636, 496, 426, 341, 248, 190, 162, 164, 130, 151, 129, 124, 137, 122, 128, 137, 182, 178, 286, 357, 493, 640, 790, 914, 1025, 1114, 1149, 1184, 1089, 925, 807, 665, 501, 435, 295, 216, 141, 131, 99, 105, 89, 109, 92, 95, 83, 82, 88, 101, 47, 83, 49, 83, 88, 104, 101, 93, 61, 68, 100, 68, 68, 53, 74, 46, 65, 52, 47, 61, 71, 61, 86, 87, 113, 128, 84, 83, 103, 72, 94, 39, 38, 73, 44, 38, 16, 42, 40, 69, 62, 58, 45, 89, 92, 63, 73, 64, 82, 100, 112, 99, 79, 120, 100, 91, 96, 115, 104, 55, 92, 112, 99, 77, 85, 89, 83, 90, 121, 131, 120, 117, 111, 88, 125, 113, 110, 106, 92, 81, 101, 113, 90, 101, 112, 109, 68, 83, 97, 81, 106, 117, 122, 88, 112, 133, 111, 110, 102, 133, 149, 140, 180, 244, 278, 336, 471, 604, 836, 902, 1059, 1146, 1246, 1217, 1163, 1039, 911, 693, 541, 438, 351, 232, 207, 180, 140, 132, 114, 109, 84, 121, 100, 88, 124, 90, 85, 93, 77, 78, 78, 106, 87, 90, 91, 104, 92, 74, 64, 85, 90, 102, 86, 105, 97, 131, 120, 86, 118, 108, 100, 134, 98, 135, 115, 121, 120, 114, 129, 102, 116, 133, 141, 149, 125, 90, 125, 103, 91, 121, 105, 112, 73, 129, 79, 82, 94, 104, 77, 79, 90, 104, 67, 81, 81, 73, 89, 69, 86, 107, 79, 69, 120, 165, 220, 279, 362, 483, 664, 789, 956, 966, 1051, 1130, 1028, 960, 826, 753, 654, 572, 564, 661, 759, 823, 917, 1032, 1215, 1288, 1229, 1147, 1051, 961, 697, 605, 474, 376, 286, 205, 129, 162, 134, 127, 92, 115, 75, 90, 87, 71, 56, 87, 44, 53, 67, 77, 66, 71, 77, 71, 68, 92, 96, 70, 80, 71, 76, 65, 50, 100, 96, 153, 129, 115, 139, 143, 165, 158, 158, 132, 176, 139, 135, 150, 127, 117, 139, 165, 176, 267, 351, 501, 679, 830, 1042, 1172, 1351, 1434, 1442, 1433, 1227, 1081, 895, 707, 586, 434, 311, 268, 222, 199, 179, 140, 136, 125, 158, 140, 175, 186, 243, 278, 357, 435, 574, 748, 888, 933, 1034, 1057, 1106, 1025, 852, 730, 561, 438, 365, 313, 233, 186, 162, 164, 184, 141, 138, 132, 118, 125, 126, 112, 145, 190, 240, 352, 445, 601, 760, 870, 914, 942, 969, 968, 852, 696, 535, 449, 405, 297, 238, 152, 125, 113, 115, 126, 129, 97, 95, 87, 100, 95, 78, 62, 87, 72, 94, 74, 93, 85, 66, 87, 78, 71, 80, 92, 67, 75, 86, 85, 73, 73, 67, 87, 84, 86, 96, 78, 88, 115, 127, 112, 130, 107, 102, 92, 71, 80, 74, 63, 68, 57, 55, 87, 89, 78, 69, 81, 72, 66, 99, 74, 79, 81, 97, 94, 102, 94, 93, 131, 118, 107, 110, 113, 109, 113, 91, 90, 121, 126, 121, 139, 158, 202, 231, 304, 464, 526, 641, 744, 863, 816, 874, 861, 852, 680, 593, 487, 409, 236, 220, 162, 164, 165, 113, 116, 98, 109, 80, 91, 80, 77, 72, 78, 92, 120, 144, 134, 129, 162, 145, 146, 151, 141, 144, 131, 122, 152, 119, 131, 138, 147, 142, 224, 287, 360, 484, 634, 839, 1101, 1322, 1603, 1812, 1881, 1910, 1719, 1629, 1347, 1138, 898, 665, 499, 376, 265, 239, 178, 168, 164, 134, 138, 124, 121, 76, 77, 76, 98, 93, 63, 74, 58, 71, 83, 121, 120, 104, 84, 105, 95, 98, 131, 142, 129, 169, 215, 307, 385, 546, 605, 803, 846, 946, 1096, 1054, 993, 972, 879, 667, 570, 525, 451, 471, 523, 641, 700, 868, 963, 981, 1015, 1069, 1000, 986, 747, 742, 619, 454, 343, 227, 222, 137, 139, 109, 116, 119, 104, 107, 88, 74, 63, 65, 95, 76, 115, 93, 110, 111, 100, 105, 84, 85, 59, 73, 86, 61, 76, 69, 80, 74, 63, 92, 108, 91, 110, 132, 110, 117, 119, 113, 101, 101, 112, 138, 99, 87, 113, 89, 109, 97, 121, 114, 122, 105, 97, 125, 104, 91, 85, 89, 57, 69, 111, 82, 86, 68, 75, 79, 76, 107, 95, 106, 116, 123, 123, 124, 130, 110, 114, 91, 138, 174, 137, 212, 263, 351, 433, 540, 595, 799, 884, 856, 840, 922, 835, 696, 562, 511, 449, 403, 439, 499, 687, 742, 828, 1026, 1067, 1037, 1045, 949, 899, 776, 616, 620, 454, 523, 519, 583, 714, 922, 1033, 1243, 1385, 1373, 1426, 1340, 1180, 945, 827, 605, 431, 339, 257, 223, 152, 129, 98, 113, 108, 105, 90, 94, 75, 78, 93, 78, 84, 101, 90, 87, 100, 75, 112, 87, 90, 95, 79, 88, 93, 45, 69, 60, 75, 86, 83, 78, 92, 75, 111, 64, 97, 91, 79, 75, 69, 87, 64, 87, 69, 103, 87, 116, 95, 118, 128, 113, 108, 117, 94, 90, 92, 59, 76, 67, 85, 76, 85, 53, 94, 72, 87, 98, 81, 102, 83, 77, 90, 91, 84, 73, 64, 69, 81, 85, 89, 84, 102, 114, 112, 125, 118, 121, 114, 112, 139, 115, 118, 91, 95, 96, 75, 81, 51, 96, 54, 93, 84, 93, 72, 107, 115, 106, 101, 113, 118, 131, 116, 122, 120, 131, 111, 123, 122, 125, 132, 123, 121, 125, 114, 127, 108, 85, 118, 122, 116, 109, 109, 125, 108, 109, 78, 85, 87, 112, 94, 104, 99, 94, 107, 91, 109, 120, 116, 113, 139, 121, 111, 75, 82, 87, 115, 108, 115, 136, 135, 117, 134, 125, 118, 136, 141, 185, 146, 240, 345, 495, 694, 860, 1248, 1408, 1689, 1825, 2129, 2002, 1831, 1783, 1600, 1254, 1094, 896, 708, 635, 732, 798, 859, 967, 1157, 1343, 1286, 1232, 1144, 1006, 867, 684, 573, 394, 331, 261, 190, 181, 156, 184, 130, 111, 122, 115, 120, 103, 105, 136, 121, 117, 114, 105, 73, 118, 102, 109, 97, 90, 93, 109, 93, 89, 119, 119, 125, 88, 122, 147, 128, 151, 117, 125, 108, 111, 119, 109, 131, 101, 114, 118, 135, 128, 96, 140, 179, 238, 280, 361, 539, 717, 877, 912, 1089, 1163, 1193, 1191, 1181, 987, 784, 660, 647, 533, 608, 577, 726, 874, 983, 1199, 1283, 1272, 1417, 1316, 1282, 1038, 860, 647, 498, 389, 353, 250, 161, 152, 111, 90, 111, 97, 85, 120, 112, 129, 152, 175, 282, 364, 459, 604, 768, 1039, 1156, 1417, 1657, 1670, 1507, 1570, 1345, 1224, 992, 751, 608, 432, 332, 262, 210, 181, 129, 122, 137, 99, 86, 84, 83, 77, 112, 110, 88, 99, 103, 66, 102, 104, 63, 75, 83, 48, 97, 78, 91, 82, 90, 82, 117, 90, 103, 124, 101, 128, 133, 137, 106, 108, 84, 89, 81, 77, 81, 98, 88, 73, 100, 96, 100, 96, 91, 96, 87, 61, 77, 63, 33, 50, 64, 92, 81, 78, 70, 57, 79, 61, 51, 64, 47, 70, 52, 68, 68, 49, 78, 65, 66, 70, 92, 88, 124, 106, 139, 139, 124, 108, 96, 115, 97, 105, 71, 80, 89, 94, 84, 122, 138, 114, 143, 121, 119, 126, 115, 107, 77, 84, 75, 91, 70, 52, 64, 78, 75, 38, 55, 77, 57, 89, 93, 83, 117, 86, 79, 119, 102, 114, 114, 116, 125, 81, 64, 121, 147, 151, 250, 361, 454, 526, 700, 896, 1014, 1116, 1153, 1217, 1224, 1141, 1024, 796, 692, 607, 559, 514, 614, 700, 841, 964, 1077, 1102, 1150, 1083, 1056, 917, 774, 648, 516, 499, 464, 531, 629, 706, 794, 939, 1088, 1121, 1154, 1129, 1064, 856, 713, 584, 452, 415, 282, 202, 181, 158, 135, 128, 67, 93, 66, 67, 61, 79, 77, 64, 34, 72, 26, 31, 41, 44, 68, 81, 89, 74, 84, 100, 91, 61, 47, 66, 64, 64, 32, 56, 57, 41, 50, 73, 80, 74, 73, 76, 115, 96, 78, 97, 88, 75, 102, 83, 94, 86, 87, 94, 132, 126, 143, 136, 125, 128, 120, 97, 105, 83, 76, 97, 79, 78, 79, 50, 76, 71, 74, 82, 86, 89, 104, 109, 101, 136, 167, 167, 152, 154, 177, 164, 216, 224, 275, 352, 525, 801, 1046, 1184, 1589, 1881, 2198, 2435, 2454, 2416, 2076, 1944, 1636, 1258, 987, 693, 561, 397, 239, 190, 171, 189, 125, 123, 129, 105, 144, 140, 110, 148, 132, 101, 111, 121, 120, 97, 98, 125, 106, 108, 83, 82, 91, 56, 66, 58, 82, 54, 72, 73, 78, 92, 68, 69, 93, 102, 81, 67, 111, 87, 79, 85, 87, 63, 51, 82, 76, 80, 100, 70, 60, 95, 96, 87, 77, 72, 67, 95, 60, 88, 84, 90, 59, 84, 113, 94, 92, 115, 146, 85, 87, 88, 84, 66, 106, 116, 129, 173, 264, 342, 450, 619, 706, 809, 897, 987, 881, 937, 829, 723, 605, 521, 487, 491, 500, 497, 600, 721, 808, 943, 1020, 1157, 1122, 1093, 938, 782, 681, 523, 459, 398, 447, 478, 632, 689, 902, 1048, 1146, 1289, 1256, 1299, 1135, 1081, 830, 725, 603, 573, 518, 434, 509, 649, 705, 899, 1042, 1091, 1244, 1279, 1297, 1115, 1026, 904, 777, 555, 482, 464, 427, 438, 617, 678, 785, 933, 1045, 1164, 1154, 1152, 989, 881, 755, 561, 461, 345, 243, 190, 141, 125, 133, 106, 108, 88, 114, 143, 114, 108, 96, 104, 105, 120, 116, 103, 94, 108, 114, 110, 111, 99, 113, 102, 134, 116, 86, 170, 104, 57, 96, 101, 89, 83, 77, 88, 68, 66, 93, 77, 69, 58, 82, 75, 73, 73, 54, 62, 47, 51, 81, 44, 50, 41, 43, 73, 73, 67, 51, 87, 84, 86, 87, 79, 99, 90, 85, 80, 88, 84, 93, 82, 116, 135, 119, 136, 137, 152, 154, 172, 163, 158, 133, 145, 149, 138, 190, 256, 289, 380, 573, 690, 874, 1115, 1245, 1400, 1444, 1426, 1296, 1244, 1151, 950, 834, 592, 426, 355, 272, 238, 199, 157, 132, 156, 152, 133, 105, 82, 116, 107, 148, 218, 243, 358, 453, 638, 787, 952, 1122, 1149, 1290, 1289, 1267, 1167, 1008, 864, 732, 523, 443, 293, 182, 165, 173, 186, 82, 75, 96, 109, 115, 96, 98, 90, 65, 80, 70, 62, 82, 101, 99, 117, 148, 141, 119, 119, 112, 113, 125, 143, 125, 130, 95, 122, 158, 198, 189, 274, 363, 462, 597, 758, 854, 1000, 1014, 972, 925, 851, 777, 605, 507, 394, 247, 222, 184, 128, 115, 103, 129, 110, 93, 89, 119, 112, 101, 107, 141, 151, 195, 258, 370, 484, 664, 825, 1101, 1310, 1388, 1437, 1579, 1329, 1365, 1303, 962, 785, 609, 496, 341, 299, 248, 186, 182, 189, 133, 150, 183, 166, 162, 184, 146, 163, 142, 215, 206, 228, 315, 471, 587, 769, 935, 1055, 1135, 1207, 1370, 1317, 1253, 1042, 915, 722, 556, 466, 349, 311, 208, 187, 166, 170, 167, 173, 162, 185, 207, 171, 152, 154, 156, 155, 111, 146, 182, 207, 338, 403, 506, 710, 1002, 1132, 1414, 1627, 1839, 1784, 1845, 1710, 1539, 1241, 928, 724, 622, 454, 309, 253, 224, 199, 160, 134, 112, 126, 101, 129, 95, 95, 85, 51, 75, 51, 68, 62, 91, 62, 87, 83, 68, 77, 103, 68, 82, 120, 94, 54, 53, 46, 80, 62, 80, 89, 101, 80, 120, 103, 111, 110, 130, 120, 121, 138, 134, 68, 66, 92, 87, 97, 85, 69, 69, 92, 107, 76, 85, 83, 92, 90, 116, 73, 73, 76, 70, 71, 86, 93, 96, 107, 80, 96, 142, 122, 154, 137, 103, 107, 109, 123, 115, 114, 109, 104, 121, 141, 102, 100, 120, 109, 141, 129, 146, 148, 133, 131, 140, 150, 139, 186, 132, 166, 242, 299, 394, 612, 785, 961, 1288, 1505, 1858, 2225, 2370, 2485, 2382, 2178, 2052, 1657, 1385, 1177, 895, 754, 713, 676, 720, 912, 918, 1061, 1093, 1117, 1185, 1127, 1004, 821, 696, 582, 436, 331, 268, 234, 197, 159, 139, 109, 119, 130, 100, 128, 126, 125, 160, 189, 233, 234, 349, 406, 579, 678, 714, 893, 963, 999, 977, 1101, 979, 852, 690, 573, 416, 387, 244, 198, 167, 141, 129, 83, 121, 98, 57, 65, 81, 80, 85, 98, 70, 101, 63, 73, 98, 79, 84, 99, 80, 87, 96, 77, 88, 64, 68, 60, 66, 77, 65, 65, 96, 90, 141, 130, 136, 116, 89, 123, 113, 123, 121, 122, 115, 113, 142, 119, 121, 104, 117, 138, 146, 200, 298, 377, 516, 688, 883, 1062, 1288, 1348, 1398, 1483, 1528, 1294, 1173, 1008, 806, 665, 448, 385, 265, 191, 123, 105, 98, 117, 76, 67, 85, 93, 87, 57, 66, 69, 88, 65, 72, 65, 83, 73, 84, 38, 95, 84, 47, 58, 61, 34, 29, 72, 39, 61, 57, 80, 58, 66, 71, 71, 100, 92, 80, 86, 78, 73, 83, 57, 68, 64, 29, 47, 68, 39, 12, 28, 58, 38, 15, 57, 29, 59, 50, 63, 64, 91, 51, 71, 58, 43, 53, 74, 51, 60, 55, 73, 94, 67, 68, 72, 110, 70, 75, 79, 109, 103, 103, 99, 100, 79, 93, 94, 90, 110, 75, 69, 83, 78, 75, 70, 71, 78, 88, 49, 80, 90, 86, 92, 93, 99, 94, 100, 70, 95, 100, 86, 123, 80, 90, 82, 87, 81, 80, 70, 76, 72, 77, 52, 64, 46, 72, 53, 67, 76, 83, 91, 66, 58, 75, 88, 62, 85, 64, 66, 96, 83, 87, 109, 84, 128, 120, 87, 100, 92, 87, 70, 116, 145, 179, 238, 325, 337, 470, 563, 744, 803, 1006, 1010, 924, 966, 855, 715, 664, 556, 437, 451, 410, 427, 400, 583, 690, 745, 775, 827, 808, 833, 853, 747, 598, 431, 348, 307, 227, 182, 130, 128, 145, 136, 159, 134, 116, 158, 125, 172, 152, 154, 155, 159, 220, 290, 324, 447, 615, 763, 960, 1106, 1181, 1299, 1397, 1398, 1419, 1175, 989, 835, 711, 486, 323, 288, 265, 174, 163, 141, 136, 104, 91, 104, 122, 114, 108, 112, 119, 116, 125, 132, 88, 116, 89, 79, 78, 96, 95, 91, 88, 121, 138, 121, 145, 96, 116, 113, 112, 106, 102, 88, 73, 70, 80, 96, 117, 101, 78, 116, 76, 91, 114, 87, 65, 85, 61, 65, 78, 95, 85, 90, 91, 89, 123, 126, 115, 114, 109, 102, 134, 78, 79, 104, 71, 69, 92, 82, 92, 75, 82, 121, 122, 145, 120, 117, 153, 117, 120, 100, 94, 73, 74, 82, 85, 88, 60, 92, 97, 118, 137, 165, 169, 161, 167, 139, 152, 125, 129, 135, 161, 198, 219, 288, 381, 602, 722, 957, 1166, 1300, 1442, 1521, 1530, 1497, 1365, 1181, 1102, 852, 743, 595, 528, 473, 518, 600, 644, 730, 803, 884, 1033, 969, 932, 932, 732, 738, 573, 534, 456, 479, 479, 506, 635, 891, 923, 1159, 1121, 1385, 1344, 1333, 1220, 1101, 941, 774, 621, 461, 315, 242, 196, 151, 134, 122, 117, 87, 93, 84, 71, 91, 90, 85, 73, 63, 16, 53, 62, 67, 59, 67, 82, 104, 110, 98, 125, 104, 124, 121, 78, 129, 100, 52, 88, 70, 97, 69, 38, 78, 67, 45, 73, 72, 54, 59, 77, 58, 84, 67, 60, 84, 77, 58, 64, 72, 96, 84, 117, 117, 82, 111, 87, 96, 108, 117, 134, 115, 83, 101, 87, 90, 88, 94, 124, 89, 123, 143, 152, 145, 164, 162, 175, 180, 156, 146, 122, 137, 120, 126, 98, 82, 102, 85, 91, 88, 113, 121, 126, 161, 129, 117, 145, 145, 110, 168, 184, 181, 266, 301, 479, 581, 712, 1110, 1313, 1639, 1921, 2243, 2533, 2467, 2604, 2359, 2128, 1931, 1600, 1422, 1155, 954, 829, 736, 729, 861, 986, 1028, 1187, 1227, 1289, 1259, 1224, 1087, 947, 760, 621, 514, 442, 296, 235, 197, 152, 165, 190, 170, 123, 125, 164, 143, 138, 155, 143, 192, 316, 354, 467, 628, 736, 881, 1021, 1053, 1272, 1241, 1225, 1116, 1029, 977, 758, 611, 391, 328, 247, 189, 170, 164, 150, 149, 148, 171, 157, 161, 113, 127, 126, 100, 141, 92, 125, 126, 145, 147, 122, 130, 159, 171, 176, 136, 109, 150, 122, 135, 83, 70, 95, 69, 76, 68, 80, 92, 79, 85, 69, 81, 88, 67, 64, 94, 64, 59, 58, 43, 62, 82, 54, 77, 65, 76, 112, 134, 132, 120, 140, 159, 135, 152, 138, 96, 105, 110, 76, 77, 69, 77, 72, 76, 90, 101, 95, 93, 120, 113, 91, 110, 126, 104, 132, 99, 148, 154, 224, 267, 313, 433, 580, 718, 784, 859, 957, 1033, 997, 904, 843, 728, 643, 527, 488, 443, 482, 518, 606, 726, 875, 984, 1094, 1130, 1122, 1044, 1013, 1025, 815, 615, 502, 386, 340, 219, 204, 146, 109, 151, 103, 88, 99, 92, 94, 102, 109, 91, 70, 60, 68, 65, 76, 110, 118, 115, 114, 136, 141, 166, 145, 159, 162, 151, 153, 195, 176, 170, 178, 175, 225, 302, 388, 485, 628, 754, 1013, 1175, 1397, 1576, 1684, 1749, 1627, 1588, 1407, 1220, 896, 769, 589, 534, 407, 331, 279, 219, 181, 182, 205, 191, 154, 199, 152, 146, 138, 170, 166, 174, 166, 179, 251, 301, 383, 536, 784, 984, 1129, 1291, 1570, 1653, 1707, 1718, 1605, 1531, 1284, 1114, 973, 733, 542, 440, 336, 266, 248, 234, 172, 183, 169, 124, 122, 144, 166, 187, 179, 191, 261, 276, 433, 504, 664, 857, 1007, 1104, 1245, 1366, 1268, 1241, 1157, 1030, 904, 779, 632, 517, 503, 530, 595, 610, 692, 880, 1015, 1079, 1194, 1203, 1106, 998, 952, 827, 770, 682, 548, 458, 504, 502, 568, 680, 865, 1044, 1189, 1255, 1205, 1249, 1213, 1095, 982, 861, 672, 526, 416, 336, 233, 198, 159, 131, 129, 126, 81, 86, 79, 74, 98, 95, 78, 80, 84, 81, 78, 85, 102, 74, 72, 61, 102, 91, 67, 78, 82, 74, 88, 78, 77, 76, 76, 88, 84, 93, 111, 115, 134, 134, 149, 117, 145, 124, 109, 108, 121, 119, 82, 94, 92, 100, 69, 94, 88, 80, 97, 43, 82, 71, 66, 88, 73, 105, 119, 90, 99, 88, 98, 83, 84, 99, 111, 107, 116, 105, 89, 94, 127, 96, 124, 127, 108, 122, 156, 174, 182, 159, 154, 188, 208, 183, 279, 431, 590, 638, 769, 963, 1206, 1520, 1699, 1782, 1916, 1893, 1638, 1609, 1288, 1073, 914, 679, 496, 404, 225, 210, 180, 149, 137, 146, 143, 108, 132, 103, 100, 130, 136, 118, 104, 114, 144, 105, 127, 117, 162, 142, 165, 158, 146, 152, 152, 153, 131, 133, 159, 191, 252, 290, 399, 537, 698, 937, 1255, 1449, 1708, 1926, 2109, 2165, 2204, 2195, 2053, 1696, 1447, 1154, 864, 685, 532, 429, 301, 237, 226, 164, 137, 113, 98, 81, 105, 110, 120, 100, 102, 128, 85, 110, 128, 128, 85, 85, 78, 91, 78, 42, 61, 43, 65, 63, 45, 75, 85, 68, 96, 112, 107, 99, 92, 136, 119, 117, 109, 104, 131, 147, 150, 135, 128, 125, 151, 143, 169, 230, 267, 338, 391, 506, 636, 803, 978, 1108, 1147, 1291, 1381, 1315, 1214, 1174, 1084, 937, 742, 622, 465, 366, 320, 202, 145, 122, 88, 114, 95, 76, 81, 80, 104, 104, 104, 95, 121, 112, 102, 108, 92, 73, 137, 129, 100, 92, 94, 107, 117, 114, 101, 112, 98, 105, 95, 94, 109, 111, 105, 95, 119, 149, 110, 119, 111, 127, 131, 102, 160, 141, 144, 189, 190, 238, 278, 383, 536, 603, 756, 900, 1091, 1150, 1281, 1265, 1365, 1251, 1038, 920, 747, 592, 474, 398, 336, 237, 255, 192, 163, 156, 124, 125, 137, 119, 79, 94, 90, 68, 85, 70, 74, 86, 58, 91, 94, 105, 76, 99, 100, 116, 127, 127, 86, 104, 118, 118, 138, 185, 244, 308, 381, 505, 611, 719, 840, 952, 1040, 1068, 1066, 945, 820, 801, 636, 514, 482, 432, 450, 467, 496, 596, 726, 717, 783, 898, 921, 907, 831, 770, 749, 612, 561, 409, 353, 284, 275, 176, 194, 201, 188, 181, 147, 162, 130, 135, 95, 117, 117, 105, 102, 140, 184, 280, 279, 362, 478, 559, 668, 805, 884, 878, 924, 871, 838, 715, 666, 552, 490, 370, 340, 276, 222, 163, 193, 134, 158, 134, 111, 129, 102, 89, 112, 138, 140, 99, 126, 130, 125, 146, 123, 121, 140, 128, 93, 97, 97, 82, 105, 73, 84, 94, 83, 93, 124, 109, 113, 130, 136, 108, 121, 103, 118, 97, 116, 116, 135, 113, 126, 129, 108, 105, 115, 125, 167, 149, 228, 288, 429, 515, 665, 758, 977, 1047, 1215, 1265, 1341, 1340, 1274, 1159, 968, 883, 724, 555, 496, 353, 289, 227, 223, 162, 156, 105, 144, 130, 106, 102, 143, 138, 127, 108, 78, 103, 103, 87, 109, 126, 109, 73, 93, 90, 66, 60, 81, 99, 76, 80, 86, 91, 95, 103, 95, 99, 92, 83, 114, 94, 84, 91, 105, 91, 95, 77, 81, 84, 91, 71, 102, 105, 120, 79, 73, 93, 114, 91, 101, 120, 88, 58, 88, 96, 87, 89, 126, 114, 106, 108, 106, 112, 75, 125, 101, 106, 127, 116, 109, 125, 102, 113, 138, 115, 131, 113, 110, 102, 138, 99, 120, 99, 106, 110, 78, 86, 48, 79, 66, 78, 132, 116, 104, 122, 103, 136, 153, 130, 154, 107, 105, 159, 135, 137, 130, 144, 121, 120, 146, 138, 149, 148, 165, 183, 162, 152, 174, 179, 224, 276, 341, 417, 606, 805, 969, 1251, 1381, 1604, 1830, 1836, 1901, 1809, 1622, 1454, 1354, 1076, 884, 686, 549, 461, 313, 224, 202, 151, 142, 140, 136, 148, 137, 109, 122, 117, 123, 98, 107, 126, 104, 113, 136, 110, 112, 109, 115, 121, 132, 150, 133, 104, 106, 114, 130, 92, 127, 103, 117, 88, 95, 113, 94, 99, 114, 132, 101, 99, 87, 83, 64, 53, 65, 77, 51, 54, 46, 50, 53, 50, 58, 73, 86, 88, 66, 94, 93, 94, 83, 93, 76, 52, 46, 57, 63, 87, 100, 88, 102, 82, 89, 81, 77, 98, 66, 60, 82, 59, 76, 71, 91, 95, 91, 82, 105, 103, 114, 123, 120, 97, 148, 110, 140, 184, 129, 165, 138, 175, 181, 209, 264, 344, 405, 542, 722, 808, 990, 1014, 1071, 1271, 1234, 1174, 1054, 1046, 955, 708, 641, 541, 528, 518, 613, 685, 732, 889, 922, 1017, 1073, 1125, 1076, 952, 901, 883, 731, 605, 570, 492, 500, 511, 661, 809, 845, 941, 1029, 1122, 1141, 1156, 1068, 965, 799, 796, 613, 483, 408, 288, 258, 170, 139, 127, 115, 89, 101, 68, 93, 82, 110, 77, 121, 80, 105, 99, 47, 71, 120, 120, 115, 105, 106, 122, 104, 95, 89, 107, 77, 90, 129, 92, 89, 67, 108, 103, 91, 76, 97, 65, 73, 91, 54, 42, 71, 65, 64, 84, 72, 68, 41, 55, 92, 44, 70, 76, 54, 69, 69, 86, 75, 96, 75, 72, 76, 105, 112, 90, 119, 111, 128, 115, 144, 117, 117, 141, 111, 128, 123, 129, 121, 115, 109, 127, 113, 118, 104, 113, 124, 108, 95, 100, 101, 106, 132, 123, 103, 124, 92, 141, 125, 136, 101, 97, 77, 92, 98, 58, 102, 90, 87, 94, 145, 145, 120, 169, 153, 165, 143, 120, 127, 141, 159, 161, 182, 172, 233, 289, 368, 465, 541, 654, 763, 859, 864, 998, 945, 942, 905, 878, 715, 565, 606, 621, 512, 591, 576, 728, 785, 888, 1007, 1118, 1049, 1093, 1077, 1000, 877, 796, 703, 585, 562, 536, 522, 617, 706, 778, 1002, 1120, 1219, 1375, 1333, 1349, 1294, 1228, 1176, 1070, 855, 703, 553, 488, 408, 270, 262, 223, 201, 182, 188, 165, 143, 132, 173, 162, 146, 179, 210, 247, 289, 334, 435, 555, 654, 842, 935, 1096, 1084, 1365, 1346, 1272, 1200, 1115, 1026, 920, 724, 552, 457, 431, 353, 254, 211, 196, 161, 195, 204, 173, 161, 169, 182, 154, 194, 207, 240, 263, 328, 405, 518, 577, 678, 839, 879, 1015, 1018, 1108, 1082, 1056, 901, 827, 683, 529, 457, 391, 294, 274, 224, 185, 153, 136, 133, 150, 140, 131, 135, 120, 106, 115, 108, 86, 68, 69, 58, 93, 92, 113, 93, 91, 96, 111, 86, 136, 105, 92, 85, 92, 105, 108, 76, 77, 91, 76, 93, 91, 105, 119, 106, 121, 142, 115, 103, 88, 94, 72, 74, 74, 95, 70, 45, 70, 77, 67, 56, 77, 103, 101, 116, 103, 142, 127, 117, 119, 112, 89, 104, 74, 77, 62, 64, 63, 93, 73, 78, 113, 89, 113, 101, 113, 106, 116, 119, 123, 64, 88, 65, 76, 82, 84, 79, 61, 109, 81, 120, 81, 75, 90, 83, 92, 78, 101, 94, 112, 126, 133, 160, 211, 264, 351, 463, 519, 647, 728, 776, 835, 954, 990, 875, 808, 696, 689, 664, 535, 542, 517, 534, 555, 650, 689, 727, 725, 814, 869, 881, 891, 775, 740, 653, 594, 420, 366, 302, 242, 203, 167, 167, 153, 128, 80, 102, 81, 53, 91, 71, 55, 60, 75, 59, 78, 95, 63, 70, 74, 86, 64, 55, 85, 78, 89, 98, 107, 67, 114, 99, 96, 98, 104, 90, 98, 86, 75, 98, 111, 70, 69, 88, 86, 67, 91, 82, 91, 90, 86, 74, 122, 86, 94, 107, 84, 125, 111, 137, 119, 117, 122, 116, 118, 99, 98, 91, 103, 124, 105, 100, 95, 103, 101, 115, 96, 117, 126, 126, 138, 126, 111, 166, 166, 169, 158, 182, 178, 136, 170, 130, 185, 258, 327, 457, 601, 765, 897, 1190, 1447, 1628, 1990, 2012, 2275, 2280, 2148, 2088, 1892, 1602, 1413, 1189, 931, 680, 575, 459, 330, 238, 216, 179, 132, 130, 131, 120, 93, 131, 122, 75, 96, 105, 107, 72, 63, 57, 107, 104, 98, 90, 106, 83, 90, 68, 69, 66, 65, 70, 60, 72, 38, 59, 50, 74, 74, 62, 63, 73, 65, 58, 76, 67, 80, 74, 75, 56, 85, 126, 103, 90, 96, 110, 124, 126, 159, 136, 94, 120, 148, 128, 176, 148, 111, 147, 136, 187, 237, 251, 358, 492, 562, 636, 730, 784, 907, 930, 889, 884, 755, 654, 618, 532, 421, 382, 261, 181, 164, 145, 157, 136, 83, 106, 107, 104, 101, 107, 90, 114, 118, 108, 98, 129, 118, 117, 104, 123, 108, 115, 112, 132, 127, 131, 88, 105, 99, 95, 113, 109, 102, 113, 87, 102, 68, 113, 99, 78, 77, 63, 52, 81, 87, 78, 61, 84, 91, 87, 124, 65, 49, 66, 85, 87, 60, 70, 76, 75, 90, 84, 71, 103, 62, 72, 108, 82, 79, 105, 99, 103, 120, 59, 90, 116, 117, 125, 99, 129, 100, 97, 119, 100, 115, 99, 102, 113, 112, 110, 120, 127, 124, 151, 129, 159, 140, 196, 147, 134, 184, 209, 165, 177, 198, 184, 228, 254, 293, 348, 460, 613, 806, 1102, 1251, 1326, 1560, 1931, 1927, 2096, 2076, 2044, 1814, 1755, 1509, 1279, 1162, 1005, 867, 824, 818, 823, 928, 911, 1015, 1086, 1157, 1210, 1122, 1092, 992, 872, 714, 643, 516, 443, 326, 294, 209, 160, 127, 99, 84, 126, 96, 90, 94, 93, 93, 84, 89, 91, 109, 108, 81, 75, 96, 67, 73, 45, 91, 66, 77, 86, 70, 46, 40, 70, 75, 64, 88, 89, 70, 57, 69, 60, 60, 67, 67, 77, 68, 81, 59, 88, 86, 66, 51, 82, 59, 64, 74, 73, 65, 36, 62, 62, 52, 92, 61, 69, 64, 55, 89, 78, 62, 96, 48, 77, 53, 58, 98, 112, 74, 100, 88, 42, 43, 50, 77, 55, 57, 63, 26, 19, 39, 41, 41, 56, 44, 38, 59, 77, 50, 52, 92, 75, 48, 70, 61, 89, 60, 56, 66, 82, 67, 106, 107, 82, 78, 80, 68, 93, 85, 97, 67, 77, 81, 97, 99, 104, 65, 109, 76, 54, 85, 89, 93, 95, 71, 81, 82, 62, 73, 49, 39, 75, 72, 73, 101, 74, 64, 88, 76, 92, 100, 72, 101, 67, 109, 107, 86, 72, 99, 71, 92, 93, 92, 88, 89, 116, 106, 83, 86, 72, 93, 102, 91, 76, 62, 70, 90, 113, 84, 78, 73, 66, 65, 77, 75, 63, 67, 40, 7, 32, 73, 40, 41, 69, 71, 49, 65, 69, 82, 64, 72, 90, 110, 85, 77, 110, 98, 127, 99, 96, 120, 134, 110, 145, 131, 131, 87, 103, 106, 108, 98, 89, 60, 81, 66, 75, 61, 81, 75, 81, 58, 76, 85, 89, 64, 67, 50, 87, 62, 82, 77, 72, 60, 74, 82, 97, 105, 112, 116, 147, 108, 119, 139, 130, 111, 155, 183, 168, 201, 255, 331, 445, 510, 590, 815, 924, 992, 1232, 1275, 1299, 1277, 1405, 1285, 1134, 1068, 920, 840, 730, 725, 766, 738, 802, 868, 1020, 1128, 1375, 1441, 1376, 1403, 1263, 1172, 1051, 939, 802, 668, 539, 418, 292, 318, 220, 186, 179, 140, 117, 103, 118, 108, 112, 85, 113, 114, 94, 107, 97, 125, 89, 99, 84, 99, 91, 109, 68, 91, 77, 81, 75, 67, 88, 81, 59, 71, 69, 52, 57, 43, 51, 98, 62, 68, 66, 66, 95, 68, 62, 60, 35, 51, 95, 77, 67, 62, 77, 59, 74, 70, 72, 124, 89, 102, 91, 122, 116, 76, 84, 68, 113, 76, 95, 107, 85, 101, 83, 72, 102, 96, 91, 99, 73, 84, 113, 114, 134, 96, 120, 102, 138, 159, 211, 241, 323, 391, 477, 658, 757, 890, 969, 1101, 1098, 1121, 1134, 1143, 1017, 921, 854, 681, 632, 486, 349, 299, 241, 196, 156, 147, 146, 153, 121, 102, 109, 128, 112, 100, 82, 77, 74, 100, 104, 108, 90, 74, 105, 88, 86, 107, 80, 134, 99, 75, 99, 84, 101, 118, 114, 111, 89, 82, 113, 126, 103, 132, 87, 84, 96, 90, 76, 51, 58, 59, 60, 60, 67, 47, 64, 60, 67, 64, 91, 47, 58, 51, 76, 81, 68, 78, 72, 73, 83, 45, 83, 94, 89, 82, 102, 94, 115, 108, 109, 118, 141, 139, 139, 123, 143, 145, 145, 122, 134, 175, 215, 274, 299, 336, 433, 549, 625, 659, 771, 787, 808, 926, 794, 722, 688, 568, 491, 493, 394, 292, 229, 210, 177, 190, 183, 160, 160, 168, 167, 200, 157, 182, 170, 179, 206, 261, 340, 435, 550, 744, 860, 1169, 1360, 1700, 1846, 2014, 2189, 2410, 2423, 2327, 2207, 1915, 1698, 1525, 1202, 991, 880, 647, 551, 434, 333, 237, 195, 179, 176, 199, 175, 162, 175, 151, 132, 130, 96, 89, 79, 68, 73, 83, 78, 71, 84, 73, 80, 101, 90, 71, 104, 86, 88, 97, 102, 95, 123, 86, 127, 119, 153, 109, 104, 142, 111, 103, 108, 120, 103, 110, 95, 116, 93, 85, 95, 110, 101, 131, 112, 106, 94, 125, 113, 96, 91, 95, 83, 66, 94, 63, 95, 106, 86, 67, 52, 61, 57, 60, 49, 72, 31, 35, 58, 59, 61, 84, 62, 57, 48, 96, 78, 90, 103, 79, 77, 75, 89, 99, 79, 85, 115, 128, 125, 98, 89, 70, 96, 109, 110, 103, 92, 99, 96, 89, 94, 83, 80, 84, 69, 69, 98, 108, 110, 77, 75, 81, 105, 61, 75, 97, 95, 86, 65, 88, 111, 78, 113, 98, 120, 101, 85, 119, 123, 136, 142, 168, 176, 198, 249, 283, 380, 420, 478, 534, 632, 678, 664, 729, 674, 645, 651, 558, 513, 472, 492, 438, 542, 553, 635, 607, 784, 779, 831, 896, 878, 761, 831, 758, 638, 664, 520, 468, 418, 343, 248, 230, 151, 164, 140, 158, 124, 141, 170, 144, 120, 164, 191, 217, 205, 246, 271, 337, 428, 570, 684, 791, 906, 1051, 1120, 1065, 1113, 1020, 1058, 1027, 1012, 881, 784, 632, 658, 593, 617, 615, 681, 723, 782, 901, 887, 1011, 1034, 1006, 983, 876, 814, 703, 563, 513, 448, 357, 281, 255, 240, 213, 188, 172, 138, 164, 145, 175, 162, 118, 165, 174, 182, 239, 293, 375, 458, 553, 626, 787, 895, 1054, 1155, 1140, 1171, 1243, 1271, 1204, 1013, 892, 716, 654, 523, 424, 323, 256, 224, 169, 168, 166, 101, 97, 86, 93, 99, 96, 95, 98, 136, 142, 65, 112, 108, 72, 79, 110, 81, 103, 98, 122, 118, 114, 91, 97, 74, 61, 91, 103, 143, 97, 114, 93, 129, 103, 111, 134, 98, 98, 88, 87, 95, 97, 92, 69, 83, 83, 68, 45, 62, 83, 77, 97, 83, 100, 73, 68, 106, 49, 91, 73, 73, 57, 104, 96, 144, 159, 184, 217, 243, 309, 400, 518, 585, 595, 665, 794, 877, 821, 868, 869, 797, 735, 645, 585, 467, 394, 318, 247, 251, 193, 191, 167, 194, 188, 151, 154, 159, 172, 203, 164, 217, 258, 256, 330, 433, 490, 670, 807, 856, 1038, 1129, 1200, 1278, 1310, 1233, 1173, 1095, 1050, 987, 831, 717, 610, 518, 395, 298, 239, 186, 143, 126, 118, 106, 95, 97, 83, 107, 93, 93, 75, 79, 66, 102, 75, 65, 95, 107, 87, 86, 97, 68, 94, 77, 76, 84, 80, 87, 93, 93, 82, 59, 69, 65, 77, 77, 67, 90, 84, 60, 81, 64, 67, 66, 37, 48, 50, 65, 58, 46, 67, 51, 51, 33, 26, 3, 28, 42, 45, 59, 33, 35, 41, 37, 73, 67, 80, 67, 70, 113, 81, 79, 94, 94, 124, 121, 109, 104, 115, 136, 125, 145, 149, 124, 123, 134, 109, 128, 160, 211, 250, 276, 351, 405, 459, 587, 647, 878, 976, 964, 1138, 1217, 1272, 1241, 1230, 1144, 982, 917, 878, 705, 634, 492, 351, 293, 243, 185, 196, 137, 110, 128, 90, 124, 121, 112, 98, 74, 94, 63, 61, 86, 88, 107, 70, 71, 74, 56, 107, 97, 90, 94, 64, 85, 98, 118, 101, 105, 88, 96, 119, 119, 112, 110, 95, 102, 83, 85, 69, 62, 84, 100, 89, 62, 97, 80, 56, 61, 72, 56, 64, 69, 65, 34, 57, 44, 64, 68, 51, 93, 78, 35, 43, 45, 56, 37, 28, 29, 24, 21, 29, 65, 59, 48, 42, 48, 32, 37, 54, 30, 47, 16, 36, 41, 56, 45, 63, 61, 71, 80, 77, 55, 74, 67, 61, 76, 76, 101, 89, 107, 134, 151, 133, 93, 118, 135, 102, 93, 73, 107, 86, 90, 55, 66, 74, 85, 62, 66, 72, 39, 30, 39, 60, 55, 50, 21, 41, 39, 54, 45, 35, 22, 52, 69, 46, 54, 58, 71, 64, 93, 66, 83, 64, 69, 60, 63, 84, 55, 61, 71, 73, 87, 59, 58, 49, 45, 76, 63, 72, 40, 73, 61, 50, 68, 50, 59, 77, 61, 72, 65, 76, 50, 68, 67, 76, 66, 80, 81, 87, 75, 71, 84, 72, 92, 96, 92, 98, 77, 72, 83, 102, 66, 95, 121, 111, 82, 66, 99, 63, 69, 76, 69, 68, 59, 61, 69, 72, 79, 80, 97, 88, 85, 78, 93, 86, 107, 103, 78, 123, 95, 61, 71, 82, 56, 85, 64, 102, 50, 54, 75, 66, 60, 94, 85, 79, 104, 112, 72, 81, 81, 106, 100, 83, 85, 97, 59, 57, 70, 77, 86, 61, 64, 64, 64, 66, 76, 95, 66, 84, 65, 78, 74, 92, 84, 96, 85, 102, 137, 93, 89, 90, 116, 113, 126, 119, 104, 153, 152, 128, 138, 137, 191, 159, 173, 163, 165, 145, 218, 209, 217, 214, 309, 337, 477, 497, 671, 833, 1021, 1136, 1233, 1275, 1434, 1562, 1593, 1492, 1396, 1394, 1245, 1073, 1049, 962, 838, 943, 835, 741, 825, 903, 933, 1050, 1134, 1084, 1041, 1068, 977, 949, 791, 791, 786, 705, 657, 637, 765, 863, 909, 990, 1068, 1158, 1251, 1146, 1088, 1014, 940, 786, 717, 597, 543, 446, 358, 270, 241, 182, 142, 118, 125, 134, 97, 103, 113, 93, 97, 92, 88, 86, 80, 74, 86, 80, 97, 107, 104, 88, 59, 98, 75, 55, 90, 105, 106, 118, 92, 102, 81, 79, 103, 102, 92, 94, 112, 135, 121, 121, 128, 126, 150, 188, 149, 134, 222, 220, 246, 302, 376, 369, 493, 633, 771, 804, 899, 874, 928, 998, 978, 1067, 1009, 881, 774, 709, 598, 478, 390, 331, 279, 238, 174, 172, 169, 143, 146, 166, 96, 124, 125, 97, 82, 82, 64, 75, 103, 93, 83, 92, 95, 90, 105, 91, 113, 122, 131, 134, 125, 131, 200, 156, 190, 223, 223, 368, 430, 523, 562, 664, 725, 737, 712, 872, 923, 833, 741, 720, 664, 592, 601, 555, 519, 466, 533, 533, 552, 565, 565, 557, 615, 678, 616, 616, 595, 586, 478, 396, 375, 363, 319, 275, 211, 173, 129, 100, 115, 134, 142, 123, 109, 118, 141, 148, 105, 124, 125, 141, 127, 127, 127, 135, 123, 159, 152, 135, 121, 106, 140, 151, 139, 159, 139, 146, 201, 195, 218, 265, 322, 377, 492, 664, 736, 870, 948, 1086, 1251, 1268, 1334, 1360, 1403, 1319, 1190, 1107, 1027, 879, 777, 621, 497, 425, 323, 249, 200, 168, 101, 88, 107, 92, 83, 58, 61, 57, 58, 79, 85, 87, 62, 67, 51, 70, 76, 78, 88, 82, 115, 107, 112, 112, 96, 90, 79, 95, 96, 94, 97, 95, 65, 77, 73, 89, 80, 59, 46, 74, 89, 80, 74, 85, 41, 90, 101, 56, 74, 68, 81, 87, 86, 68, 77, 79, 83, 75, 86, 104, 93, 99, 98, 97, 113, 112, 156, 188, 232, 258, 318, 375, 438, 547, 666, 771, 856, 889, 1014, 1108, 1010, 1082, 990, 1013, 834, 748, 678, 583, 425, 357, 279, 251, 217, 205, 180, 152, 111, 93, 99, 93, 97, 90, 111, 54, 93, 115, 98, 88, 98, 78, 90, 59, 74, 60, 81, 79, 99, 65, 63, 83, 77, 35, 68, 78, 67, 90, 87, 60, 73, 110, 93, 115, 115, 102, 102, 79, 84, 96, 97, 131, 137, 173, 213, 241, 342, 406, 541, 643, 831, 898, 1056, 1074, 1122, 1272, 1231, 1186, 1148, 1112, 1045, 825, 765, 589, 524, 445, 382, 257, 180, 152, 135, 126, 112, 71, 135, 112, 82, 80, 133, 82, 62, 76, 54, 66, 61, 61, 39, 36, 66, 56, 54, 81, 53, 57, 68, 69, 74, 79, 55, 60, 48, 57, 40, 39, 67, 72, 71, 52, 74, 86, 63, 71, 84, 90, 102, 65, 71, 85, 108, 103, 112, 76, 97, 74, 64, 83, 112, 91, 115, 94, 85, 55, 67, 53, 106, 71, 69, 69, 54, 64, 85, 50, 60, 70, 90, 73, 94, 84, 79, 88, 98, 77, 55, 64, 72, 52, 74, 57, 93, 62, 75, 47, 50, 78, 84, 101, 109, 119, 147, 90, 133, 125, 129, 208, 242, 288, 327, 350, 419, 550, 607, 541, 718, 769, 839, 858, 844, 732, 716, 708, 604, 532, 497, 385, 324, 264, 219, 211, 228, 175, 122, 138, 144, 139, 160, 176, 196, 202, 222, 287, 315, 379, 491, 537, 603, 709, 836, 824, 849, 957, 910, 918, 788, 699, 732, 634, 540, 468, 402, 345, 264, 264, 216, 186, 146, 128, 124, 110, 140, 116, 137, 146, 123, 155, 211, 205, 208, 260, 338, 424, 465, 536, 541, 607, 616, 703, 647, 658, 591, 574, 612, 557, 461, 441, 472, 489, 570, 598, 696, 698, 789, 853, 926, 922, 854, 910, 841, 720, 629, 567, 498, 434, 407, 286, 261, 222, 184, 144, 117, 125, 107, 92, 121, 88, 91, 98, 90, 67, 102, 71, 63, 59, 89, 32, 90, 59, 80, 58, 81, 59, 70, 69, 61, 79, 54, 80, 63, 59, 89, 69, 91, 64, 64, 68, 66, 68, 34, 44, 58, 42, 64, 72, 67, 64, 48, 55, 41, 95, 76, 53, 70, 79, 84, 82, 89, 105, 83, 92, 91, 103, 124, 100, 87, 137, 180, 185, 235, 243, 290, 347, 407, 439, 496, 622, 632, 637, 675, 731, 686, 620, 588, 603, 562, 482, 367, 338, 273, 261, 231, 195, 167, 153, 156, 171, 144, 162, 148, 133, 140, 198, 201, 229, 278, 324, 381, 430, 502, 558, 703, 702, 706, 777, 748, 788, 754, 702, 664, 544, 520, 453, 382, 330, 282, 230, 174, 169, 100, 112, 103, 91, 102, 87, 102, 73, 77, 68, 78, 63, 58, 55, 89, 72, 65, 88, 84, 91, 68, 81, 93, 98, 106, 88, 90, 65, 97, 65, 73, 72, 96, 98, 75, 55, 77, 86, 89, 77, 70, 75, 66, 93, 92, 61, 77, 68, 81, 63, 78, 70, 44, 82, 58, 53, 58, 75, 78, 49, 36, 62, 61, 60, 47, 17, 34, 47, 36, 36, 29, 54, 37, 37, 47, 49, 42, 48, 47, 54, 57, 54, 32, 43, 55, 70, 85, 59, 72, 67, 35, 42, 33, 57, 57, 55, 74, 53, 85, 118, 92, 129, 110, 116, 137, 165, 206, 247, 297, 370, 497, 612, 656, 675, 689, 779, 794, 827, 810, 790, 815, 776, 716, 682, 639, 647, 665, 611, 605, 673, 786, 801, 823, 815, 809, 827, 879, 759, 743, 659, 570, 512, 455, 328, 276, 252, 180, 175, 156, 144, 140, 116, 129, 100, 90, 109, 108, 90, 67, 80, 66, 52, 74, 81, 47, 51, 61, 56, 50, 44, 47, 35, 45, 89, 61, 41, 49, 64, 39, 43, 37, 42, 48, 27, 50, 78, 46, 30, 56, 58, 52, 63, 67, 62, 123, 135, 138, 172, 222, 228, 323, 391, 453, 579, 665, 699, 760, 726, 750, 819, 791, 751, 629, 543, 533, 424, 438, 388, 271, 216, 168, 145, 80, 82, 128, 104, 86, 75, 72, 52, 74, 81, 81, 86, 77, 122, 81, 48, 77, 59, 78, 49, 47, 60, 51, 73, 66, 72, 73, 82, 60, 42, 57, 74, 41, 66, 36, 53, 63, 67, 25, 44, 58, 39, 37, 60, 92, 60, 70, 56, 72, 59, 65, 68, 38, 52, 45, 22, 47, 62, 44, 62, 58, 43, 38, 47, 61, 61, 51, 56, 46, 46, 40, 64, 47, 59, 38, 56, 51, 15, 8, 12, 52, 12, 30, 39, 7, 16, 46, 23, 68, 55, 49, 51, 62, 67, 61, 50, 90, 78, 90, 48, 71, 39, 58, 58, 52, 55, 38, 40, 28, 39, 40, 63, 53, 54, 52, 66, 43, 38, 44, 53, 56, 48, 43, 60, 63, 45, 48, 55, 79, 59, 66, 63, 51, 58, 79, 106, 82, 83, 88, 76, 97, 85, 187, 198, 220, 292, 361, 440, 454, 553, 589, 625, 685, 778, 751, 782, 748, 686, 711, 617, 461, 383, 405, 339, 269, 219, 188, 117, 117, 71, 70, 68, 76, 87, 101, 89, 68, 99, 87, 57, 79, 88, 63, 17, 53, 97, 74, 81, 57, 69, 43, 64, 42, 90, 82, 48, 80, 55, 57, 47, 50, 40, 38, 25, 52, 36, 52, 18, 48, 33, 42, 23, 23, 24, 31, 33, 53, 18, 16, 34, 42, 41, 40, 58, 49, 67, 52, 49, 43, 94, 74, 65, 86, 67, 100, 95, 101, 64, 64, 62, 101, 90, 77, 46, 60, 43, 56, 51, 59, 45, 54, 31, 43, 47, 41, 64, 64, 56, 78, 80, 87, 125, 138, 114, 120, 86, 78, 99, 145, 141, 180, 222, 251, 288, 361, 470, 548, 638, 784, 883, 886, 988, 1081, 1151, 1155, 1159, 1033, 1004, 973, 945, 841, 738, 662, 504, 411, 332, 292, 206, 184, 167, 151, 120, 96, 66, 114, 104, 90, 92, 123, 95, 76, 97, 91, 105, 136, 128, 111, 124, 135, 106, 142, 164, 179, 184, 206, 253, 309, 377, 476, 582, 734, 830, 1073, 1311, 1395, 1538, 1600, 1728, 1938, 1851, 1745, 1833, 1687, 1474, 1233, 1153, 1026, 855, 695, 563, 434, 395, 307, 257, 248, 174, 105, 121, 102, 125, 78, 99, 106, 73, 108, 78, 66, 99, 91, 73, 48, 31, 33, 27, 36, 70, 75, 94, 165, 138, 180, 222, 265, 324, 415, 416, 487, 530, 596, 724, 725, 739, 689, 689, 725, 693, 618, 567, 512, 454, 413, 336, 253, 222, 193, 159, 149, 139, 95, 114, 118, 93, 95, 87, 70, 102, 93, 62, 49, 43, 58, 24, 42, 68, 57, 38, 59, 44, 89, 64, 55, 70, 70, 62, 59, 47, 52, 54, 59, 77, 80, 105, 69, 63, 59, 72, 81, 82, 78, 46, 78, 58, 53, 61, 54, 43, 55, 66, 66, 55, 41, 54, 39, 66, 52, 68, 78, 44, 31, 22, 52, 25, 49, 68, 34, 59, 57, 76, 37, 20, 29, 26, 48, 40, 24, 20, 34, 28, 31, 47, 24, 52, 51, 42, 55, 63, 46, 67, 47, 62, 50, 51, 34, 52, 66, 57, 52, 69, 70, 49, 71, 62, 66, 59, 52, 51, 59, 41, 59, 63, 84, 82, 84, 59, 75, 96, 84, 96, 93, 73, 86, 115, 97, 83, 70, 89, 82, 79, 102, 84, 82, 96, 86, 69, 75, 79, 53, 63, 89, 106, 109, 91, 98, 109, 113, 118, 157, 127, 96, 116, 111, 147, 147, 167, 179, 222, 217, 268, 352, 419, 477, 648, 786, 770, 934, 1059, 1212, 1285, 1298, 1331, 1367, 1303, 1259, 1220, 1079, 995, 791, 650, 561, 478, 388, 327, 290, 255, 205, 142, 117, 121, 120, 75, 94, 105, 66, 95, 61, 96, 75, 79, 103, 98, 96, 121, 104, 122, 102, 105, 117, 106, 102, 97, 109, 73, 87, 89, 84, 89, 80, 102, 99, 107, 95, 98, 79, 90, 81, 74, 57, 80, 90, 85, 63, 61, 88, 82, 93, 103, 85, 123, 127, 97, 134, 118, 164, 118, 134, 160, 167, 235, 247, 283, 345, 379, 444, 566, 589, 666, 789, 913, 925, 907, 931, 1052, 1050, 955, 877, 900, 777, 656, 573, 526, 433, 325, 317, 311, 269, 228, 206, 215, 184, 181, 166, 191, 188, 272, 278, 329, 361, 434, 477, 505, 627, 687, 773, 861, 857, 865, 812, 828, 803, 763, 785, 742, 658, 684, 652, 673, 694, 680, 747, 811, 883, 828, 832, 829, 775, 760, 765, 722, 682, 624, 565, 482, 412, 346, 295, 194, 159, 146, 177, 145, 106, 64, 96, 67, 65, 50, 39, 74, 85, 44, 51, 35, 55, 55, 58, 46, 67, 74, 70, 77, 108, 96, 154, 175, 180, 193, 251, 302, 369, 416, 416, 555, 564, 657, 643, 660, 648, 701, 647, 615, 596, 570, 550, 497, 444, 394, 328, 259, 218, 179, 151, 131, 113, 100, 110, 128, 85, 69, 78, 56, 82, 97, 93, 80, 61, 68, 69, 70, 85, 54, 62, 54, 79, 96, 69, 53, 45, 40, 39, 51, 61, 73, 74, 67, 65, 73, 62, 59, 78, 81, 72, 89, 79, 53, 63, 64, 56, 35, 54, 57, 56, 61, 50, 67, 30, 48, 63, 35, 23, 41, 60, 51, 59, 55, 58, 62, 41, 78, 39, 56, 38, 50, 69, 77, 38, 45, 53, 78, 60, 50, 63, 56, 78, 73, 122, 130, 158, 157, 144, 159, 262, 284, 377, 417, 442, 559, 709, 798, 873, 962, 1046, 1083, 1155, 1038, 1071, 1037, 941, 897, 809, 712, 636, 536, 459, 376, 324, 272, 242, 201, 146, 147, 153, 118, 134, 168, 162, 220, 256, 288, 308, 352, 476, 579, 625, 674, 660, 630, 691, 700, 654, 607, 532, 466, 457, 417, 416, 341, 265, 243, 176, 160, 138, 139, 110, 108, 82, 104, 108, 90, 49, 61, 67, 51, 59, 40, 83, 50, 55, 13, 49, 46, 84, 82, 74, 58, 80, 91, 106, 113, 116, 103, 122, 91, 105, 121, 103, 125, 126, 87, 94, 117, 126, 145, 173, 143, 156, 174, 180, 261, 302, 274, 353, 489, 546, 560, 645, 675, 761, 807, 870, 857, 826, 791, 724, 697, 616, 569, 488, 410, 391, 347, 255, 220, 185, 118, 136, 145, 104, 87, 99, 69, 60, 65, 84, 91, 48, 45, 29, 39, 87, 73, 71, 63, 54, 38, 39, 42, 41, 37, 31, 22, 34, 36, 23, 37, 33, 26, 30, 36, 30, 44, 48, 37, 18, 30, 58, 45, 41, 50, 74, 54, 29, 51, 84, 74, 42, 61, 116, 71, 66, 99, 98, 120, 113, 156, 183, 188, 198, 212, 223, 252, 338, 423, 497, 604, 736, 835, 947, 1062, 1235, 1222, 1325, 1366, 1364, 1330, 1366, 1187, 1130, 1131, 987, 808, 713, 581, 555, 455, 377, 323, 235, 207, 174, 145, 135, 111, 99, 123, 101, 81, 85, 73, 111, 85, 87, 71, 78, 60, 66, 71, 59, 86, 82, 75, 88, 91, 147, 135, 161, 207, 256, 293, 304, 405, 474, 492, 510, 583, 648, 678, 670, 704, 731, 736, 696, 746, 616, 640, 613, 628, 695, 677, 660, 658, 783, 800, 818, 832, 840, 812, 762, 756, 801, 661, 652, 722, 662, 672, 708, 722, 712, 752, 752, 814, 966, 939, 938, 821, 758, 793, 697, 661, 635, 505, 414, 342, 305, 294, 215, 166, 159, 126, 124, 105, 114, 128, 82, 91, 101, 71, 42, 70, 74, 37, 60, 88, 76, 75, 99, 103, 81, 80, 93, 87, 81, 84, 58, 50, 40, 36, 47, 36, 14, 7, 39, 27, 33, 38, 38, 34, 39, 31, 47, 29, 26, 16, 47, 26, 43, 31, 34, 24, 33, 40, 43, 49, 25, 22, 42, 26, 57, 26, 30, 47, 29, 17, 20, 21, 33, 29, 26, 55, 49, 33, 47, 26, 56, 56, 38, 72, 60, 46, 65, 34, 62, 80, 69, 32, 52, 54, 48, 34, 42, 38, 45, 36, 57, 50, 32, 2, 12, 35, 18, 6, 27, 33, 29, 51, 18, 33, 30, 32, 24, 4, 32, 41, 15, 7, 31, 69, 18, 2, 29, 44, 46, 25, 11, -2, 18, 33, 7, 10, 56, 31, 37, 12, 36, 32, 34, 66, 23, 9, 37, 42, 47, 33, 63, 60, 34, 60, 71, 49, 48, 39, 80, 55, 41, 63, 63, 56, 66, 51, 47, 56, 45, 63, 57, 35, 43, 55, 53, 41, 51, 56, 56, 35, 48, 31, 57, 42, 45, 38, 59, 20, 27, 65, 28, 65, 42, 39, 42, 21, 16, 35, 43, 27, 10, 9, 29, 16, 45, 30, 25, 18, 36, 23, 43, 44, 63, 25, 54, 61, 60, 56, 35, 36, 46, 54, 29, 30, 12, 34, 43, 37, 52, 45, 43, 44, 64, 57, 66, 46, 70, 28, 6, 38, 59, 23, 36, 29, 27, 73, 41, 20, 10, 18, 43, 32, 36, 11, 26, 31, 39, 32, 36, 17, 27, 23, 48, 35, 43, 22, 71, 64, 83, 53, 60, 43, 17, 43, 57, 43, 72, 57, 44, 55, 41, 39, 37, 31, 55, 76, 70, 68, 88, 43, 73, 73, 89, 111, 57, 61, 52, 89, 112, 59, 86, 114, 92, 69, 96, 112, 144, 171, 191, 223, 258, 285, 330, 392, 496, 527, 570, 607, 637, 742, 727, 755, 789, 760, 749, 764, 690, 590, 518, 494, 431, 364, 352, 302, 244, 229, 268, 237, 194, 195, 176, 214, 196, 230, 308, 366, 392, 445, 500, 600, 603, 661, 724, 819, 802, 837, 802, 791, 803, 704, 626, 590, 541, 458, 393, 298, 309, 277, 246, 146, 95, 112, 97, 66, 65, 79, 55, 61, 68, 45, 39, 40, 60, 52, 42, 3, 32, 47, 41, 39, 26, 41, 54, 38, 1, 16, 29, 34, 14, 21, 32, 24, 44, 38, 49, 51, 46, 40, 48, 73, 94, 94, 85, 136, 176, 178, 212, 256, 295, 380, 444, 497, 522, 511, 543, 606, 640, 665, 728, 667, 601, 585, 526, 472, 411, 367, 273, 261, 218, 185, 156, 132, 123, 98, 81, 67, 65, 30, 53, 33, 41, 53, 48, 68, 69, 75, 64, 54, 44, 45, 50, 50, 51, 43, 44, 46, 45, 61, 30, 45, 50, 74, 38, 42, 54, 56, 54, 37, 49, 15, 48, 48, 50, 25, 6, 12, 30, 35, 31, 31, 26, 21, 24, 38, 40, 27, 29, 42, 51, 55, 33, 65, 59, 50, 39, 62, 80, 74, 26, 36, 24, 16, 20, 33, 21, 50, 41, 40, 46, 64, 29, 36, 54, 34, 36, 35, 57, 70, 73, 60, 46, 50, 40, 22, 39, 38, 64, 59, 66, 53, 35, 39, 44, 44, 68, 58, 43, 58, 53, 51, 58, 47, 64, 40, 44, 37, 29, 29, 52, 59, 59, 59, 53, 42, 33, 58, 45, 48, 47, 36, 16, 32, 67, 41, 53, 50, 29, 27, 16, 71, 52, 14, 48, 51, 28, 44, 51, 52, 51, 58, 30, 51, 48, 50, 43, 46, 46, 52, 60, 78, 77, 78, 70, 63, 74, 45, 75, 66, 76, 67, 80, 85, 78, 106, 61, 78, 61, 74, 75, 83, 84, 74, 79, 64, 60, 66, 75, 51, 43, 50, 41, 38, 49, 45, 58, 54, 83, 49, 92, 55, 88, 70, 80, 76, 79, 73, 90, 131, 152, 157, 241, 259, 278, 328, 326, 384, 432, 501, 603, 679, 696, 808, 869, 930, 925, 863, 849, 889, 825, 660, 599, 573, 518, 434, 410, 371, 332, 265, 244, 229, 160, 200, 198, 184, 212, 250, 287, 321, 378, 392, 481, 607, 650, 727, 784, 946, 1024, 1145, 1204, 1250, 1240, 1107, 1074, 1187, 1055, 1014, 961, 841, 679, 625, 572, 492, 401, 335, 246, 227, 223, 188, 149, 124, 109, 106, 107, 92, 93, 80, 82, 72, 85, 54, 69, 73, 72, 52, 61, 80, 96, 108, 122, 129, 161, 195, 189, 242, 274, 376, 386, 409, 468, 537, 623, 594, 622, 623, 628, 582, 555, 480, 452, 390, 383, 331, 306, 252, 206, 174, 147, 151, 114, 99, 85, 59, 46, 34, 39, 50, 41, 11, 55, 63, 58, 51, 31, 37, 43, 43, 60, 30, 29, 51, 48, 45, 53, 32, 51, 42, 75, 58, 28, 27, 76, 32, 51, 54, 42, 43, 62, 52, 40, 58, 60, 61, 47, 56, 82, 91, 81, 85, 80, 112, 101, 107, 128, 136, 129, 183, 169, 182, 177, 186, 219, 319, 359, 378, 451, 530, 577, 658, 800, 863, 902, 1073, 1138, 1219, 1282, 1147, 1184, 1184, 1119, 1047, 893, 846, 744, 607, 555, 508, 427, 333, 277, 240, 234, 222, 152, 151, 122, 119, 77, 98, 92, 32, 63, 58, 18, 72, 51, 29, 35, 55, 47, 46, 34, 48, 28, 13, 60, 68, 60, 81, 60, 58, 62, 84, 57, 54, 79, 78, 99, 91, 119, 113, 148, 192, 209, 223, 247, 276, 294, 366, 373, 398, 439, 456, 519, 562, 537, 475, 431, 457, 414, 399, 334, 307, 277, 276, 221, 188, 127, 104, 78, 85, 70, 68, 75, 45, 37, 60, 30, 33, 13, 38, 49, 18, 32, 35, 38, 36, 7, 17, 9, 31, 21, 29, -4, 49, 7, 20, 33, 26, 30, 26, 33, 54, 75, 68, 70, 71, 55, 70, 79, 80, 121, 91, 111, 83, 65, 64, 59, 73, 80, 74, 72, 52, 83, 45, 66, 82, 77, 50, 48, 38, 57, 58, 61, 28, 27, 37, 40, 36, 32, 33, 51, 51, 71, 80, 76, 96, 84, 109, 133, 123, 149, 182, 236, 230, 260, 341, 382, 407, 435, 455, 483, 546, 511, 464, 475, 489, 459, 443, 427, 361, 328, 285, 254, 277, 237, 217, 166, 169, 161, 184, 177, 196, 201, 202, 253, 296, 391, 378, 475, 509, 568, 767, 885, 862, 979, 965, 1042, 1105, 1139, 1134, 1199, 1087, 1084, 961, 843, 822, 759, 666, 560, 467, 424, 345, 318, 262, 275, 183, 173, 185, 187, 164, 229, 212, 238, 284, 320, 343, 395, 428, 447, 466, 477, 528, 579, 539, 482, 418, 489, 390, 338, 313, 303, 303, 273, 206, 191, 156, 125, 96, 73, 65, 56, 31, 58, 65, 88, 86, 71, 80, 78, 85, 72, 73, 56, 56, 81, 67, 73, 89, 58, 92, 64, 95, 118, 94, 72, 77, 77, 56, 89, 75, 46, 64, 87, 56, 79, 76, 80, 51, 62, 66, 61, 85, 77, 73, 61, 68, 21, 61, 46, 54, 59, 69, 69, 55, 54, 54, 54, 59, 60, 39, 43, 47, 56, 37, 31, 19, 49, 47, 55, 74, 38, 32, 90, 100, 76, 50, 74, 97, 111, 83, 86, 74, 124, 116, 120, 109, 98, 90, 102, 127, 92, 106, 125, 156, 169, 155, 187, 246, 304, 330, 373, 400, 402, 467, 499, 553, 598, 599, 554, 584, 556, 487, 499, 411, 440, 376, 351, 282, 239, 200, 234, 175, 152, 148, 114, 57, 60, 78, 65, 39, 52, 46, 66, 39, 45, 41, 19, 66, 54, 61, 41, 37, 53, 50, 22, 20, 30, 51, 49, 52, 16, 59, 42, 63, 51, 47, 59, 72, 46, 45, 45, 31, 53, 30, 52, 37, 40, 53, 68, 54, 60, 35, 56, 63, 41, 48, 46, 61, 54, 53, 71, 52, 53, 45, 65, 71, 42, 63, 58, 43, 58, 52, 50, 47, 39, 57, 65, 76, 60, 56, 115, 82, 86, 79, 52, 46, 52, 49, 90, 106, 74, 68, 67, 90, 54, 98, 113, 120, 166, 174, 177, 194, 241, 264, 306, 358, 413, 508, 531, 526, 620, 574, 575, 577, 564, 573, 544, 438, 412, 363, 336, 336, 299, 318, 274, 244, 217, 233, 196, 185, 198, 174, 171, 195, 205, 260, 295, 384, 410, 490, 480, 485, 592, 616, 571, 603, 611, 602, 582, 511, 518, 499, 461, 397, 371, 332, 271, 248, 263, 201, 183, 181, 157, 187, 215, 237, 255, 271, 260, 256, 304, 371, 427, 476, 457, 449, 465, 501, 555, 492, 544, 544, 562, 556, 539, 535, 556, 574, 542, 528, 555, 566, 568, 529, 520, 554, 552, 565, 548, 596, 535, 492, 480, 444, 344, 341, 296, 275, 237, 202, 193, 186, 144, 123, 140, 114, 73, 105, 104, 69, 90, 62, 103, 73, 92, 83, 112, 115, 106, 87, 120, 141, 137, 117, 169, 147, 158, 219, 203, 222, 297, 283, 307, 353, 390, 462, 524, 589, 641, 702, 718, 714, 737, 768, 700, 703, 676, 631, 584, 575, 534, 476, 365, 355, 344, 287, 241, 223, 193, 158, 171, 133, 91, 99, 99, 83, 95, 91, 89, 80, 56, 62, 59, 61, 101, 77, 64, 70, 90, 75, 79, 86, 95, 64, 53, 77, 87, 75, 56, 70, 60, 54, 71, 78, 60, 67, 56, 63, 91, 80, 71, 90, 74, 49, 59, 47, 54, 87, 85, 97, 56, 58, 66, 74, 46, 89, 93, 74, 58, 90, 76, 61, 82, 82, 100, 107, 64, 99, 89, 99, 127, 147, 172, 162, 186, 199, 283, 311, 365, 498, 530, 542, 581, 712, 801, 839, 885, 919, 918, 1022, 1075, 1045, 1080, 1044, 1072, 1125, 1093, 1080, 1047, 981, 921, 933, 894, 835, 822, 781, 760, 729, 737, 669, 663, 612, 568, 527, 516, 447, 391, 340, 341, 277, 198, 173, 156, 131, 154, 140, 105, 107, 111, 124, 121, 94, 103, 82, 92, 95, 81, 98, 104, 138, 141, 141, 131, 195, 160, 162, 216, 224, 288, 317, 344, 423, 524, 570, 639, 686, 719, 763, 852, 859, 937, 906, 843, 851, 767, 733, 716, 718, 597, 546, 554, 448, 337, 366, 367, 309, 280, 263, 248, 311, 332, 307, 314, 335, 392, 491, 604, 606, 609, 828, 852, 913, 964, 965, 1053, 1103, 1098, 1110, 1119, 1074, 983, 1008, 1017, 934, 943, 928, 988, 910, 913, 880, 838, 882, 871, 849, 826, 765, 733, 689, 663, 576, 492, 409, 415, 323, 313, 295, 260, 183, 161, 179, 161, 117, 117, 87, 73, 97, 83, 77, 57, 70, 49, 67, 84, 59, 48, 54, 84, 51, 53, 99, 41, 58, 27, 44, 29, 54, 41, 75, 97, 78, 68, 106, 113, 120, 134, 141, 209, 191, 219, 288, 308, 364, 417, 449, 494, 576, 640, 628, 622, 668, 763, 803, 711, 648, 566, 564, 540, 471, 431, 407, 381, 344, 269, 261, 217, 207, 147, 161, 160, 200, 267, 213, 270, 289, 301, 303, 379, 440, 403, 445, 418, 425, 502, 455, 492, 470, 442, 447, 402, 342, 344, 320, 302, 255, 206, 178, 196, 173, 174, 121, 75, 103, 76, 76, 89, 79, 61, 42, 59, 67, 68, 68, 56, 95, 68, 66, 90, 79, 59, 53, 63, 69, 74, 51, 51, 53, 54, 62, 89, 48, 67, 63, 80, 78, 60, 56, 63, 89, 67, 81, 75, 97, 86, 77, 121, 87, 85, 81, 91, 135, 129, 125, 115, 147, 153, 159, 155, 156, 195, 239, 267, 334, 378, 464, 563, 614, 698, 686, 762, 911, 1030, 1020, 1155, 1071, 1120, 1112, 1021, 1089, 969, 954, 926, 812, 686, 605, 517, 484, 447, 364, 338, 257, 237, 197, 155, 143, 136, 114, 77, 87, 79, 82, 69, 78, 82, 76, 110, 94, 127, 92, 101, 105, 162, 163, 213, 227, 232, 249, 300, 331, 376, 468, 422, 481, 461, 506, 565, 563, 571, 552, 523, 510, 467, 442, 409, 345, 306, 288, 287, 240, 212, 170, 179, 117, 111, 122, 103, 75, 119, 100, 72, 85, 107, 108, 92, 105, 87, 95, 58, 72, 80, 105, 86, 105, 103, 123, 175, 196, 286, 299, 292, 311, 358, 436, 453, 407, 480, 431, 480, 477, 499, 444, 435, 411, 405, 346, 342, 288, 242, 223, 242, 207, 204, 189, 184, 174, 199, 218, 229, 209, 197, 246, 344, 336, 324, 397, 469, 442, 465, 518, 621, 557, 543, 537, 512, 531, 534, 441, 415, 374, 324, 266, 238, 244, 202, 199, 160, 113, 86, 93, 88, 91, 58, 33, 53, 70, 77, 47, 70, 72, 46, 77, 54, 60, 88, 81, 52, 81, 80, 93, 66, 81, 70, 74, 111, 116, 102, 75, 78, 102, 97, 89, 99, 108, 88, 99, 76, 78, 130, 105, 124, 97, 100, 76, 104, 89, 84, 95, 115, 114, 91, 73, 117, 109, 80, 55, 94, 77, 96, 83, 74, 85, 81, 101, 80, 60, 66, 79, 77, 81, 59, 87, 72, 77, 124, 95, 67, 53, 66, 89, 74, 44, 61, 61, 63, 64, 65, 44, 56, 65, 57, 64, 29, 53, 95, 115, 135, 101, 98, 111, 124, 127, 148, 148, 126, 169, 182, 234, 276, 299, 367, 392, 439, 514, 565, 667, 622, 668, 747, 873, 947, 902, 887, 820, 826, 786, 848, 757, 687, 649, 603, 583, 506, 400, 407, 345, 321, 292, 244, 238, 261, 271, 239, 267, 352, 360, 394, 444, 444, 519, 522, 555, 613, 639, 644, 584, 570, 548, 519, 515, 479, 448, 417, 389, 357, 316, 283, 264, 223, 210, 208, 184, 181, 161, 178, 200, 229, 259, 308, 336, 353, 344, 395, 386, 396, 385, 404, 434, 408, 391, 399, 391, 399, 411, 332, 273, 265, 246, 226, 161, 136, 143, 135, 94, 87, 93, 105, 104, 74, 53, 51, 62, 60, 58, 80, 64, 45, 53, 56, 64, 58, 93, 76, 61, 19, 64, 58, 65, 65, 74, 54, 43, 71, 73, 75, 68, 88, 85, 72, 68, 63, 67, 87, 83, 85, 58, 70, 91, 86, 60, 66, 74, 82, 80, 57, 79, 44, 74, 52, 75, 82, 63, 83, 64, 53, 56, 29, 52, 29, 48, 52, 74, 36, 70, 59, 50, 45, 32, 31, 42, 26, 23, 44, 34, 29, 12, 45, 62, 83, 71, 55, 21, 52, 41, 9, 53, 34, 39, 35, 26, 48, 41, 25, 76, 74, 58, 24, 40, 41, 27, 38, 26, 14, 45, 22, 50, 31, 39, 14, 30, 24, 34, 26, 50, 43, 14, 34, 51, 33, 34, 68, 47, 51, 80, 93, 62, 63, 74, 64, 70, 87, 80, 46, 56, 72, 55, 57, 49, 25, 49, 38, 87, 72, 62, 50, 35, 42, 31, 60, 40, 49, 75, 40, 49, 31, 39, 42, 47, 15, 28, 16, 36, 51, 32, 41, 50, 32, 26, 28, 58, 75, 66, 83, 89, 77, 72, 89, 86, 100, 97, 115, 141, 165, 178, 204, 182, 229, 206, 254, 289, 338, 434, 396, 408, 495, 561, 612, 758, 798, 759, 834, 868, 826, 776, 776, 805, 782, 778, 689, 555, 573, 512, 466, 456, 409, 357, 310, 301, 279, 291, 282, 307, 390, 370, 401, 441, 462, 488, 560, 589, 632, 651, 636, 625, 587, 658, 605, 500, 548, 545, 467, 412, 372, 355, 367, 320, 301, 251, 219, 204, 217, 249, 281, 273, 308, 301, 352, 401, 386, 484, 508, 500, 589, 683, 741, 787, 798, 791, 874, 809, 832, 811, 707, 728, 652, 608, 487, 473, 480, 473, 434, 425, 342, 246, 270, 266, 264, 271, 292, 303, 301, 302, 323, 356, 424, 425, 446, 486, 453, 469, 566, 545, 509, 473, 505, 446, 390, 402, 368, 335, 316, 263, 301, 258, 224, 171, 153, 154, 128, 121, 91, 115, 148, 119, 102, 79, 89, 99, 95, 125, 108, 88, 83, 100, 110, 68, 53, 70, 90, 72, 72, 59, 41, 55, 55, 40, 48, 68, 54, 64, 51, 23, 51, 53, 42, 58, 47, 67, 27, 31, 33, 26, 30, 29, 25, 41, 42, 44, 44, 104, 96, 46, 62, 73, 46, 31, 53, 53, 59, 38, 55, 62, 49, 33, 52, 39, 44, 60, 61, 77, 60, 59, 56, 53, 61, 71, 72, 26, 61, 42, 48, 30, 24, 49, 42, 60, 35, 39, 25, 59, 46, 57, 50, 50, 62, 59, 34, 53, 70, 66, 86, 75, 107, 125, 176, 171, 197, 216, 234, 264, 298, 291, 321, 400, 381, 342, 324, 414, 382, 410, 406, 345, 321, 300, 324, 274, 233, 200, 204, 184, 146, 131, 137, 106, 111, 88, 78, 68, 52, 69, 65, 44, 49, 57, 71, 48, 50, 52, 52, 31, 42, 46, 57, 41, 37, 47, 68, 68, 43, 41, 32, 35, 42, 28, 13, 26, 52, 60, 35, 39, 32, 57, 49, 37, 38, 43, 11, 33, 35, 38, 27, 8, 30, 54, 45, 35, 21, 19, 26, 24, 42, 10, 27, 36, 39, 43, 44, 56, 39, 42, 17, 70, 66, 84, 68, 80, 51, 94, 88, 67, 74, 63, 40, 56, 54, 82, 109, 71, 62, 86, 44, 55, 48, 23, 44, 61, 19, 41, 46, 51, 38, 45, 40, 39, 44, 44, 57, 59, 57, 49, 54, 60, 41, 54, 56, 64, 65, 71, 90, 97, 98, 83, 119, 146, 171, 178, 200, 217, 259, 284, 326, 348, 387, 384, 387, 411, 465, 496, 495, 519, 515, 496, 486, 466, 423, 381, 348, 321, 343, 314, 243, 244, 235, 175, 167, 135, 153, 152, 136, 111, 107, 109, 104, 102, 97, 81, 82, 71, 85, 72, 71, 68, 65, 90, 49, 60, 61, 73, 32, 44, 39, 60, 28, 42, 31, 57, 63, 60, 25, 39, 35, 43, 49, 50, 32, 14, 39, 16, 28, 21, 30, 32, 31, 4, 43, 30, 29, 32, 67, 66, 44, 62, 41, 42, 68, 58, 69, 43, 54, 66, 50, 24, 24, 56, 43, 59, 26, 45, 30, 38, 28, 29, 61, 41, 43, 60, 40, 41, 41, 37, 48, 37, 41, 64, 70, 67, 43, 39, 67, 51, 54, 75, 48, 38, 66, 67, 59, 41, 67, 78, 110, 90, 57, 35, 51, 61, 53, 50, 56, 63, 75, 77, 77, 59, 51, 49, 53, 61, 69, 57, 43, 61, 58, 40, 79, 71, 87, 115, 95, 115, 63, 131, 178, 205, 225, 227, 229, 297, 334, 358, 382, 439, 482, 481, 503, 501, 484, 508, 511, 552, 483, 440, 462, 446, 410, 346, 306, 275, 271, 262, 195, 203, 170, 161, 140, 126, 129, 122, 93, 110, 122, 119, 112, 98, 97, 97, 67, 102, 120, 106, 131, 149, 199, 218, 175, 179, 272, 276, 291, 353, 373, 393, 490, 368, 407, 408, 365, 372, 375, 396, 369, 315, 285, 321, 267, 233, 230, 189, 205, 171, 127, 120, 138, 83, 98, 126, 83, 88, 67, 70, 86, 79, 92, 84, 56, 40, 66, 56, 43, 85, 88, 81, 68, 63, 114, 78, 77, 72, 76, 89, 89, 96, 101, 105, 120, 145, 145, 160, 142, 179, 203, 258, 292, 300, 309, 338, 381, 373, 360, 379, 393, 416, 403, 408, 402, 348, 358, 395, 345, 287, 237, 233, 251, 250, 263, 210, 219, 160, 187, 205, 212, 252, 239, 280, 292, 324, 384, 356, 349, 424, 407, 424, 448, 494, 461, 466, 484, 434, 375, 330, 317, 317, 284, 248, 227, 177, 160, 131, 101, 108, 106, 75, 71, 66, 62, 49, 44, 51, 39, 37, 65, 30, 43, 50, 32, 41, 24, 40, 46, 40, 57, 44, 41, 30, 31, 67, 51, 29, 51, 34, 41, 26, 17, 46, 13, 22, 43, 42, 53, 34, 58, 46, 49, 35, 68, 43, 47, 35, 51, 56, 74, 44, 48, 58, 81, 71, 72, 73, 74, 67, 90, 103, 83, 84, 103, 113, 116, 142, 138, 140, 186, 218, 218, 205, 234, 270, 338, 338, 312, 397, 426, 464, 458, 453, 421, 445, 389, 372, 364, 407, 367, 332, 289, 274, 278, 264, 244, 230, 232, 228, 186, 174, 202, 217, 201, 217, 249, 300, 336, 333, 351, 378, 405, 423, 385, 379, 456, 492, 476, 501, 544, 566, 554, 585, 529, 580, 598, 536, 555, 541, 486, 450, 523, 529, 450, 408, 433, 356, 373, 350, 310, 259, 213, 243, 236, 180, 140, 147, 100, 73, 91, 101, 76, 50, 51, 44, 53, 54, 52, 33, 51, 38, 40, 65, 61, 20, 36, 65, 76, 44, 62, 52, 39, 39, 39, 39, 64, 71, 71, 37, 30, 38, 60, 82, 62, 57, 63, 66, 58, 40, 30, 35, 46, 54, 46, 9, 27, 32, 22, 32, 30, 43, 31, 9, 37, 18, 43, 32, 39, 28, 32, 41, 63, 40, 39, 56, 51, 24, 37, 59, 80, 82, 34, 52, 61, 53, 82, 94, 78, 106, 111, 104, 134, 124, 152, 179, 194, 210, 257, 329, 328, 335, 374, 394, 418, 474, 436, 417, 427, 389, 388, 364, 370, 305, 335, 319, 300, 247, 238, 183, 143, 130, 123, 141, 139, 147, 133, 91, 58, 63, 66, 78, 73, 60, 74, 52, 68, 43, 50, 55, 49, 32, 36, 32, 56, 66, 64, 50, 56, 78, 67, 85, 70, 83, 89, 113, 110, 136, 184, 203, 156, 170, 228, 256, 230, 297, 349, 342, 361, 300, 317, 338, 387, 389, 358, 342, 396, 339, 301, 305, 275, 270, 235, 216, 193, 141, 116, 130, 132, 117, 89, 57, 62, 65, 84, 75, 31, 61, 63, 43, 53, 39, 64, 82, 41, 65, 66, 78, 77, 79, 78, 64, 59, 85, 89, 82, 98, 76, 93, 83, 96, 88, 115, 136, 162, 155, 177, 184, 211, 256, 290, 360, 380, 363, 393, 445, 446, 427, 437, 445, 452, 412, 400, 459, 438, 380, 363, 330, 304, 291, 259, 187, 195, 198, 199, 160, 124, 100, 80, 88, 98, 76, 83, 67, 71, 88, 43, 58, 38, 33, 35, 44, 30, 50, 38, 35, 47, 44, 39, 25, 31, 22, 39, 25, 32, 36, 21, 37, 36, 57, 43, 57, 43, 35, 29, 39, 24, 36, 56, 54, 29, 39, 52, 62, 75, 39, 51, 35, 55, 58, 51, 66, 67, 82, 65, 112, 103, 104, 142, 152, 136, 180, 200, 220, 225, 287, 326, 272, 406, 425, 408, 421, 484, 528, 483, 550, 533, 560, 555, 526, 609, 591, 513, 526, 479, 484, 356, 386, 399, 366, 285, 319, 319, 300, 278, 310, 288, 278, 291, 309, 356, 345, 380, 429, 400, 372, 348, 434, 414, 347, 312, 333, 304, 281, 272, 274, 257, 202, 201, 164, 157, 175, 149, 152, 127, 100, 109, 96, 107, 60, 77, 67, 63, 85, 80, 50, 51, 52, 27, 41, 58, 88, 72, 66, 76, 79, 64, 78, 91, 94, 103, 78, 97, 91, 129, 132, 178, 181, 217, 238, 247, 285, 336, 351, 363, 352, 469, 470, 473, 465, 462, 473, 490, 561, 561, 501, 485, 467, 475, 518, 521, 388, 327, 310, 256, 256, 216, 203, 173, 141, 133, 146, 120, 104, 87, 76, 59, 49, 62, 60, 50, 35, 43, 48, 23, 36, 43, 29, 0, 39, 35, 22, 23, 28, 21, 8, 18, 46, 35, 29, 44, 15, 38, -7, 11, 26, 29, 8, 7, 12, 19, 40, 21, -3, 6, 24, 9, 15, 29, 39, 23, 22, 25, 21, 15, 28, 3, 8, 37, 37, 32, 53, 36, 32, 50, 59, 57, 45, 20, 11, 13, 33, 42, 25, 50, 30, 35, 28, 59, 48, 18, 25, 28, 18, 16, 13, 30, 31, 31, 33, 31, 40, 45, 33, 19, 12, 19, 45, 43, 31, 2, 44, 31, 25, 18, 46, 77, 56, 31, 31, 28, 15, 30, 59, 63, 62, 69, 42, 38, 50, 58, 37, 38, 56, 15, 52, 22, 38, 31, 35, 36, 38, 43, 27, 34, 54, 18, 43, 34, 38, 33, 24, 32, 37, 45, 19, 26, 40, 55, 31, 44, 53, 57, 68, 58, 93, 86, 89, 92, 123, 141, 105, 142, 159, 177, 192, 209, 244, 261, 301, 322, 336, 346, 352, 344, 415, 417, 389, 410, 393, 319, 330, 332, 319, 333, 277, 247, 219, 226, 176, 160, 160, 131, 126, 118, 98, 104, 98, 104, 83, 84, 119, 91, 95, 115, 137, 147, 134, 165, 177, 179, 213, 224, 221, 235, 285, 245, 297, 310, 291, 335, 306, 345, 341, 407, 368, 351, 355, 342, 302, 226, 277, 258, 264, 251, 229, 204, 235, 254, 219, 200, 189, 260, 275, 266, 320, 331, 359, 334, 322, 387, 373, 313, 361, 396, 365, 385, 353, 313, 275, 284, 280, 227, 257, 222, 181, 177, 153, 135, 118, 114, 113, 79, 106, 83, 39, 38, 56, 59, 33, 46, 49, 61, 27, 25, 18, 21, 21, 6, 17, 45, 46, 39, 22, 8, 20, 4, 50, 12, -6, 8, 12, 6, 9, 13, 22, 6, 16, 17, 11, 11, 25, -6, 17, 27, 5, 11, 25, 16, 10, 16, 34, 22, 6, 16, 13, 17, 25, 54, 12, 1, 25, 31, 6, 13, -5, -1, 30, 22, 29, 38, 35, 40, 17, 31, 41, 6, 28, 26, 28, 2, 43, 41, 15, 22, 19, 48, 44, 21, 57, 35, 30, 60, 40, 46, 36, 37, 58, 60, 39, 58, 62, 31, 58, 57, 48, 73, 64, 60, 55, 49, 71, 61, 11, 46, 29, 31))
        self.assertEqual(record.annotations["abif_raw"]['DATA3'], (16, -8, 7, -21, 5, 0, 12, 4, -7, 6, -7, 25, 25, -6, -2, 7, 1, -3, 1, 0, -22, -8, -6, 27, -13, 8, -11, -8, 11, 19, 10, 11, 3, -5, 9, 6, -15, 33, 0, 3, -12, 3, -18, 1, -11, -21, -8, 2, 0, -14, 1, 7, 8, 8, 17, 0, 8, -4, 3, 5, 1, -8, -1, 7, 15, -2, 6, 7, 4, 14, 12, -3, -5, -4, -5, 19, 18, 18, 0, 12, 12, 6, 11, 5, 1, -3, 2, 11, 0, 7, 14, -4, -4, 0, -2, 3, 5, 17, 0, 8, -2, 11, 22, 22, -23, -11, 14, 10, 14, -8, 2, 9, 8, -5, -8, 11, 21, 5, -5, 9, -2, -1, 6, 15, -20, 5, 12, 19, -8, 26, 21, 27, 18, 9, 15, -1, 0, 4, 17, 25, 4, 4, -11, 14, 0, 20, 32, 4, 8, 4, -7, 15, 2, 11, -1, 8, -2, -3, 0, 14, 14, 29, -6, 2, 19, -5, 32, -7, 33, -11, 1, -13, 18, 28, 13, 22, 22, 16, 21, 25, 11, 20, 14, 40, 29, 16, 3, 24, 21, 13, 37, 25, 9, 7, 37, 6, 27, 28, 16, 43, 26, 16, 20, 11, 15, 2, 7, 14, 28, 10, 38, 16, 19, 18, 40, 0, 2, 5, 23, 30, 14, 23, 21, 23, 22, 18, 31, 23, 32, 16, 28, 15, 21, 50, 14, 23, 21, 15, 0, -1, 3, 6, 28, 27, 1, 17, 9, 20, 9, 20, 16, 21, 15, 7, 13, 6, 21, 7, 13, 6, 24, 29, 21, 17, 22, -2, 27, 24, 2, 47, 30, 22, 5, 18, 25, 17, 0, 37, 26, 4, 36, 11, 17, 26, 19, 24, 29, 26, 22, 29, 16, 6, 29, 14, 30, 25, 18, 2, 11, 30, 12, 18, 20, 37, 19, -3, 12, 20, 22, 30, 22, -1, 37, 30, 20, 30, 35, 39, 20, 18, 30, 22, 21, 20, 25, 21, 33, 21, -6, 29, 11, 3, 15, 11, 19, 15, 25, 5, 13, 14, 40, 28, 23, 6, 3, 32, 22, 17, 15, 33, 16, 25, 8, 29, 1, 18, 18, 41, -1, 19, 31, 12, 19, 13, 16, 21, 4, 39, 9, 17, 13, 22, 27, 7, 33, 13, 18, 44, 17, 26, 37, 34, 42, 31, 15, 32, 33, 24, 14, 17, 30, 16, 18, 41, 12, 44, 29, 34, 52, 31, 27, 30, 16, 22, 35, 45, 16, 16, 26, 30, 26, 26, 24, 23, 25, 32, 21, 15, 7, 11, 28, 34, 5, 22, 18, 40, 16, 7, 30, 2, 18, 6, 17, 23, 29, 17, 40, 38, 8, 12, 23, 9, -2, 33, 28, 21, 20, 30, 23, 39, 24, 7, 16, 33, 28, 31, 18, 14, 33, 18, 24, 30, 22, 33, 15, 19, 26, 24, 26, 27, 38, 15, 32, 24, 41, 38, 24, 38, 16, 24, 24, 3, 21, 19, 21, 11, 36, 24, 24, 58, 29, 25, 30, 35, 25, 27, 16, 38, 30, 19, 26, 39, 24, 6, 16, 27, 18, 26, 7, 24, 25, 27, 10, 26, 24, 36, 45, 7, 39, 37, 17, 23, 18, 4, 33, 37, 48, 50, 15, 20, 34, 12, 42, 40, 23, 16, 18, 25, 33, 42, 15, 29, 34, 32, 46, 28, 51, 27, 42, 38, 26, 20, 24, 1, 21, 38, 19, 23, 35, 25, 32, 26, 22, 20, 33, 50, 0, 5, 18, 17, 46, 49, 59, 26, 41, 12, 26, 30, 16, 21, 19, 3, 6, 33, 25, 26, 14, 34, 22, 41, 18, 14, 28, 10, 38, 9, 20, 17, 15, 27, 20, 27, 30, 50, 42, 38, 29, 17, 37, 44, 20, 32, 13, 26, 17, 42, 26, 36, 22, 25, 15, 9, 22, -4, 43, 27, 10, 40, 34, 17, 23, 26, 38, 35, 44, 32, 33, 33, 29, 52, 30, 30, 48, 0, 46, 36, 7, 30, 17, 31, 37, 37, 8, 25, 42, 51, 35, 23, 35, 28, 10, 30, 25, 10, 33, 35, 34, 19, 18, 12, 23, 9, 30, 25, 21, 15, 27, 29, 27, 26, 3, 35, 1, 24, 27, 10, 5, 41, 37, 30, 33, 34, 14, 17, 12, 25, 33, 30, 36, 20, 19, 33, 17, 22, 32, 22, 17, 21, 23, 16, 0, 12, 29, 24, 23, 20, 37, 52, 33, 22, 27, 2, -2, 37, 36, 49, 30, 17, 37, 24, 11, 7, 20, 16, 19, 21, 13, 16, 21, 24, 38, 38, 20, 7, 34, 15, 33, 49, 5, 6, 25, 43, 17, 0, 27, 34, 4, 25, 14, 23, 28, 27, 35, 32, 46, 44, 6, 22, 30, 23, 25, 34, 42, 29, 47, 34, 17, 13, 13, 37, 19, 38, 13, 39, 15, 48, 32, 34, 22, 37, 49, 32, 23, 27, 17, 57, 43, 29, 37, 29, 32, 42, 62, 77, 56, 71, 54, 77, 81, 98, 83, 116, 125, 137, 156, 167, 137, 150, 138, 129, 87, 84, 108, 77, 81, 69, 56, 61, 43, 60, 47, 56, 24, 50, 28, 31, 41, 40, 49, 39, 39, 53, 26, 35, 27, 31, 22, 31, 18, 39, 30, 37, 50, 30, 26, 37, 34, 40, 31, 44, 40, 41, 49, 30, 52, 45, 45, 31, 20, 26, 47, 48, 26, 46, 54, 32, 43, 39, 40, 30, 11, 25, 47, 67, 34, 35, 22, 25, 25, 61, 17, 52, 41, 40, 40, 32, 37, 20, 39, 49, 51, 27, 26, 29, 23, 21, 34, 29, 29, 23, 33, 56, 51, 31, 45, 29, 46, 41, 34, 37, 50, 31, 50, 41, 21, 49, 37, 35, 33, 32, 27, 26, 45, 42, 28, 49, 25, 57, 35, 20, 37, 49, 23, 34, 24, 33, 1, 54, 30, 27, 42, 17, 36, 11, 44, 24, 36, 17, 33, 44, 23, 43, 38, 20, 58, 40, 32, 19, 31, 19, 59, 38, 46, 35, 24, 48, 37, 26, 32, 25, 45, 31, 28, 32, 17, 44, 51, 11, 30, 32, 51, 38, 25, 33, 58, 26, 21, 21, 34, 37, 20, 25, 43, 14, 43, 21, 37, 14, 37, 36, 39, 39, 54, 24, 22, 52, 12, 33, 30, 47, 33, 35, 27, 30, 15, 37, 37, 15, 36, 20, 33, 22, 27, 41, 36, 21, 17, 25, 20, 22, 25, 43, 19, 24, 23, 24, 16, 23, 44, 36, 32, 16, 37, 22, 30, 39, 37, 2, 24, 36, 9, 5, 23, 22, 17, 7, 25, 23, 52, 20, 23, 54, 14, 27, 30, 24, 30, 36, 39, 15, 21, 45, 13, 30, 20, 38, 30, 23, 25, 41, 25, 13, 29, 24, 33, 39, 37, 51, 22, 29, 44, 57, 35, 17, 39, 34, 45, 41, 24, 34, 33, 42, 27, 36, 42, 25, 36, 35, 42, 61, 50, 49, 44, 34, 38, 57, 22, 25, 46, 55, 46, 49, 33, 45, 55, 31, 43, 42, 45, 34, 49, 36, 48, 58, 37, 49, 36, 26, 46, 29, 52, 56, 28, 46, 48, 48, 71, 59, 57, 51, 56, 50, 47, 57, 51, 46, 65, 31, 25, 30, 37, 63, 29, 19, 31, 33, 55, 51, 48, 28, 44, 27, 42, 26, 28, 41, 42, 71, 42, 53, 37, 58, 49, 50, 56, 56, 44, 50, 60, 42, 68, 41, 26, 64, 42, 42, 51, 12, 35, 47, 48, 26, 42, 57, 42, 33, 32, 55, 64, 34, 50, 32, 40, 43, 46, 70, 15, 48, 41, 53, 34, 47, 33, 33, 32, 53, 46, 67, 25, 55, 38, 32, 20, 20, 58, 41, 20, 41, 42, 25, 42, 54, 22, 15, 51, 37, 46, 11, 47, 60, 15, 32, 12, 47, 50, 52, 39, 14, 52, 30, 52, 26, 50, 39, 36, 33, 46, 37, 45, 35, 25, 28, 44, 34, 48, 65, 51, 33, 35, 44, 56, 40, 55, 48, 48, 54, 59, 59, 58, 32, 63, 69, 48, 36, 37, 32, 46, 34, 22, 40, 39, 34, 30, 26, 50, 53, 37, 28, 22, 35, 42, 12, 12, 30, 13, 26, 21, 31, 23, 27, 19, 13, 25, 44, 55, 33, 12, 33, 24, 13, 11, 28, 21, 35, 9, 36, 38, 34, 33, 35, 56, 49, 10, 45, 38, 7, 29, 22, 47, 33, 6, 22, 31, 41, 32, 22, 26, 19, 17, 47, 37, 26, 28, 61, 42, 39, 37, 30, 28, 31, 23, 36, 6, 29, 17, 31, 40, 35, 35, 19, 30, 37, 58, 36, 47, 48, 48, 48, 50, 34, 28, 29, 46, 47, 23, 35, 34, 60, 35, 44, 52, 27, 46, 13, 32, 52, 40, 34, 42, 51, 51, 29, 33, 43, 22, 30, 34, 31, 26, 37, 43, 50, 27, 23, 16, 29, 30, 10, 24, 39, 35, 17, 39, 25, 33, 25, 31, 53, 40, 40, 37, 17, 22, 17, 31, 40, 34, 22, 44, 16, 21, 48, 29, 38, 34, 7, 25, 27, 29, 29, 30, 52, 33, 39, 5, 47, 29, 18, 39, 13, 30, 34, 44, 28, 43, 42, 14, 25, 38, 56, 42, 30, 25, 14, 21, 49, 40, 18, 34, 52, 25, 22, 58, 26, 30, 19, 21, 15, 39, 32, 32, 19, 47, 42, 36, 35, 18, 28, 50, 29, 29, 29, 28, 38, 46, 37, 30, 35, 30, 43, 47, 29, 30, 54, 33, 31, 22, 8, 37, 48, 36, 43, 24, 33, 19, 28, 39, 29, 31, 32, 14, 21, 30, 38, 50, 17, 23, 17, 32, 39, 46, 32, 27, 37, 57, 14, 25, 31, 23, 7, 30, 46, 30, 38, 39, 50, 8, 30, 38, 41, 31, 37, 35, 51, 31, 36, 57, 24, 46, 35, 41, 36, 14, 38, 26, 31, 27, 35, 46, 30, 25, 30, 28, 42, 22, 44, 21, 43, 35, 44, 42, 29, 41, 26, 31, 48, 34, 28, 29, 2, 29, 10, 17, 21, 33, 21, 26, 3, 13, 0, 0, -1, 12, 0, 2, 22, -5, 13, 15, -6, 24, 26, 11, 26, 50, 29, 26, 42, 38, 30, 40, 32, 19, 26, 39, 45, 38, 45, 44, 30, 35, 50, 40, 24, 36, 49, 22, 39, 45, 38, 17, 30, 32, 26, 17, 33, 35, 37, 45, 46, 40, 49, 57, 28, 50, 31, 1, 39, 38, 41, 37, 31, 26, 26, 30, 46, 21, 17, 31, 35, 36, 33, 33, 23, 31, 42, 25, 35, 35, 36, -1, 30, 38, 16, 49, 39, 59, 23, 49, 19, 49, 30, 23, 39, 12, 41, 30, 38, 38, 40, 50, 55, 33, 51, 41, 33, 34, 40, 63, 46, 46, 40, 30, 26, 11, 23, 40, 49, 18, 38, 16, 49, 53, 38, 44, 41, 39, 39, 31, 49, 41, 3, 67, 46, 23, 33, 23, 29, 34, 35, 49, 30, 49, 35, 25, 21, 32, 19, 35, 10, 51, 33, 28, 36, 48, 20, 39, 15, 32, 55, 51, 43, 15, 47, 40, 49, 11, 0, 10, 33, 42, 40, 67, 27, 25, 8, 39, 23, 23, 36, 20, 49, 29, 44, 37, 45, 36, 49, 35, 30, 43, 25, 26, 44, 22, 29, 59, 36, 40, 36, 47, 35, 28, 34, 45, 34, 28, 26, 24, 40, 39, 50, 57, 20, 37, 21, 36, 46, 15, 46, 33, 40, 33, 16, 3, 20, 36, 13, 30, 35, 30, 63, 35, 51, 51, 36, 52, 61, 27, 31, 52, 32, 44, 21, 15, 24, 21, 35, 21, 37, 51, 19, 39, 38, 45, 54, 24, 18, 10, -1, 39, 21, 33, 32, 51, 56, 47, 21, 32, 16, 26, 38, 50, 30, 35, 41, 50, 29, 45, 46, 17, 24, 32, 29, 41, 34, 44, 23, 23, 19, 8, 15, 25, 32, 27, 34, 27, 19, 35, 40, 28, 18, 27, 30, 18, 47, 35, 30, 34, 18, 38, 32, 48, 35, 31, 47, 33, 24, 29, 30, 37, 52, 41, 40, 23, 36, 39, 27, 22, 10, 23, 38, 23, 27, 23, 26, 24, 21, 38, 39, 29, 24, 18, 25, 35, 44, 21, 36, 13, 24, 44, 36, 26, 39, 50, 60, 11, 28, 18, 38, 45, 59, 47, 35, 30, 38, 42, 14, 36, 34, 52, 20, 13, 27, 14, 12, 30, 4, 17, 33, 32, 39, 39, 18, 35, 38, 28, 47, 21, 29, 39, 29, 26, 51, 41, 49, 28, 36, 31, 31, 46, 21, 36, 37, 49, 27, 28, 52, 27, 41, 18, 44, 46, 50, 47, 58, 43, 51, 35, 35, 46, 46, 33, 44, 47, 21, 52, 42, 48, 41, 54, 30, 39, 28, 39, 28, 43, 31, 31, 38, 37, 28, 33, 43, 33, 30, 33, 8, 51, 50, 28, 23, 26, 44, 34, 40, 27, 28, 28, 40, 40, 50, 43, 27, 42, 41, 38, 26, 62, 31, 24, 42, 31, 26, 57, 20, 35, 45, 30, 50, 43, 26, 34, 46, 38, 60, 32, 44, 50, 39, 46, 9, 38, -1, 7, 36, 31, 37, 32, 15, 8, 20, 43, 17, 16, 14, 7, 33, 49, 35, 31, 24, 31, 35, 14, 35, 41, 38, 40, 23, 15, 47, 52, 29, 29, 53, 26, 40, 71, 22, 51, 33, 30, 34, 42, 43, 23, 39, 29, 29, 41, 35, 42, 42, 27, 34, 34, 39, 39, 27, 29, 30, 47, 43, 20, 19, 37, 41, 51, 41, 18, 44, 6, 38, 10, 52, 40, 43, 39, 28, 28, 39, 41, 32, 17, 13, 25, 38, 26, 48, 45, 32, 45, 35, 28, 24, 26, -2, 31, 16, 29, 32, 20, 30, 41, 39, 23, 32, 36, 36, 33, 52, 35, 51, 33, 19, 41, 27, 28, 41, 23, 29, 39, 51, 51, 48, 32, 38, 35, 32, 11, 59, 37, 35, 28, 25, 38, 51, 54, 44, 53, 55, 29, 55, 40, 31, 43, 37, 11, 58, 39, 41, 59, 52, 42, 18, 23, 29, 39, 33, 21, 38, 34, 20, 21, 29, 32, 31, 32, 47, 29, 24, 15, 28, 37, 21, 7, 32, 31, 31, 36, 35, 7, 26, 26, 38, 65, 51, 60, 21, 40, 42, 42, 74, 51, 37, 59, 46, 50, 49, 63, 51, 76, 56, 62, 58, 61, 75, 78, 62, 105, 58, 86, 76, 61, 69, 93, 78, 62, 77, 81, 102, 100, 72, 94, 105, 98, 105, 101, 80, 92, 104, 98, 66, 83, 69, 85, 89, 106, 84, 63, 52, 73, 87, 83, 71, 79, 51, 76, 86, 83, 75, 67, 70, 99, 84, 88, 89, 73, 67, 60, 92, 71, 78, 87, 69, 87, 73, 64, 67, 98, 85, 69, 81, 76, 89, 79, 70, 83, 103, 100, 70, 95, 82, 93, 81, 88, 79, 95, 89, 74, 91, 84, 80, 89, 112, 97, 86, 86, 90, 61, 90, 74, 111, 84, 74, 94, 87, 78, 65, 107, 59, 76, 61, 70, 66, 77, 85, 62, 74, 77, 38, 61, 75, 32, 44, 60, 40, 14, 24, 21, 38, 49, 36, 70, 70, 42, 71, 60, 74, 67, 63, 77, 110, 110, 91, 85, 129, 120, 91, 82, 102, 91, 133, 118, 126, 128, 105, 143, 157, 142, 157, 170, 153, 208, 263, 260, 317, 325, 395, 419, 495, 595, 638, 718, 727, 866, 933, 961, 968, 998, 1013, 1031, 981, 907, 886, 760, 756, 680, 586, 508, 464, 456, 374, 327, 325, 257, 232, 248, 267, 264, 272, 267, 376, 373, 395, 447, 547, 625, 734, 830, 943, 1046, 1142, 1243, 1358, 1392, 1490, 1540, 1564, 1618, 1641, 1563, 1536, 1540, 1489, 1402, 1270, 1177, 1040, 988, 851, 716, 611, 516, 471, 399, 338, 317, 259, 256, 244, 220, 167, 185, 212, 154, 151, 181, 160, 135, 149, 135, 148, 141, 150, 127, 143, 125, 119, 83, 114, 124, 113, 159, 108, 149, 148, 197, 237, 262, 299, 329, 412, 486, 525, 654, 732, 844, 997, 1075, 1143, 1168, 1274, 1218, 1205, 1151, 1000, 956, 857, 729, 639, 590, 517, 470, 511, 465, 527, 549, 654, 761, 816, 908, 971, 1022, 1024, 1111, 1146, 1098, 981, 944, 830, 707, 625, 544, 450, 342, 307, 261, 250, 250, 230, 265, 299, 373, 446, 522, 569, 722, 817, 939, 1008, 1099, 1221, 1253, 1303, 1239, 1197, 1059, 968, 854, 761, 581, 462, 432, 330, 282, 237, 205, 156, 185, 179, 192, 202, 256, 259, 328, 374, 434, 534, 626, 727, 838, 993, 1153, 1315, 1380, 1451, 1486, 1497, 1462, 1365, 1231, 1163, 969, 832, 701, 607, 462, 399, 362, 289, 244, 205, 212, 203, 172, 190, 173, 195, 163, 177, 147, 194, 150, 162, 164, 171, 169, 165, 188, 160, 169, 152, 165, 174, 161, 191, 191, 177, 238, 275, 318, 410, 519, 609, 726, 872, 983, 1164, 1335, 1384, 1447, 1560, 1603, 1563, 1542, 1484, 1393, 1254, 1227, 1241, 1305, 1360, 1446, 1535, 1519, 1496, 1541, 1466, 1388, 1239, 1109, 867, 803, 603, 482, 411, 290, 255, 224, 196, 174, 154, 150, 162, 183, 192, 202, 224, 291, 284, 391, 414, 545, 621, 774, 949, 1057, 1285, 1506, 1599, 1689, 1782, 1797, 1723, 1806, 1645, 1643, 1506, 1547, 1628, 1636, 1644, 1709, 1832, 1862, 1850, 1714, 1555, 1490, 1309, 1056, 850, 674, 490, 374, 315, 247, 185, 163, 139, 125, 113, 116, 115, 119, 118, 116, 143, 125, 105, 138, 144, 133, 159, 142, 146, 161, 127, 160, 146, 136, 171, 170, 156, 183, 177, 194, 178, 185, 181, 227, 240, 310, 351, 486, 587, 749, 882, 1075, 1227, 1475, 1602, 1693, 1819, 1756, 1677, 1568, 1416, 1205, 1021, 839, 610, 539, 421, 354, 337, 327, 384, 451, 559, 696, 848, 1092, 1252, 1468, 1614, 1696, 1748, 1822, 1690, 1641, 1468, 1227, 1030, 841, 704, 497, 399, 316, 215, 208, 149, 128, 173, 114, 121, 149, 123, 112, 152, 138, 142, 202, 179, 189, 204, 223, 226, 299, 295, 369, 484, 591, 752, 958, 1109, 1241, 1435, 1568, 1658, 1622, 1574, 1559, 1417, 1266, 1110, 927, 793, 597, 490, 414, 342, 266, 222, 224, 220, 197, 157, 174, 192, 148, 156, 163, 108, 117, 102, 80, 94, 106, 84, 104, 81, 120, 108, 116, 106, 106, 109, 100, 117, 145, 115, 138, 115, 112, 131, 133, 152, 138, 152, 163, 193, 138, 155, 162, 161, 159, 168, 176, 191, 173, 195, 193, 211, 199, 207, 198, 242, 205, 179, 187, 198, 189, 220, 221, 218, 228, 273, 308, 360, 483, 547, 722, 889, 1091, 1350, 1509, 1690, 1896, 2036, 2013, 2016, 1902, 1751, 1564, 1395, 1330, 1292, 1310, 1377, 1500, 1721, 1840, 1870, 2020, 2065, 1972, 1775, 1664, 1426, 1160, 981, 780, 565, 495, 350, 328, 288, 249, 203, 167, 145, 166, 187, 160, 167, 123, 157, 130, 139, 147, 138, 104, 136, 114, 126, 145, 121, 140, 125, 125, 163, 151, 173, 174, 157, 196, 184, 241, 255, 279, 291, 361, 488, 586, 781, 930, 1161, 1397, 1622, 1836, 1946, 2086, 2074, 2023, 1820, 1642, 1411, 1273, 1211, 1229, 1264, 1429, 1691, 1814, 2044, 2250, 2379, 2360, 2273, 2132, 1960, 1581, 1307, 992, 784, 638, 521, 394, 315, 290, 267, 274, 253, 279, 317, 390, 466, 508, 649, 782, 956, 1083, 1360, 1521, 1680, 1844, 1814, 1828, 1632, 1439, 1270, 1152, 921, 802, 615, 477, 362, 307, 229, 273, 231, 225, 246, 265, 215, 243, 214, 243, 180, 184, 202, 161, 196, 192, 206, 142, 177, 171, 167, 162, 132, 157, 184, 161, 147, 148, 213, 187, 162, 179, 182, 193, 190, 180, 209, 234, 198, 223, 197, 228, 201, 208, 247, 228, 220, 255, 233, 250, 275, 254, 252, 297, 327, 422, 429, 592, 739, 909, 1220, 1495, 1705, 1882, 2164, 2202, 2182, 2074, 1884, 1718, 1409, 1266, 1153, 1034, 1076, 1234, 1459, 1641, 1939, 2126, 2335, 2407, 2535, 2467, 2230, 1992, 1723, 1495, 1269, 1164, 1152, 1212, 1381, 1631, 1851, 2044, 2175, 2294, 2303, 2270, 2064, 1777, 1457, 1176, 952, 743, 562, 438, 337, 275, 267, 263, 218, 227, 261, 240, 272, 318, 321, 427, 508, 671, 831, 1087, 1344, 1467, 1761, 1960, 1973, 2015, 1810, 1658, 1485, 1174, 916, 737, 563, 433, 322, 295, 275, 220, 213, 220, 187, 153, 136, 172, 161, 141, 159, 121, 161, 120, 189, 161, 167, 135, 143, 179, 176, 147, 138, 180, 148, 180, 220, 197, 200, 207, 203, 244, 258, 286, 313, 335, 396, 516, 658, 841, 1093, 1369, 1561, 1909, 2152, 2243, 2463, 2392, 2154, 1887, 1694, 1429, 1330, 1184, 1143, 1251, 1365, 1718, 1966, 2205, 2558, 2764, 2756, 2725, 2608, 2251, 1986, 1617, 1266, 980, 792, 623, 450, 370, 326, 261, 237, 251, 254, 277, 306, 306, 369, 460, 563, 722, 891, 1125, 1308, 1529, 1698, 1804, 1947, 1905, 1802, 1666, 1498, 1201, 1053, 861, 698, 571, 494, 437, 345, 337, 331, 330, 331, 305, 336, 383, 373, 388, 408, 395, 415, 408, 458, 498, 532, 540, 556, 576, 609, 683, 790, 885, 1024, 1279, 1441, 1814, 2073, 2373, 2504, 2739, 2827, 2813, 2606, 2460, 2219, 2051, 1937, 1891, 1825, 1906, 2113, 2397, 2682, 2880, 3058, 3180, 3163, 2999, 2777, 2517, 2213, 1833, 1565, 1331, 1054, 983, 926, 872, 780, 705, 745, 676, 636, 609, 557, 528, 533, 496, 500, 435, 436, 433, 400, 380, 338, 358, 362, 405, 388, 486, 600, 734, 869, 1164, 1360, 1601, 1869, 2082, 2074, 2095, 1980, 1772, 1629, 1292, 1037, 780, 579, 480, 354, 303, 269, 227, 242, 234, 222, 219, 236, 305, 315, 385, 463, 586, 790, 886, 1186, 1488, 1630, 1826, 2091, 2143, 2058, 1953, 1756, 1512, 1276, 1010, 768, 595, 512, 366, 312, 276, 273, 250, 260, 241, 311, 360, 437, 599, 727, 975, 1215, 1520, 1765, 1893, 2062, 2152, 2097, 1942, 1709, 1388, 1169, 890, 658, 545, 383, 295, 216, 184, 177, 209, 197, 164, 168, 213, 211, 186, 217, 200, 198, 168, 174, 158, 176, 160, 171, 162, 140, 137, 133, 125, 144, 118, 136, 139, 139, 126, 133, 110, 153, 174, 147, 124, 157, 158, 168, 180, 166, 179, 170, 176, 157, 164, 151, 179, 160, 138, 156, 162, 154, 177, 138, 168, 215, 225, 311, 347, 446, 640, 847, 1081, 1266, 1579, 1760, 2016, 2123, 2014, 1890, 1755, 1516, 1191, 896, 717, 545, 372, 292, 253, 188, 207, 191, 199, 170, 172, 186, 169, 183, 173, 156, 170, 176, 188, 151, 173, 192, 174, 175, 177, 143, 156, 149, 153, 132, 114, 115, 91, 124, 73, 91, 109, 119, 140, 130, 148, 125, 120, 132, 142, 157, 166, 175, 149, 114, 173, 201, 176, 172, 183, 170, 193, 194, 172, 212, 216, 249, 273, 307, 380, 486, 633, 859, 1065, 1369, 1582, 1864, 2038, 2078, 2038, 1986, 1663, 1425, 1180, 860, 694, 487, 428, 300, 235, 181, 165, 112, 135, 143, 143, 129, 113, 152, 161, 136, 159, 157, 180, 174, 175, 194, 186, 204, 157, 195, 191, 212, 248, 262, 341, 401, 536, 660, 827, 1067, 1233, 1312, 1507, 1652, 1628, 1612, 1475, 1281, 1002, 870, 631, 528, 422, 311, 279, 293, 253, 281, 266, 282, 302, 384, 465, 597, 763, 1028, 1248, 1546, 1817, 1961, 2238, 2295, 2218, 2062, 1796, 1557, 1202, 867, 701, 564, 481, 337, 305, 293, 278, 285, 258, 283, 273, 367, 410, 532, 673, 922, 1197, 1588, 1864, 2199, 2514, 2623, 2566, 2512, 2255, 1907, 1520, 1221, 1051, 904, 873, 898, 1075, 1351, 1637, 1978, 2327, 2558, 2675, 2663, 2432, 2296, 2018, 1607, 1274, 1076, 955, 917, 982, 1105, 1380, 1654, 1985, 2210, 2380, 2413, 2252, 2093, 1902, 1633, 1243, 945, 695, 542, 420, 378, 322, 260, 261, 211, 198, 180, 176, 162, 149, 176, 164, 163, 162, 105, 150, 154, 182, 152, 152, 146, 92, 104, 123, 101, 90, 80, 100, 78, 102, 139, 155, 162, 229, 218, 243, 202, 251, 255, 282, 335, 371, 522, 687, 958, 1290, 1678, 2022, 2406, 2798, 2979, 2832, 2707, 2484, 2244, 1875, 1476, 1207, 1080, 1042, 1061, 1256, 1506, 1850, 2172, 2413, 2692, 2619, 2684, 2519, 2267, 1898, 1560, 1265, 1087, 859, 899, 980, 1168, 1424, 1783, 2089, 2363, 2543, 2516, 2503, 2434, 2075, 1806, 1407, 1091, 761, 612, 449, 372, 266, 239, 234, 247, 209, 187, 194, 202, 181, 186, 189, 188, 198, 205, 200, 185, 168, 193, 187, 220, 199, 182, 165, 178, 176, 146, 128, 128, 163, 117, 119, 142, 172, 152, 168, 172, 196, 199, 182, 205, 177, 198, 199, 200, 234, 197, 234, 251, 230, 264, 255, 238, 233, 303, 318, 368, 525, 703, 961, 1278, 1717, 2076, 2495, 2739, 2807, 2751, 2638, 2328, 1975, 1647, 1318, 1109, 924, 919, 925, 1091, 1338, 1621, 2063, 2225, 2496, 2553, 2522, 2404, 2151, 1666, 1455, 1085, 849, 579, 478, 340, 253, 232, 245, 233, 191, 201, 206, 167, 184, 176, 187, 214, 184, 156, 203, 192, 172, 184, 179, 183, 169, 205, 194, 207, 183, 154, 111, 156, 154, 178, 173, 201, 197, 171, 229, 233, 247, 261, 302, 299, 277, 312, 343, 341, 374, 440, 550, 740, 944, 1278, 1675, 2001, 2295, 2643, 2778, 2771, 2625, 2423, 2093, 1701, 1389, 1050, 793, 554, 406, 367, 273, 248, 236, 251, 223, 204, 181, 174, 191, 161, 121, 128, 162, 172, 144, 140, 148, 177, 210, 183, 206, 155, 231, 211, 230, 203, 216, 279, 319, 443, 559, 768, 919, 1267, 1627, 1954, 2126, 2358, 2416, 2327, 2240, 1942, 1646, 1353, 1104, 925, 863, 910, 1136, 1254, 1641, 1958, 2206, 2427, 2672, 2584, 2522, 2241, 1962, 1625, 1220, 978, 625, 543, 425, 327, 333, 272, 221, 225, 214, 209, 252, 226, 235, 238, 210, 252, 250, 258, 248, 251, 271, 286, 248, 201, 218, 218, 209, 223, 270, 297, 396, 457, 627, 830, 1117, 1388, 1625, 1876, 2134, 2170, 2242, 2151, 1813, 1545, 1226, 953, 731, 554, 442, 365, 302, 288, 281, 255, 230, 235, 245, 270, 312, 379, 491, 673, 899, 1225, 1474, 1915, 2204, 2360, 2507, 2365, 2300, 1998, 1693, 1356, 1028, 770, 547, 439, 367, 313, 270, 213, 226, 199, 167, 153, 172, 137, 136, 160, 129, 131, 109, 155, 114, 166, 145, 207, 155, 168, 172, 156, 191, 223, 258, 285, 382, 487, 658, 925, 1168, 1531, 1822, 2057, 2333, 2449, 2326, 2223, 1922, 1625, 1306, 1047, 822, 757, 701, 847, 1015, 1282, 1741, 2055, 2443, 2619, 2694, 2718, 2436, 2174, 1855, 1547, 1218, 918, 665, 521, 423, 310, 286, 233, 233, 172, 185, 173, 176, 158, 134, 139, 147, 134, 137, 165, 186, 135, 148, 132, 157, 163, 138, 129, 123, 119, 104, 135, 140, 161, 175, 194, 210, 244, 199, 204, 193, 199, 168, 128, 137, 98, 126, 114, 140, 126, 125, 118, 115, 134, 178, 162, 141, 136, 145, 168, 134, 152, 115, 112, 130, 100, 111, 107, 110, 48, 62, 99, 137, 38, 89, 120, 124, 111, 124, 150, 152, 136, 160, 177, 185, 207, 180, 210, 156, 166, 171, 180, 175, 215, 221, 213, 270, 373, 444, 599, 788, 1066, 1366, 1710, 2031, 2237, 2259, 2122, 2031, 1787, 1443, 1133, 854, 654, 427, 378, 262, 209, 190, 166, 180, 151, 186, 170, 139, 166, 145, 177, 184, 165, 147, 167, 171, 154, 139, 151, 151, 173, 164, 130, 104, 103, 105, 104, 71, 83, 120, 99, 102, 154, 133, 147, 163, 204, 187, 163, 148, 150, 173, 202, 159, 176, 167, 145, 182, 196, 172, 183, 187, 208, 187, 209, 185, 248, 351, 482, 651, 942, 1248, 1613, 2061, 2379, 2568, 2745, 2775, 2609, 2208, 1891, 1546, 1188, 866, 694, 454, 318, 226, 176, 215, 175, 210, 164, 222, 225, 205, 190, 215, 216, 220, 254, 278, 331, 458, 525, 743, 1038, 1260, 1460, 1772, 2088, 2090, 2098, 2000, 1833, 1630, 1268, 1002, 869, 685, 660, 745, 846, 1054, 1374, 1695, 2043, 2389, 2611, 2589, 2501, 2343, 2044, 1616, 1280, 1006, 690, 551, 389, 275, 234, 191, 211, 205, 160, 179, 187, 224, 213, 211, 195, 180, 237, 276, 324, 409, 546, 774, 1005, 1294, 1556, 1915, 2079, 2084, 2074, 2004, 1776, 1469, 1257, 932, 832, 723, 686, 815, 929, 1216, 1508, 1889, 2153, 2358, 2357, 2332, 2124, 1854, 1565, 1241, 922, 718, 494, 401, 295, 273, 232, 174, 201, 201, 181, 199, 186, 203, 331, 412, 605, 786, 1108, 1428, 1748, 2115, 2340, 2480, 2470, 2373, 2144, 1811, 1517, 1130, 918, 741, 726, 795, 1005, 1310, 1601, 1928, 2261, 2471, 2711, 2793, 2602, 2241, 1906, 1606, 1205, 846, 662, 475, 348, 270, 192, 187, 184, 148, 141, 140, 142, 113, 97, 92, 110, 110, 102, 96, 98, 103, 88, 117, 131, 94, 107, 110, 90, 107, 120, 110, 91, 112, 127, 99, 100, 99, 97, 120, 104, 108, 121, 122, 99, 131, 101, 150, 118, 120, 123, 127, 124, 147, 135, 107, 151, 153, 146, 145, 142, 164, 123, 114, 118, 137, 113, 94, 125, 98, 111, 105, 134, 150, 152, 116, 144, 139, 145, 163, 203, 200, 273, 350, 498, 723, 1000, 1329, 1597, 1881, 2208, 2411, 2402, 2370, 2057, 1772, 1474, 1199, 920, 812, 798, 793, 954, 1226, 1636, 1969, 2352, 2809, 2961, 3056, 2913, 2661, 2368, 1883, 1395, 1144, 752, 629, 432, 303, 189, 187, 144, 130, 113, 106, 118, 72, 112, 86, 89, 111, 102, 91, 73, 91, 89, 68, 84, 125, 124, 110, 116, 78, 79, 58, 100, 76, 74, 91, 69, 95, 90, 94, 73, 80, 77, 72, 111, 144, 113, 134, 114, 123, 142, 129, 119, 131, 127, 113, 150, 107, 125, 110, 95, 120, 109, 85, 126, 109, 123, 98, 112, 117, 127, 102, 122, 178, 189, 165, 171, 154, 152, 138, 156, 161, 150, 153, 151, 137, 104, 124, 134, 132, 141, 140, 147, 132, 117, 126, 140, 107, 135, 145, 146, 149, 140, 166, 172, 158, 195, 198, 179, 155, 153, 128, 155, 199, 225, 227, 325, 376, 510, 702, 929, 1253, 1564, 1770, 2040, 2115, 2173, 2173, 2006, 1789, 1440, 1095, 947, 730, 644, 703, 784, 1091, 1425, 1744, 2154, 2432, 2723, 2823, 2762, 2596, 2296, 1898, 1635, 1226, 764, 568, 434, 297, 198, 212, 127, 98, 108, 155, 112, 103, 111, 115, 88, 107, 114, 95, 101, 88, 96, 114, 104, 113, 109, 127, 124, 124, 126, 153, 135, 106, 107, 134, 155, 166, 131, 124, 138, 127, 146, 120, 107, 114, 109, 91, 115, 106, 85, 118, 104, 91, 100, 108, 114, 101, 130, 126, 84, 121, 128, 100, 115, 119, 133, 130, 131, 107, 107, 100, 105, 105, 107, 83, 92, 107, 91, 99, 87, 70, 125, 115, 118, 131, 81, 119, 119, 143, 142, 120, 143, 105, 148, 133, 145, 171, 199, 228, 263, 350, 485, 662, 867, 1143, 1386, 1649, 1978, 2209, 2274, 2165, 1988, 1671, 1461, 1156, 839, 616, 432, 332, 292, 186, 187, 135, 137, 137, 131, 135, 123, 110, 121, 131, 111, 131, 137, 128, 120, 151, 137, 70, 103, 110, 119, 91, 87, 87, 90, 136, 146, 149, 147, 137, 128, 138, 108, 135, 125, 94, 120, 101, 141, 136, 151, 137, 154, 157, 166, 165, 150, 177, 171, 133, 142, 149, 149, 124, 128, 116, 110, 105, 89, 54, 59, 82, 74, 49, 74, 52, 64, 74, 89, 91, 110, 107, 99, 109, 116, 133, 127, 139, 159, 174, 158, 140, 146, 139, 128, 135, 153, 170, 162, 142, 207, 177, 154, 182, 213, 302, 396, 496, 686, 979, 1248, 1591, 1777, 2032, 2225, 2312, 2248, 2030, 1795, 1523, 1185, 901, 590, 409, 298, 220, 185, 169, 133, 153, 154, 163, 118, 132, 150, 149, 156, 143, 127, 129, 138, 122, 142, 108, 136, 113, 123, 117, 157, 121, 163, 123, 129, 124, 161, 161, 141, 136, 119, 75, 109, 77, 127, 133, 99, 95, 93, 108, 114, 131, 99, 116, 161, 142, 143, 139, 150, 151, 165, 136, 194, 154, 163, 195, 253, 308, 426, 618, 884, 1195, 1593, 1934, 2130, 2599, 2853, 2882, 2661, 2427, 2117, 1719, 1400, 985, 724, 538, 347, 268, 206, 187, 220, 199, 172, 183, 176, 201, 165, 129, 164, 146, 151, 156, 162, 137, 123, 131, 109, 131, 112, 139, 105, 151, 136, 134, 140, 102, 153, 146, 140, 152, 131, 140, 183, 177, 187, 149, 186, 143, 207, 168, 175, 179, 139, 166, 192, 233, 257, 341, 421, 608, 662, 942, 1226, 1545, 1767, 1963, 2050, 2098, 1970, 1658, 1435, 1171, 895, 694, 510, 350, 226, 194, 173, 161, 161, 93, 166, 156, 147, 139, 128, 178, 153, 165, 162, 181, 170, 155, 132, 119, 137, 147, 137, 157, 173, 182, 160, 172, 175, 182, 237, 214, 267, 335, 423, 575, 784, 929, 1281, 1607, 1970, 2185, 2205, 2220, 2060, 1846, 1585, 1332, 1043, 892, 784, 782, 879, 1156, 1420, 1735, 2201, 2547, 2717, 2873, 2899, 2628, 2318, 1817, 1569, 1208, 923, 661, 512, 343, 255, 197, 177, 191, 136, 150, 119, 158, 141, 137, 133, 153, 175, 135, 163, 130, 132, 107, 102, 95, 91, 106, 108, 76, 103, 108, 85, 97, 87, 110, 104, 106, 122, 133, 126, 132, 161, 157, 154, 142, 168, 200, 205, 237, 265, 371, 509, 657, 872, 1240, 1578, 1945, 2277, 2508, 2627, 2669, 2579, 2317, 1989, 1584, 1201, 898, 637, 449, 361, 266, 211, 190, 181, 239, 218, 225, 233, 207, 190, 164, 141, 187, 154, 164, 162, 168, 154, 164, 160, 141, 94, 143, 131, 137, 141, 133, 137, 128, 142, 177, 116, 149, 161, 143, 166, 149, 161, 167, 129, 153, 172, 130, 184, 159, 172, 182, 142, 146, 180, 165, 123, 127, 168, 140, 123, 118, 149, 154, 152, 146, 189, 220, 240, 320, 454, 625, 804, 1085, 1426, 1807, 1973, 2157, 2278, 2260, 2141, 1882, 1646, 1299, 1055, 788, 593, 469, 352, 285, 229, 195, 179, 185, 174, 158, 144, 150, 160, 155, 161, 171, 175, 159, 199, 197, 174, 196, 197, 190, 178, 173, 157, 163, 171, 162, 152, 170, 156, 136, 188, 142, 143, 170, 175, 182, 207, 223, 209, 227, 251, 262, 250, 261, 299, 351, 382, 423, 516, 733, 1046, 1239, 1672, 2157, 2658, 2967, 3118, 3140, 3115, 2836, 2457, 2096, 1727, 1284, 1110, 896, 913, 998, 1215, 1499, 1860, 2156, 2463, 2757, 2832, 2792, 2594, 2283, 1966, 1559, 1274, 1164, 927, 849, 998, 1278, 1532, 1918, 2254, 2739, 2970, 3104, 3085, 2863, 2750, 2276, 1782, 1440, 1182, 982, 969, 963, 1111, 1338, 1615, 2004, 2270, 2553, 2824, 2782, 2716, 2515, 2209, 1885, 1521, 1234, 986, 876, 964, 1133, 1360, 1634, 2082, 2536, 2730, 2970, 3010, 2977, 2661, 2390, 2024, 1633, 1302, 1074, 988, 944, 1079, 1282, 1537, 1868, 2211, 2457, 2612, 2581, 2464, 2293, 1995, 1563, 1365, 1036, 750, 610, 437, 324, 252, 230, 176, 174, 148, 154, 136, 142, 127, 144, 167, 156, 159, 128, 173, 167, 150, 179, 153, 171, 158, 163, 176, 159, 168, 201, 188, 220, 165, 187, 221, 237, 211, 202, 196, 248, 219, 219, 204, 194, 263, 296, 403, 565, 750, 1019, 1397, 1801, 2250, 2540, 2907, 3012, 3108, 2909, 2661, 2235, 1811, 1408, 1082, 1028, 864, 928, 1019, 1424, 1713, 2098, 2532, 2786, 2959, 3135, 2996, 2792, 2383, 2101, 1617, 1178, 953, 650, 443, 363, 241, 208, 233, 179, 204, 185, 179, 171, 163, 152, 146, 126, 168, 120, 152, 162, 184, 176, 221, 203, 231, 207, 170, 182, 169, 173, 135, 123, 152, 128, 162, 159, 132, 139, 152, 157, 178, 171, 220, 203, 233, 193, 210, 223, 210, 225, 207, 192, 192, 207, 215, 241, 259, 259, 253, 303, 281, 313, 298, 277, 313, 321, 255, 275, 288, 353, 376, 513, 681, 963, 1258, 1611, 2087, 2529, 2856, 3058, 3119, 3064, 2806, 2451, 1984, 1667, 1282, 1082, 921, 962, 1092, 1291, 1715, 2085, 2473, 2916, 3090, 3345, 3317, 3152, 2802, 2430, 2092, 1674, 1286, 1150, 1052, 1186, 1325, 1625, 2090, 2423, 2697, 2887, 2935, 3064, 2785, 2491, 2143, 1661, 1477, 1096, 942, 947, 1027, 1196, 1443, 1747, 2059, 2395, 2508, 2573, 2579, 2493, 2169, 1868, 1466, 1162, 877, 645, 494, 319, 308, 306, 275, 237, 225, 257, 188, 188, 209, 172, 176, 180, 182, 178, 149, 144, 118, 147, 153, 129, 136, 122, 97, 90, 123, 129, 120, 140, 117, 159, 143, 165, 142, 163, 134, 153, 166, 144, 131, 105, 117, 109, 132, 112, 138, 109, 162, 155, 121, 181, 183, 182, 111, 157, 119, 126, 159, 101, 96, 94, 119, 119, 129, 113, 140, 130, 138, 158, 116, 144, 139, 107, 107, 110, 107, 88, 119, 111, 92, 113, 111, 173, 124, 154, 123, 135, 117, 105, 109, 85, 92, 70, 100, 94, 104, 87, 124, 91, 146, 169, 186, 207, 248, 230, 240, 230, 211, 237, 214, 239, 300, 373, 510, 748, 1052, 1288, 1640, 2037, 2334, 2714, 2751, 2737, 2615, 2465, 1921, 1618, 1221, 916, 639, 525, 432, 355, 282, 273, 248, 255, 267, 283, 248, 277, 258, 252, 219, 243, 225, 242, 243, 299, 445, 545, 723, 996, 1181, 1555, 1843, 1960, 2033, 2022, 1928, 1796, 1474, 1269, 1020, 737, 510, 414, 301, 231, 224, 227, 195, 205, 236, 195, 177, 222, 171, 171, 167, 164, 171, 146, 145, 131, 168, 200, 199, 199, 225, 290, 271, 279, 303, 277, 249, 245, 236, 221, 291, 366, 443, 602, 824, 1109, 1443, 1814, 2129, 2264, 2464, 2480, 2361, 2193, 1885, 1698, 1350, 1004, 769, 542, 394, 351, 301, 272, 227, 199, 180, 218, 184, 167, 137, 145, 134, 117, 96, 137, 147, 115, 140, 138, 158, 155, 163, 176, 157, 150, 164, 132, 118, 104, 120, 114, 113, 82, 99, 92, 155, 101, 114, 159, 161, 148, 186, 181, 152, 142, 145, 150, 132, 115, 161, 123, 172, 195, 181, 221, 229, 215, 250, 226, 242, 210, 200, 146, 173, 222, 271, 378, 569, 762, 1051, 1321, 1650, 2050, 2400, 2554, 2600, 2624, 2359, 2196, 1771, 1412, 1103, 815, 632, 447, 346, 265, 227, 181, 200, 183, 215, 211, 221, 190, 205, 192, 229, 243, 341, 391, 564, 822, 1077, 1346, 1765, 2009, 2291, 2430, 2426, 2419, 2118, 1811, 1566, 1256, 874, 681, 487, 401, 316, 264, 264, 201, 199, 184, 219, 223, 196, 177, 162, 173, 180, 142, 159, 154, 139, 132, 180, 170, 189, 211, 194, 192, 219, 199, 198, 184, 196, 177, 162, 164, 147, 149, 156, 133, 167, 186, 199, 195, 249, 214, 223, 275, 201, 259, 241, 247, 241, 270, 386, 506, 681, 936, 1343, 1720, 2176, 2599, 2901, 3178, 3083, 3171, 2932, 2498, 2208, 1825, 1381, 1127, 889, 817, 871, 1031, 1257, 1571, 1908, 2333, 2550, 2766, 2835, 2669, 2511, 2201, 1823, 1501, 1270, 1020, 895, 916, 1103, 1235, 1537, 1821, 2235, 2562, 2782, 2901, 2952, 2774, 2619, 2226, 1763, 1441, 1073, 902, 648, 398, 363, 324, 276, 245, 259, 274, 262, 221, 242, 181, 224, 233, 265, 260, 260, 355, 452, 520, 711, 950, 1236, 1477, 1752, 1952, 2072, 2157, 2123, 1961, 1761, 1456, 1212, 895, 681, 504, 385, 352, 265, 238, 190, 189, 185, 186, 169, 182, 143, 154, 143, 103, 91, 105, 109, 83, 113, 120, 119, 111, 108, 95, 119, 71, 80, 124, 84, 109, 109, 108, 106, 122, 93, 94, 132, 109, 124, 107, 141, 94, 133, 94, 109, 91, 63, 127, 119, 105, 119, 73, 82, 115, 128, 119, 115, 92, 135, 101, 147, 110, 117, 132, 147, 126, 108, 112, 110, 135, 125, 125, 143, 132, 120, 127, 107, 125, 87, 121, 119, 110, 86, 113, 146, 102, 92, 118, 104, 106, 118, 168, 182, 160, 161, 183, 142, 144, 154, 169, 182, 186, 210, 276, 379, 490, 665, 841, 1063, 1277, 1531, 1744, 1916, 1888, 1863, 1714, 1498, 1327, 1076, 864, 647, 478, 356, 294, 243, 191, 172, 174, 184, 153, 192, 143, 135, 154, 142, 167, 164, 141, 200, 187, 154, 144, 162, 181, 181, 187, 191, 187, 166, 200, 168, 141, 145, 95, 101, 111, 118, 105, 125, 134, 133, 167, 183, 236, 212, 179, 205, 251, 251, 251, 274, 321, 368, 390, 516, 596, 806, 1145, 1491, 1743, 2154, 2486, 2684, 2955, 2915, 2750, 2451, 2248, 1921, 1551, 1268, 1047, 908, 922, 1080, 1273, 1666, 2049, 2526, 2890, 3217, 3374, 3372, 3176, 2968, 2705, 2196, 1685, 1449, 1163, 1035, 941, 1052, 1259, 1570, 2003, 2321, 2590, 2964, 3131, 3145, 3091, 2854, 2402, 1957, 1500, 1176, 920, 693, 536, 439, 316, 250, 227, 206, 185, 149, 188, 201, 146, 172, 114, 88, 131, 100, 125, 90, 157, 144, 170, 152, 143, 148, 137, 179, 154, 179, 210, 197, 161, 213, 155, 162, 199, 182, 157, 172, 150, 221, 249, 223, 227, 275, 238, 281, 268, 250, 274, 258, 299, 307, 390, 442, 664, 841, 1100, 1335, 1650, 2073, 2286, 2410, 2582, 2583, 2407, 2175, 1857, 1599, 1196, 835, 643, 518, 316, 318, 247, 193, 220, 213, 156, 195, 198, 223, 206, 226, 216, 201, 201, 154, 184, 165, 146, 157, 190, 149, 160, 160, 143, 103, 167, 168, 158, 153, 122, 160, 153, 142, 170, 172, 185, 199, 226, 205, 194, 206, 197, 203, 213, 235, 258, 288, 377, 489, 625, 988, 1308, 1488, 1817, 2092, 2299, 2393, 2368, 2309, 2069, 1824, 1512, 1190, 890, 687, 486, 378, 307, 274, 210, 168, 161, 179, 183, 187, 187, 161, 176, 128, 151, 177, 146, 155, 155, 160, 134, 111, 103, 154, 156, 150, 140, 168, 210, 211, 170, 173, 161, 141, 157, 132, 151, 126, 149, 159, 145, 183, 201, 214, 183, 209, 199, 213, 215, 222, 275, 274, 337, 413, 604, 806, 1072, 1399, 1591, 1840, 2162, 2317, 2380, 2303, 2203, 1952, 1609, 1232, 993, 730, 541, 426, 361, 247, 199, 170, 150, 203, 162, 113, 183, 143, 145, 138, 136, 149, 146, 141, 156, 129, 176, 159, 172, 181, 190, 187, 229, 233, 228, 215, 200, 210, 235, 177, 207, 254, 260, 313, 423, 571, 791, 962, 1152, 1441, 1552, 1691, 1823, 1876, 1770, 1610, 1448, 1183, 962, 835, 629, 432, 336, 284, 235, 221, 171, 181, 161, 150, 168, 163, 155, 124, 175, 141, 129, 130, 123, 124, 114, 143, 158, 204, 192, 172, 171, 167, 168, 184, 173, 183, 155, 155, 164, 157, 161, 131, 158, 203, 224, 256, 263, 291, 275, 266, 256, 242, 275, 258, 347, 488, 669, 888, 1107, 1500, 1896, 2343, 2535, 2811, 3109, 3126, 2929, 2697, 2252, 2040, 1679, 1346, 1107, 1011, 1013, 1116, 1373, 1643, 1999, 2403, 2777, 3014, 3264, 3181, 3098, 2815, 2509, 2074, 1640, 1292, 975, 735, 532, 407, 364, 252, 292, 237, 248, 196, 219, 207, 207, 187, 184, 165, 157, 177, 125, 116, 119, 127, 142, 124, 155, 112, 145, 147, 128, 102, 126, 123, 135, 143, 137, 113, 94, 111, 115, 129, 117, 149, 81, 110, 103, 95, 95, 89, 88, 79, 47, 87, 55, 90, 66, 78, 81, 120, 137, 152, 145, 176, 170, 139, 178, 156, 135, 146, 140, 128, 116, 123, 102, 78, 88, 130, 107, 88, 87, 118, 101, 105, 115, 108, 135, 122, 114, 99, 86, 133, 134, 128, 145, 177, 166, 192, 194, 206, 184, 188, 212, 220, 188, 149, 140, 174, 175, 156, 168, 123, 156, 146, 144, 146, 123, 135, 159, 172, 156, 180, 152, 188, 167, 169, 133, 149, 175, 199, 164, 178, 192, 234, 220, 240, 210, 231, 217, 231, 278, 235, 306, 332, 447, 658, 808, 1065, 1364, 1703, 2086, 2413, 2629, 2716, 2765, 2730, 2391, 1990, 1746, 1350, 1067, 744, 601, 485, 360, 278, 226, 193, 192, 223, 212, 224, 239, 215, 206, 220, 203, 208, 219, 256, 327, 426, 571, 728, 1005, 1366, 1579, 1873, 2115, 2332, 2403, 2374, 2138, 2017, 1814, 1515, 1281, 1010, 820, 774, 713, 833, 955, 1147, 1357, 1606, 1887, 2145, 2269, 2242, 2181, 2008, 1776, 1441, 1257, 934, 717, 565, 380, 353, 293, 190, 148, 151, 192, 134, 135, 142, 142, 197, 169, 145, 133, 118, 140, 141, 136, 169, 192, 169, 173, 155, 170, 161, 153, 168, 156, 179, 153, 154, 127, 149, 171, 190, 192, 223, 208, 228, 213, 259, 235, 196, 195, 169, 152, 130, 163, 119, 144, 139, 166, 168, 163, 133, 113, 181, 215, 186, 151, 185, 172, 160, 161, 126, 141, 141, 95, 146, 113, 96, 119, 120, 135, 87, 131, 109, 82, 84, 74, 105, 102, 94, 89, 82, 75, 113, 85, 93, 71, 110, 87, 115, 120, 82, 54, 103, 96, 82, 87, 100, 110, 111, 131, 129, 105, 146, 158, 127, 140, 126, 161, 118, 97, 126, 136, 103, 110, 86, 79, 114, 140, 150, 164, 160, 155, 166, 168, 190, 199, 173, 187, 201, 192, 208, 213, 213, 234, 271, 358, 481, 590, 672, 989, 1359, 1640, 1939, 2167, 2308, 2411, 2436, 2205, 1988, 1746, 1424, 1127, 867, 661, 467, 366, 279, 196, 198, 160, 130, 143, 137, 100, 93, 96, 107, 117, 133, 108, 140, 113, 127, 145, 139, 110, 119, 137, 127, 132, 98, 92, 120, 133, 153, 116, 131, 164, 123, 148, 158, 143, 163, 127, 141, 132, 138, 138, 139, 122, 74, 153, 110, 122, 73, 123, 104, 125, 117, 141, 155, 123, 111, 160, 163, 157, 102, 127, 117, 107, 107, 98, 131, 101, 139, 151, 98, 115, 128, 116, 115, 89, 116, 116, 115, 111, 74, 91, 94, 68, 108, 65, 105, 121, 136, 133, 130, 140, 164, 156, 144, 160, 164, 163, 165, 151, 130, 153, 235, 244, 260, 347, 447, 587, 736, 891, 1076, 1232, 1447, 1602, 1717, 1642, 1526, 1489, 1384, 1169, 995, 724, 582, 505, 383, 310, 271, 246, 222, 209, 187, 170, 170, 185, 125, 163, 116, 156, 141, 106, 122, 93, 112, 157, 167, 147, 129, 135, 140, 131, 113, 123, 135, 122, 153, 87, 136, 107, 119, 105, 97, 136, 127, 149, 127, 130, 162, 156, 158, 179, 128, 148, 172, 182, 122, 129, 124, 133, 151, 176, 132, 151, 135, 147, 126, 122, 123, 110, 93, 87, 141, 112, 126, 112, 130, 153, 173, 154, 182, 198, 202, 221, 254, 252, 246, 212, 211, 197, 201, 214, 211, 281, 317, 372, 532, 698, 889, 1177, 1493, 1726, 1991, 2270, 2388, 2498, 2424, 2277, 1987, 1753, 1516, 1264, 1001, 876, 864, 802, 906, 1081, 1258, 1541, 1724, 2001, 2101, 2207, 2330, 2253, 2057, 1914, 1542, 1199, 991, 754, 529, 411, 370, 278, 225, 182, 141, 162, 113, 153, 157, 119, 147, 153, 104, 138, 139, 101, 92, 114, 112, 143, 135, 104, 108, 147, 126, 126, 125, 193, 131, 109, 151, 88, 102, 144, 148, 132, 96, 170, 113, 150, 159, 141, 116, 136, 161, 143, 168, 193, 177, 167, 182, 131, 110, 127, 140, 119, 134, 134, 158, 209, 167, 140, 173, 213, 187, 175, 213, 227, 235, 287, 291, 408, 560, 667, 870, 1168, 1419, 1745, 2028, 2215, 2254, 2295, 2117, 1978, 1820, 1568, 1282, 971, 766, 584, 392, 330, 211, 192, 196, 187, 144, 137, 186, 139, 171, 138, 153, 140, 181, 158, 120, 126, 89, 131, 92, 122, 167, 121, 113, 111, 124, 150, 146, 123, 121, 96, 100, 118, 100, 132, 113, 154, 124, 113, 81, 126, 163, 127, 195, 178, 171, 131, 161, 196, 195, 196, 166, 181, 160, 143, 169, 123, 172, 128, 117, 136, 128, 112, 131, 128, 103, 121, 137, 106, 94, 121, 98, 116, 112, 140, 120, 110, 123, 126, 138, 100, 140, 153, 139, 175, 99, 95, 133, 144, 119, 122, 126, 151, 104, 162, 145, 105, 114, 139, 115, 102, 127, 116, 137, 141, 136, 115, 129, 109, 97, 80, 115, 124, 140, 128, 156, 150, 138, 168, 158, 161, 177, 177, 163, 141, 160, 151, 123, 95, 84, 81, 46, 70, 84, 67, 83, 92, 47, 64, 67, 59, 71, 104, 96, 131, 101, 87, 87, 154, 106, 128, 113, 125, 158, 157, 179, 176, 206, 265, 397, 476, 671, 792, 1069, 1325, 1551, 1636, 1834, 1865, 1848, 1752, 1624, 1495, 1239, 961, 745, 590, 503, 372, 312, 264, 181, 161, 154, 171, 167, 122, 134, 136, 135, 133, 128, 136, 126, 135, 134, 116, 141, 153, 132, 138, 98, 141, 138, 155, 149, 119, 137, 128, 91, 101, 108, 91, 84, 125, 114, 114, 116, 114, 151, 172, 146, 105, 149, 158, 155, 116, 91, 115, 125, 116, 107, 80, 121, 70, 141, 150, 148, 132, 138, 171, 177, 131, 135, 189, 206, 205, 160, 184, 150, 142, 143, 132, 130, 139, 139, 142, 118, 129, 111, 150, 131, 120, 103, 110, 135, 133, 124, 97, 106, 126, 141, 92, 133, 171, 184, 193, 233, 216, 228, 253, 230, 226, 198, 214, 219, 246, 305, 372, 438, 545, 710, 895, 1094, 1283, 1460, 1703, 1718, 1759, 1774, 1681, 1453, 1293, 1019, 885, 722, 562, 412, 321, 282, 244, 213, 192, 186, 192, 188, 220, 197, 184, 184, 161, 167, 154, 140, 124, 110, 138, 179, 129, 155, 178, 207, 212, 213, 166, 177, 177, 197, 223, 266, 354, 356, 525, 686, 954, 1138, 1478, 1690, 1926, 2216, 2345, 2347, 2359, 2238, 1980, 1770, 1521, 1236, 953, 761, 667, 467, 354, 309, 216, 240, 263, 196, 196, 193, 235, 244, 206, 182, 166, 174, 167, 196, 183, 141, 168, 151, 151, 181, 156, 227, 234, 239, 183, 209, 221, 167, 183, 180, 234, 273, 336, 385, 486, 659, 822, 975, 1177, 1319, 1546, 1680, 1706, 1796, 1743, 1642, 1509, 1300, 1031, 889, 649, 562, 455, 351, 264, 242, 238, 214, 216, 206, 228, 190, 238, 228, 188, 206, 211, 246, 250, 312, 402, 543, 683, 883, 1084, 1303, 1508, 1772, 1985, 2089, 2010, 1956, 1800, 1677, 1462, 1202, 1000, 750, 599, 464, 384, 302, 249, 236, 222, 208, 169, 160, 177, 168, 192, 149, 153, 178, 164, 127, 133, 147, 140, 141, 147, 150, 167, 140, 181, 173, 175, 159, 176, 183, 143, 155, 169, 183, 203, 207, 207, 274, 269, 219, 243, 227, 254, 244, 282, 283, 303, 333, 377, 469, 631, 777, 1007, 1340, 1628, 1955, 2138, 2350, 2502, 2689, 2535, 2422, 2282, 2000, 1630, 1380, 1097, 860, 725, 486, 465, 387, 272, 248, 225, 253, 242, 261, 258, 232, 248, 275, 278, 290, 359, 407, 491, 716, 874, 1015, 1286, 1476, 1871, 2116, 2346, 2419, 2435, 2392, 2140, 1911, 1742, 1480, 1208, 1038, 947, 914, 938, 992, 1131, 1342, 1613, 1983, 2138, 2225, 2395, 2309, 2150, 2019, 1790, 1485, 1179, 926, 887, 646, 475, 370, 264, 260, 206, 175, 193, 223, 210, 186, 204, 214, 196, 227, 172, 208, 205, 218, 184, 183, 180, 180, 185, 170, 161, 177, 147, 154, 165, 163, 142, 133, 111, 132, 120, 115, 117, 122, 163, 129, 128, 140, 161, 161, 98, 180, 154, 174, 175, 138, 152, 155, 148, 173, 152, 146, 156, 145, 158, 137, 160, 166, 154, 119, 157, 120, 154, 137, 143, 174, 188, 188, 169, 175, 183, 202, 212, 187, 199, 174, 147, 177, 144, 154, 149, 154, 142, 178, 168, 166, 174, 200, 176, 267, 250, 209, 224, 230, 210, 213, 213, 245, 284, 276, 278, 381, 415, 597, 782, 960, 1184, 1529, 1776, 1949, 2096, 2314, 2385, 2296, 2180, 2013, 1822, 1567, 1331, 998, 776, 661, 529, 463, 384, 328, 282, 333, 303, 277, 292, 282, 269, 277, 307, 325, 385, 523, 658, 820, 1023, 1280, 1497, 1913, 2181, 2389, 2542, 2410, 2367, 2360, 2168, 1840, 1567, 1201, 961, 800, 574, 469, 392, 360, 290, 257, 276, 256, 260, 232, 255, 260, 277, 308, 377, 426, 503, 671, 842, 1174, 1347, 1670, 1882, 2241, 2420, 2605, 2702, 2689, 2603, 2475, 2127, 1865, 1549, 1174, 922, 763, 642, 477, 391, 293, 292, 308, 310, 249, 245, 267, 239, 242, 201, 215, 202, 208, 198, 183, 159, 153, 143, 219, 168, 215, 182, 217, 224, 239, 257, 192, 238, 245, 250, 250, 289, 309, 357, 458, 590, 724, 901, 1060, 1262, 1554, 1768, 2025, 2266, 2305, 2274, 2227, 2199, 2048, 1840, 1583, 1357, 1109, 1027, 974, 1036, 1169, 1323, 1545, 1918, 2127, 2437, 2667, 2671, 2815, 2689, 2509, 2334, 1981, 1601, 1296, 1065, 870, 669, 510, 474, 344, 306, 313, 322, 267, 312, 238, 239, 222, 203, 210, 199, 179, 230, 254, 219, 248, 222, 230, 200, 204, 212, 193, 198, 199, 167, 171, 150, 140, 154, 165, 170, 124, 139, 160, 131, 173, 173, 197, 177, 182, 201, 162, 158, 144, 165, 176, 183, 163, 207, 179, 209, 198, 259, 219, 275, 276, 284, 295, 302, 274, 290, 301, 293, 298, 335, 432, 485, 576, 763, 1031, 1231, 1483, 1803, 2080, 2287, 2595, 2725, 2607, 2539, 2379, 2067, 1883, 1574, 1308, 1043, 826, 664, 472, 423, 355, 312, 260, 243, 233, 182, 212, 204, 183, 140, 186, 147, 138, 91, 117, 138, 132, 141, 104, 128, 127, 102, 82, 78, 98, 103, 117, 97, 120, 72, 107, 116, 95, 115, 110, 126, 106, 153, 160, 122, 142, 145, 175, 145, 125, 170, 146, 158, 144, 141, 165, 173, 196, 171, 185, 198, 210, 213, 208, 218, 178, 189, 205, 179, 159, 135, 145, 170, 165, 160, 127, 158, 156, 149, 175, 179, 196, 181, 217, 191, 151, 213, 215, 202, 177, 190, 176, 177, 195, 200, 180, 189, 148, 185, 161, 158, 166, 153, 187, 127, 154, 161, 145, 130, 156, 119, 109, 96, 91, 134, 160, 148, 183, 170, 136, 117, 222, 158, 151, 185, 148, 182, 135, 152, 160, 184, 177, 196, 237, 188, 141, 205, 203, 129, 187, 196, 214, 196, 160, 170, 169, 149, 229, 154, 165, 204, 178, 206, 215, 218, 226, 221, 226, 244, 225, 224, 213, 234, 302, 345, 376, 481, 669, 796, 1020, 1275, 1539, 1842, 2234, 2414, 2627, 2749, 2623, 2454, 2327, 2053, 1884, 1560, 1344, 1255, 1147, 1194, 1328, 1493, 1801, 2076, 2324, 2784, 3047, 3262, 3465, 3421, 3250, 2971, 2767, 2348, 2017, 1687, 1359, 989, 794, 533, 435, 367, 396, 258, 258, 227, 187, 236, 197, 206, 163, 153, 192, 164, 216, 173, 159, 137, 136, 110, 154, 152, 119, 158, 141, 139, 186, 217, 211, 166, 177, 160, 182, 150, 152, 126, 218, 175, 179, 167, 180, 247, 142, 188, 122, 147, 165, 166, 161, 114, 147, 126, 107, 94, 144, 159, 151, 134, 135, 143, 105, 122, 98, 117, 126, 130, 103, 122, 131, 125, 127, 136, 122, 74, 117, 95, 111, 115, 86, 133, 155, 134, 125, 141, 121, 126, 160, 140, 119, 152, 147, 158, 157, 129, 188, 213, 212, 220, 234, 217, 224, 288, 299, 269, 318, 422, 509, 585, 702, 865, 1162, 1429, 1705, 1866, 2132, 2234, 2315, 2445, 2339, 2215, 1922, 1664, 1436, 1267, 949, 793, 586, 480, 405, 310, 292, 249, 242, 277, 271, 219, 197, 202, 212, 234, 203, 185, 230, 167, 170, 217, 209, 195, 228, 225, 231, 281, 260, 280, 290, 306, 320, 304, 313, 380, 407, 425, 520, 655, 819, 981, 1182, 1376, 1530, 1734, 2047, 2050, 2149, 2091, 1913, 1810, 1619, 1391, 1175, 1002, 775, 653, 531, 434, 359, 313, 260, 275, 259, 226, 220, 244, 249, 260, 255, 238, 286, 293, 367, 442, 508, 619, 803, 992, 1269, 1296, 1623, 1847, 2084, 2052, 2050, 2139, 2005, 1781, 1685, 1400, 1277, 1168, 1042, 1066, 1129, 1357, 1459, 1627, 1910, 2053, 2291, 2626, 2680, 2588, 2585, 2365, 2123, 1926, 1677, 1491, 1303, 1314, 1256, 1315, 1458, 1622, 1913, 2063, 2225, 2582, 2628, 2689, 2700, 2567, 2334, 2158, 1904, 1660, 1474, 1279, 1298, 1416, 1465, 1705, 1872, 2041, 2217, 2441, 2576, 2605, 2440, 2450, 2161, 1895, 1648, 1343, 1166, 1000, 807, 574, 486, 428, 415, 323, 266, 303, 289, 244, 290, 248, 207, 278, 258, 271, 255, 268, 247, 202, 203, 250, 234, 225, 182, 226, 225, 179, 151, 191, 169, 153, 148, 151, 145, 113, 132, 131, 140, 157, 131, 134, 163, 129, 112, 131, 132, 136, 183, 127, 158, 164, 167, 164, 169, 206, 241, 185, 207, 193, 213, 163, 183, 203, 166, 161, 142, 153, 112, 159, 143, 137, 130, 67, 82, 124, 120, 112, 96, 126, 106, 98, 109, 141, 140, 143, 153, 108, 122, 136, 147, 143, 129, 138, 136, 109, 107, 144, 142, 149, 125, 138, 182, 129, 139, 151, 147, 157, 145, 160, 184, 195, 201, 223, 232, 196, 238, 232, 224, 272, 264, 253, 289, 288, 365, 425, 528, 642, 865, 1186, 1390, 1733, 2016, 2109, 2377, 2360, 2438, 2402, 2245, 2235, 2071, 1784, 1495, 1294, 915, 696, 572, 526, 434, 378, 361, 336, 308, 294, 321, 318, 303, 288, 305, 284, 352, 342, 407, 499, 613, 713, 879, 1089, 1295, 1581, 1695, 1986, 2071, 2082, 2046, 1886, 1799, 1625, 1370, 1221, 928, 772, 581, 491, 433, 413, 251, 239, 291, 238, 239, 202, 198, 163, 216, 201, 212, 172, 153, 144, 157, 174, 172, 170, 167, 147, 172, 155, 143, 136, 155, 161, 173, 139, 138, 154, 118, 114, 141, 113, 122, 144, 151, 146, 156, 185, 174, 190, 237, 226, 210, 211, 234, 236, 313, 382, 400, 494, 598, 816, 1008, 1196, 1332, 1688, 1888, 1960, 2057, 2101, 2101, 1931, 1749, 1617, 1388, 1188, 975, 709, 627, 495, 410, 364, 303, 248, 250, 256, 233, 188, 191, 203, 148, 147, 153, 143, 137, 137, 95, 125, 126, 127, 174, 153, 170, 194, 204, 187, 185, 176, 217, 196, 216, 216, 257, 281, 350, 419, 499, 586, 734, 858, 1075, 1249, 1352, 1451, 1548, 1613, 1483, 1504, 1414, 1297, 1112, 937, 814, 686, 525, 408, 390, 307, 253, 229, 212, 220, 168, 136, 183, 188, 183, 210, 242, 211, 166, 210, 219, 186, 157, 158, 158, 172, 180, 128, 117, 125, 138, 89, 138, 114, 126, 135, 108, 106, 96, 110, 110, 109, 125, 157, 147, 121, 143, 167, 170, 168, 205, 208, 189, 198, 219, 237, 217, 222, 212, 264, 220, 213, 256, 295, 285, 269, 288, 285, 295, 295, 318, 405, 468, 531, 670, 849, 1091, 1249, 1471, 1846, 2087, 2243, 2364, 2541, 2492, 2478, 2265, 2085, 1797, 1631, 1427, 1296, 1240, 1201, 1294, 1420, 1622, 1851, 2016, 2394, 2507, 2700, 2814, 2694, 2620, 2483, 2255, 2008, 1770, 1562, 1380, 1267, 1334, 1362, 1480, 1565, 1752, 1942, 2119, 2260, 2265, 2266, 2318, 2190, 2096, 1766, 1474, 1368, 1183, 940, 697, 537, 451, 364, 283, 267, 236, 240, 236, 192, 170, 158, 188, 185, 173, 200, 196, 211, 205, 209, 161, 147, 224, 202, 152, 162, 205, 240, 220, 198, 129, 127, 159, 171, 118, 113, 133, 152, 143, 110, 159, 154, 138, 175, 146, 120, 149, 173, 183, 187, 231, 206, 215, 237, 347, 393, 476, 562, 764, 965, 1154, 1307, 1533, 1750, 1731, 1847, 1916, 1875, 1801, 1616, 1410, 1151, 981, 855, 723, 607, 445, 400, 312, 286, 275, 285, 244, 199, 186, 227, 214, 208, 243, 227, 179, 197, 160, 156, 152, 188, 153, 150, 154, 159, 162, 141, 130, 145, 146, 148, 131, 123, 144, 160, 165, 202, 179, 176, 230, 213, 216, 209, 269, 265, 259, 265, 253, 276, 317, 356, 405, 527, 688, 804, 1087, 1253, 1468, 1737, 1826, 1957, 1942, 1931, 1901, 1900, 1745, 1573, 1438, 1208, 1023, 800, 632, 531, 446, 374, 330, 252, 217, 210, 180, 173, 196, 176, 186, 166, 169, 176, 166, 193, 209, 201, 188, 230, 219, 228, 220, 240, 264, 247, 261, 254, 290, 287, 307, 382, 449, 544, 668, 811, 1028, 1306, 1454, 1613, 1759, 1925, 2044, 2041, 2031, 1929, 1692, 1551, 1274, 1173, 953, 796, 659, 554, 384, 359, 326, 274, 311, 279, 290, 325, 322, 298, 338, 334, 402, 468, 643, 686, 842, 995, 1152, 1422, 1651, 2020, 2230, 2308, 2409, 2412, 2356, 2372, 2141, 1953, 1702, 1517, 1382, 1216, 1131, 1186, 1240, 1370, 1488, 1700, 1947, 2101, 2296, 2380, 2384, 2349, 2277, 2003, 1741, 1485, 1255, 1071, 853, 747, 561, 463, 415, 352, 258, 239, 226, 222, 220, 190, 189, 176, 154, 162, 179, 127, 154, 158, 178, 157, 168, 134, 187, 172, 200, 189, 168, 156, 144, 166, 137, 180, 178, 148, 172, 173, 161, 164, 202, 193, 182, 175, 189, 208, 250, 211, 213, 259, 234, 230, 290, 293, 366, 387, 515, 642, 768, 934, 1148, 1419, 1609, 1765, 1908, 2052, 2151, 2014, 1852, 1792, 1693, 1437, 1195, 1047, 850, 639, 505, 381, 308, 300, 244, 211, 159, 172, 173, 162, 146, 175, 126, 88, 110, 121, 115, 90, 113, 124, 125, 111, 111, 102, 120, 138, 159, 163, 163, 142, 142, 147, 136, 121, 118, 151, 161, 127, 156, 139, 147, 165, 147, 112, 159, 169, 144, 129, 139, 145, 135, 157, 141, 111, 130, 129, 145, 159, 158, 171, 193, 196, 182, 158, 254, 209, 188, 220, 228, 246, 268, 302, 335, 308, 426, 525, 615, 723, 892, 1043, 1227, 1355, 1582, 1702, 1734, 1683, 1750, 1622, 1611, 1290, 1136, 979, 878, 641, 503, 475, 391, 277, 234, 237, 233, 220, 196, 191, 205, 188, 164, 169, 189, 169, 186, 177, 202, 168, 193, 180, 179, 173, 191, 202, 190, 229, 220, 223, 199, 229, 257, 275, 331, 381, 458, 608, 688, 759, 1000, 1191, 1493, 1680, 1783, 1886, 1933, 1959, 2003, 1863, 1776, 1541, 1343, 1230, 983, 786, 640, 563, 465, 405, 301, 230, 279, 252, 174, 203, 212, 171, 215, 157, 150, 132, 159, 157, 131, 138, 118, 133, 150, 153, 147, 157, 143, 181, 139, 164, 188, 165, 164, 174, 146, 122, 159, 138, 184, 146, 150, 156, 139, 144, 162, 138, 195, 165, 176, 182, 189, 169, 195, 186, 194, 155, 163, 197, 207, 173, 224, 190, 203, 249, 264, 307, 275, 268, 245, 236, 272, 279, 356, 364, 414, 557, 624, 763, 1023, 1131, 1284, 1499, 1639, 1810, 1901, 1966, 1982, 1922, 1720, 1544, 1448, 1331, 1114, 907, 753, 611, 542, 419, 324, 293, 256, 215, 209, 211, 200, 234, 232, 241, 262, 252, 286, 327, 358, 472, 612, 759, 916, 1019, 1210, 1434, 1561, 1741, 1732, 1778, 1734, 1680, 1535, 1454, 1263, 1182, 958, 798, 667, 538, 465, 369, 281, 248, 203, 168, 194, 206, 177, 195, 155, 151, 198, 179, 179, 162, 136, 189, 170, 141, 147, 158, 186, 195, 184, 187, 163, 195, 177, 184, 191, 169, 212, 205, 172, 210, 158, 184, 225, 247, 191, 244, 211, 236, 255, 273, 249, 249, 261, 323, 356, 401, 472, 530, 647, 844, 1039, 1151, 1394, 1630, 1774, 1971, 2155, 2191, 2266, 2116, 1973, 1848, 1637, 1472, 1335, 1272, 1223, 1235, 1319, 1306, 1465, 1608, 1829, 1900, 2074, 2198, 2158, 2168, 2056, 1868, 1800, 1589, 1332, 1111, 951, 783, 606, 569, 402, 372, 325, 347, 249, 195, 174, 173, 166, 181, 119, 142, 123, 159, 101, 122, 104, 122, 108, 96, 105, 112, 128, 95, 73, 115, 110, 98, 89, 87, 92, 111, 88, 75, 88, 72, 85, 81, 86, 82, 101, 91, 97, 82, 51, 91, 105, 93, 105, 85, 94, 103, 86, 107, 127, 120, 146, 159, 166, 176, 114, 135, 154, 174, 159, 172, 149, 172, 152, 176, 156, 148, 164, 132, 148, 194, 136, 162, 142, 125, 128, 120, 154, 148, 162, 119, 178, 156, 139, 142, 189, 174, 171, 158, 123, 114, 171, 135, 137, 133, 120, 149, 146, 140, 167, 126, 104, 130, 151, 130, 189, 121, 105, 133, 161, 187, 140, 151, 181, 162, 154, 138, 134, 128, 116, 125, 148, 108, 125, 156, 149, 149, 138, 134, 169, 165, 161, 199, 187, 235, 261, 301, 335, 400, 418, 562, 738, 912, 1067, 1275, 1350, 1523, 1729, 1698, 1660, 1723, 1677, 1548, 1473, 1269, 1034, 907, 744, 577, 509, 438, 375, 285, 312, 260, 249, 198, 191, 198, 223, 277, 300, 335, 382, 470, 585, 653, 848, 1087, 1235, 1458, 1669, 1822, 1904, 1939, 1895, 1928, 1805, 1609, 1493, 1285, 1131, 942, 747, 614, 488, 445, 349, 268, 251, 202, 179, 195, 197, 159, 155, 155, 185, 139, 130, 145, 130, 119, 164, 132, 141, 127, 121, 136, 141, 193, 134, 149, 147, 171, 137, 136, 93, 134, 133, 140, 138, 179, 150, 151, 164, 162, 154, 179, 185, 186, 189, 222, 235, 282, 344, 437, 508, 624, 742, 893, 1040, 1311, 1465, 1687, 1755, 1937, 2033, 2044, 1958, 1794, 1709, 1579, 1326, 1161, 954, 762, 606, 532, 438, 368, 281, 239, 180, 170, 162, 139, 170, 132, 151, 137, 105, 130, 115, 98, 93, 113, 117, 107, 118, 96, 118, 107, 96, 105, 124, 89, 112, 95, 133, 129, 97, 112, 110, 135, 96, 125, 142, 119, 109, 129, 177, 127, 161, 173, 129, 144, 116, 158, 163, 179, 158, 159, 189, 185, 156, 153, 155, 163, 190, 208, 188, 171, 161, 202, 225, 200, 233, 257, 315, 371, 402, 534, 651, 806, 883, 1144, 1295, 1441, 1525, 1733, 1821, 1815, 1752, 1739, 1735, 1494, 1272, 1141, 968, 886, 688, 512, 440, 383, 289, 264, 200, 199, 192, 189, 160, 124, 136, 141, 174, 135, 157, 154, 161, 158, 146, 156, 157, 145, 177, 201, 186, 205, 180, 206, 219, 237, 278, 336, 378, 403, 453, 554, 713, 910, 1034, 1287, 1449, 1653, 1813, 1867, 2006, 1976, 1955, 1930, 1731, 1641, 1477, 1351, 1196, 1157, 1203, 1281, 1256, 1404, 1560, 1582, 1759, 2010, 2168, 2118, 2239, 2221, 1985, 1935, 1745, 1551, 1350, 1172, 916, 719, 647, 507, 438, 355, 324, 291, 235, 195, 220, 175, 189, 170, 122, 155, 159, 164, 143, 125, 140, 159, 180, 165, 208, 209, 218, 221, 191, 241, 222, 261, 276, 323, 362, 401, 511, 642, 713, 906, 1040, 1250, 1426, 1613, 1668, 1858, 1933, 1971, 1935, 1879, 1789, 1643, 1405, 1208, 1072, 926, 794, 610, 573, 473, 375, 353, 344, 304, 261, 318, 257, 265, 261, 255, 297, 314, 326, 333, 445, 529, 599, 777, 875, 1221, 1339, 1429, 1567, 1640, 1884, 1865, 1865, 1823, 1633, 1517, 1415, 1291, 1118, 974, 824, 666, 496, 415, 365, 310, 275, 245, 236, 245, 233, 176, 146, 182, 129, 114, 131, 167, 146, 112, 140, 113, 123, 145, 117, 99, 106, 141, 107, 121, 98, 90, 135, 94, 103, 89, 106, 149, 122, 150, 135, 123, 120, 122, 195, 123, 143, 167, 142, 146, 150, 137, 140, 143, 121, 121, 112, 101, 129, 104, 116, 105, 113, 87, 97, 124, 111, 126, 102, 107, 110, 145, 121, 127, 144, 134, 163, 191, 153, 192, 163, 182, 192, 199, 142, 187, 127, 102, 126, 133, 144, 129, 135, 114, 128, 111, 129, 116, 105, 113, 121, 129, 159, 120, 147, 115, 121, 64, 99, 107, 138, 146, 164, 136, 155, 171, 122, 179, 141, 145, 182, 159, 129, 131, 122, 105, 114, 135, 142, 134, 127, 109, 105, 106, 136, 140, 132, 132, 113, 108, 142, 108, 121, 130, 133, 116, 166, 177, 148, 160, 168, 176, 160, 202, 223, 214, 243, 243, 266, 253, 317, 396, 467, 549, 714, 829, 934, 1079, 1309, 1469, 1586, 1783, 1870, 1796, 1760, 1759, 1683, 1454, 1282, 1141, 1011, 793, 697, 624, 490, 378, 347, 342, 310, 274, 232, 244, 249, 244, 212, 233, 295, 293, 338, 404, 430, 516, 651, 748, 879, 1123, 1223, 1376, 1565, 1626, 1650, 1655, 1773, 1759, 1677, 1607, 1389, 1302, 1089, 959, 791, 649, 569, 490, 380, 301, 301, 297, 228, 176, 150, 193, 186, 220, 170, 182, 208, 228, 131, 146, 145, 155, 121, 115, 163, 159, 112, 142, 124, 139, 114, 140, 123, 152, 124, 140, 166, 133, 171, 197, 205, 143, 209, 202, 154, 161, 210, 178, 136, 159, 159, 165, 178, 139, 131, 123, 131, 156, 126, 132, 136, 104, 122, 129, 116, 165, 141, 130, 103, 113, 143, 111, 142, 178, 174, 191, 215, 193, 245, 169, 203, 252, 281, 277, 277, 345, 348, 374, 390, 452, 623, 707, 823, 956, 1152, 1283, 1562, 1715, 1914, 2002, 2118, 2056, 2078, 1983, 1943, 1781, 1659, 1558, 1460, 1382, 1331, 1397, 1435, 1692, 1820, 2009, 2198, 2321, 2282, 2416, 2472, 2394, 2219, 2184, 1939, 1783, 1549, 1354, 1140, 855, 669, 518, 474, 408, 319, 241, 212, 217, 189, 150, 161, 123, 98, 146, 120, 107, 130, 148, 113, 119, 137, 152, 170, 153, 149, 147, 150, 141, 170, 146, 150, 166, 158, 139, 156, 190, 151, 114, 168, 161, 169, 204, 139, 175, 174, 151, 156, 168, 152, 151, 152, 215, 185, 155, 166, 160, 160, 149, 137, 132, 154, 139, 129, 130, 134, 122, 136, 128, 123, 103, 143, 129, 145, 147, 134, 129, 171, 147, 160, 177, 142, 155, 155, 139, 153, 142, 120, 162, 140, 129, 102, 137, 129, 118, 142, 140, 126, 126, 123, 126, 130, 142, 150, 96, 155, 147, 143, 164, 111, 139, 138, 134, 126, 130, 127, 127, 120, 126, 135, 131, 136, 156, 115, 99, 106, 104, 118, 128, 161, 110, 124, 152, 115, 122, 191, 154, 194, 158, 183, 167, 151, 175, 150, 165, 150, 171, 137, 190, 195, 173, 191, 161, 196, 223, 237, 221, 199, 249, 288, 336, 380, 432, 466, 660, 702, 863, 1050, 1191, 1438, 1586, 1726, 1912, 1893, 2014, 2021, 1947, 1860, 1777, 1716, 1587, 1426, 1300, 1433, 1415, 1459, 1587, 1730, 1940, 2015, 2299, 2388, 2469, 2484, 2546, 2484, 2401, 2231, 2184, 1955, 1796, 1593, 1497, 1474, 1563, 1516, 1620, 1657, 1882, 2060, 2157, 2179, 2305, 2190, 2141, 2102, 1860, 1708, 1648, 1625, 1453, 1435, 1362, 1347, 1440, 1505, 1579, 1723, 1788, 1840, 1794, 1828, 1828, 1787, 1621, 1484, 1417, 1215, 1009, 886, 788, 653, 565, 485, 444, 402, 355, 280, 276, 228, 265, 238, 182, 196, 187, 169, 159, 159, 140, 153, 126, 162, 157, 169, 137, 132, 137, 134, 119, 102, 117, 119, 121, 131, 114, 127, 122, 119, 121, 126, 137, 117, 117, 142, 133, 131, 124, 102, 120, 119, 129, 132, 145, 136, 75, 129, 129, 157, 171, 132, 145, 184, 180, 226, 204, 225, 185, 135, 150, 182, 189, 196, 197, 188, 193, 172, 176, 202, 188, 201, 208, 200, 210, 269, 227, 278, 279, 347, 394, 489, 596, 708, 792, 880, 1093, 1266, 1513, 1688, 1891, 1890, 2024, 2066, 2143, 2095, 1996, 1896, 1785, 1668, 1583, 1477, 1508, 1603, 1640, 1614, 1728, 1943, 2102, 2136, 2136, 2486, 2503, 2349, 2273, 2065, 1939, 1787, 1467, 1230, 1041, 1003, 793, 607, 570, 447, 411, 334, 342, 352, 319, 326, 306, 308, 323, 341, 376, 451, 552, 574, 699, 861, 901, 1079, 1234, 1351, 1444, 1531, 1634, 1667, 1627, 1622, 1488, 1417, 1221, 1175, 982, 906, 707, 589, 450, 418, 323, 250, 259, 240, 223, 252, 245, 206, 193, 236, 202, 167, 190, 169, 140, 153, 176, 171, 164, 169, 170, 188, 176, 156, 163, 163, 178, 155, 159, 155, 148, 131, 122, 142, 137, 139, 108, 159, 168, 168, 132, 145, 151, 156, 240, 230, 281, 311, 348, 404, 501, 595, 715, 794, 957, 1026, 1149, 1346, 1428, 1549, 1588, 1538, 1656, 1601, 1462, 1265, 1166, 1015, 951, 825, 622, 472, 388, 333, 331, 252, 244, 190, 164, 201, 138, 123, 147, 163, 179, 152, 161, 146, 141, 127, 167, 120, 160, 148, 136, 176, 158, 204, 223, 246, 271, 305, 317, 405, 502, 613, 747, 840, 868, 1126, 1219, 1367, 1463, 1567, 1685, 1682, 1728, 1657, 1511, 1433, 1297, 1119, 982, 858, 735, 658, 504, 430, 368, 330, 216, 235, 209, 232, 215, 145, 182, 143, 137, 172, 203, 178, 171, 118, 197, 179, 132, 166, 155, 193, 206, 215, 210, 220, 226, 306, 417, 439, 532, 681, 804, 897, 1104, 1223, 1392, 1550, 1668, 1752, 1751, 1824, 1817, 1814, 1770, 1645, 1326, 1117, 1031, 832, 755, 599, 465, 440, 378, 356, 312, 244, 239, 173, 152, 199, 203, 204, 143, 129, 166, 139, 129, 165, 131, 138, 130, 171, 186, 124, 130, 151, 151, 171, 153, 161, 163, 142, 116, 127, 139, 139, 131, 132, 131, 115, 121, 128, 116, 112, 117, 105, 101, 89, 108, 73, 105, 71, 101, 95, 97, 93, 86, 79, 100, 98, 87, 91, 101, 98, 96, 82, 120, 111, 124, 90, 127, 83, 84, 130, 108, 101, 116, 94, 99, 110, 148, 165, 178, 128, 116, 195, 228, 198, 248, 355, 355, 429, 458, 517, 651, 746, 948, 1059, 1262, 1477, 1513, 1658, 1752, 1840, 1883, 1856, 1730, 1726, 1539, 1418, 1389, 1315, 1297, 1338, 1386, 1414, 1515, 1548, 1614, 1790, 1866, 1865, 1775, 1784, 1759, 1595, 1439, 1368, 1142, 1064, 892, 776, 626, 561, 478, 362, 312, 234, 208, 213, 203, 201, 183, 180, 150, 153, 168, 164, 138, 171, 178, 159, 169, 190, 168, 194, 223, 239, 255, 236, 300, 324, 376, 449, 508, 565, 715, 889, 1092, 1205, 1315, 1454, 1561, 1673, 1818, 1750, 1845, 1792, 1779, 1708, 1581, 1524, 1498, 1424, 1422, 1340, 1356, 1516, 1673, 1686, 1790, 1850, 2057, 2154, 2126, 2136, 2013, 1992, 1892, 1751, 1726, 1574, 1456, 1495, 1511, 1500, 1634, 1623, 1739, 1753, 1869, 1875, 1932, 2031, 1882, 1692, 1558, 1505, 1374, 1203, 991, 872, 685, 609, 525, 431, 332, 254, 263, 234, 168, 181, 176, 159, 122, 105, 139, 123, 125, 126, 135, 139, 152, 118, 139, 151, 191, 153, 181, 182, 240, 258, 297, 342, 356, 435, 500, 600, 657, 828, 1011, 1154, 1275, 1467, 1491, 1443, 1549, 1450, 1479, 1450, 1378, 1303, 1180, 1078, 888, 799, 720, 593, 508, 397, 349, 283, 262, 191, 190, 125, 91, 133, 108, 127, 135, 105, 123, 143, 119, 131, 153, 112, 115, 128, 125, 124, 110, 108, 106, 148, 123, 93, 109, 156, 143, 126, 70, 129, 167, 187, 137, 134, 150, 158, 202, 139, 133, 126, 128, 99, 163, 136, 110, 138, 166, 169, 134, 104, 118, 120, 143, 146, 152, 148, 128, 130, 117, 116, 106, 142, 175, 136, 175, 140, 163, 199, 181, 186, 187, 193, 169, 214, 188, 234, 253, 353, 365, 496, 501, 605, 651, 781, 976, 1186, 1209, 1385, 1594, 1613, 1682, 1742, 1709, 1543, 1525, 1509, 1355, 1171, 1037, 936, 661, 586, 568, 484, 416, 296, 238, 247, 201, 171, 167, 188, 160, 134, 121, 111, 157, 107, 113, 123, 108, 87, 70, 70, 94, 107, 94, 86, 66, 74, 75, 87, 100, 73, 80, 60, 60, 104, 80, 59, 76, 86, 56, 79, 62, 91, 78, 72, 66, 53, 93, 90, 115, 103, 72, 66, 116, 82, 92, 68, 76, 108, 97, 93, 121, 113, 70, 146, 132, 156, 139, 133, 125, 132, 123, 116, 140, 130, 145, 125, 125, 115, 160, 177, 148, 164, 144, 139, 144, 163, 145, 134, 144, 108, 136, 103, 142, 131, 123, 139, 129, 142, 140, 187, 184, 142, 156, 149, 101, 125, 145, 137, 156, 120, 150, 117, 144, 168, 193, 204, 244, 230, 217, 297, 304, 318, 430, 476, 557, 717, 820, 995, 1182, 1287, 1452, 1695, 1720, 1813, 1969, 2110, 1996, 1946, 1943, 1799, 1778, 1793, 1604, 1634, 1605, 1464, 1450, 1667, 1732, 1799, 1899, 2010, 2018, 2078, 2187, 2232, 2100, 2002, 1859, 1819, 1725, 1559, 1482, 1400, 1410, 1386, 1445, 1496, 1632, 1730, 1740, 1765, 1697, 1713, 1808, 1689, 1531, 1479, 1358, 1224, 1086, 920, 846, 712, 568, 468, 405, 360, 345, 300, 291, 278, 316, 336, 396, 452, 519, 588, 653, 805, 915, 1045, 1205, 1282, 1438, 1608, 1606, 1676, 1632, 1651, 1586, 1493, 1378, 1197, 1090, 972, 847, 768, 662, 522, 425, 425, 322, 259, 269, 281, 218, 237, 201, 184, 224, 196, 213, 200, 157, 168, 163, 148, 163, 175, 177, 141, 156, 174, 180, 158, 147, 168, 170, 145, 129, 141, 122, 130, 141, 141, 99, 117, 117, 115, 127, 112, 130, 125, 118, 149, 145, 144, 113, 128, 157, 130, 155, 120, 136, 132, 97, 126, 87, 102, 77, 93, 34, 92, 80, 85, 90, 94, 98, 102, 57, 31, 86, 85, 73, 101, 86, 82, 117, 60, 39, 79, 119, 98, 110, 109, 103, 100, 78, 87, 79, 113, 88, 62, 90, 73, 109, 107, 70, 133, 104, 94, 103, 74, 95, 97, 121, 103, 77, 97, 85, 93, 68, 73, 63, 60, 60, 81, 97, 79, 84, 85, 54, 58, 34, 65, 59, 80, 69, 73, 54, 77, 74, 101, 75, 71, 98, 64, 87, 104, 91, 86, 82, 78, 65, 105, 74, 77, 88, 94, 52, 87, 75, 67, 52, 48, 61, 52, 81, 64, 44, 54, 100, 89, 70, 65, 78, 74, 116, 111, 99, 112, 113, 100, 101, 99, 100, 88, 107, 113, 57, 78, 108, 89, 100, 91, 70, 80, 104, 117, 94, 134, 103, 152, 122, 104, 93, 127, 84, 103, 165, 104, 100, 120, 80, 92, 130, 160, 113, 139, 126, 146, 138, 161, 141, 139, 138, 164, 141, 125, 148, 202, 147, 138, 184, 147, 215, 255, 209, 283, 280, 324, 367, 409, 481, 600, 693, 885, 1061, 1132, 1256, 1275, 1392, 1497, 1596, 1646, 1600, 1464, 1461, 1309, 1145, 1061, 946, 851, 747, 592, 515, 430, 300, 251, 195, 192, 183, 182, 164, 137, 110, 146, 138, 93, 99, 72, 102, 123, 103, 135, 82, 104, 164, 140, 85, 100, 128, 117, 102, 136, 100, 99, 140, 129, 143, 117, 154, 158, 147, 161, 130, 183, 123, 138, 123, 120, 107, 118, 97, 122, 113, 116, 108, 111, 117, 123, 95, 135, 135, 159, 137, 153, 153, 178, 181, 207, 202, 245, 259, 283, 348, 385, 470, 587, 699, 794, 910, 967, 1007, 1104, 1131, 1172, 1144, 1223, 1158, 1124, 1060, 987, 829, 787, 691, 564, 464, 376, 324, 319, 261, 227, 229, 165, 172, 146, 91, 154, 143, 152, 115, 117, 114, 92, 84, 105, 94, 107, 96, 113, 93, 105, 89, 117, 106, 93, 110, 121, 144, 105, 148, 145, 149, 138, 143, 164, 182, 186, 137, 154, 186, 176, 138, 192, 190, 175, 206, 187, 200, 204, 196, 172, 209, 227, 252, 274, 254, 295, 278, 254, 261, 319, 332, 366, 472, 481, 550, 654, 753, 855, 994, 1068, 1210, 1377, 1316, 1499, 1612, 1602, 1674, 1614, 1556, 1602, 1405, 1502, 1521, 1457, 1393, 1455, 1520, 1559, 1554, 1645, 1771, 1833, 1861, 1860, 1774, 1830, 1752, 1725, 1518, 1388, 1300, 1170, 993, 802, 672, 603, 525, 391, 327, 266, 272, 249, 237, 200, 158, 130, 166, 178, 171, 126, 158, 152, 154, 175, 159, 168, 148, 196, 176, 171, 180, 240, 256, 281, 281, 354, 441, 484, 618, 688, 858, 928, 1002, 1162, 1176, 1229, 1418, 1360, 1436, 1365, 1289, 1161, 1021, 942, 901, 785, 622, 557, 479, 453, 387, 288, 247, 180, 216, 165, 155, 131, 128, 113, 111, 110, 75, 58, 76, 98, 80, 73, 86, 84, 67, 62, 92, 91, 107, 94, 115, 116, 74, 81, 91, 60, 115, 75, 73, 84, 76, 50, 74, 62, 68, 83, 88, 92, 118, 107, 92, 86, 159, 109, 109, 139, 128, 158, 121, 150, 141, 118, 160, 158, 173, 149, 174, 130, 158, 157, 168, 151, 98, 127, 139, 154, 137, 127, 115, 131, 109, 130, 100, 143, 119, 121, 113, 145, 122, 135, 150, 142, 148, 179, 246, 239, 195, 227, 234, 254, 271, 326, 357, 414, 480, 547, 645, 816, 916, 998, 1050, 1185, 1332, 1432, 1478, 1592, 1570, 1650, 1535, 1552, 1502, 1473, 1343, 1154, 986, 887, 767, 692, 599, 497, 429, 381, 361, 312, 301, 349, 312, 299, 341, 337, 391, 425, 493, 586, 618, 696, 790, 927, 954, 1034, 1177, 1281, 1361, 1341, 1338, 1407, 1318, 1299, 1378, 1314, 1305, 1306, 1269, 1341, 1362, 1362, 1418, 1411, 1484, 1583, 1673, 1808, 1788, 1790, 1757, 1701, 1690, 1660, 1575, 1492, 1467, 1458, 1417, 1437, 1577, 1521, 1522, 1619, 1663, 1726, 1692, 1770, 1692, 1713, 1628, 1573, 1531, 1533, 1478, 1544, 1509, 1430, 1465, 1480, 1611, 1603, 1609, 1691, 1709, 1637, 1684, 1606, 1457, 1316, 1281, 1118, 1031, 871, 772, 699, 550, 447, 411, 368, 309, 291, 292, 277, 314, 324, 314, 320, 443, 483, 478, 525, 662, 738, 806, 938, 1074, 1175, 1232, 1229, 1292, 1411, 1324, 1307, 1366, 1407, 1334, 1283, 1298, 1202, 1222, 1348, 1290, 1378, 1436, 1427, 1497, 1545, 1565, 1569, 1541, 1431, 1405, 1296, 1259, 1082, 931, 875, 837, 749, 655, 527, 396, 372, 328, 273, 285, 229, 198, 166, 129, 154, 130, 133, 128, 110, 125, 91, 108, 126, 123, 98, 121, 109, 77, 73, 76, 51, 94, 73, 74, 94, 82, 84, 76, 66, 66, 94, 76, 82, 65, 47, 68, 103, 111, 76, 73, 49, 63, 49, 83, 75, 97, 100, 87, 79, 92, 101, 81, 82, 76, 78, 104, 99, 52, 103, 91, 97, 101, 92, 78, 117, 112, 92, 124, 111, 95, 105, 118, 122, 101, 131, 135, 113, 89, 93, 137, 112, 135, 103, 116, 111, 108, 97, 125, 77, 92, 114, 88, 128, 129, 165, 142, 122, 121, 127, 104, 148, 171, 131, 102, 133, 123, 130, 98, 121, 156, 86, 107, 109, 97, 136, 109, 107, 105, 51, 91, 97, 79, 73, 103, 55, 56, 97, 78, 45, 89, 92, 53, 81, 90, 111, 98, 80, 103, 95, 112, 121, 133, 90, 125, 122, 129, 129, 157, 141, 133, 99, 155, 131, 145, 138, 162, 133, 151, 164, 169, 168, 156, 158, 109, 145, 138, 126, 91, 124, 93, 121, 129, 108, 147, 95, 89, 128, 129, 120, 121, 89, 132, 90, 80, 105, 91, 89, 97, 89, 99, 91, 111, 76, 116, 78, 124, 121, 111, 86, 119, 104, 119, 89, 118, 85, 93, 107, 104, 104, 139, 166, 142, 180, 164, 130, 198, 253, 244, 290, 323, 360, 423, 487, 655, 688, 751, 857, 975, 1084, 1249, 1260, 1360, 1475, 1580, 1595, 1492, 1496, 1573, 1489, 1450, 1419, 1388, 1437, 1485, 1423, 1467, 1467, 1581, 1529, 1675, 1620, 1636, 1706, 1649, 1550, 1478, 1423, 1381, 1245, 1035, 975, 834, 704, 587, 551, 453, 393, 351, 315, 294, 250, 242, 209, 216, 200, 198, 186, 167, 179, 136, 129, 148, 115, 128, 115, 114, 81, 117, 92, 126, 104, 104, 102, 136, 99, 125, 84, 77, 102, 122, 106, 105, 114, 118, 123, 140, 97, 103, 120, 148, 140, 145, 159, 158, 152, 160, 195, 178, 201, 236, 205, 232, 239, 281, 330, 398, 379, 394, 485, 518, 627, 719, 938, 1019, 1106, 1186, 1301, 1454, 1608, 1722, 1618, 1749, 1713, 1595, 1532, 1421, 1435, 1239, 1138, 1031, 980, 779, 646, 545, 439, 411, 356, 303, 275, 220, 194, 202, 146, 117, 158, 125, 122, 164, 101, 138, 138, 153, 100, 162, 146, 133, 126, 153, 119, 113, 131, 119, 94, 98, 120, 91, 88, 118, 144, 118, 90, 98, 86, 73, 136, 127, 69, 89, 75, 91, 80, 88, 89, 105, 121, 92, 102, 102, 104, 125, 101, 103, 122, 106, 106, 104, 147, 105, 112, 125, 132, 112, 122, 98, 112, 123, 137, 145, 146, 122, 169, 141, 153, 169, 145, 137, 141, 115, 139, 186, 151, 161, 166, 166, 187, 174, 194, 175, 175, 129, 134, 142, 172, 177, 163, 130, 105, 112, 112, 99, 108, 83, 98, 86, 118, 93, 112, 90, 106, 121, 89, 102, 104, 150, 130, 139, 138, 143, 174, 224, 228, 222, 259, 238, 284, 338, 425, 493, 536, 596, 700, 732, 845, 990, 1049, 1189, 1254, 1302, 1389, 1422, 1366, 1408, 1292, 1292, 1209, 1106, 1098, 1015, 805, 758, 669, 589, 469, 413, 340, 312, 253, 217, 183, 183, 173, 184, 147, 136, 160, 160, 146, 144, 147, 136, 164, 160, 166, 195, 190, 218, 223, 246, 306, 346, 343, 370, 454, 483, 479, 651, 716, 839, 999, 1040, 1148, 1264, 1331, 1421, 1462, 1614, 1519, 1435, 1488, 1347, 1229, 1239, 1069, 949, 832, 800, 748, 618, 543, 521, 392, 312, 306, 297, 241, 218, 229, 181, 220, 211, 224, 183, 159, 172, 192, 175, 162, 150, 140, 136, 136, 110, 152, 124, 105, 158, 129, 148, 168, 149, 122, 158, 153, 152, 119, 142, 108, 116, 134, 120, 122, 123, 127, 122, 92, 91, 54, 87, 105, 110, 76, 80, 97, 92, 101, 99, 101, 104, 83, 104, 118, 101, 78, 75, 77, 60, 85, 92, 99, 75, 97, 70, 90, 99, 73, 79, 66, 98, 80, 81, 128, 90, 127, 110, 110, 80, 91, 109, 135, 131, 151, 183, 177, 185, 215, 224, 257, 315, 396, 523, 511, 590, 671, 695, 793, 877, 936, 937, 1093, 1159, 1140, 1134, 1146, 1143, 1114, 1060, 937, 845, 800, 687, 555, 488, 463, 421, 370, 362, 289, 242, 239, 243, 226, 179, 188, 144, 122, 157, 113, 135, 105, 116, 145, 127, 110, 105, 117, 121, 103, 77, 117, 126, 111, 72, 91, 111, 114, 141, 178, 148, 153, 126, 131, 109, 89, 138, 141, 148, 137, 99, 138, 120, 105, 96, 94, 80, 96, 120, 120, 131, 85, 77, 112, 101, 81, 99, 127, 85, 121, 125, 130, 149, 85, 111, 131, 115, 129, 165, 159, 137, 126, 150, 114, 161, 142, 139, 93, 101, 131, 131, 112, 65, 81, 136, 107, 78, 85, 76, 101, 86, 104, 81, 98, 111, 98, 106, 119, 81, 42, 77, 82, 91, 91, 105, 98, 115, 90, 93, 93, 79, 67, 100, 100, 130, 120, 93, 97, 109, 107, 116, 136, 94, 104, 102, 116, 114, 74, 110, 116, 122, 117, 86, 117, 101, 96, 102, 88, 96, 79, 102, 99, 118, 108, 87, 122, 79, 118, 120, 100, 117, 125, 137, 158, 142, 106, 121, 113, 130, 119, 125, 165, 123, 115, 100, 149, 122, 112, 121, 132, 113, 160, 151, 121, 145, 100, 97, 143, 150, 158, 145, 127, 93, 117, 147, 152, 144, 146, 176, 173, 202, 201, 209, 245, 239, 257, 299, 402, 449, 501, 615, 661, 737, 850, 968, 1031, 1148, 1220, 1300, 1283, 1429, 1441, 1453, 1504, 1466, 1391, 1421, 1397, 1365, 1277, 1307, 1380, 1411, 1369, 1423, 1394, 1478, 1555, 1564, 1430, 1346, 1302, 1275, 1216, 1135, 1060, 986, 923, 827, 735, 654, 511, 431, 397, 350, 315, 264, 216, 217, 172, 161, 142, 139, 118, 155, 152, 155, 132, 99, 91, 100, 105, 117, 92, 130, 106, 100, 91, 110, 87, 99, 87, 115, 119, 98, 107, 86, 97, 104, 110, 104, 81, 59, 100, 56, 91, 67, 84, 34, 84, 83, 82, 56, 53, 42, 44, 63, 58, 77, 80, 48, 46, 53, 48, 63, 57, 45, 39, 46, 62, 97, 53, 98, 68, 74, 61, 79, 66, 104, 88, 92, 74, 77, 93, 66, 61, 107, 73, 73, 118, 94, 65, 71, 53, 75, 92, 65, 90, 60, 84, 100, 70, 104, 72, 101, 126, 106, 90, 98, 89, 79, 109, 121, 127, 95, 117, 138, 114, 92, 98, 104, 108, 87, 117, 111, 111, 97, 71, 102, 93, 132, 115, 101, 136, 121, 122, 112, 129, 97, 89, 82, 128, 121, 148, 104, 117, 113, 106, 111, 133, 121, 138, 90, 83, 82, 115, 126, 97, 89, 92, 93, 73, 93, 119, 126, 123, 116, 115, 100, 113, 120, 120, 130, 104, 103, 174, 158, 138, 152, 128, 154, 184, 199, 281, 311, 347, 368, 415, 488, 536, 599, 677, 746, 795, 792, 851, 1013, 974, 1015, 1045, 972, 989, 929, 831, 818, 822, 738, 601, 543, 457, 408, 365, 304, 322, 312, 227, 207, 210, 216, 142, 174, 136, 165, 166, 159, 157, 139, 159, 155, 139, 153, 177, 171, 184, 192, 148, 162, 150, 148, 135, 136, 136, 102, 99, 136, 108, 108, 91, 85, 115, 101, 97, 101, 108, 92, 111, 94, 82, 97, 97, 111, 111, 108, 82, 37, 85, 80, 110, 83, 77, 79, 112, 98, 72, 85, 85, 78, 80, 107, 127, 109, 150, 142, 205, 211, 200, 249, 284, 288, 340, 412, 389, 460, 509, 584, 676, 783, 845, 882, 977, 1018, 1051, 1044, 1045, 1080, 993, 986, 952, 883, 826, 794, 651, 565, 538, 481, 413, 385, 352, 298, 269, 238, 219, 200, 204, 167, 173, 166, 188, 159, 155, 157, 165, 148, 144, 157, 164, 168, 117, 96, 83, 109, 107, 148, 119, 116, 132, 140, 156, 143, 136, 124, 120, 117, 85, 112, 110, 95, 104, 143, 133, 118, 101, 105, 154, 151, 124, 134, 123, 117, 119, 130, 143, 109, 174, 166, 191, 180, 142, 155, 175, 168, 198, 169, 190, 205, 227, 212, 243, 254, 271, 312, 344, 442, 427, 504, 585, 624, 722, 822, 904, 971, 1123, 1215, 1247, 1352, 1488, 1521, 1496, 1492, 1342, 1273, 1290, 1181, 1074, 1031, 931, 839, 764, 646, 532, 558, 464, 412, 342, 326, 284, 272, 271, 232, 194, 220, 234, 189, 184, 195, 160, 168, 143, 151, 155, 144, 165, 136, 101, 103, 143, 125, 134, 146, 147, 166, 155, 151, 173, 188, 180, 152, 104, 136, 136, 123, 164, 158, 137, 144, 183, 157, 194, 139, 129, 143, 141, 120, 158, 151, 153, 158, 175, 214, 173, 180, 180, 190, 209, 181, 170, 192, 242, 278, 269, 286, 335, 348, 388, 446, 464, 511, 609, 692, 792, 825, 929, 1023, 1104, 1277, 1332, 1431, 1408, 1510, 1629, 1727, 1698, 1650, 1607, 1671, 1718, 1615, 1624, 1645, 1687, 1728, 1709, 1825, 1798, 1849, 1976, 1927, 1874, 1879, 2051, 1957, 1978, 1863, 1899, 1800, 1717, 1709, 1692, 1684, 1657, 1752, 1746, 1685, 1719, 1734, 1785, 1716, 1709, 1680, 1727, 1770, 1621, 1603, 1604, 1523, 1568, 1601, 1556, 1513, 1554, 1570, 1568, 1522, 1450, 1456, 1424, 1387, 1283, 1214, 1181, 1078, 1031, 893, 786, 704, 617, 578, 525, 457, 372, 329, 315, 263, 272, 229, 174, 183, 226, 209, 185, 184, 189, 214, 188, 179, 172, 152, 103, 116, 153, 141, 110, 135, 109, 124, 147, 198, 172, 155, 167, 179, 216, 205, 252, 264, 324, 310, 316, 387, 406, 472, 630, 698, 737, 809, 873, 980, 1033, 1034, 1108, 1120, 1168, 1162, 1137, 1172, 1050, 977, 918, 893, 803, 690, 641, 540, 514, 480, 413, 350, 345, 272, 263, 243, 234, 231, 219, 214, 230, 181, 188, 154, 192, 158, 209, 203, 214, 196, 188, 203, 175, 221, 211, 200, 232, 186, 170, 191, 192, 170, 177, 150, 141, 191, 196, 195, 151, 215, 232, 238, 256, 251, 267, 244, 260, 261, 294, 305, 297, 296, 322, 461, 471, 531, 562, 617, 689, 740, 882, 973, 1051, 1163, 1173, 1283, 1380, 1499, 1493, 1600, 1674, 1670, 1632, 1682, 1694, 1750, 1754, 1691, 1722, 1796, 1684, 1673, 1806, 1867, 1859, 1927, 2007, 1960, 1978, 2006, 1979, 1879, 1846, 1867, 1778, 1764, 1700, 1660, 1656, 1700, 1700, 1714, 1687, 1625, 1621, 1627, 1675, 1589, 1467, 1427, 1389, 1267, 1186, 1038, 878, 845, 786, 781, 656, 576, 557, 451, 352, 298, 307, 272, 273, 272, 243, 267, 256, 243, 211, 197, 237, 229, 205, 223, 197, 231, 231, 271, 221, 224, 222, 246, 230, 186, 210, 264, 260, 222, 294, 269, 292, 309, 367, 371, 413, 434, 467, 537, 608, 698, 780, 862, 940, 920, 949, 1116, 1238, 1304, 1358, 1416, 1531, 1551, 1537, 1559, 1540, 1578, 1579, 1580, 1585, 1618, 1538, 1592, 1574, 1644, 1698, 1644, 1630, 1675, 1707, 1687, 1608, 1507, 1451, 1352, 1167, 1076, 980, 839, 821, 766, 644, 602, 496, 523, 474, 447, 433, 436, 428, 429, 513, 563, 590, 662, 684, 731, 839, 868, 828, 922, 984, 951, 909, 970, 904, 850, 825, 840, 763, 706, 695, 564, 544, 488, 435, 348, 357, 304, 268, 230, 221, 202, 180, 176, 155, 128, 173, 193, 157, 163, 191, 219, 212, 226, 262, 301, 306, 355, 378, 355, 410, 515, 619, 698, 823, 815, 856, 919, 903, 1054, 1090, 1056, 1093, 1045, 956, 985, 982, 882, 773, 747, 646, 552, 492, 448, 391, 326, 292, 235, 241, 215, 187, 178, 174, 104, 159, 136, 131, 129, 123, 74, 122, 123, 97, 110, 119, 130, 127, 180, 164, 140, 124, 135, 105, 116, 150, 130, 160, 144, 145, 150, 146, 135, 125, 151, 149, 113, 149, 150, 132, 174, 167, 182, 182, 147, 133, 162, 118, 165, 210, 185, 168, 192, 177, 172, 163, 204, 207, 212, 202, 215, 235, 267, 254, 276, 351, 422, 426, 553, 599, 648, 695, 784, 948, 1002, 1111, 1131, 1266, 1409, 1422, 1612, 1627, 1583, 1530, 1569, 1595, 1596, 1558, 1633, 1683, 1679, 1650, 1681, 1703, 1601, 1732, 1649, 1650, 1599, 1490, 1434, 1409, 1347, 1255, 1228, 1175, 1108, 1041, 954, 906, 801, 680, 590, 529, 537, 494, 485, 455, 473, 457, 444, 502, 514, 588, 581, 714, 727, 840, 827, 960, 1053, 1075, 1167, 1158, 1185, 1190, 1264, 1276, 1214, 1250, 1303, 1296, 1330, 1413, 1366, 1371, 1355, 1383, 1376, 1443, 1308, 1347, 1354, 1335, 1245, 1206, 1254, 1217, 1027, 901, 795, 757, 641, 539, 476, 527, 412, 367, 328, 241, 269, 255, 228, 200, 150, 168, 187, 128, 147, 154, 176, 144, 130, 136, 120, 106, 115, 142, 125, 149, 124, 122, 136, 111, 125, 134, 121, 88, 126, 107, 109, 114, 114, 105, 90, 92, 81, 128, 129, 97, 86, 106, 130, 140, 97, 108, 147, 88, 88, 120, 94, 113, 91, 109, 123, 119, 131, 130, 139, 125, 130, 117, 109, 114, 134, 142, 101, 161, 112, 162, 157, 122, 143, 172, 163, 159, 126, 113, 139, 132, 156, 123, 122, 153, 113, 101, 133, 119, 155, 165, 118, 132, 120, 112, 171, 147, 169, 182, 213, 187, 229, 267, 310, 304, 351, 415, 453, 493, 554, 611, 683, 749, 846, 928, 882, 902, 932, 934, 904, 934, 834, 882, 825, 800, 690, 648, 608, 569, 519, 413, 361, 299, 272, 242, 220, 211, 202, 183, 130, 120, 100, 116, 91, 80, 97, 121, 103, 111, 96, 113, 69, 77, 107, 101, 64, 70, 109, 78, 85, 103, 98, 89, 55, 133, 129, 103, 51, 79, 98, 106, 129, 101, 106, 149, 133, 97, 146, 137, 175, 171, 175, 235, 260, 281, 262, 256, 408, 439, 474, 487, 547, 601, 670, 770, 781, 829, 893, 950, 970, 964, 955, 994, 992, 979, 924, 885, 807, 756, 682, 586, 524, 488, 424, 391, 403, 345, 310, 251, 248, 221, 183, 211, 233, 159, 181, 177, 183, 201, 164, 190, 207, 225, 243, 249, 279, 325, 368, 433, 443, 540, 525, 606, 710, 821, 835, 913, 915, 974, 1098, 1167, 1126, 1185, 1198, 1194, 1151, 1085, 985, 1016, 958, 847, 783, 739, 691, 572, 529, 511, 503, 391, 359, 375, 417, 434, 447, 456, 510, 515, 542, 577, 568, 669, 708, 752, 819, 828, 879, 922, 898, 950, 908, 844, 828, 803, 730, 700, 647, 589, 574, 581, 495, 397, 371, 332, 248, 268, 188, 174, 128, 147, 196, 148, 165, 133, 112, 106, 126, 122, 136, 110, 102, 81, 150, 108, 109, 123, 89, 140, 141, 125, 137, 135, 142, 155, 149, 131, 146, 129, 166, 132, 149, 148, 147, 148, 148, 156, 204, 179, 196, 167, 222, 242, 223, 262, 329, 361, 394, 415, 478, 513, 598, 696, 724, 838, 892, 863, 976, 1043, 1101, 1094, 1122, 1074, 1053, 1050, 1010, 1018, 926, 856, 776, 731, 619, 591, 586, 545, 497, 448, 424, 419, 316, 325, 300, 284, 273, 245, 217, 232, 215, 269, 242, 249, 274, 315, 296, 328, 384, 419, 456, 536, 565, 587, 667, 724, 852, 996, 980, 1035, 1088, 1175, 1262, 1257, 1265, 1318, 1270, 1376, 1372, 1376, 1367, 1386, 1398, 1447, 1506, 1475, 1562, 1517, 1505, 1561, 1445, 1432, 1426, 1285, 1255, 1313, 1154, 1044, 969, 856, 796, 688, 646, 558, 533, 422, 368, 320, 311, 260, 215, 220, 201, 164, 187, 176, 145, 149, 140, 108, 125, 113, 113, 120, 161, 119, 131, 149, 152, 134, 128, 106, 125, 144, 152, 117, 143, 125, 127, 174, 145, 111, 119, 114, 153, 150, 139, 129, 104, 117, 94, 113, 83, 119, 137, 134, 177, 144, 175, 136, 114, 114, 156, 146, 149, 136, 95, 136, 124, 99, 91, 129, 131, 90, 86, 76, 82, 82, 85, 117, 118, 103, 113, 91, 96, 84, 101, 111, 88, 97, 129, 106, 112, 158, 147, 121, 111, 166, 161, 192, 176, 183, 205, 205, 221, 242, 273, 306, 351, 434, 467, 487, 581, 621, 749, 788, 778, 873, 968, 988, 989, 1018, 1099, 1068, 1119, 1091, 1029, 965, 882, 869, 823, 701, 654, 646, 626, 581, 439, 382, 417, 297, 256, 283, 271, 235, 189, 173, 131, 155, 139, 134, 154, 99, 134, 109, 126, 122, 100, 108, 130, 115, 83, 83, 110, 145, 108, 136, 133, 160, 129, 117, 134, 125, 136, 97, 116, 112, 145, 119, 127, 143, 145, 154, 116, 101, 94, 138, 136, 96, 131, 135, 106, 105, 125, 120, 127, 128, 117, 119, 123, 97, 111, 76, 106, 118, 142, 110, 110, 108, 136, 100, 105, 95, 99, 89, 108, 94, 115, 107, 130, 142, 104, 112, 77, 69, 84, 62, 107, 121, 127, 91, 81, 129, 105, 88, 129, 100, 95, 108, 86, 108, 112, 116, 71, 71, 85, 58, 90, 73, 49, 50, 71, 77, 56, 75, 60, 83, 90, 83, 70, 102, 52, 75, 67, 84, 125, 93, 94, 70, 74, 108, 95, 91, 132, 133, 99, 81, 123, 96, 122, 106, 141, 134, 100, 126, 134, 158, 93, 127, 154, 110, 112, 135, 141, 122, 105, 119, 151, 162, 137, 116, 124, 145, 150, 136, 147, 94, 73, 77, 120, 104, 146, 101, 110, 94, 96, 64, 88, 81, 77, 124, 103, 90, 103, 110, 113, 125, 97, 89, 119, 163, 93, 114, 111, 126, 133, 97, 104, 103, 113, 120, 92, 148, 136, 119, 81, 95, 113, 109, 84, 113, 124, 137, 94, 113, 129, 111, 115, 133, 109, 164, 135, 88, 119, 129, 117, 145, 140, 109, 118, 184, 140, 127, 146, 139, 140, 218, 179, 165, 243, 230, 236, 273, 338, 342, 361, 437, 450, 501, 554, 594, 659, 777, 855, 915, 990, 1076, 1094, 1178, 1315, 1360, 1323, 1260, 1315, 1376, 1401, 1387, 1361, 1309, 1316, 1332, 1370, 1408, 1410, 1411, 1391, 1302, 1261, 1278, 1231, 1145, 1088, 1032, 1017, 945, 810, 774, 683, 626, 576, 528, 402, 388, 319, 287, 257, 224, 174, 205, 173, 164, 158, 196, 197, 206, 209, 224, 277, 267, 252, 296, 306, 383, 449, 515, 542, 553, 621, 721, 775, 789, 850, 870, 925, 919, 949, 1005, 986, 929, 936, 917, 854, 850, 829, 757, 644, 612, 554, 485, 478, 402, 331, 320, 339, 256, 205, 209, 219, 192, 180, 217, 193, 200, 207, 136, 190, 232, 237, 224, 248, 280, 311, 313, 377, 424, 421, 430, 460, 547, 611, 623, 581, 594, 659, 672, 703, 711, 663, 677, 684, 654, 642, 567, 470, 488, 463, 371, 349, 332, 305, 243, 208, 191, 143, 157, 115, 121, 123, 109, 124, 122, 132, 115, 96, 105, 91, 66, 111, 111, 56, 92, 109, 117, 114, 95, 85, 100, 88, 55, 113, 135, 125, 90, 151, 153, 162, 123, 90, 109, 119, 149, 157, 181, 206, 200, 257, 200, 287, 320, 344, 370, 413, 423, 528, 618, 636, 671, 727, 821, 866, 971, 1011, 1053, 1119, 1063, 1114, 1127, 1065, 1003, 965, 911, 848, 817, 787, 750, 693, 561, 493, 452, 457, 394, 320, 304, 297, 279, 204, 218, 174, 177, 160, 168, 125, 100, 131, 113, 95, 94, 98, 104, 88, 57, 123, 113, 105, 132, 111, 114, 108, 167, 187, 170, 194, 174, 211, 266, 273, 334, 331, 394, 464, 477, 577, 611, 690, 760, 791, 843, 857, 880, 884, 890, 925, 931, 888, 888, 897, 842, 778, 729, 678, 594, 506, 501, 442, 404, 371, 313, 273, 237, 221, 199, 164, 163, 141, 154, 98, 98, 126, 129, 98, 105, 96, 115, 81, 74, 68, 75, 93, 81, 85, 109, 112, 72, 109, 69, 96, 71, 111, 107, 91, 101, 92, 119, 107, 122, 93, 106, 77, 100, 91, 129, 110, 95, 69, 139, 108, 149, 177, 147, 130, 141, 138, 115, 131, 142, 110, 131, 145, 159, 178, 233, 232, 247, 259, 256, 318, 347, 386, 479, 505, 542, 644, 606, 690, 779, 814, 862, 941, 972, 1015, 1083, 1161, 1163, 1125, 1166, 1236, 1219, 1215, 1213, 1264, 1270, 1233, 1247, 1269, 1285, 1258, 1213, 1141, 1180, 1095, 1022, 1011, 971, 900, 795, 711, 680, 565, 512, 467, 397, 389, 371, 319, 269, 300, 234, 214, 211, 180, 182, 180, 191, 158, 145, 110, 134, 175, 127, 179, 132, 120, 112, 142, 129, 129, 137, 119, 137, 122, 117, 96, 127, 115, 139, 126, 135, 146, 144, 170, 172, 168, 175, 221, 216, 261, 318, 365, 355, 385, 464, 512, 527, 586, 700, 737, 819, 836, 932, 1005, 981, 1022, 1062, 1013, 985, 1074, 1072, 974, 1001, 972, 893, 813, 769, 691, 631, 561, 499, 427, 423, 420, 366, 347, 265, 230, 237, 209, 177, 178, 132, 114, 130, 122, 105, 169, 97, 98, 59, 146, 116, 107, 91, 130, 119, 139, 121, 159, 131, 118, 115, 102, 93, 113, 132, 181, 160, 182, 138, 176, 169, 219, 215, 239, 298, 286, 366, 388, 431, 488, 528, 649, 713, 812, 841, 907, 926, 966, 1031, 1042, 1152, 1220, 1211, 1180, 1195, 1178, 1126, 1020, 968, 957, 868, 821, 750, 649, 574, 518, 530, 424, 463, 374, 307, 272, 270, 272, 182, 195, 178, 158, 154, 189, 167, 179, 135, 127, 142, 150, 104, 121, 111, 101, 135, 145, 87, 111, 127, 143, 101, 146, 103, 96, 106, 107, 139, 127, 134, 132, 73, 76, 93, 102, 96, 108, 105, 72, 58, 99, 99, 94, 87, 97, 82, 93, 98, 109, 77, 123, 102, 108, 118, 117, 143, 136, 165, 194, 182, 183, 225, 285, 312, 358, 408, 420, 462, 539, 596, 594, 652, 670, 694, 721, 713, 746, 756, 736, 789, 733, 681, 612, 591, 537, 475, 480, 452, 407, 374, 344))
        self.assertEqual(record.annotations["abif_raw"]['DATA4'], (-7, 4, -13, 8, -7, -1, -6, -6, -1, -6, 3, -22, -10, -5, 2, -4, -12, -4, -2, -5, 12, 0, -4, -25, 0, -15, 3, -4, -12, -11, -1, -13, -13, -3, -13, -14, 8, -20, -5, 6, 2, -15, 7, -12, 7, 13, -9, -5, -4, 5, -6, -6, -17, -14, -17, -9, -15, -7, -8, -9, -6, -9, -12, -17, -17, -9, -19, 1, -9, -4, -10, -10, -8, -12, 3, -13, -16, -12, -9, -6, -5, -14, 4, -8, -6, -7, -6, -15, -3, 0, -21, 1, -5, 0, -16, -8, -4, -11, -13, -1, -8, -20, -17, -16, 2, 3, -14, -14, -10, -3, -11, -6, -15, 8, -4, -8, -20, -1, 5, -15, -11, -2, -14, -15, 8, -7, -7, -17, -1, -17, -12, -20, -3, -7, -7, 0, -2, -4, -18, -5, 2, 0, 4, -10, -3, -10, -11, -13, -10, 9, 0, -12, -1, -6, 3, -7, 10, -1, 8, -6, -9, -13, 22, -10, -3, 9, -11, 7, -1, 13, 5, 20, 1, -12, -4, -1, 16, 29, 36, 13, 11, 2, 21, 3, 7, 20, 10, -4, 14, 4, -1, 7, 7, 6, -3, 7, 9, -1, 5, -2, 0, 10, -3, 1, 13, 13, 16, 9, 4, 14, -6, 18, 6, 4, -10, 18, 25, 16, 1, 2, 3, 0, 0, -5, 0, 10, -7, 13, 4, 7, 10, 6, 11, -16, 19, 5, -4, 12, 27, 34, 17, 22, -1, 2, 22, 10, 22, 11, 6, 6, 11, 3, 10, 22, 15, 14, 8, 18, 7, 15, 7, 2, 10, 14, 15, 30, 13, 12, 21, -4, 11, 17, 22, 19, 16, 16, 23, 10, 20, 21, 8, 23, 13, 3, 13, 12, 13, 14, 4, 10, 9, 16, 15, 25, 5, 7, 12, 22, 25, -1, 20, 22, 13, 6, 19, 29, 22, 18, 16, 20, 13, 39, 7, 15, 21, 5, 8, 0, 9, 17, 11, 19, 17, 29, 14, 18, 9, 25, 41, 12, 16, 25, 20, 19, 15, 17, 15, 18, 26, 16, -2, 18, 17, 25, 35, 12, 20, 12, 19, 12, 22, 12, 12, 16, 43, 16, 17, 2, 30, 20, 4, 12, 14, 20, 13, 19, 25, 9, 28, 20, 22, 27, 6, 23, 15, 16, 17, 8, 16, 13, 13, 15, 7, 13, 11, 5, 10, 10, 17, 15, 14, 28, 20, 12, 12, 7, 13, 14, 3, 21, 15, 5, 20, 13, 21, 18, 23, 23, 13, 9, 18, 10, 15, 20, 22, 15, 29, 18, 33, 35, 9, 14, 30, 25, 19, 12, 20, 32, 11, 23, 11, 20, 14, 13, 12, 26, 4, 12, 29, 28, 19, 28, 41, 15, 28, 20, 20, 9, 15, 10, 14, 34, 27, 14, 25, 20, 24, 16, 11, 14, 15, 15, 24, 15, 20, 30, 14, 30, 13, 10, 5, 33, 19, 22, 6, 15, 23, 9, 18, 16, 26, 43, 19, 34, 30, 32, 17, 16, 22, -2, 19, 14, 22, 18, 18, 20, 15, 14, 8, 25, 14, 20, 18, 31, 22, 13, 34, 20, 28, 22, 17, 16, 22, 19, 23, 17, -1, 23, 10, 15, 35, 21, 19, 34, 12, 6, 12, 8, 25, 19, 5, 26, 14, 11, 26, 17, 26, 22, 10, 8, 18, 12, 15, 17, 4, 16, -2, 26, 8, 8, 13, 27, 1, 28, 26, 9, 24, 17, 6, 8, 14, 11, 12, 26, 13, 8, 34, 29, 23, 16, 7, 0, -11, 10, -5, 18, 14, 1, 26, 19, 27, 22, 29, 13, 19, 18, 22, 11, 13, 11, 19, 30, 11, 31, 0, 27, 10, 16, 20, 4, 0, 9, 10, 2, -2, 6, 3, 16, 12, -4, 17, 0, 23, 12, 12, 7, 11, 0, 14, 9, 11, 11, 4, 34, 6, 12, 14, 4, 13, 20, 12, 22, 8, 4, 5, 5, -6, 4, 12, -4, 15, 12, 1, 20, 2, 7, 23, -6, 18, 16, 10, 11, 26, 14, -5, 4, 5, 11, 1, 18, 19, 5, 5, 24, 6, -5, 9, 13, 4, 9, 10, 13, 0, 7, 16, 25, 6, 3, 9, 14, 27, 0, 14, 9, 9, 28, 18, 4, 1, 11, 7, 5, 11, 17, 10, -1, -6, 1, 4, 21, 18, 8, 8, 14, 4, 6, 5, 10, 8, 1, 19, 13, -1, 6, 2, 3, 0, -21, 13, -5, -4, 12, 23, 3, 2, -8, 1, 14, 1, 10, 18, 15, 6, 10, 15, 6, 19, 6, 11, 0, -10, -7, 15, 12, 1, 15, -12, -6, 15, 5, 1, -15, 5, 34, -7, -8, 11, 0, 26, 3, -2, 10, 1, 10, -1, 2, 17, 3, 7, 13, 13, 5, 7, 8, 0, 1, -7, 10, 18, 0, 10, 2, 14, 4, 24, -2, 20, 14, 20, 15, 8, 20, 31, 30, 42, 12, 30, 35, 49, 47, 45, 45, 47, 47, 65, 69, 84, 77, 106, 110, 125, 132, 148, 147, 174, 181, 189, 160, 140, 112, 137, 104, 98, 83, 68, 47, 56, 52, 35, 35, 37, 27, 48, 24, 37, 38, 15, 12, 11, 20, 15, 9, 19, 12, 25, 19, 16, 19, 31, 16, 26, 26, 9, 17, 24, 21, 28, 21, 26, 18, 25, 18, 11, 34, 19, 37, 28, 48, 40, 37, 18, 27, 34, 26, 19, 33, 32, 25, 19, 28, 39, 37, 21, 3, 20, 21, 42, 40, 30, 9, 42, 17, 23, 25, 34, 21, 17, 26, 12, 23, 15, 28, 27, 28, 22, 36, 17, 26, 16, 13, 23, 15, 14, 14, 18, 21, 23, 8, 12, 10, 16, 23, 4, 17, 35, 10, 21, 5, 16, 11, 24, 23, 17, 18, 23, 12, 31, 6, 15, 29, 13, 13, 27, 5, 27, 23, 40, 0, 18, 26, 13, 29, 10, 31, 9, 32, 21, 26, 25, 0, 28, 10, 12, 27, 11, 12, 26, 32, 18, 23, 5, 16, 16, 14, 21, 10, 21, 29, 16, 21, 14, 15, 24, 21, 29, 6, 8, 27, 8, 14, 3, 17, 5, 21, -3, 17, 16, 18, 14, 4, 14, 12, 4, 19, 15, 21, 11, 27, 19, 11, 2, 15, 11, 14, 19, 8, 16, 7, 22, 1, 18, 12, 16, 15, 20, 3, 14, 29, 11, 15, 2, 14, 10, 5, 9, 10, 16, 10, 6, 19, 12, -1, 25, 9, 23, 9, 12, 17, 1, 7, 4, 19, 2, 19, 5, 6, 5, 19, 0, 2, 20, 28, 6, 6, 13, 24, 13, 11, -1, 19, 18, 5, 19, 19, 0, 15, 9, 5, 7, 21, 13, 5, 13, 22, 12, 15, 14, 23, 22, 28, 18, 35, 15, 31, 28, 21, 3, 16, 30, 40, 15, 19, 21, 33, 15, 24, 16, 26, 27, 30, 22, 12, 34, 35, 27, 38, 36, 37, 38, 27, 32, 37, 25, 32, 32, 38, 58, 59, 37, 39, 58, 45, 39, 31, 16, 42, 24, 29, 38, 36, 21, 21, 24, 18, 37, 24, 26, 29, 25, 29, 24, 18, 35, 19, 21, 17, 19, 24, 15, 20, 24, 31, 27, 20, 34, 39, 15, 33, 33, 43, 28, 16, 38, 45, 36, 39, 23, 25, 25, 35, 22, 36, 32, 49, 46, 33, 40, 18, 35, 31, 37, 19, 29, 30, 13, 25, 28, 31, 15, 19, 22, 36, 37, 16, 31, 22, 24, 49, 29, 34, 25, 32, 19, 17, 36, 25, 27, 24, 12, 30, 28, 34, 33, 43, 23, 9, 35, 31, 38, 15, 32, 24, 44, 32, 30, 20, 19, 15, 27, 19, 16, 25, 29, 22, -5, 19, 29, 22, 19, 37, 16, 9, 24, 22, 21, 32, 16, 39, 20, 5, 21, 30, 38, 12, 12, 5, 18, 24, 13, 22, 11, 28, 5, 17, 21, 15, 30, 15, 25, 15, 31, 27, 13, 23, 20, 14, 20, 22, 36, 27, 26, 33, 31, 38, 26, 30, 24, 16, 28, 31, 21, 18, 19, 25, 24, 39, 10, 15, 31, 28, 24, 20, 30, 37, 28, 24, 24, 39, 46, 27, 37, 54, 52, 44, 46, 40, 39, 29, 42, 32, 39, 35, 32, 16, 10, 14, 26, 19, 15, 28, 24, 26, 28, 17, 32, 16, 0, 13, 6, 17, 5, -2, 20, 1, 0, 19, 7, 19, 4, 3, 34, 15, 9, 10, 12, 17, 17, 19, 30, 6, 6, 27, 30, 3, 20, 14, 22, 27, 18, 20, 21, 12, 31, 22, 23, 27, 15, 18, 6, 17, 18, 20, 15, 18, 5, 13, 8, 12, 7, 29, 19, 19, 13, 11, 20, 19, 24, 11, 22, 9, 1, 27, 19, 35, 22, 12, 25, 15, 10, 1, 3, 18, 18, 9, 21, 28, 13, 15, 19, 8, 12, 8, 14, 24, 24, 22, 15, 28, 16, 4, 12, 32, 1, 21, 5, 14, 13, 1, 2, 20, 10, 24, 14, 10, 17, 0, 18, 29, 12, 16, 26, 22, -2, 18, 12, 20, 18, 19, 18, 9, 8, 16, 7, 16, 24, 3, 21, 17, 16, 13, 6, 18, 13, 15, 9, 11, 34, 13, 18, 8, 11, 5, 26, 26, 18, 1, -4, 14, 16, 6, 17, 21, 4, 24, 22, 31, 12, 22, 16, 2, 23, 19, 8, 10, 9, 5, 22, 19, -2, 13, 8, 14, 10, 15, 11, 7, 21, 13, 13, 7, 7, 13, 25, 4, 26, 21, 17, 29, 11, -1, 19, 5, 18, 15, 30, 23, 16, 24, 19, 16, 17, 21, 19, 20, 10, 12, 20, 29, 11, 16, -1, 16, 16, 10, 4, 14, 15, 15, 24, 30, 6, 5, 11, 4, 8, -4, 29, 16, 20, -4, 11, 10, 12, 7, 25, 12, 8, 18, 11, 12, 0, 8, 25, 26, 15, 16, 20, 21, 5, 5, 11, 9, 19, 2, 11, 0, 25, 3, 9, 4, 13, 4, 11, 15, 12, -8, 22, 13, 18, 33, 11, 22, 13, 19, 14, 13, 12, 39, 22, 29, 37, 38, 27, 29, 39, 31, 50, 32, 35, 43, 42, 25, 43, 29, 15, 19, 24, 16, 17, 30, 13, 21, 21, 22, 19, 11, 17, 16, 21, 33, 21, 12, 13, 29, 21, 18, 20, 24, 9, 19, 34, 18, 9, 27, 33, 26, 13, 6, 19, 15, 22, 24, 5, 31, 11, 26, 51, 23, 24, 8, 20, 27, 18, 25, 31, 18, 28, 43, 30, 18, 22, 6, 16, 27, 30, 17, 21, 18, 29, 14, 36, 29, 22, 34, 2, 20, 4, 25, 14, 24, 7, 29, 28, 13, 26, 18, 13, 19, 17, 19, 17, 6, 20, 12, 33, 31, 31, 29, 36, 42, 36, 34, 29, 36, 35, 32, 18, 25, 35, 17, 25, 5, 11, 23, 5, 8, 12, 26, 19, 10, 23, 38, 10, 14, 22, 21, 31, 25, 25, 22, 7, 20, 13, 16, 36, 20, 18, 31, 14, 22, 5, 10, 19, 9, 6, 27, 19, 28, 19, 10, 17, 8, 23, 14, 19, 0, 26, 43, 27, 24, 19, 27, -2, 19, 29, 28, 16, 14, 20, 14, 29, 17, 22, 11, 22, 7, 16, 15, 18, 16, 9, 13, 24, 17, 28, 24, 0, 13, 18, 15, 18, 24, 21, 23, 2, 10, 21, 23, 21, 15, 17, 16, -2, 18, 5, 33, 21, 3, 25, 4, 18, 10, 18, 22, 32, 18, 20, 37, 13, 10, 14, 4, 8, 9, 8, 22, 19, 7, 8, 17, 9, 17, 3, 24, 20, 16, 25, 30, 21, 23, 7, 21, 15, 2, 18, -3, 22, 22, 30, 38, 18, 29, 18, 13, 8, 3, 12, 25, 16, 31, 21, 12, 8, 14, 16, 23, 0, 12, 12, 10, 30, 21, 23, 22, 17, 10, 7, 23, 15, 23, 24, 24, 18, 12, 13, 10, 24, 29, 15, 12, -2, 19, 28, 17, 17, 6, 8, 27, 25, 19, 1, 14, 0, 6, 11, 11, 5, 17, 8, 8, 6, -5, 6, 14, 19, 16, 6, 13, 11, 13, 23, 0, 21, 4, 18, 20, 21, 21, 6, 13, 0, 16, 28, 11, -3, -1, 5, -1, 10, 18, 1, 10, 1, 15, -7, -5, 26, 5, 13, 0, 3, -10, 4, 0, 6, 1, 2, 9, 3, 6, 7, 0, 23, 15, 14, 24, 2, 22, 11, 11, 24, 0, 11, 29, 16, 16, 13, 10, 18, 9, 26, 26, 22, 11, 15, 16, 19, 17, 22, 26, 21, 34, 29, 22, 18, 38, 37, 22, 31, 42, 45, 35, 50, 29, 30, 27, 35, 31, 28, 39, 23, 24, 37, 26, 30, 36, 16, 29, 16, 26, 15, 31, 32, 25, 25, 29, 20, 24, 20, 29, 22, 28, 33, 22, 17, 19, 23, 44, 13, 10, 21, 23, 28, 6, 10, 13, 27, 30, 19, 22, 23, 13, 15, 26, 20, 29, 12, 37, 11, 31, 26, 35, 30, 21, 5, 31, 24, 11, 29, 17, 12, 28, 20, 14, 20, 5, 32, 26, 1, 9, 10, 32, 18, 24, 31, 16, 19, 9, 5, 35, 30, 23, 12, 30, 28, 41, 35, 25, 7, 16, 9, 15, 21, 18, 36, 8, 1, 7, 13, 36, 25, 5, 16, 21, 14, 8, 22, 10, -4, 24, 0, 23, 18, 20, 9, 10, 29, 20, 16, 16, 11, 17, 0, 17, 15, 21, 19, 7, 16, 18, 5, 16, 15, 11, 30, 34, 4, 4, 14, 7, 19, 8, 45, 14, 45, 7, 17, 15, 8, 13, 17, 12, 15, 14, 26, 31, 32, 9, 28, 0, 18, 21, 0, 10, 18, 20, 27, 37, 13, 24, 19, 19, 22, 11, 5, 15, 19, 8, 19, 22, 12, -6, 27, 7, 22, 18, 15, 25, 26, 14, 23, 23, 22, 12, 6, 17, 18, 18, 15, 19, 38, 3, 10, 19, 23, 37, 26, 21, 15, 14, 10, 12, 30, 17, 20, 17, 6, 22, 47, 4, 23, 11, 11, 12, 23, 36, 22, 13, 15, 29, 28, 15, 23, 38, 20, 45, 32, 25, 27, 22, 27, 27, 40, 30, 27, 38, 47, 41, 39, 39, 44, 50, 45, 44, 41, 45, 24, 44, 34, 67, 42, 53, 50, 27, 61, 50, 31, 50, 55, 54, 61, 47, 46, 58, 44, 56, 48, 45, 59, 59, 40, 41, 40, 48, 62, 60, 39, 48, 51, 48, 38, 52, 28, 43, 35, 54, 39, 40, 52, 65, 47, 31, 52, 62, 42, 39, 44, 48, 40, 49, 52, 56, 42, 49, 44, 51, 40, 48, 40, 36, 42, 34, 47, 50, 40, 48, 28, 52, 39, 45, 52, 40, 42, 42, 36, 52, 47, 59, 53, 61, 44, 45, 55, 55, 35, 41, 52, 50, 56, 42, 46, 60, 53, 44, 47, 57, 46, 63, 50, 54, 63, 44, 55, 58, 63, 36, 66, 60, 50, 65, 67, 57, 65, 44, 66, 63, 57, 53, 55, 67, 50, 66, 67, 67, 52, 57, 67, 75, 73, 79, 58, 102, 100, 65, 138, 134, 121, 153, 179, 201, 207, 235, 242, 250, 242, 254, 264, 296, 289, 261, 245, 267, 249, 225, 210, 229, 203, 217, 242, 256, 272, 298, 349, 424, 439, 510, 621, 728, 843, 874, 979, 1110, 1168, 1208, 1296, 1325, 1290, 1330, 1361, 1338, 1269, 1220, 1157, 1126, 1111, 1039, 1062, 1124, 1049, 1086, 971, 932, 870, 807, 712, 671, 537, 473, 413, 354, 319, 272, 241, 214, 203, 192, 160, 163, 158, 155, 144, 146, 166, 147, 154, 156, 182, 160, 162, 151, 169, 170, 177, 201, 227, 259, 263, 297, 379, 497, 544, 624, 703, 864, 912, 1014, 1174, 1167, 1294, 1347, 1400, 1409, 1378, 1378, 1319, 1255, 1113, 1033, 973, 805, 786, 685, 645, 549, 535, 521, 490, 553, 560, 655, 679, 721, 760, 815, 803, 823, 837, 775, 715, 649, 602, 568, 452, 399, 326, 296, 229, 205, 164, 131, 123, 140, 132, 104, 117, 113, 118, 114, 124, 140, 114, 129, 126, 110, 136, 104, 104, 119, 112, 114, 107, 106, 100, 119, 99, 90, 112, 114, 111, 121, 98, 127, 132, 148, 141, 149, 126, 132, 157, 155, 112, 130, 140, 128, 145, 135, 122, 138, 134, 121, 136, 146, 135, 145, 136, 136, 159, 192, 209, 224, 277, 317, 352, 410, 477, 559, 620, 694, 833, 881, 914, 964, 1030, 986, 930, 884, 759, 731, 622, 532, 491, 397, 334, 332, 292, 343, 365, 435, 539, 604, 689, 817, 917, 1057, 1044, 1155, 1110, 1170, 1117, 1087, 987, 883, 843, 783, 786, 835, 795, 847, 924, 964, 969, 1023, 1024, 1021, 906, 823, 738, 626, 522, 455, 390, 327, 305, 238, 209, 208, 182, 163, 157, 143, 131, 139, 133, 134, 114, 114, 104, 95, 96, 96, 81, 83, 87, 89, 65, 75, 58, 65, 91, 92, 77, 65, 100, 82, 80, 86, 89, 58, 77, 83, 76, 74, 62, 98, 63, 52, 77, 73, 74, 65, 67, 63, 86, 97, 48, 99, 81, 74, 80, 82, 70, 66, 95, 82, 78, 79, 79, 62, 87, 78, 76, 79, 62, 62, 60, 85, 78, 75, 96, 117, 92, 92, 105, 114, 99, 101, 90, 109, 102, 119, 115, 112, 119, 134, 138, 115, 164, 183, 205, 217, 272, 372, 443, 547, 633, 773, 896, 963, 1018, 1165, 1230, 1187, 1154, 1162, 1102, 1128, 1063, 1108, 1176, 1207, 1238, 1272, 1304, 1237, 1154, 1058, 967, 804, 672, 551, 443, 339, 310, 254, 196, 181, 141, 118, 136, 128, 126, 112, 124, 104, 106, 126, 123, 117, 96, 133, 126, 167, 154, 160, 158, 162, 140, 166, 148, 144, 151, 133, 146, 160, 147, 133, 173, 179, 177, 169, 213, 223, 257, 339, 398, 478, 580, 736, 879, 1072, 1208, 1285, 1342, 1482, 1512, 1415, 1357, 1288, 1183, 1174, 1175, 1092, 1239, 1250, 1349, 1397, 1424, 1366, 1258, 1328, 1151, 952, 851, 689, 506, 423, 327, 268, 224, 198, 180, 199, 189, 228, 242, 341, 445, 517, 650, 799, 966, 1172, 1291, 1410, 1495, 1522, 1507, 1396, 1336, 1128, 1043, 853, 679, 575, 450, 373, 367, 374, 416, 466, 545, 629, 781, 894, 1012, 1113, 1226, 1223, 1201, 1203, 1081, 985, 781, 703, 526, 426, 350, 274, 239, 230, 203, 163, 168, 144, 154, 140, 143, 114, 140, 126, 148, 116, 140, 111, 149, 138, 103, 114, 112, 127, 142, 133, 147, 149, 159, 128, 101, 100, 124, 103, 118, 130, 102, 112, 129, 105, 121, 95, 113, 103, 111, 105, 91, 107, 101, 128, 127, 116, 142, 119, 125, 140, 156, 129, 153, 158, 151, 155, 153, 141, 136, 152, 142, 110, 148, 144, 114, 112, 105, 122, 130, 141, 128, 134, 206, 234, 299, 377, 517, 618, 757, 880, 1018, 1152, 1261, 1236, 1297, 1225, 1141, 1090, 1040, 1053, 1015, 1042, 1177, 1289, 1458, 1544, 1662, 1711, 1631, 1556, 1491, 1289, 1089, 840, 674, 563, 396, 336, 270, 234, 192, 177, 173, 146, 133, 155, 143, 169, 152, 138, 159, 155, 121, 140, 144, 171, 159, 168, 191, 180, 226, 275, 340, 427, 527, 679, 851, 1023, 1237, 1431, 1554, 1674, 1702, 1699, 1656, 1455, 1189, 1048, 837, 606, 495, 411, 334, 303, 243, 224, 225, 211, 229, 250, 298, 343, 466, 520, 696, 893, 1044, 1307, 1489, 1594, 1709, 1700, 1566, 1555, 1331, 1171, 897, 719, 578, 455, 353, 311, 313, 278, 319, 413, 456, 641, 748, 927, 1080, 1209, 1283, 1408, 1404, 1377, 1280, 1189, 1122, 988, 956, 965, 985, 1099, 1191, 1359, 1391, 1434, 1400, 1441, 1314, 1171, 1002, 834, 672, 531, 386, 339, 260, 226, 169, 164, 152, 143, 136, 132, 140, 111, 121, 126, 132, 124, 129, 128, 137, 159, 138, 116, 114, 137, 119, 132, 105, 105, 124, 118, 126, 131, 122, 123, 110, 93, 138, 106, 136, 121, 113, 116, 120, 128, 123, 112, 126, 127, 118, 123, 128, 123, 135, 117, 140, 157, 97, 143, 138, 145, 149, 124, 119, 126, 146, 131, 144, 129, 126, 141, 124, 106, 109, 84, 114, 115, 106, 107, 129, 129, 140, 147, 131, 160, 171, 202, 245, 303, 314, 458, 561, 758, 939, 1147, 1278, 1507, 1594, 1634, 1503, 1417, 1340, 1242, 1139, 1061, 1095, 1094, 1228, 1315, 1506, 1664, 1762, 1796, 1660, 1672, 1491, 1410, 1375, 1334, 1251, 1443, 1550, 1730, 1757, 1773, 1879, 1791, 1615, 1439, 1245, 1015, 779, 620, 475, 365, 275, 238, 214, 200, 202, 199, 183, 180, 158, 158, 179, 192, 179, 164, 185, 189, 197, 182, 181, 174, 180, 170, 201, 204, 193, 246, 271, 345, 410, 515, 703, 896, 1117, 1278, 1500, 1694, 1839, 1882, 1817, 1678, 1507, 1342, 1092, 919, 789, 655, 547, 474, 464, 469, 479, 452, 451, 529, 497, 544, 525, 541, 546, 546, 566, 543, 589, 581, 570, 534, 586, 531, 491, 496, 502, 466, 450, 440, 466, 447, 427, 417, 391, 381, 363, 370, 343, 329, 305, 299, 299, 303, 259, 227, 227, 224, 242, 234, 200, 198, 178, 192, 152, 166, 141, 142, 130, 151, 158, 142, 153, 154, 149, 132, 139, 154, 151, 140, 119, 143, 131, 113, 94, 119, 119, 127, 104, 113, 96, 91, 126, 107, 90, 98, 86, 106, 83, 94, 100, 100, 91, 88, 93, 103, 100, 103, 102, 98, 113, 124, 72, 96, 110, 124, 108, 90, 103, 100, 119, 121, 129, 131, 116, 159, 243, 267, 373, 487, 691, 895, 1144, 1369, 1540, 1664, 1776, 1760, 1682, 1470, 1288, 1034, 790, 574, 467, 349, 254, 189, 163, 126, 117, 114, 116, 118, 119, 140, 93, 119, 137, 144, 114, 135, 120, 117, 121, 111, 116, 88, 89, 77, 120, 104, 103, 105, 102, 99, 117, 113, 115, 132, 116, 148, 128, 131, 160, 196, 266, 358, 436, 570, 723, 960, 1183, 1275, 1402, 1493, 1527, 1432, 1310, 1140, 1007, 784, 606, 457, 336, 264, 207, 159, 148, 140, 140, 123, 134, 136, 131, 118, 127, 125, 111, 103, 100, 118, 103, 90, 92, 77, 104, 91, 96, 82, 85, 56, 47, 75, 86, 70, 65, 50, 50, 52, 60, 72, 55, 78, 60, 52, 47, 69, 84, 73, 60, 73, 87, 100, 62, 77, 110, 106, 120, 132, 123, 111, 127, 149, 190, 256, 322, 410, 594, 778, 980, 1225, 1474, 1617, 1736, 1699, 1734, 1551, 1367, 1127, 910, 703, 501, 398, 301, 253, 182, 171, 167, 167, 150, 142, 120, 147, 114, 104, 110, 94, 68, 76, 91, 57, 62, 58, 78, 74, 84, 73, 83, 68, 57, 75, 51, 75, 70, 90, 82, 80, 76, 59, 80, 44, 68, 88, 56, 42, 29, 47, 60, 52, 58, 69, 73, 78, 79, 89, 85, 72, 89, 91, 74, 90, 79, 84, 94, 100, 89, 117, 101, 106, 115, 118, 81, 104, 87, 92, 90, 121, 100, 102, 100, 121, 111, 85, 77, 92, 105, 82, 60, 69, 61, 72, 78, 62, 64, 68, 65, 54, 98, 60, 72, 63, 32, 53, 64, 64, 55, 74, 96, 52, 56, 69, 69, 81, 72, 77, 76, 62, 73, 75, 65, 49, 73, 53, 70, 42, 63, 58, 48, 74, 85, 80, 74, 77, 80, 67, 70, 62, 63, 87, 73, 88, 91, 75, 67, 74, 52, 71, 54, 67, 37, 50, 60, 41, 50, 55, 83, 59, 87, 46, 72, 73, 80, 60, 68, 81, 85, 91, 98, 93, 114, 133, 115, 132, 96, 100, 88, 75, 78, 56, 90, 69, 66, 83, 67, 70, 73, 94, 107, 104, 95, 92, 88, 106, 107, 84, 75, 82, 85, 74, 69, 70, 68, 65, 97, 91, 57, 70, 70, 64, 77, 70, 68, 43, 64, 61, 78, 100, 67, 70, 57, 77, 82, 73, 80, 96, 104, 144, 145, 156, 218, 293, 394, 546, 720, 901, 1087, 1214, 1432, 1483, 1422, 1303, 1223, 1017, 812, 606, 471, 371, 277, 223, 183, 149, 141, 117, 147, 121, 150, 130, 115, 113, 121, 144, 88, 86, 110, 94, 104, 106, 69, 89, 90, 100, 63, 98, 116, 103, 110, 85, 83, 96, 126, 89, 83, 102, 98, 103, 89, 80, 109, 92, 100, 100, 101, 99, 98, 80, 93, 94, 79, 92, 79, 52, 102, 75, 113, 90, 101, 100, 92, 90, 86, 80, 90, 108, 85, 104, 108, 100, 109, 114, 77, 80, 87, 101, 98, 106, 161, 170, 277, 325, 496, 621, 814, 981, 1223, 1329, 1457, 1404, 1411, 1291, 1086, 849, 701, 532, 429, 335, 249, 185, 149, 126, 112, 127, 89, 124, 93, 96, 99, 110, 102, 100, 103, 81, 113, 119, 114, 120, 115, 88, 112, 108, 113, 112, 126, 128, 139, 128, 142, 140, 138, 167, 129, 142, 142, 151, 130, 140, 145, 128, 152, 139, 146, 156, 134, 176, 125, 174, 131, 144, 147, 140, 174, 160, 168, 173, 189, 182, 166, 153, 173, 155, 160, 150, 190, 168, 155, 147, 174, 173, 174, 185, 234, 304, 364, 501, 693, 877, 1057, 1406, 1470, 1624, 1694, 1671, 1616, 1416, 1164, 1002, 785, 545, 442, 342, 257, 191, 146, 155, 161, 166, 150, 112, 117, 105, 113, 121, 139, 101, 98, 137, 137, 147, 129, 133, 120, 118, 81, 94, 82, 102, 103, 79, 107, 139, 170, 271, 375, 485, 700, 926, 1165, 1388, 1591, 1664, 1708, 1614, 1422, 1230, 1057, 794, 613, 472, 356, 243, 220, 159, 149, 130, 127, 136, 136, 141, 122, 140, 115, 120, 138, 138, 114, 135, 149, 140, 138, 140, 130, 137, 133, 107, 109, 120, 112, 119, 126, 138, 129, 131, 104, 98, 97, 69, 101, 63, 89, 67, 62, 66, 55, 80, 76, 97, 77, 92, 88, 91, 82, 84, 81, 83, 81, 71, 75, 61, 64, 58, 79, 127, 107, 97, 104, 90, 112, 102, 72, 103, 84, 110, 90, 102, 96, 127, 105, 108, 106, 106, 82, 116, 107, 88, 73, 84, 78, 81, 68, 64, 62, 110, 80, 94, 83, 81, 92, 72, 73, 71, 83, 80, 70, 108, 73, 107, 86, 87, 88, 112, 94, 77, 101, 115, 109, 105, 86, 100, 112, 198, 211, 299, 483, 606, 772, 1072, 1234, 1398, 1518, 1639, 1605, 1452, 1288, 1060, 815, 656, 520, 370, 279, 209, 144, 127, 145, 144, 123, 132, 131, 109, 127, 109, 80, 88, 94, 100, 106, 60, 120, 146, 115, 111, 123, 109, 132, 82, 112, 117, 95, 102, 102, 105, 109, 109, 141, 116, 91, 113, 113, 83, 106, 102, 81, 103, 86, 100, 97, 110, 93, 99, 87, 104, 102, 116, 80, 109, 89, 76, 44, 65, 61, 86, 85, 58, 57, 71, 42, 65, 86, 80, 66, 81, 69, 76, 62, 56, 97, 80, 82, 63, 86, 95, 81, 114, 84, 102, 112, 87, 110, 104, 123, 112, 108, 106, 104, 108, 128, 152, 235, 282, 359, 564, 754, 931, 1152, 1406, 1479, 1621, 1643, 1464, 1371, 1175, 937, 699, 554, 393, 327, 226, 188, 174, 135, 117, 122, 109, 97, 103, 95, 116, 105, 92, 101, 78, 87, 90, 82, 75, 73, 88, 80, 104, 93, 84, 108, 89, 109, 105, 93, 125, 113, 90, 120, 99, 117, 88, 77, 71, 76, 66, 67, 65, 52, 61, 74, 59, 74, 64, 62, 83, 94, 99, 59, 88, 133, 134, 162, 198, 294, 426, 606, 800, 956, 1136, 1340, 1477, 1478, 1482, 1361, 1192, 999, 815, 620, 471, 342, 274, 238, 198, 163, 163, 156, 152, 144, 126, 106, 105, 81, 76, 84, 67, 85, 90, 85, 80, 96, 98, 118, 131, 93, 137, 141, 124, 127, 153, 206, 304, 402, 564, 773, 1002, 1276, 1536, 1836, 1992, 2062, 1993, 1887, 1624, 1460, 1100, 812, 627, 419, 332, 277, 205, 195, 139, 130, 113, 127, 134, 99, 83, 106, 107, 76, 115, 101, 80, 83, 107, 81, 95, 74, 77, 86, 77, 67, 53, 78, 76, 93, 89, 79, 95, 95, 105, 130, 115, 103, 114, 124, 93, 62, 87, 77, 74, 58, 50, 82, 76, 72, 69, 61, 66, 64, 70, 59, 58, 57, 64, 39, 56, 42, 45, 67, 60, 61, 74, 81, 66, 69, 88, 71, 65, 80, 46, 48, 75, 37, 56, 59, 57, 71, 67, 78, 83, 65, 93, 66, 76, 56, 79, 40, 68, 51, 63, 60, 47, 59, 59, 48, 69, 69, 60, 63, 44, 73, 57, 77, 71, 67, 66, 54, 83, 65, 74, 79, 77, 100, 73, 70, 89, 85, 80, 86, 76, 64, 75, 108, 91, 86, 89, 89, 109, 90, 110, 99, 84, 99, 94, 73, 90, 82, 77, 78, 80, 82, 78, 71, 74, 79, 78, 66, 90, 84, 78, 79, 77, 92, 89, 74, 80, 91, 90, 80, 122, 160, 192, 248, 370, 427, 555, 692, 861, 1007, 1107, 1159, 1137, 1061, 893, 801, 637, 486, 382, 263, 227, 162, 147, 127, 100, 63, 88, 96, 67, 95, 110, 67, 76, 76, 90, 102, 77, 75, 73, 84, 69, 93, 82, 82, 54, 78, 75, 63, 59, 57, 54, 67, 46, 63, 80, 76, 61, 70, 70, 51, 78, 73, 84, 84, 104, 104, 106, 106, 129, 121, 118, 103, 109, 125, 130, 105, 134, 140, 127, 110, 107, 148, 114, 109, 113, 112, 87, 95, 90, 37, 61, 72, 91, 92, 94, 94, 92, 101, 85, 110, 106, 120, 100, 121, 96, 110, 82, 96, 96, 104, 107, 69, 83, 115, 136, 222, 289, 420, 591, 746, 961, 1182, 1287, 1387, 1425, 1396, 1329, 1180, 1008, 792, 598, 456, 344, 252, 215, 185, 182, 144, 152, 142, 121, 144, 145, 149, 171, 224, 302, 433, 541, 747, 914, 1227, 1416, 1564, 1593, 1661, 1568, 1367, 1186, 1008, 793, 624, 480, 336, 283, 236, 196, 168, 191, 145, 136, 110, 136, 126, 106, 111, 106, 113, 113, 82, 101, 113, 96, 111, 73, 100, 91, 65, 79, 68, 60, 67, 73, 53, 80, 60, 77, 81, 83, 72, 59, 98, 78, 97, 75, 81, 88, 102, 87, 117, 92, 109, 127, 109, 67, 107, 115, 123, 99, 117, 107, 93, 109, 120, 133, 151, 127, 122, 184, 170, 244, 325, 423, 606, 754, 1021, 1266, 1426, 1624, 1755, 1785, 1739, 1605, 1429, 1099, 926, 805, 695, 688, 785, 962, 1115, 1412, 1616, 1855, 1868, 1868, 1790, 1729, 1510, 1163, 986, 735, 585, 456, 315, 280, 247, 223, 205, 210, 174, 194, 191, 175, 194, 235, 336, 460, 583, 778, 1051, 1252, 1427, 1575, 1746, 1753, 1706, 1483, 1219, 1010, 804, 595, 428, 393, 283, 242, 184, 159, 158, 127, 130, 106, 136, 114, 91, 92, 93, 90, 119, 102, 55, 103, 77, 84, 85, 70, 70, 78, 76, 88, 68, 72, 81, 63, 98, 92, 97, 87, 107, 84, 85, 79, 95, 102, 99, 93, 99, 98, 115, 101, 144, 145, 133, 230, 216, 305, 398, 603, 769, 990, 1170, 1472, 1654, 1759, 1727, 1607, 1508, 1253, 990, 810, 623, 466, 365, 291, 210, 173, 147, 144, 108, 105, 91, 76, 117, 115, 84, 81, 73, 106, 108, 92, 81, 66, 71, 74, 77, 91, 78, 87, 71, 74, 79, 86, 82, 74, 85, 68, 83, 88, 68, 92, 82, 81, 98, 97, 82, 101, 102, 108, 166, 222, 298, 436, 631, 779, 1001, 1301, 1508, 1675, 1830, 1811, 1732, 1541, 1411, 1202, 906, 716, 503, 395, 286, 194, 172, 150, 118, 129, 99, 98, 116, 98, 89, 95, 87, 96, 79, 62, 98, 75, 81, 98, 115, 99, 96, 101, 96, 103, 105, 107, 112, 150, 163, 158, 213, 265, 346, 434, 613, 814, 999, 1235, 1422, 1526, 1630, 1568, 1439, 1323, 1082, 910, 692, 542, 380, 311, 253, 194, 172, 139, 132, 118, 112, 97, 124, 116, 97, 109, 114, 98, 128, 101, 112, 118, 103, 96, 79, 76, 91, 60, 82, 86, 107, 75, 65, 74, 101, 80, 101, 82, 91, 120, 104, 48, 92, 89, 87, 74, 76, 89, 75, 73, 87, 105, 105, 129, 124, 151, 131, 118, 138, 158, 113, 136, 147, 161, 210, 285, 397, 512, 642, 858, 1168, 1463, 1656, 1828, 1930, 1948, 1957, 1746, 1471, 1254, 959, 744, 656, 625, 641, 717, 923, 1155, 1339, 1532, 1654, 1719, 1803, 1690, 1501, 1262, 1059, 834, 645, 475, 361, 264, 232, 208, 160, 168, 145, 139, 123, 114, 129, 96, 98, 69, 108, 103, 98, 110, 102, 117, 102, 116, 81, 114, 100, 124, 100, 106, 131, 125, 139, 137, 159, 191, 220, 312, 406, 540, 719, 968, 1174, 1420, 1679, 1880, 1821, 1820, 1607, 1469, 1210, 992, 812, 596, 463, 360, 271, 253, 211, 198, 137, 154, 118, 137, 129, 121, 112, 108, 121, 131, 130, 125, 136, 126, 126, 86, 119, 104, 99, 88, 96, 76, 98, 92, 80, 99, 113, 100, 83, 86, 90, 95, 89, 60, 99, 46, 66, 97, 79, 85, 111, 107, 109, 148, 116, 131, 114, 152, 133, 90, 87, 87, 115, 86, 90, 90, 84, 88, 85, 93, 98, 91, 93, 104, 135, 101, 89, 46, 92, 66, 81, 75, 109, 95, 114, 126, 121, 94, 121, 95, 80, 85, 74, 88, 74, 66, 58, 50, 64, 76, 79, 63, 70, 63, 89, 69, 61, 74, 96, 89, 93, 84, 80, 118, 95, 111, 107, 118, 124, 201, 212, 269, 396, 552, 712, 936, 1290, 1529, 1838, 1989, 2088, 2061, 1929, 1650, 1409, 1140, 897, 670, 529, 342, 249, 216, 177, 157, 164, 129, 125, 160, 135, 111, 126, 139, 95, 108, 95, 77, 112, 99, 102, 111, 81, 97, 89, 65, 88, 76, 66, 89, 53, 76, 58, 55, 79, 70, 79, 80, 85, 86, 91, 72, 61, 77, 92, 73, 75, 98, 90, 91, 90, 87, 91, 55, 62, 77, 64, 54, 71, 89, 102, 59, 109, 74, 103, 111, 59, 97, 104, 87, 80, 59, 64, 85, 63, 43, 41, 72, 64, 66, 82, 74, 66, 80, 58, 72, 93, 95, 82, 74, 91, 78, 94, 78, 101, 125, 113, 117, 111, 152, 194, 245, 328, 488, 668, 851, 1039, 1316, 1411, 1565, 1542, 1472, 1317, 1143, 978, 810, 691, 472, 353, 269, 217, 162, 121, 130, 118, 112, 108, 105, 113, 122, 114, 94, 111, 129, 125, 107, 92, 89, 64, 80, 82, 74, 93, 60, 86, 66, 87, 94, 86, 111, 86, 97, 116, 104, 104, 72, 61, 80, 70, 82, 84, 95, 77, 82, 84, 81, 88, 123, 72, 60, 89, 60, 85, 77, 51, 41, 75, 54, 65, 58, 42, 66, 42, 88, 75, 53, 72, 58, 71, 85, 80, 72, 73, 62, 85, 82, 77, 82, 82, 81, 55, 76, 80, 84, 73, 75, 76, 80, 85, 67, 66, 82, 91, 80, 104, 79, 83, 61, 106, 85, 89, 96, 97, 84, 81, 60, 49, 64, 64, 71, 68, 81, 76, 83, 74, 109, 104, 102, 96, 118, 117, 86, 86, 73, 87, 59, 50, 55, 50, 56, 68, 46, 41, 54, 65, 61, 74, 47, 49, 76, 74, 56, 72, 75, 76, 86, 68, 92, 90, 72, 100, 99, 96, 83, 93, 108, 106, 114, 95, 95, 93, 77, 84, 95, 82, 76, 81, 90, 93, 119, 92, 96, 98, 131, 95, 119, 116, 107, 134, 142, 146, 163, 246, 369, 469, 690, 865, 1057, 1231, 1536, 1701, 1749, 1809, 1671, 1477, 1196, 988, 794, 586, 485, 339, 287, 207, 168, 160, 128, 120, 120, 104, 104, 112, 109, 98, 108, 100, 117, 87, 94, 113, 82, 65, 82, 55, 67, 89, 71, 80, 76, 54, 42, 38, 32, 38, 49, 42, 62, 44, 73, 61, 72, 61, 75, 75, 68, 73, 93, 72, 78, 58, 62, 49, 70, 104, 85, 71, 72, 70, 58, 67, 71, 72, 70, 88, 88, 80, 97, 86, 114, 117, 92, 122, 107, 76, 105, 113, 84, 77, 93, 88, 77, 74, 84, 102, 97, 96, 100, 100, 93, 95, 115, 81, 85, 76, 75, 86, 76, 81, 88, 71, 73, 92, 116, 126, 125, 121, 123, 106, 122, 145, 142, 124, 131, 137, 139, 164, 157, 142, 179, 201, 211, 287, 391, 498, 744, 830, 1105, 1336, 1565, 1755, 1821, 1876, 1736, 1516, 1271, 971, 779, 649, 455, 373, 265, 201, 183, 177, 174, 124, 127, 81, 118, 116, 97, 100, 98, 98, 95, 92, 87, 71, 88, 83, 82, 69, 83, 70, 78, 51, 42, 68, 81, 87, 76, 80, 92, 102, 82, 101, 65, 86, 98, 91, 62, 51, 77, 81, 81, 72, 52, 83, 90, 70, 50, 58, 61, 77, 49, 59, 70, 68, 59, 48, 59, 67, 84, 80, 78, 101, 102, 98, 122, 117, 130, 105, 100, 98, 110, 120, 150, 177, 260, 368, 481, 641, 857, 1083, 1356, 1530, 1559, 1644, 1595, 1482, 1324, 1160, 894, 720, 549, 363, 287, 221, 171, 152, 118, 126, 125, 113, 118, 99, 82, 73, 118, 105, 74, 85, 106, 78, 103, 91, 109, 83, 80, 79, 71, 79, 85, 83, 81, 78, 75, 61, 89, 65, 77, 52, 73, 55, 75, 93, 104, 92, 95, 110, 86, 100, 94, 100, 82, 77, 88, 90, 111, 88, 107, 82, 106, 77, 108, 77, 77, 75, 81, 70, 77, 59, 61, 73, 73, 54, 73, 67, 74, 64, 66, 70, 53, 65, 76, 70, 79, 77, 57, 45, 71, 61, 53, 74, 87, 70, 67, 61, 54, 57, 67, 63, 69, 61, 70, 78, 62, 67, 80, 81, 84, 73, 61, 67, 46, 70, 77, 87, 73, 72, 76, 60, 55, 57, 67, 45, 75, 67, 67, 73, 84, 101, 116, 99, 114, 132, 115, 128, 112, 124, 137, 114, 127, 130, 136, 199, 249, 384, 495, 635, 798, 1023, 1197, 1428, 1554, 1493, 1512, 1503, 1231, 1087, 860, 676, 540, 378, 285, 206, 187, 177, 152, 161, 126, 115, 114, 113, 116, 94, 68, 71, 65, 72, 81, 75, 65, 78, 85, 62, 61, 41, 55, 63, 73, 71, 86, 99, 73, 86, 72, 75, 92, 104, 107, 115, 91, 97, 96, 85, 109, 88, 58, 64, 58, 73, 40, 73, 40, 75, 57, 85, 74, 99, 76, 67, 67, 95, 67, 81, 57, 76, 69, 86, 77, 58, 41, 42, 70, 77, 66, 66, 64, 66, 52, 60, 61, 60, 46, 46, 51, 62, 51, 60, 55, 67, 58, 80, 88, 96, 78, 64, 87, 82, 89, 81, 76, 78, 78, 72, 59, 82, 63, 81, 99, 94, 69, 106, 92, 82, 99, 93, 95, 72, 83, 54, 84, 61, 56, 70, 61, 58, 57, 72, 70, 69, 92, 83, 70, 89, 93, 81, 93, 86, 85, 105, 102, 103, 70, 96, 106, 63, 76, 73, 95, 53, 90, 96, 96, 100, 105, 130, 102, 96, 102, 128, 115, 119, 137, 115, 111, 115, 113, 128, 140, 135, 136, 161, 168, 238, 256, 428, 567, 752, 989, 1157, 1358, 1689, 1754, 1895, 1786, 1676, 1546, 1307, 1137, 924, 776, 628, 440, 333, 268, 241, 201, 150, 145, 153, 103, 131, 105, 104, 96, 93, 109, 96, 110, 102, 103, 102, 108, 125, 108, 100, 81, 90, 85, 79, 99, 78, 69, 89, 84, 101, 97, 108, 105, 125, 113, 114, 95, 108, 96, 76, 58, 108, 85, 64, 50, 78, 76, 71, 81, 90, 108, 98, 102, 107, 113, 113, 125, 147, 119, 133, 118, 138, 115, 141, 167, 194, 264, 307, 469, 610, 776, 956, 1143, 1363, 1321, 1447, 1461, 1415, 1272, 1034, 892, 718, 569, 429, 366, 263, 229, 168, 161, 155, 120, 138, 98, 106, 102, 99, 103, 99, 116, 99, 88, 100, 103, 122, 112, 110, 107, 102, 91, 103, 95, 81, 81, 74, 71, 75, 75, 78, 81, 67, 79, 88, 85, 73, 76, 108, 101, 120, 107, 83, 92, 113, 106, 79, 99, 93, 90, 97, 92, 111, 104, 73, 86, 69, 77, 61, 79, 80, 48, 81, 51, 85, 72, 91, 83, 72, 67, 95, 93, 97, 90, 115, 125, 117, 92, 97, 102, 144, 181, 212, 278, 379, 526, 682, 893, 1151, 1432, 1679, 1760, 1918, 1837, 1794, 1628, 1417, 1253, 993, 702, 523, 434, 321, 287, 243, 197, 179, 137, 161, 152, 143, 167, 128, 138, 168, 138, 136, 145, 152, 138, 140, 135, 104, 100, 112, 83, 108, 123, 108, 96, 143, 116, 89, 102, 132, 88, 73, 112, 101, 86, 103, 78, 96, 110, 106, 91, 98, 141, 98, 124, 124, 98, 145, 144, 126, 122, 109, 137, 113, 130, 117, 141, 122, 129, 155, 140, 159, 155, 156, 164, 142, 193, 249, 264, 364, 530, 621, 924, 1046, 1195, 1521, 1722, 1824, 1871, 1757, 1593, 1380, 1204, 961, 770, 670, 654, 644, 734, 847, 1054, 1287, 1441, 1699, 1821, 1785, 1846, 1773, 1540, 1361, 1195, 911, 686, 531, 393, 316, 248, 232, 224, 180, 207, 168, 193, 175, 169, 162, 170, 148, 196, 200, 206, 288, 346, 521, 668, 910, 1230, 1331, 1607, 1822, 1875, 1801, 1799, 1729, 1505, 1232, 1021, 741, 625, 506, 355, 278, 243, 194, 193, 144, 130, 120, 106, 91, 102, 78, 95, 82, 110, 89, 63, 83, 96, 93, 102, 88, 98, 82, 90, 89, 97, 92, 87, 74, 74, 98, 64, 83, 66, 72, 66, 78, 50, 89, 66, 71, 85, 60, 115, 74, 103, 120, 94, 105, 90, 84, 108, 141, 129, 124, 126, 110, 122, 120, 141, 117, 125, 171, 189, 264, 367, 517, 702, 852, 1131, 1270, 1480, 1577, 1677, 1711, 1579, 1493, 1278, 1018, 816, 672, 488, 356, 301, 249, 185, 183, 159, 126, 136, 94, 98, 113, 108, 107, 108, 95, 57, 80, 68, 114, 83, 71, 93, 62, 101, 89, 117, 118, 122, 154, 162, 159, 160, 150, 142, 160, 183, 229, 291, 424, 567, 695, 957, 1197, 1341, 1586, 1705, 1671, 1728, 1613, 1486, 1324, 1222, 954, 762, 559, 396, 314, 249, 152, 164, 143, 123, 106, 106, 97, 95, 111, 91, 90, 77, 90, 65, 75, 100, 89, 76, 71, 76, 65, 64, 66, 74, 64, 50, 68, 82, 53, 59, 89, 69, 79, 60, 99, 98, 83, 81, 87, 76, 85, 100, 102, 107, 108, 73, 98, 84, 76, 57, 73, 104, 54, 77, 72, 83, 76, 104, 99, 73, 80, 48, 61, 62, 76, 95, 81, 51, 61, 50, 64, 84, 43, 71, 63, 107, 67, 72, 80, 49, 77, 64, 69, 50, 63, 96, 73, 93, 109, 88, 96, 100, 132, 143, 102, 114, 122, 101, 103, 124, 124, 119, 113, 92, 108, 119, 111, 96, 130, 138, 113, 143, 132, 152, 184, 150, 156, 192, 252, 259, 369, 484, 634, 822, 1098, 1209, 1532, 1775, 1778, 1833, 1836, 1741, 1547, 1373, 1110, 876, 743, 691, 612, 657, 745, 822, 1019, 1205, 1349, 1532, 1646, 1571, 1463, 1399, 1249, 1073, 856, 604, 522, 436, 327, 259, 192, 185, 145, 138, 124, 114, 120, 90, 104, 80, 88, 93, 81, 80, 81, 80, 86, 72, 99, 82, 82, 84, 76, 90, 79, 55, 57, 63, 76, 74, 62, 61, 66, 51, 60, 67, 64, 88, 83, 97, 97, 77, 111, 93, 93, 67, 72, 66, 81, 81, 81, 47, 62, 83, 88, 63, 88, 81, 85, 85, 99, 91, 99, 91, 98, 87, 104, 145, 162, 216, 293, 419, 601, 752, 924, 1189, 1284, 1443, 1605, 1579, 1584, 1493, 1245, 1080, 878, 672, 500, 419, 316, 248, 227, 181, 163, 134, 145, 141, 130, 106, 109, 99, 81, 86, 77, 96, 84, 100, 87, 122, 110, 101, 88, 103, 125, 127, 106, 101, 84, 86, 107, 85, 96, 126, 107, 124, 109, 71, 89, 74, 81, 85, 88, 55, 53, 57, 72, 40, 58, 80, 73, 58, 78, 72, 92, 78, 86, 76, 95, 59, 92, 109, 85, 110, 85, 81, 112, 107, 105, 103, 88, 88, 75, 107, 92, 94, 107, 110, 97, 105, 103, 109, 137, 141, 152, 149, 197, 243, 314, 414, 522, 716, 890, 1178, 1400, 1540, 1746, 1745, 1700, 1680, 1593, 1334, 1115, 918, 775, 606, 502, 408, 281, 224, 187, 164, 173, 138, 112, 123, 107, 101, 94, 115, 100, 90, 85, 63, 84, 101, 82, 93, 111, 98, 111, 105, 90, 109, 147, 130, 149, 178, 220, 329, 397, 542, 653, 873, 1100, 1303, 1308, 1381, 1430, 1451, 1322, 1227, 1004, 832, 650, 507, 426, 292, 229, 217, 176, 159, 162, 146, 129, 103, 116, 106, 84, 84, 83, 87, 100, 95, 79, 65, 72, 82, 89, 67, 75, 75, 84, 65, 81, 69, 51, 64, 76, 66, 82, 64, 74, 79, 51, 60, 84, 58, 44, 60, 55, 48, 37, 65, 67, 38, 48, 65, 59, 47, 52, 52, 70, 69, 81, 38, 55, 66, 25, 46, 60, 70, 59, 77, 58, 72, 77, 70, 67, 79, 89, 88, 97, 105, 102, 118, 94, 89, 92, 89, 89, 102, 105, 122, 74, 89, 85, 99, 115, 82, 113, 112, 127, 123, 73, 92, 90, 88, 87, 79, 51, 74, 76, 93, 92, 99, 89, 87, 110, 90, 114, 80, 75, 78, 114, 78, 63, 70, 50, 51, 58, 65, 56, 52, 75, 70, 85, 98, 96, 90, 97, 89, 111, 95, 117, 115, 127, 131, 210, 278, 344, 465, 630, 794, 1039, 1213, 1369, 1471, 1477, 1491, 1462, 1347, 1083, 973, 802, 660, 517, 385, 322, 239, 203, 188, 173, 167, 140, 122, 123, 123, 91, 103, 112, 91, 103, 84, 91, 88, 79, 91, 77, 67, 107, 101, 91, 55, 84, 62, 79, 74, 60, 88, 53, 64, 76, 79, 69, 76, 94, 88, 65, 76, 68, 74, 72, 89, 93, 60, 85, 76, 70, 96, 110, 113, 109, 113, 96, 99, 118, 102, 112, 93, 83, 65, 86, 83, 43, 65, 67, 62, 72, 82, 92, 98, 93, 104, 124, 114, 128, 134, 95, 120, 96, 120, 146, 114, 142, 164, 169, 197, 278, 311, 441, 567, 761, 1006, 1115, 1343, 1492, 1630, 1649, 1659, 1569, 1411, 1228, 1021, 845, 690, 526, 398, 322, 307, 240, 198, 142, 158, 102, 118, 118, 116, 100, 113, 90, 90, 95, 83, 90, 85, 87, 80, 99, 62, 68, 60, 64, 74, 69, 66, 79, 57, 51, 77, 88, 85, 99, 68, 82, 89, 65, 64, 86, 82, 71, 99, 76, 64, 93, 99, 71, 77, 86, 86, 74, 79, 94, 82, 66, 84, 81, 85, 83, 74, 91, 112, 96, 108, 122, 105, 120, 139, 128, 113, 122, 113, 118, 108, 110, 100, 96, 80, 82, 88, 81, 111, 96, 109, 101, 116, 123, 129, 120, 131, 121, 104, 139, 151, 119, 101, 90, 133, 127, 101, 123, 119, 108, 150, 147, 231, 309, 416, 566, 768, 913, 1099, 1274, 1475, 1508, 1533, 1393, 1395, 1292, 1063, 900, 694, 563, 451, 378, 280, 255, 218, 197, 146, 144, 124, 128, 118, 120, 106, 55, 70, 95, 89, 97, 103, 93, 73, 64, 87, 75, 92, 68, 62, 90, 93, 66, 100, 95, 106, 86, 94, 127, 100, 86, 115, 116, 119, 129, 154, 150, 132, 119, 139, 153, 141, 142, 170, 208, 194, 275, 314, 475, 550, 668, 832, 1014, 1210, 1409, 1545, 1608, 1620, 1514, 1435, 1245, 1058, 965, 810, 721, 674, 666, 735, 845, 1017, 1158, 1340, 1566, 1652, 1762, 1762, 1653, 1541, 1375, 1209, 1013, 766, 595, 470, 378, 293, 200, 179, 148, 169, 130, 152, 131, 111, 123, 97, 73, 114, 86, 72, 88, 96, 105, 95, 61, 81, 100, 123, 125, 104, 92, 116, 102, 121, 139, 146, 197, 270, 342, 414, 617, 800, 912, 1081, 1250, 1396, 1554, 1593, 1643, 1481, 1315, 1281, 1065, 904, 738, 555, 446, 357, 289, 250, 199, 147, 142, 140, 142, 109, 112, 116, 105, 82, 75, 63, 82, 77, 57, 95, 65, 85, 89, 78, 76, 65, 73, 79, 91, 75, 78, 71, 92, 106, 78, 104, 104, 100, 99, 103, 97, 90, 82, 95, 77, 86, 67, 84, 85, 73, 46, 76, 74, 70, 84, 62, 76, 70, 68, 73, 76, 79, 87, 78, 96, 82, 93, 90, 127, 142, 130, 121, 133, 113, 86, 126, 108, 100, 91, 77, 84, 76, 85, 88, 121, 122, 112, 126, 142, 134, 126, 129, 122, 107, 113, 116, 131, 105, 110, 130, 107, 125, 143, 138, 140, 140, 189, 207, 243, 255, 337, 474, 603, 733, 951, 1115, 1256, 1399, 1514, 1502, 1468, 1426, 1319, 1111, 890, 859, 747, 591, 613, 727, 691, 887, 1093, 1309, 1497, 1674, 1834, 1800, 1713, 1690, 1564, 1432, 1261, 1096, 852, 634, 575, 422, 330, 297, 229, 215, 159, 143, 158, 140, 117, 109, 110, 96, 118, 122, 100, 129, 96, 100, 100, 77, 83, 97, 95, 107, 98, 79, 104, 95, 82, 101, 89, 78, 72, 76, 77, 92, 79, 98, 80, 96, 85, 78, 94, 62, 76, 96, 121, 116, 100, 127, 141, 165, 161, 143, 154, 155, 126, 156, 170, 126, 110, 90, 99, 82, 94, 85, 91, 80, 62, 61, 96, 70, 76, 76, 79, 70, 88, 95, 102, 121, 120, 106, 137, 132, 123, 108, 117, 117, 110, 94, 75, 86, 72, 70, 89, 89, 100, 90, 123, 139, 130, 150, 135, 133, 141, 150, 195, 222, 321, 404, 566, 692, 837, 1020, 1223, 1521, 1695, 1772, 1814, 1832, 1692, 1593, 1380, 1136, 863, 727, 624, 475, 392, 298, 234, 199, 209, 187, 149, 153, 176, 163, 154, 116, 93, 99, 91, 92, 85, 80, 91, 92, 83, 63, 97, 111, 107, 100, 113, 107, 98, 102, 95, 85, 109, 124, 127, 130, 106, 104, 113, 101, 117, 84, 89, 75, 92, 79, 79, 86, 63, 79, 81, 88, 52, 60, 74, 74, 59, 58, 85, 73, 53, 53, 61, 47, 61, 60, 68, 69, 74, 71, 64, 74, 68, 72, 84, 59, 74, 59, 46, 50, 35, 53, 61, 52, 57, 49, 43, 61, 62, 56, 55, 79, 92, 62, 75, 105, 86, 93, 92, 81, 77, 88, 71, 81, 61, 73, 77, 71, 46, 66, 70, 55, 70, 52, 53, 61, 43, 63, 56, 77, 68, 80, 76, 80, 71, 70, 68, 79, 105, 90, 91, 104, 78, 96, 119, 114, 90, 126, 82, 136, 118, 120, 140, 101, 108, 128, 120, 111, 117, 118, 106, 112, 113, 111, 128, 122, 122, 120, 113, 124, 101, 77, 126, 102, 60, 72, 94, 98, 87, 86, 122, 99, 89, 101, 107, 118, 114, 126, 118, 131, 105, 108, 101, 102, 95, 97, 118, 123, 142, 152, 121, 139, 164, 153, 180, 190, 238, 306, 387, 472, 592, 756, 885, 1114, 1279, 1389, 1626, 1678, 1713, 1672, 1567, 1277, 1126, 999, 772, 633, 488, 401, 324, 229, 193, 179, 182, 169, 137, 172, 175, 139, 134, 150, 147, 156, 176, 222, 263, 347, 465, 546, 660, 868, 1077, 1193, 1379, 1402, 1406, 1456, 1362, 1241, 1196, 1007, 881, 693, 562, 443, 345, 266, 216, 168, 160, 148, 148, 123, 84, 107, 99, 88, 89, 85, 67, 103, 80, 64, 62, 56, 62, 67, 47, 65, 87, 61, 62, 72, 59, 91, 74, 49, 85, 80, 89, 79, 83, 79, 77, 63, 94, 64, 63, 63, 83, 75, 46, 80, 66, 83, 77, 56, 46, 81, 94, 56, 84, 97, 69, 83, 78, 80, 75, 81, 59, 68, 93, 76, 57, 61, 83, 75, 71, 78, 49, 65, 58, 63, 85, 64, 71, 46, 69, 69, 62, 56, 60, 66, 58, 80, 68, 65, 65, 81, 94, 58, 105, 95, 73, 103, 67, 105, 86, 102, 93, 93, 83, 79, 80, 52, 74, 85, 63, 56, 89, 98, 77, 95, 94, 90, 96, 80, 55, 76, 72, 94, 104, 117, 115, 142, 124, 118, 145, 152, 146, 164, 172, 195, 191, 245, 312, 382, 508, 652, 844, 977, 1184, 1451, 1569, 1677, 1674, 1756, 1688, 1473, 1433, 1224, 1031, 953, 854, 786, 772, 801, 894, 999, 1160, 1371, 1581, 1765, 1869, 1945, 1947, 1813, 1718, 1574, 1316, 1126, 858, 730, 555, 420, 389, 274, 223, 208, 179, 158, 177, 151, 135, 120, 114, 122, 139, 105, 89, 119, 82, 66, 108, 80, 100, 103, 112, 114, 127, 142, 104, 165, 164, 179, 165, 164, 176, 204, 193, 287, 353, 437, 545, 659, 787, 1027, 1285, 1363, 1394, 1610, 1556, 1661, 1498, 1456, 1251, 1064, 925, 812, 627, 482, 376, 324, 259, 205, 189, 180, 154, 139, 142, 129, 133, 122, 111, 97, 116, 125, 121, 132, 129, 118, 120, 139, 153, 144, 137, 148, 138, 139, 149, 169, 173, 217, 281, 347, 502, 585, 731, 937, 1085, 1167, 1369, 1463, 1501, 1491, 1350, 1298, 1101, 960, 804, 708, 552, 450, 370, 295, 255, 227, 219, 209, 194, 177, 206, 180, 155, 157, 190, 170, 196, 179, 240, 296, 390, 494, 586, 724, 944, 1185, 1386, 1577, 1692, 1724, 1726, 1714, 1637, 1523, 1310, 1153, 919, 812, 691, 538, 438, 331, 255, 221, 179, 166, 138, 145, 145, 115, 132, 121, 128, 102, 104, 94, 86, 76, 103, 87, 65, 98, 85, 78, 95, 101, 66, 59, 94, 100, 104, 69, 98, 97, 98, 88, 85, 92, 86, 77, 81, 76, 118, 96, 104, 109, 109, 143, 112, 159, 153, 152, 188, 217, 319, 409, 423, 512, 650, 892, 1052, 1178, 1285, 1412, 1465, 1551, 1496, 1454, 1265, 1115, 975, 819, 607, 481, 401, 330, 263, 268, 208, 184, 185, 215, 199, 192, 163, 173, 199, 200, 231, 253, 296, 339, 424, 548, 736, 922, 1127, 1303, 1642, 1660, 1898, 1848, 1833, 1779, 1733, 1505, 1426, 1217, 1032, 925, 818, 798, 798, 866, 1064, 1187, 1291, 1509, 1624, 1633, 1591, 1594, 1587, 1386, 1248, 1099, 888, 728, 562, 487, 375, 285, 251, 206, 195, 170, 162, 137, 138, 121, 134, 128, 95, 123, 106, 90, 116, 97, 101, 92, 112, 104, 131, 114, 111, 145, 126, 131, 113, 106, 100, 120, 108, 113, 124, 103, 124, 115, 119, 112, 115, 133, 170, 142, 139, 174, 189, 187, 175, 226, 248, 290, 359, 378, 525, 696, 797, 1051, 1322, 1519, 1661, 1855, 1823, 1741, 1836, 1705, 1628, 1422, 1264, 1042, 917, 879, 856, 875, 919, 1049, 1175, 1299, 1516, 1525, 1680, 1667, 1622, 1585, 1422, 1266, 1121, 983, 855, 760, 793, 876, 892, 981, 1221, 1271, 1506, 1491, 1663, 1609, 1589, 1571, 1385, 1283, 1100, 966, 773, 637, 509, 385, 313, 254, 269, 219, 197, 166, 162, 134, 156, 164, 114, 139, 140, 133, 123, 118, 112, 121, 110, 101, 119, 101, 88, 103, 129, 107, 132, 112, 119, 119, 117, 108, 126, 125, 105, 109, 102, 80, 82, 96, 101, 101, 85, 98, 93, 98, 101, 98, 104, 108, 137, 129, 117, 129, 143, 139, 156, 142, 146, 126, 146, 178, 168, 161, 155, 202, 195, 224, 283, 340, 379, 456, 600, 736, 902, 1157, 1319, 1483, 1670, 1793, 2022, 1944, 1915, 1761, 1588, 1490, 1269, 1105, 948, 877, 908, 856, 908, 990, 1180, 1340, 1438, 1469, 1520, 1607, 1560, 1473, 1360, 1161, 987, 841, 783, 602, 491, 374, 319, 227, 200, 194, 174, 163, 172, 129, 117, 111, 107, 94, 118, 98, 100, 93, 81, 113, 120, 96, 82, 93, 84, 91, 91, 103, 93, 111, 79, 60, 79, 77, 88, 62, 58, 85, 89, 68, 89, 75, 72, 62, 54, 78, 95, 92, 83, 97, 85, 83, 75, 87, 70, 62, 73, 90, 97, 60, 68, 79, 103, 102, 72, 104, 84, 104, 132, 105, 101, 113, 124, 107, 108, 114, 101, 76, 77, 74, 81, 95, 78, 112, 132, 85, 79, 104, 124, 146, 86, 63, 101, 90, 96, 151, 120, 130, 153, 120, 141, 108, 147, 150, 148, 169, 194, 221, 253, 282, 318, 460, 567, 759, 959, 1108, 1207, 1478, 1674, 1802, 1844, 1796, 1656, 1608, 1448, 1225, 1096, 983, 953, 877, 839, 913, 1029, 1118, 1203, 1269, 1388, 1501, 1550, 1517, 1452, 1312, 1147, 1066, 937, 731, 583, 503, 412, 336, 296, 257, 227, 188, 216, 183, 179, 204, 173, 185, 197, 227, 213, 283, 303, 342, 497, 586, 785, 845, 1056, 1256, 1360, 1359, 1493, 1469, 1492, 1372, 1334, 1197, 1005, 799, 704, 589, 480, 385, 291, 253, 225, 192, 190, 168, 125, 151, 143, 162, 135, 130, 129, 123, 113, 108, 95, 86, 110, 109, 96, 91, 76, 63, 83, 65, 109, 113, 80, 61, 59, 96, 94, 85, 103, 72, 78, 78, 65, 70, 81, 55, 77, 46, 90, 55, 76, 69, 73, 82, 70, 97, 105, 86, 118, 96, 112, 120, 155, 155, 173, 157, 165, 183, 199, 231, 215, 236, 306, 351, 425, 515, 678, 842, 981, 1192, 1413, 1576, 1722, 1805, 1822, 1770, 1679, 1520, 1350, 1154, 1026, 873, 722, 546, 487, 377, 317, 277, 205, 213, 227, 191, 183, 182, 149, 176, 183, 182, 218, 210, 249, 352, 427, 496, 654, 793, 956, 1106, 1217, 1356, 1474, 1523, 1566, 1490, 1333, 1220, 1052, 927, 777, 610, 496, 394, 339, 244, 196, 188, 174, 154, 154, 131, 132, 128, 120, 103, 113, 128, 113, 94, 115, 92, 86, 93, 75, 64, 82, 82, 75, 94, 84, 66, 88, 110, 92, 71, 85, 81, 84, 55, 73, 93, 91, 116, 122, 157, 138, 117, 150, 126, 160, 157, 148, 160, 210, 220, 302, 344, 447, 537, 680, 804, 1070, 1213, 1277, 1438, 1527, 1540, 1580, 1511, 1466, 1303, 1117, 935, 810, 712, 528, 434, 360, 281, 235, 198, 162, 164, 160, 137, 128, 136, 104, 100, 94, 111, 88, 87, 91, 102, 87, 96, 84, 92, 112, 103, 105, 115, 110, 120, 95, 138, 130, 177, 187, 227, 289, 370, 373, 438, 583, 731, 844, 1042, 1127, 1211, 1302, 1385, 1297, 1343, 1150, 1060, 860, 756, 633, 571, 484, 373, 328, 249, 187, 154, 144, 110, 137, 106, 97, 109, 113, 95, 99, 126, 135, 111, 145, 140, 123, 117, 132, 122, 117, 102, 85, 105, 102, 87, 84, 78, 78, 80, 62, 60, 59, 75, 63, 65, 60, 69, 64, 79, 77, 98, 54, 58, 55, 74, 65, 62, 48, 46, 37, 74, 46, 52, 57, 89, 49, 56, 64, 87, 84, 107, 85, 88, 83, 128, 117, 109, 136, 129, 130, 119, 120, 119, 103, 102, 124, 115, 104, 133, 132, 122, 103, 116, 117, 128, 75, 80, 93, 96, 96, 112, 100, 110, 85, 89, 131, 144, 145, 169, 173, 148, 200, 216, 312, 362, 465, 561, 658, 814, 1034, 1221, 1354, 1417, 1475, 1511, 1464, 1438, 1355, 1255, 1093, 947, 956, 898, 836, 883, 907, 1027, 1097, 1236, 1332, 1350, 1482, 1498, 1444, 1481, 1298, 1185, 1059, 906, 770, 633, 508, 412, 317, 252, 208, 180, 176, 120, 128, 142, 127, 113, 103, 116, 109, 102, 91, 120, 112, 73, 103, 103, 105, 100, 103, 119, 145, 136, 146, 156, 130, 154, 191, 164, 199, 184, 272, 319, 386, 536, 680, 770, 906, 955, 1275, 1344, 1434, 1407, 1462, 1406, 1276, 1240, 1124, 1016, 837, 714, 587, 514, 400, 301, 247, 214, 192, 155, 117, 104, 118, 94, 85, 87, 84, 80, 100, 87, 55, 67, 82, 59, 82, 78, 75, 74, 89, 53, 74, 75, 75, 64, 80, 67, 70, 96, 71, 83, 67, 73, 86, 85, 80, 93, 113, 86, 86, 88, 73, 72, 88, 57, 79, 74, 76, 62, 60, 62, 56, 57, 52, 46, 67, 61, 64, 68, 79, 66, 75, 64, 52, 45, 54, 81, 56, 64, 64, 54, 59, 70, 66, 48, 37, 55, 59, 58, 88, 62, 90, 88, 74, 73, 79, 58, 55, 76, 84, 71, 74, 78, 77, 90, 90, 91, 85, 79, 85, 96, 92, 98, 79, 84, 60, 64, 62, 79, 67, 62, 49, 41, 46, 70, 54, 45, 44, 49, 61, 46, 72, 61, 64, 38, 52, 65, 64, 77, 65, 50, 83, 87, 71, 48, 79, 92, 87, 70, 80, 55, 49, 63, 67, 75, 38, 69, 83, 67, 62, 76, 78, 73, 80, 85, 72, 81, 73, 79, 87, 92, 92, 109, 117, 111, 86, 98, 99, 108, 106, 122, 143, 166, 212, 247, 323, 375, 480, 550, 658, 792, 968, 1058, 1119, 1208, 1259, 1145, 1186, 1085, 997, 914, 809, 723, 561, 468, 387, 288, 211, 172, 146, 137, 126, 120, 119, 106, 82, 64, 81, 83, 56, 68, 68, 64, 85, 80, 75, 84, 78, 92, 109, 73, 80, 75, 100, 108, 80, 78, 87, 103, 102, 86, 103, 90, 101, 109, 107, 90, 105, 109, 102, 114, 111, 128, 121, 138, 103, 107, 91, 108, 121, 100, 101, 107, 123, 107, 108, 123, 107, 106, 119, 130, 132, 124, 153, 182, 212, 246, 299, 375, 510, 599, 696, 777, 915, 1026, 1151, 1242, 1246, 1227, 1229, 1164, 1086, 1022, 1014, 867, 773, 708, 683, 724, 806, 898, 1102, 1171, 1310, 1522, 1480, 1649, 1611, 1583, 1537, 1461, 1259, 1089, 902, 748, 597, 500, 395, 342, 294, 223, 199, 168, 173, 163, 167, 147, 136, 122, 136, 119, 79, 105, 104, 76, 111, 103, 83, 92, 105, 91, 73, 76, 98, 93, 80, 95, 69, 63, 79, 75, 71, 80, 79, 62, 77, 72, 70, 54, 74, 63, 72, 67, 69, 98, 72, 78, 46, 64, 48, 71, 62, 71, 85, 109, 90, 118, 140, 118, 100, 107, 159, 162, 157, 161, 186, 199, 226, 191, 232, 313, 460, 498, 592, 824, 1012, 1160, 1283, 1403, 1475, 1502, 1559, 1490, 1341, 1313, 1196, 1033, 889, 729, 619, 516, 378, 320, 288, 229, 191, 183, 180, 182, 166, 172, 155, 164, 189, 208, 214, 255, 305, 376, 430, 592, 736, 823, 936, 1097, 1124, 1276, 1245, 1239, 1305, 1228, 1270, 1106, 1047, 982, 860, 789, 758, 791, 851, 895, 981, 1123, 1140, 1299, 1440, 1350, 1428, 1501, 1374, 1251, 1161, 1043, 876, 727, 595, 473, 438, 337, 279, 244, 216, 188, 171, 160, 143, 128, 115, 132, 136, 155, 122, 125, 145, 144, 110, 96, 118, 113, 124, 123, 138, 110, 134, 137, 112, 111, 122, 119, 127, 125, 105, 111, 103, 81, 90, 94, 88, 84, 77, 105, 107, 89, 108, 104, 128, 100, 142, 109, 117, 119, 123, 96, 134, 128, 123, 127, 148, 117, 137, 94, 107, 106, 135, 163, 147, 157, 170, 214, 241, 335, 445, 484, 558, 730, 841, 972, 1098, 1193, 1229, 1337, 1436, 1283, 1201, 1200, 1155, 1018, 1006, 883, 842, 785, 863, 954, 1052, 1119, 1298, 1422, 1518, 1507, 1475, 1549, 1452, 1409, 1270, 1100, 944, 873, 677, 556, 469, 415, 344, 272, 236, 218, 173, 155, 159, 140, 113, 118, 102, 104, 102, 86, 89, 104, 108, 84, 95, 118, 124, 131, 88, 153, 161, 161, 143, 157, 196, 205, 224, 255, 260, 327, 415, 435, 585, 687, 891, 1023, 1140, 1296, 1355, 1486, 1451, 1440, 1346, 1346, 1206, 1093, 937, 768, 638, 529, 462, 372, 287, 236, 219, 190, 130, 122, 112, 94, 121, 113, 81, 113, 68, 78, 87, 92, 100, 96, 86, 86, 79, 66, 59, 56, 77, 76, 98, 101, 61, 78, 80, 93, 71, 84, 79, 61, 95, 87, 83, 78, 75, 73, 68, 51, 56, 64, 57, 87, 78, 66, 78, 77, 69, 75, 50, 80, 74, 101, 99, 96, 102, 89, 80, 93, 80, 96, 92, 80, 102, 102, 94, 77, 111, 95, 128, 141, 139, 120, 97, 151, 132, 132, 137, 149, 184, 221, 257, 302, 364, 468, 550, 629, 732, 829, 945, 1038, 1097, 1254, 1321, 1358, 1279, 1175, 1106, 989, 822, 668, 638, 513, 445, 348, 310, 250, 195, 169, 173, 142, 151, 143, 115, 123, 123, 104, 98, 97, 100, 73, 70, 92, 88, 91, 83, 74, 87, 69, 71, 103, 70, 60, 95, 85, 74, 92, 68, 77, 87, 60, 86, 88, 65, 81, 81, 64, 74, 62, 78, 74, 67, 86, 85, 91, 106, 103, 83, 95, 101, 106, 113, 107, 104, 126, 132, 146, 150, 153, 131, 160, 197, 198, 265, 280, 340, 410, 510, 574, 694, 784, 960, 1115, 1149, 1242, 1295, 1323, 1314, 1235, 1108, 962, 838, 720, 627, 571, 460, 348, 266, 229, 209, 141, 131, 137, 120, 100, 92, 93, 52, 78, 68, 91, 86, 80, 72, 103, 56, 58, 51, 35, 33, 55, 54, 73, 68, 69, 60, 71, 40, 52, 47, 62, 79, 52, 54, 68, 49, 59, 69, 69, 80, 73, 65, 92, 101, 103, 83, 97, 93, 71, 85, 86, 78, 90, 93, 108, 77, 74, 96, 91, 75, 91, 87, 83, 101, 69, 62, 68, 79, 72, 97, 84, 111, 87, 88, 108, 126, 132, 114, 102, 126, 146, 131, 181, 195, 237, 278, 370, 419, 497, 585, 692, 760, 908, 1022, 1084, 1181, 1232, 1168, 1093, 1098, 1000, 867, 800, 682, 560, 418, 359, 278, 253, 196, 147, 120, 138, 99, 104, 86, 94, 84, 70, 73, 72, 64, 50, 38, 56, 49, 53, 30, 52, 62, 43, 64, 54, 43, 36, 51, 38, 79, 61, 59, 55, 85, 60, 94, 84, 80, 111, 86, 80, 87, 79, 66, 86, 75, 92, 92, 80, 73, 90, 82, 72, 65, 68, 85, 85, 76, 61, 77, 70, 92, 90, 85, 76, 52, 106, 58, 78, 48, 73, 71, 64, 49, 64, 64, 53, 60, 82, 59, 62, 59, 58, 82, 72, 54, 85, 80, 82, 57, 66, 76, 62, 87, 81, 66, 80, 57, 50, 47, 56, 65, 74, 75, 64, 61, 80, 61, 93, 79, 61, 85, 99, 80, 70, 88, 73, 77, 87, 88, 68, 82, 79, 92, 66, 66, 53, 72, 84, 59, 64, 58, 76, 91, 88, 75, 77, 82, 62, 70, 92, 87, 82, 77, 94, 73, 112, 82, 75, 86, 106, 101, 105, 103, 119, 131, 144, 133, 175, 171, 158, 208, 265, 320, 335, 427, 544, 622, 669, 891, 1038, 1210, 1277, 1368, 1412, 1442, 1384, 1435, 1380, 1251, 1068, 914, 842, 679, 655, 522, 426, 402, 334, 304, 274, 249, 228, 221, 199, 187, 182, 169, 221, 229, 284, 309, 393, 428, 524, 605, 744, 801, 1064, 1168, 1235, 1224, 1345, 1389, 1312, 1259, 1213, 1143, 1052, 859, 716, 655, 551, 406, 376, 281, 242, 205, 176, 166, 179, 165, 163, 177, 166, 157, 180, 212, 231, 256, 348, 383, 508, 550, 627, 780, 899, 947, 987, 1079, 1126, 1131, 1085, 1083, 998, 878, 827, 692, 619, 552, 498, 398, 339, 284, 235, 240, 208, 213, 207, 206, 181, 193, 225, 244, 278, 329, 391, 455, 601, 645, 707, 796, 875, 1003, 1072, 1074, 1152, 1144, 1103, 1007, 948, 817, 744, 626, 481, 419, 387, 318, 264, 270, 217, 235, 175, 181, 195, 186, 205, 229, 261, 266, 287, 362, 460, 517, 589, 767, 885, 927, 1057, 1169, 1092, 1171, 1255, 1160, 1096, 934, 894, 795, 678, 506, 488, 434, 325, 282, 255, 191, 179, 164, 134, 147, 128, 115, 126, 129, 125, 102, 80, 111, 78, 102, 119, 120, 95, 94, 89, 111, 80, 102, 80, 82, 105, 92, 86, 84, 79, 87, 95, 99, 99, 107, 114, 96, 94, 112, 113, 132, 134, 129, 152, 147, 210, 182, 259, 332, 412, 489, 554, 645, 787, 941, 1050, 1204, 1350, 1423, 1494, 1434, 1579, 1535, 1475, 1316, 1174, 982, 842, 786, 637, 511, 439, 367, 314, 256, 240, 235, 233, 186, 182, 166, 199, 179, 196, 247, 229, 280, 362, 407, 486, 592, 695, 852, 1027, 1089, 1204, 1294, 1337, 1387, 1354, 1313, 1303, 1205, 1082, 962, 846, 713, 576, 513, 421, 350, 313, 275, 276, 231, 187, 185, 201, 185, 198, 217, 271, 333, 388, 463, 521, 593, 752, 841, 940, 1000, 1172, 1138, 1179, 1282, 1136, 978, 1061, 902, 826, 677, 602, 507, 410, 388, 303, 298, 273, 256, 223, 188, 176, 183, 195, 233, 207, 257, 284, 350, 419, 481, 534, 680, 770, 914, 1010, 1145, 1155, 1200, 1202, 1334, 1151, 1016, 992, 888, 853, 701, 563, 465, 418, 361, 292, 262, 204, 176, 161, 152, 134, 142, 132, 132, 113, 103, 96, 101, 85, 97, 102, 100, 90, 88, 82, 80, 88, 102, 101, 88, 73, 75, 83, 114, 112, 107, 104, 121, 102, 122, 135, 138, 119, 128, 138, 158, 139, 171, 140, 136, 140, 158, 180, 150, 170, 192, 203, 294, 357, 393, 471, 535, 577, 717, 801, 918, 944, 1119, 1100, 1143, 1180, 1069, 1021, 1079, 958, 882, 820, 766, 678, 742, 785, 861, 900, 961, 1018, 1167, 1204, 1112, 1126, 1116, 1088, 1111, 1012, 870, 788, 672, 549, 463, 397, 319, 238, 219, 213, 176, 150, 144, 98, 98, 91, 108, 98, 112, 110, 109, 94, 92, 88, 121, 70, 95, 110, 118, 146, 111, 126, 152, 201, 216, 236, 295, 342, 406, 535, 625, 703, 820, 1010, 1083, 1112, 1181, 1186, 1113, 1096, 1117, 1039, 943, 889, 761, 640, 569, 492, 410, 341, 272, 256, 209, 204, 188, 163, 144, 171, 147, 184, 191, 201, 223, 254, 294, 381, 434, 549, 623, 720, 811, 847, 962, 1008, 1035, 1199, 1138, 1041, 1064, 996, 853, 743, 668, 564, 529, 405, 346, 284, 251, 212, 172, 153, 149, 138, 131, 103, 120, 100, 103, 113, 100, 93, 80, 80, 69, 70, 68, 64, 68, 78, 82, 104, 69, 94, 86, 95, 84, 110, 103, 99, 123, 71, 79, 88, 96, 86, 91, 87, 73, 84, 88, 93, 99, 92, 104, 70, 74, 57, 48, 64, 98, 94, 56, 67, 78, 71, 78, 62, 73, 70, 73, 79, 66, 79, 93, 101, 86, 79, 72, 78, 73, 104, 109, 106, 129, 118, 144, 130, 159, 195, 151, 184, 207, 262, 287, 346, 435, 505, 631, 748, 880, 981, 1048, 1130, 1100, 1286, 1291, 1327, 1181, 1142, 1106, 1011, 917, 795, 724, 606, 537, 456, 339, 271, 237, 189, 154, 150, 127, 135, 103, 134, 105, 101, 105, 119, 114, 86, 91, 120, 113, 106, 110, 118, 106, 99, 104, 116, 115, 112, 127, 139, 141, 144, 182, 233, 273, 320, 381, 475, 551, 609, 754, 795, 833, 866, 877, 932, 940, 859, 813, 689, 664, 612, 520, 466, 395, 326, 293, 249, 207, 182, 144, 144, 149, 113, 132, 119, 97, 88, 104, 111, 106, 93, 80, 104, 99, 105, 96, 79, 111, 115, 97, 110, 111, 103, 99, 83, 89, 104, 95, 110, 129, 134, 129, 103, 166, 148, 153, 164, 149, 176, 161, 177, 213, 245, 254, 309, 394, 416, 508, 647, 730, 887, 929, 1069, 1149, 1282, 1215, 1273, 1267, 1276, 1211, 1132, 1023, 928, 756, 649, 518, 463, 463, 427, 319, 272, 269, 232, 223, 186, 187, 205, 190, 198, 194, 272, 290, 336, 421, 498, 587, 640, 809, 909, 1084, 1211, 1243, 1212, 1210, 1303, 1247, 1203, 1072, 1056, 933, 850, 711, 606, 496, 429, 343, 305, 250, 199, 186, 157, 146, 110, 97, 96, 74, 91, 103, 78, 59, 57, 65, 84, 64, 85, 95, 91, 77, 85, 93, 70, 86, 70, 82, 92, 83, 110, 81, 80, 81, 68, 80, 107, 81, 84, 77, 85, 61, 74, 68, 81, 79, 66, 80, 77, 41, 65, 73, 88, 70, 66, 73, 84, 82, 85, 83, 78, 58, 49, 49, 45, 54, 34, 61, 77, 57, 69, 68, 70, 59, 68, 47, 70, 43, 78, 76, 76, 66, 43, 68, 43, 65, 56, 57, 78, 85, 93, 82, 86, 90, 104, 117, 132, 127, 143, 146, 178, 202, 262, 301, 317, 346, 463, 548, 577, 655, 768, 927, 1007, 1227, 1207, 1250, 1202, 1232, 1140, 1166, 1047, 983, 908, 740, 626, 500, 436, 365, 326, 255, 221, 163, 146, 154, 141, 148, 119, 108, 135, 102, 98, 118, 112, 111, 115, 112, 110, 110, 115, 165, 145, 146, 159, 206, 211, 229, 243, 304, 320, 367, 431, 551, 651, 754, 895, 984, 1072, 1113, 1084, 1218, 1253, 1157, 1178, 1227, 1090, 1032, 946, 915, 876, 836, 888, 916, 860, 965, 996, 1023, 1083, 1099, 1149, 1154, 1181, 1116, 1047, 965, 914, 811, 735, 571, 532, 400, 371, 296, 241, 223, 175, 169, 156, 119, 104, 112, 107, 89, 68, 82, 76, 70, 72, 82, 79, 83, 90, 74, 61, 78, 81, 53, 52, 64, 58, 77, 78, 60, 72, 43, 68, 58, 75, 53, 82, 90, 73, 69, 101, 89, 77, 88, 83, 73, 67, 56, 65, 81, 71, 74, 77, 61, 34, 50, 40, 47, 58, 64, 62, 34, 31, 46, 34, 53, 22, 27, 40, 42, 74, 59, 82, 46, 65, 63, 46, 69, 55, 54, 56, 72, 64, 46, 72, 62, 64, 74, 48, 64, 74, 75, 79, 53, 35, 66, 30, 56, 51, 67, 33, 54, 68, 56, 48, 44, 38, 33, 53, 66, 48, 37, 34, 59, 49, 61, 59, 54, 51, 61, 63, 56, 54, 56, 51, 41, 54, 80, 65, 57, 54, 62, 45, 41, 61, 71, 83, 78, 59, 67, 56, 42, 54, 57, 54, 76, 64, 41, 59, 69, 53, 66, 65, 63, 54, 63, 68, 69, 53, 44, 94, 79, 48, 44, 67, 75, 76, 70, 60, 47, 50, 53, 49, 69, 90, 97, 63, 45, 69, 72, 86, 56, 65, 49, 82, 95, 94, 91, 69, 41, 64, 68, 76, 47, 60, 68, 65, 37, 69, 64, 58, 50, 64, 44, 45, 52, 66, 58, 72, 56, 70, 53, 89, 57, 67, 46, 50, 49, 38, 54, 46, 39, 30, 65, 63, 76, 64, 73, 36, 59, 72, 69, 62, 96, 74, 75, 55, 75, 82, 75, 82, 93, 75, 82, 101, 84, 111, 66, 104, 94, 116, 123, 75, 117, 85, 99, 92, 105, 107, 110, 108, 99, 136, 158, 152, 172, 179, 206, 273, 285, 345, 457, 552, 625, 721, 845, 899, 1007, 1043, 1151, 1178, 1232, 1183, 1149, 1119, 1059, 1056, 953, 916, 872, 867, 929, 968, 961, 1095, 1117, 1126, 1155, 1220, 1315, 1191, 1156, 1143, 1029, 963, 806, 731, 706, 595, 491, 403, 318, 304, 207, 204, 159, 175, 148, 119, 111, 105, 101, 103, 91, 89, 120, 110, 126, 100, 117, 114, 112, 140, 144, 126, 128, 144, 155, 162, 167, 188, 229, 278, 314, 366, 420, 517, 577, 716, 845, 849, 957, 1068, 1115, 1123, 1107, 1084, 1023, 972, 890, 834, 726, 623, 524, 502, 412, 330, 279, 235, 229, 189, 182, 158, 139, 104, 119, 101, 122, 91, 106, 84, 99, 116, 112, 94, 112, 88, 88, 98, 83, 76, 85, 96, 84, 69, 114, 91, 84, 109, 79, 85, 79, 90, 98, 99, 78, 109, 88, 113, 121, 108, 96, 95, 97, 85, 112, 84, 101, 84, 84, 90, 107, 79, 103, 109, 98, 104, 116, 111, 117, 146, 144, 158, 177, 190, 244, 263, 301, 333, 418, 578, 625, 720, 769, 774, 925, 1018, 960, 1029, 956, 1041, 1043, 966, 874, 770, 704, 585, 576, 499, 375, 314, 275, 223, 183, 179, 164, 132, 88, 105, 101, 92, 100, 65, 66, 86, 60, 72, 76, 69, 64, 78, 50, 91, 64, 44, 54, 78, 69, 77, 71, 77, 79, 107, 68, 73, 79, 66, 88, 83, 82, 113, 72, 98, 85, 71, 92, 102, 78, 91, 90, 94, 106, 108, 117, 84, 90, 91, 79, 97, 79, 75, 79, 81, 102, 72, 100, 95, 82, 80, 92, 75, 102, 106, 66, 73, 106, 67, 74, 66, 79, 91, 86, 96, 103, 100, 82, 80, 99, 106, 105, 95, 129, 128, 90, 132, 150, 159, 167, 165, 198, 219, 218, 252, 356, 358, 470, 534, 618, 693, 813, 894, 975, 1015, 1074, 1114, 1166, 1119, 1085, 1048, 1066, 948, 896, 916, 850, 920, 843, 894, 1000, 1012, 1001, 1119, 1182, 1156, 1148, 1172, 1158, 1121, 1065, 1028, 955, 822, 727, 581, 529, 449, 398, 337, 286, 188, 212, 196, 147, 125, 116, 116, 110, 91, 96, 70, 93, 96, 101, 102, 91, 118, 79, 120, 100, 99, 100, 98, 116, 105, 104, 105, 121, 112, 121, 117, 113, 105, 101, 110, 123, 133, 112, 113, 102, 87, 125, 139, 134, 150, 165, 202, 229, 250, 291, 323, 377, 466, 557, 661, 728, 727, 851, 952, 937, 928, 945, 952, 953, 884, 798, 674, 596, 507, 442, 442, 359, 317, 256, 203, 182, 141, 130, 103, 106, 112, 112, 99, 94, 84, 73, 62, 72, 60, 95, 90, 68, 70, 62, 57, 63, 56, 56, 61, 67, 91, 60, 73, 71, 61, 66, 90, 72, 64, 90, 78, 78, 64, 67, 58, 78, 60, 64, 81, 64, 80, 77, 78, 57, 64, 85, 68, 56, 74, 92, 72, 60, 73, 73, 61, 68, 81, 76, 82, 59, 63, 85, 96, 88, 110, 121, 100, 93, 103, 139, 124, 149, 146, 196, 223, 249, 301, 381, 398, 460, 534, 583, 588, 677, 810, 769, 801, 881, 964, 1009, 934, 917, 888, 817, 875, 807, 760, 781, 875, 915, 889, 937, 1043, 1120, 1131, 1045, 1076, 1075, 1101, 1021, 974, 850, 786, 745, 640, 533, 463, 436, 330, 317, 266, 257, 196, 144, 156, 144, 143, 125, 145, 140, 115, 126, 136, 112, 123, 112, 132, 120, 136, 136, 132, 150, 128, 150, 163, 156, 165, 194, 214, 236, 292, 345, 384, 493, 557, 657, 698, 747, 797, 872, 949, 1005, 1038, 1002, 995, 945, 902, 885, 879, 883, 822, 851, 967, 977, 985, 984, 1015, 1043, 1048, 1045, 1090, 1140, 1186, 1105, 913, 897, 817, 794, 726, 602, 483, 415, 419, 333, 272, 251, 205, 179, 153, 152, 129, 160, 118, 109, 118, 121, 106, 132, 119, 118, 118, 107, 117, 140, 136, 170, 173, 198, 186, 191, 231, 236, 278, 305, 348, 431, 539, 617, 659, 774, 827, 994, 1072, 1183, 1122, 1125, 1181, 1237, 1186, 1136, 1088, 994, 916, 821, 644, 650, 533, 446, 370, 307, 251, 256, 212, 213, 194, 221, 211, 237, 252, 279, 309, 352, 429, 476, 509, 612, 656, 793, 841, 891, 1000, 1027, 1038, 1088, 1080, 1101, 999, 967, 922, 1030, 954, 1015, 1017, 1007, 1001, 1138, 1136, 1145, 1257, 1301, 1269, 1276, 1209, 1246, 1155, 1011, 957, 875, 746, 685, 651, 505, 409, 374, 307, 225, 227, 198, 172, 154, 136, 100, 111, 101, 83, 111, 95, 105, 94, 125, 119, 108, 94, 106, 111, 107, 109, 96, 95, 76, 127, 92, 74, 85, 110, 84, 95, 113, 108, 94, 101, 83, 84, 80, 77, 75, 80, 81, 99, 75, 71, 74, 64, 84, 84, 59, 59, 54, 63, 61, 76, 66, 51, 70, 67, 83, 49, 52, 66, 81, 47, 54, 50, 70, 55, 46, 48, 33, 44, 44, 52, 65, 32, 56, 82, 47, 60, 56, 65, 61, 46, 67, 64, 43, 59, 59, 63, 38, 53, 40, 69, 65, 61, 57, 47, 40, 53, 57, 62, 49, 44, 43, 63, 62, 56, 56, 62, 64, 68, 69, 76, 69, 69, 61, 56, 41, 57, 97, 78, 64, 82, 62, 104, 114, 117, 107, 136, 139, 186, 186, 194, 222, 287, 320, 338, 398, 474, 491, 536, 652, 683, 716, 705, 741, 680, 682, 783, 733, 649, 549, 542, 463, 439, 358, 342, 326, 277, 241, 238, 232, 200, 225, 229, 241, 245, 303, 324, 364, 428, 439, 524, 619, 724, 744, 895, 952, 1044, 997, 948, 1026, 957, 852, 867, 814, 686, 693, 635, 537, 448, 361, 303, 282, 219, 180, 163, 166, 111, 103, 124, 119, 95, 108, 116, 91, 84, 110, 83, 121, 119, 108, 78, 54, 64, 78, 70, 75, 81, 81, 62, 72, 63, 69, 65, 48, 61, 69, 68, 60, 61, 72, 65, 64, 63, 52, 70, 81, 49, 62, 77, 71, 85, 89, 79, 60, 79, 66, 68, 95, 81, 106, 83, 64, 58, 95, 76, 60, 79, 82, 116, 65, 82, 69, 81, 81, 85, 79, 76, 58, 73, 63, 81, 75, 69, 72, 55, 55, 65, 63, 80, 77, 60, 64, 56, 54, 48, 44, 72, 64, 73, 53, 83, 46, 78, 67, 64, 59, 62, 89, 82, 74, 70, 51, 73, 49, 44, 73, 57, 46, 62, 75, 76, 50, 69, 53, 73, 59, 58, 70, 60, 77, 86, 39, 57, 74, 72, 60, 73, 101, 101, 62, 69, 81, 59, 88, 76, 85, 86, 90, 105, 101, 114, 99, 106, 139, 142, 169, 211, 273, 297, 331, 391, 464, 507, 567, 623, 689, 789, 823, 952, 960, 969, 972, 965, 941, 949, 902, 807, 730, 638, 578, 510, 424, 373, 293, 261, 209, 181, 157, 145, 133, 122, 82, 102, 90, 83, 71, 102, 87, 77, 60, 47, 63, 59, 84, 61, 66, 65, 77, 81, 80, 90, 77, 68, 69, 63, 73, 91, 75, 74, 58, 80, 57, 75, 86, 66, 62, 51, 53, 77, 65, 83, 76, 70, 80, 63, 67, 67, 72, 64, 80, 69, 60, 71, 68, 66, 65, 72, 66, 78, 77, 73, 61, 72, 90, 105, 75, 80, 90, 113, 103, 123, 142, 188, 157, 215, 245, 270, 324, 357, 386, 456, 505, 670, 710, 745, 838, 849, 852, 846, 953, 902, 870, 864, 812, 736, 655, 624, 576, 473, 391, 337, 327, 306, 239, 222, 192, 197, 246, 265, 282, 302, 360, 443, 491, 538, 580, 656, 766, 816, 805, 892, 888, 1047, 1071, 1025, 989, 930, 964, 923, 965, 839, 832, 945, 920, 906, 888, 968, 1014, 979, 1033, 977, 887, 949, 904, 959, 862, 745, 690, 587, 549, 470, 415, 385, 311, 278, 230, 172, 147, 149, 124, 153, 139, 107, 83, 80, 63, 82, 74, 80, 79, 68, 58, 92, 70, 62, 70, 54, 67, 83, 71, 57, 58, 60, 60, 48, 51, 69, 70, 84, 64, 61, 60, 70, 74, 51, 64, 74, 92, 70, 66, 79, 70, 59, 64, 51, 63, 53, 49, 48, 53, 81, 82, 88, 60, 70, 42, 86, 86, 95, 91, 89, 85, 75, 72, 75, 100, 106, 96, 103, 105, 112, 124, 129, 139, 176, 202, 211, 234, 311, 344, 382, 494, 529, 632, 761, 827, 896, 933, 1035, 950, 1035, 1029, 952, 1004, 965, 863, 827, 743, 667, 601, 511, 422, 394, 367, 309, 282, 226, 218, 166, 149, 131, 98, 109, 118, 94, 89, 96, 100, 77, 73, 120, 96, 116, 120, 135, 138, 156, 172, 211, 226, 252, 284, 377, 426, 468, 495, 624, 653, 723, 790, 837, 869, 855, 939, 913, 821, 858, 837, 759, 694, 614, 550, 489, 450, 374, 333, 275, 252, 242, 243, 197, 177, 118, 129, 121, 105, 104, 93, 100, 82, 86, 102, 94, 81, 85, 94, 87, 112, 114, 124, 159, 137, 174, 182, 206, 273, 320, 311, 382, 482, 539, 591, 667, 678, 777, 806, 808, 843, 970, 928, 896, 875, 763, 717, 736, 680, 614, 543, 460, 459, 359, 308, 247, 210, 199, 164, 143, 125, 136, 128, 111, 97, 116, 86, 94, 73, 90, 94, 75, 79, 51, 65, 64, 78, 83, 84, 81, 78, 72, 75, 69, 79, 74, 75, 81, 89, 85, 108, 102, 124, 119, 124, 132, 174, 213, 235, 265, 306, 359, 413, 497, 576, 610, 667, 736, 747, 810, 805, 839, 864, 806, 731, 782, 731, 687, 631, 627, 482, 449, 374, 361, 309, 291, 263, 256, 219, 237, 281, 269, 348, 309, 415, 473, 488, 525, 587, 705, 816, 856, 888, 961, 952, 991, 947, 970, 1055, 1031, 848, 740, 655, 670, 588, 536, 464, 376, 303, 268, 249, 213, 212, 178, 159, 124, 142, 119, 124, 147, 121, 125, 118, 112, 107, 103, 93, 129, 111, 102, 112, 152, 92, 119, 149, 193, 213, 200, 218, 268, 294, 376, 409, 463, 480, 550, 627, 703, 748, 762, 860, 824, 764, 739, 718, 712, 658, 637, 593, 492, 426, 352, 338, 286, 267, 257, 223, 218, 193, 223, 210, 204, 229, 256, 271, 304, 380, 455, 467, 501, 622, 704, 686, 780, 853, 927, 981, 990, 928, 1040, 1033, 949, 922, 853, 767, 725, 705, 564, 518, 477, 402, 349, 343, 278, 257, 261, 281, 246, 285, 321, 366, 399, 458, 476, 534, 585, 644, 686, 673, 713, 772, 785, 731, 758, 761, 746, 657, 582, 587, 474, 413, 372, 357, 305, 227, 236, 191, 177, 154, 133, 121, 82, 95, 77, 83, 76, 72, 88, 84, 80, 73, 64, 73, 81, 82, 63, 71, 56, 65, 62, 83, 65, 50, 57, 38, 77, 81, 65, 66, 60, 64, 54, 81, 66, 55, 52, 49, 76, 67, 83, 102, 102, 91, 87, 83, 88, 72, 77, 89, 99, 99, 112, 105, 140, 174, 176, 197, 237, 242, 271, 294, 400, 404, 492, 609, 590, 621, 624, 743, 820, 856, 848, 881, 892, 890, 858, 830, 731, 731, 666, 589, 523, 451, 420, 410, 347, 310, 325, 259, 248, 268, 294, 266, 288, 367, 370, 433, 547, 558, 569, 633, 702, 741, 826, 762, 832, 821, 761, 715, 691, 670, 669, 611, 524, 466, 410, 383, 344, 300, 272, 243, 240, 266, 272, 265, 262, 309, 343, 417, 472, 514, 545, 591, 619, 704, 749, 877, 881, 873, 866, 855, 894, 820, 786, 748, 712, 651, 594, 529, 465, 404, 367, 328, 310, 267, 188, 153, 134, 151, 136, 126, 100, 103, 95, 115, 87, 91, 104, 129, 108, 93, 109, 116, 132, 151, 168, 136, 182, 214, 249, 290, 321, 354, 416, 499, 574, 605, 653, 669, 757, 722, 716, 734, 722, 717, 683, 671, 558, 546, 511, 490, 432, 348, 330, 248, 254, 186, 177, 186, 167, 111, 98, 94, 89, 75, 67, 81, 96, 71, 75, 89, 72, 104, 86, 95, 100, 134, 151, 149, 159, 157, 202, 281, 318, 314, 356, 444, 430, 480, 634, 712, 787, 756, 832, 924, 941, 946, 873, 912, 886, 771, 721, 708, 623, 557, 465, 460, 414, 360, 307, 229, 191, 171, 156, 113, 104, 102, 92, 72, 80, 70, 64, 48, 46, 48, 40, 53, 46, 61, 70, 72, 55, 33, 57, 58, 51, 61, 37, 50, 79, 58, 56, 31, 40, 49, 54, 57, 56, 68, 56, 63, 60, 60, 33, 65, 76, 65, 73, 53, 66, 55, 58, 67, 60, 65, 46, 45, 64, 66, 48, 48, 52, 42, 71, 57, 62, 69, 47, 67, 84, 62, 60, 57, 62, 63, 62, 41, 62, 47, 57, 67, 73, 74, 71, 65, 71, 54, 82, 43, 45, 56, 59, 61, 40, 54, 36, 35, 65, 72, 63, 61, 62, 72, 54, 67, 52, 67, 21, 55, 44, 48, 69, 55, 50, 28, 25, 56, 45, 49, 54, 33, 37, 47, 51, 59, 37, 29, 56, 32, 52, 43, 37, 25, 32, 24, 40, 46, 69, 33, 28, 38, 63, 40, 68, 58, 37, 46, 50, 56, 76, 48, 39, 48, 46, 52, 46, 36, 41, 42, 57, 40, 46, 60, 54, 58, 53, 39, 36, 65, 56, 57, 53, 35, 40, 54, 54, 32, 43, 28, 28, 19, 23, 36, 37, 30, 40, 18, 50, 29, 44, 26, 21, 34, 33, 36, 40, 57, 39, 24, 13, 32, 17, 32, 58, 50, 19, 36, 31, 37, 45, 52, 28, 27, 36, 49, 39, 46, 55, 52, 58, 57, 69, 59, 72, 70, 59, 79, 81, 56, 72, 72, 41, 53, 78, 66, 56, 70, 74, 83, 76, 72, 68, 62, 68, 62, 44, 58, 53, 54, 41, 65, 71, 75, 96, 83, 96, 74, 73, 91, 62, 58, 108, 110, 121, 101, 148, 166, 186, 202, 214, 250, 278, 373, 453, 453, 493, 531, 630, 700, 729, 725, 792, 816, 715, 714, 813, 733, 688, 691, 623, 557, 512, 482, 384, 328, 268, 239, 223, 230, 191, 164, 152, 156, 103, 97, 77, 113, 96, 99, 95, 78, 50, 73, 80, 78, 77, 64, 66, 89, 72, 79, 75, 66, 48, 70, 80, 65, 48, 45, 67, 78, 55, 71, 58, 67, 58, 76, 33, 74, 54, 46, 45, 35, 50, 52, 60, 42, 42, 64, 46, 65, 38, 55, 41, 42, 80, 64, 41, 81, 95, 58, 72, 50, 49, 69, 65, 43, 53, 58, 44, 36, 59, 55, 50, 53, 46, 47, 46, 74, 41, 50, 43, 40, 41, 34, 65, 60, 79, 61, 44, 60, 64, 50, 77, 79, 85, 77, 79, 72, 70, 78, 100, 114, 123, 106, 142, 151, 152, 185, 214, 214, 255, 343, 378, 445, 439, 491, 519, 632, 682, 725, 796, 861, 804, 831, 860, 870, 835, 792, 679, 691, 672, 605, 534, 499, 450, 397, 338, 282, 270, 213, 194, 195, 175, 144, 126, 102, 118, 102, 106, 106, 111, 95, 118, 103, 126, 143, 157, 182, 181, 206, 217, 244, 287, 355, 423, 416, 475, 524, 613, 709, 792, 761, 774, 845, 880, 835, 829, 809, 833, 788, 731, 672, 602, 570, 496, 478, 459, 432, 373, 356, 262, 267, 322, 318, 323, 341, 315, 370, 485, 452, 490, 533, 524, 617, 675, 659, 664, 762, 705, 702, 717, 751, 740, 678, 568, 559, 514, 441, 430, 399, 354, 345, 298, 296, 301, 289, 279, 322, 321, 339, 393, 401, 481, 479, 529, 615, 695, 719, 708, 790, 828, 890, 830, 760, 752, 754, 743, 707, 622, 591, 536, 407, 365, 288, 284, 258, 265, 199, 173, 155, 136, 123, 108, 91, 102, 97, 69, 66, 44, 52, 83, 69, 49, 86, 69, 59, 72, 73, 60, 53, 44, 47, 54, 38, 77, 91, 37, 69, 68, 69, 84, 49, 65, 70, 95, 60, 58, 33, 45, 66, 68, 72, 60, 83, 69, 63, 76, 66, 76, 97, 102, 82, 151, 174, 173, 160, 188, 220, 299, 330, 357, 370, 411, 495, 612, 608, 691, 749, 735, 731, 890, 821, 793, 817, 752, 741, 721, 716, 588, 516, 495, 415, 380, 334, 284, 229, 232, 198, 184, 140, 161, 141, 143, 123, 90, 81, 86, 79, 99, 92, 99, 66, 76, 70, 59, 62, 68, 61, 52, 65, 35, 46, 68, 63, 39, 49, 20, 33, 60, 43, 40, 38, 29, 75, 44, 65, 93, 79, 77, 68, 57, 80, 126, 113, 126, 155, 163, 172, 182, 202, 232, 288, 321, 373, 412, 452, 504, 527, 671, 646, 661, 738, 825, 831, 745, 834, 824, 778, 736, 676, 635, 608, 575, 527, 479, 447, 377, 322, 311, 345, 296, 269, 282, 309, 312, 361, 410, 426, 465, 559, 553, 621, 646, 712, 708, 763, 761, 776, 729, 755, 709, 687, 691, 599, 566, 489, 500, 430, 384, 344, 302, 269, 214, 183, 147, 152, 130, 106, 91, 93, 91, 66, 69, 57, 75, 70, 74, 71, 77, 73, 83, 55, 86, 79, 69, 89, 67, 98, 75, 67, 81, 93, 111, 98, 103, 140, 99, 104, 130, 147, 183, 210, 211, 249, 270, 269, 318, 365, 433, 565, 599, 627, 614, 771, 823, 805, 895, 920, 1025, 985, 1026, 1022, 1066, 1045, 1003, 1027, 943, 967, 1015, 999, 1026, 950, 957, 939, 1016, 1022, 1067, 1042, 1020, 1036, 1045, 1038, 1037, 980, 870, 971, 1075, 1095, 1042, 1021, 1076, 1036, 957, 993, 893, 873, 814, 775, 706, 667, 585, 522, 431, 394, 346, 329, 282, 234, 205, 196, 172, 150, 136, 103, 71, 103, 81, 83, 93, 79, 78, 75, 68, 61, 69, 80, 89, 96, 88, 107, 51, 69, 80, 116, 109, 99, 67, 90, 69, 80, 55, 71, 60, 74, 65, 74, 47, 48, 52, 49, 53, 44, 64, 52, 46, 49, 46, 24, 43, 35, 47, 72, 66, 37, 43, 59, 47, 67, 55, 54, 49, 63, 58, 55, 52, 53, 44, 37, 46, 61, 48, 61, 37, 47, 36, 49, 46, 47, 41, 42, 57, 35, 34, 37, 42, 38, 55, 42, 46, 44, 41, 30, 54, 54, 67, 65, 40, 25, 45, 52, 35, 35, 66, 38, 47, 46, 29, 19, 40, 47, 27, 21, 30, 30, 30, 41, 52, 70, 62, 45, 24, 46, 34, 40, 50, 43, 33, 49, 41, 40, 42, 63, 35, 57, 35, 61, 60, 63, 72, 42, 83, 63, 75, 82, 66, 53, 75, 82, 114, 130, 128, 144, 179, 194, 213, 226, 275, 349, 354, 382, 402, 430, 527, 547, 603, 645, 698, 651, 643, 627, 610, 606, 564, 521, 488, 415, 403, 406, 354, 277, 265, 232, 189, 187, 171, 139, 127, 120, 92, 96, 79, 55, 72, 64, 52, 75, 40, 45, 66, 44, 51, 51, 91, 88, 72, 97, 83, 82, 99, 64, 69, 99, 94, 115, 128, 135, 149, 181, 183, 231, 273, 298, 335, 408, 456, 506, 538, 576, 659, 666, 694, 758, 830, 817, 905, 928, 850, 839, 798, 784, 769, 666, 624, 628, 561, 448, 473, 396, 330, 283, 234, 199, 233, 195, 135, 118, 79, 91, 117, 94, 76, 64, 79, 70, 67, 80, 74, 83, 69, 97, 73, 77, 50, 77, 77, 77, 84, 83, 96, 102, 111, 117, 116, 121, 120, 156, 186, 182, 164, 199, 264, 276, 318, 319, 351, 381, 415, 493, 560, 559, 623, 661, 719, 708, 681, 660, 657, 678, 645, 693, 590, 556, 504, 488, 439, 402, 368, 345, 275, 267, 309, 334, 339, 365, 349, 414, 427, 445, 414, 513, 550, 585, 600, 677, 692, 671, 645, 672, 650, 663, 649, 558, 534, 505, 435, 398, 400, 356, 309, 309, 295, 298, 280, 282, 259, 255, 283, 290, 306, 357, 418, 491, 509, 545, 588, 662, 599, 668, 699, 806, 784, 777, 704, 700, 703, 675, 643, 591, 553, 521, 450, 378, 326, 309, 253, 199, 240, 167, 166, 146, 146, 106, 85, 86, 67, 85, 73, 84, 84, 71, 68, 80, 57, 59, 64, 43, 53, 60, 61, 63, 58, 74, 55, 96, 65, 72, 61, 76, 44, 61, 73, 75, 57, 69, 77, 71, 42, 53, 82, 53, 65, 81, 65, 96, 81, 96, 91, 79, 93, 101, 125, 129, 143, 181, 202, 208, 256, 283, 341, 342, 393, 435, 431, 420, 453, 504, 451, 516, 475, 462, 415, 409, 363, 373, 358, 332, 315, 265, 252, 210, 193, 162, 130, 111, 127, 86, 76, 85, 78, 54, 44, 69, 84, 56, 69, 82, 58, 46, 54, 58, 72, 79, 66, 92, 76, 47, 63, 65, 70, 63, 98, 120, 96, 123, 139, 160, 182, 195, 189, 222, 297, 292, 321, 356, 444, 515, 565, 569, 566, 631, 692, 662, 670, 656, 626, 604, 620, 564, 575, 519, 483, 483, 440, 395, 341, 288, 237, 240, 201, 187, 163, 172, 158, 105, 133, 101, 100, 89, 98, 69, 80, 88, 42, 93, 78, 70, 64, 72, 69, 66, 67, 72, 69, 77, 72, 73, 79, 89, 81, 96, 110, 105, 145, 158, 200, 193, 208, 254, 273, 325, 356, 389, 442, 466, 550, 588, 562, 572, 578, 691, 694, 702, 692, 676, 776, 716, 659, 595, 549, 518, 456, 412, 414, 405, 357, 351, 310, 292, 255, 281, 283, 281, 300, 340, 390, 394, 413, 502, 533, 570, 614, 598, 616, 558, 671, 679, 670, 626, 646, 614, 566, 528, 435, 437, 448, 377, 350, 317, 280, 214, 212, 181, 189, 155, 152, 126, 128, 125, 105, 108, 129, 127, 146, 169, 166, 151, 173, 230, 251, 274, 345, 334, 402, 438, 454, 473, 555, 617, 648, 738, 728, 721, 721, 749, 770, 775, 761, 703, 606, 705, 644, 531, 528, 462, 408, 417, 353, 349, 250, 272, 223, 181, 144, 160, 133, 95, 115, 130, 107, 85, 106, 82, 85, 99, 88, 74, 82, 81, 93, 79, 73, 98, 89, 79, 119, 120, 149, 131, 128, 170, 120, 177, 188, 221, 225, 249, 280, 298, 387, 391, 419, 495, 545, 591, 588, 618, 713, 739, 755, 774, 762, 766, 748, 754, 729, 687, 682, 589, 539, 483, 464, 417, 390, 360, 345, 358, 380, 394, 415, 373, 410, 418, 503, 486, 496, 508, 498, 586, 604, 665, 698, 640, 657, 621, 673, 610, 637, 547, 550, 478, 448, 437, 412, 384, 338, 327, 357, 352, 328, 370, 396, 446, 501, 462, 542, 531, 560, 623, 651, 735, 712, 713, 767, 752, 663, 702, 701, 753, 675, 590, 564, 534, 489, 478, 440, 366, 373, 355, 332, 346, 325, 340, 320, 334, 375, 374, 386, 386, 444, 518, 547, 585, 618, 668, 688, 724, 774, 794, 804, 759, 698, 722, 717, 730, 628, 614, 548, 515, 524, 431, 393, 364, 333, 308, 299, 328, 326, 309, 380, 386, 443, 478, 496, 513, 561, 597, 672, 657, 791, 806, 780, 813, 747, 792, 764, 739, 768, 675, 700, 620, 582, 589, 534, 521, 496, 465, 454, 403, 424, 363, 367, 412, 417, 455, 535, 577, 579, 611, 589, 598, 674, 772, 700, 676, 700, 687, 658, 590, 635, 603, 570, 502, 415, 428, 380, 349, 291, 262, 220, 226, 200, 157, 139, 136, 140, 148, 127, 137, 146, 143, 149, 156, 173, 180, 187, 204, 240, 273, 324, 402, 410, 421, 418, 492, 561, 559, 638, 649, 676, 716, 774, 734, 795, 802, 905, 871, 874, 884, 802, 865, 934, 938, 913, 963, 915, 900, 913, 875, 918, 894, 806, 800, 757, 723, 711, 627, 590, 529, 527, 492, 470, 378, 345, 337, 339, 327, 361, 322, 318, 324, 379, 433, 410, 490, 546, 548, 607, 626, 672, 705, 721, 740, 801, 785, 857, 879, 902, 842, 898, 923, 838, 885, 941, 880, 936, 946, 899, 927, 959, 986, 967, 973, 981, 930, 932, 964, 999, 970, 974, 987, 977, 897, 906, 959, 884, 825, 835, 848, 809, 755, 722, 719, 682, 661, 616, 560, 558, 499, 488, 440, 421, 406, 414, 448, 452, 449, 475, 473, 547, 568, 625, 608, 592, 608, 669, 649, 712, 688, 677, 675, 630, 669, 566, 540, 513, 492, 440, 411, 382, 338, 315, 286, 322, 306, 308, 305, 339, 401, 386, 441, 438, 486, 483, 522, 659, 610, 641, 680, 684, 700, 677, 690, 667, 659, 641, 591, 503, 487, 491, 490, 423, 392, 299, 267, 245, 240, 194, 180, 137, 152, 146, 108, 123, 112, 89, 106, 76, 71, 85, 77, 78, 83, 79, 67, 70, 40, 65, 81, 61, 61, 59, 95, 77, 91, 76, 40, 52, 77, 81, 76, 109, 85, 77, 62, 63, 60, 54, 62, 70, 91, 73, 94, 81, 67, 78, 89, 64, 110, 70, 85, 94, 74, 77, 100, 74, 65, 78, 88, 83, 90, 80, 68, 83, 61, 62, 91, 84, 85, 90, 82, 90, 96, 100, 136, 132, 173, 145, 180, 230, 273, 257, 265, 337, 339, 414, 451, 456, 486, 503, 555, 591, 641, 601, 611, 642, 630, 630, 604, 652, 592, 585, 507, 476, 414, 385, 339, 318, 262, 245, 222, 243, 189, 171, 145, 140, 127, 109, 101, 100, 93, 109, 104, 91, 99, 75, 87, 76, 78, 70, 77, 85, 86, 76, 67, 72, 94, 100, 104, 91, 73, 64, 79, 89, 99, 87, 85, 75, 90, 89, 78, 80, 113, 123, 121, 115, 140, 156, 132, 146, 164, 174, 234, 262, 316, 336, 351, 398, 441, 459, 475, 543, 540, 609, 646, 629, 611, 591, 575, 656, 610, 553, 515, 496, 461, 456, 411, 395, 363, 334, 280, 265, 224, 236, 216, 194, 178, 172, 147, 148, 157, 163, 148, 133, 168, 179, 186, 227, 211, 263, 313, 363, 397, 443, 472, 558, 585, 586, 615, 606, 580, 640, 733, 718, 713, 716, 714, 701, 649, 573, 574, 538, 509, 483, 429, 377, 345, 293, 300, 279, 226, 186, 170, 163, 152, 159, 123, 141, 95, 108, 154, 115, 109, 119, 109, 146, 137, 122, 109, 105, 114, 107, 104, 104, 139, 113, 109, 119, 127, 147, 172, 187, 192, 193, 218, 245, 277, 324, 314, 384, 445, 458, 490, 522, 517, 560, 614, 588, 671, 631, 661, 620, 628, 653, 610, 588, 526, 582, 510, 473, 443, 426, 405, 338, 298, 330, 301, 295, 303, 333, 366, 407, 422, 423, 473, 504, 513, 537, 600, 584, 539, 635, 706, 714, 778, 782, 807, 792, 697, 753, 829, 772, 836, 795, 843, 813, 847, 807, 782, 759, 769, 716, 664, 669, 606, 547, 521, 510, 481, 423, 352, 315, 323, 288, 250, 261, 225, 167, 184, 185, 142, 139, 125, 138, 124, 122, 109, 119, 72, 88, 108, 83, 108, 109, 122, 126, 100, 118, 130, 126, 144, 134, 117, 97, 171, 195, 200, 204, 240, 267, 292, 334, 396, 390, 383, 377, 499, 526, 574, 594, 657, 658, 688, 649, 665, 628, 599, 691, 603, 600, 557, 574, 532, 438, 456, 432, 406, 390, 386, 368, 324, 341, 401, 419, 386, 405, 413, 490, 500, 554, 485, 604, 610, 637, 615, 582, 621, 613, 559, 590, 611, 548, 498, 480, 424, 440, 371, 330, 303, 329, 284, 259, 238, 222, 193, 201, 165, 173, 147, 151, 181, 201, 172, 186, 265, 240, 251, 248, 311, 309, 327, 370, 432, 470, 544, 509, 618, 673, 628, 700, 673, 653, 691, 673, 677, 695, 596, 573, 596, 591, 570, 507, 450, 419, 419, 425, 407, 376, 423, 383, 395, 397, 437, 457, 450, 482, 504, 494, 513, 561, 510, 620, 620, 599, 559, 496, 532, 484, 484, 474, 426, 404, 366, 327, 289, 292, 252, 274, 239, 170, 165, 186, 170, 162, 131, 127, 128, 147, 140, 153, 177, 222, 216, 248, 276, 289, 324, 352, 403, 485, 503, 497, 594, 646, 664, 703, 689, 709, 765, 806, 809, 816, 904, 831, 936, 911, 960, 860, 937, 947, 901, 933, 957, 963, 863, 871, 963, 944, 913, 960, 897, 917, 1007, 923, 970, 938, 894, 865, 853, 876, 815, 847, 788, 802, 721, 664, 703, 606, 558, 537, 458, 455, 432, 419, 373, 325, 270, 248, 206, 189, 164, 136, 137, 144, 138, 155, 132, 112, 92, 103, 84, 90, 117, 96, 88, 108, 108, 81, 108, 95, 85, 119, 111, 121, 128, 132, 162, 161, 180, 173, 212, 209, 270, 310, 319, 291, 383, 388, 438, 527, 554, 547, 597, 639, 651, 769, 794, 715, 771, 858, 897))
        self.assertEqual(record.annotations["abif_raw"]['DATA5'], (0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1430, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 1469, 0, 0, 0, 0, 0, 0, 0, 0, 0, 100, 150, 40, 0, 0, 0, 0, 0, 129, 129, 129, 189, 189, 189, 229, 229, 229, 270, 270, 270, 300, 300, 290, 329, 329, 329, 350, 350, 350, 379, 379, 390, 400, 400, 419, 419, 419, 440, 440, 440, 459, 459, 459, 480, 480, 480, 490, 490, 490, 509, 509, 509, 530, 530, 530, 550, 540, 540, 559, 559, 559, 580, 580, 580, 590, 590, 590, 609, 609, 609, 619, 619, 619, 640, 640, 640, 650, 650, 650, 659, 659, 669, 669, 680, 690, 690, 690, 700, 700, 700, 709, 709, 709, 719, 730, 730, 740, 740, 740, 750, 750, 750, 759, 759, 759, 769, 769, 769, 780, 780, 780, 800, 800, 800, 800, 800, 800, 819, 819, 819, 830, 830, 830, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839, 839))
        self.assertEqual(record.annotations["abif_raw"]['DATA6'], (0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 729, 784, 791, 794, 796, 797, 798, 798, 798, 799, 799, 800, 799, 800, 799, 799, 798, 798, 797, 797, 795, 795, 794, 793, 792, 792, 790, 789, 787, 786, 785, 784, 783, 781, 781, 780, 0, 0, 0, 0, 0, 0, 0, 0, 0, 53, 71, 16, 0, 0, 0, 0, 0, 64, 64, 64, 90, 90, 90, 110, 110, 110, 127, 128, 127, 142, 142, 142, 156, 156, 156, 169, 169, 169, 181, 181, 189, 191, 191, 201, 202, 202, 212, 212, 212, 221, 221, 221, 231, 231, 231, 238, 238, 238, 247, 247, 247, 255, 255, 255, 263, 263, 264, 271, 271, 271, 278, 279, 279, 286, 286, 286, 293, 293, 293, 300, 301, 300, 307, 307, 307, 314, 314, 314, 320, 320, 325, 327, 327, 332, 332, 332, 339, 339, 339, 345, 345, 344, 351, 350, 350, 357, 356, 357, 362, 362, 362, 367, 368, 367, 373, 373, 373, 378, 378, 377, 382, 382, 382, 388, 387, 387, 392, 392, 392, 397, 397, 397, 402, 402, 402, 402, 401, 401, 400, 400, 400, 400, 399, 399, 399, 398, 398, 397, 397, 396, 396, 396, 396, 395, 395, 394, 394, 394, 393, 393, 393, 392, 392, 392, 391, 390, 390, 390, 390, 389, 388, 388, 388, 387, 387, 387, 387, 386, 386, 385, 385, 385, 385, 384, 384, 384, 383, 383, 383, 383, 383, 383, 383, 383, 382, 383, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 381, 382, 382, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 380, 380, 380, 380, 379, 379, 379, 379, 379, 378, 378, 378, 378, 378, 378, 377, 377, 377, 377, 377, 377, 377, 376, 376, 376, 376, 376, 376, 375, 375, 375, 375, 375, 375, 375, 374, 374, 375, 374, 374, 374, 374, 374, 374, 374, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 372, 373, 372, 373, 373, 372, 373, 373, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 371, 372, 372, 371, 371, 371, 372, 372, 372, 372, 371, 371, 372, 371, 372, 372, 372, 371, 372, 371, 371, 372, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 370, 371, 371, 371, 371, 371, 371, 370, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 370, 371, 370, 371, 371, 371, 371, 370, 370, 370, 371, 371, 370, 371, 371, 371, 371, 370, 371, 371, 371, 370, 371, 371, 371, 370, 371, 371, 370, 370, 371, 371, 371, 371, 370, 370, 371, 371, 370, 371, 371, 371, 371, 370, 370, 370, 370, 371, 370, 370, 370, 370, 371, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 369, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 369, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 369, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 369, 370, 370, 370, 370, 370, 370, 369, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 371, 370, 371, 370, 370, 370, 371, 371, 370, 371, 371, 371, 370, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 372, 371, 372, 371, 371, 371, 371, 372, 371, 372, 371, 371, 371, 372, 371, 372, 372, 371, 372, 371, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 373, 372, 372, 372, 373, 372, 373, 373, 373, 372, 373, 373, 373, 372, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 374, 374, 374, 374, 373, 374, 374, 374, 374, 374, 374, 374, 374, 374, 374, 374, 374, 374, 374, 374, 374, 374, 375, 374, 374, 375, 374, 375, 374, 375, 374, 375, 375, 374, 375, 375, 375, 375, 375, 375, 375, 375, 375, 375, 375, 375, 375, 375, 375, 375, 376, 375, 376, 375, 375, 376, 376, 376, 376, 376, 376, 376, 376, 376, 376, 376, 376, 376, 376, 376, 376, 377, 376, 376, 377, 377, 376, 377, 377, 376, 377, 377, 377, 377, 377, 377, 377, 377, 377, 377, 377, 377, 377, 377, 377, 377, 377, 377, 377, 377, 378, 378, 377, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 379, 378, 378, 378, 379, 379, 378, 378, 379, 379, 379, 379, 379, 379, 379, 379, 379, 379, 379, 379, 379, 379, 379, 379, 379, 379, 379, 380, 379, 379, 380, 379, 380, 380, 379, 379, 380, 380, 380, 380, 380, 380, 380, 380, 380, 380, 380, 380, 380, 380, 380, 380, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 381, 382, 382, 381, 382, 381, 382, 381, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 383, 383, 382, 383, 382, 383, 383, 383, 383, 383, 383, 383, 383, 383, 383, 383, 384, 383, 383, 383, 383, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 385, 385, 385, 384, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 386, 386, 386, 385, 386, 386, 386, 386, 386, 386, 386, 386, 386, 386, 386, 386, 386, 386, 386, 387, 386, 386, 386, 387, 387, 386, 387, 387, 387, 387, 387, 387, 387, 387, 387, 387, 387, 387, 388, 387, 387, 388, 388, 388, 388, 388, 388, 388, 388, 388, 388, 388, 388, 388, 388))
        self.assertEqual(record.annotations["abif_raw"]['DATA7'], (25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25))
        self.assertEqual(record.annotations["abif_raw"]['DATA8'], (60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 60, 60, 60, 60, 59, 59, 59, 59, 60, 60, 59, 59, 59, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60))
        self.assertEqual(record.annotations["abif_raw"]['DCHT1'], 0)
        self.assertEqual(record.annotations["abif_raw"]['DSam1'], 1)
        self.assertEqual(record.annotations["abif_raw"]['DySN1'], 'Z-BigDyeV3')
        self.assertEqual(record.annotations["abif_raw"]['Dye#1'], 4)
        self.assertEqual(record.annotations["abif_raw"]['DyeN1'], 'Dye1')
        self.assertEqual(record.annotations["abif_raw"]['DyeN2'], 'Dye2')
        self.assertEqual(record.annotations["abif_raw"]['DyeN3'], 'Dye3')
        self.assertEqual(record.annotations["abif_raw"]['DyeN4'], 'Dye4')
        self.assertEqual(record.annotations["abif_raw"]['DyeW1'], 540)
        self.assertEqual(record.annotations["abif_raw"]['DyeW2'], 568)
        self.assertEqual(record.annotations["abif_raw"]['DyeW3'], 595)
        self.assertEqual(record.annotations["abif_raw"]['DyeW4'], 615)
        self.assertEqual(record.annotations["abif_raw"]['EPVt1'], 8500)
        self.assertEqual(record.annotations["abif_raw"]['EVNT1'], 'Run Started')
        self.assertEqual(record.annotations["abif_raw"]['EVNT2'], 'Run Stopped')
        self.assertEqual(record.annotations["abif_raw"]['EVNT3'], 'Collection Started')
        self.assertEqual(record.annotations["abif_raw"]['EVNT4'], 'Collection Stopped')
        self.assertEqual(record.annotations["abif_raw"]['FWO_1'], 'GATC')
        self.assertEqual(record.annotations["abif_raw"]['GTyp1'], 'POP7                            ')
        self.assertEqual(record.annotations["abif_raw"]['HCFG1'], 'CE')
        self.assertEqual(record.annotations["abif_raw"]['HCFG2'], '37XX')
        self.assertEqual(record.annotations["abif_raw"]['HCFG3'], '3730xl')
        self.assertEqual(record.annotations["abif_raw"]['HCFG4'], 'UnitID=3;CPUBoard=ECPU500;ArraySize=96;SerialNumber=0;')
        self.assertEqual(record.annotations["abif_raw"]['InSc1'], 10)
        self.assertEqual(record.annotations["abif_raw"]['InVt1'], 1500)
        self.assertEqual(record.annotations["abif_raw"]['LANE1'], 92)
        self.assertEqual(record.annotations["abif_raw"]['LAST1'], 'BCPFileName=KB.bcp;MobilityFileName=KB_3730_POP7_BDTv3.mob;StartPoint=0;BaseOneLocation=0;StopPoint=0;StopAtPCRStop=0;AverageSpacing=0;ProcessedDataProfile=0;StopAfterNumBases=0;MaxBasesToCall=800;CallMixedBases=0;SecondaryPeakThreshold=25;NCallQvThreshold=10;StopAfterNumNs=0;MaxNumNs=20;StopAfterNumNsInXBases=0;NumNsPerXBaseInterval=5;XBaseInterval=10;')
        self.assertEqual(record.annotations["abif_raw"]['LIMS1'], '60455a08eb8311e380820060088f971b')
        self.assertEqual(record.annotations["abif_raw"]['LNTD1'], 50)
        self.assertEqual(record.annotations["abif_raw"]['LsrP1'], 25000)
        self.assertEqual(record.annotations["abif_raw"]['MCHN1'], 'AB3730-0')
        self.assertEqual(record.annotations["abif_raw"]['MODF1'], 'LongSeq50_POP7_2')
        self.assertEqual(record.annotations["abif_raw"]['MODL1'], '3730')
        self.assertEqual(record.annotations["abif_raw"]['NAVG1'], 1)
        self.assertEqual(record.annotations["abif_raw"]['NLNE1'], 96)
        self.assertEqual(record.annotations["abif_raw"]['NOIS1'], (6.595200061798096, 8.70479965209961, 11.501799583435059, 8.527000427246094))
        self.assertEqual(record.annotations["abif_raw"]['P1AM1'], (223, 140, 3036, 477, 2031, 914, 913, 553, 1275, 871, 985, 709, 22, 955, 862, 1165, 1310, 848, 720, 750, 1062, 654, 1116, 1156, 793, 1451, 909, 923, 1021, 834, 925, 876, 900, 882, 1237, 1579, 996, 1043, 2290, 977, 1065, 871, 958, 1041, 942, 1087, 1009, 820, 1135, 656, 896, 737, 696, 1672, 975, 960, 2117, 861, 1186, 976, 1122, 1233, 835, 1174, 762, 891, 951, 834, 977, 1100, 1031, 1663, 904, 1014, 1088, 1155, 1052, 1228, 1221, 800, 1396, 1643, 1107, 1267, 1644, 1165, 881, 1166, 858, 889, 876, 957, 688, 809, 1831, 844, 1117, 642, 902, 1098, 851, 2599, 1066, 624, 1023, 908, 1062, 1024, 1014, 920, 891, 943, 911, 1136, 1021, 1002, 1105, 1258, 824, 822, 1065, 951, 939, 449, 963, 1074, 2028, 975, 886, 979, 1599, 1203, 859, 958, 946, 926, 919, 880, 1015, 1327, 1079, 873, 699, 936, 867, 897, 1043, 752, 858, 897, 1108, 1203, 794, 1004, 2141, 811, 925, 964, 960, 1060, 2076, 1078, 651, 815, 965, 948, 1226, 1969, 1187, 842, 855, 980, 764, 782, 870, 1137, 1038, 1120, 671, 1051, 784, 1590, 930, 1064, 806, 798, 663, 1099, 1278, 932, 951, 664, 777, 1123, 1264, 1136, 992, 1050, 882, 1099, 776, 939, 1216, 1126, 1551, 924, 1111, 1192, 550, 839, 978, 910, 1230, 626, 847, 1255, 1062, 1163, 1021, 1121, 1001, 824, 757, 764, 1204, 1187, 968, 549, 1092, 1795, 1589, 1665, 1424, 1257, 986, 597, 736, 844, 844, 782, 861, 1457, 2050, 978, 1216, 1016, 1145, 901, 566, 833, 660, 1353, 1074, 1250, 913, 417, 1386, 1641, 1479, 1479, 966, 1035, 1259, 1794, 764, 767, 690, 768, 724, 1671, 1079, 1044, 1032, 1204, 1160, 1326, 1000, 999, 975, 1059, 685, 588, 923, 1068, 958, 610, 928, 983, 510, 693, 1171, 711, 1013, 1167, 1246, 1052, 892, 853, 1261, 1983, 955, 956, 952, 1145, 2044, 926, 877, 1066, 716, 840, 872, 790, 1374, 656, 1366, 994, 993, 853, 947, 927, 897, 1458, 651, 1627, 1530, 1019, 570, 1839, 969, 897, 1994, 1034, 916, 764, 942, 1049, 964, 1052, 797, 631, 958, 661, 717, 1776, 766, 1128, 493, 896, 1074, 881, 1747, 709, 1632, 1626, 978, 1162, 738, 702, 1725, 834, 974, 847, 913, 1069, 2028, 976, 929, 1066, 516, 882, 737, 867, 1112, 941, 1092, 1020, 891, 1107, 1056, 1176, 917, 1146, 1016, 694, 650, 941, 1146, 1139, 623, 841, 2032, 981, 869, 1197, 1044, 719, 1157, 1499, 1183, 1049, 1010, 706, 1847, 974, 1072, 1300, 1030, 928, 2246, 901, 1125, 1162, 1111, 1231, 992, 737, 1019, 752, 1208, 1261, 1121, 1260, 953, 1048, 630, 1052, 984, 1093, 1120, 698, 945, 744, 525, 2396, 1164, 1276, 1048, 1054, 1022, 828, 906, 1035, 545, 732, 927, 1025, 919, 972, 1195, 1173, 1160, 1553, 868, 771, 1018, 1226, 438, 904, 1242, 855, 1926, 1295, 978, 1156, 776, 930, 1074, 953, 1970, 870, 1352, 982, 1165, 1097, 1074, 1148, 500, 835, 1541, 996, 1158, 1031, 743, 874, 1040, 1006, 1040, 815, 571, 1051, 1683, 1481, 959, 628, 952, 1531, 1161, 1027, 1141, 1759, 1358, 1021, 1172, 978, 1090, 575, 870, 855, 884, 961, 959, 629, 947, 975, 1273, 905, 1101, 689, 923, 803, 994, 1087, 1273, 1541, 880, 900, 811, 850, 929, 716, 884, 838, 1042, 1306, 1099, 919, 1028, 559, 1368, 736, 805, 1128, 1294, 1047, 883, 1043, 622, 1242, 898, 1133, 2187, 961, 1072, 751, 1068, 973, 926, 748, 1241, 1032, 1035, 1016, 1509, 1610, 977, 1123, 1045, 1052, 1178, 901, 947, 908, 1013, 1042, 788, 1051, 1028, 661, 709, 911, 1419, 940, 834, 1127, 1181, 966, 1088, 1018, 979, 1041, 551, 1854, 467, 922, 773, 863, 1083, 781, 1628, 779, 982, 1033, 1658, 859, 987, 832, 809, 1205, 574, 912, 1200, 980, 1199, 1169, 1058, 994, 1161, 631, 1296, 726, 941, 1053, 1127, 1974, 777, 1005, 920, 1017, 1568, 822, 1008, 1081, 525, 1352, 566, 1817, 829, 962, 949, 952, 933, 837, 907, 2024, 996, 1109, 1292, 767, 495, 1142, 1330, 1401, 1144, 728, 467, 1872, 1017, 774, 1290, 1102, 2132, 1018, 1212, 1260, 926, 950, 913, 1413, 812, 921, 1141, 945, 992, 928, 1130, 656, 742, 953, 637, 1221, 1233, 586, 1058, 1061, 807, 1637, 1041, 474, 1771, 1421, 538, 1594, 1104, 472, 2138, 840, 1190, 1016, 687, 599, 1646, 862, 1384, 1948, 603, 1087, 876, 1264, 1052, 724, 855, 893, 1038, 1121, 1280, 1005, 991, 1082, 560, 1761, 952, 1150, 1041, 705, 967, 987, 1210, 1114, 1073, 785, 1121, 862, 1135, 2076, 689, 1319, 1702, 875, 1242, 526, 1581, 844, 989, 1097, 1029, 1150, 741, 971, 1746, 875, 721, 831, 808, 963, 783, 792, 1835, 967, 605, 999, 878, 1892, 933, 1044, 1281, 775, 997, 446, 887, 1003, 1008, 1110, 829, 909, 1299, 1131, 749, 906, 928, 1012, 619, 1091, 1060, 1129, 850, 1009, 717, 1001, 910, 1231, 930, 914, 1162, 1074, 1006, 952, 678, 1055, 858, 1180, 1182, 1458, 1117, 980, 1020, 915, 667, 1355, 1242, 951, 845, 1217, 1118, 468, 1191, 1001, 966, 1259, 851, 1134, 853, 1020, 1206, 1066, 868, 957, 946, 1207, 1107, 900, 888, 868, 1474, 1060, 1348, 1325, 813, 1167, 978))
        self.assertEqual(record.annotations["abif_raw"]['P1RL1'], (69, 43, 71, 78, 47, 36, 34, 6, 89, 13, 35, 28, 11, 75, 75, 28, 35, 66, 22, 65, 29, 19, 47, 37, 62, 7, 60, 63, 74, 66, 78, 72, 36, 8, 13, 62, 87, 82, 69, 34, 63, 8, 88, 12, 15, 46, 48, 50, 73, 89, 48, 87, 49, 21, 13, 26, 67, 30, 55, 77, 2, 52, 63, 44, 68, 53, 52, 70, 42, 9, 29, 51, 41, 42, 94, 64, 36, 47, 51, 75, 84, 9, 67, 8, 71, 80, 14, 46, 46, 17, 31, 85, 21, 17, 68, 36, 49, 34, 93, 40, 22, 4, 11, 6, 27, 2, 16, 73, 73, 38, 97, 39, 72, 44, 10, 96, 1, 55, 41, 57, 70, 54, 50, 43, 64, 64, 12, 64, 63, 14, 58, 61, 5, 4, 49, 55, 57, 38, 13, 43, 36, 50, 86, 73, 90, 92, 11, 91, 28, 35, 95, 64, 80, 65, 71, 77, 19, 89, 69, 29, 76, 3, 44, 56, 62, 22, 74, 83, 25, 59, 59, 67, 58, 12, 84, 54, 42, 3, 31, 55, 53, 60, 45, 29, 27, 75, 90, 16, 85, 53, 5, 14, 29, 99, 64, 52, 61, 73, 35, 98, 90, 31, 36, 60, 65, 77, 63, 25, 9, 56, 88, 77, 14, 31, 4, 98, 21, 34, 53, 49, 27, 67, 32, 51, 11, 30, 17, 85, 13, 38, 70, 86, 53, 36, 88, 69, 89, 35, 13, 74, 58, 90, 5, 61, 43, 7, 54, 16, 28, 12, 53, 99, 31, 93, 73, 4, 1, 82, 3, 9, 72, 23, 92, 35, 53, 57, 44, 39, 91, 71, 78, 17, 59, 89, 21, 62, 99, 100, 45, 81, 64, 99, 69, 6, 22, 89, 23, 17, 14, 85, 22, 78, 34, 43, 61, 46, 38, 6, 75, 53, 4, 64, 83, 80, 5, 32, 73, 73, 43, 69, 37, 33, 37, 99, 28, 79, 22, 72, 49, 97, 55, 27, 12, 81, 54, 80, 36, 85, 77, 32, 12, 2, 52, 27, 26, 71, 80, 82, 85, 49, 57, 72, 94, 43, 7, 72, 56, 46, 65, 53, 17, 32, 52, 90, 20, 25, 85, 72, 80, 86, 44, 75, 28, 39, 76, 25, 49, 66, 41, 59, 20, 30, 26, 69, 35, 43, 19, 58, 8, 59, 5, 45, 59, 40, 95, 43, 87, 92, 13, 1, 9, 77, 48, 43, 61, 53, 32, 99, 69, 68, 61, 68, 24, 79, 17, 88, 86, 17, 18, 53, 70, 78, 48, 57, 49, 68, 10, 30, 86, 50, 48, 7, 48, 40, 76, 31, 24, 29, 59, 63, 36, 91, 60, 6, 95, 28, 49, 86, 96, 54, 69, 33, 50, 87, 56, 97, 22, 57, 50, 78, 25, 71, 2, 61, 14, 48, 93, 51, 19, 40, 37, 72, 54, 58, 7, 59, 30, 60, 13, 95, 31, 45, 63, 84, 23, 70, 98, 42, 20, 70, 78, 13, 3, 29, 3, 19, 49, 57, 68, 45, 63, 63, 3, 43, 53, 25, 92, 26, 96, 95, 67, 48, 76, 22, 22, 34, 60, 31, 55, 39, 23, 37, 43, 69, 27, 77, 79, 92, 40, 70, 53, 28, 78, 65, 44, 86, 33, 13, 66, 69, 43, 61, 36, 16, 43, 67, 26, 84, 68, 93, 94, 87, 46, 11, 49, 21, 97, 30, 50, 16, 74, 76, 94, 40, 81, 69, 81, 62, 30, 7, 20, 41, 85, 62, 98, 54, 66, 92, 40, 68, 30, 62, 90, 51, 28, 82, 41, 33, 63, 46, 30, 17, 89, 57, 30, 84, 1, 39, 48, 61, 81, 18, 4, 45, 29, 35, 43, 60, 71, 20, 31, 56, 61, 60, 52, 55, 10, 91, 78, 56, 20, 11, 76, 91, 59, 97, 60, 23, 52, 94, 46, 62, 36, 25, 45, 50, 38, 50, 100, 40, 60, 49, 58, 2, 16, 89, 23, 36, 93, 83, 20, 36, 60, 64, 55, 19, 44, 64, 1, 81, 56, 18, 2, 74, 64, 28, 3, 60, 21, 48, 62, 46, 5, 74, 42, 16, 85, 87, 17, 11, 79, 46, 69, 29, 94, 52, 70, 33, 94, 14, 42, 86, 43, 18, 38, 94, 6, 18, 82, 70, 49, 77, 78, 27, 69, 18, 69, 73, 28, 32, 29, 71, 94, 15, 53, 41, 55, 54, 71, 61, 30, 98, 16, 12, 93, 38, 33, 39, 82, 30, 54, 66, 17, 12, 21, 35, 24, 12, 77, 50, 23, 88, 12, 72, 14, 65, 44, 16, 21, 36, 0, 14, 28, 88, 12, 5, 88, 12, 30, 13, 16, 62, 0, 37, 64, 14, 30, 70, 38, 0, 71, 57, 55, 59, 33, 61, 85, 49, 66, 89, 48, 59, 50, 88, 12, 77, 45, 67, 11, 52, 76, 1, 66, 59, 48, 59, 21, 79, 77, 23, 50, 77, 0, 23, 50, 65, 37, 88, 50, 88, 12, 64, 0, 71, 99, 97, 64, 70, 20, 12, 79, 32, 66, 91, 11, 2, 34, 50, 77, 23, 84, 88, 12, 76, 0, 79, 95, 55, 69, 50, 48, 33, 0, 77, 0, 23, 71, 7, 43))
        self.assertEqual(record.annotations["abif_raw"]['P1WD1'], (1591, 1871, 1360, 1044, 3800, 4000, 4200, 1215, 1218, 968, 1054, 1076, 2600, 1286, 713, 716, 816, 592, 652, 586, 559, 543, 512, 468, 481, 526, 484, 500, 477, 457, 427, 428, 462, 447, 442, 370, 447, 408, 373, 413, 404, 361, 373, 345, 379, 376, 391, 388, 370, 331, 345, 341, 350, 290, 326, 329, 291, 324, 324, 296, 307, 313, 311, 307, 316, 307, 310, 284, 279, 281, 286, 282, 284, 273, 281, 294, 268, 283, 289, 301, 310, 300, 311, 314, 287, 303, 279, 256, 282, 303, 274, 262, 268, 274, 293, 264, 253, 285, 283, 300, 260, 256, 256, 270, 279, 261, 255, 247, 240, 251, 254, 256, 232, 247, 239, 249, 264, 251, 253, 261, 243, 243, 258, 261, 268, 261, 263, 248, 247, 249, 258, 261, 245, 261, 248, 258, 248, 233, 243, 255, 247, 242, 258, 247, 250, 237, 234, 244, 250, 260, 243, 249, 240, 234, 253, 236, 235, 253, 233, 233, 247, 232, 240, 242, 243, 225, 232, 242, 248, 257, 247, 247, 247, 239, 244, 237, 234, 241, 258, 242, 261, 244, 236, 240, 239, 258, 250, 252, 238, 246, 244, 229, 235, 230, 217, 237, 239, 240, 212, 239, 242, 204, 208, 244, 238, 220, 221, 237, 242, 236, 239, 222, 243, 241, 239, 206, 197, 202, 202, 202, 211, 240, 224, 230, 202, 213, 233, 241, 237, 238, 153, 1200, 1200, 1200, 229, 235, 214, 225, 227, 222, 245, 1400, 241, 1200, 252, 249, 168, 250, 244, 238, 236, 1400, 244, 1400, 247, 251, 228, 1200, 1400, 1400, 247, 1200, 237, 232, 229, 247, 251, 245, 218, 244, 247, 244, 231, 218, 226, 241, 222, 237, 244, 247, 231, 241, 246, 245, 248, 245, 236, 262, 244, 247, 242, 222, 243, 232, 232, 239, 235, 240, 242, 247, 230, 244, 248, 237, 236, 241, 229, 251, 232, 243, 255, 258, 243, 257, 253, 240, 237, 229, 243, 249, 242, 256, 254, 252, 239, 251, 259, 250, 237, 240, 243, 268, 253, 264, 245, 237, 234, 244, 229, 255, 264, 253, 250, 261, 252, 252, 257, 244, 262, 241, 256, 246, 262, 253, 254, 250, 273, 255, 250, 252, 248, 240, 247, 259, 271, 237, 238, 252, 268, 274, 252, 235, 261, 267, 263, 254, 259, 262, 251, 253, 256, 247, 273, 257, 271, 259, 261, 274, 255, 265, 263, 266, 272, 263, 262, 261, 243, 259, 263, 241, 270, 267, 266, 253, 269, 259, 264, 269, 279, 250, 242, 248, 257, 253, 264, 277, 287, 284, 265, 253, 289, 273, 264, 270, 284, 284, 273, 285, 279, 282, 267, 270, 263, 273, 258, 252, 271, 273, 284, 283, 271, 292, 279, 303, 302, 293, 265, 277, 276, 268, 257, 280, 299, 288, 280, 303, 300, 284, 277, 277, 278, 298, 289, 295, 297, 302, 286, 293, 310, 298, 298, 288, 289, 268, 283, 296, 306, 304, 296, 270, 267, 282, 301, 297, 308, 293, 286, 297, 313, 290, 304, 309, 298, 315, 299, 277, 295, 284, 290, 273, 301, 305, 309, 309, 312, 307, 316, 299, 305, 303, 305, 290, 285, 295, 307, 330, 319, 319, 309, 315, 307, 295, 316, 280, 303, 321, 351, 324, 332, 290, 314, 309, 320, 320, 320, 320, 345, 318, 328, 322, 313, 313, 307, 331, 328, 336, 328, 348, 313, 342, 317, 316, 327, 346, 320, 356, 345, 333, 333, 333, 314, 339, 325, 323, 341, 338, 335, 329, 338, 358, 337, 386, 343, 350, 374, 332, 334, 332, 393, 367, 336, 370, 328, 354, 351, 351, 335, 340, 355, 371, 340, 357, 347, 337, 401, 426, 357, 372, 376, 378, 381, 346, 359, 354, 354, 362, 371, 375, 363, 394, 398, 402, 404, 370, 372, 346, 355, 466, 403, 375, 358, 359, 378, 362, 421, 437, 493, 409, 352, 445, 425, 378, 394, 344, 407, 378, 403, 378, 342, 361, 361, 361, 383, 364, 446, 433, 365, 383, 436, 357, 389, 398, 368, 395, 424, 387, 368, 365, 345, 407, 411, 343, 394, 355, 384, 359, 507, 472, 482, 394, 360, 362, 352, 394, 373, 393, 393, 390, 416, 408, 375, 412, 404, 404, 422, 394, 394, 418, 418, 403, 428, 394, 394, 598, 476, 422, 384, 445, 442, 563, 480, 419, 407, 427, 452, 442, 418, 410, 432, 423, 419, 404, 422, 435, 435, 435, 435, 443, 442, 391, 443, 359, 437, 449, 452, 488, 488, 488, 477, 477, 477, 488, 488, 430, 454, 398, 389, 488, 380, 438, 421, 439, 397, 488, 488, 478, 488, 488, 431, 451, 433, 426, 430, 468, 434, 480, 430, 430, 485, 421, 474, 430, 493, 473, 486, 470, 471, 537, 475, 421, 492, 507, 421, 488, 488, 469, 421, 450, 449, 482, 430, 458, 501, 485, 421, 490, 421, 421, 477, 477, 430, 477, 477, 477, 421, 475, 430, 496, 421, 488, 488, 430, 421, 548, 524, 430, 488, 430, 421, 514, 421, 507, 477, 430, 421, 551, 536, 430, 477, 477, 430, 488, 488, 513, 430, 477, 421, 488, 477, 421, 507, 421, 488, 477, 477, 477, 430, 430, 430))
        self.assertEqual(record.annotations["abif_raw"]['P2AM1'], (19, 2286, 392, 2987, 681, 1205, 708, 1194, 622, 1023, 791, 846, 752, 631, 877, 1297, 1108, 437, 123, 764, 554, 866, 466, 747, 1071, 383, 602, 184, 565, 670, 456, 87, 46, 289, 60, 652, 1038, 82, 159, 319, 457, 721, 159, 144, 133, 39, 76, 60, 102, 406, 231, 210, 61, 178, 62, 27, 24, 31, 40, 25, 67, 33, 14, 143, 186, 81, 113, 160, 80, 47, 35, 16, 23, 43, 38, 22, 41, 32, 45, 208, 222, 217, 184, 49, 167, 72, 140, 59, 64, 35, 35, 25, 35, 28, 38, 99, 46, 26, 51, 27, 51, 36, 33, 40, 60, 95, 15, 21, 18, 19, 21, 33, 45, 21, 33, 20, 21, 37, 35, 28, 28, 28, 40, 35, 37, 49, 22, 40, 38, 25, 28, 36, 50, 18, 29, 23, 57, 39, 23, 13, 22, 22, 15, 38, 34, 33, 27, 41, 20, 24, 36, 30, 65, 45, 22, 34, 22, 26, 22, 31, 29, 22, 38, 32, 22, 32, 32, 46, 19, 15, 12, 21, 29, 24, 23, 19, 15, 29, 14, 15, 40, 23, 12, 21, 35, 26, 21, 32, 19, 23, 18, 14, 22, 13, 21, 17, 17, 19, 13, 35, 22, 23, 20, 19, 30, 50, 21, 23, 23, 14, 57, 31, 35, 20, 29, 18, 45, 27, 30, 15, 23, 18, 16, 25, 28, 41, 54, 28, 29, 55, 38, 37, 16, 26, 38, 28, 26, 16, 33, 51, 55, 63, 71, 65, 34, 79, 19, 69, 20, 23, 49, 35, 32, 27, 31, 39, 55, 12, 29, 21, 54, 23, 27, 32, 22, 20, 40, 17, 28, 21, 29, 46, 27, 16, 46, 23, 20, 45, 42, 23, 15, 19, 32, 26, 31, 37, 23, 32, 40, 50, 32, 17, 31, 19, 23, 48, 55, 50, 23, 38, 28, 26, 30, 32, 42, 18, 35, 18, 20, 37, 22, 42, 19, 36, 24, 20, 16, 15, 30, 51, 14, 32, 20, 28, 31, 23, 41, 36, 44, 39, 24, 30, 23, 17, 21, 24, 14, 23, 19, 17, 28, 23, 27, 18, 23, 23, 19, 21, 27, 16, 25, 34, 34, 20, 20, 27, 21, 48, 12, 26, 55, 21, 25, 21, 23, 32, 24, 31, 28, 38, 10, 15, 23, 21, 17, 26, 28, 26, 23, 23, 26, 19, 22, 25, 17, 36, 42, 28, 27, 38, 24, 22, 15, 19, 34, 31, 28, 31, 26, 37, 28, 54, 38, 20, 25, 23, 37, 36, 16, 35, 21, 31, 24, 40, 34, 39, 36, 28, 35, 25, 28, 23, 53, 23, 18, 22, 25, 44, 29, 14, 36, 29, 36, 42, 43, 29, 39, 30, 22, 26, 27, 40, 55, 36, 37, 23, 29, 40, 35, 28, 27, 32, 40, 22, 41, 24, 22, 48, 28, 40, 53, 25, 40, 39, 45, 27, 39, 53, 34, 51, 26, 26, 29, 22, 21, 44, 30, 59, 21, 18, 20, 27, 23, 33, 20, 23, 31, 20, 30, 46, 52, 30, 58, 28, 26, 12, 67, 35, 54, 15, 32, 37, 56, 20, 22, 14, 25, 47, 34, 29, 30, 21, 16, 22, 20, 13, 32, 56, 39, 40, 23, 20, 15, 28, 19, 43, 47, 31, 38, 30, 26, 28, 33, 18, 17, 19, 41, 19, 17, 84, 76, 61, 23, 30, 31, 24, 47, 51, 75, 40, 44, 105, 79, 43, 72, 17, 25, 29, 12, 31, 32, 32, 69, 87, 174, 48, 34, 63, 93, 135, 47, 38, 100, 88, 70, 42, 30, 22, 18, 22, 111, 77, 118, 64, 55, 54, 87, 58, 37, 35, 34, 39, 107, 67, 19, 5, 92, 78, 80, 68, 28, 44, 34, 77, 42, 48, 89, 69, 55, 48, 85, 46, 79, 135, 110, 198, 78, 54, 69, 152, 69, 68, 120, 53, 51, 15, 17, 36, 109, 104, 74, 99, 50, 132, 103, 76, 163, 77, 187, 141, 67, 64, 50, 151, 224, 79, 121, 102, 81, 93, 89, 69, 156, 82, 84, 114, 336, 59, 93, 128, 94, 77, 49, 145, 91, 158, 45, 3, 77, 122, 88, 99, 139, 84, 202, 98, 40, 94, 75, 89, 207, 113, 36, 107, 180, 137, 157, 203, 37, 154, 135, 92, 114, 320, 131, 362, 185, 189, 137, 92, 76, 178, 133, 135, 293, 121, 202, 169, 157, 139, 122, 206, 169, 36, 51, 48, 76, 161, 224, 216, 205, 210, 94, 232, 206, 218, 32, 133, 238, 42, 179, 158, 90, 211, 93, 110, 229, 223, 153, 78, 205, 192, 333, 255, 277, 222, 261, 350, 162, 138, 242, 179, 157, 95, 148, 416, 289, 165, 233, 238, 359, 156, 423, 189, 123, 252, 180, 148, 248, 259, 239, 243, 371, 256, 198, 252, 282, 200, 274, 178, 175, 223, 158, 261, 387, 212, 437, 338, 389, 247, 193, 291, 26, 345, 368, 221, 164, 194, 294, 329, 517, 370, 366, 359, 273, 251, 330, 312, 446, 492, 411, 335, 310, 385, 442, 439, 325, 221, 314, 259, 286, 167, 409, 458, 311, 263, 492, 333, 481, 307, 293, 511, 428, 752, 28, 257, 545, 21, 998))
        self.assertEqual(record.annotations["abif_raw"]['P2BA1'], 'CGAGAGCCGCGCTGTCACGGTGTTACGCACACGGTTAACGACATAAAGACACATGGCGAGGTCACAACGGGAGGAAACACCTAATTAAAGAAAATAACGAGAAGTATACAGGGCGGAGAGACCCTGCAGCTTAGGGGGGCGTAAGGGGTGGGGGAGGGGGAAGGGGGAGATTGTGAGATGAGAGGGTGAGACGGCTGTCGAACAGGGTATGGGTTAGAAGAGTTGGGAATGGCCTGGTGGTGTGATGGCATGGATATAGATAAGGAGTGAGGCCGAAGGTTTAGAAAAAAGTTCAGGAAGTGTGAGAGTTTAGAGACGGGCAGAGAATGGAGGTACATTGAGAAACAGGAACACAACGCAGGTCCGAAAACTTGGAGATAAATTCTGGAATTTGAGGAGGAAGTCTTGTGAGATTAGAGTCAGGAGAGAAAGAGACAATACAGGATTAGAAAAGACAGGGAGAAAGGAAGCGTACGAATTCGAATAACGAAGGAGATAGATGGAGCGCACACGGGGACACTACCGACGGCGCAAGCGCCGTACTAAATAATAATAAACCTAGTTAAAATATCAACAGCCTCACAAATATTCGAACGGACGTCTGGCCGCTTCGTTAGACGCAGTGAAGTGCTAGAGAGCAGCATACACAGGCGCACAGTGTGAACGCTTCTCACACCAACACGGTGGAAGAGGATGCACATGAGAGTATCACTCAACTCTATACTCACTTTCGTGATATACATACTCGCGCACGCGTCGTGTGGTGTCACATCACACACGCTCTCACCAGCGTACACACAGTTATCGACACCTATAGCGGCGCTCGTATCATATGGCAC')
        self.assertEqual(record.annotations["abif_raw"]['P2RL1'], (734, 400, 400, -300, 400, -300, 400, 300, 400, -300, -300, -300, 400, 411, -429, 400, -300, -300, -300, 400, -300, 400, -300, 400, -300, 400, -300, 400, 400, -300, -300, -300, -300, 400, -300, 400, -300, 308, 400, -300, 400, -400, -300, 400, -300, 400, -266, -525, 400, -300, 400, -300, -59, 400, -300, -300, 400, -300, -187, -442, -235, 400, 400, 400, -300, -300, 400, -300, -523, 401, -300, 542, 400, 0, -73, 400, -453, -515, 434, 289, -327, 400, -300, 400, -9, 400, -300, 167, 400, -300, -300, 400, 437, -157, 400, -300, 40, 0, 500, -300, -300, -400, -363, -400, 400, -400, 400, -300, -466, 400, 500, -222, 221, 177, 123, 500, -79, 114, 400, -9, 0, -128, 627, -300, 83, 73, -567, 430, -354, -516, 400, -461, -400, 400, -28, 433, 642, -300, -458, -300, -300, -476, 125, -109, 145, 632, -300, -47, 400, 400, -300, 0, -74, -211, 664, 210, -300, -30, 434, -77, 641, -400, 522, -300, 400, -316, -100, -61, 97, -300, -470, 207, -131, -84, -300, -200, -312, 400, -327, -207, -167, -100, 0, 400, -78, 400, -426, -300, -168, 400, -141, 549, 47, 647, -110, 203, 100, -118, -146, 66, 500, -548, 507, 21, 109, 100, 567, 100, -512, 225, 255, 426, 172, -306, 30, -518, 379, 508, 357, 524, 400, 205, 400, -227, 400, 400, -456, -215, -23, -249, -326, 199, 400, -245, 65, -33, 122, -179, -45, 280, 75, -41, 7, 95, 103, -164, -300, -24, -155, -216, -83, 379, -112, 108, 109, -30, -151, 138, -200, 18, 49, -66, -118, 323, -300, 400, -85, -100, -100, -300, 400, -214, 372, -300, -114, 561, -300, 70, 216, 370, -300, 0, -13, 260, 378, 227, -133, -99, -57, -82, -106, 78, -125, 100, 126, 43, 85, 35, -36, 32, -300, -69, -100, -109, -232, 117, -52, -242, -200, -48, 50, -14, 400, 112, -300, -300, -320, 400, 400, -193, 0, -315, -16, 186, 101, 400, -172, -212, 621, -300, 0, 244, -300, -74, -224, -218, 0, 559, -300, -442, -200, -300, 618, -300, -579, 534, 249, -300, 159, 400, -64, 494, -487, -2, 404, 83, 87, -21, -300, -45, -9, 216, -200, 400, 200, 340, 400, -349, -173, -82, 97, -321, -259, 612, -300, 71, 9, 101, 114, -202, -75, -58, 400, -435, -221, -92, 257, 500, -253, -139, -459, 174, -300, -300, -172, 200, -498, -327, 400, -355, 400, -100, 226, -85, -339, 400, 13, -244, 46, 223, 179, -220, 419, 555, -300, -38, 400, -31, 200, -282, -147, -400, 257, -300, -47, -300, -300, -227, -553, 400, -83, 500, -300, 361, -300, -478, -272, -341, -343, 0, -100, 400, 433, -300, -300, -300, -283, -300, 19, 579, 400, -461, 400, 227, 400, 334, -300, 376, 400, -200, -231, 400, -300, 400, -582, -300, 344, 461, -300, 500, -533, 402, -237, 568, -249, 400, 500, -191, 280, 526, 400, 400, 400, -191, 400, 100, 400, -300, 400, 400, -454, 253, 400, 125, -300, -317, 500, -300, -189, -207, 400, 22, 61, 400, 400, 400, -300, 400, 400, -300, 400, -300, 3, 578, -300, -300, 400, -300, 400, -300, -74, -7, 400, -262, 400, 400, -300, 400, -300, -300, 100, -300, -300, -300, -35, 400, 400, -300, 400, -300, 500, -300, 400, 400, 522, -300, 500, 400, -300, -412, 400, -300, 500, 400, -300, 400, -300, -300, -580, 400, 475, 400, -300, 400, -300, 11, -300, 500, 400, -300, -300, 400, -300, 400, -300, -300, 400, -300, -525, 400, 400, -300, -300, -300, 543, 400, -400, -300, 649, -300, -300, 400, -400, 400, -300, 400, 400, -300, 400, -300, 400, -300, -300, -300, 400, -300, 400, 500, -300, 659, 400, -300, 400, -300, 400, -300, 400, -300, 400, -300, 400, -300, 400, 400, -300, 400, -539, 400, 500, -300, 400, -300, 400, 400, -300, 400, -300, -300, 500, -300, 400, -300, -367, 400, -300, 400, 400, -300, -400, -300, 400, 400, -400, -300, -300, 400, -400, -19, 400, 400, -300, -300, 530, 400, -300, 400, -300, 400, 400, 400, -300, -300, -300, 400, 500, -300, 400, -300, 500, 400, -300, -300, -300, 400, -300, -300, 400, -300, 400, -300, -300, -300, 400, -300, 400, -300, 400, -300, -300, -300, 400, -300, 500, 400, -300, -300, 400, -300, -300, -300, 400, 500, 400, -300, 400, -300, 400, -300, -300, 400, -300, -300, 400, 400, 400, -300, 480, 400, -300, 173, 400, -300, 400, -300, 400, 400, 400, -300, -300, -300, 400, -300, 400, -300, 400, -400, -300, 400, 400, -300, 400, -300, 400, -300, -300, 400, -300, 400, -300, 400, 400, -300, -300, 400, 400, -300, -300, -300, 400, -300, -300, -300, -300, -300, 400, 400, -300, 400, 400, 400, -300, 400, 400, -300, -300, -300, -300, 400, -300, 400, -300, -300, 400, 400, -300, -300, 400, 400, -300, -300, 400, -300, 400, -300, -300, -300, 400, 400, 400, -300, -300, 400, -300, 500, 400, -400, 400, -300, -300, 400, -300, -300, 400, -300, -400, -300, 500, -300, -300, 400, 400, 400, -400, -300, 200, 400, -300, 120, 1565))
        self.assertEqual(record.annotations["abif_raw"]['PBAS1'], 'NNNNNNNNNNCNNNNNNGCTNNNNNGCTCACGTTGATTGCCATATCCTCACAGATTGCCTTCTCACCATTTGTCCCTTCTGATTGATCTATCGGATCGAGTGGTATATTTAACTTTGACATTAGCTGCTTGATATACTTGAAGCATAACATCTTGTTATTGGCAATTGACACAGTTCCACGATCAAGCGTCAAATCCGTCGTTGAATCGAATAGCTTTTTTAAATTCGGATTTTCAAAAACTGTGATAGCATATAGATTTCTGAACAGTGACTTTGCCTCAATACGTCGTAAATTCCGGAACATGTTCAACGAGATAAACGGTGACGATTGACGTACCAACAGGTAGCCGGTGATCGTGTAAATGTTCGCAAATATATCCTTCAACTCGGACGCCATTCCCGAATCCTGTTTTCCGCGAATCTCGATCGTCAGATTTCCGTCAATAATATTGCATAGCCTGATCGCCTGTGCCTTCGGAAACGTATCAATGACGTGATTGATCTCGCACACAATCTCACACTTGCCAACGCATTTTCGGCATTCTCGATGATCATCCGGGTTGATTTGATAACCATCGGGACATTTATCCGAGCAAAGGCCTGCCGTCGCCTTGATAGGCACTGTTTTGTTCGAGAGCACCGGATTCAGCTGCAGACACTGCTCACGGGTCACACAACGACGTTGAAGGAGAAGGTACAGGTGAGCATCACACTTTTCGATACACTTTCCCTTGTGATAGACATTCTTGCACGCGTGGCATGCTGTCGCATCATTCACACGCTCACAACCGCCCACGCATTGATCGTGACATCGATCGCCGTTCGCATCACATCCCGGG')
        self.assertEqual(record.annotations["abif_raw"]['PBAS2'], 'NNNNNNNNNNCNNNNNNGCTNNNNNGCTCACGTTGATTGCCATATCCTCACAGATTGCCTTCTCACCATTTGTCCCTTCTGATTGATCTATCGGATCGAGTGGTATATTTAACTTTGACATTAGCTGCTTGATATACTTGAAGCATAACATCTTGTTATTGGCAATTGACACAGTTCCACGATCAAGCGTCAAATCCGTCGTTGAATCGAATAGCTTTTTTAAATTCGGATTTTCAAAAACTGTGATAGCATATAGATTTCTGAACAGTGACTTTGCCTCAATACGTCGTAAATTCCGGAACATGTTCAACGAGATAAACGGTGACGATTGACGTACCAACAGGTAGCCGGTGATCGTGTAAATGTTCGCAAATATATCCTTCAACTCGGACGCCATTCCCGAATCCTGTTTTCCGCGAATCTCGATCGTCAGATTTCCGTCAATAATATTGCATAGCCTGATCGCCTGTGCCTTCGGAAACGTATCAATGACGTGATTGATCTCGCACACAATCTCACACTTGCCAACGCATTTTCGGCATTCTCGATGATCATCCGGGTTGATTTGATAACCATCGGGACATTTATCCGAGCAAAGGCCTGCCGTCGCCTTGATAGGCACTGTTTTGTTCGAGAGCACCGGATTCAGCTGCAGACACTGCTCACGGGTCACACAACGACGTTGAAGGAGAAGGTACAGGTGAGCATCACACTTTTCGATACACTTTCCCTTGTGATAGACATTCTTGCACGCGTGGCATGCTGTCGCATCATTCACACGCTCACAACCGCCCACGCATTGATCGTGACATCGATCGCCGTTCGCATCACATCCCGGG')
        self.assertEqual(record.annotations["abif_raw"]['PCON1'], '\x02\x02\x03\x04\x06\x05\x04\x04\t\x05\n\x04\x04\t\x05\x05\x06\x13\x1e\x0b\t\x06\t\x06\x05"\x16\'\x17\x12\x11):\x18"\x13\x0e:*\x1a\x14\x1444)6:6\'\x15*"/)/>>>>>>>>%\'6,,>>>>>>>;>88$&!.;>;+;>>>>>>;\'>>>>>>;>>>>;>;>>>>>>>;>>>>;;>>>>>>>;>>>>>>>>;>>>>>>>>>>>>>>>>>>>>>>>>>>>6>>>>>>>>>>>>;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>61>>>>>>>>>>>;;>>>>>>>>>>>>>6666;>>>>>>;;>>>>>>>11>>>>;;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>;>>>>>>>>>;;>>>>>;;>>>>>>>>>>>>>>>>>>>;;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>1>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>8>>>>1>>>>>>>>>>>>>>6>>>>>>>>>>>>>>>>>>>/6>>>6:/::::6::6:::::6:::::::60::::::::::::66:::::::6::6:6:::::::::-*:::.15355)::3::322::+)0::4:.:::636*:3:36:5:5::31.$-5+1.463:4::::(5423 :\x1b221:2,,%:2+*64:25%:\x1f234:2:\')466312(25413116\'%63\x1f\' \x1661\x1762\r1\x10\'-"\'6-56 5"5/(4\'*6663%\x1b3*(4 \'"6/(6/)65*45 *(5\x1f\x1f\x1f -#\x1e(  %*/(\x11\r\x13\x18 \x15/\x1d / 4\x17\x1e %(\x1d 5 \x1f%!\x1f\x1f( %\x19\x1d\x0c\x0f\x12$\x1f\x1f3\x1a\x1e\x1f\x1d\x1e\x1f\x1e\x15 "\x1f\x1e\x1e\x12\x1e\x1a\x13\x1a!\'\x19\x1d\x1a\x1e* \x1f\x1d"\x1d\x1c\x1f\x1f\x1a"\x1f\x1a\x1b\x19&\x17/\x1f\x0e\x0e\n')
        self.assertEqual(record.annotations["abif_raw"]['PCON2'], '\x02\x02\x03\x04\x06\x05\x04\x04\t\x05\n\x04\x04\t\x05\x05\x06\x13\x1e\x0b\t\x06\t\x06\x05"\x16\'\x17\x12\x11):\x18"\x13\x0e:*\x1a\x14\x1444)6:6\'\x15*"/)/>>>>>>>>%\'6,,>>>>>>>;>88$&!.;>;+;>>>>>>;\'>>>>>>;>>>>;>;>>>>>>>;>>>>;;>>>>>>>;>>>>>>>>;>>>>>>>>>>>>>>>>>>>>>>>>>>>6>>>>>>>>>>>>;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>61>>>>>>>>>>>;;>>>>>>>>>>>>>6666;>>>>>>;;>>>>>>>11>>>>;;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>;>>>>>>>>>;;>>>>>;;>>>>>>>>>>>>>>>>>>>;;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>1>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>8>>>>1>>>>>>>>>>>>>>6>>>>>>>>>>>>>>>>>>>/6>>>6:/::::6::6:::::6:::::::60::::::::::::66:::::::6::6:6:::::::::-*:::.15355)::3::322::+)0::4:.:::636*:3:36:5:5::31.$-5+1.463:4::::(5423 :\x1b221:2,,%:2+*64:25%:\x1f234:2:\')466312(25413116\'%63\x1f\' \x1661\x1762\r1\x10\'-"\'6-56 5"5/(4\'*6663%\x1b3*(4 \'"6/(6/)65*45 *(5\x1f\x1f\x1f -#\x1e(  %*/(\x11\r\x13\x18 \x15/\x1d / 4\x17\x1e %(\x1d 5 \x1f%!\x1f\x1f( %\x19\x1d\x0c\x0f\x12$\x1f\x1f3\x1a\x1e\x1f\x1d\x1e\x1f\x1e\x15 "\x1f\x1e\x1e\x12\x1e\x1a\x13\x1a!\'\x19\x1d\x1a\x1e* \x1f\x1d"\x1d\x1c\x1f\x1f\x1a"\x1f\x1a\x1b\x19&\x17/\x1f\x0e\x0e\n')
        self.assertEqual(record.annotations["abif_raw"]['PDMF1'], 'KB_3730_POP7_BDTv3.mob')
        self.assertEqual(record.annotations["abif_raw"]['PDMF2'], 'KB_3730_POP7_BDTv3.mob')
        self.assertEqual(record.annotations["abif_raw"]['PLOC1'], (3, 27, 42, 48, 56, 64, 73, 85, 93, 102, 109, 116, 138, 149, 160, 179, 184, 200, 214, 230, 238, 258, 265, 284, 289, 302, 312, 325, 337, 345, 353, 366, 382, 396, 408, 423, 430, 443, 457, 468, 479, 487, 498, 510, 521, 535, 547, 559, 572, 581, 592, 601, 614, 629, 639, 652, 665, 677, 689, 701, 715, 727, 739, 751, 760, 770, 782, 791, 804, 818, 831, 844, 856, 868, 879, 890, 903, 916, 928, 940, 953, 966, 975, 988, 1000, 1012, 1022, 1034, 1046, 1057, 1068, 1080, 1094, 1108, 1120, 1130, 1142, 1155, 1167, 1179, 1192, 1204, 1217, 1228, 1239, 1249, 1260, 1270, 1283, 1296, 1306, 1318, 1329, 1342, 1355, 1367, 1380, 1393, 1403, 1413, 1424, 1437, 1448, 1461, 1473, 1486, 1499, 1510, 1523, 1536, 1547, 1560, 1571, 1582, 1592, 1603, 1614, 1627, 1640, 1652, 1666, 1677, 1688, 1700, 1711, 1722, 1734, 1745, 1757, 1768, 1779, 1792, 1805, 1818, 1830, 1843, 1856, 1866, 1877, 1890, 1902, 1916, 1927, 1938, 1949, 1961, 1973, 1985, 1999, 2009, 2020, 2031, 2042, 2055, 2068, 2081, 2093, 2105, 2115, 2126, 2138, 2152, 2164, 2176, 2187, 2198, 2210, 2223, 2235, 2248, 2260, 2271, 2282, 2293, 2305, 2317, 2329, 2342, 2355, 2366, 2378, 2392, 2404, 2415, 2428, 2439, 2450, 2462, 2475, 2487, 2498, 2509, 2521, 2533, 2545, 2557, 2570, 2582, 2594, 2606, 2618, 2628, 2639, 2650, 2662, 2674, 2686, 2698, 2712, 2724, 2734, 2746, 2758, 2770, 2781, 2791, 2802, 2814, 2826, 2837, 2848, 2860, 2873, 2885, 2897, 2911, 2921, 2933, 2945, 2957, 2968, 2979, 2991, 3002, 3014, 3027, 3040, 3050, 3063, 3075, 3086, 3099, 3111, 3124, 3135, 3146, 3157, 3169, 3181, 3193, 3206, 3217, 3229, 3241, 3254, 3265, 3277, 3289, 3302, 3313, 3324, 3335, 3347, 3359, 3370, 3382, 3395, 3407, 3419, 3431, 3443, 3454, 3466, 3477, 3489, 3501, 3513, 3526, 3538, 3551, 3563, 3573, 3584, 3595, 3608, 3620, 3632, 3644, 3655, 3666, 3678, 3690, 3703, 3714, 3728, 3738, 3750, 3761, 3773, 3784, 3797, 3810, 3822, 3833, 3847, 3857, 3870, 3882, 3893, 3906, 3918, 3931, 3941, 3954, 3966, 3977, 3988, 4000, 4011, 4023, 4035, 4046, 4058, 4071, 4084, 4095, 4107, 4119, 4131, 4144, 4157, 4169, 4180, 4193, 4204, 4216, 4228, 4240, 4252, 4264, 4276, 4287, 4299, 4310, 4322, 4334, 4346, 4358, 4371, 4382, 4394, 4405, 4417, 4428, 4440, 4451, 4463, 4474, 4487, 4499, 4512, 4524, 4536, 4547, 4558, 4570, 4582, 4594, 4607, 4620, 4633, 4643, 4655, 4667, 4679, 4690, 4702, 4714, 4726, 4738, 4750, 4762, 4775, 4786, 4798, 4809, 4821, 4834, 4846, 4858, 4870, 4882, 4894, 4905, 4917, 4929, 4941, 4953, 4965, 4977, 4988, 5000, 5012, 5024, 5036, 5049, 5060, 5072, 5084, 5096, 5108, 5119, 5131, 5144, 5154, 5167, 5179, 5190, 5202, 5215, 5227, 5239, 5250, 5261, 5273, 5284, 5297, 5308, 5320, 5331, 5344, 5356, 5369, 5380, 5392, 5403, 5415, 5428, 5441, 5453, 5465, 5478, 5489, 5501, 5514, 5525, 5538, 5550, 5562, 5574, 5586, 5598, 5610, 5622, 5635, 5646, 5657, 5671, 5684, 5695, 5706, 5718, 5730, 5742, 5754, 5765, 5777, 5788, 5800, 5812, 5824, 5837, 5849, 5861, 5873, 5885, 5898, 5909, 5921, 5933, 5946, 5957, 5969, 5982, 5994, 6006, 6018, 6030, 6041, 6052, 6064, 6075, 6087, 6099, 6112, 6124, 6136, 6147, 6159, 6170, 6182, 6195, 6207, 6219, 6232, 6244, 6255, 6267, 6278, 6290, 6303, 6314, 6326, 6339, 6351, 6363, 6375, 6387, 6400, 6411, 6423, 6434, 6447, 6459, 6471, 6483, 6495, 6508, 6519, 6531, 6543, 6554, 6566, 6578, 6589, 6601, 6614, 6626, 6639, 6652, 6664, 6676, 6687, 6700, 6710, 6723, 6735, 6747, 6760, 6770, 6782, 6794, 6805, 6818, 6829, 6841, 6854, 6866, 6879, 6892, 6905, 6916, 6926, 6938, 6951, 6963, 6975, 6986, 6998, 7011, 7023, 7036, 7048, 7060, 7072, 7083, 7095, 7106, 7119, 7131, 7143, 7155, 7167, 7179, 7192, 7204, 7217, 7228, 7240, 7252, 7265, 7277, 7289, 7300, 7314, 7324, 7336, 7348, 7361, 7372, 7384, 7395, 7408, 7420, 7432, 7445, 7457, 7469, 7481, 7493, 7505, 7517, 7529, 7542, 7553, 7565, 7577, 7589, 7600, 7611, 7624, 7636, 7649, 7662, 7673, 7686, 7697, 7708, 7721, 7732, 7745, 7758, 7770, 7781, 7793, 7806, 7817, 7828, 7840, 7852, 7864, 7877, 7890, 7902, 7913, 7925, 7936, 7949, 7962, 7975, 7986, 7997, 8008, 8020, 8031, 8043, 8055, 8067, 8079, 8093, 8104, 8116, 8129, 8141, 8153, 8164, 8179, 8189, 8201, 8214, 8226, 8238, 8249, 8260, 8273, 8285, 8296, 8307, 8319, 8331, 8344, 8356, 8368, 8382, 8393, 8405, 8417, 8428, 8440, 8452, 8464, 8475, 8487, 8500, 8512, 8525, 8536, 8549, 8560, 8573, 8584, 8595, 8607, 8619, 8630, 8641, 8654, 8667, 8679, 8691, 8703, 8715, 8727, 8739, 8752, 8764, 8776, 8787, 8798, 8810, 8823, 8834, 8846, 8856, 8870, 8882, 8893, 8907, 8919, 8931, 8943, 8953, 8966, 8978, 8990, 9003, 9013, 9028, 9039, 9051, 9062, 9074, 9086, 9099, 9112, 9124, 9136, 9148, 9160, 9171, 9183, 9195, 9206, 9217, 9231, 9243, 9254, 9266, 9278, 9289, 9301, 9315, 9327, 9339, 9350, 9362, 9372, 9386, 9395, 9410, 9422, 9433, 9447, 9460, 9470, 9482, 9494, 9506, 9518, 9529, 9544, 9554, 9568, 9579, 9591, 9603, 9616, 9628, 9642, 9652, 9662, 9675, 9687, 9699, 9714, 9725, 9737, 9749, 9761, 9773, 9785, 9799, 9812, 9822, 9835, 9846, 9858, 9870, 9881, 9894, 9905, 9917, 9928, 9938, 9952, 9965, 9975, 9985, 9994))
        self.assertEqual(record.annotations["abif_raw"]['PLOC2'], (3, 27, 42, 48, 56, 64, 73, 85, 93, 102, 109, 116, 138, 149, 160, 179, 184, 200, 214, 230, 238, 258, 265, 284, 289, 302, 312, 325, 337, 345, 353, 366, 382, 396, 408, 423, 430, 443, 457, 468, 479, 487, 498, 510, 521, 535, 547, 559, 572, 581, 592, 601, 614, 629, 639, 652, 665, 677, 689, 701, 715, 727, 739, 751, 760, 770, 782, 791, 804, 818, 831, 844, 856, 868, 879, 890, 903, 916, 928, 940, 953, 966, 975, 988, 1000, 1012, 1022, 1034, 1046, 1057, 1068, 1080, 1094, 1108, 1120, 1130, 1142, 1155, 1167, 1179, 1192, 1204, 1217, 1228, 1239, 1249, 1260, 1270, 1283, 1296, 1306, 1318, 1329, 1342, 1355, 1367, 1380, 1393, 1403, 1413, 1424, 1437, 1448, 1461, 1473, 1486, 1499, 1510, 1523, 1536, 1547, 1560, 1571, 1582, 1592, 1603, 1614, 1627, 1640, 1652, 1666, 1677, 1688, 1700, 1711, 1722, 1734, 1745, 1757, 1768, 1779, 1792, 1805, 1818, 1830, 1843, 1856, 1866, 1877, 1890, 1902, 1916, 1927, 1938, 1949, 1961, 1973, 1985, 1999, 2009, 2020, 2031, 2042, 2055, 2068, 2081, 2093, 2105, 2115, 2126, 2138, 2152, 2164, 2176, 2187, 2198, 2210, 2223, 2235, 2248, 2260, 2271, 2282, 2293, 2305, 2317, 2329, 2342, 2355, 2366, 2378, 2392, 2404, 2415, 2428, 2439, 2450, 2462, 2475, 2487, 2498, 2509, 2521, 2533, 2545, 2557, 2570, 2582, 2594, 2606, 2618, 2628, 2639, 2650, 2662, 2674, 2686, 2698, 2712, 2724, 2734, 2746, 2758, 2770, 2781, 2791, 2802, 2814, 2826, 2837, 2848, 2860, 2873, 2885, 2897, 2911, 2921, 2933, 2945, 2957, 2968, 2979, 2991, 3002, 3014, 3027, 3040, 3050, 3063, 3075, 3086, 3099, 3111, 3124, 3135, 3146, 3157, 3169, 3181, 3193, 3206, 3217, 3229, 3241, 3254, 3265, 3277, 3289, 3302, 3313, 3324, 3335, 3347, 3359, 3370, 3382, 3395, 3407, 3419, 3431, 3443, 3454, 3466, 3477, 3489, 3501, 3513, 3526, 3538, 3551, 3563, 3573, 3584, 3595, 3608, 3620, 3632, 3644, 3655, 3666, 3678, 3690, 3703, 3714, 3728, 3738, 3750, 3761, 3773, 3784, 3797, 3810, 3822, 3833, 3847, 3857, 3870, 3882, 3893, 3906, 3918, 3931, 3941, 3954, 3966, 3977, 3988, 4000, 4011, 4023, 4035, 4046, 4058, 4071, 4084, 4095, 4107, 4119, 4131, 4144, 4157, 4169, 4180, 4193, 4204, 4216, 4228, 4240, 4252, 4264, 4276, 4287, 4299, 4310, 4322, 4334, 4346, 4358, 4371, 4382, 4394, 4405, 4417, 4428, 4440, 4451, 4463, 4474, 4487, 4499, 4512, 4524, 4536, 4547, 4558, 4570, 4582, 4594, 4607, 4620, 4633, 4643, 4655, 4667, 4679, 4690, 4702, 4714, 4726, 4738, 4750, 4762, 4775, 4786, 4798, 4809, 4821, 4834, 4846, 4858, 4870, 4882, 4894, 4905, 4917, 4929, 4941, 4953, 4965, 4977, 4988, 5000, 5012, 5024, 5036, 5049, 5060, 5072, 5084, 5096, 5108, 5119, 5131, 5144, 5154, 5167, 5179, 5190, 5202, 5215, 5227, 5239, 5250, 5261, 5273, 5284, 5297, 5308, 5320, 5331, 5344, 5356, 5369, 5380, 5392, 5403, 5415, 5428, 5441, 5453, 5465, 5478, 5489, 5501, 5514, 5525, 5538, 5550, 5562, 5574, 5586, 5598, 5610, 5622, 5635, 5646, 5657, 5671, 5684, 5695, 5706, 5718, 5730, 5742, 5754, 5765, 5777, 5788, 5800, 5812, 5824, 5837, 5849, 5861, 5873, 5885, 5898, 5909, 5921, 5933, 5946, 5957, 5969, 5982, 5994, 6006, 6018, 6030, 6041, 6052, 6064, 6075, 6087, 6099, 6112, 6124, 6136, 6147, 6159, 6170, 6182, 6195, 6207, 6219, 6232, 6244, 6255, 6267, 6278, 6290, 6303, 6314, 6326, 6339, 6351, 6363, 6375, 6387, 6400, 6411, 6423, 6434, 6447, 6459, 6471, 6483, 6495, 6508, 6519, 6531, 6543, 6554, 6566, 6578, 6589, 6601, 6614, 6626, 6639, 6652, 6664, 6676, 6687, 6700, 6710, 6723, 6735, 6747, 6760, 6770, 6782, 6794, 6805, 6818, 6829, 6841, 6854, 6866, 6879, 6892, 6905, 6916, 6926, 6938, 6951, 6963, 6975, 6986, 6998, 7011, 7023, 7036, 7048, 7060, 7072, 7083, 7095, 7106, 7119, 7131, 7143, 7155, 7167, 7179, 7192, 7204, 7217, 7228, 7240, 7252, 7265, 7277, 7289, 7300, 7314, 7324, 7336, 7348, 7361, 7372, 7384, 7395, 7408, 7420, 7432, 7445, 7457, 7469, 7481, 7493, 7505, 7517, 7529, 7542, 7553, 7565, 7577, 7589, 7600, 7611, 7624, 7636, 7649, 7662, 7673, 7686, 7697, 7708, 7721, 7732, 7745, 7758, 7770, 7781, 7793, 7806, 7817, 7828, 7840, 7852, 7864, 7877, 7890, 7902, 7913, 7925, 7936, 7949, 7962, 7975, 7986, 7997, 8008, 8020, 8031, 8043, 8055, 8067, 8079, 8093, 8104, 8116, 8129, 8141, 8153, 8164, 8179, 8189, 8201, 8214, 8226, 8238, 8249, 8260, 8273, 8285, 8296, 8307, 8319, 8331, 8344, 8356, 8368, 8382, 8393, 8405, 8417, 8428, 8440, 8452, 8464, 8475, 8487, 8500, 8512, 8525, 8536, 8549, 8560, 8573, 8584, 8595, 8607, 8619, 8630, 8641, 8654, 8667, 8679, 8691, 8703, 8715, 8727, 8739, 8752, 8764, 8776, 8787, 8798, 8810, 8823, 8834, 8846, 8856, 8870, 8882, 8893, 8907, 8919, 8931, 8943, 8953, 8966, 8978, 8990, 9003, 9013, 9028, 9039, 9051, 9062, 9074, 9086, 9099, 9112, 9124, 9136, 9148, 9160, 9171, 9183, 9195, 9206, 9217, 9231, 9243, 9254, 9266, 9278, 9289, 9301, 9315, 9327, 9339, 9350, 9362, 9372, 9386, 9395, 9410, 9422, 9433, 9447, 9460, 9470, 9482, 9494, 9506, 9518, 9529, 9544, 9554, 9568, 9579, 9591, 9603, 9616, 9628, 9642, 9652, 9662, 9675, 9687, 9699, 9714, 9725, 9737, 9749, 9761, 9773, 9785, 9799, 9812, 9822, 9835, 9846, 9858, 9870, 9881, 9894, 9905, 9917, 9928, 9938, 9952, 9965, 9975, 9985, 9994))
        self.assertEqual(record.annotations["abif_raw"]['PSZE1'], 96)
        self.assertEqual(record.annotations["abif_raw"]['PTYP1'], '96-Well')
        self.assertEqual(record.annotations["abif_raw"]['RGOw1'], 'GW')
        self.assertEqual(record.annotations["abif_raw"]['RMXV1'], '2')
        self.assertEqual(record.annotations["abif_raw"]['RMdN1'], 'LongSeq50_POP7_2')
        self.assertEqual(record.annotations["abif_raw"]['RMdX1'], """<?xml version="1.0"?>
<Run-Module-Instance><Name>LongSeq50_POP7_2</Name><Version>2</Version><Method-Name>LongSeq50_POP7</Method-Name><Method-Version>1</Method-Version><Comment></Comment><Parameters><Parameter><Token>DC_Cap96_Fill</Token><Value>17800</Value></Parameter><Parameter><Token>DC_Cap96_EPS_ARC_SELF</Token><Value>10</Value></Parameter><Parameter><Token>DC_Prep_Time</Token><Value>600</Value></Parameter><Parameter><Token>DC_Scale_Divisor</Token><Value>2.0</Value></Parameter><Parameter><Token>DC_Down_Sample</Token><Value>1</Value></Parameter><Parameter><Token>DC_Laser_Power_Setting</Token><Value>25.0</Value></Parameter><Parameter><Token>DC_RS_Binning_Pattern</Token><Value></Value></Parameter><Parameter><Token>DC_RS_NumCaps</Token><Value></Value></Parameter><Parameter><Token>DC_RS_CSData</Token><Value></Value></Parameter><Parameter><Token>Oven_Temperature</Token><Value>60</Value></Parameter><Parameter><Token>PreRun_Voltage</Token><Value>15.0</Value></Parameter><Parameter><Token>DC_PreRun_Time</Token><Value>180</Value></Parameter><Parameter><Token>Injection_Voltage</Token><Value>1.5</Value></Parameter><Parameter><Token>DC_Injection_Time</Token><Value>10</Value></Parameter><Parameter><Token>First_ReadOut_Time</Token><Value>300</Value></Parameter><Parameter><Token>Second_ReadOut_Time</Token><Value>300</Value></Parameter><Parameter><Token>Exposure_Time</Token><Value>0</Value></Parameter><Parameter><Token>Run_Voltage</Token><Value>8.5</Value></Parameter><Parameter><Token>Voltage_Number_Of_Steps</Token><Value>40</Value></Parameter><Parameter><Token>Voltage_Step_Interval</Token><Value>15</Value></Parameter><Parameter><Token>Voltage_Tolerance</Token><Value>0.6</Value></Parameter><Parameter><Token>Current_Stability</Token><Value>15.0</Value></Parameter><Parameter><Token>DC_Ramp_Delay</Token><Value>600</Value></Parameter><Parameter><Token>DC_Data_Delay</Token><Value>405</Value></Parameter><Parameter><Token>DC_Run_Time</Token><Value>5000</Value></Parameter></Parameters></Run-Module-Instance>""")
        self.assertEqual(record.annotations["abif_raw"]['RPrN1'], 'BDV3-SeqRun')
        self.assertEqual(record.annotations["abif_raw"]['RPrV1'], '1')
        self.assertEqual(record.annotations["abif_raw"]['RUND1'], '2014-06-04')
        self.assertEqual(record.annotations["abif_raw"]['RUND2'], '2014-06-04')
        self.assertEqual(record.annotations["abif_raw"]['RUND3'], '2014-06-04')
        self.assertEqual(record.annotations["abif_raw"]['RUND4'], '2014-06-04')
        self.assertEqual(record.annotations["abif_raw"]['RUNT1'], '00:10:18')
        self.assertEqual(record.annotations["abif_raw"]['RUNT2'], '01:57:17')
        self.assertEqual(record.annotations["abif_raw"]['RUNT3'], '00:33:43')
        self.assertEqual(record.annotations["abif_raw"]['RUNT4'], '01:57:03')
        self.assertEqual(record.annotations["abif_raw"]['RunN1'], 'Run_AB3730_2014-06-04_00-10_0541')
        self.assertEqual(record.annotations["abif_raw"]['S/N%1'], (459, 535, 1082, 652))
        self.assertEqual(record.annotations["abif_raw"]['SCAN1'], 16627)
        self.assertEqual(record.annotations["abif_raw"]['SMED1'], 'May 06, 2014')
        self.assertEqual(record.annotations["abif_raw"]['SMLt1'], '1309487')
        self.assertEqual(record.annotations["abif_raw"]['SMPL1'], 'A6_1-DB3')
        self.assertEqual(record.annotations["abif_raw"]['SPAC1'], 18.261703491210938)
        self.assertEqual(record.annotations["abif_raw"]['SPAC2'], 'KB.bcp')
        self.assertEqual(record.annotations["abif_raw"]['SPAC3'], 18.261703491210938)
        self.assertEqual(record.annotations["abif_raw"]['SVER1'], '3.0')
        self.assertEqual(record.annotations["abif_raw"]['SVER2'], 'KB 1.4.0')
        self.assertEqual(record.annotations["abif_raw"]['SVER3'], '6258000-04 6258002-04 6258003-03 6258005-00')
        self.assertEqual(record.annotations["abif_raw"]['Scal1'], 2.0)
        self.assertEqual(record.annotations["abif_raw"]['Scan1'], 16627)
        self.assertEqual(record.annotations["abif_raw"]['TUBE1'], 'C12')
        self.assertEqual(record.annotations["abif_raw"]['Tmpr1'], 60)
        self.assertEqual(record.annotations["abif_raw"]['User1'], 'genewiz')
        self.assertEqual(record.annotations["abif_raw"]['phAR1'], -1.0)
        self.assertEqual(record.annotations["abif_raw"]['phCH1'], 'term')
        self.assertEqual(record.annotations["abif_raw"]['phDY1'], 'big')
        self.assertEqual(record.annotations["abif_raw"]['phQL1'], 99)
        self.assertEqual(record.annotations["abif_raw"]['phTR1'], (-1, -1))
        self.assertEqual(record.annotations["abif_raw"]['phTR2'], -1.0)

        self.assertEqual(record.annotations["abif_raw"]['AUDT1'], (56, 66, 98, 58, 40, 123, 46, -93, 88, 124, 76, 61, 54, 88, 79, 63, 84, 118, 101, 33, 64, 46, 37, 74, 94, 75, 39, 112, 61, 109, 106, 122, 115, 44, 116, 53, 47, 96, 51, 84, 54, 89, 74, 58, 48, 47, 46, 112, 57, 59, 52, 94, 59, 53, 115, 35, 111, 67, 38, 79, 112, 97, 117, 95, 108, 95, 101, 53, 32, 125, 51, 55, 82, 103, 120, 124, 102, 102, 91, 96, 35, 124, 68, 53, 120, 95, 55, 38, 79, 77, 98, 39, 126, 54, 47, 108, 68, 45, 98, 41, 106, 74, 74, 84, 77, 70, 52, 70, 105, 37, 80, 32, 112, 39, 73, 99, 102, 93, 52, 39, 85, 119, 92, 80, 99, 99, 123, 92, 65, 95, 125, 36, 65, 62, 65, 82, 53, 90, 40, 60, 112, 79, 67, 103, 96, 67, 122, 104, 66, 98, 123, 49, 66, 42, 64, 89, 37, 72, 95, 84, 81, 86, 57, 88, 51, 47, 123, 107, 36, 94, 92, 69, 43, 50, 87, 65, 101, 65, 84, 37, 123, 65, 66, 48, 50, 111, 71, 104, 110, 87, 34, 75, 117, 81, 116, 98, 39, 74, 33, 62, 121, 72, 80, 120, 37, 98, 103, 42, 112, 47, 89, 103, 110, 60, 70, 66, 102, 35, 118, 71, 57, 121, 64, 85, 41, 65, 107, 109, 36, 118, 113, 80, 74, 69, 91, 85, 93, 119, 74, 74, 90, 49, 59, 103, 112, 95, 96, 123, 65, 91, 53, 53, 91, 35, 71, 61, 110, 45, 105, 119, 56, 55, 91, 37, 41, 52, 35, 44, 117, 54, 65, 124, 68, 50, 45, 55, 103, 109, 95, 103, 40, 123, 93, 47, 65, 84, 94, 111, 103, 86, 124, 61, 91, 80, 91, 87, 48, 76, 39, 113, 32, 60, 110, 80, 55, 33, 67, 119, 82, 92, 52, 39, 104, 50, 100, 63, 55, 79, 66, 109, 120, 67, 66, 35, 118, 49, 102, 124, 40, 86, 123, 33, 65, 75, 38, 114, 124, 95, 58, 78, 105, 41, 92, 108, 90, 104, 70, 125, 74, 110, 38, 59, 124, 43, 84, 63, 67, 44, 95, 44, 113, 122, 55, 52, 42, 108, 101, 95, 77, 103, 104, 87, 126, 94, 40, 39, 59, 95, 47, 67, 93, 50, 90, 69, 60, 125, 103, 54, 77, 62, 80, 58, 89, 39, 64, 40, 110, 86, 99, 58, 39, 50, 107, 62, 49, 109, 90, 92, 103, 120, 85, 108, 63, 116, 116, 47, 59, 107, 36, 59, 126, 54, 62, 117, 100, 114, 86, 98, 72, 96, 54, 64, 118, 48, 107, 99, 32, 86, 38, 73, 112, 63, 72, 94, 80, 111, 41, 71, 114, 40, 75, 46, 76, 45, 91, 68, 87, 91, 90, 101, 104, 38, 48, 119, 35, 106, 39, 34, 76, 65, 88, 59, 58, 98, 122, 93, 81, 121, 81, 93, 98, 120, 100, 36, 88, 70, 36, 126, 56, 43, 33, 51, 64, 42, 55, 95, 54, 83, 118, 114, 115, 80, 111, 73, 88, 122, 55, 95, 108, 110, 113, 82, 57, 118, 67, 46, 86, 113, 96, 65, 32, 35, 112, 74, 69, 63, 82, 86, 117, 85, 36, 72, 84, 108, 58, 104, 126, 54, 79, 36, 64, 118, 61, 97, 105, 66, 120, 91, 70, 111, 104, 105, 88, 78, 34, 125, 97, 56, 114, 79, 114, 109, 62, 116, 125, 49, 86, 62, 65, 54, 91, 44, 33, 99, 86, 80, 32, 53, 71, 111, 45, 68, 55, 73, 32, 57, 120, 120, 115, 87, 68, 64, 52, 105, 114, 36, 71, 69, 49, 88, 122, 46, 60, 98, 91, 120, 105, 110, 74, 105, 108, 115, 62, 58, 60, 47, 46, 51, 93, 78, 69, 94, 79, 123, 105, 99, 55, 93, 99, 64, 79, 59, 125, 83, 36, 111, 97, 36, 70, 108, 64, 105, 50, 115, 111, 50, 84, 109, 96, 77, 65, 86, 105, 104, 63, 93, 39, 94, 87, 63, 77, 46, 76, 123, 111, 88, 106, 96, 81, 32, 68, 120, 54, 109, 125, 80, 122, 55, 73, 32, 90, 75, 85, 45, 106, 121, 33, 84, 87, 114, 75, 79, 123, 126, 63, 122, 59, 123, 87, 56, 60, 41, 57, 118, 75, 91, 89, 74, 40, 86, 93, 39, 71, 69, 35, 33, 35, 104, 109, 109, 102, 99, 52, 100, 38, 93, 63, 35, 38, 123, 58, 124, 101, 78, 94, 116, 112, 117, 34, 80, 126, 102, 73, 50, 80, 38, 78, 43, 69, 47, 71, 122, 94, 110, 107, 61, 46, 104, 94, 110, 64, 69, 75, 53, 108, 108, 109, 100, 36, 88, 70, 36, 126, 33, 83, 78, 109, 87, 46, 36, 55, 75, 79, 93, 101, 79, 89, 32, 126, 63, 112, 32, 93, 65, 75, 86, 83, 85, 40, 93, 121, 34, 47, 58, 66, 111, 122, 63, 120, 78, 117, 126, 41, 66, 42, 35, 107, 64, 69, 42, 70, 59, 50, 95, 47, 103, 89, 70, 105, 36, 116, 95, 83, 70, 96, 124, 122, 118, 110, 91, 121, 93, 49, 56, 96, 39, 124, 95, 110, 63, 50, 54, 63, 75, 59, 64, 87, 70, 86, 62, 65, 120, 101, 60, 62, 47, 68, 63, 42, 85, 113, 36, 91, 45, 103, 93, 65, 88, 99, 91, 33, 56, 47, 106, 45, 114, 100, 86, 32, 126, 105, 101, 35, 77, 59, 66, 76, 32, 110, 115, 117, 57, 45, 93, 72, 54, 88, 43, 57, 109, 117, 121, 37, 108, 70, 85, 46, 70, 52, 47, 111, 97, 119, 102, 84, 59, 71, 65, 122, 114, 124, 105, 107, 91, 125, 55, 45, 106, 65, 83, 95, 55, 87, 88, 55, 97, 94, 93, 42, 82, 105, 84, 75, 40, 62, 86, 113, 38, 60, 113, 95, 61, 38, 87, 46, 58, 32, 119, 84, 95, 82, 32, 120, 118, 39, 111, 62, 115, 49, 72, 119, 44, 82, 73, 91, 120, 115, 85, 51, 84, 37, 121, 99, 88, 125, 79, 33, 68, 90, 78, 117, 64, 118, 117, 108, 99, 42, 123, 100, 38, 93, 63, 35, 58, 123, 67, 115, 85, 124, 125, 38, 95, 91, 51, 65, 52, 122, 87, 50, 96, 38, 84, 65, 93, 66, 71, 109, 98, 44, 116, 74, 76, 86, 60, 97, 73, 106, 96, 69, 64, 63, 84, 99, 71, 43, 85, 88, 32, 37, 61, 70, 86, 118, 42, 80, 75, 64, 124, 81, 36, 79, 69, 72, 98, 88, 92, 34, 66, 45, 56, 39, 105, 50, 123, 38, 91, 121, 51, 40, 32, 93, 39, 110, 103, 103, 54, 49, 67, 56, 100, 107, 55, 89, 90, 70, 85, 63, 45, 52, 36, 38, 76, 107, 102, 71, 59, 55, 111, 70, 96, 124, 122, 118, 110, 91, 89, 60, 44, 104, 96, 43, 109, 38, 119, 69, 54, 71, 122, 32, 122, 70, 47, 34, 87, 116, 105, 120, 121, 36, 124, 70, 82, 114, 116, 108, 113, 97, 120, 107, 97, 98, 67, 68, 107, 36, 79, 69, 45, 96, 123, 115, 105, 79, 125, 76, 54, 105, 121, 90, 110, 115, 122, 51, 80, 60, 125, 117, 94, 90, 113, 78, 88, 43, 57, 95, 101, 32, 86, 38, 38, 79, 69, 54, 79, 76, 60, 115, 83, 97, 121, 68, 82, 33, 95, 84, 120, 35, 123, 38, 36, 121, 101, 50, 38, 100, 112, 59, 73, 94, 65, 110, 109, 64, 110, 110, 34, 69, 107, 81, 116, 65, 38, 98, 36, 83, 88, 116, 109, 120, 43, 57, 72, 103, 110, 98, 43, 78, 52, 43, 48, 110, 119, 107, 73, 54, 32, 43, 70, 119, 77, 122, 37, 70, 84, 33, 115, 122, 94, 96, 73, 102, 57, 47, 113, 119, 60, 81, 103, 40, 43, 107, 115, 57, 63, 79, 94, 63, 72, 89, 84, 119, 124, 93, 106, 76, 75, 112, 100, 70, 109, 126, 59, 91, 99, 77, 109, 78, 50, 67, 103, 71, 122, 94, 110, 55, 97, 67, 63, 75, 59, 64, 87, 70, 86, 62, 65, 120, 101, 60, 113, 70, 111, 59, 88, 46, 107, 72, 45, 107, 95, 82, 74, 37, 118, 58, 81, 62, 119, 62, 54, 63, 124, 36, 108, 81, 37, 58, 88, 71, 84, 116, 114, 87, 86, 42, 0))
        self.assertEqual(record.annotations["abif_raw"]['PXLB1'], 3)
        self.assertEqual(record.annotations["abif_raw"]['RMdV1'], '2')
        self.assertEqual(record.annotations["abif_raw"]['RGNm1'], 'Results_WellPosition')
        self.assertEqual(record.annotations["abif_raw"]['DATA9'], (111, 114, 120, 132, 149, 171, 193, 210, 224, 240, 264, 296, 333, 371, 404, 431, 455, 475, 497, 524, 558, 597, 640, 683, 726, 767, 807, 845, 883, 919, 953, 985, 1016, 1047, 1079, 1111, 1143, 1172, 1196, 1215, 1227, 1235, 1239, 1239, 1235, 1226, 1209, 1184, 1151, 1111, 1069, 1029, 993, 964, 939, 915, 884, 843, 789, 724, 655, 589, 534, 495, 470, 454, 440, 419, 390, 353, 312, 275, 246, 227, 217, 213, 211, 207, 201, 195, 191, 191, 196, 206, 219, 231, 242, 252, 262, 273, 284, 296, 307, 316, 323, 332, 345, 364, 386, 408, 427, 438, 441, 437, 430, 423, 418, 416, 414, 411, 407, 401, 395, 391, 388, 388, 388, 386, 381, 372, 358, 340, 319, 298, 277, 257, 239, 223, 206, 189, 171, 154, 137, 123, 112, 104, 100, 100, 103, 111, 122, 134, 146, 156, 164, 172, 183, 199, 224, 254, 288, 320, 347, 366, 379, 389, 397, 408, 421, 434, 442, 443, 435, 416, 389, 356, 321, 286, 255, 227, 204, 183, 162, 142, 121, 101, 85, 75, 71, 74, 79, 86, 92, 98, 104, 115, 130, 152, 179, 208, 239, 268, 294, 318, 339, 359, 378, 395, 411, 424, 431, 431, 421, 400, 370, 332, 291, 253, 218, 188, 162, 139, 117, 96, 78, 64, 57, 54, 54, 56, 58, 62, 69, 81, 100, 126, 155, 185, 214, 240, 266, 295, 330, 371, 414, 456, 489, 509, 514, 504, 483, 456, 424, 391, 357, 321, 283, 245, 208, 175, 150, 133, 127, 131, 144, 163, 190, 224, 264, 310, 361, 412, 460, 499, 525, 535, 528, 508, 478, 443, 403, 361, 317, 271, 223, 177, 136, 103, 80, 65, 55, 49, 44, 39, 37, 38, 46, 61, 82, 106, 132, 160, 191, 230, 279, 339, 408, 480, 547, 602, 640, 661, 665, 654, 629, 592, 542, 480, 411, 339, 270, 209, 159, 119, 88, 63, 44, 29, 20, 15, 15, 16, 17, 16, 14, 10, 8, 7, 7, 8, 9, 8, 7, 6, 5, 7, 9, 12, 13, 13, 13, 13, 14, 18, 23, 28, 33, 36, 37, 37, 36, 36, 38, 42, 51, 63, 79, 100, 125, 156, 193, 235, 282, 329, 372, 406, 424, 424, 407, 373, 328, 277, 226, 179, 137, 101, 72, 50, 34, 24, 19, 20, 23, 27, 30, 32, 33, 34, 36, 39, 42, 45, 48, 51, 56, 64, 77, 99, 128, 167, 214, 268, 328, 390, 451, 506, 551, 580, 589, 575, 540, 484, 414, 335, 255, 183, 121, 74, 42, 22, 12, 8, 7, 8, 9, 10, 12, 15, 19, 23, 28, 30, 31, 30, 28, 27, 27, 30, 32, 32, 29, 21, 11, 1, 0, 4, 23, 56, 102, 164, 244, 344, 467, 607, 753, 887, 989, 1043, 1040, 981, 878, 747, 605, 465, 338, 228, 140, 74, 30, 6, 0, 0, 0, 3, 5, 6, 7, 9, 11, 13, 15, 18, 20, 24, 29, 34, 37, 38, 36, 31, 24, 17, 12, 9, 9, 10, 11, 10, 8, 5, 2, 0, 0, 0, 2, 4, 6, 8, 10, 14, 19, 26, 32, 37, 39, 36, 29, 20, 12, 4, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 3, 7, 11, 15, 18, 19, 18, 16, 13, 10, 7, 4, 1, 0, 0, 1, 5, 10, 16, 22, 28, 32, 35, 37, 37, 35, 31, 26, 19, 14, 10, 8, 9, 13, 18, 23, 27, 30, 31, 31, 29, 28, 27, 28, 28, 29, 30, 29, 29, 28, 29, 30, 33, 35, 38, 41, 44, 46, 48, 50, 50, 49, 46, 42, 37, 32, 27, 23, 18, 13, 7, 3, 1, 2, 10, 27, 52, 88, 132, 184, 239, 292, 337, 370, 384, 377, 349, 304, 248, 187, 130, 81, 45, 22, 10, 8, 13, 22, 33, 44, 54, 60, 63, 60, 54, 46, 36, 28, 23, 22, 24, 28, 33, 36, 36, 32, 25, 16, 9, 3, 1, 0, 0, 0, 0, 2, 23, 71, 155, 282, 446, 638, 836, 1015, 1149, 1216, 1205, 1117, 967, 778, 579, 395, 244, 134, 63, 24, 5, 0, 0, 0, 2, 6, 13, 19, 25, 28, 28, 26, 22, 19, 16, 15, 15, 15, 15, 15, 16, 18, 19, 19, 17, 14, 10, 6, 4, 5, 9, 16, 25, 33, 40, 45, 47, 47, 43, 38, 30, 21, 13, 6, 2, 0, 0, 0, 1, 1, 2, 1, 1, 0, 0, 0, 0, 0, 1, 2, 3, 3, 3, 2, 1, 1, 0, 0, 0, 0, 0, 0, 3, 8, 14, 20, 23, 23, 19, 13, 7, 2, 0, 0, 0, 0, 0, 1, 3, 6, 12, 18, 25, 30, 33, 33, 30, 25, 18, 11, 6, 2, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 1, 4, 8, 15, 22, 29, 36, 42, 48, 53, 56, 56, 53, 46, 36, 24, 13, 5, 0, 0, 0, 4, 9, 15, 20, 24, 27, 28, 28, 27, 27, 28, 29, 32, 34, 35, 33, 28, 21, 14, 9, 7, 7, 7, 5, 0, 0, 0, 31, 99, 210, 362, 540, 722, 882, 995, 1044, 1022, 931, 785, 607, 424, 259, 132, 50, 8, 0, 0, 1, 3, 2, 1, 4, 11, 19, 28, 33, 35, 35, 34, 32, 29, 25, 21, 16, 12, 9, 8, 8, 7, 6, 4, 2, 0, 0, 0, 2, 4, 6, 7, 6, 6, 5, 5, 6, 9, 13, 18, 22, 24, 24, 20, 14, 7, 2, 0, 0, 3, 6, 8, 9, 7, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 3, 8, 15, 21, 24, 22, 16, 8, 1, 0, 1, 3, 3, 0, 0, 4, 42, 123, 253, 425, 621, 811, 963, 1051, 1059, 984, 841, 655, 457, 279, 140, 51, 7, 0, 0, 3, 6, 5, 3, 3, 5, 7, 8, 7, 5, 2, 0, 0, 1, 3, 5, 5, 4, 1, 0, 0, 0, 2, 5, 5, 0, 0, 3, 42, 131, 281, 486, 724, 957, 1143, 1244, 1238, 1125, 931, 695, 459, 260, 119, 37, 2, 0, 0, 3, 4, 5, 9, 16, 26, 34, 38, 37, 33, 29, 26, 27, 29, 32, 34, 34, 33, 30, 28, 26, 24, 24, 25, 29, 35, 44, 53, 60, 63, 61, 55, 46, 36, 29, 26, 26, 30, 35, 39, 40, 39, 36, 32, 28, 26, 24, 24, 22, 20, 18, 16, 14, 14, 14, 15, 15, 15, 14, 14, 15, 17, 21, 23, 23, 20, 14, 7, 3, 2, 3, 3, 1, 0, 0, 17, 64, 144, 252, 375, 488, 567, 594, 560, 472, 349, 215, 100, 26, 10, 56, 158, 297, 447, 579, 667, 693, 653, 558, 427, 289, 167, 76, 22, 0, 0, 2, 10, 17, 21, 25, 29, 31, 31, 30, 26, 22, 17, 13, 10, 7, 6, 7, 8, 9, 9, 7, 3, 0, 0, 0, 4, 5, 2, 0, 0, 11, 54, 130, 237, 359, 473, 556, 588, 565, 493, 390, 277, 173, 92, 39, 12, 5, 8, 14, 16, 11, 1, 0, 0, 39, 122, 253, 422, 608, 778, 900, 949, 916, 811, 657, 484, 322, 190, 96, 37, 7, 0, 1, 13, 23, 22, 7, 0, 0, 62, 236, 534, 937, 1385, 1789, 2054, 2114, 1951, 1602, 1149, 691, 320, 98, 43, 133, 320, 540, 737, 864, 900, 842, 711, 537, 357, 202, 90, 25, 0, 0, 2, 5, 4, 2, 3, 11, 23, 35, 43, 44, 38, 29, 19, 11, 7, 6, 9, 14, 20, 26, 31, 36, 39, 39, 36, 31, 24, 16, 8, 3, 0, 0, 0, 2, 6, 10, 14, 16, 16, 14, 11, 9, 8, 9, 12, 15, 16, 15, 12, 8, 3, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 4, 7, 10, 13, 15, 16, 15, 13, 10, 7, 4, 2, 0, 0, 0, 0, 0, 0, 2, 6, 10, 15, 21, 27, 32, 37, 41, 42, 40, 35, 27, 18, 10, 3, 0, 0, 2, 11, 24, 38, 51, 61, 63, 58, 47, 33, 20, 10, 5, 4, 5, 6, 7, 6, 4, 3, 2, 1, 0, 0, 0, 0, 4, 14, 27, 40, 50, 54, 49, 39, 27, 16, 7, 2, 0, 0, 0, 0, 4, 8, 8, 2, 0, 0, 29, 109, 245, 429, 631, 812, 929, 954, 880, 727, 530, 332, 169, 60, 5, 0, 0, 12, 23, 30, 33, 35, 35, 33, 28, 19, 11, 3, 0, 0, 1, 4, 8, 11, 14, 17, 21, 26, 30, 34, 36, 36, 32, 26, 18, 10, 4, 1, 0, 0, 0, 0, 0, 1, 2, 3, 5, 5, 5, 4, 2, 1, 0, 0, 0, 0, 0, 1, 2, 2, 1, 0, 0, 0, 1, 7, 14, 19, 19, 15, 10, 13, 35, 82, 155, 245, 335, 406, 438, 425, 368, 282, 188, 104, 44, 10, 0, 0, 2, 4, 3, 0, 0, 0, 1, 4, 6, 6, 4, 2, 2, 9, 21, 35, 43, 39, 19, 0, 0, 46, 188, 434, 768, 1140, 1476, 1699, 1753, 1623, 1341, 976, 607, 305, 105, 8, 0, 0, 15, 18, 10, 0, 0, 4, 19, 35, 47, 51, 49, 45, 40, 36, 33, 30, 27, 25, 24, 24, 22, 18, 11, 3, 0, 2, 9, 16, 14, 2, 0, 0, 61, 204, 433, 725, 1026, 1268, 1389, 1356, 1176, 897, 587, 313, 119, 16, 0, 0, 13, 20, 14, 4, 0, 0, 4, 11, 15, 14, 9, 3, 0, 0, 0, 0, 1, 0, 0, 0, 1, 5, 10, 14, 17, 17, 15, 13, 11, 13, 16, 22, 28, 33, 36, 36, 34, 29, 24, 18, 12, 7, 4, 2, 3, 6, 11, 16, 20, 24, 26, 26, 25, 24, 21, 18, 16, 18, 23, 32, 44, 56, 67, 72, 71, 63, 50, 36, 23, 13, 7, 5, 7, 12, 19, 27, 32, 33, 27, 16, 5, 0, 1, 10, 19, 19, 6, 0, 0, 42, 165, 365, 617, 873, 1071, 1163, 1125, 971, 742, 494, 276, 120, 32, 0, 0, 4, 9, 7, 1, 0, 0, 0, 3, 7, 7, 4, 0, 0, 0, 2, 7, 9, 5, 0, 0, 10, 63, 158, 289, 434, 562, 640, 649, 586, 470, 329, 194, 89, 25, 0, 0, 9, 19, 21, 15, 7, 0, 0, 0, 5, 14, 24, 34, 40, 43, 42, 39, 34, 29, 25, 23, 21, 19, 16, 14, 12, 13, 17, 24, 32, 40, 45, 44, 38, 28, 17, 6, 0, 0, 3, 12, 22, 32, 39, 43, 45, 45, 45, 44, 44, 42, 38, 34, 29, 25, 21, 18, 14, 11, 7, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 5, 10, 15, 22, 28, 34, 39, 42, 43, 42, 38, 33, 27, 22, 18, 18, 19, 21, 23, 24, 25, 27, 28, 31, 33, 33, 31, 26, 21, 17, 17, 23, 35, 50, 66, 77, 80, 75, 64, 49, 37, 30, 30, 35, 41, 43, 39, 29, 20, 18, 24, 33, 33, 17, 0, 0, 35, 189, 466, 844, 1257, 1610, 1810, 1801, 1586, 1225, 810, 437, 169, 25, 0, 0, 19, 32, 27, 13, 4, 4, 14, 26, 35, 37, 33, 27, 21, 18, 16, 17, 17, 17, 17, 16, 16, 17, 20, 23, 27, 31, 32, 31, 27, 21, 14, 8, 2, 0, 0, 1, 4, 10, 15, 20, 24, 28, 30, 31, 29, 24, 15, 5, 0, 3, 14, 25, 26, 13, 0, 0, 43, 194, 458, 814, 1201, 1535, 1731, 1736, 1548, 1212, 808, 426, 139, 0, 0, 104, 308, 542, 750, 885, 917, 844, 687, 488, 293, 139, 43, 0, 0, 2, 11, 15, 15, 17, 23, 32, 40, 44, 43, 36, 28, 21, 17, 16, 15, 15, 14, 12, 10, 9, 11, 13, 16, 17, 17, 16, 16, 17, 20, 26, 33, 39, 41, 39, 32, 21, 10, 2, 0, 4, 14, 25, 31, 30, 21, 11, 6, 11, 22, 28, 19, 0, 0, 9, 122, 343, 661, 1022, 1344, 1543, 1565, 1405, 1107, 752, 423, 178, 39, 0, 0, 10, 20, 17, 6, 0, 0, 5, 14, 19, 19, 13, 6, 0, 0, 0, 0, 1, 1, 2, 2, 2, 3, 4, 5, 7, 10, 13, 15, 14, 12, 8, 4, 1, 0, 0, 0, 0, 0, 0, 0, 5, 13, 22, 29, 32, 30, 26, 21, 19, 16, 10, 0, 0, 0, 36, 117, 241, 391, 534, 634, 663, 614, 499, 351, 206, 92, 23, 0, 0, 6, 15, 16, 13, 8, 4, 5, 7, 9, 9, 7, 5, 2, 0, 0, 0, 0, 0, 0, 0, 1, 4, 9, 15, 20, 23, 24, 21, 15, 9, 4, 1, 0, 0, 0, 0, 0, 0, 0, 1, 4, 8, 11, 12, 10, 7, 3, 1, 0, 0, 1, 3, 5, 8, 11, 14, 16, 15, 13, 9, 6, 6, 9, 11, 10, 3, 0, 0, 25, 95, 210, 356, 506, 627, 688, 674, 590, 460, 313, 181, 82, 23, 0, 0, 3, 10, 14, 16, 16, 17, 17, 16, 14, 10, 5, 2, 0, 0, 0, 0, 1, 2, 2, 2, 1, 0, 0, 0, 3, 10, 17, 23, 26, 26, 22, 17, 14, 15, 21, 29, 37, 44, 46, 43, 37, 29, 21, 15, 12, 11, 11, 12, 13, 14, 16, 16, 13, 6, 0, 0, 14, 63, 148, 263, 389, 500, 567, 573, 516, 412, 287, 170, 82, 30, 7, 3, 4, 4, 3, 5, 9, 12, 7, 0, 0, 9, 83, 227, 433, 670, 882, 1016, 1034, 931, 738, 504, 285, 122, 28, 0, 0, 6, 13, 11, 4, 2, 6, 16, 27, 33, 32, 25, 16, 7, 1, 0, 0, 0, 1, 5, 9, 14, 18, 20, 19, 15, 10, 5, 2, 0, 0, 0, 1, 5, 11, 18, 26, 31, 33, 31, 26, 19, 12, 6, 2, 0, 0, 0, 2, 5, 8, 11, 13, 15, 16, 17, 17, 16, 14, 12, 8, 4, 1, 0, 0, 1, 5, 9, 12, 14, 14, 12, 9, 7, 5, 6, 7, 9, 12, 14, 15, 14, 10, 5, 3, 4, 10, 16, 18, 10, 0, 0, 11, 83, 216, 399, 600, 772, 871, 869, 768, 596, 398, 219, 89, 17, 0, 0, 6, 11, 9, 2, 0, 0, 0, 4, 10, 14, 16, 19, 24, 29, 35, 38, 34, 21, 3, 0, 2, 51, 147, 283, 437, 575, 661, 671, 603, 475, 321, 178, 72, 13, 0, 0, 6, 10, 7, 0, 0, 1, 11, 25, 35, 38, 32, 19, 5, 0, 0, 2, 12, 16, 9, 0, 0, 20, 109, 267, 479, 705, 891, 988, 968, 837, 632, 406, 209, 73, 4, 0, 0, 15, 26, 28, 25, 22, 19, 16, 11, 6, 3, 6, 15, 29, 46, 59, 66, 64, 55, 41, 26, 13, 5, 0, 0, 0, 2, 7, 14, 21, 28, 33, 33, 28, 19, 9, 1, 0, 0, 4, 8, 7, 0, 0, 0, 29, 94, 191, 307, 416, 491, 512, 472, 382, 267, 155, 68, 16, 0, 0, 3, 8, 8, 4, 0, 0, 0, 0, 0, 0, 0, 5, 16, 33, 51, 67, 74, 72, 59, 41, 22, 7, 0, 0, 3, 12, 24, 36, 44, 46, 42, 31, 17, 5, 0, 4, 16, 31, 40, 38, 24, 7, 1, 24, 88, 191, 319, 446, 541, 580, 552, 466, 346, 219, 113, 41, 5, 0, 0, 5, 6, 4, 0, 0, 0, 0, 1, 3, 2, 1, 0, 0, 0, 2, 4, 6, 7, 7, 6, 4, 2, 0, 0, 0, 2, 10, 23, 39, 53, 62, 62, 54, 38, 21, 7, 0, 0, 4, 12, 17, 20, 18, 12, 7, 2, 0, 0, 0, 0, 0, 0, 2, 7, 13, 19, 22, 22, 17, 11, 4, 0, 0, 0, 6, 13, 20, 24, 25, 22, 16, 9, 3, 0, 0, 0, 2, 3, 4, 3, 3, 4, 6, 11, 16, 22, 26, 27, 24, 20, 13, 7, 3, 1, 1, 4, 7, 10, 13, 15, 17, 17, 18, 18, 17, 16, 15, 14, 12, 12, 11, 12, 14, 16, 19, 21, 23, 25, 27, 28, 30, 33, 35, 35, 34, 30, 23, 15, 7, 2, 1, 5, 14, 26, 40, 53, 63, 69, 70, 64, 55, 44, 34, 27, 21, 15, 6, 0, 0, 11, 57, 137, 245, 364, 467, 528, 531, 474, 367, 234, 105, 10, 0, 25, 162, 371, 617, 847, 1004, 1045, 961, 775, 538, 308, 131, 28, 0, 0, 8, 17, 14, 4, 0, 1, 15, 34, 52, 62, 63, 56, 44, 30, 17, 8, 3, 0, 0, 0, 0, 4, 12, 23, 37, 49, 58, 60, 55, 44, 30, 17, 7, 1, 0, 0, 0, 0, 1, 2, 3, 5, 5, 5, 4, 3, 3, 5, 7, 9, 11, 11, 9, 7, 6, 6, 8, 9, 10, 10, 7, 3, 0, 0, 2, 10, 20, 31, 40, 44, 43, 37, 27, 17, 9, 4, 3, 8, 15, 24, 33, 38, 40, 37, 31, 22, 14, 9, 7, 8, 11, 13, 15, 14, 12, 10, 9, 10, 13, 18, 24, 31, 38, 44, 47, 47, 43, 34, 24, 13, 5, 3, 7, 17, 31, 46, 59, 67, 69, 65, 58, 51, 45, 42, 42, 43, 43, 40, 33, 24, 16, 11, 12, 19, 28, 37, 42, 45, 50, 57, 62, 58, 37, 2, 0, 1, 123, 374, 745, 1179, 1581, 1849, 1906, 1737, 1390, 959, 547, 233, 49, 0, 0, 17, 45, 58, 58, 48, 31, 9, 0, 0, 51, 179, 384, 642, 901, 1099, 1184, 1131, 955, 703, 439, 217, 70, 0, 0, 1, 17, 21, 15, 6, 1, 3, 11, 21, 29, 31, 30, 26, 21, 17, 14, 10, 6, 3, 1, 2, 9, 21, 39, 59, 78, 91, 94, 83, 60, 28, 0, 0, 14, 81, 191, 326, 458, 554, 589, 554, 459, 331, 201, 96, 29, 0, 0, 0, 7, 8, 5, 0, 0, 0, 0, 5, 13, 23, 33, 44, 53, 58, 58, 55, 47, 38, 28, 21, 16, 15, 18, 24, 32, 40, 45, 47, 44, 38, 32, 28, 28, 32, 39, 44, 45, 40, 29, 17, 6, 0, 0, 3, 8, 13, 16, 16, 14, 13, 11, 11, 11, 12, 12, 11, 10, 10, 11, 14, 17, 19, 15, 5, 0, 0, 17, 71, 158, 264, 365, 436, 455, 417, 333, 228, 126, 49, 6, 0, 1, 16, 27, 30, 26, 17, 8, 2, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 2, 7, 15, 24, 34, 42, 46, 45, 39, 30, 20, 11, 4, 0, 0, 0, 2, 4, 5, 5, 4, 3, 1, 1, 2, 5, 10, 15, 20, 24, 25, 23, 19, 13, 8, 3, 2, 3, 4, 5, 4, 1, 0, 1, 10, 20, 26, 19, 0, 0, 0, 77, 235, 468, 738, 983, 1141, 1165, 1050, 828, 561, 312, 127, 24, 0, 0, 10, 20, 20, 16, 15, 19, 28, 37, 40, 38, 32, 23, 16, 12, 9, 8, 8, 8, 9, 11, 12, 13, 13, 11, 8, 4, 1, 0, 2, 8, 16, 25, 33, 40, 44, 47, 46, 41, 29, 10, 0, 0, 25, 107, 241, 410, 578, 706, 759, 721, 605, 442, 275, 137, 47, 3, 0, 0, 1, 8, 14, 20, 20, 9, 0, 0, 28, 152, 377, 686, 1026, 1317, 1483, 1476, 1297, 996, 652, 345, 128, 14, 0, 0, 16, 24, 17, 3, 0, 0, 8, 24, 41, 53, 61, 65, 64, 59, 50, 37, 24, 11, 2, 0, 0, 5, 14, 24, 31, 35, 34, 29, 21, 13, 5, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 6, 14, 21, 27, 30, 29, 28, 30, 37, 46, 52, 47, 27, 0, 0, 14, 116, 301, 553, 824, 1052, 1175, 1160, 1011, 771, 502, 265, 99, 12, 0, 0, 11, 17, 13, 3, 0, 0, 0, 2, 8, 15, 24, 35, 47, 58, 64, 64, 57, 45, 30, 16, 5, 0, 0, 2, 10, 23, 37, 49, 58, 60, 56, 46, 34, 22, 12, 5, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 1, 2, 3, 3, 2, 2, 2, 4, 8, 13, 18, 20, 19, 16, 12, 7, 3, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 2, 9, 19, 30, 39, 43, 41, 34, 23, 12, 4, 0, 0, 4, 11, 16, 17, 10, 0, 0, 2, 44, 124, 238, 368, 484, 557, 568, 514, 410, 284, 164, 73, 19, 0, 0, 2, 7, 6, 2, 0, 0, 0, 2, 5, 6, 4, 1, 0, 0, 0, 3, 6, 5, 0, 0, 1, 30, 91, 182, 291, 395, 467, 486, 447, 361, 250, 143, 60, 11, 0, 0, 6, 15, 19, 19, 18, 17, 17, 17, 18, 19, 23, 29, 36, 43, 46, 45, 39, 29, 18, 9, 2, 0, 0, 0, 1, 2, 2, 1, 1, 0, 0, 0, 2, 6, 10, 13, 15, 14, 12, 8, 4, 1, 0, 0, 0, 0, 2, 4, 8, 12, 17, 20, 22, 22, 20, 18, 16, 15, 15, 16, 16, 15, 13, 10, 7, 4, 4, 7, 14, 25, 38, 50, 60, 64, 62, 53, 41, 27, 15, 7, 5, 8, 18, 33, 49, 62, 68, 62, 44, 18, 0, 0, 43, 137, 277, 443, 599, 710, 744, 690, 556, 374, 188, 50, 0, 60, 226, 466, 726, 946, 1069, 1068, 945, 736, 493, 271, 108, 17, 0, 0, 22, 40, 45, 40, 30, 20, 12, 7, 4, 2, 2, 2, 3, 4, 6, 7, 7, 8, 9, 11, 15, 21, 27, 33, 36, 35, 31, 24, 15, 7, 1, 0, 0, 3, 9, 15, 19, 21, 21, 22, 23, 25, 25, 21, 15, 9, 8, 15, 26, 32, 24, 1, 0, 0, 101, 314, 633, 1010, 1362, 1599, 1654, 1512, 1213, 839, 481, 208, 48, 0, 0, 11, 25, 23, 10, 0, 0, 0, 0, 8, 15, 20, 23, 26, 26, 25, 22, 18, 13, 10, 9, 10, 13, 17, 20, 22, 22, 21, 18, 15, 11, 7, 3, 1, 0, 1, 3, 6, 9, 12, 13, 13, 11, 9, 8, 8, 10, 13, 14, 14, 11, 7, 2, 0, 1, 5, 8, 10, 9, 7, 5, 6, 9, 12, 10, 2, 0, 0, 31, 108, 230, 381, 531, 645, 693, 664, 567, 428, 280, 153, 64, 15, 0, 0, 0, 3, 5, 5, 4, 0, 0, 0, 10, 55, 136, 250, 379, 494, 568, 581, 531, 432, 311, 194, 102, 43, 15, 7, 8, 10, 8, 4, 0, 0, 0, 0, 1, 2, 2, 1, 0, 0, 0, 2, 4, 6, 8, 8, 6, 4, 3, 2, 3, 6, 10, 14, 16, 15, 12, 8, 4, 1, 0, 0, 0, 3, 5, 8, 10, 13, 15, 18, 22, 25, 27, 27, 24, 20, 16, 14, 16, 24, 35, 46, 54, 56, 52, 44, 36, 28, 21, 13, 4, 0, 0, 32, 106, 226, 383, 549, 689, 764, 750, 645, 471, 273, 105, 17, 42, 187, 428, 713, 979, 1166, 1230, 1161, 980, 733, 476, 257, 103, 19, 0, 0, 3, 16, 22, 18, 5, 0, 0, 28, 139, 336, 603, 897, 1153, 1307, 1317, 1181, 935, 642, 369, 162, 41, 0, 0, 7, 20, 23, 19, 15, 15, 18, 21, 23, 20, 16, 11, 8, 7, 8, 9, 7, 3, 0, 0, 10, 46, 110, 201, 306, 408, 483, 512, 489, 418, 317, 209, 115, 48, 11, 0, 0, 1, 4, 4, 5, 8, 12, 18, 22, 24, 23, 21, 20, 21, 24, 30, 36, 43, 49, 54, 55, 50, 40, 26, 13, 6, 8, 16, 20, 14, 0, 0, 18, 122, 327, 625, 976, 1304, 1528, 1585, 1459, 1181, 827, 480, 211, 49, 0, 0, 11, 26, 25, 12, 1, 0, 10, 25, 37, 39, 32, 20, 8, 1, 1, 6, 9, 7, 0, 0, 5, 48, 132, 256, 400, 537, 633, 667, 628, 528, 393, 252, 133, 51, 8, 0, 0, 5, 8, 8, 7, 7, 8, 10, 11, 9, 6, 3, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 4, 10, 16, 21, 22, 19, 14, 7, 2, 0, 0, 1, 4, 6, 9, 11, 13, 14, 13, 11, 8, 6, 5, 7, 10, 15, 19, 21, 21, 19, 16, 13, 10, 8, 7, 5, 4, 2, 1, 0, 0, 0, 0, 2, 7, 14, 21, 28, 33, 34, 31, 25, 18, 12, 9, 9, 10, 10, 5, 0, 0, 15, 61, 143, 258, 388, 509, 589, 605, 546, 424, 267, 123, 43, 67, 215, 474, 798, 1118, 1359, 1463, 1406, 1204, 910, 594, 318, 125, 22, 0, 0, 11, 18, 13, 3, 0, 0, 0, 7, 13, 14, 13, 10, 7, 7, 9, 9, 7, 2, 0, 0, 19, 65, 139, 232, 327, 404, 442, 432, 378, 293, 197, 111, 48, 11, 0, 0, 4, 11, 15, 17, 18, 18, 17, 14, 10, 5, 2, 1, 4, 10, 17, 23, 23, 17, 6, 0, 5, 40, 114, 231, 380, 535, 663, 730, 713, 607, 435, 241, 82, 12, 66, 250, 536, 866, 1167, 1369, 1428, 1333, 1110, 814, 513, 261, 90, 3, 0, 0, 25, 42, 42, 25, 0, 0, 2, 76, 225, 448, 719, 991, 1205, 1310, 1278, 1116, 864, 580, 324, 137, 30, 0, 0, 8, 17, 15, 6, 0, 0, 0, 2, 6, 7, 5, 2, 0, 0, 0, 1, 4, 7, 9, 11, 10, 7, 3, 0, 0, 0, 3, 7, 6, 0, 0, 0, 27, 89, 186, 311, 442, 554, 620, 626, 570, 466, 338, 213, 110, 41, 7, 0, 9, 25, 39, 44, 36, 15, 0, 0, 30, 147, 349, 620, 915, 1171, 1323, 1333, 1196, 951, 658, 382, 172, 46, 0, 0, 4, 14, 15, 7, 0, 0, 0, 0, 8, 19, 33, 48, 62, 70, 71, 64, 50, 33, 17, 5, 0, 0, 1, 8, 16, 24, 30, 31, 28, 20, 11, 3, 0, 0, 5, 13, 21, 26, 27, 25, 20, 15, 10, 4, 0, 0, 0, 0, 11, 21, 21, 7, 0, 0, 50, 197, 444, 768, 1117, 1415, 1594, 1609, 1459, 1183, 845, 519, 258, 89, 8, 0, 0, 15, 23, 22, 18, 15, 16, 19, 21, 19, 15, 11, 7, 5, 4, 4, 3, 2, 1, 1, 1, 1, 0, 0, 0, 0, 1, 4, 6, 4, 0, 0, 3, 32, 88, 170, 269, 364, 435, 463, 441, 376, 283, 185, 100, 40, 8, 0, 0, 2, 4, 4, 1, 0, 1, 3, 5, 5, 4, 2, 0, 0, 1, 3, 5, 6, 6, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 4, 6, 6, 5, 3, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 4, 10, 17, 23, 26, 26, 23, 17, 11, 6, 3, 2, 2, 3, 3, 3, 4, 4, 4, 3, 3, 2, 2, 2, 4, 7, 10, 13, 16, 18, 21, 23, 26, 27, 28, 27, 24, 20, 16, 13, 11, 12, 15, 18, 22, 24, 23, 20, 15, 9, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 4, 5, 7, 7, 6, 5, 3, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 4, 9, 14, 19, 21, 20, 16, 10, 4, 0, 0, 0, 3, 6, 10, 12, 12, 10, 7, 4, 1, 0, 0, 0, 1, 2, 4, 5, 7, 10, 13, 16, 19, 20, 19, 15, 10, 5, 1, 0, 0, 3, 7, 11, 13, 12, 9, 5, 1, 0, 0, 2, 5, 8, 9, 9, 7, 6, 5, 5, 6, 8, 10, 13, 16, 21, 26, 33, 40, 46, 49, 49, 46, 41, 33, 25, 16, 9, 5, 3, 3, 4, 4, 2, 1, 5, 23, 62, 128, 218, 320, 416, 484, 504, 468, 381, 264, 145, 56, 22, 56, 152, 292, 444, 576, 657, 672, 617, 508, 370, 232, 119, 43, 5, 0, 0, 8, 11, 10, 7, 7, 9, 13, 16, 15, 10, 4, 0, 0, 2, 4, 2, 0, 0, 12, 57, 140, 259, 400, 537, 642, 691, 674, 595, 473, 334, 204, 103, 38, 6, 0, 0, 3, 4, 2, 0, 0, 0, 0, 2, 2, 1, 0, 0, 1, 3, 5, 7, 7, 7, 6, 6, 9, 15, 22, 28, 33, 34, 33, 31, 29, 29, 30, 30, 28, 23, 16, 9, 3, 1, 3, 6, 9, 11, 11, 9, 8, 7, 9, 10, 11, 11, 8, 5, 2, 0, 0, 0, 0, 0, 0, 0, 2, 7, 14, 22, 30, 36, 39, 40, 39, 37, 34, 31, 28, 25, 22, 21, 20, 21, 22, 22, 21, 18, 14, 12, 10, 9, 5, 0, 0, 2, 30, 89, 179, 293, 411, 508, 562, 557, 496, 393, 271, 158, 71, 18, 0, 0, 7, 19, 28, 33, 35, 37, 38, 37, 33, 27, 20, 12, 6, 3, 2, 3, 6, 8, 9, 9, 8, 7, 6, 6, 7, 7, 7, 6, 4, 2, 1, 0, 0, 0, 0, 1, 3, 5, 8, 11, 14, 16, 17, 16, 14, 12, 10, 8, 7, 5, 3, 2, 4, 8, 14, 20, 24, 23, 19, 16, 17, 22, 27, 24, 10, 0, 0, 40, 170, 396, 703, 1043, 1350, 1557, 1619, 1525, 1301, 1000, 684, 407, 200, 73, 13, 0, 2, 10, 14, 12, 9, 7, 9, 14, 19, 23, 24, 23, 18, 13, 7, 2, 0, 0, 1, 7, 15, 24, 33, 39, 40, 36, 28, 18, 9, 2, 0, 0, 0, 1, 1, 0, 0, 0, 2, 8, 15, 21, 24, 24, 20, 13, 6, 1, 0, 0, 1, 4, 7, 11, 15, 18, 20, 20, 18, 16, 15, 14, 11, 6, 0, 0, 2, 32, 93, 185, 298, 414, 509, 561, 560, 505, 411, 298, 187, 97, 37, 6, 0, 0, 3, 5, 2, 0, 0, 9, 39, 93, 173, 272, 378, 475, 544, 569, 544, 474, 372, 258, 153, 72, 21, 0, 0, 8, 21, 30, 34, 33, 28, 22, 15, 9, 4, 2, 0, 0, 0, 1, 1, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 4, 6, 8, 9, 10, 12, 15, 21, 28, 37, 45, 51, 54, 54, 51, 45, 36, 27, 17, 8, 2, 0, 0, 0, 0, 0, 3, 17, 46, 97, 169, 255, 342, 413, 454, 455, 417, 350, 267, 185, 115, 64, 32, 16, 10, 9, 10, 10, 9, 8, 6, 4, 3, 1, 0, 0, 0, 0, 3, 8, 13, 20, 25, 29, 31, 32, 33, 33, 34, 34, 31, 22, 11, 0, 0, 20, 76, 171, 303, 457, 607, 724, 784, 772, 690, 556, 399, 248, 128, 48, 8, 0, 0, 4, 6, 4, 1, 0, 0, 0, 1, 4, 6, 8, 10, 12, 14, 16, 17, 18, 18, 17, 15, 14, 13, 15, 19, 22, 24, 23, 19, 15, 18, 37, 76, 137, 213, 291, 354, 386, 378, 332, 259, 177, 101, 45, 12, 0, 0, 2, 8, 13, 20, 28, 36, 43, 45, 43, 35, 25, 15, 6, 1, 0, 0, 0, 2, 3, 3, 4, 4, 4, 4, 3, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0, 2, 4, 6, 7, 7, 5, 3, 1, 0, 0, 0, 0, 0, 0, 2, 4, 7, 9, 11, 11, 9, 7, 6, 5, 6, 8, 9, 9, 6, 1, 0, 2, 21, 65, 137, 235, 347, 456, 541, 583, 571, 507, 405, 285, 172, 83, 26, 0, 0, 0, 4, 4, 1, 0, 0, 4, 12, 20, 24, 23, 18, 10, 4, 0, 0, 0, 0, 1, 0, 0, 0, 3, 9, 18, 29, 40, 49, 55, 56, 53, 46, 36, 25, 15, 8, 5, 6, 10, 16, 21, 25, 26, 24, 20, 14, 8, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 3, 3, 3, 4, 4, 5, 5, 5, 6, 6, 6, 6, 6, 6, 5, 3, 2, 0, 0, 0, 0, 1, 2, 3, 4, 3, 3, 5, 9, 15, 24, 31, 37, 38, 35, 29, 21, 14, 9, 6, 6, 7, 9, 10, 11, 10, 10, 9, 8, 8, 8, 6, 4, 2, 1, 3, 8, 13, 15, 9, 0, 0, 8, 66, 182, 353, 561, 770, 938, 1030, 1024, 926, 757, 556, 359, 197, 85, 24, 0, 0, 1, 3, 2, 0, 0, 0, 0, 3, 5, 5, 6, 9, 12, 17, 22, 25, 24, 20, 14, 8, 2, 0, 0, 1, 5, 10, 15, 19, 21, 20, 17, 11, 6, 1, 0, 0, 1, 1, 0, 0, 0, 6, 26, 62, 113, 174, 237, 289, 321, 324, 298, 249, 185, 120, 65, 26, 4, 0, 0, 5, 14, 24, 34, 42, 48, 50, 46, 38, 26, 15, 6, 0, 0, 0, 5, 12, 19, 25, 27, 26, 23, 20, 18, 19, 23, 27, 28, 24, 15, 10, 22, 69, 169, 328, 540, 779, 1003, 1168, 1235, 1188, 1035, 811, 562, 335, 163, 56, 6, 0, 0, 4, 5, 2, 0, 4, 11, 20, 25, 26, 21, 14, 7, 3, 2, 3, 6, 10, 13, 14, 13, 10, 5, 2, 0, 0, 0, 1, 0, 0, 0, 12, 42, 97, 176, 272, 370, 455, 509, 522, 492, 423, 330, 230, 139, 69, 25, 3, 0, 0, 2, 4, 4, 4, 4, 5, 7, 9, 12, 14, 17, 21, 25, 29, 33, 36, 36, 34, 29, 22, 14, 9, 8, 12, 16, 16, 8, 0, 0, 16, 89, 223, 419, 656, 899, 1105, 1238, 1274, 1209, 1058, 848, 616, 397, 218, 94, 23, 0, 0, 3, 8, 5, 0, 0, 10, 56, 141, 267, 422, 588, 736, 840, 878, 842, 738, 587, 417, 259, 133, 50, 8, 0, 0, 4, 9, 10, 10, 10, 12, 14, 16, 15, 14, 12, 11, 10, 10, 10, 10, 10, 10, 11, 11, 11, 9, 6, 3, 1, 3, 7, 13, 19, 22, 22, 18, 12, 6, 1, 0, 0, 1, 3, 4, 6, 7, 9, 10, 10, 9, 6, 3, 1, 0, 0, 3, 12, 26, 50, 84, 130, 184, 242, 294, 331, 342, 324, 279, 217, 155, 110, 95, 120, 183, 273, 374, 466, 531, 556, 536, 476, 386, 284, 185, 103, 46, 13, 0, 0, 0, 2, 2, 1, 0, 0, 0, 0, 2, 3, 3, 3, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 3, 5, 8, 11, 14, 16, 16, 12, 6, 0, 0, 6, 35, 88, 165, 256, 349, 427, 476, 488, 461, 400, 317, 227, 143, 76, 31, 6, 0, 0, 2, 4, 3, 2, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 4, 4, 5, 7, 9, 11, 12, 12, 11, 9, 8, 6, 5, 4, 2, 1, 1, 3, 6, 10, 12, 14, 13, 10, 7, 4, 3, 4, 6, 10, 14, 18, 21, 23, 25, 26, 28, 28, 27, 23, 19, 15, 13, 12, 10, 6, 0, 0, 6, 47, 131, 264, 439, 633, 817, 958, 1029, 1018, 927, 776, 592, 407, 245, 123, 46, 8, 0, 0, 8, 15, 19, 18, 16, 13, 11, 11, 14, 21, 30, 40, 48, 52, 50, 45, 42, 45, 63, 98, 151, 217, 287, 350, 395, 413, 401, 361, 301, 230, 160, 99, 54, 26, 13, 10, 11, 10, 5, 0, 0, 14, 61, 148, 273, 425, 584, 722, 814, 845, 809, 716, 584, 436, 294, 176, 90, 36, 9, 0, 0, 0, 2, 3, 4, 6, 8, 10, 12, 12, 12, 11, 10, 9, 8, 8, 8, 9, 12, 15, 20, 27, 33, 38, 39, 35, 26, 15, 8, 16, 48, 117, 227, 375, 546, 716, 857, 942, 953, 889, 761, 596, 423, 268, 149, 70, 26, 8, 2, 2, 2, 0, 0, 0, 0, 2, 8, 16, 24, 31, 36, 36, 33, 26, 18, 12, 8, 8, 12, 20, 29, 38, 44, 47, 46, 41, 35, 28, 22, 16, 12, 9, 6, 3, 1, 0, 0, 0, 1, 2, 2, 1, 0, 0, 2, 6, 12, 17, 19, 17, 13, 11, 17, 36, 71, 121, 182, 244, 298, 336, 350, 340, 307, 257, 198, 140, 89, 51, 28, 18, 17, 19, 21, 19, 15, 8, 3, 0, 0, 1, 4, 6, 7, 6, 5, 4, 4, 4, 5, 5, 5, 4, 3, 1, 0, 0, 0, 0, 0, 1, 3, 4, 4, 4, 2, 1, 0, 0, 1, 4, 8, 11, 12, 11, 8, 4, 1, 0, 0, 2, 5, 7, 7, 6, 4, 3, 3, 4, 7, 10, 12, 11, 9, 5, 1, 0, 0, 3, 9, 15, 19, 20, 18, 13, 8, 3, 0, 0, 0, 2, 4, 6, 8, 10, 12, 14, 17, 18, 19, 18, 17, 15, 13, 12, 12, 12, 13, 13, 14, 14, 14, 14, 13, 13, 11, 9, 6, 4, 2, 0, 0, 0, 1, 2, 3, 3, 3, 3, 2, 2, 1, 0, 0, 0, 0, 2, 5, 8, 11, 13, 14, 15, 15, 15, 14, 11, 8, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 2, 6, 12, 18, 22, 23, 22, 19, 15, 12, 11, 11, 12, 13, 14, 13, 12, 11, 10, 10, 10, 10, 9, 8, 7, 5, 4, 3, 3, 5, 6, 6, 6, 5, 3, 1, 1, 1, 1, 2, 3, 4, 4, 3, 1, 0, 0, 5, 26, 70, 144, 250, 382, 526, 662, 768, 823, 816, 746, 627, 479, 329, 198, 100, 38, 8, 0, 0, 0, 1, 1, 2, 4, 9, 13, 17, 20, 21, 22, 23, 25, 26, 26, 22, 16, 9, 3, 0, 0, 4, 10, 15, 19, 20, 19, 16, 13, 11, 9, 8, 6, 5, 4, 4, 6, 10, 16, 21, 26, 29, 30, 29, 28, 26, 23, 18, 12, 4, 0, 3, 20, 55, 109, 178, 254, 326, 383, 415, 416, 387, 332, 260, 182, 111, 54, 17, 1, 0, 8, 18, 25, 26, 21, 14, 8, 5, 5, 8, 11, 14, 16, 16, 15, 14, 13, 13, 11, 10, 7, 5, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 8, 12, 14, 16, 16, 15, 14, 12, 11, 10, 10, 9, 8, 7, 7, 8, 10, 12, 13, 13, 12, 10, 8, 5, 2, 0, 0, 0, 2, 5, 9, 15, 22, 32, 44, 60, 81, 109, 145, 188, 235, 280, 314, 327, 316, 282, 237, 198, 184, 210, 279, 386, 511, 631, 723, 769, 762, 705, 609, 489, 363, 245, 147, 75, 29, 6, 0, 0, 1, 3, 3, 1, 0, 1, 4, 10, 18, 26, 33, 38, 39, 36, 31, 24, 17, 11, 6, 3, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 3, 7, 11, 17, 21, 23, 23, 20, 14, 8, 4, 1, 2, 4, 9, 13, 16, 19, 19, 18, 17, 15, 13, 11, 10, 8, 6, 4, 2, 1, 0, 0, 0, 1, 2, 2, 2, 1, 0, 0, 0, 3, 12, 29, 59, 101, 154, 214, 273, 323, 354, 360, 341, 301, 245, 184, 127, 78, 42, 19, 6, 0, 0, 0, 1, 1, 1, 1, 1, 1, 2, 4, 5, 6, 6, 7, 12, 26, 57, 107, 180, 272, 373, 472, 555, 608, 623, 600, 541, 456, 357, 257, 168, 96, 46, 17, 5, 3, 4, 5, 4, 1, 0, 0, 3, 9, 15, 19, 20, 18, 15, 12, 10, 12, 15, 20, 25, 29, 33, 37, 41, 45, 49, 50, 49, 44, 37, 28, 19, 12, 7, 4, 4, 4, 5, 6, 5, 4, 3, 2, 1, 1, 3, 4, 7, 9, 11, 13, 13, 12, 11, 8, 5, 2, 1, 1, 2, 5, 7, 9, 11, 11, 12, 12, 14, 15, 16, 16, 13, 8, 3, 0, 1, 12, 37, 76, 129, 194, 264, 332, 388, 423, 430, 407, 360, 300, 246, 218, 229, 283, 375, 483, 586, 658, 685, 662, 597, 507, 413, 337, 295, 293, 328, 390, 461, 524, 562, 566, 534, 470, 384, 290, 199, 123, 65, 28, 8, 0, 0, 0, 1, 3, 5, 8, 10, 11, 11, 9, 6, 4, 2, 0, 0, 6, 22, 58, 122, 217, 340, 479, 613, 720, 781, 785, 734, 640, 521, 396, 281, 187, 116, 67, 36, 17, 8, 5, 6, 10, 15, 19, 22, 24, 25, 25, 24, 23, 21, 19, 18, 16, 14, 11, 8, 5, 3, 1, 0, 0, 0, 0, 1, 2, 3, 3, 4, 4, 5, 6, 6, 7, 5, 2, 0, 0, 8, 34, 82, 152, 240, 336, 426, 497, 538, 543, 511, 448, 366, 276, 192, 121, 68, 33, 13, 4, 1, 0, 1, 1, 2, 2, 2, 2, 3, 4, 4, 5, 5, 5, 3, 2, 1, 1, 1, 2, 3, 3, 2, 2, 0, 0, 0, 0, 0, 1, 2, 2, 2, 1, 0, 0, 0, 1, 2, 3, 3, 3, 2, 1, 0, 0, 0, 0, 0, 1, 2, 3, 6, 10, 15, 20, 25, 28, 28, 26, 22, 16, 11, 6, 3, 0, 0, 0, 0, 1, 2, 3, 4, 3, 2, 1, 0, 0, 0, 1, 3, 6, 8, 10, 12, 13, 13, 13, 12, 11, 9, 9, 9, 11, 12, 12, 10, 8, 10, 18, 38, 73, 123, 184, 247, 303, 342, 357, 349, 320, 279, 237, 203, 186, 189, 210, 246, 288, 330, 364, 383, 383, 366, 336, 299, 268, 250, 254, 280, 325, 378, 429, 465, 479, 467, 428, 370, 298, 224, 153, 93, 48, 19, 3, 0, 1, 5, 8, 10, 9, 6, 5, 5, 7, 11, 17, 24, 30, 36, 40, 43, 43, 42, 39, 35, 30, 25, 21, 18, 16, 15, 15, 14, 13, 10, 7, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 2, 8, 23, 49, 86, 133, 183, 229, 265, 283, 283, 264, 230, 187, 142, 100, 65, 39, 21, 11, 5, 1, 0, 0, 0, 7, 21, 44, 75, 113, 153, 191, 222, 241, 246, 234, 208, 171, 128, 87, 52, 27, 12, 6, 5, 6, 9, 10, 10, 9, 7, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 3, 3, 2, 1, 0, 0, 0, 1, 3, 5, 6, 6, 5, 5, 7, 12, 21, 33, 50, 72, 102, 143, 193, 250, 307, 355, 384, 391, 376, 347, 318, 301, 309, 346, 411, 495, 587, 671, 736, 769, 764, 720, 640, 535, 417, 301, 197, 115, 58, 22, 5, 0, 0, 4, 7, 8, 8, 7, 6, 5, 5, 7, 9, 12, 14, 15, 15, 13, 12, 11, 11, 12, 13, 14, 14, 13, 13, 18, 34, 67, 122, 202, 306, 424, 545, 651, 728, 765, 758, 709, 627, 524, 412, 304, 209, 133, 78, 43, 23, 14, 11, 10, 10, 9, 9, 9, 11, 13, 16, 17, 16, 12, 8, 4, 4, 13, 34, 68, 115, 172, 233, 292, 343, 378, 395, 390, 365, 322, 267, 206, 146, 93, 52, 23, 7, 0, 0, 0, 0, 1, 1, 1, 2, 3, 3, 3, 2, 1, 1, 2, 7, 17, 32, 53, 82, 118, 158, 199, 238, 268, 286, 288, 274, 244, 202, 154, 106, 65, 33, 12, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 8, 14, 20, 25, 27, 28, 28, 27, 26, 24, 22, 18, 14, 8, 4, 0, 0, 0, 2, 4, 6, 5, 2, 0, 2, 12, 35, 74, 131, 203, 284, 366, 439, 495, 527, 531, 507, 459, 392, 315, 237, 164, 104, 58, 27, 10, 3, 2, 5, 8, 11, 12, 12, 11, 11, 13, 15, 17, 18, 17, 14, 9, 4, 1, 0, 2, 5, 9, 10, 10, 10, 10, 15, 28, 49, 80, 118, 161, 205, 246, 280, 304, 317, 318, 309, 295, 283, 278, 288, 316, 361, 418, 481, 538, 580, 600, 592, 556, 495, 417, 330, 242, 163, 98, 51, 21, 5, 0, 0, 0, 0, 0, 0, 0, 2, 6, 11, 15, 16, 14, 11, 8, 6, 7, 9, 11, 12, 10, 7, 3, 0, 0, 1, 3, 6, 8, 8, 6, 3, 2, 3, 10, 22, 40, 66, 103, 154, 223, 312, 420, 541, 663, 768, 840, 865, 838, 764, 654, 525, 396, 281, 186, 115, 66, 33, 14, 4, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 4, 8, 13, 16, 18, 16, 11, 6, 3, 3, 8, 18, 35, 59, 96, 149, 224, 317, 423, 527, 612, 664, 673, 639, 570, 480, 383, 290, 208, 140, 87, 48, 22, 6, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 7, 14, 24, 38, 58, 84, 116, 151, 187, 219, 242, 253, 251, 236, 212, 182, 151, 120, 93, 72, 56, 45, 39, 37, 39, 44, 54, 70, 92, 121, 155, 189, 222, 247, 263, 267, 259, 240, 212, 180, 145, 111, 80, 54, 35, 23, 16, 13, 15, 22, 36, 60, 96, 143, 197, 253, 303, 342, 364, 370, 361, 338, 304, 262, 214, 165, 117, 75, 43, 23, 15, 15, 21, 30, 37, 41, 43, 42, 38, 33, 27, 22, 18, 17, 19, 21, 25, 26, 26, 24, 21, 18, 14, 11, 8, 5, 3, 2, 3, 5, 8, 9, 9, 7, 4, 3, 6, 11, 21, 33, 50, 73, 105, 147, 200, 257, 312, 353, 374, 370, 344, 303, 258, 221, 202, 204, 228, 268, 314, 356, 386, 398, 388, 359, 315, 262, 207, 155, 110, 74, 47, 29, 16, 9, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 7, 11, 16, 21, 25, 27, 28, 28, 27, 23, 18, 12, 7, 3, 0, 0, 0, 0, 0, 1, 3, 7, 12, 19, 26, 33, 38, 41, 43, 44, 48, 56, 69, 88, 110, 135, 161, 183, 201, 212, 215, 208, 194, 172, 146, 118, 91, 67, 46, 30, 18, 9, 4, 0, 0, 0, 1, 3, 5, 6, 5, 3, 1, 1, 6, 16, 33, 58, 91, 133, 182, 239, 301, 366, 428, 483, 520, 535, 522, 482, 420, 344, 266, 196, 140, 99, 73, 56, 46, 37, 30, 24, 20, 16, 13, 11, 8, 4, 2, 0, 0, 1, 7, 17, 33, 54, 79, 106, 132, 156, 174, 188, 196, 197, 191, 176, 152, 121, 87, 54, 27, 9, 1, 0, 4, 7, 8, 7, 4, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 3, 8, 15, 22, 28, 32, 35, 36, 34, 32, 28, 24, 20, 16, 13, 11, 10, 9, 8, 8, 8, 10, 13, 16, 19, 20, 19, 16, 12, 8, 5, 4, 6, 13, 31, 63, 116, 191, 286, 393, 503, 604, 687, 744, 770, 765, 729, 664, 577, 475, 370, 270, 184, 116, 68, 37, 19, 9, 3, 0, 0, 0, 0, 1, 3, 6, 8, 9, 10, 8, 6, 4, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 2, 5, 8, 11, 13, 13, 12, 9, 6, 4, 2, 2, 3, 7, 12, 19, 27, 38, 53, 72, 99, 134, 177, 225, 273, 318, 353, 376, 386, 384, 373, 357, 341, 331, 334, 354, 392, 444, 499, 546, 574, 577, 554, 510, 456, 403, 360, 332, 322, 325, 337, 349, 356, 352, 335, 306, 267, 223, 178, 136, 98, 65, 39, 20, 8, 2, 0, 1, 4, 6, 7, 8, 8, 7, 7, 6, 5, 4, 3, 2, 2, 1, 0, 0, 0, 1, 4, 9, 14, 19, 23, 26, 27, 28, 29, 28, 25, 22, 17, 13, 9, 8, 8, 9, 10, 10, 9, 7, 6, 7, 9, 11, 13, 13, 11, 8, 5, 2, 0, 0, 0, 0, 1, 2, 2, 1, 1, 1, 2, 5, 11, 18, 26, 32, 34, 32, 27, 19, 10, 4, 0, 0, 0, 1, 2, 3, 5, 8, 12, 16, 20, 23, 25, 24, 23, 22, 21, 22, 24, 27, 30, 36, 45, 60, 83, 114, 149, 185, 216, 239, 250, 249, 239, 221, 197, 170, 140, 110, 81, 55, 33, 17, 7, 2, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 4, 15, 32, 53, 76, 98, 120, 141, 162, 183, 204, 221, 228, 223, 205, 174, 135, 95, 60, 32, 14, 5, 2, 3, 6, 10, 12, 13, 11, 8, 4, 2, 1, 1, 3, 5, 8, 14, 23, 39, 63, 95, 132, 171, 208, 241, 268, 286, 294, 291, 276, 249, 213, 172, 129, 91, 60, 38, 23, 15, 12, 10, 8, 7, 5, 4, 3, 3, 3, 4, 6, 9, 13, 18, 25, 35, 47, 63, 84, 109, 139, 170, 200, 226, 246, 260, 270, 279, 291, 310, 339, 378, 426, 480, 534, 580, 609, 616, 598, 554, 492, 419, 344, 274, 213, 161, 116, 79, 49, 29, 19, 21, 32, 51, 74, 98, 123, 147, 169, 187, 200, 207, 204, 194, 178, 156, 131, 106, 81, 60, 43, 32, 27, 27, 28, 29, 28, 25, 21, 15, 11, 8, 7, 9, 15, 24, 36, 50, 65, 82, 102, 127, 158, 193, 228, 259, 281, 292, 296, 299, 311, 341, 391, 460, 538, 615, 679, 722, 739, 730, 699, 649, 584, 510, 430, 348, 267, 192, 127, 74, 37, 13, 2, 0, 0, 1, 3, 3, 3, 2, 2, 2, 3, 2, 2, 1, 0, 0, 0, 0, 1, 3, 4, 4, 4, 3, 1, 0, 1, 3, 6, 11, 17, 22, 27, 32, 39, 48, 62, 82, 108, 140, 177, 214, 247, 272, 286, 291, 290, 288, 293, 307, 334, 373, 420, 471, 520, 561, 588, 598, 586, 554, 505, 443, 376, 310, 247, 191, 141, 99, 65, 44, 36, 44, 67, 103, 149, 201, 257, 313, 369, 420, 462, 489, 496, 479, 440, 385, 321, 258, 201, 155, 120, 93, 74, 61, 52, 48, 50, 57, 69, 85, 106, 130, 157, 183, 205, 220, 225, 220, 208, 190, 170, 149, 127, 104, 80, 57, 35, 18, 6, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 3, 4, 4, 4, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 7, 12, 17, 22, 25, 26, 26, 25, 23, 20, 17, 13, 9, 6, 4, 4, 4, 6, 7, 7, 6, 4, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 3, 2, 1, 0, 0, 0, 2, 4, 6, 7, 6, 4, 2, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 3, 5, 7, 9, 10, 12, 13, 15, 18, 23, 31, 42, 57, 77, 101, 126, 150, 169, 181, 185, 182, 174, 164, 153, 140, 125, 108, 88, 67, 48, 32, 22, 16, 14, 12, 11, 9, 7, 6, 6, 8, 10, 11, 13, 14, 16, 18, 19, 20, 20, 18, 16, 15, 13, 13, 13, 11, 9, 7, 4, 2, 2, 4, 6, 8, 9, 9, 8, 7, 7, 9, 13, 17, 21, 23, 24, 23, 20, 16, 12, 8, 5, 3, 2, 2, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 6, 8, 8, 8, 7, 5, 4, 3, 2, 1, 1, 0, 0, 1, 1, 2, 2, 2, 1, 0, 0, 0, 0, 1, 3, 4, 5, 5, 3, 2, 0, 0, 0, 1, 2, 3, 3, 2, 2, 1, 0, 0, 0, 0, 0, 1, 3, 4, 5, 5, 4, 2, 2, 3, 5, 8, 11, 12, 10, 7, 4, 2, 3, 5, 8, 12, 14, 16, 17, 17, 18, 19, 21, 23, 26, 29, 30, 30, 26, 21, 17, 15, 21, 34, 55, 80, 106, 131, 157, 189, 231, 290, 366, 450, 532, 596, 629, 625, 585, 521, 446, 375, 314, 266, 227, 192, 158, 125, 95, 75, 68, 75, 96, 127, 164, 204, 244, 285, 324, 358, 384, 399, 400, 384, 355, 314, 266, 216, 168, 125, 88, 60, 40, 28, 22, 22, 25, 27, 28, 27, 24, 19, 16, 14, 15, 18, 22, 27, 31, 32, 32, 30, 29, 28, 29, 33, 39, 47, 55, 65, 76, 90, 107, 127, 146, 163, 175, 179, 175, 165, 149, 132, 115, 100, 86, 74, 64, 56, 49, 44, 40, 37, 34, 31, 28, 24, 21, 19, 18, 18, 19, 20, 21, 22, 24, 25, 26, 25, 24, 21, 18, 14, 12, 11, 10, 9, 8, 7, 5, 4, 2, 1, 1, 0, 0, 0, 0, 1, 2, 2, 2, 1, 1, 0, 0, 0, 1, 2, 3, 2, 1, 0, 0, 0, 1, 2, 2, 2, 1, 0, 0, 0, 2, 5, 9, 12, 13, 13, 11, 10, 11, 14, 20, 28, 37, 44, 47, 46, 41, 34, 26, 23, 26, 39, 63, 97, 140, 187, 234, 278, 318, 355, 392, 430, 469, 501, 521, 520, 495, 446, 383, 315, 252, 200, 162, 134, 112, 92, 71, 52, 35, 24, 19, 18, 21, 23, 24, 23, 21, 18, 17, 18, 20, 23, 24, 23, 22, 20, 20, 22, 27, 34, 42, 52, 65, 85, 111, 144, 179, 212, 236, 248, 248, 240, 227, 214, 203, 191, 177, 158, 134, 106, 80, 58, 44, 38, 40, 48, 61, 79, 101, 127, 154, 179, 202, 218, 228, 232, 231, 225, 215, 201, 183, 161, 136, 112, 92, 78, 75, 82, 101, 129, 164, 205, 249, 297, 348, 402, 458, 513, 561, 597, 613, 608, 583, 543, 498, 457, 426, 407, 397, 390, 383, 370, 352, 331, 307, 281, 254, 226, 195, 163, 133, 106, 83, 66, 52, 40, 29, 20, 12, 6, 4, 5, 8, 12, 16, 18, 18, 17, 15, 14, 14, 16, 19, 24, 31, 39, 48, 61, 80, 106, 142, 189, 248, 313, 380, 442, 494, 532, 555, 564, 560, 543, 514, 474, 423, 365, 304, 244, 189, 142, 103, 74, 52, 37, 26, 19, 13, 10, 8, 10, 15, 24, 33, 43, 52, 60, 70, 85, 106, 135, 172, 212, 255, 297, 333, 363, 382, 389, 382, 363, 333, 298, 259, 221, 184, 150, 119, 92, 68, 49, 36, 26, 20, 15, 11, 9, 8, 9, 12, 16, 20, 23, 26, 30, 37, 48, 64, 84, 103, 120, 133, 141, 144, 145, 144, 140, 135, 127, 115, 100, 82, 65, 48, 34, 23, 14, 9, 6, 5, 8, 12, 17, 21, 22, 22, 19, 16, 14, 13, 13, 15, 15, 15, 13, 11, 8, 6, 4, 3, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 6, 7, 7, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, 3, 6, 8, 9, 8, 6, 4, 2, 1, 2, 3, 5, 7, 8, 8, 8, 7, 6, 6, 5, 5, 5, 5, 7, 8, 9, 9, 8, 6, 3, 0, 0, 0, 5, 12, 21, 30, 38, 45, 52, 61, 74, 92, 113, 134, 152, 165, 174, 180, 183, 184, 184, 179, 168, 150, 128, 102, 78, 57, 40, 29, 20, 14, 9, 5, 3, 1, 0, 0, 0, 0, 0, 1, 1, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 2, 4, 5, 5, 5, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 5, 6, 6, 4, 1, 0, 0, 2, 5, 7, 8, 7, 5, 2, 0, 0, 0, 2, 4, 6, 8, 9, 9, 9, 9, 10, 11, 13, 14, 15, 15, 14, 12, 11, 12, 17, 26, 41, 58, 76, 94, 111, 128, 145, 164, 184, 203, 220, 233, 240, 240, 235, 224, 206, 183, 155, 125, 97, 73, 56, 46, 41, 38, 35, 29, 21, 13, 5, 0, 0, 0, 1, 3, 3, 3, 2, 2, 2, 3, 5, 7, 9, 10, 9, 7, 6, 5, 6, 8, 11, 14, 16, 17, 15, 13, 11, 9, 9, 9, 10, 11, 11, 10, 9, 8, 9, 11, 15, 21, 28, 37, 47, 60, 74, 90, 107, 124, 141, 157, 169, 178, 181, 179, 172, 160, 144, 126, 109, 93, 80, 67, 56, 46, 36, 29, 23, 20, 18, 17, 16, 14, 12, 9, 7, 6, 6, 6, 6, 4, 3, 1, 1, 1, 3, 5, 7, 8, 8, 9, 10, 13, 16, 19, 22, 24, 27, 33, 43, 55, 68, 80, 91, 101, 114, 135, 167, 209, 256, 303, 340, 363, 369, 362, 344, 319, 289, 258, 224, 191, 159, 130, 107, 88, 73, 60, 46, 32, 18, 8, 1, 0, 1, 4, 6, 7, 6, 4, 4, 6, 10, 15, 18, 19, 16, 12, 9, 9, 16, 29, 45, 63, 79, 94, 108, 124, 143, 167, 192, 214, 230, 238, 236, 228, 217, 204, 191, 177, 160, 142, 122, 105, 94, 91, 96, 107, 123, 141, 161, 184, 211, 240, 270, 298, 321, 335, 341, 339, 329, 314, 294, 268, 238, 206, 173, 143, 116, 94, 77, 63, 53, 44, 36, 29, 23, 18, 15, 13, 12, 12, 12, 12, 12, 11, 11, 10, 10, 10, 9, 7, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 4, 9, 13, 16, 18, 18, 18, 20, 27, 38, 54, 71, 88, 101, 110, 114, 116, 117, 118, 119, 117, 112, 103, 90, 75, 62, 52, 45, 40, 37, 32, 27, 21, 16, 11, 9, 9, 9, 10, 12, 13, 13, 14, 13, 12, 10, 8, 9, 11, 17, 24, 32, 39, 45, 52, 60, 71, 86, 105, 126, 147, 168, 188, 207, 226, 245, 262, 274, 279, 274, 260, 237, 209, 179, 150, 125, 105, 88, 74, 61, 49, 39, 30, 24, 20, 18, 17, 18, 20, 23, 29, 37, 49, 64, 81, 99, 116, 132, 147, 162, 177, 190, 201, 207, 209, 204, 195, 181, 164, 148, 132, 116, 101, 85, 69, 52, 37, 24, 14, 9, 8, 8, 10, 10, 10, 8, 6, 3, 1, 0, 0, 0, 0, 3, 6, 9, 12, 14, 17, 22, 29, 39, 51, 65, 78, 91, 103, 115, 129, 145, 162, 176, 186, 190, 187, 177, 163, 147, 130, 113, 97, 82, 68, 56, 45, 36, 27, 21, 15, 11, 10, 10, 11, 12, 13, 12, 10, 6, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 5, 7, 7, 6, 5, 4, 5, 8, 12, 16, 18, 16, 12, 8, 4, 3, 6, 11, 16, 18, 18, 15, 9, 4, 0, 0, 1, 4, 7, 9, 10, 9, 8, 7, 7, 8, 8, 7, 6, 4, 2, 1, 1, 2, 2, 2, 1, 1, 0, 0, 1, 2, 4, 5, 6, 7, 7, 8, 8, 7, 7, 6, 6, 6, 5, 5, 5, 4, 5, 6, 8, 11, 13, 13, 11, 9, 7, 7, 9, 14, 21, 28, 35, 42, 51, 63, 78, 96, 115, 136, 157, 179, 201, 224, 245, 263, 277, 286, 290, 294, 299, 306, 313, 320, 324, 323, 318, 311, 304, 297, 291, 282, 270, 253, 231, 204, 174, 144, 117, 92, 73, 59, 50, 44, 39, 32, 24, 15, 8, 4, 2, 1))
        self.assertEqual(record.annotations["abif_raw"]['DATA10'], (7, 6, 5, 4, 2, 1, 0, 0, 0, 0, 1, 4, 7, 10, 12, 12, 10, 6, 3, 1, 1, 3, 6, 9, 13, 18, 25, 34, 45, 55, 62, 65, 64, 60, 56, 54, 55, 59, 65, 71, 78, 86, 96, 109, 124, 140, 155, 167, 175, 180, 186, 194, 204, 216, 227, 237, 244, 251, 259, 273, 291, 314, 336, 354, 362, 361, 354, 345, 340, 341, 347, 356, 363, 363, 355, 340, 320, 297, 275, 255, 236, 219, 204, 189, 176, 164, 153, 143, 134, 125, 117, 109, 101, 93, 83, 73, 64, 56, 52, 50, 51, 51, 49, 45, 37, 30, 24, 22, 24, 29, 34, 39, 40, 39, 36, 33, 29, 26, 24, 21, 19, 18, 18, 19, 21, 22, 22, 22, 20, 17, 15, 13, 12, 11, 12, 13, 14, 15, 15, 15, 14, 14, 13, 13, 13, 14, 16, 17, 19, 21, 22, 23, 25, 28, 33, 37, 41, 43, 44, 45, 49, 57, 71, 90, 114, 139, 167, 196, 227, 260, 294, 328, 360, 391, 421, 450, 478, 502, 519, 524, 518, 500, 476, 450, 424, 398, 371, 339, 300, 256, 210, 167, 132, 107, 91, 80, 72, 63, 54, 45, 38, 35, 33, 32, 30, 27, 22, 19, 17, 18, 21, 26, 30, 33, 35, 36, 37, 40, 42, 43, 43, 40, 36, 30, 24, 20, 19, 19, 20, 23, 24, 25, 24, 22, 20, 18, 17, 17, 18, 18, 18, 15, 12, 7, 3, 0, 0, 2, 5, 7, 9, 8, 6, 3, 2, 2, 3, 6, 8, 10, 10, 9, 6, 4, 2, 3, 6, 12, 20, 31, 45, 64, 87, 115, 147, 181, 219, 259, 303, 349, 396, 441, 479, 504, 515, 508, 486, 452, 409, 361, 310, 261, 213, 170, 132, 100, 73, 52, 36, 25, 18, 14, 13, 13, 14, 16, 19, 23, 28, 32, 34, 32, 28, 21, 14, 9, 7, 8, 10, 12, 13, 12, 10, 6, 3, 1, 0, 1, 4, 10, 18, 28, 39, 52, 67, 87, 113, 145, 182, 221, 260, 294, 322, 346, 363, 374, 373, 359, 328, 284, 231, 178, 133, 100, 78, 63, 52, 40, 26, 14, 7, 5, 9, 15, 21, 23, 22, 19, 15, 12, 10, 9, 8, 6, 3, 1, 0, 0, 1, 1, 1, 0, 0, 1, 3, 5, 7, 7, 6, 4, 1, 0, 0, 0, 2, 4, 6, 8, 10, 13, 15, 16, 14, 10, 5, 0, 0, 2, 11, 26, 45, 68, 94, 124, 160, 206, 262, 332, 411, 497, 582, 656, 710, 736, 729, 690, 624, 541, 450, 359, 275, 201, 139, 88, 49, 22, 7, 0, 0, 2, 7, 14, 22, 32, 41, 48, 50, 47, 40, 30, 21, 14, 9, 7, 6, 3, 1, 0, 0, 3, 10, 17, 23, 26, 24, 19, 12, 6, 3, 1, 2, 3, 4, 6, 10, 19, 34, 55, 82, 115, 152, 194, 241, 292, 341, 383, 412, 422, 411, 380, 334, 281, 227, 177, 134, 98, 69, 46, 31, 26, 35, 59, 101, 160, 234, 316, 399, 474, 533, 567, 572, 545, 493, 423, 344, 268, 200, 144, 100, 67, 42, 24, 11, 3, 0, 0, 0, 0, 4, 9, 15, 22, 26, 26, 23, 18, 13, 12, 16, 23, 33, 42, 49, 51, 50, 47, 45, 44, 45, 46, 47, 45, 41, 35, 29, 22, 17, 13, 11, 8, 6, 4, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 11, 31, 65, 113, 170, 232, 289, 333, 358, 362, 344, 308, 259, 202, 145, 92, 51, 25, 16, 26, 51, 89, 135, 187, 241, 295, 346, 387, 413, 417, 396, 350, 286, 214, 146, 92, 56, 37, 31, 28, 25, 17, 7, 0, 2, 19, 52, 104, 177, 270, 384, 518, 662, 804, 924, 1001, 1021, 975, 868, 717, 543, 373, 227, 118, 48, 12, 0, 0, 0, 2, 1, 0, 0, 0, 0, 2, 2, 2, 1, 0, 0, 0, 3, 6, 10, 12, 13, 12, 8, 5, 1, 0, 0, 2, 5, 7, 8, 7, 5, 3, 1, 0, 0, 0, 0, 2, 4, 7, 10, 13, 17, 22, 26, 30, 32, 31, 26, 19, 11, 4, 1, 1, 2, 5, 6, 5, 4, 2, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 5, 7, 7, 6, 4, 1, 0, 0, 1, 3, 4, 5, 4, 2, 1, 0, 0, 0, 2, 4, 5, 6, 5, 3, 1, 0, 0, 1, 5, 10, 12, 11, 6, 0, 1, 14, 46, 99, 170, 251, 332, 403, 454, 479, 476, 443, 386, 311, 229, 150, 84, 37, 10, 0, 0, 1, 3, 2, 0, 0, 0, 0, 4, 8, 14, 21, 35, 60, 101, 162, 239, 326, 410, 477, 515, 514, 476, 408, 320, 229, 147, 82, 38, 13, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 6, 9, 12, 12, 11, 8, 4, 1, 0, 0, 0, 0, 0, 2, 4, 7, 10, 12, 13, 14, 14, 14, 15, 15, 14, 11, 8, 5, 2, 0, 0, 0, 0, 0, 0, 1, 5, 8, 11, 13, 12, 10, 8, 8, 12, 18, 25, 30, 33, 32, 28, 22, 16, 11, 8, 5, 3, 1, 0, 0, 0, 2, 6, 11, 17, 23, 29, 34, 36, 36, 33, 27, 19, 11, 5, 1, 0, 0, 0, 0, 0, 0, 2, 4, 8, 12, 15, 17, 18, 19, 19, 19, 18, 17, 17, 16, 17, 20, 24, 28, 33, 36, 37, 37, 36, 34, 32, 28, 24, 19, 14, 11, 11, 15, 20, 25, 28, 26, 20, 11, 3, 0, 0, 3, 4, 1, 0, 0, 29, 101, 229, 413, 636, 868, 1068, 1198, 1232, 1163, 1004, 788, 554, 341, 176, 69, 14, 0, 0, 3, 6, 7, 8, 13, 21, 31, 39, 44, 45, 44, 42, 41, 43, 46, 49, 51, 51, 48, 43, 38, 34, 32, 28, 23, 15, 8, 11, 36, 97, 201, 345, 514, 683, 820, 899, 905, 838, 713, 556, 394, 250, 138, 63, 21, 5, 5, 14, 25, 35, 41, 43, 42, 41, 41, 45, 51, 58, 63, 64, 60, 53, 44, 36, 32, 32, 34, 37, 37, 33, 27, 26, 38, 74, 142, 243, 369, 503, 621, 702, 728, 693, 605, 480, 344, 217, 117, 50, 13, 0, 0, 1, 4, 5, 4, 3, 2, 2, 2, 3, 5, 9, 13, 17, 21, 23, 23, 24, 26, 28, 31, 34, 34, 32, 29, 25, 23, 23, 27, 33, 39, 42, 43, 40, 34, 26, 17, 10, 6, 7, 12, 17, 16, 7, 0, 0, 28, 122, 292, 536, 825, 1114, 1347, 1476, 1471, 1335, 1097, 806, 519, 279, 114, 25, 0, 0, 6, 12, 8, 0, 0, 0, 7, 23, 40, 53, 60, 60, 54, 45, 37, 32, 30, 29, 29, 26, 21, 15, 11, 10, 11, 12, 10, 2, 0, 0, 28, 98, 212, 357, 512, 647, 733, 754, 708, 608, 475, 334, 208, 111, 46, 12, 0, 0, 1, 3, 4, 4, 6, 9, 14, 20, 25, 28, 30, 31, 31, 32, 34, 38, 43, 47, 50, 49, 45, 38, 29, 21, 16, 13, 15, 18, 23, 27, 31, 34, 35, 35, 34, 33, 33, 33, 35, 36, 36, 33, 28, 20, 12, 6, 5, 7, 9, 9, 3, 0, 0, 26, 94, 211, 370, 549, 714, 832, 875, 832, 713, 545, 363, 201, 83, 17, 0, 0, 10, 11, 1, 0, 8, 70, 189, 360, 555, 735, 858, 892, 830, 688, 500, 311, 155, 53, 3, 0, 0, 9, 12, 8, 1, 0, 0, 0, 1, 3, 4, 6, 9, 13, 17, 19, 19, 16, 12, 7, 2, 0, 0, 0, 4, 11, 18, 19, 12, 0, 0, 11, 67, 171, 317, 482, 632, 730, 748, 684, 554, 398, 264, 191, 205, 301, 454, 618, 746, 799, 764, 650, 487, 314, 166, 63, 9, 0, 0, 9, 15, 14, 10, 6, 3, 2, 2, 3, 3, 4, 5, 6, 7, 6, 5, 4, 2, 0, 0, 0, 3, 11, 21, 33, 42, 46, 43, 35, 23, 13, 7, 5, 8, 13, 18, 22, 24, 23, 21, 17, 12, 6, 2, 0, 2, 7, 13, 16, 17, 15, 12, 12, 14, 15, 11, 1, 0, 0, 51, 163, 340, 565, 796, 980, 1071, 1042, 900, 682, 441, 230, 83, 8, 0, 0, 11, 10, 0, 0, 30, 108, 232, 386, 540, 658, 713, 692, 601, 463, 312, 177, 79, 24, 6, 10, 20, 26, 24, 16, 8, 2, 1, 1, 2, 1, 0, 0, 3, 8, 11, 9, 1, 0, 1, 42, 129, 263, 431, 601, 735, 800, 779, 676, 519, 345, 190, 78, 16, 0, 0, 4, 11, 13, 14, 17, 22, 28, 31, 30, 24, 18, 12, 10, 11, 15, 19, 22, 24, 24, 22, 19, 16, 13, 10, 8, 7, 6, 6, 5, 4, 2, 1, 0, 0, 0, 0, 1, 4, 8, 13, 16, 18, 17, 13, 9, 6, 5, 7, 11, 17, 23, 29, 34, 39, 42, 44, 44, 43, 41, 38, 35, 31, 27, 23, 18, 13, 8, 3, 0, 0, 0, 3, 9, 15, 20, 23, 23, 19, 13, 7, 4, 6, 11, 18, 23, 23, 19, 11, 5, 4, 8, 13, 12, 3, 0, 0, 37, 138, 304, 518, 743, 928, 1030, 1024, 913, 725, 506, 299, 139, 40, 0, 0, 7, 16, 13, 0, 0, 8, 72, 190, 353, 535, 699, 807, 831, 767, 629, 453, 276, 132, 38, 0, 0, 5, 19, 21, 9, 0, 0, 22, 104, 238, 409, 587, 731, 805, 791, 693, 537, 362, 204, 88, 21, 0, 0, 3, 10, 12, 13, 15, 20, 25, 28, 28, 25, 21, 18, 18, 18, 18, 16, 12, 7, 2, 1, 3, 9, 15, 20, 22, 20, 15, 9, 3, 0, 0, 0, 2, 5, 7, 7, 4, 0, 0, 0, 0, 6, 11, 9, 0, 0, 6, 59, 166, 327, 524, 717, 861, 916, 866, 724, 530, 340, 209, 171, 233, 369, 533, 670, 738, 719, 619, 469, 306, 165, 66, 13, 0, 0, 7, 11, 8, 3, 0, 0, 1, 8, 16, 24, 30, 36, 39, 38, 31, 16, 0, 0, 8, 66, 171, 317, 481, 627, 718, 731, 662, 530, 368, 213, 93, 20, 0, 0, 5, 19, 22, 14, 0, 0, 1, 59, 173, 340, 536, 719, 844, 875, 801, 640, 435, 243, 111, 72, 126, 251, 406, 544, 628, 640, 579, 466, 330, 199, 96, 30, 0, 0, 2, 12, 16, 10, 0, 0, 1, 49, 143, 282, 448, 608, 726, 773, 736, 624, 467, 300, 157, 58, 7, 0, 0, 8, 12, 10, 3, 0, 0, 0, 5, 12, 17, 19, 18, 15, 10, 6, 3, 1, 0, 0, 0, 1, 6, 13, 22, 28, 30, 27, 20, 12, 4, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 2, 6, 11, 15, 19, 21, 21, 19, 16, 12, 8, 4, 2, 3, 7, 14, 21, 27, 29, 28, 24, 19, 15, 13, 12, 12, 13, 13, 12, 12, 12, 11, 10, 9, 8, 9, 11, 14, 13, 6, 0, 0, 15, 80, 197, 358, 538, 697, 796, 806, 726, 577, 396, 226, 98, 22, 0, 0, 5, 12, 12, 6, 1, 0, 0, 4, 8, 11, 13, 12, 9, 6, 4, 2, 3, 6, 10, 15, 19, 21, 22, 22, 21, 21, 21, 22, 24, 27, 29, 31, 32, 31, 29, 25, 21, 17, 14, 12, 12, 13, 14, 14, 12, 9, 4, 0, 0, 0, 3, 10, 12, 7, 0, 0, 15, 76, 184, 330, 486, 614, 677, 658, 560, 415, 270, 173, 160, 237, 385, 559, 709, 789, 778, 678, 519, 341, 184, 73, 13, 0, 0, 7, 11, 8, 2, 0, 0, 0, 0, 3, 4, 4, 4, 5, 5, 4, 2, 0, 0, 0, 6, 17, 29, 41, 48, 48, 43, 37, 32, 30, 26, 16, 1, 0, 0, 54, 171, 349, 563, 769, 917, 967, 906, 751, 543, 332, 159, 47, 0, 0, 4, 19, 22, 11, 0, 0, 17, 88, 206, 359, 517, 644, 707, 689, 596, 451, 292, 153, 54, 2, 0, 0, 12, 22, 21, 10, 0, 0, 5, 64, 167, 305, 453, 575, 639, 627, 544, 413, 267, 141, 52, 6, 0, 0, 7, 10, 8, 2, 0, 0, 0, 1, 4, 5, 5, 3, 2, 0, 0, 1, 2, 2, 1, 0, 0, 0, 3, 10, 19, 27, 31, 30, 24, 16, 7, 1, 0, 0, 0, 1, 2, 1, 0, 0, 0, 0, 3, 9, 14, 18, 20, 17, 13, 10, 8, 9, 10, 7, 0, 0, 0, 32, 100, 204, 330, 452, 541, 574, 542, 452, 329, 203, 98, 30, 0, 0, 1, 8, 10, 5, 0, 0, 0, 9, 25, 42, 54, 56, 47, 31, 16, 8, 9, 16, 19, 10, 0, 0, 28, 142, 342, 612, 901, 1141, 1267, 1245, 1080, 819, 530, 276, 100, 10, 0, 0, 14, 20, 14, 3, 0, 0, 0, 5, 7, 5, 2, 0, 3, 7, 11, 9, 2, 0, 1, 38, 117, 239, 387, 531, 634, 666, 616, 496, 341, 196, 104, 92, 163, 295, 451, 586, 664, 666, 595, 470, 324, 188, 86, 24, 0, 0, 3, 8, 7, 2, 0, 0, 2, 10, 17, 21, 21, 16, 10, 4, 0, 0, 0, 0, 2, 4, 8, 13, 19, 24, 26, 26, 23, 19, 14, 11, 10, 11, 15, 19, 22, 22, 20, 15, 9, 4, 1, 0, 0, 0, 1, 3, 5, 7, 9, 13, 17, 20, 20, 15, 5, 0, 0, 25, 91, 195, 324, 450, 540, 566, 517, 405, 266, 142, 76, 91, 187, 336, 495, 616, 664, 621, 499, 332, 169, 64, 52, 147, 332, 559, 769, 904, 928, 835, 657, 441, 241, 95, 15, 0, 0, 11, 17, 14, 5, 0, 0, 0, 4, 11, 16, 17, 16, 12, 8, 4, 1, 0, 0, 0, 2, 4, 6, 7, 8, 7, 5, 2, 0, 0, 0, 5, 13, 21, 28, 30, 28, 22, 13, 6, 1, 0, 0, 0, 3, 5, 6, 7, 8, 7, 6, 4, 1, 0, 0, 0, 2, 7, 13, 21, 28, 32, 33, 31, 26, 18, 11, 5, 1, 0, 0, 0, 2, 5, 10, 15, 20, 23, 25, 25, 24, 24, 25, 26, 28, 28, 26, 23, 20, 18, 17, 18, 19, 20, 19, 16, 11, 5, 2, 1, 4, 9, 16, 22, 26, 27, 26, 23, 19, 12, 4, 0, 0, 1, 12, 25, 28, 17, 0, 0, 18, 129, 334, 619, 932, 1200, 1349, 1334, 1157, 865, 542, 276, 135, 142, 276, 476, 667, 784, 792, 695, 526, 336, 171, 60, 5, 0, 1, 12, 16, 12, 6, 4, 7, 13, 19, 21, 19, 14, 9, 5, 4, 5, 8, 14, 21, 28, 34, 37, 34, 26, 15, 6, 3, 7, 13, 15, 9, 0, 0, 15, 83, 205, 371, 550, 698, 771, 745, 625, 446, 262, 133, 101, 181, 351, 563, 755, 871, 880, 782, 609, 407, 222, 87, 11, 0, 0, 10, 25, 28, 17, 0, 0, 0, 63, 202, 410, 658, 894, 1058, 1106, 1023, 832, 586, 344, 155, 40, 0, 0, 5, 15, 14, 6, 0, 0, 0, 6, 12, 13, 11, 6, 2, 0, 0, 0, 3, 7, 14, 21, 28, 32, 32, 28, 20, 11, 3, 0, 0, 3, 8, 13, 15, 13, 9, 4, 1, 0, 0, 1, 3, 4, 4, 2, 1, 0, 0, 0, 0, 0, 0, 0, 4, 12, 23, 35, 43, 44, 38, 26, 13, 3, 0, 4, 15, 29, 41, 47, 47, 40, 29, 18, 9, 3, 0, 0, 0, 0, 1, 4, 7, 10, 12, 12, 12, 13, 16, 19, 17, 10, 0, 0, 14, 79, 199, 364, 547, 703, 788, 775, 665, 491, 307, 173, 132, 195, 340, 513, 654, 713, 668, 533, 354, 187, 88, 88, 186, 352, 534, 677, 742, 713, 601, 442, 276, 138, 47, 4, 0, 0, 8, 10, 6, 1, 0, 0, 0, 0, 2, 5, 9, 13, 17, 18, 16, 12, 7, 3, 0, 0, 0, 0, 3, 7, 12, 15, 16, 14, 9, 4, 1, 4, 12, 24, 35, 43, 45, 40, 30, 19, 8, 1, 0, 0, 4, 7, 8, 6, 1, 0, 0, 10, 26, 38, 36, 15, 0, 0, 52, 225, 515, 886, 1263, 1551, 1670, 1586, 1321, 951, 571, 264, 71, 0, 0, 8, 28, 30, 17, 2, 0, 3, 17, 31, 38, 37, 30, 21, 12, 6, 2, 0, 0, 0, 1, 3, 5, 8, 11, 13, 13, 11, 8, 4, 1, 0, 0, 1, 4, 7, 9, 8, 7, 5, 4, 4, 5, 6, 8, 9, 12, 15, 19, 19, 13, 1, 0, 0, 40, 122, 243, 383, 509, 585, 588, 515, 387, 242, 127, 82, 125, 247, 414, 575, 679, 691, 603, 439, 249, 92, 20, 57, 197, 400, 610, 766, 827, 776, 629, 429, 231, 89, 41, 101, 250, 448, 641, 775, 812, 740, 579, 372, 177, 49, 24, 108, 275, 474, 645, 740, 734, 634, 472, 295, 144, 45, 0, 0, 3, 15, 18, 15, 10, 8, 11, 16, 21, 23, 22, 20, 20, 23, 27, 32, 34, 33, 27, 19, 10, 3, 0, 0, 0, 2, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 7, 10, 12, 11, 9, 6, 5, 7, 12, 19, 25, 31, 37, 42, 43, 38, 24, 3, 0, 0, 58, 181, 365, 582, 786, 926, 966, 895, 734, 525, 317, 150, 43, 0, 0, 3, 20, 26, 18, 0, 0, 0, 48, 159, 325, 521, 707, 836, 873, 808, 659, 466, 275, 125, 33, 0, 0, 4, 14, 14, 7, 0, 0, 0, 4, 13, 22, 28, 30, 29, 27, 27, 26, 22, 12, 0, 0, 4, 60, 164, 307, 462, 591, 659, 645, 554, 414, 261, 131, 43, 0, 0, 0, 7, 18, 22, 17, 2, 0, 0, 33, 142, 321, 547, 775, 948, 1016, 962, 802, 580, 354, 170, 51, 0, 0, 0, 16, 26, 27, 20, 6, 0, 0, 16, 92, 225, 406, 603, 770, 864, 858, 753, 578, 378, 200, 74, 9, 0, 2, 17, 26, 28, 28, 30, 31, 23, 4, 0, 0, 63, 221, 465, 758, 1033, 1215, 1251, 1129, 887, 591, 318, 119, 13, 0, 0, 15, 24, 19, 6, 0, 0, 0, 0, 5, 8, 10, 11, 13, 13, 12, 10, 6, 2, 0, 0, 1, 6, 14, 24, 32, 38, 39, 35, 26, 16, 7, 1, 0, 0, 5, 10, 14, 15, 14, 10, 5, 0, 0, 1, 7, 16, 25, 33, 36, 36, 32, 26, 21, 19, 21, 27, 33, 36, 33, 24, 12, 5, 8, 19, 30, 29, 12, 0, 0, 56, 227, 513, 878, 1249, 1533, 1647, 1557, 1284, 905, 524, 238, 106, 135, 282, 476, 641, 721, 697, 585, 423, 257, 122, 36, 0, 0, 5, 16, 19, 12, 0, 0, 0, 39, 121, 244, 391, 536, 644, 687, 653, 550, 404, 251, 123, 39, 0, 0, 0, 9, 11, 6, 0, 0, 0, 2, 9, 15, 18, 18, 16, 11, 6, 2, 0, 0, 2, 11, 21, 32, 38, 39, 32, 22, 13, 10, 14, 20, 22, 14, 0, 0, 8, 84, 228, 433, 661, 861, 980, 986, 878, 685, 459, 252, 101, 18, 0, 0, 9, 15, 12, 4, 0, 0, 0, 3, 7, 8, 7, 5, 2, 0, 0, 0, 0, 0, 0, 2, 6, 11, 15, 18, 17, 14, 9, 5, 1, 0, 0, 0, 0, 0, 0, 0, 2, 4, 6, 7, 6, 4, 1, 0, 0, 2, 5, 8, 11, 14, 17, 20, 23, 26, 28, 28, 27, 24, 20, 16, 11, 7, 4, 3, 4, 7, 12, 16, 20, 22, 23, 23, 23, 23, 23, 22, 21, 18, 13, 8, 3, 0, 0, 1, 5, 9, 11, 12, 10, 8, 7, 7, 9, 11, 14, 18, 22, 25, 22, 13, 0, 0, 3, 51, 144, 276, 424, 552, 625, 621, 537, 398, 246, 127, 76, 105, 204, 338, 464, 542, 551, 490, 379, 249, 131, 47, 2, 0, 0, 14, 28, 34, 30, 16, 0, 0, 4, 71, 200, 385, 599, 795, 923, 946, 858, 682, 467, 262, 109, 21, 0, 0, 8, 18, 20, 18, 17, 21, 27, 31, 29, 23, 15, 11, 14, 23, 35, 45, 50, 47, 38, 27, 16, 10, 10, 14, 21, 27, 30, 28, 21, 13, 6, 2, 3, 8, 17, 27, 36, 42, 43, 39, 32, 21, 11, 3, 0, 2, 11, 24, 39, 51, 56, 54, 44, 30, 16, 5, 1, 4, 12, 23, 34, 44, 51, 54, 49, 34, 11, 0, 0, 45, 163, 349, 578, 804, 970, 1032, 970, 800, 567, 332, 155, 74, 99, 210, 365, 509, 596, 599, 515, 371, 213, 92, 52, 114, 269, 479, 691, 846, 903, 849, 703, 504, 304, 143, 42, 0, 0, 4, 14, 15, 9, 2, 0, 0, 0, 0, 2, 6, 11, 19, 27, 34, 36, 33, 26, 17, 8, 2, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 4, 8, 13, 18, 22, 24, 24, 21, 16, 10, 4, 2, 3, 11, 23, 38, 53, 63, 67, 64, 53, 40, 27, 18, 13, 11, 8, 3, 0, 0, 3, 15, 25, 25, 10, 0, 0, 57, 223, 499, 857, 1231, 1535, 1689, 1649, 1424, 1075, 691, 369, 180, 148, 255, 441, 633, 764, 796, 723, 573, 389, 218, 90, 16, 0, 0, 12, 23, 23, 10, 0, 0, 13, 83, 208, 378, 564, 725, 819, 821, 728, 565, 374, 199, 72, 5, 0, 0, 21, 33, 34, 25, 15, 8, 9, 16, 28, 41, 51, 57, 57, 51, 40, 28, 16, 7, 1, 0, 0, 2, 3, 4, 3, 2, 2, 1, 1, 0, 0, 0, 2, 11, 23, 37, 49, 54, 51, 41, 28, 14, 5, 0, 0, 0, 1, 2, 6, 11, 16, 16, 9, 0, 0, 13, 70, 173, 312, 459, 577, 630, 601, 494, 342, 192, 94, 82, 162, 313, 491, 644, 729, 723, 631, 482, 315, 168, 64, 10, 0, 0, 8, 12, 9, 2, 0, 0, 0, 5, 14, 22, 29, 32, 33, 33, 34, 36, 35, 26, 7, 0, 0, 46, 172, 374, 626, 878, 1069, 1148, 1092, 918, 674, 421, 211, 72, 5, 0, 0, 10, 17, 18, 18, 17, 10, 0, 0, 1, 67, 203, 412, 666, 912, 1092, 1159, 1095, 918, 676, 424, 214, 72, 1, 0, 0, 18, 27, 22, 4, 0, 0, 37, 137, 292, 482, 669, 806, 855, 799, 651, 451, 255, 117, 74, 134, 276, 455, 617, 715, 719, 629, 470, 289, 137, 62, 86, 207, 392, 591, 750, 825, 799, 683, 511, 326, 169, 62, 8, 0, 0, 8, 11, 7, 1, 0, 0, 0, 4, 8, 9, 8, 7, 6, 5, 3, 1, 0, 0, 1, 8, 17, 28, 38, 44, 44, 39, 31, 23, 15, 9, 5, 3, 1, 0, 0, 1, 4, 8, 11, 13, 13, 13, 13, 14, 16, 19, 19, 16, 11, 5, 3, 7, 17, 26, 26, 12, 0, 0, 34, 153, 358, 630, 916, 1149, 1265, 1230, 1055, 787, 496, 247, 79, 0, 0, 1, 20, 26, 17, 5, 0, 2, 15, 29, 39, 41, 35, 28, 23, 24, 28, 25, 12, 0, 0, 31, 156, 379, 685, 1023, 1319, 1496, 1506, 1345, 1058, 717, 402, 167, 35, 0, 0, 11, 20, 15, 3, 0, 0, 10, 24, 33, 33, 25, 15, 5, 0, 0, 0, 0, 4, 9, 15, 21, 25, 25, 23, 20, 18, 18, 17, 13, 4, 0, 0, 26, 100, 225, 393, 578, 745, 854, 880, 814, 674, 493, 310, 158, 57, 6, 0, 0, 6, 9, 7, 4, 4, 8, 15, 20, 23, 23, 22, 20, 19, 18, 17, 16, 15, 16, 17, 18, 17, 15, 12, 12, 15, 17, 16, 7, 0, 0, 19, 94, 228, 408, 604, 769, 860, 851, 744, 569, 372, 196, 73, 8, 0, 0, 9, 14, 10, 3, 0, 0, 0, 4, 5, 5, 3, 4, 8, 14, 16, 12, 1, 0, 0, 45, 136, 271, 427, 567, 655, 662, 586, 448, 288, 152, 79, 85, 163, 286, 416, 516, 560, 539, 464, 355, 238, 137, 65, 23, 7, 7, 14, 19, 21, 16, 5, 0, 0, 12, 65, 155, 275, 403, 512, 574, 575, 515, 410, 287, 172, 83, 28, 3, 0, 0, 4, 6, 7, 9, 15, 24, 33, 39, 41, 37, 28, 17, 7, 0, 0, 1, 8, 18, 27, 33, 34, 30, 23, 14, 9, 8, 12, 16, 14, 6, 0, 0, 27, 109, 248, 429, 623, 784, 875, 870, 774, 610, 419, 243, 109, 30, 0, 0, 4, 11, 12, 10, 7, 7, 10, 15, 19, 19, 16, 11, 5, 1, 0, 0, 3, 7, 11, 14, 16, 17, 16, 13, 10, 7, 4, 3, 4, 7, 11, 14, 17, 16, 13, 9, 5, 4, 5, 9, 13, 17, 19, 19, 19, 19, 20, 21, 24, 26, 27, 28, 29, 29, 29, 27, 23, 16, 9, 3, 0, 0, 3, 11, 20, 29, 36, 42, 44, 44, 40, 34, 26, 18, 14, 15, 22, 29, 29, 16, 0, 0, 16, 113, 292, 542, 822, 1070, 1228, 1254, 1142, 923, 653, 392, 187, 58, 0, 0, 2, 13, 16, 11, 5, 2, 5, 11, 18, 24, 27, 29, 31, 32, 32, 31, 27, 21, 16, 13, 13, 16, 20, 22, 21, 18, 12, 7, 2, 0, 0, 0, 0, 0, 2, 5, 8, 11, 11, 8, 4, 1, 0, 1, 4, 9, 13, 13, 11, 6, 2, 0, 0, 2, 6, 10, 13, 13, 13, 13, 15, 19, 20, 15, 5, 0, 0, 35, 118, 250, 418, 589, 723, 785, 754, 637, 467, 292, 163, 115, 159, 279, 434, 575, 659, 659, 576, 436, 281, 160, 108, 142, 255, 413, 572, 688, 730, 687, 574, 421, 264, 134, 48, 5, 0, 0, 7, 9, 5, 0, 0, 0, 4, 11, 18, 22, 24, 23, 20, 17, 14, 13, 13, 13, 12, 11, 8, 5, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 7, 12, 18, 25, 30, 33, 33, 32, 29, 27, 26, 26, 26, 25, 23, 20, 17, 14, 14, 17, 23, 29, 33, 34, 29, 21, 11, 3, 0, 2, 8, 18, 28, 36, 39, 38, 30, 16, 2, 0, 5, 44, 120, 228, 356, 478, 565, 594, 557, 463, 341, 228, 162, 167, 246, 377, 521, 633, 677, 637, 521, 363, 211, 111, 97, 177, 336, 532, 718, 848, 893, 844, 718, 546, 366, 208, 94, 25, 0, 0, 3, 12, 14, 7, 0, 0, 9, 67, 174, 328, 508, 682, 809, 861, 825, 709, 543, 362, 202, 86, 19, 0, 0, 6, 14, 13, 5, 0, 0, 21, 79, 177, 308, 454, 586, 675, 700, 652, 543, 398, 250, 125, 42, 1, 0, 2, 15, 23, 22, 16, 8, 2, 0, 0, 3, 8, 15, 20, 24, 25, 22, 17, 10, 3, 0, 0, 5, 14, 22, 29, 31, 29, 22, 15, 8, 2, 0, 0, 0, 2, 8, 17, 27, 35, 38, 35, 27, 16, 6, 0, 0, 0, 6, 14, 22, 27, 28, 25, 19, 11, 6, 3, 2, 2, 1, 0, 0, 1, 5, 7, 6, 0, 0, 0, 29, 92, 191, 312, 432, 523, 562, 541, 466, 361, 257, 185, 163, 198, 276, 374, 464, 520, 524, 477, 387, 278, 171, 85, 30, 3, 0, 0, 4, 5, 3, 0, 0, 0, 0, 1, 1, 1, 1, 3, 6, 11, 16, 19, 21, 20, 18, 13, 9, 5, 3, 2, 2, 3, 4, 4, 4, 5, 8, 13, 20, 26, 31, 32, 30, 25, 19, 14, 11, 11, 12, 15, 19, 22, 24, 24, 22, 22, 24, 29, 32, 28, 13, 0, 0, 31, 149, 360, 651, 979, 1277, 1476, 1528, 1420, 1179, 864, 546, 282, 105, 15, 0, 0, 11, 16, 11, 1, 0, 0, 0, 6, 12, 16, 17, 16, 12, 8, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 6, 10, 15, 19, 23, 26, 29, 32, 33, 31, 23, 9, 0, 0, 21, 83, 181, 306, 432, 533, 583, 569, 497, 384, 259, 146, 64, 16, 0, 0, 2, 7, 9, 9, 10, 14, 19, 25, 30, 32, 33, 31, 28, 25, 22, 20, 19, 20, 21, 21, 19, 15, 9, 4, 0, 0, 2, 5, 8, 10, 9, 6, 3, 1, 0, 2, 4, 6, 7, 9, 10, 11, 14, 17, 20, 22, 23, 24, 23, 21, 19, 17, 16, 17, 20, 25, 31, 36, 38, 38, 38, 38, 38, 36, 28, 13, 0, 0, 33, 137, 319, 569, 852, 1115, 1299, 1359, 1279, 1082, 817, 551, 347, 245, 255, 352, 491, 620, 695, 694, 618, 489, 339, 200, 95, 31, 2, 0, 1, 5, 6, 5, 5, 7, 11, 14, 15, 12, 8, 3, 0, 0, 0, 0, 2, 3, 3, 2, 1, 0, 0, 0, 1, 3, 4, 5, 5, 3, 2, 0, 0, 0, 0, 1, 2, 3, 3, 3, 4, 5, 8, 11, 13, 13, 11, 8, 5, 2, 0, 0, 0, 0, 0, 0, 1, 3, 4, 5, 5, 4, 4, 5, 7, 11, 14, 16, 17, 16, 15, 15, 15, 16, 17, 19, 20, 20, 20, 19, 18, 15, 11, 7, 3, 0, 0, 2, 7, 12, 16, 19, 20, 19, 18, 16, 14, 14, 15, 17, 20, 23, 24, 23, 21, 17, 14, 13, 14, 16, 17, 17, 15, 10, 6, 2, 0, 0, 0, 0, 0, 1, 2, 4, 6, 10, 14, 18, 22, 28, 33, 39, 42, 43, 40, 34, 26, 16, 8, 2, 0, 0, 2, 4, 6, 7, 6, 4, 2, 1, 1, 3, 6, 9, 13, 16, 20, 21, 20, 13, 3, 0, 3, 39, 119, 245, 409, 586, 741, 839, 853, 779, 635, 460, 306, 217, 222, 320, 483, 665, 810, 879, 853, 741, 572, 388, 225, 105, 35, 4, 0, 2, 5, 6, 5, 7, 11, 17, 23, 25, 24, 20, 16, 12, 10, 9, 7, 5, 3, 1, 0, 0, 1, 3, 5, 6, 6, 6, 4, 2, 0, 0, 0, 2, 7, 14, 21, 25, 26, 24, 21, 17, 14, 13, 13, 12, 10, 7, 5, 5, 8, 10, 9, 3, 0, 0, 23, 84, 187, 326, 481, 623, 724, 762, 728, 630, 490, 336, 197, 91, 26, 0, 0, 4, 11, 14, 12, 7, 4, 4, 7, 13, 18, 22, 24, 24, 23, 21, 19, 19, 21, 24, 29, 32, 33, 31, 25, 17, 9, 3, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 2, 3, 5, 6, 8, 10, 13, 17, 22, 28, 33, 35, 32, 22, 9, 0, 0, 21, 73, 156, 263, 376, 473, 532, 541, 497, 412, 305, 197, 108, 44, 8, 0, 0, 5, 16, 20, 14, 0, 0, 1, 68, 215, 449, 752, 1078, 1364, 1549, 1591, 1481, 1245, 938, 622, 350, 155, 42, 0, 0, 8, 22, 28, 27, 21, 14, 8, 4, 3, 3, 5, 8, 10, 12, 12, 11, 12, 15, 21, 29, 36, 41, 43, 40, 34, 27, 21, 16, 15, 18, 22, 27, 30, 31, 29, 25, 20, 17, 15, 13, 12, 10, 8, 5, 2, 0, 0, 0, 0, 0, 2, 3, 5, 7, 9, 11, 14, 17, 20, 22, 24, 24, 24, 23, 22, 21, 20, 17, 15, 12, 10, 10, 12, 14, 15, 15, 13, 11, 9, 8, 8, 10, 13, 15, 16, 13, 9, 4, 1, 9, 32, 77, 143, 225, 308, 373, 405, 395, 347, 277, 209, 168, 169, 216, 297, 391, 474, 524, 530, 493, 421, 328, 230, 141, 71, 25, 2, 0, 2, 7, 7, 1, 0, 0, 32, 101, 209, 344, 488, 612, 692, 713, 670, 574, 449, 325, 231, 192, 215, 294, 405, 518, 598, 623, 586, 495, 374, 249, 142, 67, 24, 7, 6, 10, 11, 8, 0, 0, 0, 28, 90, 194, 334, 494, 649, 766, 820, 795, 697, 548, 378, 222, 103, 31, 0, 0, 0, 6, 9, 10, 11, 14, 18, 22, 24, 24, 23, 23, 22, 21, 20, 17, 15, 15, 18, 22, 28, 33, 35, 33, 28, 21, 13, 7, 3, 2, 4, 7, 10, 10, 8, 5, 3, 1, 1, 0, 0, 0, 4, 23, 64, 131, 222, 327, 428, 508, 548, 540, 484, 391, 282, 175, 88, 31, 4, 0, 8, 15, 15, 6, 0, 0, 27, 96, 210, 357, 517, 665, 773, 823, 807, 731, 610, 464, 318, 191, 95, 34, 4, 0, 0, 3, 4, 3, 1, 1, 3, 7, 12, 15, 17, 17, 15, 14, 12, 10, 9, 8, 7, 8, 8, 9, 10, 9, 7, 5, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 6, 11, 17, 23, 28, 33, 36, 39, 40, 39, 35, 29, 21, 11, 2, 0, 8, 39, 102, 199, 328, 476, 622, 740, 808, 810, 746, 628, 477, 323, 187, 87, 27, 0, 0, 2, 9, 12, 11, 8, 6, 6, 10, 15, 19, 22, 23, 22, 21, 21, 24, 28, 34, 38, 40, 38, 33, 26, 20, 16, 15, 16, 18, 18, 16, 11, 6, 1, 0, 1, 4, 7, 9, 9, 7, 4, 1, 0, 0, 0, 1, 2, 3, 3, 1, 0, 0, 0, 1, 4, 7, 9, 8, 7, 6, 7, 14, 24, 35, 45, 50, 49, 43, 34, 25, 19, 14, 12, 10, 8, 5, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 4, 5, 6, 5, 5, 4, 3, 3, 3, 3, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 5, 8, 9, 10, 10, 10, 11, 12, 14, 17, 19, 19, 18, 14, 9, 4, 1, 0, 0, 1, 4, 8, 13, 18, 21, 24, 23, 21, 17, 12, 6, 2, 0, 0, 0, 2, 7, 12, 17, 20, 22, 22, 19, 14, 9, 5, 1, 0, 0, 1, 2, 3, 3, 4, 5, 8, 12, 16, 20, 23, 24, 25, 25, 26, 28, 33, 39, 44, 46, 44, 35, 20, 3, 0, 9, 58, 153, 294, 469, 653, 816, 928, 966, 925, 816, 667, 512, 387, 319, 317, 374, 463, 551, 604, 601, 540, 440, 333, 258, 243, 295, 403, 535, 652, 719, 718, 649, 530, 388, 251, 140, 64, 21, 3, 0, 1, 3, 2, 2, 2, 5, 9, 12, 14, 13, 11, 9, 8, 10, 13, 17, 19, 19, 16, 13, 12, 14, 19, 24, 25, 22, 15, 10, 14, 36, 83, 158, 257, 370, 482, 575, 633, 645, 605, 520, 405, 279, 165, 77, 23, 1, 1, 10, 19, 21, 17, 9, 2, 0, 0, 2, 8, 14, 17, 17, 13, 6, 0, 2, 19, 59, 124, 211, 309, 401, 471, 504, 495, 448, 378, 301, 236, 198, 191, 212, 253, 298, 336, 355, 350, 320, 270, 208, 143, 85, 40, 12, 0, 0, 8, 18, 26, 32, 36, 37, 38, 39, 39, 37, 34, 30, 26, 20, 13, 5, 0, 0, 13, 56, 135, 251, 397, 556, 706, 821, 880, 873, 799, 672, 513, 350, 207, 100, 33, 2, 0, 0, 3, 5, 3, 0, 0, 0, 0, 3, 7, 13, 19, 24, 27, 28, 26, 22, 18, 14, 10, 7, 4, 2, 1, 1, 2, 5, 7, 9, 11, 11, 10, 8, 6, 3, 2, 1, 2, 2, 2, 0, 0, 2, 22, 65, 140, 245, 372, 501, 611, 678, 689, 640, 542, 415, 283, 167, 81, 29, 5, 0, 0, 1, 2, 3, 6, 10, 13, 16, 15, 13, 10, 6, 4, 3, 3, 2, 2, 2, 4, 6, 9, 10, 10, 11, 11, 10, 7, 0, 0, 0, 25, 86, 187, 323, 479, 628, 744, 805, 799, 728, 606, 457, 306, 176, 81, 24, 0, 0, 1, 6, 8, 7, 4, 1, 0, 0, 0, 1, 2, 2, 3, 5, 5, 5, 4, 2, 0, 0, 0, 0, 1, 4, 7, 10, 12, 15, 16, 17, 17, 17, 17, 17, 17, 17, 15, 13, 10, 7, 4, 1, 0, 0, 0, 2, 6, 10, 13, 14, 13, 11, 8, 4, 1, 0, 0, 0, 0, 3, 5, 6, 6, 6, 8, 20, 48, 97, 167, 255, 348, 433, 493, 516, 497, 439, 354, 258, 166, 91, 40, 12, 1, 0, 0, 0, 0, 8, 34, 87, 167, 269, 380, 479, 550, 579, 561, 502, 414, 314, 216, 134, 72, 32, 10, 1, 0, 0, 0, 0, 3, 17, 46, 94, 158, 231, 302, 357, 385, 378, 338, 277, 213, 167, 156, 190, 264, 362, 460, 533, 562, 541, 476, 382, 279, 183, 106, 53, 21, 6, 1, 0, 2, 4, 5, 5, 4, 3, 2, 2, 2, 2, 2, 1, 1, 2, 4, 7, 10, 11, 11, 8, 4, 1, 0, 0, 0, 0, 0, 0, 0, 1, 2, 5, 7, 9, 9, 8, 5, 2, 0, 0, 7, 27, 61, 112, 177, 249, 321, 379, 416, 422, 398, 346, 277, 202, 132, 77, 39, 18, 9, 7, 5, 4, 5, 13, 35, 77, 139, 217, 303, 384, 447, 481, 480, 444, 379, 296, 209, 131, 69, 28, 6, 0, 0, 0, 2, 1, 0, 0, 0, 0, 2, 7, 12, 16, 20, 20, 19, 15, 11, 8, 7, 6, 7, 8, 9, 10, 10, 11, 11, 10, 9, 6, 3, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 2, 6, 10, 14, 14, 12, 7, 2, 0, 0, 3, 7, 8, 5, 0, 0, 8, 38, 91, 167, 256, 346, 422, 474, 494, 482, 444, 390, 334, 292, 276, 290, 333, 393, 454, 499, 511, 485, 421, 333, 235, 145, 75, 31, 11, 9, 15, 22, 25, 23, 19, 13, 9, 6, 5, 4, 3, 2, 2, 2, 3, 4, 5, 5, 4, 2, 0, 0, 0, 0, 2, 3, 3, 0, 0, 0, 18, 59, 126, 214, 311, 403, 473, 508, 503, 461, 390, 304, 216, 137, 76, 34, 10, 0, 0, 0, 3, 8, 13, 17, 20, 21, 19, 15, 10, 7, 7, 8, 11, 13, 14, 12, 9, 6, 2, 0, 0, 0, 1, 5, 9, 14, 17, 16, 13, 9, 4, 0, 0, 0, 1, 3, 4, 5, 5, 5, 5, 6, 5, 4, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 2, 6, 11, 16, 20, 21, 20, 16, 10, 5, 1, 0, 0, 1, 3, 4, 5, 5, 4, 4, 3, 3, 4, 4, 4, 5, 6, 7, 8, 8, 6, 2, 0, 0, 8, 32, 74, 135, 211, 295, 375, 441, 480, 486, 455, 393, 311, 220, 137, 70, 26, 3, 0, 0, 6, 13, 17, 18, 17, 15, 12, 10, 9, 9, 10, 11, 13, 13, 12, 10, 7, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 5, 10, 15, 20, 24, 27, 27, 26, 23, 18, 13, 9, 6, 3, 2, 0, 0, 0, 1, 5, 11, 19, 25, 28, 26, 19, 9, 0, 0, 9, 44, 106, 195, 304, 423, 536, 630, 693, 717, 700, 642, 551, 439, 319, 206, 114, 48, 11, 0, 0, 3, 9, 13, 16, 18, 20, 23, 25, 27, 27, 23, 16, 6, 0, 0, 23, 83, 191, 348, 547, 765, 971, 1131, 1217, 1214, 1122, 960, 757, 547, 358, 209, 105, 43, 12, 0, 0, 0, 4, 9, 14, 17, 18, 16, 12, 7, 2, 0, 0, 2, 13, 34, 69, 120, 183, 255, 327, 390, 434, 454, 445, 409, 351, 280, 204, 134, 77, 36, 14, 7, 9, 14, 19, 20, 16, 10, 4, 0, 0, 0, 3, 7, 10, 10, 9, 7, 4, 4, 6, 9, 14, 19, 21, 21, 19, 15, 11, 8, 5, 3, 3, 3, 4, 5, 6, 5, 4, 2, 0, 0, 1, 3, 4, 5, 5, 3, 2, 0, 0, 0, 0, 0, 0, 0, 2, 4, 5, 5, 5, 3, 2, 2, 4, 7, 11, 13, 12, 9, 5, 1, 0, 2, 6, 12, 18, 23, 26, 27, 27, 25, 23, 22, 23, 23, 23, 22, 19, 15, 11, 10, 11, 17, 25, 32, 38, 39, 37, 30, 21, 11, 4, 4, 15, 44, 98, 181, 295, 432, 579, 717, 823, 878, 870, 801, 682, 534, 380, 243, 136, 65, 25, 7, 1, 0, 0, 0, 0, 2, 9, 18, 28, 36, 42, 45, 43, 39, 33, 27, 22, 19, 19, 22, 26, 29, 30, 28, 24, 18, 12, 8, 7, 10, 15, 20, 25, 28, 28, 25, 20, 17, 17, 24, 41, 71, 118, 184, 267, 363, 462, 551, 614, 640, 622, 562, 471, 365, 261, 172, 106, 63, 39, 27, 20, 17, 19, 30, 58, 106, 175, 259, 348, 428, 486, 510, 500, 459, 402, 346, 307, 294, 309, 347, 395, 441, 474, 487, 478, 446, 395, 330, 256, 182, 115, 61, 25, 5, 0, 0, 1, 2, 2, 1, 0, 0, 0, 0, 1, 1, 0, 1, 5, 18, 42, 80, 132, 194, 259, 319, 367, 397, 407, 398, 371, 329, 276, 216, 154, 98, 52, 20, 3, 0, 1, 7, 12, 14, 14, 12, 10, 11, 12, 15, 16, 14, 11, 7, 2, 0, 0, 2, 7, 12, 17, 19, 19, 17, 14, 11, 7, 5, 3, 2, 2, 1, 0, 0, 0, 0, 1, 2, 4, 5, 6, 7, 7, 7, 6, 4, 2, 0, 0, 0, 0, 2, 4, 6, 10, 22, 49, 99, 175, 277, 394, 512, 612, 679, 703, 681, 618, 525, 416, 305, 205, 125, 67, 31, 11, 0, 0, 0, 10, 41, 94, 167, 251, 332, 394, 427, 428, 401, 353, 295, 235, 177, 127, 84, 50, 27, 13, 6, 6, 7, 8, 7, 5, 2, 0, 0, 0, 1, 5, 11, 18, 27, 34, 40, 43, 42, 39, 34, 30, 26, 25, 24, 24, 25, 28, 38, 60, 100, 158, 235, 321, 406, 476, 518, 526, 499, 443, 367, 285, 205, 137, 83, 45, 20, 7, 1, 0, 0, 0, 1, 3, 6, 9, 11, 12, 11, 9, 5, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 7, 9, 10, 10, 8, 8, 11, 16, 22, 28, 30, 27, 23, 25, 42, 85, 160, 268, 401, 544, 680, 789, 857, 874, 841, 763, 652, 524, 393, 275, 176, 101, 50, 19, 4, 0, 3, 9, 16, 20, 22, 21, 18, 13, 7, 3, 0, 0, 2, 11, 27, 54, 90, 137, 191, 250, 308, 358, 393, 410, 405, 383, 351, 320, 301, 301, 323, 361, 404, 440, 458, 452, 424, 382, 338, 306, 297, 315, 358, 416, 475, 520, 538, 524, 477, 405, 319, 231, 152, 90, 47, 23, 14, 13, 16, 17, 14, 9, 3, 0, 2, 10, 21, 32, 40, 42, 38, 30, 19, 10, 3, 0, 0, 0, 1, 2, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 3, 5, 7, 9, 10, 12, 13, 14, 15, 15, 14, 13, 10, 8, 6, 5, 3, 2, 0, 0, 0, 0, 1, 2, 3, 3, 2, 1, 0, 0, 0, 1, 2, 2, 2, 1, 1, 0, 0, 0, 0, 2, 3, 4, 4, 4, 3, 3, 4, 6, 8, 11, 13, 14, 15, 15, 16, 16, 16, 16, 14, 13, 11, 10, 9, 9, 10, 10, 10, 9, 7, 6, 4, 4, 4, 5, 5, 5, 4, 2, 1, 0, 0, 0, 0, 0, 0, 0, 1, 4, 8, 12, 16, 17, 15, 11, 5, 1, 0, 0, 3, 8, 13, 17, 18, 16, 12, 7, 3, 1, 1, 1, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 2, 6, 10, 14, 16, 16, 15, 12, 10, 9, 8, 6, 5, 5, 6, 9, 15, 21, 27, 30, 30, 28, 24, 20, 17, 14, 11, 9, 9, 14, 26, 48, 80, 122, 174, 233, 296, 358, 415, 458, 480, 476, 445, 391, 322, 249, 183, 128, 88, 61, 43, 33, 31, 39, 61, 97, 151, 218, 295, 372, 441, 491, 516, 511, 477, 419, 347, 270, 197, 133, 83, 45, 20, 6, 0, 0, 1, 3, 5, 5, 4, 2, 1, 1, 2, 3, 3, 3, 2, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 11, 30, 59, 98, 145, 198, 253, 308, 357, 395, 413, 407, 376, 324, 257, 188, 125, 77, 44, 24, 13, 6, 2, 0, 0, 2, 7, 14, 19, 21, 19, 15, 10, 6, 4, 4, 6, 9, 11, 13, 14, 14, 12, 10, 7, 4, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 7, 13, 19, 22, 22, 19, 13, 6, 1, 0, 0, 2, 5, 7, 8, 9, 9, 10, 12, 13, 14, 13, 11, 10, 8, 7, 7, 8, 8, 8, 9, 11, 13, 14, 14, 12, 9, 6, 5, 5, 8, 11, 13, 13, 11, 8, 4, 3, 2, 3, 4, 5, 5, 4, 3, 4, 6, 8, 10, 10, 9, 7, 5, 5, 7, 10, 15, 20, 22, 23, 21, 20, 20, 21, 24, 26, 27, 27, 27, 26, 26, 27, 27, 26, 22, 15, 8, 2, 0, 2, 9, 17, 22, 22, 17, 8, 1, 0, 7, 23, 43, 64, 85, 109, 143, 194, 265, 352, 442, 519, 567, 575, 544, 482, 402, 318, 242, 177, 126, 86, 56, 35, 24, 22, 31, 51, 85, 137, 210, 303, 412, 524, 628, 707, 751, 756, 723, 659, 574, 477, 378, 283, 198, 129, 77, 42, 23, 16, 15, 18, 20, 20, 17, 13, 8, 4, 3, 4, 9, 15, 23, 34, 52, 80, 123, 179, 243, 307, 359, 389, 392, 368, 325, 271, 215, 165, 123, 89, 62, 40, 23, 10, 2, 0, 0, 0, 3, 5, 7, 9, 9, 8, 8, 7, 8, 9, 10, 10, 10, 9, 7, 6, 5, 5, 6, 7, 9, 9, 10, 12, 14, 18, 24, 30, 35, 38, 40, 40, 40, 42, 49, 61, 83, 117, 167, 237, 327, 434, 548, 655, 737, 780, 776, 724, 635, 524, 407, 300, 212, 145, 99, 69, 49, 36, 26, 18, 11, 6, 4, 3, 4, 5, 5, 3, 1, 0, 1, 4, 10, 15, 18, 17, 13, 8, 6, 8, 16, 28, 43, 61, 81, 107, 139, 178, 221, 261, 291, 307, 306, 288, 257, 218, 176, 135, 97, 64, 37, 17, 5, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 7, 13, 20, 26, 30, 32, 30, 27, 22, 18, 14, 13, 13, 14, 14, 14, 13, 10, 7, 5, 2, 1, 0, 0, 0, 0, 1, 3, 5, 7, 11, 18, 32, 55, 88, 129, 175, 219, 256, 282, 293, 291, 275, 249, 216, 180, 142, 107, 77, 53, 36, 24, 18, 19, 28, 49, 85, 138, 207, 291, 384, 479, 567, 640, 689, 707, 691, 643, 567, 473, 373, 277, 192, 125, 75, 41, 21, 13, 16, 29, 53, 88, 130, 178, 226, 267, 295, 307, 301, 280, 248, 210, 171, 134, 100, 70, 43, 22, 7, 0, 0, 2, 9, 15, 19, 20, 17, 13, 10, 9, 11, 15, 21, 25, 26, 25, 21, 17, 13, 10, 9, 9, 10, 12, 14, 15, 15, 14, 12, 8, 5, 2, 2, 3, 6, 8, 9, 8, 6, 3, 1, 0, 0, 0, 0, 2, 5, 9, 12, 14, 17, 19, 23, 27, 32, 34, 33, 29, 22, 15, 10, 11, 18, 34, 57, 87, 125, 169, 217, 264, 305, 334, 346, 339, 314, 275, 230, 184, 142, 106, 76, 51, 31, 16, 6, 0, 0, 0, 0, 1, 2, 2, 2, 2, 2, 1, 1, 0, 0, 0, 0, 0, 1, 2, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 2, 2, 1, 1, 0, 0, 0, 0, 2, 5, 9, 12, 14, 13, 11, 7, 3, 0, 0, 0, 0, 3, 6, 11, 19, 32, 55, 90, 138, 196, 257, 311, 349, 363, 351, 317, 270, 220, 177, 143, 120, 104, 88, 70, 49, 29, 16, 19, 40, 80, 132, 190, 246, 292, 325, 343, 348, 339, 319, 288, 248, 203, 155, 110, 73, 46, 31, 28, 34, 47, 67, 92, 122, 154, 187, 217, 240, 257, 266, 271, 273, 274, 275, 275, 272, 268, 263, 261, 266, 276, 290, 301, 303, 290, 263, 223, 177, 132, 95, 67, 48, 34, 23, 14, 7, 1, 0, 2, 6, 12, 16, 19, 21, 23, 26, 30, 36, 41, 47, 54, 67, 90, 127, 177, 239, 304, 364, 409, 434, 437, 420, 389, 347, 301, 252, 204, 159, 118, 82, 54, 33, 19, 11, 7, 6, 5, 4, 3, 3, 3, 5, 8, 12, 15, 17, 17, 15, 12, 9, 5, 2, 0, 0, 2, 5, 7, 10, 10, 9, 7, 5, 4, 4, 5, 7, 8, 9, 10, 10, 11, 11, 12, 12, 11, 10, 8, 7, 6, 6, 8, 14, 28, 53, 91, 142, 204, 273, 341, 404, 458, 502, 538, 567, 590, 605, 609, 600, 578, 544, 503, 462, 427, 399, 379, 364, 349, 330, 303, 268, 225, 177, 129, 85, 50, 27, 16, 14, 17, 21, 21, 18, 14, 10, 9, 12, 18, 24, 27, 29, 30, 36, 52, 82, 128, 188, 259, 333, 404, 463, 504, 524, 521, 496, 454, 400, 339, 277, 219, 167, 125, 93, 71, 58, 56, 64, 86, 123, 177, 246, 325, 409, 488, 554, 600, 623, 623, 605, 575, 540, 507, 481, 465, 459, 460, 466, 469, 465, 447, 415, 367, 310, 249, 191, 141, 102, 74, 54, 39, 27, 17, 8, 3, 0, 0, 0, 1, 4, 6, 8, 9, 8, 6, 3, 1, 0, 0, 0, 1, 3, 5, 8, 15, 27, 45, 69, 101, 142, 192, 248, 307, 362, 403, 424, 421, 395, 352, 301, 248, 198, 153, 113, 77, 47, 27, 19, 26, 46, 77, 111, 145, 175, 200, 221, 237, 249, 255, 253, 242, 221, 193, 161, 130, 101, 75, 54, 36, 22, 12, 5, 1, 0, 0, 0, 1, 4, 7, 10, 12, 12, 10, 6, 2, 0, 0, 1, 4, 6, 7, 6, 4, 3, 4, 7, 11, 14, 17, 18, 18, 19, 21, 24, 27, 26, 23, 19, 17, 21, 37, 67, 113, 173, 245, 324, 408, 489, 561, 619, 656, 667, 652, 611, 548, 471, 387, 305, 230, 166, 115, 76, 48, 27, 13, 4, 0, 0, 0, 2, 4, 5, 7, 9, 14, 22, 35, 54, 77, 106, 138, 174, 211, 247, 279, 303, 316, 314, 297, 267, 230, 190, 151, 116, 87, 61, 40, 23, 11, 4, 3, 7, 14, 19, 22, 21, 16, 9, 2, 0, 0, 7, 24, 51, 83, 119, 154, 186, 214, 236, 253, 263, 264, 255, 235, 206, 170, 133, 100, 74, 56, 46, 42, 42, 46, 54, 67, 87, 115, 150, 190, 231, 269, 299, 317, 321, 309, 284, 248, 206, 162, 122, 87, 58, 36, 20, 10, 3, 0, 0, 0, 0, 0, 2, 3, 6, 8, 10, 10, 11, 11, 12, 14, 17, 21, 23, 24, 22, 20, 18, 17, 19, 22, 25, 27, 27, 25, 22, 21, 21, 23, 25, 25, 24, 21, 17, 13, 12, 13, 15, 15, 15, 12, 9, 7, 7, 10, 14, 17, 18, 16, 12, 8, 4, 2, 2, 2, 2, 0, 0, 0, 3, 10, 19, 26, 31, 30, 25, 20, 19, 26, 43, 69, 103, 144, 192, 246, 306, 369, 428, 476, 507, 517, 506, 478, 438, 390, 338, 284, 228, 174, 125, 85, 57, 44, 46, 63, 92, 132, 179, 228, 275, 314, 339, 348, 340, 320, 291, 258, 226, 195, 165, 135, 103, 72, 45, 26, 20, 30, 53, 87, 123, 156, 180, 195, 203, 206, 210, 213, 215, 212, 200, 177, 146, 110, 75, 46, 26, 14, 9, 7, 6, 5, 3, 0, 0, 0, 0, 2, 4, 6, 6, 6, 5, 3, 2, 2, 3, 5, 8, 11, 13, 14, 14, 13, 12, 12, 12, 14, 15, 15, 14, 12, 10, 8, 8, 8, 10, 11, 11, 9, 6, 3, 0, 0, 0, 2, 4, 5, 5, 3, 1, 0, 0, 0, 1, 5, 8, 11, 12, 10, 7, 3, 0, 0, 0, 3, 6, 9, 9, 8, 6, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 4, 8, 14, 19, 23, 26, 25, 23, 18, 13, 8, 5, 4, 6, 8, 10, 11, 10, 8, 6, 5, 5, 5, 5, 4, 2, 1, 0, 0, 0, 0, 0, 0, 1, 4, 9, 14, 19, 21, 20, 20, 21, 26, 35, 48, 61, 73, 82, 92, 106, 129, 166, 217, 279, 347, 409, 459, 489, 499, 489, 464, 428, 384, 335, 283, 232, 183, 140, 108, 89, 83, 92, 114, 147, 189, 235, 281, 322, 353, 370, 370, 357, 333, 303, 270, 237, 204, 170, 136, 104, 79, 64, 61, 71, 88, 111, 135, 162, 194, 234, 285, 345, 407, 461, 497, 507, 491, 451, 399, 343, 292, 249, 213, 183, 154, 126, 103, 88, 83, 89, 106, 131, 160, 191, 222, 252, 276, 292, 296, 288, 269, 242, 213, 184, 158, 134, 111, 89, 68, 50, 36, 28, 26, 27, 30, 33, 35, 35, 35, 34, 32, 30, 26, 22, 17, 12, 9, 7, 6, 6, 6, 7, 8, 8, 8, 8, 8, 7, 5, 2, 0, 0, 0, 4, 10, 18, 24, 27, 25, 20, 14, 8, 4, 3, 4, 5, 6, 7, 9, 11, 14, 17, 19, 19, 17, 13, 9, 5, 2, 0, 0, 0, 0, 2, 5, 7, 8, 8, 6, 3, 0, 0, 4, 14, 28, 47, 68, 91, 113, 134, 154, 172, 188, 203, 213, 218, 215, 205, 186, 162, 135, 109, 86, 67, 52, 40, 30, 21, 13, 8, 6, 6, 8, 10, 10, 8, 6, 3, 3, 4, 8, 11, 13, 13, 10, 5, 1, 0, 0, 1, 5, 8, 10, 10, 7, 4, 2, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 6, 11, 18, 24, 27, 26, 23, 18, 15, 15, 19, 24, 28, 28, 23, 16, 8, 2, 0, 0, 0, 2, 4, 5, 6, 7, 9, 10, 10, 9, 8, 8, 8, 9, 10, 13, 16, 23, 33, 49, 69, 92, 116, 139, 162, 185, 210, 236, 261, 282, 292, 291, 276, 251, 219, 187, 157, 131, 108, 89, 72, 58, 48, 41, 38, 36, 34, 32, 28, 24, 21, 20, 19, 20, 19, 17, 14, 9, 5, 2, 2, 3, 5, 7, 9, 9, 7, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 5, 9, 11, 13, 13, 12, 10, 7, 4, 2, 1, 0, 0, 0, 0, 0, 1, 3, 3, 3, 2, 1, 0, 2, 5, 9, 12, 13, 12, 8, 4, 2, 4, 9, 15, 21, 24, 22, 17, 10, 5, 3, 6, 11, 16, 19, 18, 14, 9, 5, 2, 2, 3, 5, 8, 10, 13, 17, 24, 34, 48, 67, 91, 120, 153, 186, 217, 245, 266, 281, 289, 291, 285, 273, 252, 225, 194, 160, 128, 100, 77, 60, 47, 40, 36, 34, 33, 31, 28, 24, 21, 19, 20, 24, 30, 39, 51, 68, 90, 119, 152, 183, 208, 224, 228, 224, 214, 202, 190, 179, 165, 148, 128, 105, 82, 61, 45, 33, 26, 23, 22, 22, 21, 20, 17, 13, 9, 7, 7, 10, 15, 20, 23, 23, 21, 19, 17, 17, 19, 23, 26, 30, 37, 48, 64, 86, 111, 137, 160, 178, 193, 204, 212, 217, 218, 212, 201, 183, 162, 141, 120, 102, 86, 74, 65, 61, 64, 75, 92, 113, 135, 158, 179, 200, 222, 242, 257, 264, 260, 244, 218, 185, 151, 119, 91, 68, 49, 33, 21, 11, 4, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 2, 3, 5, 6, 6, 5, 4, 2, 0, 0, 0, 0, 1, 3, 5, 6, 6, 5, 4, 3, 2, 3, 5, 8, 12, 15, 17, 19, 19, 18, 18, 19, 22, 28, 35, 44, 53, 63, 75, 91, 113, 142, 174, 205, 231, 247, 251, 244, 229, 209, 189, 170, 155, 141, 128, 114, 99, 84, 69, 58, 55, 62, 80, 105, 133, 159, 177, 189, 195, 201, 211, 228, 250, 274, 295, 308, 311, 307, 297, 286, 276, 267, 258, 248, 233, 214, 191, 166, 141, 117, 96, 77, 59, 42, 27, 16, 7, 2, 0, 0, 0, 0, 0, 0, 1, 3, 5, 7, 8, 8, 8, 6, 5, 4, 4, 4, 5, 7, 10, 12, 14, 15, 14, 11, 7, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 4, 5, 6, 6, 5, 4, 4, 6, 9, 13, 18, 22, 26, 29, 35, 44, 60, 84, 116, 152, 188, 218, 238, 245, 242, 231, 216, 201, 187, 171, 153, 131, 106, 81, 60, 46, 38, 34, 31, 27, 22, 15, 10, 8, 8, 9, 9, 7, 4, 1, 0, 0, 2, 5, 7, 8, 10, 12, 16, 22, 30, 40, 52, 66, 82, 101, 121, 140, 156, 168, 177, 183, 188, 192, 194, 192, 184, 170, 150, 126, 100, 77, 57, 40, 29, 20, 14, 10, 7, 5, 3, 2, 2, 2, 4, 7, 11, 14, 17, 19, 20, 20, 19, 19, 18, 19, 19, 21, 22, 24, 28, 35, 48, 68, 96, 129, 164, 194, 217, 231, 237, 239, 239, 239, 237, 230, 217, 197, 170, 142, 115, 91, 72, 57, 44, 33, 25, 19, 15, 14, 13, 11, 9, 5, 2, 0, 0, 0, 1, 2, 2, 2, 1, 0, 0, 0, 0, 2, 3, 5, 5, 5, 3, 1, 0, 0, 2, 5, 8, 10, 10, 8, 5, 2, 1, 3, 7, 13, 20, 26, 31, 36, 44, 56, 75, 101, 131, 162, 191, 214, 232, 247, 260, 274, 291, 307, 318, 321, 312, 291, 261, 227, 195, 168, 148, 134, 124, 117, 112, 113, 121, 136, 158, 181, 202, 216, 220, 215, 203, 186, 169, 151, 135, 120, 105, 91, 78, 67, 58, 50, 42, 35, 29, 24, 19, 16, 13, 10, 6, 3, 1, 2, 6, 11, 17, 22, 24, 23, 20, 16, 15, 18, 26, 38, 54, 72, 93, 115, 140, 165, 190, 213, 231, 244, 253, 260, 267, 276, 285, 290, 288, 273, 245, 206, 162, 121, 88, 65, 53, 46, 41, 35, 26, 17, 8, 2, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 2, 1, 0, 0, 0, 1, 5, 11, 15, 18, 17, 13, 8, 2, 0, 0, 2, 7, 11, 13, 13, 9, 5, 1, 0, 0, 1, 3, 6, 8, 8, 7, 5, 2, 0, 0, 0, 1, 5, 8, 10, 11, 9, 6, 3, 3, 7, 12, 18, 23, 23, 21, 15, 9, 4, 1, 1, 2, 4, 6, 8, 8, 7, 6, 5, 5, 4, 4, 3, 3, 3, 2, 2, 3, 5, 8, 13, 19, 25, 30, 34, 37, 40, 45, 54, 67, 84, 104, 126, 148, 169, 188, 203, 214, 220, 220, 214, 205, 192, 177, 161, 142, 123, 103, 84, 66, 52, 42, 36, 31, 29, 27, 27, 28, 31, 35, 42, 50, 61, 73, 86, 99, 111, 121, 129, 136, 144, 155, 170, 187, 200, 207, 203, 190, 170, 148, 129, 116, 107, 100, 94, 88, 85, 87, 97, 114, 135, 156, 172, 183, 190, 193, 197, 199, 200, 197, 189, 174, 156, 136, 116, 99, 84, 72, 61, 51, 41, 32, 24, 17, 13, 11, 10, 10, 9, 7, 4, 2, 0, 0, 1, 3, 5, 5, 4, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 2, 3, 5, 5, 5, 4, 1, 0, 0, 0, 3, 7, 11, 13, 13, 10, 6, 3, 0, 0, 0, 1, 3, 6, 8, 10, 12, 13, 14, 15, 15, 16, 17, 17, 17, 17, 17, 19, 21, 23, 24, 26, 27))
        self.assertEqual(record.annotations["abif_raw"]['DATA11'], (0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 1, 3, 4, 4, 3, 2, 1, 0, 0, 0, 0, 0, 1, 2, 4, 5, 6, 5, 3, 2, 0, 0, 2, 4, 8, 11, 12, 12, 11, 9, 8, 8, 10, 14, 20, 27, 33, 40, 45, 49, 52, 54, 58, 63, 70, 81, 94, 107, 120, 131, 140, 148, 157, 169, 184, 200, 215, 226, 233, 237, 238, 241, 245, 250, 254, 254, 250, 240, 227, 213, 201, 190, 180, 170, 158, 144, 129, 114, 102, 93, 86, 81, 75, 67, 58, 48, 39, 33, 29, 29, 30, 31, 32, 34, 36, 40, 45, 50, 56, 61, 66, 72, 81, 94, 111, 131, 153, 174, 196, 216, 237, 258, 278, 298, 317, 335, 351, 366, 380, 393, 403, 410, 414, 414, 411, 407, 402, 397, 389, 377, 360, 339, 315, 291, 266, 242, 218, 194, 169, 144, 121, 101, 84, 71, 61, 54, 48, 42, 37, 33, 29, 25, 22, 20, 17, 16, 15, 15, 15, 14, 13, 11, 9, 7, 6, 7, 8, 10, 11, 11, 9, 7, 4, 1, 0, 0, 0, 1, 1, 1, 0, 0, 1, 4, 10, 16, 24, 32, 39, 47, 56, 66, 80, 98, 120, 146, 175, 206, 237, 266, 291, 312, 327, 334, 333, 324, 307, 283, 255, 224, 193, 163, 137, 115, 97, 82, 72, 68, 69, 78, 94, 116, 141, 167, 192, 214, 234, 252, 270, 285, 296, 299, 293, 276, 251, 220, 187, 155, 126, 99, 76, 55, 38, 24, 15, 10, 9, 12, 19, 30, 45, 63, 84, 107, 134, 163, 194, 227, 261, 293, 322, 343, 354, 353, 338, 313, 278, 239, 198, 158, 122, 90, 65, 44, 29, 17, 9, 4, 1, 0, 2, 6, 12, 19, 28, 38, 49, 62, 79, 102, 131, 168, 212, 259, 307, 352, 388, 413, 424, 422, 407, 379, 342, 298, 250, 203, 158, 120, 88, 63, 45, 32, 23, 17, 13, 12, 13, 14, 16, 17, 18, 17, 16, 14, 13, 12, 13, 14, 16, 17, 17, 17, 15, 14, 13, 11, 10, 8, 7, 5, 5, 7, 14, 26, 45, 71, 103, 141, 184, 229, 275, 321, 363, 400, 426, 439, 435, 414, 380, 339, 302, 276, 269, 282, 310, 346, 383, 412, 428, 430, 415, 385, 342, 290, 233, 175, 122, 78, 45, 23, 10, 3, 1, 0, 0, 0, 1, 3, 6, 11, 16, 23, 31, 43, 60, 87, 124, 173, 232, 297, 362, 420, 465, 492, 499, 487, 460, 427, 394, 369, 358, 365, 389, 423, 462, 496, 516, 515, 491, 444, 379, 302, 223, 150, 89, 44, 16, 3, 0, 0, 0, 1, 0, 0, 0, 0, 1, 2, 4, 4, 5, 7, 8, 10, 11, 11, 10, 9, 8, 8, 10, 13, 15, 17, 17, 14, 10, 5, 0, 0, 2, 12, 30, 59, 99, 153, 218, 292, 368, 440, 497, 531, 538, 513, 461, 389, 304, 219, 143, 82, 38, 11, 0, 0, 6, 28, 67, 122, 193, 274, 358, 435, 496, 533, 543, 523, 477, 409, 328, 242, 163, 97, 48, 18, 3, 0, 0, 1, 1, 1, 0, 0, 0, 2, 6, 9, 13, 15, 14, 11, 7, 5, 9, 22, 50, 93, 151, 221, 297, 369, 430, 472, 491, 485, 454, 404, 341, 270, 201, 138, 86, 48, 23, 9, 4, 4, 7, 11, 14, 15, 14, 12, 8, 5, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 5, 7, 7, 6, 5, 4, 4, 5, 7, 10, 13, 16, 19, 21, 22, 22, 20, 17, 13, 10, 6, 4, 1, 0, 0, 1, 9, 27, 60, 110, 179, 265, 361, 456, 536, 588, 600, 571, 507, 421, 333, 264, 228, 235, 282, 357, 445, 526, 582, 602, 580, 521, 435, 334, 234, 146, 80, 35, 11, 1, 0, 0, 0, 0, 0, 0, 0, 1, 2, 2, 2, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 5, 21, 52, 104, 180, 276, 383, 488, 572, 619, 617, 566, 476, 368, 266, 197, 176, 211, 296, 413, 538, 647, 716, 731, 689, 598, 473, 338, 213, 114, 47, 11, 0, 0, 0, 2, 2, 3, 9, 25, 56, 105, 174, 260, 355, 444, 514, 550, 547, 505, 432, 343, 251, 167, 99, 49, 18, 2, 0, 1, 9, 17, 23, 25, 24, 21, 17, 13, 11, 11, 12, 12, 11, 8, 4, 1, 0, 1, 3, 6, 8, 10, 10, 10, 10, 11, 13, 15, 18, 20, 21, 22, 22, 22, 22, 21, 21, 20, 19, 16, 12, 6, 0, 0, 8, 38, 97, 187, 304, 433, 553, 641, 678, 656, 578, 462, 332, 217, 143, 124, 167, 263, 395, 537, 661, 741, 759, 711, 606, 466, 322, 207, 146, 152, 223, 341, 478, 604, 692, 722, 688, 600, 474, 336, 208, 107, 41, 7, 0, 0, 1, 4, 4, 1, 0, 0, 4, 30, 82, 164, 273, 395, 510, 595, 632, 611, 537, 425, 299, 181, 89, 30, 3, 0, 0, 3, 4, 2, 0, 0, 0, 0, 1, 3, 5, 6, 6, 6, 5, 5, 4, 5, 6, 6, 7, 7, 7, 8, 8, 7, 4, 0, 0, 8, 38, 99, 196, 324, 467, 601, 698, 737, 705, 609, 469, 319, 195, 127, 134, 216, 358, 530, 697, 821, 878, 856, 760, 612, 442, 279, 148, 59, 12, 0, 0, 3, 6, 4, 0, 0, 3, 27, 75, 149, 245, 354, 459, 542, 587, 587, 540, 456, 353, 248, 156, 86, 41, 18, 11, 13, 20, 29, 39, 47, 55, 60, 64, 68, 72, 77, 81, 83, 82, 78, 76, 84, 114, 173, 266, 391, 532, 669, 777, 833, 826, 757, 643, 512, 400, 337, 342, 418, 549, 704, 846, 942, 968, 918, 804, 650, 487, 341, 232, 164, 132, 124, 127, 130, 128, 121, 112, 103, 95, 88, 82, 75, 66, 54, 39, 23, 9, 1, 8, 38, 100, 195, 317, 449, 568, 650, 675, 636, 541, 412, 274, 153, 65, 15, 0, 0, 1, 7, 8, 4, 0, 0, 1, 27, 81, 166, 281, 411, 534, 628, 671, 653, 577, 459, 323, 193, 92, 27, 0, 0, 3, 8, 6, 0, 0, 15, 68, 161, 288, 430, 562, 656, 690, 655, 561, 428, 285, 159, 68, 16, 0, 0, 2, 6, 7, 6, 7, 10, 14, 18, 19, 17, 14, 9, 5, 3, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 4, 7, 10, 12, 13, 13, 12, 9, 5, 0, 0, 0, 0, 4, 6, 3, 0, 0, 12, 57, 140, 258, 398, 533, 632, 672, 641, 548, 414, 270, 144, 56, 9, 0, 0, 5, 8, 7, 2, 0, 0, 2, 6, 10, 12, 13, 13, 12, 11, 9, 7, 4, 2, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 10, 13, 16, 17, 15, 11, 7, 5, 6, 8, 8, 4, 0, 0, 11, 59, 147, 273, 419, 555, 652, 684, 643, 539, 399, 254, 132, 49, 6, 0, 0, 5, 7, 4, 0, 0, 0, 4, 11, 16, 18, 17, 14, 11, 9, 6, 3, 0, 0, 5, 29, 78, 153, 250, 356, 449, 510, 523, 483, 400, 291, 179, 87, 26, 0, 0, 3, 8, 6, 0, 0, 23, 87, 195, 339, 496, 635, 723, 740, 680, 558, 402, 246, 118, 35, 0, 0, 3, 16, 18, 9, 0, 0, 24, 103, 234, 407, 592, 749, 841, 841, 746, 577, 374, 183, 50, 3, 54, 191, 382, 584, 750, 844, 843, 747, 580, 380, 197, 73, 37, 96, 233, 413, 590, 719, 771, 736, 626, 470, 306, 164, 66, 13, 0, 0, 6, 9, 7, 2, 0, 0, 1, 7, 12, 15, 15, 13, 10, 6, 3, 2, 2, 2, 2, 0, 0, 0, 1, 8, 17, 22, 18, 6, 0, 0, 42, 144, 305, 507, 712, 874, 950, 919, 787, 585, 363, 176, 68, 61, 155, 324, 525, 709, 832, 865, 799, 651, 454, 255, 100, 24, 46, 161, 340, 542, 717, 825, 841, 765, 618, 438, 263, 125, 39, 0, 0, 2, 9, 11, 8, 4, 4, 8, 14, 20, 24, 25, 24, 24, 23, 22, 20, 17, 12, 7, 2, 0, 0, 2, 6, 11, 16, 20, 23, 25, 25, 23, 20, 16, 14, 16, 21, 25, 23, 12, 0, 0, 18, 96, 235, 424, 629, 807, 911, 913, 809, 624, 402, 196, 54, 7, 63, 207, 403, 602, 757, 831, 809, 700, 533, 350, 189, 75, 14, 0, 0, 7, 12, 10, 6, 3, 5, 10, 16, 20, 22, 21, 21, 21, 21, 20, 17, 14, 10, 7, 6, 8, 10, 13, 16, 20, 24, 30, 35, 35, 26, 11, 0, 0, 34, 125, 274, 465, 665, 829, 918, 908, 802, 627, 426, 242, 104, 24, 0, 0, 6, 15, 14, 8, 1, 0, 0, 2, 8, 12, 13, 11, 9, 7, 8, 11, 14, 12, 4, 0, 0, 28, 102, 225, 385, 555, 697, 775, 763, 662, 495, 303, 137, 39, 38, 136, 309, 514, 702, 825, 856, 789, 645, 462, 281, 136, 43, 1, 0, 2, 11, 14, 12, 10, 13, 22, 33, 42, 46, 43, 36, 28, 22, 18, 14, 7, 0, 0, 10, 62, 161, 303, 464, 613, 711, 731, 668, 539, 376, 217, 93, 18, 0, 0, 9, 23, 23, 10, 0, 0, 21, 105, 244, 423, 607, 753, 822, 795, 680, 509, 325, 167, 59, 5, 0, 0, 8, 11, 7, 1, 0, 0, 1, 5, 6, 4, 0, 0, 0, 5, 10, 10, 3, 0, 0, 34, 116, 249, 418, 590, 723, 782, 747, 624, 441, 242, 78, 0, 0, 108, 294, 514, 715, 850, 888, 823, 676, 487, 299, 148, 50, 4, 0, 0, 8, 9, 5, 0, 0, 0, 2, 8, 13, 15, 15, 13, 10, 6, 2, 0, 0, 1, 8, 17, 26, 33, 35, 32, 25, 16, 7, 2, 0, 0, 0, 2, 4, 6, 9, 11, 12, 12, 10, 7, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 8, 15, 22, 26, 25, 20, 13, 8, 8, 11, 14, 10, 0, 0, 0, 46, 145, 293, 466, 626, 730, 748, 676, 533, 360, 198, 79, 13, 0, 0, 7, 13, 11, 5, 2, 3, 9, 16, 21, 23, 23, 20, 17, 14, 11, 8, 5, 2, 0, 0, 0, 2, 7, 11, 16, 20, 23, 25, 25, 25, 22, 17, 10, 5, 4, 9, 15, 19, 13, 0, 0, 2, 64, 189, 372, 584, 778, 903, 926, 839, 668, 458, 261, 112, 26, 0, 0, 5, 11, 9, 4, 2, 3, 6, 4, 0, 0, 17, 71, 169, 306, 458, 592, 675, 681, 607, 466, 294, 130, 16, 0, 40, 184, 383, 594, 767, 857, 844, 733, 557, 362, 191, 72, 10, 0, 0, 5, 7, 5, 2, 2, 4, 3, 0, 0, 12, 62, 158, 296, 453, 594, 683, 691, 612, 463, 282, 116, 9, 0, 73, 231, 429, 616, 745, 784, 728, 594, 421, 249, 113, 28, 0, 0, 7, 22, 25, 14, 0, 0, 7, 75, 200, 371, 559, 722, 815, 810, 704, 522, 308, 118, 1, 0, 81, 263, 489, 703, 853, 901, 839, 688, 488, 290, 133, 36, 0, 0, 4, 13, 13, 7, 0, 0, 0, 0, 3, 5, 4, 3, 2, 1, 1, 1, 2, 2, 2, 2, 1, 0, 0, 0, 0, 1, 3, 4, 5, 6, 6, 5, 5, 5, 7, 10, 13, 15, 15, 13, 11, 8, 6, 4, 3, 2, 0, 0, 0, 0, 7, 12, 12, 4, 0, 0, 35, 123, 265, 441, 615, 743, 787, 732, 588, 391, 191, 40, 0, 29, 186, 415, 666, 877, 995, 992, 872, 670, 440, 235, 90, 13, 0, 0, 10, 16, 12, 3, 0, 0, 0, 1, 4, 5, 4, 3, 3, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 6, 9, 11, 12, 12, 12, 11, 9, 8, 6, 5, 3, 2, 0, 0, 0, 0, 3, 6, 11, 16, 20, 23, 23, 22, 20, 16, 13, 10, 9, 8, 9, 10, 11, 10, 9, 7, 7, 8, 12, 16, 20, 21, 18, 14, 11, 10, 11, 10, 5, 0, 0, 14, 69, 171, 313, 474, 618, 709, 719, 641, 491, 306, 130, 11, 0, 53, 219, 443, 674, 855, 942, 913, 779, 578, 363, 181, 59, 1, 0, 0, 12, 15, 10, 1, 0, 0, 0, 2, 4, 5, 4, 4, 5, 7, 8, 10, 10, 11, 11, 12, 13, 14, 14, 13, 11, 8, 5, 2, 0, 0, 0, 0, 1, 2, 3, 3, 3, 4, 5, 7, 8, 9, 9, 8, 6, 4, 2, 1, 0, 0, 0, 0, 2, 5, 9, 11, 10, 6, 0, 0, 0, 3, 9, 11, 5, 0, 0, 20, 93, 217, 379, 548, 681, 741, 711, 599, 437, 266, 126, 36, 0, 0, 2, 12, 13, 6, 0, 0, 0, 4, 11, 15, 15, 12, 7, 2, 0, 0, 1, 5, 11, 15, 17, 16, 12, 7, 2, 0, 1, 4, 10, 15, 19, 21, 22, 21, 20, 18, 16, 14, 11, 7, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 4, 9, 14, 18, 18, 14, 8, 1, 0, 0, 4, 10, 12, 6, 0, 0, 21, 94, 220, 386, 559, 699, 765, 740, 629, 464, 287, 139, 41, 0, 0, 2, 12, 14, 8, 1, 0, 0, 3, 8, 10, 9, 6, 3, 2, 3, 5, 9, 12, 13, 12, 10, 7, 4, 1, 0, 0, 0, 0, 3, 6, 8, 8, 5, 0, 0, 0, 4, 12, 16, 10, 0, 0, 12, 91, 237, 438, 658, 843, 943, 929, 805, 606, 386, 194, 64, 0, 0, 0, 15, 20, 14, 5, 0, 0, 4, 10, 13, 12, 9, 4, 1, 1, 2, 4, 5, 6, 6, 7, 9, 13, 16, 18, 16, 12, 7, 6, 9, 12, 12, 5, 0, 0, 20, 90, 207, 360, 516, 636, 686, 650, 540, 386, 229, 103, 25, 0, 0, 4, 11, 11, 4, 0, 0, 5, 13, 21, 24, 21, 14, 5, 0, 0, 0, 6, 14, 16, 9, 0, 0, 18, 89, 211, 373, 541, 672, 727, 685, 553, 363, 171, 31, 0, 60, 235, 471, 707, 882, 950, 898, 744, 533, 320, 149, 42, 0, 0, 4, 14, 13, 6, 0, 0, 1, 8, 14, 16, 13, 8, 2, 0, 0, 0, 0, 2, 4, 5, 3, 0, 0, 0, 0, 4, 12, 15, 9, 0, 0, 13, 89, 229, 421, 630, 805, 898, 882, 761, 570, 359, 178, 55, 0, 0, 2, 19, 26, 23, 15, 8, 6, 9, 13, 17, 18, 16, 12, 8, 4, 2, 2, 3, 6, 9, 11, 13, 13, 12, 11, 10, 10, 11, 13, 16, 19, 21, 21, 17, 10, 2, 0, 0, 1, 9, 13, 10, 0, 0, 3, 59, 168, 324, 500, 655, 748, 753, 668, 519, 345, 187, 73, 12, 0, 0, 7, 11, 8, 1, 0, 0, 4, 14, 22, 28, 29, 27, 23, 19, 15, 12, 11, 11, 12, 13, 15, 15, 13, 9, 7, 9, 16, 25, 28, 21, 4, 0, 0, 62, 195, 397, 637, 864, 1018, 1055, 958, 751, 485, 231, 52, 0, 61, 237, 470, 697, 857, 909, 840, 666, 435, 209, 48, 0, 74, 261, 511, 760, 944, 1014, 953, 776, 531, 281, 89, 1, 38, 185, 402, 632, 815, 903, 873, 731, 513, 279, 92, 5, 45, 204, 442, 695, 895, 987, 947, 786, 549, 301, 110, 25, 65, 215, 428, 641, 795, 849, 796, 654, 467, 281, 133, 40, 0, 0, 3, 11, 11, 5, 0, 0, 0, 6, 13, 18, 19, 17, 14, 13, 14, 18, 24, 27, 26, 22, 15, 11, 13, 18, 23, 19, 6, 0, 0, 48, 168, 358, 592, 818, 976, 1018, 927, 723, 459, 206, 31, 0, 66, 265, 526, 781, 964, 1029, 965, 793, 565, 336, 154, 42, 0, 0, 6, 17, 18, 10, 0, 0, 0, 2, 11, 20, 28, 32, 33, 31, 26, 19, 12, 7, 3, 1, 1, 3, 7, 12, 18, 25, 29, 32, 31, 28, 25, 24, 25, 29, 34, 37, 38, 38, 39, 42, 43, 38, 23, 1, 0, 4, 80, 230, 444, 686, 898, 1022, 1017, 880, 642, 368, 130, 0, 0, 138, 382, 665, 915, 1066, 1078, 950, 715, 440, 200, 61, 58, 190, 416, 669, 875, 976, 943, 787, 551, 302, 107, 18, 55, 203, 418, 637, 799, 861, 813, 672, 481, 288, 134, 38, 0, 0, 7, 19, 23, 18, 11, 6, 6, 9, 13, 14, 13, 9, 5, 1, 0, 0, 1, 4, 8, 12, 15, 16, 15, 12, 8, 3, 0, 0, 0, 4, 9, 15, 18, 19, 18, 13, 8, 3, 0, 0, 0, 2, 5, 9, 11, 11, 9, 6, 3, 0, 0, 0, 2, 5, 8, 9, 9, 7, 4, 2, 0, 0, 1, 3, 3, 4, 4, 6, 12, 21, 29, 31, 22, 3, 0, 0, 56, 183, 371, 590, 788, 910, 922, 820, 634, 415, 217, 77, 5, 0, 0, 15, 24, 24, 21, 19, 16, 9, 0, 0, 5, 57, 159, 303, 464, 602, 679, 671, 583, 438, 277, 138, 43, 0, 0, 2, 15, 22, 21, 15, 8, 4, 3, 5, 8, 12, 16, 22, 29, 36, 42, 43, 35, 18, 0, 0, 11, 85, 218, 397, 587, 744, 825, 808, 697, 525, 335, 171, 59, 3, 0, 0, 10, 15, 11, 3, 0, 0, 0, 2, 8, 12, 16, 18, 18, 16, 13, 9, 5, 2, 0, 0, 0, 1, 4, 9, 14, 17, 17, 14, 9, 6, 6, 8, 12, 16, 19, 22, 25, 29, 29, 21, 4, 0, 0, 41, 151, 322, 528, 722, 851, 879, 797, 630, 426, 236, 95, 15, 0, 0, 3, 14, 19, 17, 9, 0, 0, 0, 46, 152, 313, 504, 681, 796, 814, 730, 569, 377, 200, 73, 6, 0, 0, 13, 21, 18, 11, 4, 0, 1, 5, 9, 13, 17, 21, 26, 29, 31, 29, 25, 19, 13, 10, 9, 9, 9, 9, 11, 15, 23, 30, 32, 22, 2, 0, 0, 71, 217, 432, 681, 907, 1049, 1064, 943, 715, 439, 187, 22, 0, 68, 258, 496, 719, 867, 901, 811, 622, 386, 167, 27, 6, 109, 312, 561, 789, 938, 971, 885, 708, 489, 282, 124, 31, 0, 0, 2, 10, 12, 10, 7, 2, 0, 0, 0, 39, 122, 252, 412, 568, 679, 714, 662, 538, 375, 216, 92, 19, 0, 0, 7, 15, 14, 7, 0, 0, 0, 0, 2, 3, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 4, 4, 4, 4, 4, 5, 6, 6, 5, 4, 2, 0, 0, 0, 0, 1, 3, 4, 3, 2, 1, 3, 7, 12, 14, 9, 0, 0, 0, 42, 127, 251, 396, 528, 612, 625, 563, 444, 300, 165, 66, 11, 0, 0, 6, 10, 9, 3, 0, 0, 4, 9, 15, 19, 21, 24, 26, 26, 25, 20, 15, 9, 4, 2, 2, 3, 3, 3, 5, 9, 16, 22, 21, 12, 0, 0, 21, 102, 246, 443, 657, 841, 945, 936, 809, 592, 341, 122, 0, 1, 140, 383, 669, 926, 1086, 1109, 988, 755, 471, 207, 33, 0, 90, 303, 573, 828, 999, 1045, 958, 769, 531, 303, 130, 30, 0, 0, 9, 17, 15, 6, 0, 0, 0, 1, 6, 9, 10, 10, 12, 14, 18, 21, 24, 25, 24, 21, 19, 17, 16, 17, 19, 23, 28, 32, 33, 26, 11, 0, 0, 26, 114, 261, 450, 645, 798, 866, 830, 699, 511, 313, 149, 43, 0, 0, 2, 16, 21, 20, 16, 15, 17, 20, 22, 21, 17, 14, 11, 10, 10, 10, 10, 10, 10, 10, 9, 9, 9, 13, 18, 22, 21, 12, 0, 0, 9, 74, 197, 366, 551, 709, 798, 792, 692, 528, 342, 178, 64, 5, 0, 0, 10, 15, 11, 5, 2, 3, 7, 10, 11, 8, 5, 3, 6, 11, 18, 24, 26, 24, 19, 13, 7, 3, 0, 1, 5, 11, 18, 22, 17, 5, 0, 0, 34, 127, 273, 453, 628, 753, 792, 734, 594, 413, 236, 99, 20, 0, 0, 7, 15, 13, 6, 0, 0, 0, 3, 9, 14, 17, 19, 20, 21, 23, 25, 25, 21, 11, 0, 0, 7, 54, 142, 263, 399, 518, 591, 598, 536, 423, 288, 163, 68, 14, 0, 0, 5, 10, 10, 6, 1, 0, 0, 0, 3, 8, 13, 18, 22, 25, 25, 23, 18, 13, 8, 5, 5, 7, 13, 22, 31, 35, 30, 14, 0, 0, 29, 128, 296, 514, 742, 925, 1012, 973, 813, 570, 309, 100, 1, 41, 211, 467, 740, 958, 1066, 1040, 893, 669, 427, 219, 77, 6, 0, 0, 15, 22, 18, 9, 2, 0, 2, 6, 8, 9, 7, 6, 5, 5, 5, 6, 5, 4, 3, 3, 3, 2, 2, 2, 2, 1, 1, 1, 0, 0, 0, 0, 1, 5, 11, 18, 24, 27, 26, 22, 16, 10, 4, 1, 0, 0, 0, 1, 3, 4, 4, 3, 1, 0, 0, 2, 7, 15, 23, 30, 34, 35, 33, 29, 24, 19, 15, 12, 10, 8, 8, 10, 12, 15, 18, 20, 21, 21, 21, 19, 17, 14, 14, 16, 22, 27, 26, 16, 0, 0, 5, 75, 208, 399, 613, 801, 914, 920, 816, 634, 421, 228, 89, 14, 0, 0, 5, 12, 13, 11, 9, 6, 0, 0, 0, 36, 120, 254, 423, 595, 727, 784, 747, 622, 440, 245, 86, 4, 18, 123, 293, 481, 640, 728, 725, 636, 488, 321, 173, 67, 10, 0, 0, 8, 12, 10, 4, 0, 1, 5, 11, 16, 20, 22, 23, 22, 21, 18, 14, 10, 8, 10, 15, 23, 33, 41, 45, 44, 37, 27, 17, 9, 4, 4, 6, 11, 16, 21, 25, 26, 26, 22, 17, 12, 7, 3, 2, 3, 4, 5, 4, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 7, 11, 14, 16, 16, 13, 9, 4, 0, 0, 1, 6, 12, 17, 18, 16, 13, 12, 14, 18, 19, 12, 0, 0, 4, 62, 178, 343, 531, 697, 798, 804, 715, 555, 368, 198, 76, 10, 0, 0, 9, 14, 10, 2, 0, 0, 1, 8, 14, 17, 15, 12, 7, 4, 3, 3, 6, 8, 11, 13, 15, 16, 15, 14, 12, 9, 6, 3, 0, 0, 0, 2, 6, 10, 14, 16, 15, 12, 9, 5, 2, 2, 3, 5, 7, 8, 8, 7, 5, 3, 1, 0, 0, 0, 2, 6, 9, 11, 12, 11, 10, 10, 10, 9, 4, 0, 0, 8, 46, 118, 221, 339, 449, 523, 544, 504, 415, 299, 182, 88, 27, 0, 0, 3, 11, 13, 10, 4, 0, 0, 0, 4, 9, 13, 15, 15, 13, 11, 9, 7, 6, 4, 2, 1, 1, 4, 9, 14, 19, 21, 21, 18, 15, 12, 11, 11, 13, 14, 14, 12, 9, 5, 1, 0, 0, 2, 8, 15, 22, 27, 30, 30, 29, 29, 27, 21, 10, 0, 0, 15, 81, 199, 361, 540, 696, 791, 795, 706, 543, 347, 167, 49, 20, 86, 229, 413, 588, 713, 754, 706, 583, 420, 255, 121, 36, 0, 0, 2, 10, 11, 6, 0, 0, 0, 1, 5, 8, 9, 10, 10, 11, 11, 10, 8, 7, 6, 6, 7, 8, 8, 9, 10, 12, 15, 19, 22, 23, 22, 19, 14, 9, 5, 1, 1, 3, 9, 16, 19, 16, 4, 0, 0, 34, 121, 258, 428, 596, 720, 768, 725, 604, 438, 269, 130, 40, 0, 0, 1, 10, 13, 11, 7, 5, 5, 7, 8, 8, 7, 6, 7, 9, 10, 11, 10, 7, 4, 2, 0, 1, 3, 5, 9, 12, 16, 20, 23, 26, 28, 27, 25, 21, 16, 12, 8, 6, 5, 5, 6, 6, 5, 4, 3, 2, 2, 3, 6, 9, 12, 13, 12, 11, 9, 8, 7, 8, 9, 9, 9, 8, 7, 6, 6, 7, 8, 7, 5, 3, 1, 0, 2, 6, 12, 18, 24, 27, 26, 23, 18, 11, 6, 2, 0, 0, 0, 0, 1, 2, 3, 2, 1, 0, 0, 0, 1, 6, 9, 7, 0, 0, 2, 42, 123, 243, 382, 511, 598, 619, 569, 461, 324, 191, 87, 24, 0, 0, 2, 7, 8, 4, 0, 0, 0, 0, 3, 6, 7, 8, 8, 8, 8, 7, 6, 3, 1, 0, 0, 0, 2, 5, 9, 11, 11, 10, 7, 4, 2, 0, 0, 0, 0, 0, 1, 3, 7, 12, 16, 18, 18, 15, 10, 6, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 5, 9, 14, 19, 23, 22, 15, 4, 0, 0, 28, 95, 198, 324, 447, 538, 573, 542, 454, 331, 205, 100, 31, 0, 0, 1, 10, 13, 12, 8, 4, 3, 4, 4, 2, 0, 0, 0, 5, 11, 14, 9, 0, 0, 4, 53, 152, 298, 470, 632, 746, 782, 731, 607, 442, 274, 137, 46, 2, 0, 0, 10, 14, 13, 10, 6, 5, 6, 7, 9, 9, 10, 10, 11, 12, 13, 13, 9, 1, 0, 0, 23, 79, 169, 284, 406, 508, 565, 561, 496, 389, 264, 150, 65, 15, 0, 0, 0, 4, 7, 8, 6, 1, 0, 0, 12, 65, 160, 291, 439, 570, 652, 662, 599, 479, 333, 194, 87, 23, 0, 0, 2, 8, 8, 4, 0, 0, 0, 0, 1, 3, 5, 7, 8, 9, 9, 9, 8, 9, 10, 11, 13, 14, 16, 20, 22, 21, 13, 0, 0, 2, 52, 154, 308, 492, 673, 808, 863, 826, 705, 531, 347, 188, 76, 15, 0, 0, 1, 8, 11, 9, 3, 0, 0, 7, 55, 147, 285, 452, 615, 736, 782, 737, 608, 426, 239, 95, 31, 65, 186, 362, 545, 687, 752, 727, 622, 467, 302, 160, 62, 10, 0, 0, 6, 10, 10, 8, 9, 12, 17, 20, 22, 20, 17, 14, 11, 9, 7, 5, 3, 1, 0, 0, 0, 0, 0, 2, 5, 7, 9, 10, 10, 9, 7, 6, 5, 5, 5, 5, 3, 2, 0, 0, 0, 3, 8, 14, 19, 22, 21, 18, 13, 7, 2, 0, 1, 5, 10, 15, 17, 18, 17, 17, 17, 16, 10, 1, 0, 0, 34, 111, 234, 390, 552, 686, 759, 754, 672, 534, 371, 219, 101, 28, 0, 0, 1, 13, 18, 12, 0, 0, 0, 41, 135, 279, 456, 633, 768, 828, 796, 681, 512, 331, 174, 64, 6, 0, 0, 7, 16, 16, 7, 0, 0, 6, 58, 159, 309, 492, 676, 822, 895, 875, 766, 594, 400, 224, 94, 20, 0, 0, 8, 18, 21, 18, 13, 10, 9, 9, 10, 11, 11, 11, 11, 13, 14, 15, 14, 9, 2, 0, 2, 31, 94, 197, 333, 485, 622, 715, 738, 680, 552, 383, 216, 97, 62, 126, 278, 481, 686, 840, 904, 866, 736, 552, 356, 188, 71, 11, 0, 0, 11, 16, 13, 7, 3, 4, 11, 20, 28, 33, 33, 29, 23, 17, 11, 7, 3, 1, 0, 0, 2, 5, 9, 13, 14, 13, 10, 7, 6, 8, 12, 17, 21, 25, 28, 31, 32, 30, 22, 9, 0, 0, 24, 101, 229, 400, 586, 748, 848, 861, 784, 638, 457, 281, 139, 48, 4, 0, 0, 6, 8, 4, 0, 0, 0, 0, 2, 3, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 5, 8, 11, 12, 12, 10, 8, 7, 9, 12, 18, 24, 29, 32, 31, 27, 20, 14, 8, 4, 3, 4, 7, 11, 16, 19, 21, 23, 23, 23, 22, 22, 22, 21, 20, 18, 16, 14, 13, 12, 11, 9, 6, 3, 0, 0, 0, 3, 8, 13, 16, 17, 15, 12, 9, 9, 12, 16, 20, 23, 25, 24, 22, 19, 17, 16, 16, 18, 20, 21, 20, 18, 15, 14, 15, 17, 16, 11, 0, 0, 4, 53, 153, 305, 489, 670, 806, 862, 819, 686, 496, 299, 146, 80, 122, 268, 489, 734, 950, 1085, 1111, 1022, 843, 614, 386, 199, 74, 10, 0, 0, 7, 11, 9, 5, 3, 6, 10, 13, 12, 8, 5, 4, 7, 13, 19, 22, 21, 17, 13, 10, 10, 13, 19, 23, 25, 22, 17, 10, 4, 0, 0, 1, 4, 6, 7, 5, 3, 1, 0, 0, 1, 2, 3, 3, 2, 0, 0, 0, 1, 3, 5, 7, 8, 8, 8, 6, 5, 4, 6, 10, 14, 17, 15, 7, 0, 0, 20, 82, 189, 335, 499, 649, 753, 784, 736, 620, 463, 300, 160, 63, 11, 0, 0, 8, 15, 17, 16, 14, 13, 14, 15, 17, 19, 22, 26, 31, 34, 33, 27, 15, 4, 5, 28, 84, 178, 303, 440, 565, 649, 674, 634, 537, 406, 270, 151, 66, 17, 0, 0, 0, 4, 7, 6, 2, 0, 0, 8, 45, 116, 225, 359, 496, 607, 664, 651, 569, 439, 294, 175, 117, 139, 240, 398, 572, 720, 803, 799, 711, 562, 392, 246, 163, 167, 256, 407, 582, 734, 821, 819, 731, 579, 407, 265, 192, 210, 310, 465, 627, 752, 806, 777, 674, 524, 361, 215, 106, 38, 6, 0, 2, 10, 16, 21, 25, 29, 32, 35, 37, 36, 34, 31, 27, 23, 18, 13, 8, 5, 2, 1, 2, 3, 4, 4, 4, 3, 3, 4, 7, 11, 16, 21, 25, 28, 29, 27, 24, 19, 14, 9, 5, 2, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 5, 5, 4, 2, 1, 0, 0, 1, 3, 4, 5, 6, 6, 7, 8, 10, 12, 15, 19, 22, 23, 19, 9, 0, 0, 13, 70, 172, 315, 480, 639, 758, 813, 791, 698, 555, 392, 239, 117, 38, 0, 0, 0, 12, 20, 19, 11, 0, 0, 4, 45, 124, 241, 380, 517, 622, 671, 652, 569, 442, 300, 172, 77, 21, 0, 0, 3, 9, 10, 8, 6, 5, 6, 9, 11, 12, 11, 11, 10, 9, 8, 7, 6, 4, 3, 1, 0, 0, 0, 1, 4, 9, 12, 10, 4, 0, 0, 17, 69, 161, 286, 427, 558, 649, 680, 644, 550, 419, 279, 157, 69, 18, 0, 0, 3, 8, 8, 5, 1, 0, 0, 0, 3, 5, 7, 8, 10, 11, 12, 10, 5, 0, 0, 7, 38, 96, 178, 276, 373, 451, 492, 487, 438, 356, 258, 163, 86, 35, 7, 0, 0, 1, 5, 8, 12, 17, 22, 25, 25, 21, 15, 8, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 8, 13, 19, 24, 27, 28, 28, 27, 25, 25, 25, 26, 23, 16, 5, 0, 1, 32, 103, 218, 368, 531, 676, 771, 790, 727, 596, 429, 271, 163, 135, 198, 336, 515, 689, 813, 856, 808, 684, 519, 354, 233, 182, 210, 306, 440, 576, 680, 726, 703, 617, 488, 342, 207, 103, 36, 4, 0, 0, 3, 5, 4, 3, 5, 9, 15, 19, 22, 24, 24, 23, 21, 18, 14, 9, 5, 2, 1, 1, 0, 0, 0, 0, 0, 3, 5, 3, 0, 0, 4, 37, 103, 203, 326, 451, 553, 607, 602, 539, 434, 310, 191, 97, 35, 4, 0, 0, 7, 13, 15, 16, 16, 15, 14, 12, 10, 8, 6, 5, 4, 3, 2, 2, 1, 2, 4, 8, 13, 19, 24, 25, 20, 10, 0, 0, 11, 57, 139, 251, 379, 501, 593, 638, 629, 568, 469, 351, 234, 135, 63, 19, 0, 0, 0, 2, 3, 3, 4, 6, 11, 16, 20, 23, 24, 24, 21, 16, 7, 0, 0, 12, 56, 135, 246, 377, 505, 605, 654, 642, 572, 460, 329, 204, 103, 37, 3, 0, 0, 5, 8, 5, 0, 0, 13, 58, 141, 261, 408, 560, 688, 764, 769, 699, 571, 413, 265, 164, 137, 190, 310, 464, 613, 718, 752, 709, 601, 455, 302, 170, 76, 22, 0, 0, 2, 4, 4, 0, 0, 0, 0, 3, 8, 12, 15, 16, 16, 14, 11, 9, 8, 8, 10, 12, 13, 12, 12, 12, 12, 13, 13, 9, 2, 0, 0, 25, 83, 178, 301, 437, 559, 642, 667, 629, 536, 409, 274, 156, 69, 19, 0, 0, 1, 4, 4, 1, 0, 0, 0, 0, 1, 1, 2, 3, 5, 7, 8, 8, 7, 5, 3, 3, 3, 4, 6, 7, 8, 7, 6, 4, 2, 1, 1, 2, 4, 6, 8, 10, 11, 11, 11, 10, 7, 2, 0, 3, 22, 65, 136, 232, 343, 448, 527, 561, 541, 472, 368, 252, 147, 67, 19, 0, 0, 1, 6, 7, 6, 6, 7, 11, 15, 17, 16, 14, 11, 9, 8, 8, 6, 1, 0, 0, 19, 66, 147, 256, 382, 503, 595, 639, 627, 560, 454, 331, 212, 116, 49, 12, 0, 0, 2, 5, 6, 4, 2, 0, 0, 0, 1, 2, 5, 7, 10, 12, 13, 12, 11, 9, 7, 5, 3, 3, 4, 6, 9, 13, 16, 18, 18, 16, 14, 12, 13, 15, 20, 26, 30, 31, 27, 18, 6, 0, 0, 20, 69, 148, 253, 370, 483, 569, 613, 606, 551, 457, 344, 231, 134, 62, 19, 0, 0, 0, 5, 7, 3, 0, 0, 5, 38, 99, 187, 294, 402, 493, 548, 558, 521, 446, 347, 243, 150, 77, 30, 6, 0, 0, 1, 4, 5, 5, 6, 6, 7, 8, 9, 11, 13, 16, 18, 19, 18, 18, 17, 16, 16, 17, 18, 19, 19, 19, 18, 14, 8, 1, 0, 10, 46, 115, 218, 346, 481, 597, 667, 674, 618, 511, 383, 267, 194, 182, 234, 335, 457, 571, 649, 671, 635, 549, 431, 304, 189, 100, 42, 11, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 6, 9, 12, 14, 15, 15, 14, 14, 15, 15, 16, 17, 16, 13, 10, 7, 4, 4, 6, 9, 13, 17, 18, 17, 14, 10, 7, 5, 5, 6, 7, 8, 7, 5, 3, 3, 5, 8, 12, 16, 17, 17, 14, 11, 7, 4, 2, 0, 0, 0, 2, 4, 5, 3, 0, 0, 5, 30, 81, 160, 260, 368, 464, 531, 555, 532, 465, 370, 263, 163, 83, 30, 3, 0, 0, 5, 5, 0, 0, 0, 28, 88, 180, 298, 423, 533, 607, 630, 598, 519, 409, 290, 181, 95, 38, 9, 0, 0, 1, 4, 4, 3, 2, 1, 2, 3, 5, 7, 10, 13, 14, 13, 12, 9, 7, 6, 5, 5, 5, 4, 4, 4, 3, 1, 0, 0, 7, 34, 88, 171, 282, 405, 521, 607, 646, 629, 559, 450, 325, 206, 110, 45, 11, 0, 0, 1, 3, 2, 0, 0, 0, 0, 1, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 5, 6, 8, 9, 10, 11, 13, 14, 16, 17, 16, 15, 12, 10, 9, 8, 6, 2, 0, 0, 7, 33, 84, 160, 258, 365, 466, 542, 578, 566, 510, 418, 309, 202, 113, 50, 14, 0, 0, 0, 2, 2, 2, 3, 5, 8, 10, 10, 9, 8, 7, 7, 6, 4, 0, 0, 6, 33, 87, 171, 279, 398, 506, 583, 610, 584, 510, 408, 304, 224, 189, 208, 278, 383, 499, 600, 662, 671, 625, 532, 412, 286, 173, 88, 33, 7, 0, 0, 2, 3, 1, 0, 0, 0, 4, 9, 13, 17, 17, 15, 10, 4, 0, 0, 15, 49, 107, 191, 295, 406, 509, 584, 618, 603, 542, 448, 338, 230, 138, 71, 30, 11, 6, 8, 11, 10, 5, 0, 0, 7, 41, 103, 192, 300, 410, 505, 568, 588, 562, 497, 405, 301, 202, 118, 57, 20, 2, 0, 1, 4, 6, 4, 2, 0, 0, 0, 2, 3, 4, 4, 2, 1, 0, 0, 0, 0, 0, 2, 5, 8, 11, 14, 15, 15, 14, 11, 8, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 4, 9, 15, 20, 23, 23, 20, 14, 9, 4, 1, 0, 0, 0, 0, 1, 2, 2, 3, 3, 1, 0, 0, 0, 3, 7, 11, 15, 16, 15, 11, 7, 3, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 7, 4, 0, 0, 3, 23, 66, 136, 230, 338, 444, 527, 572, 570, 521, 437, 334, 231, 141, 73, 30, 7, 0, 0, 1, 1, 0, 0, 0, 11, 41, 91, 163, 251, 345, 433, 501, 539, 539, 502, 433, 344, 248, 159, 87, 38, 10, 0, 0, 1, 6, 10, 12, 13, 13, 11, 8, 5, 2, 1, 0, 0, 0, 0, 0, 3, 7, 12, 18, 21, 23, 22, 19, 15, 12, 9, 7, 5, 4, 3, 2, 1, 1, 1, 1, 1, 0, 1, 2, 4, 7, 10, 12, 15, 17, 18, 18, 17, 15, 17, 28, 58, 113, 196, 302, 419, 527, 605, 636, 613, 542, 442, 340, 266, 241, 274, 357, 471, 591, 689, 744, 745, 692, 594, 468, 334, 211, 113, 47, 11, 0, 0, 1, 3, 2, 0, 0, 0, 1, 4, 7, 10, 11, 12, 11, 10, 9, 8, 8, 8, 8, 9, 10, 11, 11, 11, 11, 12, 13, 14, 16, 16, 15, 13, 9, 6, 3, 2, 1, 0, 0, 0, 0, 1, 3, 6, 9, 11, 12, 12, 10, 8, 5, 3, 2, 1, 2, 2, 2, 2, 2, 2, 2, 3, 4, 5, 5, 5, 4, 3, 2, 2, 2, 2, 2, 2, 1, 0, 0, 0, 0, 3, 7, 10, 14, 15, 16, 14, 13, 11, 10, 9, 9, 9, 9, 8, 8, 5, 2, 0, 1, 12, 42, 95, 173, 273, 383, 486, 563, 599, 588, 533, 450, 359, 286, 250, 263, 325, 426, 543, 653, 729, 756, 725, 643, 529, 411, 317, 271, 282, 346, 442, 543, 618, 648, 625, 557, 462, 368, 296, 263, 274, 322, 392, 464, 519, 543, 529, 480, 404, 315, 227, 151, 93, 55, 33, 23, 17, 14, 10, 7, 3, 2, 2, 4, 7, 8, 8, 7, 4, 2, 0, 0, 0, 0, 1, 2, 2, 3, 3, 2, 2, 2, 1, 1, 0, 0, 0, 0, 4, 10, 16, 21, 24, 24, 22, 19, 17, 15, 15, 15, 14, 13, 10, 7, 5, 2, 2, 5, 17, 42, 88, 157, 248, 354, 461, 552, 610, 624, 593, 526, 440, 358, 303, 292, 329, 409, 513, 617, 696, 728, 704, 627, 513, 382, 257, 153, 79, 35, 14, 7, 5, 3, 2, 2, 10, 29, 64, 117, 185, 265, 347, 421, 476, 502, 495, 455, 386, 300, 210, 128, 64, 22, 1, 0, 2, 10, 16, 17, 15, 11, 8, 7, 8, 11, 14, 17, 18, 17, 14, 11, 8, 5, 2, 1, 0, 0, 0, 0, 0, 0, 0, 4, 13, 31, 62, 108, 171, 247, 329, 407, 466, 496, 488, 443, 369, 277, 186, 107, 50, 16, 1, 0, 0, 2, 3, 3, 2, 3, 3, 3, 3, 2, 1, 0, 0, 0, 2, 9, 25, 54, 99, 162, 241, 327, 411, 478, 516, 517, 480, 413, 326, 235, 152, 86, 41, 15, 2, 0, 0, 1, 4, 5, 7, 8, 8, 8, 7, 5, 3, 0, 0, 0, 5, 24, 58, 111, 183, 270, 364, 455, 527, 570, 573, 534, 459, 360, 255, 159, 84, 35, 9, 0, 0, 1, 2, 1, 0, 0, 0, 1, 4, 7, 9, 10, 11, 12, 11, 10, 8, 6, 4, 2, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0, 2, 4, 8, 17, 33, 65, 117, 190, 281, 379, 467, 528, 551, 531, 475, 400, 327, 277, 262, 286, 342, 413, 480, 527, 542, 521, 470, 397, 313, 229, 154, 93, 48, 20, 4, 0, 0, 2, 5, 7, 7, 7, 7, 8, 10, 11, 11, 10, 8, 9, 19, 44, 88, 151, 230, 318, 402, 473, 518, 532, 514, 469, 407, 344, 296, 278, 297, 350, 426, 508, 573, 606, 598, 552, 482, 406, 346, 316, 323, 363, 424, 489, 540, 563, 552, 505, 432, 342, 249, 165, 97, 49, 19, 4, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 3, 6, 7, 6, 5, 7, 17, 43, 89, 155, 234, 317, 390, 443, 468, 464, 434, 385, 323, 256, 190, 129, 78, 40, 15, 2, 0, 0, 0, 1, 2, 4, 5, 7, 8, 8, 7, 5, 3, 1, 0, 0, 1, 4, 7, 11, 14, 16, 17, 16, 13, 10, 7, 6, 6, 7, 8, 9, 10, 10, 10, 8, 6, 4, 4, 5, 7, 11, 15, 16, 15, 12, 7, 3, 1, 2, 10, 26, 55, 101, 164, 243, 331, 416, 484, 524, 528, 495, 432, 349, 261, 178, 110, 60, 28, 10, 2, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 3, 2, 1, 0, 0, 1, 4, 8, 12, 15, 16, 15, 13, 10, 9, 9, 10, 14, 17, 20, 22, 21, 19, 15, 12, 8, 6, 7, 9, 12, 16, 20, 22, 21, 18, 13, 8, 5, 4, 5, 9, 12, 14, 13, 11, 11, 18, 38, 76, 134, 212, 303, 397, 484, 549, 585, 587, 557, 503, 439, 379, 338, 328, 352, 405, 477, 548, 602, 623, 605, 550, 473, 393, 329, 296, 300, 337, 394, 453, 499, 521, 513, 477, 418, 344, 265, 188, 120, 67, 29, 7, 0, 0, 8, 24, 49, 88, 141, 210, 290, 372, 445, 495, 516, 503, 459, 392, 314, 235, 164, 107, 64, 36, 20, 13, 12, 16, 20, 23, 25, 24, 21, 19, 17, 18, 19, 22, 23, 23, 21, 18, 15, 11, 9, 7, 5, 5, 5, 6, 8, 11, 14, 16, 18, 18, 17, 14, 10, 6, 2, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 3, 5, 7, 7, 6, 5, 3, 2, 2, 3, 4, 6, 7, 7, 7, 7, 6, 6, 4, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 2, 2, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 6, 7, 8, 7, 6, 4, 2, 0, 0, 0, 0, 2, 4, 7, 10, 11, 11, 10, 8, 6, 4, 3, 3, 5, 7, 11, 14, 17, 19, 19, 17, 14, 11, 10, 9, 10, 11, 11, 11, 12, 20, 39, 73, 124, 192, 269, 347, 415, 464, 486, 480, 445, 388, 317, 241, 167, 104, 56, 24, 6, 0, 0, 0, 1, 1, 0, 0, 0, 3, 7, 10, 12, 12, 10, 7, 4, 4, 6, 10, 15, 19, 21, 22, 20, 17, 13, 9, 7, 6, 5, 5, 4, 4, 5, 7, 9, 11, 11, 10, 7, 5, 9, 22, 47, 85, 133, 187, 241, 290, 328, 352, 358, 346, 316, 271, 216, 158, 105, 61, 30, 11, 3, 1, 2, 4, 5, 5, 4, 3, 3, 2, 2, 2, 2, 3, 3, 4, 5, 7, 11, 14, 17, 19, 20, 21, 21, 22, 23, 24, 25, 25, 25, 27, 29, 32, 34, 36, 36, 33, 31, 30, 35, 48, 73, 111, 161, 221, 286, 348, 401, 438, 453, 447, 424, 390, 357, 335, 331, 348, 384, 431, 481, 521, 542, 538, 507, 451, 377, 294, 211, 139, 81, 42, 18, 7, 5, 5, 7, 8, 8, 9, 11, 13, 15, 15, 13, 11, 8, 8, 12, 25, 49, 87, 138, 201, 269, 335, 387, 419, 424, 401, 355, 295, 229, 168, 115, 74, 45, 25, 12, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 5, 6, 6, 5, 3, 1, 0, 0, 0, 0, 0, 0, 1, 3, 6, 9, 12, 14, 14, 13, 13, 14, 16, 19, 21, 22, 22, 20, 17, 15, 13, 11, 11, 11, 10, 9, 7, 5, 4, 5, 7, 12, 17, 22, 23, 22, 20, 18, 21, 34, 57, 93, 140, 196, 258, 321, 382, 434, 472, 489, 482, 449, 394, 322, 246, 173, 112, 68, 40, 26, 21, 22, 24, 29, 39, 55, 82, 120, 168, 221, 274, 321, 354, 373, 375, 365, 346, 324, 305, 294, 297, 315, 349, 392, 438, 475, 494, 490, 463, 422, 378, 343, 327, 333, 357, 393, 428, 454, 465, 457, 435, 404, 373, 350, 340, 348, 373, 407, 443, 471, 481, 469, 435, 381, 315, 244, 176, 117, 71, 39, 19, 11, 10, 14, 22, 35, 55, 84, 123, 172, 226, 279, 325, 359, 377, 378, 365, 343, 319, 302, 297, 308, 334, 369, 405, 433, 443, 433, 403, 357, 302, 245, 190, 142, 101, 67, 42, 23, 11, 4, 1, 1, 2, 3, 5, 7, 8, 7, 6, 4, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 1, 1, 0, 0, 0, 0, 0, 1, 2, 3, 4, 6, 7, 7, 7, 7, 7, 7, 8, 9, 10, 10, 8, 5, 2, 0, 0, 2, 5, 9, 13, 16, 16, 15, 13, 11, 10, 10, 10, 11, 11, 10, 9, 7, 4, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 5, 7, 9, 11, 12, 12, 12, 12, 13, 15, 18, 20, 22, 22, 19, 15, 10, 5, 2, 1, 2, 4, 6, 8, 9, 9, 7, 6, 3, 2, 0, 0, 0, 0, 2, 4, 6, 7, 7, 6, 3, 1, 0, 0, 1, 2, 4, 4, 5, 6, 9, 15, 26, 44, 68, 102, 145, 196, 254, 313, 366, 405, 427, 429, 415, 390, 364, 344, 336, 341, 359, 386, 416, 444, 463, 466, 451, 416, 364, 299, 230, 164, 110, 69, 43, 28, 22, 21, 21, 21, 21, 19, 17, 13, 9, 6, 3, 2, 2, 3, 4, 5, 4, 3, 2, 2, 2, 3, 4, 5, 7, 8, 9, 10, 12, 14, 15, 16, 18, 20, 22, 25, 28, 32, 40, 54, 78, 117, 170, 237, 312, 385, 449, 493, 512, 505, 475, 425, 364, 297, 231, 169, 115, 72, 41, 19, 7, 1, 0, 0, 0, 2, 4, 7, 11, 14, 16, 16, 14, 11, 7, 4, 2, 2, 3, 4, 5, 6, 6, 4, 3, 1, 0, 0, 0, 0, 2, 3, 5, 5, 6, 6, 6, 6, 7, 7, 7, 6, 5, 5, 6, 9, 13, 16, 19, 19, 16, 13, 11, 11, 14, 18, 22, 25, 26, 26, 24, 22, 20, 18, 15, 12, 8, 4, 1, 0, 0, 0, 1, 1, 1, 1, 1, 3, 5, 8, 11, 14, 17, 22, 29, 40, 57, 82, 117, 162, 214, 271, 325, 370, 401, 414, 408, 384, 347, 300, 248, 196, 145, 100, 62, 33, 13, 3, 0, 2, 5, 8, 9, 9, 7, 6, 8, 11, 17, 23, 29, 34, 39, 48, 63, 90, 130, 183, 246, 312, 373, 420, 447, 450, 430, 392, 343, 288, 235, 187, 144, 107, 75, 49, 30, 18, 13, 14, 18, 23, 26, 27, 24, 20, 15, 10, 7, 5, 6, 9, 12, 16, 18, 18, 16, 14, 12, 10, 9, 8, 8, 6, 4, 2, 0, 0, 0, 0, 2, 3, 4, 5, 4, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 1, 1, 0, 0, 3, 10, 23, 42, 67, 96, 128, 162, 198, 236, 273, 306, 331, 341, 332, 305, 263, 211, 159, 113, 79, 56, 43, 35, 30, 25, 19, 13, 7, 4, 3, 4, 6, 7, 7, 6, 3, 1, 0, 0, 3, 7, 11, 14, 16, 15, 14, 13, 12, 11, 10, 9, 7, 4, 3, 1, 1, 1, 1, 1, 1, 1, 2, 3, 5, 6, 8, 9, 11, 13, 14, 16, 16, 16, 14, 12, 9, 6, 4, 2, 0, 0, 0, 0, 1, 2, 2, 2, 2, 1, 0, 0, 0, 2, 3, 3, 3, 2, 1, 1, 2, 4, 7, 9, 9, 9, 7, 6, 5, 5, 6, 7, 7, 7, 5, 3, 2, 1, 1, 2, 3, 4, 6, 7, 8, 10, 11, 12, 13, 13, 13, 12, 12, 11, 10, 10, 9, 9, 9, 10, 11, 13, 14, 14, 13, 12, 10, 8, 8, 10, 11, 13, 14, 14, 14, 17, 25, 40, 64, 96, 134, 178, 225, 272, 317, 357, 386, 402, 403, 390, 367, 343, 324, 319, 328, 349, 374, 395, 406, 403, 386, 358, 325, 288, 250, 210, 169, 128, 90, 57, 32, 16, 8, 5, 6, 8, 9, 9, 8, 6, 5, 5, 5, 5, 5, 5, 4, 3, 4, 4, 5, 6, 6, 5, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 2, 3, 5, 7, 8, 8, 8, 6, 5, 3, 1, 0, 0, 1, 3, 6, 8, 9, 8, 7, 5, 4, 4, 5, 7, 9, 10, 10, 9, 7, 5, 2, 1, 0, 0, 0, 2, 3, 5, 6, 7, 7, 7, 6, 5, 3, 2, 1, 2, 6, 14, 26, 41, 60, 83, 109, 138, 171, 205, 238, 265, 284, 291, 284, 266, 237, 201, 162, 124, 90, 63, 43, 30, 22, 17, 15, 12, 11, 10, 10, 11, 14, 17, 21, 24, 26, 26, 24, 20, 16, 12, 8, 6, 4, 3, 2, 1, 1, 1, 1, 2, 3, 4, 4, 4, 2, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 4, 9, 14, 21, 28, 38, 54, 76, 108, 146, 188, 228, 262, 285, 296, 294, 280, 257, 226, 191, 154, 119, 88, 62, 42, 28, 18, 12, 8, 5, 5, 6, 9, 11, 12, 12, 10, 6, 2, 0, 0, 0, 3, 6, 8, 8, 7, 5, 3, 0, 0, 0, 0, 2, 3, 4, 4, 3, 1, 0, 0, 2, 5, 9, 12, 14, 15, 13, 11, 9, 8, 8, 11, 16, 25, 37, 51, 69, 92, 122, 162, 212, 269, 327, 377, 413, 428, 420, 393, 352, 305, 256, 210, 168, 130, 97, 69, 47, 32, 23, 19, 18, 18, 18, 16, 14, 11, 9, 7, 4, 2, 0, 0, 0, 3, 7, 12, 16, 17, 14, 9, 3, 0, 0, 3, 8, 13, 15, 13, 9, 4, 1, 2, 5, 10, 15, 17, 17, 15, 12, 11, 12, 16, 22, 29, 39, 51, 66, 87, 113, 146, 185, 229, 277, 326, 371, 407, 431, 441, 438, 426, 411, 399, 394, 400, 415, 437, 463, 488, 509, 520, 521, 509, 488, 460, 432, 411, 399, 399, 408, 421, 435, 443, 444, 438, 425, 410, 396, 386, 381, 382, 386, 391, 395, 394, 387, 373, 351, 321, 285, 244, 202, 161, 123, 91, 64, 43, 27, 16, 9, 7, 9, 13, 18, 21, 20, 17, 12, 6, 2, 0, 0, 0, 1, 3, 5, 8, 11, 13, 14, 17, 23, 35, 54, 81, 115, 153, 191, 228, 262, 290, 311, 321, 320, 307, 281, 246, 207, 166, 129, 97, 71, 51, 37, 27, 20, 17, 14, 13, 13, 13, 13, 14, 16, 18, 20, 22, 24, 25, 25, 24, 22, 17, 13, 8, 6, 6, 10, 16, 24, 31, 35, 35, 33, 29, 26, 27, 32, 42, 58, 78, 102, 131, 165, 202, 243, 285, 326, 362, 393, 417, 433, 440, 441, 435, 427, 420, 418, 426, 443, 468, 495, 516, 526, 522, 504, 478, 450, 426, 410, 405, 408, 417, 429, 437, 438, 428, 405, 372, 331, 285, 240, 198, 161, 127, 97, 69, 47, 30, 21, 18, 20, 24, 26, 26, 24, 23, 23, 24, 27, 30, 30, 29, 26, 24, 23, 24, 26, 29, 32, 36, 42, 53, 68, 89, 115, 143, 173, 205, 240, 277, 314, 348, 377, 398, 408, 410, 404, 396, 388, 384, 387, 398, 416, 437, 454, 460, 451, 423, 378, 323, 264, 209, 162, 125, 97, 76, 60, 50, 45, 49, 62, 83, 111, 143, 174, 203, 226, 241, 248, 248, 241, 229, 212, 190, 166, 139, 112, 86, 62, 42, 26, 15, 8, 3, 0, 0, 0, 2, 7, 12, 16, 20, 23, 28, 38, 55, 81, 113, 150, 188, 223, 253, 275, 289, 294, 288, 273, 248, 215, 177, 138, 101, 69, 44, 26, 14, 6, 2, 0, 0, 0, 0, 0, 0, 0, 1, 3, 6, 8, 9, 8, 6, 5, 4, 4, 6, 7, 8, 8, 6, 5, 4, 6, 8, 11, 12, 12, 11, 10, 10, 11, 13, 16, 18, 18, 17, 14, 12, 10, 11, 16, 26, 40, 58, 80, 105, 134, 168, 209, 256, 305, 351, 388, 411, 419, 416, 408, 401, 401, 409, 422, 436, 446, 446, 436, 417, 393, 368, 345, 323, 301, 275, 244, 206, 166, 127, 94, 70, 57, 53, 55, 62, 74, 91, 114, 145, 181, 218, 253, 280, 297, 305, 307, 307, 309, 315, 324, 334, 342, 347, 350, 351, 352, 353, 353, 346, 329, 300, 261, 215, 168, 126, 92, 68, 51, 39, 29, 20, 12, 7, 5, 5, 5, 6, 5, 4, 2, 1, 1, 2, 4, 5, 5, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 2, 2, 2, 1, 0, 0, 0, 1, 3, 5, 6, 6, 5, 3, 2, 1, 2, 4, 6, 9, 11, 13, 13, 12, 9, 7, 4, 2, 1, 2, 3, 5, 5, 4, 3, 1, 0, 0, 4, 9, 16, 23, 30, 40, 55, 75, 103, 136, 171, 204, 230, 246, 252, 249, 241, 228, 213, 195, 172, 146, 117, 88, 61, 39, 23, 13, 7, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 3, 4, 5, 7, 10, 15, 22, 31, 42, 56, 73, 94, 118, 144, 171, 198, 223, 245, 262, 273, 275, 268, 251, 224, 192, 157, 124, 95, 74, 58, 46, 38, 30, 24, 19, 15, 13, 12, 12, 12, 13, 15, 20, 27, 38, 54, 73, 95, 119, 147, 176, 207, 239, 270, 297, 318, 328, 327, 314, 293, 264, 232, 198, 163, 130, 100, 75, 58, 51, 52, 59, 71, 84, 97, 111, 128, 149, 174, 201, 225, 243, 249, 245, 231, 211, 190, 170, 152, 134, 115, 92, 68, 44, 24, 9, 2, 1, 3, 7, 8, 8, 6, 3, 0, 0, 0, 1, 3, 5, 7, 9, 11, 13, 13, 14, 13, 12, 11, 11, 11, 11, 12, 12, 13, 14, 16, 19, 25, 33, 44, 60, 79, 104, 133, 164, 197, 229, 258, 281, 297, 305, 302, 291, 270, 243, 213, 182, 153, 129, 110, 95, 83, 71, 60, 48, 38, 29, 23, 21, 21, 23, 28, 33, 40, 49, 61, 76, 96, 120, 150, 183, 217, 251, 281, 306, 323, 333, 337, 337, 336, 338, 344, 357, 373, 389, 403, 410, 408, 398, 380, 356, 327, 294, 259, 222, 185, 149, 115, 86, 62, 44, 31, 22, 18, 15, 13, 10, 7, 4, 1, 1, 2, 5, 8, 11, 11, 11, 10, 9, 9, 9, 10, 10, 11, 11, 12, 12, 11, 9, 6, 3, 0, 0, 2, 6, 11, 16, 18, 18, 16, 13, 10, 8, 6, 5, 4, 3, 1, 0, 0, 0, 0, 1, 1, 2, 1, 1, 0, 0, 0, 0, 2, 4, 6, 9, 12, 14, 15, 16, 16, 17, 20, 28, 42, 61, 85, 112, 140, 167, 194, 220, 246, 269, 288, 298, 299, 288, 268, 241, 212, 183, 157, 133, 111, 89, 69, 52, 37, 26, 19, 13, 10, 7, 4, 3, 3, 3, 4, 4, 2, 1, 0, 0, 2, 6, 10, 13, 13, 11, 7, 4, 3, 3, 6, 9, 11, 12, 10, 8, 5, 3, 2, 3, 6, 8, 9, 9, 7, 5, 2, 0, 0, 1, 4, 6, 7, 6, 4, 1, 0, 0, 2, 5, 7, 7, 6, 4, 1, 0, 0, 0, 2, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 3, 4, 4, 5, 6, 7, 9, 10, 11, 11, 10, 9, 7, 6, 6, 7, 10, 12, 15, 17, 16, 14, 11, 7, 4, 3, 3, 3, 3, 3, 1, 0, 0, 0, 1, 2, 3, 4, 5, 6, 6, 7, 7, 7, 6, 5, 5, 5, 6, 7, 8, 7, 6, 4, 2, 1, 2, 4, 6, 8, 9, 10, 9, 8, 7, 6, 6, 7, 9, 11, 12, 13, 12, 11, 10, 11, 13, 17, 24, 32, 41, 51, 61, 73, 88, 108, 132, 161, 195, 231, 266, 299, 326, 345, 354, 356, 351, 344, 338, 336, 340, 348, 357, 362, 362, 354, 339, 320, 299, 276, 251, 224, 196, 166, 135, 106, 80, 57, 38, 23, 12, 6, 4, 7, 11, 17, 22, 26, 29, 33, 40, 51, 68, 90, 115, 141, 166, 189, 210, 227, 242, 253, 259, 260, 256, 245, 228, 207, 182, 155, 128, 102, 79, 60, 45, 34, 26, 21, 19, 18, 18, 18, 17, 18, 20, 23, 29, 37, 47, 59, 73, 87, 102, 117, 131, 143, 154, 164, 172, 178, 180, 176, 167, 153, 135, 115, 95, 75, 58, 42, 27, 15, 6, 1, 0, 1, 4, 6, 7, 5, 3, 1, 0, 0, 1, 2, 3, 2, 0, 0, 0, 2, 6, 10, 12, 11, 9, 5, 3, 3, 6, 11, 18, 24, 31, 39, 49, 62, 78, 97, 119, 143, 171, 201, 234, 264, 289, 304, 307, 300, 283, 261, 237, 213, 189, 165, 141, 116, 92, 71, 55, 43, 35, 29, 24, 19, 14, 9, 5, 2, 0, 0, 0, 0, 0, 0, 0, 1, 2, 4, 7, 10, 13, 17, 22, 28, 37, 51, 69, 93, 121, 149, 176, 198, 215, 227, 234, 240, 243, 242, 237, 225, 205, 180, 152, 124, 99, 78, 61, 47, 35, 25, 16, 9, 5, 3, 2, 3, 3, 3, 2, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 2, 3, 4, 4, 3, 2, 1, 0, 0, 0, 3, 7, 12, 16, 17, 14, 10, 5, 1, 0, 3, 8, 14, 19, 25, 30, 38, 49, 63, 81, 100, 120, 140, 160, 182, 205, 229, 253, 273, 288, 298, 303, 305, 307, 310, 315, 320, 325, 327, 325, 320, 310, 297, 280, 259, 234, 205, 173, 141, 111, 86, 67, 54, 45, 39, 34, 29, 24, 19, 15, 13, 12, 11, 11, 11, 11, 10, 10, 9, 9, 8, 7, 6, 4, 3, 3, 4, 6, 7, 8, 8, 9, 11, 17, 24, 34, 45, 56, 69, 85, 105, 129, 158, 188, 217, 240, 258, 269, 276, 279, 280, 277, 269, 253, 230, 200, 168, 137, 111, 91, 76, 66, 56, 45, 34, 23, 13, 6, 3, 2, 2, 2, 1, 0, 0, 0, 2, 6, 9, 11, 11, 9, 6, 3, 1, 2, 4, 8, 11, 14, 16, 18, 20, 24, 33, 46, 64, 88, 116, 145, 172, 197, 220, 242, 264, 286, 307, 322, 329, 326, 310, 286, 255, 222, 191, 163, 138, 117, 99, 82, 66, 51, 37, 26, 18, 14, 13, 14, 16, 16, 15, 12, 8, 5, 3, 3, 4, 5, 6, 6, 5, 4, 4, 5, 6, 7, 8, 7, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 4, 6, 9, 13, 19, 29, 44, 62, 84, 105, 124, 139, 152, 162, 172, 183, 194, 203, 204, 192, 165, 133, 106, 87, 76, 71))
        self.assertEqual(record.annotations["abif_raw"]['DATA12'], (12, 11, 8, 5, 1, 0, 2, 7, 14, 21, 24, 24, 22, 20, 20, 23, 29, 37, 44, 49, 52, 54, 55, 57, 61, 65, 69, 72, 72, 70, 67, 65, 64, 67, 73, 81, 88, 93, 93, 89, 83, 77, 72, 71, 72, 73, 73, 71, 67, 62, 57, 53, 50, 49, 50, 51, 54, 57, 60, 62, 64, 67, 71, 77, 87, 98, 108, 118, 126, 135, 147, 164, 187, 215, 243, 270, 294, 314, 332, 351, 373, 398, 424, 449, 469, 485, 494, 499, 502, 505, 510, 517, 524, 528, 526, 518, 503, 484, 464, 444, 427, 412, 401, 393, 390, 392, 400, 409, 417, 421, 417, 406, 388, 368, 348, 330, 312, 295, 275, 251, 225, 197, 171, 147, 128, 113, 100, 89, 79, 70, 62, 55, 49, 45, 41, 38, 35, 32, 30, 28, 28, 28, 28, 28, 28, 29, 30, 32, 34, 37, 39, 39, 37, 33, 30, 28, 28, 31, 36, 42, 46, 48, 50, 54, 62, 75, 95, 119, 145, 172, 200, 227, 257, 288, 323, 359, 395, 430, 462, 491, 516, 537, 554, 566, 572, 570, 561, 543, 517, 484, 446, 405, 366, 329, 297, 269, 243, 218, 195, 173, 155, 144, 142, 149, 164, 185, 208, 230, 252, 272, 290, 306, 320, 328, 330, 325, 311, 292, 268, 242, 216, 190, 165, 139, 115, 91, 69, 51, 35, 24, 17, 14, 12, 13, 13, 14, 13, 13, 13, 14, 16, 19, 22, 23, 23, 22, 20, 18, 16, 15, 14, 14, 14, 15, 15, 14, 13, 11, 10, 10, 10, 10, 11, 11, 12, 13, 15, 17, 21, 24, 27, 29, 30, 30, 28, 27, 25, 23, 23, 23, 23, 24, 24, 23, 22, 22, 21, 22, 23, 23, 23, 21, 19, 18, 18, 21, 27, 35, 46, 58, 71, 86, 104, 126, 153, 185, 223, 265, 308, 350, 386, 411, 423, 419, 401, 370, 332, 290, 246, 202, 161, 123, 89, 63, 47, 42, 50, 71, 103, 144, 191, 243, 297, 349, 398, 440, 471, 488, 489, 472, 439, 395, 347, 303, 269, 249, 245, 256, 279, 311, 348, 383, 412, 430, 431, 413, 377, 327, 270, 214, 165, 125, 96, 75, 61, 50, 41, 34, 28, 24, 23, 22, 23, 23, 23, 21, 18, 14, 11, 8, 6, 4, 4, 3, 2, 1, 0, 0, 0, 0, 1, 3, 4, 5, 5, 5, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 19, 42, 77, 122, 177, 239, 304, 368, 422, 462, 481, 478, 457, 426, 396, 378, 378, 400, 437, 479, 516, 534, 527, 493, 435, 361, 280, 203, 136, 83, 45, 21, 8, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 5, 11, 16, 21, 24, 25, 23, 20, 16, 13, 11, 10, 10, 11, 11, 10, 9, 7, 4, 3, 6, 16, 36, 71, 123, 193, 277, 371, 465, 546, 603, 627, 615, 572, 510, 445, 395, 371, 379, 416, 470, 528, 574, 596, 589, 549, 481, 393, 295, 200, 118, 57, 19, 1, 0, 0, 0, 0, 0, 11, 39, 90, 165, 261, 369, 478, 574, 647, 686, 686, 647, 575, 477, 368, 258, 160, 83, 31, 5, 6, 30, 75, 138, 215, 302, 390, 470, 529, 558, 551, 507, 432, 339, 242, 154, 85, 39, 13, 3, 2, 4, 5, 5, 3, 2, 2, 4, 7, 10, 11, 10, 7, 3, 0, 0, 2, 7, 12, 17, 18, 16, 11, 5, 1, 0, 0, 1, 3, 4, 3, 2, 1, 0, 0, 0, 0, 0, 0, 1, 2, 4, 5, 7, 9, 12, 14, 17, 19, 20, 19, 19, 17, 15, 13, 10, 6, 2, 0, 0, 0, 1, 0, 0, 0, 4, 23, 60, 120, 200, 293, 387, 468, 523, 543, 524, 474, 406, 341, 298, 294, 334, 416, 524, 634, 722, 766, 756, 690, 580, 445, 308, 187, 96, 38, 9, 0, 0, 0, 1, 2, 4, 8, 12, 16, 17, 15, 10, 6, 2, 1, 0, 0, 0, 0, 9, 34, 82, 160, 268, 400, 541, 671, 769, 815, 799, 722, 597, 446, 296, 167, 74, 20, 0, 0, 1, 5, 3, 0, 0, 13, 55, 131, 242, 380, 529, 666, 766, 811, 790, 706, 576, 423, 275, 153, 68, 18, 0, 0, 6, 35, 87, 166, 270, 387, 498, 582, 622, 610, 553, 466, 373, 300, 268, 284, 346, 437, 535, 613, 651, 638, 577, 477, 357, 239, 138, 65, 21, 1, 0, 0, 2, 3, 1, 0, 0, 0, 2, 6, 10, 13, 14, 14, 11, 7, 4, 1, 0, 1, 2, 4, 4, 4, 3, 2, 1, 1, 1, 2, 3, 4, 4, 3, 3, 3, 4, 4, 5, 5, 6, 8, 9, 10, 11, 10, 9, 9, 9, 10, 11, 11, 11, 9, 7, 4, 2, 1, 1, 0, 0, 0, 0, 1, 4, 5, 2, 0, 0, 19, 69, 157, 281, 425, 564, 670, 720, 704, 628, 515, 398, 311, 280, 317, 411, 535, 653, 730, 745, 697, 605, 505, 432, 414, 461, 558, 677, 780, 835, 822, 738, 602, 439, 280, 150, 61, 13, 0, 0, 6, 11, 12, 11, 11, 14, 21, 29, 34, 36, 33, 26, 19, 13, 9, 7, 4, 0, 0, 5, 38, 108, 222, 373, 543, 704, 825, 880, 860, 769, 629, 467, 314, 191, 110, 70, 63, 77, 100, 123, 141, 154, 164, 171, 178, 186, 193, 198, 200, 197, 189, 177, 164, 153, 144, 140, 137, 135, 132, 126, 119, 110, 102, 95, 87, 80, 73, 65, 58, 52, 48, 45, 42, 38, 32, 24, 16, 10, 7, 7, 9, 12, 15, 17, 18, 17, 16, 14, 11, 8, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 4, 4, 2, 0, 0, 0, 1, 6, 8, 5, 0, 0, 13, 69, 175, 328, 509, 685, 818, 877, 847, 735, 568, 383, 217, 95, 24, 0, 0, 3, 8, 6, 1, 0, 0, 4, 12, 19, 21, 19, 13, 7, 2, 0, 0, 1, 3, 4, 2, 0, 0, 0, 0, 3, 6, 4, 0, 0, 4, 40, 114, 227, 370, 519, 646, 722, 731, 670, 553, 407, 260, 137, 53, 9, 0, 0, 4, 7, 6, 4, 3, 4, 5, 7, 7, 5, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 3, 1, 0, 0, 0, 3, 8, 10, 7, 0, 0, 9, 61, 164, 315, 497, 676, 814, 877, 851, 742, 577, 393, 226, 101, 27, 0, 0, 5, 14, 17, 16, 12, 9, 7, 6, 4, 3, 2, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, 2, 3, 3, 3, 2, 1, 0, 0, 0, 0, 0, 0, 1, 3, 6, 9, 11, 11, 9, 7, 6, 6, 8, 12, 17, 21, 22, 21, 18, 14, 12, 11, 13, 16, 19, 21, 20, 18, 14, 9, 5, 2, 0, 0, 0, 1, 2, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 4, 5, 4, 4, 3, 1, 0, 0, 0, 0, 1, 3, 5, 5, 5, 4, 3, 2, 3, 5, 7, 8, 8, 6, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 5, 12, 18, 24, 25, 23, 18, 11, 5, 1, 0, 0, 0, 0, 2, 4, 7, 11, 14, 15, 15, 12, 8, 4, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 2, 3, 2, 0, 0, 0, 0, 3, 8, 9, 3, 0, 0, 21, 86, 198, 347, 506, 641, 716, 713, 632, 495, 337, 191, 82, 19, 0, 0, 5, 12, 14, 13, 12, 13, 15, 17, 17, 15, 10, 6, 3, 3, 4, 6, 9, 11, 12, 12, 11, 10, 9, 8, 8, 8, 8, 9, 10, 12, 13, 12, 10, 7, 3, 0, 0, 2, 5, 8, 10, 11, 10, 9, 9, 10, 11, 11, 8, 4, 1, 2, 5, 8, 7, 0, 0, 0, 37, 118, 244, 400, 555, 673, 723, 693, 590, 442, 284, 149, 56, 8, 0, 0, 6, 8, 5, 0, 0, 0, 4, 10, 16, 19, 19, 17, 14, 12, 11, 12, 15, 19, 23, 27, 31, 33, 33, 32, 29, 27, 25, 24, 25, 27, 30, 32, 32, 31, 29, 27, 28, 30, 36, 42, 47, 49, 47, 42, 34, 27, 22, 21, 22, 21, 16, 4, 0, 0, 35, 121, 259, 435, 618, 767, 848, 840, 744, 586, 402, 232, 104, 27, 0, 0, 5, 12, 12, 6, 0, 0, 5, 17, 29, 37, 38, 30, 18, 6, 0, 2, 8, 12, 7, 0, 0, 16, 90, 222, 400, 593, 755, 844, 836, 735, 568, 379, 209, 86, 17, 0, 0, 6, 13, 12, 9, 6, 8, 14, 21, 28, 32, 34, 33, 30, 25, 20, 17, 16, 17, 20, 24, 25, 24, 19, 13, 7, 3, 0, 0, 0, 0, 1, 3, 5, 7, 9, 9, 8, 6, 2, 0, 0, 0, 3, 7, 12, 15, 15, 14, 10, 7, 6, 7, 9, 13, 17, 19, 19, 17, 14, 9, 5, 2, 0, 0, 1, 3, 6, 7, 7, 5, 2, 0, 0, 1, 6, 10, 13, 12, 7, 1, 0, 0, 4, 10, 12, 5, 0, 0, 24, 102, 234, 407, 587, 732, 805, 786, 679, 515, 334, 175, 65, 7, 0, 0, 10, 17, 15, 9, 2, 0, 0, 2, 7, 13, 18, 23, 24, 24, 20, 16, 11, 9, 9, 12, 16, 19, 21, 20, 17, 13, 9, 6, 5, 5, 7, 9, 12, 14, 15, 14, 12, 9, 6, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 6, 10, 14, 14, 11, 6, 2, 1, 6, 11, 14, 8, 0, 0, 12, 78, 200, 369, 554, 713, 805, 803, 709, 549, 365, 198, 78, 13, 0, 0, 7, 11, 8, 2, 0, 0, 2, 7, 9, 8, 5, 2, 1, 2, 5, 8, 11, 13, 15, 17, 18, 19, 17, 14, 9, 5, 2, 2, 2, 2, 1, 0, 0, 0, 0, 6, 11, 9, 1, 0, 0, 39, 127, 264, 430, 591, 709, 751, 706, 588, 427, 262, 127, 39, 0, 0, 4, 15, 18, 14, 6, 0, 0, 1, 4, 6, 4, 0, 0, 0, 7, 15, 17, 8, 0, 0, 25, 118, 282, 502, 737, 933, 1038, 1023, 892, 681, 444, 234, 87, 10, 0, 0, 11, 16, 11, 2, 0, 0, 2, 10, 15, 17, 14, 10, 5, 2, 0, 0, 0, 0, 1, 5, 10, 16, 21, 23, 22, 18, 12, 6, 2, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 6, 7, 7, 6, 3, 1, 0, 0, 0, 1, 3, 5, 7, 8, 8, 6, 4, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 4, 5, 6, 7, 7, 7, 6, 5, 5, 6, 8, 10, 13, 15, 17, 17, 15, 13, 11, 8, 6, 4, 3, 3, 4, 6, 8, 8, 7, 4, 0, 0, 0, 3, 7, 7, 2, 0, 0, 22, 80, 177, 299, 425, 522, 567, 544, 462, 343, 215, 107, 35, 1, 0, 0, 6, 8, 5, 0, 0, 0, 3, 7, 10, 11, 10, 8, 6, 5, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 7, 12, 16, 20, 21, 21, 20, 20, 20, 22, 24, 27, 28, 27, 24, 18, 12, 7, 2, 0, 0, 0, 2, 7, 12, 17, 20, 19, 14, 7, 2, 1, 5, 10, 11, 5, 0, 0, 19, 87, 204, 358, 520, 652, 721, 706, 614, 469, 308, 165, 62, 5, 0, 0, 10, 21, 21, 9, 0, 0, 11, 84, 212, 384, 569, 725, 813, 808, 713, 554, 372, 207, 86, 19, 0, 0, 7, 12, 11, 5, 1, 0, 3, 9, 14, 17, 16, 12, 8, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 4, 6, 9, 12, 16, 19, 20, 19, 14, 7, 0, 0, 0, 5, 12, 13, 6, 0, 0, 31, 120, 267, 459, 658, 813, 881, 836, 686, 473, 258, 105, 62, 143, 325, 557, 773, 915, 946, 865, 698, 491, 291, 133, 36, 0, 0, 9, 24, 25, 12, 0, 0, 16, 102, 248, 440, 639, 798, 874, 847, 725, 541, 342, 173, 59, 3, 0, 0, 10, 13, 8, 0, 0, 0, 2, 7, 10, 9, 6, 3, 0, 0, 0, 0, 0, 0, 2, 5, 9, 12, 12, 9, 3, 0, 0, 0, 7, 14, 14, 5, 0, 0, 32, 128, 284, 479, 675, 819, 874, 824, 684, 492, 298, 142, 42, 0, 0, 2, 10, 11, 5, 0, 0, 0, 3, 7, 9, 8, 5, 1, 0, 0, 0, 2, 5, 6, 7, 6, 3, 0, 0, 0, 1, 9, 14, 11, 0, 0, 1, 56, 171, 343, 548, 743, 880, 920, 855, 701, 500, 300, 141, 40, 0, 0, 2, 12, 13, 7, 0, 0, 0, 1, 8, 13, 14, 10, 4, 0, 0, 0, 7, 12, 9, 0, 0, 1, 51, 154, 308, 489, 657, 770, 797, 731, 590, 412, 240, 107, 27, 0, 0, 4, 10, 8, 2, 0, 0, 4, 13, 20, 22, 19, 13, 6, 1, 0, 0, 0, 1, 2, 3, 5, 7, 9, 9, 8, 6, 2, 0, 1, 4, 8, 11, 12, 10, 9, 10, 14, 18, 17, 7, 0, 0, 29, 122, 281, 492, 715, 898, 989, 958, 808, 576, 323, 117, 11, 30, 166, 376, 601, 781, 869, 849, 731, 552, 358, 190, 73, 12, 0, 0, 10, 14, 10, 2, 0, 0, 0, 9, 17, 22, 21, 15, 5, 0, 0, 0, 9, 15, 13, 0, 0, 0, 57, 177, 358, 570, 767, 897, 927, 848, 683, 477, 279, 126, 34, 0, 0, 4, 10, 9, 2, 0, 0, 6, 16, 26, 30, 29, 23, 15, 8, 3, 2, 3, 5, 8, 10, 10, 8, 6, 3, 1, 0, 0, 1, 5, 11, 18, 24, 28, 29, 26, 20, 14, 7, 2, 0, 0, 1, 5, 10, 13, 14, 12, 8, 3, 0, 0, 3, 10, 17, 22, 23, 19, 13, 7, 2, 0, 0, 0, 0, 0, 1, 3, 4, 5, 3, 0, 0, 0, 0, 9, 18, 18, 6, 0, 0, 36, 151, 340, 580, 818, 993, 1057, 990, 813, 575, 338, 151, 37, 0, 0, 7, 19, 20, 12, 4, 0, 2, 7, 13, 15, 15, 11, 8, 4, 2, 1, 0, 0, 0, 0, 0, 1, 2, 2, 1, 1, 0, 1, 3, 5, 7, 7, 6, 4, 2, 0, 0, 0, 2, 5, 8, 9, 9, 8, 6, 4, 2, 1, 0, 0, 0, 0, 0, 2, 5, 7, 7, 5, 1, 0, 0, 0, 8, 14, 13, 2, 0, 0, 37, 132, 279, 457, 626, 742, 774, 714, 579, 407, 239, 109, 30, 0, 0, 2, 8, 9, 6, 4, 6, 11, 17, 19, 17, 12, 6, 1, 0, 0, 1, 5, 9, 12, 14, 14, 12, 8, 4, 0, 0, 0, 3, 7, 10, 11, 11, 8, 5, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 4, 3, 3, 2, 1, 0, 1, 1, 2, 2, 3, 3, 4, 5, 7, 8, 9, 9, 8, 6, 4, 2, 0, 0, 0, 1, 5, 11, 16, 19, 20, 18, 13, 7, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 4, 5, 7, 8, 10, 12, 14, 16, 17, 16, 13, 10, 8, 8, 8, 9, 8, 5, 1, 0, 4, 11, 17, 15, 3, 0, 0, 42, 149, 316, 521, 718, 856, 895, 826, 667, 464, 266, 115, 26, 0, 0, 6, 13, 12, 5, 0, 0, 5, 12, 16, 16, 12, 6, 2, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 2, 4, 6, 7, 6, 4, 3, 2, 3, 4, 5, 5, 4, 2, 0, 0, 1, 4, 10, 15, 19, 22, 22, 20, 16, 12, 8, 5, 5, 5, 8, 11, 15, 17, 18, 16, 13, 9, 4, 0, 0, 0, 5, 13, 21, 28, 30, 28, 21, 14, 11, 14, 20, 22, 16, 1, 0, 0, 61, 189, 376, 593, 788, 910, 923, 822, 638, 420, 223, 81, 7, 0, 0, 12, 17, 12, 3, 0, 0, 2, 9, 15, 15, 13, 8, 3, 0, 0, 0, 1, 5, 8, 12, 14, 14, 12, 9, 6, 4, 2, 2, 2, 3, 2, 2, 1, 0, 0, 0, 0, 1, 3, 5, 7, 7, 6, 6, 8, 14, 19, 19, 10, 0, 0, 11, 83, 215, 394, 588, 749, 834, 819, 708, 533, 339, 172, 58, 2, 0, 0, 11, 15, 10, 1, 0, 0, 3, 11, 17, 18, 16, 12, 8, 6, 5, 5, 5, 5, 3, 1, 0, 0, 3, 8, 13, 17, 18, 17, 14, 11, 9, 9, 11, 14, 16, 16, 14, 10, 6, 3, 0, 0, 0, 0, 0, 0, 1, 2, 3, 3, 2, 2, 1, 0, 0, 1, 2, 2, 2, 1, 0, 0, 0, 1, 2, 4, 5, 5, 5, 4, 3, 3, 3, 4, 4, 2, 1, 0, 0, 2, 6, 11, 14, 15, 14, 14, 15, 18, 19, 14, 2, 0, 0, 40, 137, 286, 466, 638, 757, 790, 726, 584, 404, 230, 96, 19, 0, 0, 8, 17, 15, 8, 0, 0, 0, 0, 4, 5, 5, 3, 1, 0, 0, 1, 3, 6, 9, 12, 15, 18, 19, 19, 17, 13, 9, 4, 1, 0, 0, 0, 2, 4, 6, 7, 8, 7, 6, 5, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 5, 7, 7, 6, 4, 2, 0, 0, 0, 1, 4, 8, 11, 12, 12, 10, 6, 3, 1, 0, 0, 0, 0, 1, 1, 2, 3, 4, 6, 8, 10, 11, 11, 10, 7, 4, 1, 0, 0, 1, 6, 13, 19, 23, 23, 18, 10, 4, 3, 8, 15, 17, 10, 0, 0, 11, 86, 224, 414, 622, 801, 905, 909, 812, 642, 443, 258, 119, 35, 0, 0, 2, 8, 7, 2, 0, 0, 1, 8, 15, 20, 21, 18, 14, 8, 3, 0, 0, 0, 4, 10, 15, 19, 21, 19, 15, 10, 5, 1, 0, 0, 1, 4, 7, 9, 10, 9, 9, 11, 15, 18, 16, 7, 0, 0, 17, 85, 204, 361, 526, 661, 730, 713, 617, 466, 301, 158, 58, 7, 0, 0, 7, 10, 7, 0, 0, 0, 3, 10, 17, 21, 22, 19, 15, 9, 5, 2, 0, 0, 0, 0, 0, 1, 4, 7, 11, 14, 15, 14, 13, 11, 10, 10, 10, 9, 8, 5, 3, 0, 0, 0, 0, 3, 5, 5, 3, 0, 0, 0, 0, 9, 16, 15, 3, 0, 0, 46, 160, 338, 556, 765, 913, 959, 890, 726, 512, 301, 136, 35, 0, 0, 5, 15, 17, 13, 10, 12, 20, 28, 33, 32, 26, 18, 11, 8, 9, 13, 16, 18, 17, 14, 11, 8, 6, 5, 5, 7, 9, 12, 16, 20, 24, 27, 28, 27, 24, 19, 12, 7, 7, 11, 18, 22, 18, 4, 0, 0, 50, 163, 336, 543, 740, 874, 903, 813, 626, 392, 176, 41, 23, 127, 324, 560, 770, 901, 921, 832, 662, 458, 266, 121, 33, 0, 0, 0, 10, 17, 19, 16, 6, 0, 0, 10, 84, 221, 414, 630, 822, 938, 947, 846, 664, 449, 251, 105, 22, 0, 0, 7, 13, 11, 3, 0, 0, 0, 3, 8, 11, 12, 11, 10, 8, 7, 6, 4, 3, 1, 0, 0, 0, 0, 0, 0, 1, 4, 8, 12, 14, 13, 10, 6, 5, 8, 14, 18, 15, 3, 0, 0, 42, 144, 302, 494, 679, 810, 852, 793, 649, 461, 274, 126, 35, 0, 0, 3, 11, 11, 5, 0, 0, 0, 1, 5, 7, 6, 2, 0, 1, 6, 15, 23, 25, 17, 1, 0, 1, 56, 168, 332, 526, 708, 837, 880, 826, 689, 505, 317, 161, 56, 5, 0, 0, 8, 10, 7, 0, 0, 0, 0, 3, 6, 6, 4, 2, 0, 0, 0, 0, 0, 0, 1, 1, 1, 2, 4, 6, 9, 11, 11, 10, 8, 4, 1, 0, 0, 1, 3, 5, 5, 5, 3, 1, 0, 0, 0, 0, 0, 0, 1, 2, 3, 3, 3, 1, 0, 0, 0, 2, 7, 12, 17, 21, 22, 22, 21, 21, 21, 20, 19, 15, 9, 4, 2, 5, 13, 19, 19, 9, 0, 0, 28, 117, 270, 470, 680, 849, 931, 899, 758, 543, 312, 129, 39, 65, 192, 381, 575, 720, 777, 737, 615, 448, 278, 138, 47, 4, 0, 1, 8, 10, 6, 0, 0, 0, 0, 4, 7, 9, 8, 7, 5, 3, 1, 0, 0, 0, 0, 0, 0, 1, 4, 8, 12, 13, 13, 10, 6, 3, 2, 2, 4, 4, 2, 0, 0, 0, 4, 11, 14, 8, 0, 0, 13, 83, 210, 383, 569, 725, 807, 794, 688, 520, 333, 171, 59, 3, 0, 0, 11, 16, 12, 3, 0, 0, 0, 5, 13, 19, 22, 22, 20, 16, 13, 10, 10, 12, 15, 17, 17, 15, 11, 6, 2, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 12, 14, 17, 18, 18, 14, 7, 0, 0, 0, 4, 12, 15, 8, 0, 0, 15, 89, 223, 406, 606, 776, 875, 877, 782, 618, 426, 248, 114, 33, 0, 0, 2, 7, 7, 3, 0, 0, 1, 3, 3, 2, 1, 3, 8, 12, 11, 3, 0, 0, 30, 110, 238, 400, 562, 685, 736, 701, 589, 432, 269, 132, 42, 0, 0, 1, 10, 13, 9, 2, 0, 0, 0, 4, 8, 9, 8, 6, 3, 1, 0, 0, 1, 2, 3, 3, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 5, 8, 12, 15, 17, 17, 14, 11, 7, 5, 5, 7, 12, 17, 21, 22, 20, 15, 9, 3, 0, 0, 1, 6, 11, 15, 17, 16, 14, 9, 5, 2, 1, 2, 3, 3, 1, 0, 0, 0, 2, 10, 14, 10, 0, 0, 1, 51, 153, 302, 475, 634, 738, 760, 696, 563, 397, 237, 111, 33, 0, 0, 2, 9, 10, 6, 1, 0, 0, 1, 5, 8, 9, 9, 7, 5, 3, 1, 0, 0, 0, 0, 2, 3, 3, 3, 2, 1, 0, 0, 0, 2, 6, 10, 15, 19, 21, 20, 18, 13, 8, 4, 0, 0, 0, 2, 8, 12, 15, 14, 11, 7, 5, 8, 12, 12, 6, 0, 0, 15, 84, 208, 378, 564, 726, 821, 826, 739, 585, 403, 233, 104, 28, 0, 0, 3, 8, 7, 2, 0, 0, 0, 4, 7, 8, 6, 4, 1, 0, 0, 0, 0, 2, 3, 3, 3, 3, 2, 1, 1, 1, 2, 3, 4, 5, 5, 4, 2, 1, 0, 0, 0, 2, 6, 10, 16, 20, 24, 26, 26, 23, 18, 12, 6, 2, 0, 1, 4, 11, 18, 25, 29, 29, 24, 16, 9, 7, 10, 14, 14, 6, 0, 0, 19, 92, 217, 381, 553, 691, 759, 740, 640, 487, 320, 174, 70, 15, 0, 0, 5, 8, 5, 0, 0, 0, 0, 4, 6, 7, 5, 2, 0, 0, 1, 3, 6, 10, 13, 16, 17, 16, 15, 13, 13, 15, 16, 15, 7, 0, 0, 17, 80, 192, 346, 518, 671, 764, 770, 685, 527, 340, 175, 79, 82, 183, 359, 562, 740, 846, 854, 766, 606, 417, 240, 106, 27, 0, 0, 6, 12, 11, 5, 0, 0, 0, 3, 5, 4, 1, 0, 2, 6, 9, 7, 0, 0, 0, 40, 124, 256, 419, 586, 718, 784, 768, 675, 528, 362, 210, 96, 28, 0, 0, 1, 6, 6, 3, 0, 0, 0, 0, 3, 4, 4, 3, 1, 0, 0, 0, 2, 6, 10, 14, 16, 15, 13, 10, 6, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 9, 16, 23, 27, 28, 25, 18, 10, 3, 0, 0, 3, 11, 19, 27, 32, 33, 29, 22, 13, 5, 1, 2, 7, 11, 11, 4, 0, 0, 32, 107, 229, 385, 547, 676, 738, 713, 602, 434, 255, 115, 55, 96, 229, 422, 626, 791, 878, 870, 775, 619, 439, 270, 138, 53, 11, 0, 0, 3, 4, 3, 2, 3, 6, 9, 12, 12, 11, 10, 9, 9, 9, 8, 6, 4, 2, 1, 1, 3, 3, 3, 2, 1, 1, 4, 10, 19, 29, 38, 43, 44, 41, 33, 24, 15, 8, 3, 0, 0, 0, 0, 0, 0, 0, 4, 9, 17, 24, 28, 29, 27, 22, 16, 10, 5, 0, 0, 0, 0, 4, 11, 15, 10, 0, 0, 3, 59, 172, 341, 542, 734, 872, 919, 864, 721, 528, 329, 164, 55, 2, 0, 1, 15, 21, 18, 11, 3, 0, 0, 0, 2, 6, 8, 10, 11, 12, 12, 12, 13, 15, 18, 20, 20, 18, 15, 10, 6, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 4, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 6, 9, 11, 12, 11, 9, 6, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 5, 7, 9, 12, 14, 16, 18, 19, 20, 21, 20, 19, 18, 17, 17, 18, 20, 22, 23, 23, 20, 15, 10, 5, 2, 0, 2, 5, 10, 15, 19, 23, 25, 24, 20, 15, 8, 3, 2, 4, 9, 12, 10, 2, 0, 0, 38, 123, 258, 430, 610, 758, 838, 828, 731, 571, 388, 221, 96, 23, 0, 0, 1, 9, 12, 10, 3, 0, 0, 1, 40, 118, 236, 382, 532, 652, 716, 708, 630, 502, 353, 213, 104, 35, 3, 0, 0, 5, 6, 3, 0, 0, 0, 0, 1, 3, 2, 2, 0, 0, 0, 0, 1, 2, 4, 5, 5, 5, 3, 2, 1, 0, 0, 0, 0, 0, 0, 2, 4, 5, 6, 6, 5, 4, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 4, 6, 6, 6, 4, 2, 1, 0, 0, 0, 1, 3, 4, 4, 4, 3, 2, 0, 0, 0, 3, 10, 15, 15, 9, 0, 0, 15, 77, 191, 349, 530, 696, 808, 838, 776, 636, 453, 273, 141, 93, 144, 283, 480, 687, 854, 940, 924, 813, 635, 433, 249, 111, 30, 0, 0, 4, 10, 10, 5, 0, 0, 0, 3, 5, 6, 4, 2, 2, 5, 10, 15, 15, 10, 0, 0, 7, 53, 144, 277, 438, 599, 726, 790, 776, 687, 546, 382, 228, 109, 35, 1, 0, 0, 5, 6, 3, 0, 0, 4, 8, 12, 14, 15, 16, 17, 15, 9, 0, 0, 2, 43, 129, 257, 412, 564, 679, 728, 700, 603, 463, 311, 177, 80, 23, 0, 0, 0, 6, 11, 12, 9, 0, 0, 0, 28, 106, 235, 406, 591, 752, 852, 870, 803, 668, 498, 327, 183, 81, 23, 0, 0, 1, 5, 5, 2, 0, 0, 0, 1, 3, 3, 2, 1, 0, 0, 0, 2, 4, 6, 7, 7, 5, 3, 0, 0, 0, 0, 3, 6, 5, 0, 0, 4, 38, 112, 229, 378, 534, 666, 741, 742, 666, 534, 375, 224, 106, 31, 0, 0, 0, 10, 17, 16, 8, 0, 0, 6, 60, 167, 325, 517, 707, 854, 922, 893, 772, 590, 392, 230, 143, 150, 246, 401, 573, 713, 787, 777, 688, 544, 380, 228, 111, 38, 4, 0, 1, 6, 8, 5, 1, 0, 0, 0, 4, 8, 13, 17, 19, 19, 18, 16, 15, 14, 13, 12, 9, 6, 3, 4, 8, 14, 18, 15, 5, 0, 0, 35, 122, 262, 443, 634, 796, 890, 892, 802, 642, 454, 285, 179, 161, 233, 373, 539, 682, 760, 752, 659, 509, 345, 214, 154, 179, 281, 430, 587, 711, 771, 754, 668, 532, 377, 232, 118, 44, 7, 0, 0, 6, 10, 11, 10, 9, 9, 10, 11, 10, 9, 8, 9, 11, 14, 17, 19, 18, 16, 12, 8, 4, 1, 0, 0, 0, 1, 3, 7, 13, 18, 22, 24, 22, 17, 13, 11, 11, 11, 8, 1, 0, 4, 43, 128, 266, 446, 641, 812, 917, 931, 847, 686, 491, 312, 195, 166, 228, 356, 510, 647, 729, 737, 671, 551, 403, 259, 141, 60, 16, 0, 0, 2, 4, 3, 0, 0, 0, 0, 3, 6, 8, 8, 7, 6, 4, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 1, 1, 1, 0, 0, 0, 0, 0, 0, 2, 5, 10, 15, 17, 17, 13, 8, 3, 0, 0, 2, 5, 8, 10, 10, 10, 11, 11, 11, 9, 7, 6, 7, 10, 12, 9, 1, 0, 0, 41, 132, 275, 456, 641, 790, 866, 848, 743, 579, 403, 261, 188, 199, 284, 416, 554, 660, 707, 683, 597, 469, 327, 197, 97, 33, 2, 0, 0, 7, 10, 7, 0, 0, 0, 33, 102, 209, 347, 495, 625, 711, 733, 687, 584, 446, 301, 174, 80, 24, 0, 0, 1, 8, 11, 12, 11, 10, 9, 8, 7, 5, 2, 1, 0, 0, 0, 1, 2, 2, 3, 3, 2, 1, 0, 0, 0, 0, 0, 1, 2, 4, 5, 6, 7, 8, 11, 16, 22, 24, 20, 8, 0, 0, 28, 106, 235, 406, 592, 757, 864, 891, 833, 705, 537, 364, 214, 104, 36, 4, 0, 0, 4, 7, 6, 0, 0, 0, 18, 73, 168, 300, 452, 597, 705, 750, 721, 625, 485, 331, 192, 88, 26, 0, 0, 1, 6, 8, 7, 6, 6, 7, 8, 7, 5, 2, 1, 1, 2, 4, 5, 4, 3, 2, 2, 2, 4, 6, 10, 13, 15, 15, 10, 2, 0, 0, 28, 93, 199, 340, 495, 637, 737, 773, 738, 639, 499, 346, 207, 101, 35, 4, 0, 0, 4, 5, 2, 0, 0, 0, 4, 6, 7, 5, 2, 0, 1, 3, 4, 1, 0, 0, 15, 62, 144, 261, 395, 523, 616, 654, 627, 544, 424, 294, 177, 89, 34, 8, 0, 0, 0, 0, 0, 1, 6, 14, 22, 28, 30, 28, 22, 15, 9, 5, 2, 0, 0, 0, 0, 0, 0, 1, 1, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 7, 12, 18, 23, 26, 26, 24, 21, 18, 17, 18, 20, 23, 24, 23, 19, 13, 7, 1, 0, 0, 1, 6, 10, 9, 2, 0, 0, 30, 99, 208, 349, 498, 627, 705, 714, 655, 545, 413, 296, 225, 217, 273, 377, 501, 613, 683, 694, 643, 541, 410, 274, 157, 72, 21, 0, 0, 1, 5, 7, 6, 6, 7, 8, 10, 10, 10, 11, 13, 16, 19, 19, 14, 4, 0, 0, 27, 90, 189, 318, 458, 585, 674, 708, 682, 601, 485, 354, 231, 131, 61, 20, 2, 0, 0, 1, 1, 0, 0, 0, 0, 1, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1, 2, 4, 6, 9, 11, 12, 11, 9, 7, 4, 2, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 4, 6, 6, 6, 4, 1, 0, 0, 1, 4, 8, 11, 13, 13, 12, 9, 6, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 4, 5, 5, 4, 2, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 4, 5, 5, 4, 4, 4, 5, 4, 2, 0, 0, 8, 40, 101, 189, 298, 411, 508, 571, 588, 554, 478, 375, 263, 161, 81, 29, 4, 0, 0, 2, 3, 2, 0, 0, 0, 0, 3, 5, 7, 7, 7, 7, 6, 6, 7, 8, 8, 9, 9, 10, 11, 13, 15, 18, 19, 20, 20, 19, 17, 14, 11, 9, 7, 6, 5, 4, 2, 0, 0, 5, 28, 74, 146, 241, 349, 453, 536, 578, 571, 513, 416, 304, 205, 150, 158, 234, 364, 520, 665, 763, 792, 745, 633, 483, 325, 186, 85, 25, 0, 0, 4, 11, 13, 11, 8, 6, 5, 6, 8, 10, 10, 9, 7, 6, 4, 3, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 3, 4, 5, 8, 13, 18, 20, 17, 9, 0, 0, 12, 64, 158, 288, 438, 581, 691, 745, 735, 664, 548, 411, 275, 159, 74, 23, 0, 0, 1, 5, 5, 0, 0, 0, 23, 73, 151, 254, 367, 474, 556, 597, 591, 539, 453, 353, 263, 206, 199, 245, 338, 455, 568, 649, 677, 645, 560, 439, 308, 188, 97, 38, 8, 0, 0, 4, 7, 10, 12, 14, 17, 19, 20, 20, 19, 19, 20, 22, 24, 25, 24, 20, 16, 10, 5, 1, 1, 2, 6, 10, 13, 16, 18, 19, 21, 22, 23, 22, 19, 15, 11, 7, 4, 1, 0, 0, 7, 32, 81, 158, 260, 376, 487, 572, 614, 601, 538, 440, 334, 249, 213, 237, 319, 440, 569, 673, 725, 714, 642, 527, 393, 264, 157, 81, 35, 12, 3, 1, 0, 0, 0, 0, 0, 2, 3, 4, 6, 8, 12, 16, 18, 16, 10, 3, 4, 24, 73, 158, 275, 411, 544, 650, 707, 705, 643, 536, 405, 274, 161, 78, 28, 5, 0, 0, 0, 0, 0, 0, 0, 2, 5, 6, 6, 4, 1, 0, 0, 1, 3, 4, 6, 6, 6, 5, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 6, 9, 11, 11, 9, 6, 4, 4, 6, 10, 13, 15, 14, 11, 6, 2, 0, 1, 11, 32, 72, 134, 220, 326, 437, 537, 603, 621, 584, 501, 387, 266, 158, 78, 28, 5, 0, 2, 5, 6, 5, 4, 2, 2, 2, 3, 3, 3, 3, 2, 2, 2, 2, 3, 3, 4, 4, 3, 3, 1, 0, 0, 0, 0, 2, 5, 8, 10, 11, 10, 9, 8, 8, 9, 9, 7, 4, 3, 10, 33, 83, 162, 268, 389, 504, 591, 631, 616, 549, 445, 324, 209, 116, 52, 16, 2, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 0, 0, 0, 0, 0, 3, 7, 11, 14, 16, 17, 15, 13, 11, 10, 10, 12, 13, 14, 13, 10, 8, 5, 4, 5, 7, 10, 12, 12, 11, 7, 3, 0, 0, 6, 26, 65, 126, 209, 308, 410, 499, 557, 573, 541, 467, 365, 255, 155, 78, 28, 4, 0, 0, 2, 3, 1, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 2, 4, 6, 8, 8, 8, 6, 5, 4, 3, 4, 4, 3, 2, 0, 0, 0, 1, 3, 5, 6, 5, 4, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 1, 0, 0, 0, 0, 1, 2, 2, 0, 0, 6, 31, 84, 168, 281, 409, 533, 632, 686, 686, 632, 538, 422, 303, 200, 120, 66, 35, 19, 12, 8, 4, 0, 0, 0, 18, 59, 127, 222, 335, 452, 554, 623, 647, 622, 551, 449, 332, 219, 124, 56, 16, 0, 0, 1, 4, 2, 0, 0, 5, 31, 80, 152, 241, 336, 424, 489, 520, 513, 469, 397, 311, 222, 143, 82, 40, 16, 4, 0, 0, 0, 5, 20, 50, 99, 168, 253, 345, 431, 497, 529, 520, 469, 387, 287, 188, 104, 45, 12, 0, 0, 1, 1, 0, 0, 12, 45, 104, 188, 289, 393, 484, 544, 563, 536, 470, 377, 274, 177, 99, 45, 14, 1, 0, 0, 1, 1, 0, 0, 0, 2, 5, 7, 8, 8, 7, 4, 2, 1, 0, 1, 2, 3, 2, 1, 0, 0, 0, 0, 0, 0, 1, 11, 35, 82, 157, 261, 385, 514, 630, 710, 739, 709, 626, 506, 371, 242, 136, 62, 19, 1, 0, 1, 4, 2, 0, 0, 4, 33, 88, 172, 280, 399, 512, 601, 650, 651, 605, 521, 412, 297, 191, 107, 50, 18, 4, 0, 0, 0, 0, 11, 41, 95, 174, 271, 373, 464, 528, 553, 536, 480, 397, 302, 209, 131, 73, 35, 16, 7, 3, 1, 0, 0, 9, 37, 88, 166, 265, 373, 473, 548, 583, 572, 518, 432, 329, 228, 140, 74, 31, 9, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 2, 1, 1, 0, 0, 0, 1, 3, 6, 9, 13, 16, 20, 22, 24, 22, 18, 12, 6, 1, 0, 7, 23, 53, 99, 163, 242, 330, 414, 479, 511, 502, 453, 378, 296, 231, 200, 211, 261, 337, 418, 485, 523, 523, 487, 420, 333, 241, 155, 85, 37, 10, 0, 0, 1, 3, 3, 2, 0, 0, 0, 0, 1, 3, 4, 2, 0, 0, 5, 31, 82, 157, 251, 351, 443, 511, 546, 543, 504, 439, 356, 267, 183, 111, 57, 22, 4, 0, 0, 0, 0, 0, 5, 22, 57, 111, 184, 270, 359, 438, 495, 518, 504, 453, 375, 284, 194, 117, 60, 24, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 2, 5, 8, 10, 10, 8, 5, 2, 0, 0, 0, 2, 3, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 3, 2, 0, 0, 0, 2, 6, 11, 12, 9, 3, 0, 4, 28, 76, 150, 243, 346, 443, 522, 572, 588, 568, 517, 443, 355, 264, 178, 106, 53, 19, 2, 0, 0, 1, 4, 5, 7, 8, 10, 11, 11, 10, 8, 7, 5, 2, 0, 0, 3, 19, 53, 105, 173, 249, 322, 381, 414, 418, 392, 343, 279, 211, 147, 94, 53, 27, 11, 3, 0, 0, 1, 2, 3, 3, 4, 6, 9, 11, 13, 13, 13, 11, 8, 6, 6, 7, 10, 15, 19, 22, 21, 17, 11, 5, 5, 16, 44, 94, 167, 258, 360, 459, 542, 595, 608, 578, 512, 419, 317, 219, 138, 80, 43, 24, 15, 11, 6, 0, 0, 1, 23, 69, 141, 235, 342, 445, 531, 585, 602, 579, 522, 440, 346, 250, 164, 95, 46, 16, 2, 0, 0, 0, 1, 0, 0, 0, 0, 2, 4, 5, 5, 5, 4, 2, 2, 2, 3, 4, 4, 4, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 2, 4, 5, 5, 5, 3, 1, 0, 0, 0, 0, 1, 2, 2, 2, 2, 2, 2, 1, 1, 0, 0, 1, 2, 3, 3, 3, 3, 5, 8, 12, 17, 24, 38, 63, 105, 169, 254, 351, 449, 531, 582, 593, 562, 495, 404, 303, 207, 126, 65, 26, 5, 0, 1, 7, 12, 15, 14, 12, 10, 10, 12, 16, 20, 22, 21, 19, 21, 32, 60, 110, 181, 269, 361, 446, 511, 545, 546, 515, 461, 396, 336, 292, 274, 285, 322, 376, 434, 483, 511, 509, 476, 415, 335, 249, 168, 100, 52, 22, 6, 1, 0, 0, 0, 0, 0, 1, 3, 6, 8, 8, 7, 5, 3, 1, 0, 0, 0, 0, 1, 3, 5, 8, 11, 12, 11, 10, 8, 5, 4, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 4, 3, 1, 1, 1, 2, 4, 5, 5, 5, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 3, 5, 6, 6, 4, 2, 1, 0, 0, 1, 2, 3, 3, 2, 2, 3, 4, 5, 6, 6, 5, 3, 1, 0, 0, 1, 3, 5, 6, 6, 6, 6, 7, 8, 9, 9, 9, 7, 4, 2, 1, 0, 1, 1, 0, 0, 0, 0, 2, 4, 6, 7, 7, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 4, 5, 6, 8, 10, 12, 13, 14, 13, 12, 10, 8, 6, 5, 3, 1, 0, 0, 3, 16, 43, 88, 151, 229, 316, 402, 473, 519, 531, 510, 460, 395, 333, 290, 278, 301, 353, 420, 485, 531, 548, 532, 483, 411, 327, 241, 163, 99, 52, 22, 6, 0, 0, 0, 1, 3, 6, 9, 13, 16, 18, 18, 16, 13, 10, 6, 6, 13, 31, 66, 120, 193, 279, 367, 445, 499, 521, 508, 462, 392, 311, 230, 159, 102, 60, 33, 17, 7, 2, 0, 0, 2, 6, 10, 14, 16, 15, 13, 9, 6, 4, 4, 5, 6, 7, 7, 7, 7, 9, 11, 14, 17, 17, 17, 15, 12, 9, 7, 6, 6, 6, 7, 7, 8, 7, 4, 1, 1, 8, 26, 58, 106, 167, 237, 308, 373, 425, 458, 468, 453, 414, 356, 285, 211, 141, 83, 41, 15, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 6, 8, 11, 14, 16, 16, 14, 10, 6, 3, 1, 2, 5, 8, 10, 11, 9, 6, 3, 1, 0, 1, 4, 7, 9, 10, 9, 8, 8, 9, 10, 12, 14, 13, 13, 14, 21, 39, 70, 119, 183, 259, 339, 412, 466, 493, 490, 459, 409, 353, 305, 278, 278, 305, 352, 409, 464, 504, 522, 512, 474, 414, 337, 256, 178, 113, 64, 31, 13, 5, 1, 0, 0, 0, 0, 3, 7, 11, 15, 16, 16, 15, 14, 14, 16, 18, 20, 21, 21, 19, 16, 12, 9, 6, 4, 3, 4, 8, 18, 37, 69, 115, 176, 246, 318, 382, 428, 447, 437, 398, 339, 268, 196, 133, 82, 45, 22, 9, 2, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 2, 4, 5, 5, 4, 2, 1, 0, 0, 1, 1, 2, 1, 1, 0, 0, 0, 1, 2, 2, 1, 0, 0, 0, 2, 5, 7, 8, 5, 2, 0, 0, 8, 23, 46, 74, 107, 145, 188, 236, 287, 335, 372, 391, 387, 361, 324, 287, 266, 269, 297, 345, 400, 449, 480, 486, 466, 423, 364, 298, 230, 167, 113, 70, 39, 19, 8, 6, 8, 12, 15, 18, 18, 18, 18, 19, 20, 20, 18, 14, 11, 10, 16, 32, 61, 104, 160, 224, 291, 351, 396, 419, 418, 397, 364, 330, 306, 296, 305, 330, 366, 405, 441, 468, 478, 467, 435, 383, 317, 245, 175, 116, 70, 39, 20, 10, 7, 7, 9, 10, 10, 10, 9, 10, 12, 16, 21, 24, 24, 24, 25, 34, 57, 97, 155, 227, 308, 388, 459, 513, 545, 551, 530, 484, 416, 333, 246, 164, 95, 45, 15, 2, 1, 7, 17, 29, 47, 73, 112, 166, 231, 302, 368, 419, 447, 451, 434, 402, 368, 342, 333, 345, 378, 425, 477, 524, 553, 558, 535, 485, 416, 336, 255, 181, 119, 72, 39, 18, 6, 0, 0, 0, 1, 4, 8, 13, 18, 21, 21, 19, 14, 10, 6, 5, 6, 9, 12, 14, 14, 12, 9, 6, 4, 3, 3, 3, 4, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 2, 1, 1, 0, 0, 0, 0, 1, 3, 6, 9, 13, 18, 28, 43, 65, 96, 135, 180, 227, 270, 305, 325, 329, 314, 284, 242, 194, 146, 103, 69, 46, 32, 26, 24, 26, 31, 45, 71, 115, 175, 249, 325, 393, 440, 462, 455, 425, 377, 320, 258, 198, 142, 93, 53, 24, 6, 0, 0, 0, 5, 9, 12, 14, 15, 14, 11, 8, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 6, 7, 7, 7, 7, 6, 5, 5, 5, 5, 6, 7, 8, 8, 7, 6, 4, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 4, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 4, 5, 5, 4, 4, 3, 3, 2, 1, 0, 0, 0, 3, 12, 27, 50, 79, 117, 163, 217, 276, 335, 388, 428, 447, 444, 416, 368, 305, 237, 171, 112, 66, 33, 13, 3, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 3, 5, 6, 7, 6, 5, 4, 3, 2, 2, 1, 1, 0, 0, 0, 0, 1, 2, 3, 3, 2, 1, 0, 0, 0, 0, 0, 1, 2, 2, 1, 0, 0, 0, 0, 2, 5, 11, 18, 28, 44, 69, 104, 150, 205, 264, 320, 366, 397, 409, 400, 373, 331, 278, 219, 161, 107, 62, 29, 9, 4, 11, 30, 58, 93, 135, 184, 237, 294, 347, 390, 417, 425, 413, 387, 356, 330, 316, 317, 331, 353, 377, 396, 407, 406, 391, 362, 319, 267, 210, 153, 104, 64, 36, 19, 10, 6, 3, 2, 1, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 4, 4, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 2, 3, 4, 5, 4, 3, 1, 0, 0, 2, 3, 3, 3, 4, 11, 27, 56, 101, 160, 228, 298, 362, 413, 446, 458, 449, 422, 378, 323, 263, 204, 151, 106, 71, 46, 27, 15, 7, 2, 0, 0, 0, 0, 1, 2, 2, 3, 5, 10, 18, 33, 57, 89, 131, 181, 235, 289, 338, 375, 396, 398, 380, 345, 296, 241, 186, 137, 97, 68, 48, 33, 23, 15, 8, 3, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 6, 15, 31, 55, 88, 130, 178, 230, 282, 329, 365, 386, 389, 376, 347, 307, 260, 211, 162, 116, 76, 44, 21, 6, 0, 0, 2, 4, 5, 5, 3, 1, 0, 0, 0, 1, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 1, 7, 19, 40, 73, 115, 165, 217, 267, 309, 339, 357, 360, 349, 327, 293, 250, 201, 151, 104, 65, 36, 20, 16, 22, 38, 62, 95, 137, 188, 246, 306, 363, 410, 441, 451, 438, 403, 352, 291, 227, 167, 116, 76, 47, 28, 16, 10, 8, 8, 11, 13, 15, 15, 13, 9, 5, 3, 3, 5, 9, 16, 24, 36, 54, 80, 116, 161, 212, 263, 307, 338, 352, 347, 325, 289, 243, 194, 146, 103, 68, 43, 26, 16, 13, 16, 23, 37, 58, 87, 125, 170, 222, 277, 332, 381, 419, 440, 441, 421, 383, 331, 272, 212, 156, 107, 68, 40, 24, 21, 32, 54, 85, 123, 163, 203, 241, 276, 304, 324, 332, 324, 301, 264, 218, 169, 123, 84, 55, 34, 20, 11, 5, 1, 0, 0, 0, 1, 1, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 5, 7, 8, 8, 6, 3, 0, 0, 1, 4, 9, 16, 26, 40, 60, 86, 119, 157, 200, 246, 291, 333, 365, 385, 387, 370, 337, 290, 238, 186, 139, 100, 70, 49, 36, 31, 36, 52, 80, 119, 165, 216, 264, 304, 332, 346, 344, 328, 300, 264, 222, 178, 136, 96, 64, 40, 27, 24, 32, 49, 73, 105, 142, 186, 233, 281, 325, 360, 380, 384, 370, 342, 303, 259, 213, 169, 129, 93, 62, 38, 20, 8, 1, 0, 0, 1, 2, 4, 5, 6, 7, 8, 9, 11, 17, 29, 51, 84, 126, 173, 221, 262, 293, 312, 316, 308, 290, 264, 233, 197, 160, 124, 89, 60, 36, 19, 8, 2, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 6, 13, 22, 33, 48, 69, 99, 140, 193, 253, 313, 365, 400, 412, 402, 372, 330, 283, 235, 189, 147, 108, 73, 43, 21, 6, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 2, 3, 4, 3, 3, 2, 1, 0, 0, 0, 0, 0, 0, 1, 3, 5, 6, 6, 5, 3, 1, 0, 0, 0, 3, 5, 7, 7, 6, 4, 2, 0, 0, 0, 0, 1, 2, 3, 4, 4, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 3, 3, 3, 2, 2, 1, 0, 0, 0, 0, 0, 1, 2, 2, 3, 3, 3, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 4, 6, 9, 10, 10, 8, 6, 3, 1, 2, 4, 7, 10, 12, 11, 9, 5, 1, 0, 0, 3, 7, 11, 13, 12, 8, 4, 2, 2, 4, 9, 15, 23, 33, 49, 73, 106, 147, 192, 237, 277, 308, 330, 341, 341, 331, 310, 278, 237, 192, 145, 104, 71, 48, 34, 26, 22, 18, 15, 11, 8, 5, 5, 5, 6, 7, 8, 9, 10, 9, 7, 5, 3, 1, 1, 3, 5, 7, 7, 7, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 5, 8, 10, 11, 9, 7, 4, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 5, 6, 6, 6, 6, 7, 7, 8, 9, 10, 12, 16, 24, 36, 53, 75, 103, 136, 174, 217, 262, 306, 342, 366, 375, 368, 347, 316, 278, 237, 197, 159, 124, 93, 67, 47, 32, 21, 14, 10, 7, 6, 6, 8, 12, 16, 21, 26, 33, 43, 59, 82, 115, 156, 203, 252, 298, 336, 363, 375, 372, 357, 330, 297, 259, 219, 180, 144, 112, 85, 67, 58, 59, 71, 91, 117, 146, 176, 205, 234, 262, 287, 307, 318, 316, 301, 272, 233, 190, 149, 114, 89, 72, 63, 60, 62, 70, 85, 108, 140, 181, 225, 270, 309, 339, 357, 362, 354, 333, 300, 259, 213, 165, 121, 85, 58, 40, 30, 24, 19, 14, 8, 3, 0, 0, 0, 3, 5, 7, 6, 5, 2, 0, 0, 0, 2, 5, 8, 10, 10, 9, 6, 3, 1, 0, 1, 2, 3, 3, 2, 1, 0, 0, 3, 9, 17, 26, 36, 49, 68, 94, 129, 171, 217, 262, 302, 333, 352, 359, 354, 337, 310, 273, 229, 182, 137, 98, 68, 48, 36, 30, 25, 21, 16, 12, 9, 9, 9, 10, 10, 9, 6, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 2, 3, 3, 4, 6, 8, 12, 17, 23, 32, 44, 60, 81, 107, 139, 175, 216, 257, 296, 328, 348, 355, 346, 325, 294, 257, 219, 181, 145, 113, 86, 66, 55, 53, 63, 82, 111, 145, 183, 222, 259, 290, 313, 326, 328, 318, 300, 274, 244, 211, 177, 143, 111, 82, 57, 36, 20, 10, 4, 1, 0, 0, 0, 2, 4, 7, 9, 10, 10, 9, 9, 8, 8, 9, 10, 11, 12, 14, 16, 19, 23, 29, 37, 50, 69, 95, 127, 166, 208, 252, 294, 333, 367, 394, 413, 423, 424, 416, 401, 384, 368, 358, 356, 362, 375, 390, 402, 406, 402, 392, 381, 376, 381, 394, 412, 427, 434, 427, 407, 375, 335, 291, 246, 202, 161, 125, 95, 70, 51, 37, 26, 17, 10, 5, 2, 1, 2, 4, 6, 7, 9, 10, 11, 13, 15, 17, 19, 18, 17, 14, 10, 7, 4, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 5, 5, 4, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 2, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 4, 10, 20, 31, 44, 58, 75, 97, 126, 161, 198, 235, 263, 280, 282, 271, 249, 221, 191, 161, 134, 108, 85, 64, 46, 31, 19, 11, 5, 2, 1, 0, 0, 0, 0, 0, 1, 2, 4, 6, 7, 6, 5, 2, 0, 0, 1, 5, 12, 25, 43, 66, 92, 120, 149, 180, 212, 248, 285, 320, 350, 370, 375, 364, 341, 308, 271, 232, 194, 157, 123, 91, 63, 40, 23, 11, 5, 1, 0, 0, 0, 0, 1, 3, 4, 4, 3, 1, 0, 0, 1, 4, 7, 9, 11, 14, 17, 22, 29, 38, 48, 59, 74, 93, 119, 152, 188, 223, 254, 276, 288, 290, 285, 272, 254, 230, 199, 164, 128, 96, 73, 62, 63, 73, 88, 105, 124, 145, 169, 196, 225, 253, 275, 288, 288, 274, 249, 217, 183, 150, 123, 102, 86, 73, 63, 54, 48, 49, 58, 78, 107, 142, 180, 217, 249, 275, 295, 307, 313, 310, 298, 278, 250, 216, 178, 140, 106, 77, 54, 38, 26, 17, 10, 5, 1, 0, 0, 0, 1, 2, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 1, 0, 0, 1, 6, 16, 32, 51, 74, 99, 123, 145, 163, 177, 184, 186, 181, 172, 160, 146, 131, 115, 98, 79, 60, 41, 24, 11, 3, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 4, 9, 15, 21, 27, 35, 47, 66, 94, 129, 168, 207, 240, 263, 273, 273, 264, 251, 236, 221, 204, 183, 159, 131, 101, 74, 51, 35, 25, 20, 17, 14, 10, 6, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 6, 11, 18, 29, 44, 63, 88, 115, 141, 165, 187, 207, 227, 249, 272, 291, 301, 297, 277, 244, 206, 168, 136, 112, 95, 81, 68, 57, 50, 52, 65, 88, 119, 152, 182, 208, 229, 244, 255, 261, 262, 255, 240, 218, 191, 162, 134, 108, 84, 64, 47, 32, 20, 12, 7, 4, 4, 5, 7, 10, 16, 25, 37, 52, 70, 91, 116, 144, 176, 209, 241, 270, 293, 308, 316, 314, 303, 284, 260, 231, 200, 170, 141, 115, 90, 68, 48, 31, 17, 8, 3, 2, 3, 4, 5, 5, 4, 2, 0, 0, 0, 1, 4, 8, 13, 16, 19, 20, 22, 26, 33, 45, 62, 83, 109, 137, 168, 198, 229, 257, 282, 302, 316, 321, 317, 302, 276, 241, 200, 159, 124, 99, 86, 85, 92, 104, 117, 128, 139, 152, 169, 189, 213, 236, 255, 267, 269, 261, 245, 222, 194, 164, 135, 110, 91, 80, 78, 84, 96, 114, 135, 159, 185, 212, 238, 261, 279, 291, 297, 296, 288, 272, 250, 223, 191, 159, 130, 107, 91, 83, 80, 80, 82, 85, 93, 106, 126, 154, 188, 225, 260, 288, 308, 316, 315, 306, 291, 272, 249, 223, 192, 158, 124, 94, 72, 60, 59, 69, 85, 106, 129, 154, 182, 213, 246, 278, 305, 322, 328, 322, 308, 288, 266, 245, 225, 205, 183, 160, 136, 115, 99, 94, 100, 118, 144, 174, 205, 231, 253, 268, 276, 279, 276, 267, 253, 233, 208, 181, 152, 123, 95, 70, 49, 32, 20, 14, 11, 11, 13, 15, 16, 16, 18, 24, 35, 51, 73, 97, 121, 144, 165, 186, 207, 229, 252, 274, 293, 308, 318, 324, 327, 331, 337, 345, 354, 361, 364, 362, 355, 343, 326, 306, 283, 257, 228, 198, 168, 140, 116, 95, 80, 70, 67, 71, 84, 103, 128, 155, 183, 208, 230, 250, 269, 287, 305, 321, 333, 340, 341, 338, 335, 334, 336, 342, 349, 355, 359, 361, 361, 362, 364, 367, 371, 372, 369, 362, 351, 337, 323, 310, 297, 285, 270, 252, 231, 205, 178, 151, 129, 114, 108, 113, 126, 145, 166, 186, 204, 220, 235, 249, 261, 269, 272, 267, 253, 231, 204, 173, 142, 113, 90, 73, 64, 63, 69, 81, 96, 114, 134, 156, 181, 207, 234, 257, 275, 283, 281, 271, 254, 233, 211, 189, 166, 141, 116, 89, 65, 44, 29, 19, 14, 12, 10, 7, 5, 2, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 6, 14, 24, 36, 49, 65, 82, 102, 125, 149, 172, 193, 211, 225, 236, 244, 249, 249, 242, 226, 202, 171, 138, 106, 80, 61, 48, 38, 29, 21, 13, 6, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 7, 17, 33, 53, 75, 98, 122, 146, 169, 192, 212, 227, 236, 238, 232, 220, 204, 186, 167, 147, 127, 108, 89, 71, 55, 41, 29, 21, 15, 10, 8, 5, 3, 2, 3, 10, 23, 45, 73, 106, 137, 165, 186, 202, 215, 229, 244, 260, 272, 278, 272, 257, 233, 204, 175, 148, 124, 102, 82, 63, 45, 29, 16, 7, 2, 0, 0, 1, 4, 6, 8, 9, 9, 8, 6, 3, 1, 0, 0, 0, 2, 4, 8, 13, 18, 26, 37, 51, 70, 91, 115, 139, 162, 183, 202, 218, 231, 239, 242, 240, 233, 222, 208, 191, 171, 149, 124, 98, 74, 56, 48, 51, 65, 86, 110, 131, 147, 158, 168, 180, 197, 220, 244, 265, 279, 285, 285, 282, 281, 285, 294, 303, 309, 309, 301, 287, 268, 246, 224, 201, 177, 152, 128, 104, 83, 66, 52, 41, 32, 25, 18, 12, 8, 4, 2, 1, 0, 0, 0, 0, 1, 4, 7, 8, 7, 4, 1, 0, 0, 8, 21, 37, 53, 68, 81, 93, 109, 131, 158, 188, 217, 239, 252, 255, 251, 242, 231, 218, 204, 188, 168, 146, 124, 105, 91, 82, 78, 79, 86, 98, 117, 140, 165, 190, 209, 222, 229, 230, 227, 222, 215, 204, 189, 169, 147, 124, 103, 86, 72, 61, 50, 38, 26, 15, 8, 6, 10, 18, 27, 36, 42, 47, 54, 65, 83, 109, 141, 175, 206, 232, 249, 257, 259, 255, 249, 240, 229, 215, 198, 177, 155, 134, 116, 103, 97, 97, 102, 109, 117, 127, 138, 151, 167, 184, 200, 211, 216, 213, 202, 187, 168, 149, 130, 113, 96, 80, 65, 52, 40, 29, 19, 11, 4, 0, 0, 2, 7, 15, 25, 36, 48, 64, 83, 108, 136, 165, 192, 214, 232, 246, 259, 272, 287, 303, 317, 329, 337, 341, 341, 339, 335, 331, 327, 324, 323, 326, 332, 340, 348, 352, 351, 345, 336, 326, 317, 310, 303, 293, 278, 257, 231, 203, 178, 156, 138, 120, 101, 80, 57, 35, 18, 7, 3, 3, 5, 6, 5, 3, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 3, 10, 20, 32, 42, 48, 54, 63, 85, 119, 158, 191, 213, 226, 233))


class TestAbiWrongMode(unittest.TestCase):

    def test_file_mode(self):
        """Test if exception is raised if file is not opened in 'rb' mode."""
        open_files_wrong_mode(test_data)
        for trace in test_data:
            self.assertRaises(ValueError, SeqIO.read, test_data[trace]['handle'], 'abi')
        close_files(test_data)


class TestAbiFake(unittest.TestCase):

    def test_file_type(self):
        """Test if error is raised if filetype is not ABIF."""
        open_files(test_data_fake)
        for trace in test_data_fake:
            self.assertRaises(IOError, SeqIO.read, test_data_fake[trace]['handle'], 'abi')
        close_files(test_data_fake)


if __name__ == "__main__":
    runner = unittest.TextTestRunner(verbosity=2)
    unittest.main(testRunner=runner)
