/*******************************************************************************
 * Copyright (c) 2020, 2021 Christoph Läubrich and others.
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Christoph Läubrich - initial API and implementation
 *******************************************************************************/
package org.eclipse.tycho.test.target;

import static org.junit.Assert.assertFalse;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.maven.it.VerificationException;
import org.apache.maven.it.Verifier;
import org.eclipse.tycho.core.osgitools.DefaultBundleReader;
import org.eclipse.tycho.core.osgitools.OsgiManifest;
import org.eclipse.tycho.test.AbstractTychoIntegrationTest;
import org.junit.Assert;
import org.junit.Ignore;
import org.junit.Test;

public class TargetPlatformLocationsTest extends AbstractTychoIntegrationTest {

	@Test
	public void testMavenLocation() throws Exception {
		Verifier verifier = getVerifier("target.maven", false, true);
		verifier.executeGoal("verify");
		verifier.verifyErrorFreeLog();
	}

	@Test
	public void testMavenLocationScopes() throws Exception {
		Verifier verifier = getVerifier("target.maven-scopes", false, true);
		verifier.executeGoal("verify");
		verifier.verifyErrorFreeLog();
	}

	@Test
	public void testMavenArtifactHaveMavenRepoPath() throws Exception {
		Verifier verifier = getVerifier("target.maven", false, true);
		verifier.addCliOption("-DoutputAbsoluteArtifactFilename=true");
		verifier.executeGoal("dependency:list");
		verifier.verifyErrorFreeLog();
		assertFalse("Location for Maven deps should not resolve to cache",
				Files.readString(Path.of(verifier.getBasedir(), verifier.getLogFileName())).contains("p2/osgi"));
	}

	@Test
	public void testMavenArtifactHaveMavenDepsCoordinates() throws Exception {
		Verifier verifier = getVerifier("target.maven", false, true);
		verifier.executeGoal("dependency:list");
		verifier.verifyErrorFreeLog();
		verifier.verifyTextInLog("commons-lang:commons-lang:jar:2.4:compile"); // this is a weak assert, should
																				// be
		// improved
	}

	@Test
	public void testMavenLocationMulti() throws Exception {
		Verifier verifier = getVerifier("target.mavenMulti", false, true);
		verifier.executeGoal("verify");
		verifier.verifyErrorFreeLog();
	}

	@Test
	@Ignore(value = "This test is flaky on the buildserver")
	public void testMavenLocationRepositories() throws Exception {
		Verifier verifier = getVerifier("target.mavenRepos", false, true);
		verifier.executeGoal("verify");
		verifier.verifyErrorFreeLog();
	}

	@Test
	public void testDirectoryLocation() throws Exception {
		Verifier verifier = getVerifier("target.directory", false, true);
		verifier.executeGoal("verify");
		verifier.verifyErrorFreeLog();
	}

	@Test
	public void testTargetPlatformArtifactCaching() throws Exception {
		Verifier verifier = getVerifier("target.artifact.caching", false, true);
		verifier.getCliOptions().add("-Dtycho.localArtifacts=default");

		File annotBundleManifestFile = new File(verifier.getBasedir(),
				"target.test/plugins/osgi.annotation.bundle_0.0.1/META-INF/MANIFEST.MF");
		DefaultBundleReader reader = new DefaultBundleReader();
		OsgiManifest annotBundleManifest = reader.loadManifest(annotBundleManifestFile);
		Assert.assertEquals("org.osgi.annotation.versioning", annotBundleManifest.getValue("Export-Package"));
		verifier.executeGoal("verify");
		verifier.verifyErrorFreeLog();

		List<String> out = Files.lines(annotBundleManifestFile.toPath())
				.filter(line -> !line.contains("Export-Package")).collect(Collectors.toList());
		Files.write(annotBundleManifestFile.toPath(), out, StandardOpenOption.WRITE,
				StandardOpenOption.TRUNCATE_EXISTING);

		try {
			verifier.executeGoal("verify");
			Assert.fail("Reference to the restricted package did not fail the build");
		} catch (VerificationException expected) {
			verifier.verifyTextInLog(
					"Access restriction: The type 'Version' is not API (restriction on classpath entry");
		}
	}

	@Test
	public void testMavenLocationAutogeneratedFeature() throws Exception {
		Verifier verifier = getVerifier("target.maven.autofeature", false, true);
		verifier.executeGoal("verify");
		verifier.verifyErrorFreeLog();
	}

	@Test
	public void testMavenLocationTransitiveFeature() throws Exception {
		Verifier verifier = getVerifier("target.maven-deps", false, true);
		verifier.executeGoal("verify");
		verifier.verifyErrorFreeLog();
	}
}
