/**********************************************************************
  Copyright(c) 2011-2016 Intel Corporation All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions 
  are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of Intel Corporation nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**********************************************************************/

#include "aes_xts.h"
#include <stdlib.h>
#include <openssl/evp.h>

#define TEST_LEN  (1024*1024)
#define TEST_LOOPS 100
#ifndef RANDOMS
# define RANDOMS  100
#endif

/* Generates random data for keys, tweak and plaintext */
void mk_rand_data(unsigned char *k1, unsigned char *k2, unsigned char *k3, unsigned char *p,
		  int n)
{
	int i;
	for (i = 0; i < 16; i++) {
		*k1++ = rand();
		*k2++ = rand();
		*k3++ = rand();
	}
	for (i = 0; i < n; i++)
		*p++ = rand();

}

/* Wrapper for OpenSSL EVP AES-XTS 128 encryption */
static inline
    int openssl_aes_128_xts_enc(EVP_CIPHER_CTX * ctx, unsigned char *key, unsigned char *iv,
				int len, unsigned char *pt, unsigned char *ct)
{
	int outlen, tmplen;
	if (!EVP_EncryptInit_ex(ctx, EVP_aes_128_xts(), NULL, key, iv))
		printf("\n ERROR!! \n");
	if (!EVP_EncryptUpdate(ctx, ct, &outlen, (const unsigned char *)pt, len))
		printf("\n ERROR!! \n");
	if (!EVP_EncryptFinal_ex(ctx, ct + outlen, &tmplen))
		printf("\n ERROR!! \n");

	return 0;
}

/* Wrapper for OpenSSL EVP AES-XTS 128 decryption */
static inline
    int openssl_aes_128_xts_dec(EVP_CIPHER_CTX * ctx, unsigned char *key, unsigned char *iv,
				int len, unsigned char *ct, unsigned char *dt)
{
	int outlen, tmplen;
	if (!EVP_DecryptInit_ex(ctx, EVP_aes_128_xts(), NULL, key, iv))
		printf("\n ERROR!! \n");
	if (!EVP_DecryptUpdate(ctx, dt, &outlen, (const unsigned char *)ct, len))
		printf("\n ERROR!! \n");
	if (!EVP_DecryptFinal_ex(ctx, dt + outlen, &tmplen))
		printf("\n ERROR!! \n");

	return 0;
}

int main(void)
{

	unsigned char key1[16], key2[16], tinit[16];
	unsigned char *pt, *ct, *dt, *refct, *refdt;
	unsigned char keyssl[32];	/* SSL takes both keys together */
	unsigned int rand_len, t;
	int i, j, k;

	/* Initialise our cipher context, which can use same input vectors */
	EVP_CIPHER_CTX ctx;
	EVP_CIPHER_CTX_init(&ctx);

	/* Allocate space for input and output buffers */
	pt = malloc(TEST_LEN);
	ct = malloc(TEST_LEN);
	dt = malloc(TEST_LEN);
	refct = malloc(TEST_LEN);
	refdt = malloc(TEST_LEN);

	if (NULL == pt || NULL == ct || NULL == dt || NULL == refct || NULL == refdt) {
		printf("malloc of testsize failed\n");
		return -1;
	}

	/**************************** FIXED LENGTH TEST *************************/
	printf("aes_xts_128_rand_ossl test, %d sets of length %d: ", TEST_LOOPS, TEST_LEN);

	// Loop over the vectors
	for (i = 0; i < TEST_LOOPS; i++) {

		mk_rand_data(key1, key2, tinit, pt, TEST_LEN);

		/* Set up key for the SSL engine */
		for (k = 0; k < 16; k++) {
			keyssl[k] = key1[k];
			keyssl[k + 16] = key2[k];
		}

		/* Encrypt using each method */
		XTS_AES_128_enc(key2, key1, tinit, TEST_LEN, pt, ct);
		openssl_aes_128_xts_enc(&ctx, keyssl, tinit, TEST_LEN, pt, refct);

		/* Carry out comparison of the calculated ciphertext with 
		 * the reference
		 */
		for (j = 0; j < TEST_LEN; j++) {

			if (ct[j] != refct[j]) {
				printf("XTS_AES_128_enc failed at byte %d! \n", j);
				return -1;
			}
		}

		/* Decrypt using each method */
		XTS_AES_128_dec(key2, key1, tinit, TEST_LEN, ct, dt);
		openssl_aes_128_xts_dec(&ctx, keyssl, tinit, TEST_LEN, refct, refdt);

		for (j = 0; j < TEST_LEN; j++) {

			if (dt[j] != refdt[j]) {
				printf("XTS_AES_128_dec failed at byte %d! \n", j);
				return -1;
			}
		}
		printf(".");
		fflush(0);
	}
	printf("Pass\n");

	/**************************** RANDOM LENGTH TEST *************************/
	printf("aes_xts_128_rand_ossl test, %d sets of random lengths: ", RANDOMS);

	/* Run tests with random size */

	for (t = 0; t < RANDOMS; t++) {

		rand_len = rand() % (TEST_LEN);
		mk_rand_data(key1, key2, tinit, pt, rand_len);

		/* Set up key for the SSL engine */
		for (k = 0; k < 16; k++) {
			keyssl[k] = key1[k];
			keyssl[k + 16] = key2[k];
		}

		/* Encrypt using each method */
		XTS_AES_128_enc(key2, key1, tinit, rand_len, pt, ct);
		openssl_aes_128_xts_enc(&ctx, keyssl, tinit, rand_len, pt, refct);

		/* Carry out comparison of the calculated ciphertext with 
		 * the reference
		 */
		for (j = 0; j < rand_len; j++) {

			if (ct[j] != refct[j]) {
				printf("XTS_AES_128_enc failed at byte %d! \n", j);
				return -1;
			}
		}

		/* Decrypt using each method */
		XTS_AES_128_dec(key2, key1, tinit, rand_len, ct, dt);
		openssl_aes_128_xts_dec(&ctx, keyssl, tinit, rand_len, refct, refdt);

		for (j = 0; j < rand_len; j++) {

			if (dt[j] != refdt[j]) {
				printf("XTS_AES_128_dec failed at byte %d! \n", j);
				return -1;
			}
		}
		printf(".");
		fflush(0);
	}
	printf("Pass\n");

	printf("aes_xts_128_rand_ossl: All tests passed\n");

	return 0;
}
