#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/delay.h>

#include "kmartian.h"
#include "../martian.h"
#include "marsio.h"

u8 mars_read_register (struct martian *mdev, u8 reg)
{
	outb (  reg,

		(reg < 0x80) || ((reg >= 0xa0) && (reg < 0xd0)) ? 
		mdev->params.BaseAddress : 
		mdev->params.BaseAddress2
	);

	return inb(mdev->params.BaseAddressData);
}


void mars_write_register(struct martian *mdev, u8 reg, u8 val) 
{
	outb(reg, mdev->params.BaseAddress);
	return outb(val, mdev->params.BaseAddressData);
}


void mars_write_byte(struct martian *mdev, u16 addr, u8 val)
{
	outb(val, mdev->params.BaseAddress + addr);
}

void mars_write_word(struct martian *mdev, u16 addr, u16 val)
{
	outw(val, mdev->params.BaseAddress + addr);
}

void mars_write_dword(struct martian *mdev, u16 addr, u32 val)
{
	outl(val, mdev->params.BaseAddress + addr);
}

#define CHECK_FIFO_CORRESPOND_HEAD(fifo) \
({	\
	int incor = 0;							\
	int size = sizeof mcb->io_##fifo##_buff 			\
		/ sizeof mcb->io_##fifo##_buff[0];			\
	int uhead = (mcb->io_##fifo##_wptr - mcb->io_##fifo##_buff_um)	\
		/ sizeof mcb->io_##fifo##_buff[0];			\
	int utail = (mcb->io_##fifo##_rptr - mcb->io_##fifo##_buff_um)	\
		/ sizeof mcb->io_##fifo##_buff[0];			\
	if (uhead != mdev->fifo.head) { 				\
		mdev->fifo.head = uhead;				\
		incor = 1;						\
	}								\
	(!incor) && (utail >= 0) && (utail < size);	\
})

#define KFIFO_UPDATE_TAIL(fifo)						\
mdev->fifo.tail = (mcb->io_##fifo##_rptr - mcb->io_##fifo##_buff_um) 	\
			/ sizeof mcb->io_##fifo##_buff[0];

#define UFIFO_UPDATE_HEAD(fifo)						\
mcb->io_##fifo##_wptr = mcb->io_##fifo##_buff_um + 			\
	(sizeof mcb->io_##fifo##_buff[0]) * mdev->fifo.head;

static inline unsigned char mdual_port_rx (struct martian *mdev) {
	struct martian_common *mcb = mdev->common;

	register __u16 *startptr;
	__u8 reg, rb0;
	int num;

	mcb->dual_port_rx++;

	rb0 = mars_read_register(mdev, 0xb0);
	num = rb0 & 0x1f;
	if (num == 0)
		return mars_read_register(mdev, 0xb0);

	ASSERT (CHECK_FIFO_CORRESPOND_HEAD (dce_rx), 
			mcb->rx_incorrespondence++;);

	KFIFO_UPDATE_TAIL (dce_rx);

	mcb->rx += num;
	startptr = (u16 *) mfifo_head (&mdev->dce_rx); 
	reg = 0x90; 
	
	while (--num != -1) {
		u16 val = mars_read_register(mdev, reg++);
		/* on BE should be transformed to LE */
		val = cpu_to_le16 (val);
		mfifo_put (&mdev->dce_rx, &val);

		if (mfifo_empty (&mdev->dce_rx))
			mcb->rx_overrun++;
	}

	/* update modem's */
	UFIFO_UPDATE_HEAD (dce_rx);

	
	if (rb0 & 0x20)
		*startptr |= cpu_to_le16 (0x4000);

	if ((rb0 & 0xc0) && ! mfifo_empty (&mdev->dce_rx)) {

		u16 *lastptr = (u16 *) mfifo_lastwr (&mdev->dce_rx);

		if (rb0 & 0x80) 
			*lastptr |= cpu_to_le16 (0x8000);

		if (rb0 & 0x40) 
			*lastptr |= cpu_to_le16 (0xa800);
	}

	// ASSERT (mfifo_ok (&mdev->dce_rx), mcb->kfifo_rx_bad++;);

	return mars_read_register(mdev, 0xb0);
}

static __u8 Last_rbs_tick; 
static inline int mio_dual_port_rx (struct martian *mdev)
{
	struct martian_common *mcb = mdev->common;
	u8 rb0 = mars_read_register(mdev, 0xb0);
	u8 reg = 0x90; 
	u8 num = 16;

	u32 io_dce_rx_rptr = mcb->io_dce_rx_rptr;
	u32 io_dce_rx_wptr = mcb->io_dce_rx_wptr;

	if (mcb->init_RBS_tick_flag) {
		rb0 &= 0x7;
		mcb->RBS_tick_rx = rb0;
		mcb->init_RBS_tick_flag = 0;
	}
	else if ((Last_rbs_tick + 4) % 6 != rb0)
		return mars_read_register(mdev, 0xb0);

	Last_rbs_tick = rb0;

	ASSERT (CHECK_FIFO_CORRESPOND_HEAD (pdm_rx), 
			mcb->pdm_rx_incorrespondence++;);

	KFIFO_UPDATE_TAIL (pdm_rx);

	do {
		u8 val = mars_read_register(mdev, reg++);
		mfifo_put (&mdev->pdm_rx, &val);

		/* if (mfifo_full (&mcb->pdm_rx)) */
		if (mfifo_empty (&mdev->pdm_rx))
			mcb->pdm_rx_overrun++;
	} while (--num);

	UFIFO_UPDATE_HEAD (pdm_rx);

	if (test_bit (KMSETTING_DEBUG, &mdev->state)) {
		if (io_dce_rx_rptr != mcb->io_dce_rx_rptr) {
			mcb->pdm_dirty++;
			mcb->latest_rptr = mcb->io_dce_rx_rptr;
		}

		if (io_dce_rx_wptr != mcb->io_dce_rx_wptr) {
			mcb->pdm_dirty += 0x1000;
			mcb->latest_wptr = mcb->io_dce_rx_wptr;
		}

		if ((io_dce_rx_wptr != mcb->io_dce_rx_wptr) || (io_dce_rx_wptr != mcb->io_dce_rx_wptr)) 
			set_bit (KMSTATE_FIFO_FAULT, &mdev->state);
	}

	mcb->io_dual_port_rx++;
	mcb->pdm_rx += (16 - num);

	return mars_read_register(mdev, 0xb0);
}

#define CRL_BYTE(w)	((w) >> 8)
#define CHECK_FIFO_CORRESPOND_TAIL(fifo) \
({	\
	int incor = 0;							\
	int size = sizeof mcb->io_##fifo##_buff				\
		/ sizeof mcb->io_##fifo##_buff[0];			\
	int uhead = (mcb->io_##fifo##_wptr - mcb->io_##fifo##_buff_um)	\
		/ sizeof mcb->io_##fifo##_buff[0];			\
	int utail = (mcb->io_##fifo##_rptr - mcb->io_##fifo##_buff_um)	\
		/ sizeof mcb->io_##fifo##_buff[0];			\
	if (utail != mdev->fifo.tail) { 				\
		mdev->fifo.tail = utail;				\
		incor = 1;						\
	}								\
	(!incor) && (uhead >= 0) && (uhead < size);	\
})

#define KFIFO_UPDATE_HEAD(fifo)						\
mdev->fifo.head = (mcb->io_##fifo##_wptr - mcb->io_##fifo##_buff_um) 	\
			/ sizeof mcb->io_##fifo##_buff[0];

#define UFIFO_UPDATE_TAIL(fifo)						\
mcb->io_##fifo##_rptr = mcb->io_##fifo##_buff_um + 			\
	(sizeof mcb->io_##fifo##_buff[0]) * mdev->fifo.tail;

#define	CF_CRL_BYTE(w)	(le16_to_cpu (w) >> 8)
#define	CF_DATA_BYTE(w)	(le16_to_cpu (w) & 0xff)

static inline void mdual_port_tx (struct martian *mdev)
{
	struct martian_common *mcb = mdev->common;

	unsigned char reg;
	unsigned char crl, num;

	mcb->dual_port_tx++;

	mars_read_register(mdev, 0x32);

	reg = 0x80;
	
	crl = CF_CRL_BYTE (*(u16 *) mfifo_tail (&mdev->dce_tx));
	crl &= (crl & 0x20) ? 0x20 : 0x40;


	ASSERT (CHECK_FIFO_CORRESPOND_TAIL (dce_tx), 
			mcb->tx_incorrespondence++;);

	KFIFO_UPDATE_HEAD (dce_tx);

	while (reg <= 0x85 && ! mfifo_empty (&mdev->dce_tx)) {
		u16 val;
		mfifo_get (&mdev->dce_tx, &val);
		mars_write_register(mdev, reg++, CF_DATA_BYTE(val));

		if (CF_CRL_BYTE (val) & 0x80) {
			crl |= 0x80;
			break;
		}

		if (0x40 & CF_CRL_BYTE (* (u16 *) mfifo_tail (&mdev->dce_tx))) 
			break;
	}

	UFIFO_UPDATE_TAIL (dce_tx);

	num = reg - 0x80; 
	if (num != 0) {
		crl |= num;
		mcb->tx += num;

		mcb->dp_dsp_data_in_progress = 1;
		mars_write_register(mdev, 0xb0, crl);
	}
	else {
		mcb->dp_dsp_data_in_progress = 0;
		mars_write_register(mdev, 0xb7, 1);
	}

	// ASSERT (mfifo_ok(&mdev->dce_tx), mcb->kfifo_tx_bad++;);
}

int process_stream (struct martian *mdev) {
	struct martian_common *mcb = mdev->common;
	u32 io_dce_rx_rptr = mcb->io_dce_rx_rptr;
	u32 io_dce_rx_wptr = mcb->io_dce_rx_wptr;

	if (mars_read_register(mdev, 0xb6) & 0x10) {
		if (mcb->S[0x6b] == 1 && ! mcb->V34Mode && mcb->S[0x96] == 0)
			/* return 0; */
			mio_dual_port_rx (mdev);
		else {
			mdual_port_rx (mdev);
			io_dce_rx_wptr = mcb->io_dce_rx_wptr;
		}
	}

	if (mars_read_register(mdev, 0xb6) & 0x1) 
		mdual_port_tx (mdev);

	if (test_bit (KMSETTING_DEBUG, &mdev->state)) {
		if (io_dce_rx_rptr != mcb->io_dce_rx_rptr) {
			mcb->dirty++;
			mcb->latest_rptr = mcb->io_dce_rx_rptr;
		}

		if (io_dce_rx_wptr != mcb->io_dce_rx_wptr) {
			mcb->dirty += 0x1000;
			mcb->latest_wptr = mcb->io_dce_rx_wptr;
		}

		if ((io_dce_rx_wptr != mcb->io_dce_rx_wptr) || (io_dce_rx_wptr != mcb->io_dce_rx_wptr)) 
			set_bit (KMSTATE_FIFO_FAULT, &mdev->state);
	}

	if (mars_read_register(mdev, 0xb6) & 0x20) {
		u8 reg = 0xb3; 
		u8 buf[3];
		struct mcirc *circ = &mcb->dcp_circ;
		int i, c, written;

		for (i = 0; reg >= 0xb1; reg--)
			buf[i++] = mars_read_register(mdev, reg);
		
		c = CIRC_SPACE_TO_END(circ->head, circ->tail, sizeof mcb->dcp_buf);
		if (c > 3)
			c = 3;

		for (i = 0; i < c; i++)
			mcb->dcp_buf[circ->head + i] = buf[i];

		written = c;
		circ->head = (circ->head + c) & (sizeof mcb->dcp_buf - 1);
		if (3 - written) {
			c = CIRC_SPACE(circ->head, circ->tail, sizeof mcb->dcp_buf);

			if (c > 3 - written) 
				c = 3 - written;
				
			for (i = 0; i < c; i++)
				mcb->dcp_buf[i] = buf[written + i];

			circ->head = c;
		}

		mars_read_register(mdev, 0xb1);
		return 0;
	}

	return 1;
}

void process_ring(struct martian *mdev) 
{
	struct martian_common *mcb = mdev->common;
	if (mcb->dp_sleep == 1) {
		mars_write_register(mdev, 0xcb, 0x1d);
		mcb->dp_bamil_rd7 = 0xff;
	}
	mcb->dp_ring_int_count++;

	if (mdev->params.dsp_mars3 == 4) 
		mars_write_register(mdev, 0xad, 0x21);

	if (mcb->dp_ring_int_count <= ((mdev->params.dsp_mars3 == 4) ? 1 : 2))
	{
		mars_write_register(mdev, 0xd8, 0xff);
		if (mdev->params.dsp_mars3 == 4)
			mars_write_register(mdev, 0xad, 1);
	}
	else {
		mcb->dp_bamil_rd7 |= 2;
		//mcb->dp_ring_int_count = 0;
	}
}

static u8 mars_read_register_rem(struct martian *mdev, u8 reg)
{
	mdev->common->BaseValue	       = reg;
	mdev->common->BaseAddressIndex = 
		(reg < 0x80) || ((reg >= 0xa0) && (reg < 0xd0)) ? 
		mdev->params.BaseAddress : 
		mdev->params.BaseAddress2;

	outb(reg, mdev->common->BaseAddressIndex);
	return inb(mdev->params.BaseAddressData);
}


static void mars_write_register_rem(struct martian *mdev, u8 reg, u8 val) 
{
	mdev->common->BaseValue		= reg;
	mdev->common->BaseAddressIndex	= mdev->params.BaseAddress; 
	outb(reg, mdev->params.BaseAddress);
	outb(val, mdev->params.BaseAddressData);
}

u8 mars_read_register_safe (struct martian *mdev, u8 reg) 
{
	u8 val;
	unsigned long flags;

	mars_lock(mdev, flags);
	val = mars_read_register/*_rem*/(mdev, reg);
	mars_unlock(mdev, flags);

	return val;
}


void mars_write_register_safe (struct martian *mdev, u8 reg, u8 val) 
{
	unsigned long flags;

	mars_lock(mdev, flags);
	mars_write_register/*_rem*/(mdev, reg, val);
	mars_unlock(mdev, flags);
}

void mars_reg_andor (struct martian *mdev, u8 reg, u8 val_and, u8 val_or) 
{
	u8 val;
	unsigned long flags;

	mars_lock (mdev, flags);
	val = mars_read_register (mdev, reg);
	mars_write_register (mdev, reg, val_or | (val_and & val));
	mars_unlock (mdev, flags);
}

#define DSP_TIMEOUT	19	/* ms */
#define UDELAY		1

int wait_for_core_read (struct martian *mdev)
{
	unsigned dls = (DSP_TIMEOUT * 1000) / UDELAY;

	do {
		if (mars_read_register_safe (mdev, 0xd8) & 0x10) 
			return 0;

		udelay (UDELAY);
	} while (--dls);

	return -1;
}

u8 mars_read_dsp_reg (struct martian *mdev, u8 reg)
{
	unsigned long flags;
	unsigned dls = (DSP_TIMEOUT * 1000) / UDELAY;
	u8 byte_e;

	mdev->common->dp_byte_f = 0;

	mars_lock (mdev, flags);
	mars_write_register(mdev, 0xd8, 0x18);
	mars_write_register(mdev, 0x36, reg);
	mars_write_register(mdev, 0x37, 0x6);
	mars_unlock (mdev, flags);

	do {
		mdev->common->dp_byte_f = mars_read_register_safe (mdev, 0x35);
		if (mdev->common->dp_byte_f == 2) 
			break;

		udelay (UDELAY);
	} while (--dls);

	byte_e = mars_read_register_safe (mdev, 0x34);

	if (mdev->common->dp_byte_f != 2) {
		mars_write_register_safe (mdev, 0xd8, 0xff);
		byte_e = 0;
	}

	return byte_e;
}

u8 mars_readpoll_dsp_reg (struct martian *mdev, u8 type, u8 reg, u8 exp)
{
	unsigned dls;
	u8 byte_f = 0;
	//dp_cmd_timer = x_current_time();

	mars_write_register_safe (mdev, 0x18, 0xd8);
	mars_write_register_safe (mdev, 0x36, reg);
	mars_write_register_safe (mdev, 0x37, type);

	dls = (199 * 1000) / UDELAY;
	do {
		byte_f = mars_read_register_safe (mdev, 0x35);
		if (byte_f == exp) 
			break;

		udelay (UDELAY);
	} while (--dls);

	mdev->common->dp_byte_f = byte_f;
	return byte_f;
}



void mars_write_dsp_ram (struct martian *mdev, u16 addr, u16 val)
{
	unsigned long flags;

	mars_lock (mdev, flags);
	mars_write_register(mdev, 0xd8, 0x18);
	mars_write_register(mdev, 0x32, val & 0xff);
	mars_write_register(mdev, 0x33, (val >> 8) & 0xff);
	mars_write_register(mdev, 0x34, addr & 0xff);
	mars_write_register(mdev, 0x35, (addr >> 8) & 0xff);
	mars_write_register(mdev, 0x37, 1);
	mars_unlock (mdev, flags);

	wait_for_core_read (mdev);
}

/* side-effect: mdev->byte_f */
u16 mars_read_dsp_ram (struct martian *mdev, u16 addr) 
{
	struct martian_common *mcb = mdev->common;
	unsigned long flags;
	unsigned dls = (DSP_TIMEOUT * 1000) / UDELAY;
	u8 byte_d, byte_e;

	mars_lock (mdev, flags);
	mars_write_register(mdev, 0xd8, 0x18);
	mars_write_register(mdev, 0x34, addr & 0xff);
	mars_write_register(mdev, 0x35, (addr >> 8) & 0xff);
	/* cmd */
	mars_write_register(mdev, 0x37, 4);
	mars_unlock (mdev, flags);

	/* inline wait but 0x8 */
	do {
		if (mars_read_register_safe (mdev, 0xd8) & 0x8) 
			break;

		udelay (UDELAY);
	} while (--dls);

	mars_write_register_safe (mdev, 0xd8, 0x8);

	byte_d = mars_read_register_safe (mdev, 0x33);
	byte_e = mars_read_register_safe (mdev, 0x34);

	mcb->dp_byte_f = mars_read_register_safe (mdev, 0x35);

	if (mcb->dp_byte_f == 1) 
		return byte_d + (byte_e << 8);

	else {
		mars_write_register_safe (mdev, 0xd8, 0xff);
		if (mcb->x_dsp_mars == 1) 
			mars_write_word(mdev, 0x3c, 0x1ff);

		return 0;
	}
}


/* side-effect: mdev->byte_f */
void mars_command (struct martian *mdev, 
			u8 cmd, u8 arg1, u8 arg2) 
{
	struct martian_common *mcb = mdev->common;
	unsigned long flags;

	if (cmd == 0xe || cmd == 0xc) {
		if ((mars_read_dsp_reg (mdev, 0xb5) & 1) == 0) 
			mars_read_register_safe (mdev, 0xb0);

		if ((mars_read_dsp_reg (mdev, 0xb5) & 0x10) == 0) {
			mars_write_register_safe (mdev, 0xb6, mdev->params.DCPAudioOn ? 0xcf : 0xef);
			mars_read_dsp_reg (mdev, 0xb0);
			mars_write_register_safe (mdev, 0xb0, 0);
		}

		mars_write_register_safe (mdev, 0xb7, 0xff);

		if (mdev->params.DCPAudioOn) {
			mars_write_register_safe (mdev, 0xb6, 0xce);
			mars_read_register_safe (mdev, 0xb1);
		}
		else
			mars_write_register_safe (mdev, 0xb6, 0xee);

		mars_write_register_safe (mdev, 0xd7, 0xfb);
		mcb->dp_bamil_rd7 = 0xfb;
		mcb->dp_dsp_data_in_progress = 0;
	}

	mars_lock (mdev, flags);
	mars_write_register(mdev, 0xd8, 0x18);
	mars_write_register(mdev, 0x35, arg2);
	mars_write_register(mdev, 0x36, arg1);
	mars_write_register(mdev, 0x37, cmd);
	mars_unlock (mdev, flags);

	wait_for_core_read (mdev);
}

int mars_command_long (struct martian *mdev, 
			u8 cmd, u8 arg1, u8 arg2, u8 arg3, u8 arg4) 
{
	unsigned long flags;

	mars_lock (mdev, flags);
	mars_write_register(mdev, 0xd8, 0x18);
	mars_write_register(mdev, 0x33, arg4);
	mars_write_register(mdev, 0x34, arg3);
	mars_write_register(mdev, 0x35, arg2);
	mars_write_register(mdev, 0x36, arg1);
	mars_write_register(mdev, 0x37, cmd);
	mars_unlock (mdev, flags);

	return wait_for_core_read (mdev);
}

void mars_clear_dsp_ram (struct martian *mdev)
{
	u16 addr = 0x1000;

	do 
		mars_write_dsp_ram (mdev, addr, 0x0);

	while (++addr < 0x4000);
}

static inline void mars_dld_dsp_begin (struct martian *mdev, struct martian_common *mcb)
{
	u8 bamil_rd7 = mdev->params.DCPAudioOn ? 0xfb : 0xff;


	mars_write_register_safe (mdev, 0xd8, 0xff);
	mars_write_register_safe (mdev, IR, bamil_rd7);
	mcb->dp_bamil_rd7 = bamil_rd7;
	mars_command_long (mdev, 0x29, 0, 0, 0, 0);
	mars_write_dsp_ram (mdev, 0x39, 0);
}

static inline void mars_dld_dsp_end (struct martian *mdev, struct martian_common *mcb)
{
	{
		unsigned dls = (9 * 1000) / UDELAY;
		do {
			if (mars_read_register_safe (mdev, 0xd8) & 0x8) 
				break;

			udelay (UDELAY);
		} while (--dls);
	}

	mars_write_dsp_ram (mdev, 0x3d, 0x186);

	if (mdev->params.DCPAudioOn) {
		mars_write_register_safe (mdev, 0xb7, 0xdf);
		mars_write_register_safe (mdev, 0xd8, mcb->dp_bamil_rd7);
	}
	else {
		mars_write_register_safe (mdev, 0xb7, 0xff);
		mars_write_register_safe (mdev, 0xd8, 0xff);
	}
}

static inline void mars_dld_dsp_chunk_header (struct martian *mdev, u16 *dptr)
{
	unsigned long flags;
	int idx;

	mars_lock (mdev, flags);
	mars_write_register(mdev, 0x30, 1);

	for (idx = 0; idx <= 5; idx += 2) {
		u16 val = *dptr;
		mars_write_register(mdev, 0x31 + idx, val & 0xff);
		mars_write_register(mdev, 0x32 + idx, (val >> 8) & 0xff);
		dptr++;
	}

	mars_write_register(mdev, 0xd8, 0x18);
	mars_write_register(mdev, 0x37, 0x1a);
	mars_unlock (mdev, flags);

	wait_for_core_read (mdev);
}

static inline void mars_dld_dsp_chunk_transfer (struct martian *mdev, u16 *dptr, int bytes, int chunk)
{
	unsigned long flags;
	int idx = 0;

	mars_lock (mdev, flags);
	while (idx < bytes) {
		mars_write_register(mdev, 0x30 + idx, *dptr & 0xff);
		mars_write_register(mdev, 0x31 + idx, (*dptr >> 8) & 0xff);
		idx += 2; dptr++;
	}

	mars_write_register(mdev, 0xd8, 0x18);
	mars_write_register(mdev, 0x37, bytes | (chunk ? 0xc0 : 0xc8));
	mars_unlock (mdev, flags);

	wait_for_core_read (mdev);
}

/* side-effect: mdev->byte_f */
int mars_download_dsp (struct martian *mdev, u16 *dptr, u16 num) 
{
	struct martian_common *mcb = mdev->common;
	u16 rd7_bamil_save = mcb->dp_bamil_rd7;
	int res = 0;

	mars_dld_dsp_begin (mdev, mcb);

	while (num != 0) {
		int chunk = dptr[1];
		if (chunk + 3 > num) {
			res = -EFAULT;
			break;
		}

		num -= (chunk + 3);
		
		mars_dld_dsp_chunk_header (mdev, dptr);
		dptr += 3;

		while (chunk > 2) {
			chunk -= 3;
			mars_dld_dsp_chunk_transfer (mdev, dptr, 6, chunk);
			dptr += 3;
		}
		mars_dld_dsp_chunk_transfer (mdev, dptr, chunk << 1 , chunk);
	}

	mars_dld_dsp_end (mdev, mcb);
	
	mcb->dp_bamil_rd7 = rd7_bamil_save;
	mars_write_register_safe (mdev, 0xd7, mcb->dp_bamil_rd7);

	return res;
}

/* side-effect: mdev->byte_f */
int mars_download_dsp_user (struct martian *mdev, u16 __user *dptr, u16 num) 
{
	struct martian_common *mcb = mdev->common;
	u16 rd7_bamil_save = mcb->dp_bamil_rd7;
	int res = 0;

	mars_dld_dsp_begin (mdev, mcb);

	while (num != 0) {
		int chunk;
		u16 kdata[3];

		if (copy_from_user (kdata, dptr, sizeof kdata)) {
			res = -EFAULT;
			break;
		}

		chunk = kdata[1];
		if (chunk + 3 > num) {
			res = -EFAULT;
			break;
		}

		num -= (chunk + 3);
		
		mars_dld_dsp_chunk_header (mdev, kdata);
		dptr += 3;

		while (chunk > 2) {
			if (copy_from_user (kdata, dptr, 6)) {
				res = -EFAULT;
				goto finalize;
			}
			chunk -= 3;
			dptr += 3;
			mars_dld_dsp_chunk_transfer (mdev, kdata, 6, chunk);
		}

		if (copy_from_user (kdata, dptr, (chunk << 1))) {
			res = -EFAULT;
			goto finalize;
		}

		mars_dld_dsp_chunk_transfer (mdev, kdata, chunk << 1 , chunk);
	} /* while (num) */

finalize:
	mars_dld_dsp_end (mdev, mcb);
	
	mcb->dp_bamil_rd7 = rd7_bamil_save;
	mars_write_register_safe (mdev, 0xd7, mcb->dp_bamil_rd7);

	return res;
}

u16 mars_dsp_rom_checksum (struct martian *mdev)
{
	u8 byte_d, byte_e;
	u8 byte_f = 0;
	unsigned dls;

	mars_command (mdev, 0x10, 0, 0);

	dls = (99 * 1000) / UDELAY;
	do {
		byte_f = mars_read_register_safe (mdev, 0x35);
		if (byte_f == 5) 
			break;

		udelay (UDELAY);
	} while (--dls);

	byte_d = mars_read_register_safe (mdev, 0x33);
	byte_e = mars_read_register_safe (mdev, 0x34);

	if (byte_f == 5)
		mdev->params.dp_version = byte_d + (byte_e << 8);
	else {
		mdev->params.dp_version = 0;
		MERROR("dp version failure\n");
	}

	mdev->common->dp_version = mdev->params.dp_version;

	byte_f = 0;
	mars_command (mdev, 0x14, 0, 0);

	dls = (199 * 1000) / UDELAY;
	do {
		byte_f = mars_read_register_safe (mdev, 0x35);
		if (byte_f == 3) 
			break;

		udelay (UDELAY);
	} while (--dls);

	mdev->common->dp_byte_f = byte_f;
	if (byte_f != 3) {
		MERROR("dp version failure\n");
		return 0;
	}

	return mars_read_dsp_ram (mdev, 0x49);
}

