/*
    $Id: opcodes.c 3191 2025-04-08 16:42:51Z soci $

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
#include "opcodes.h"

static const uint16_t opcode_w65816[] = {
    0, 1, 258, 515, 516, 517, 516, 774, 515, 519, 520, 521, 522, 1035, 1292,
    525, 526, 1536, 1537, 1539, 1807, 2051, 1540, 16, 1041, 16, 2322, 2323,
    1041, 1541, 2580, 1542, 1543, 21, 2827, 2820, 2821, 2820, 2827, 2823, 2824,
    2825, 2826, 2828, 2830, 1544, 1545, 2582, 1546, 1548, 3075, 3332, 3588,
    3853, 4119, 4352, 4632, 4889, 282, 5147, 5148, 1549, 29, 4352, 5406, 5637,
    1311, 1550, 1551, 1552, 1554, 1555, 1557, 1563, 1564, 1565, 1566, 1567,
    1568, 1569, 5922, 6179, 1807, 292, 293, 1574, 1575, 1576, 32, 1577, 1578,
    1579, 1825, 258, 282, 6444, 1544, 4352, 6701, 6927, 2051, 1582, 1583, 1584,
    1585, 1586, 1583, 1584, 1587, 1587, 7185, 1588, 7195, 1588, 1589, 1590,
    1591, 1592, 1593, 1594, 1595, 1830, 1582, 1596
};

static const uint32_t mnemonic_w65816[] = {
    0x616463, 0x616e64, 0x61736c, 0x626363, 0x626373, 0x626571, 0x626765, 0x626974,
    0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x627261, 0x62726b, 0x62726c, 0x627663,
    0x627673, 0x636c63, 0x636c64, 0x636c69, 0x636c70, 0x636c72, 0x636c76, 0x636d70,
    0x636f70, 0x637061, 0x637078, 0x637079, 0x637370, 0x646561, 0x646563, 0x646578,
    0x646579, 0x656f72, 0x676363, 0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69,
    0x676e65, 0x67706c, 0x677261, 0x677663, 0x677673, 0x686c74, 0x696e61, 0x696e63,
    0x696e78, 0x696e79, 0x6a6d6c, 0x6a6d70, 0x6a736c, 0x6a7372, 0x6c6461, 0x6c6472,
    0x6c6478, 0x6c6479, 0x6c7372, 0x6d766e, 0x6d7670, 0x6e6f70, 0x6f7261, 0x6f7272,
    0x706561, 0x706569, 0x706572, 0x706861, 0x706862, 0x706864, 0x70686b, 0x706870,
    0x706878, 0x706879, 0x706c61, 0x706c62, 0x706c64, 0x706c70, 0x706c78, 0x706c79,
    0x707368, 0x70756c, 0x726570, 0x726f6c, 0x726f72, 0x727469, 0x72746c, 0x727473,
    0x736263, 0x736563, 0x736564, 0x736569, 0x736570, 0x73686c, 0x736872, 0x737461,
    0x737470, 0x737472, 0x737478, 0x737479, 0x73747a, 0x737761, 0x746164, 0x746173,
    0x746178, 0x746179, 0x746364, 0x746373, 0x746461, 0x746463, 0x747262, 0x747361,
    0x747362, 0x747363, 0x747378, 0x747861, 0x747873, 0x747879, 0x747961, 0x747978,
    0x776169, 0x77646d, 0x786261, 0x786365
};

static const uint16_t disasm_w65816[] = {
    525, 2366, 536, 2622, 1392, 1342, 1282, 4414, 327, 574, 2, 325, 1136, 1086,
    1026, 830, 4875, 3646, 3902, 2878, 1390, 2110, 2050, 4158, 273, 3134, 47,
    363, 1134, 1854, 1794, 1598, 7477, 2305, 820, 2561, 1287, 1281, 1363, 4353,
    333, 513, 83, 332, 1031, 1025, 1107, 769, 4873, 3585, 3841, 2817, 2055,
    2049, 2131, 4097, 345, 3073, 30, 369, 1799, 1793, 1875, 1537, 341, 2337,
    633, 2593, 7740, 1313, 1338, 4385, 323, 545, 58, 326, 7475, 1057, 1082, 801,
    4879, 3617, 3873, 2849, 7739, 2081, 2106, 4129, 275, 3105, 329, 362, 818,
    1825, 1850, 1569, 343, 2304, 4674, 2560, 1380, 1280, 1364, 4352, 330, 512,
    84, 342, 6707, 1024, 1108, 768, 4880, 3584, 3840, 2816, 2148, 2048, 2132,
    4096, 347, 3072, 335, 365, 6195, 1792, 1876, 1536, 4876, 2399, 4622, 2655,
    1379, 1375, 1378, 4447, 288, 519, 371, 324, 1123, 1119, 1122, 863, 4867,
    3679, 3935, 2911, 2147, 2143, 3426, 4191, 374, 3167, 372, 373, 1124, 1887,
    1892, 1631, 569, 2358, 568, 2614, 1337, 1334, 1336, 4406, 361, 566, 360,
    331, 1081, 1078, 1080, 822, 4868, 3638, 3894, 2870, 2105, 2102, 3384, 4150,
    278, 3126, 370, 375, 1849, 1846, 3128, 1590, 539, 2327, 594, 2583, 1307,
    1303, 1310, 4375, 305, 535, 287, 376, 1051, 1047, 1054, 791, 4874, 3607,
    3863, 2839, 1345, 2071, 2078, 4119, 274, 3095, 328, 352, 6451, 1815, 1822,
    1559, 538, 2392, 604, 2648, 1306, 1368, 1327, 4440, 304, 600, 317, 378,
    1050, 1112, 1071, 856, 4869, 3672, 3928, 2904, 576, 2136, 2095, 4184, 346,
    3160, 334, 379, 6197, 1880, 1839, 1624
};

static const uint8_t alias_w65816[] = {
    0, 1, 2, 3, 4, 5, 4, 7, 3, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 82,
    100, 22, 23, 24, 23, 26, 27, 24, 30, 30, 31, 32, 33, 34, 35, 36, 35, 34, 39,
    40, 41, 42, 43, 44, 96, 47, 47, 48, 49, 51, 51, 52, 53, 54, 55, 56, 57, 58,
    59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
    78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 2, 58, 95, 96,
    97, 98, 99, 100, 122, 106, 107, 104, 105, 106, 107, 109, 109, 110, 113, 112,
    113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123
};

const struct cpu_s w65816 = {
    "65816",
    mnemonic_w65816,
    opcode_w65816,
    disasm_w65816,
    alias_w65816,
    0x384840b,
    124,
    0xffffff,
    51, 14, 55, 54, 56, 57, -1, 97, 95, 98, 99, 100, 23, 25, 26, 27, -1, 106,
    -1, -1, 0, 88, 1, 63, 33, 7, 62, 112, 110, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1
};

static const uint16_t opcode_c6502[] = {
    7424, 7425, 258, 515, 516, 517, 516, 7174, 515, 519, 520, 521, 1035, 525,
    526, 1536, 1537, 1539, 1540, 7440, 7440, 2322, 2323, 2621, 1542, 1543, 7445,
    2827, 2820, 2821, 2820, 2827, 2823, 2824, 2825, 2828, 2830, 2622, 1546,
    1548, 7684, 7949, 8255, 4352, 4672, 4929, 282, 1549, 7453, 4352, 1550, 1555,
    1564, 1567, 6210, 6211, 292, 293, 1574, 1576, 7456, 1577, 1578, 1579, 258,
    282, 8476, 4352, 6701, 6927, 1585, 1586, 1589, 1590, 1591, 1593
};

static const uint32_t mnemonic_c6502[] = {
    0x616463, 0x616e64, 0x61736c, 0x626363, 0x626373, 0x626571, 0x626765, 0x626974,
    0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x62726b, 0x627663, 0x627673, 0x636c63,
    0x636c64, 0x636c69, 0x636c76, 0x636d70, 0x637061, 0x637078, 0x637079, 0x646563,
    0x646578, 0x646579, 0x656f72, 0x676363, 0x676373, 0x676571, 0x676765, 0x676c74,
    0x676d69, 0x676e65, 0x67706c, 0x677663, 0x677673, 0x696e63, 0x696e78, 0x696e79,
    0x6a6d70, 0x6a7372, 0x6c6461, 0x6c6472, 0x6c6478, 0x6c6479, 0x6c7372, 0x6e6f70,
    0x6f7261, 0x6f7272, 0x706861, 0x706870, 0x706c61, 0x706c70, 0x707368, 0x70756c,
    0x726f6c, 0x726f72, 0x727469, 0x727473, 0x736263, 0x736563, 0x736564, 0x736569,
    0x73686c, 0x736872, 0x737461, 0x737472, 0x737478, 0x737479, 0x746178, 0x746179,
    0x747378, 0x747861, 0x747873, 0x747961
};

static const uint16_t disasm_c6502[] = {
    524, 2352, 303, 303, 303, 1328, 1282, 303, 307, 560, 2, 303, 303, 1072,
    1026, 303, 4875, 3632, 303, 303, 303, 2096, 2050, 303, 271, 3120, 303, 303,
    303, 1840, 1794, 303, 7465, 2305, 303, 303, 1287, 1281, 1336, 303, 309, 513,
    56, 303, 1031, 1025, 1080, 303, 4873, 3585, 303, 303, 303, 2049, 2104, 303,
    317, 3073, 303, 303, 303, 1793, 1848, 303, 314, 2330, 303, 303, 303, 1306,
    1326, 303, 306, 538, 46, 303, 7464, 1050, 1070, 303, 4877, 3610, 303, 303,
    303, 2074, 2094, 303, 273, 3098, 303, 303, 303, 1818, 1838, 303, 315, 2304,
    303, 303, 303, 1280, 1337, 303, 308, 512, 57, 303, 6696, 1024, 1081, 303,
    4878, 3584, 303, 303, 303, 2048, 2105, 303, 319, 3072, 303, 303, 303, 1792,
    1849, 303, 303, 2370, 303, 303, 1349, 1346, 1348, 303, 281, 303, 329, 303,
    1093, 1090, 1092, 303, 4867, 3650, 303, 303, 2117, 2114, 3396, 303, 331,
    3138, 330, 303, 303, 1858, 303, 303, 557, 2346, 556, 303, 1325, 1322, 1324,
    303, 327, 554, 326, 303, 1069, 1066, 1068, 303, 4868, 3626, 303, 303, 2093,
    2090, 3372, 303, 274, 3114, 328, 303, 1837, 1834, 3116, 303, 534, 2323, 303,
    303, 1302, 1299, 1303, 303, 295, 531, 280, 303, 1046, 1043, 1047, 303, 4874,
    3603, 303, 303, 303, 2067, 2071, 303, 272, 3091, 303, 303, 303, 1811, 1815,
    303, 533, 2364, 303, 303, 1301, 1340, 1317, 303, 294, 572, 303, 303, 1045,
    1084, 1061, 303, 4869, 3644, 303, 303, 303, 2108, 2085, 303, 318, 3132, 303,
    303, 303, 1852, 1829, 303
};

static const uint8_t alias_c6502[] = {
    0, 1, 2, 3, 4, 5, 4, 7, 3, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 19,
    21, 22, 23, 24, 25, 26, 27, 28, 29, 28, 27, 32, 33, 34, 35, 36, 37, 38, 39,
    40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58,
    59, 60, 61, 62, 63, 2, 46, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75
};

const struct cpu_s c6502 = {
    "6502",
    mnemonic_c6502,
    opcode_c6502,
    disasm_c6502,
    alias_c6502,
    0x1848001,
    76,
    0xffff,
    40, -1, 43, 42, 44, 45, -1, 67, 66, 68, 69, -1, 19, 20, 21, 22, -1, -1, -1,
    -1, 0, 60, 1, 49, 26, 7, 48, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1
};

static const uint16_t opcode_c65c02[] = {
    8704, 8705, 258, 515, 516, 517, 516, 774, 515, 519, 520, 521, 522, 1035,
    525, 526, 1536, 1537, 1539, 2051, 1540, 8720, 8720, 2322, 2323, 1541, 2580,
    1542, 1543, 8725, 2827, 2820, 2821, 2820, 2827, 2823, 2824, 2825, 2826,
    2828, 2830, 1545, 2582, 1546, 1548, 8964, 7949, 9279, 4352, 4672, 4929, 282,
    1549, 8733, 4352, 1550, 1555, 1557, 1563, 1564, 1567, 1568, 1569, 6212,
    6213, 292, 293, 1574, 1576, 8736, 1577, 1578, 1579, 258, 282, 9500, 4352,
    6701, 6927, 2051, 1585, 1586, 7185, 7195, 1589, 1590, 1591, 1593
};

static const uint32_t mnemonic_c65c02[] = {
    0x616463, 0x616e64, 0x61736c, 0x626363, 0x626373, 0x626571, 0x626765, 0x626974,
    0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x627261, 0x62726b, 0x627663, 0x627673,
    0x636c63, 0x636c64, 0x636c69, 0x636c72, 0x636c76, 0x636d70, 0x637061, 0x637078,
    0x637079, 0x646561, 0x646563, 0x646578, 0x646579, 0x656f72, 0x676363, 0x676373,
    0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65, 0x67706c, 0x677261, 0x677663,
    0x677673, 0x696e61, 0x696e63, 0x696e78, 0x696e79, 0x6a6d70, 0x6a7372, 0x6c6461,
    0x6c6472, 0x6c6478, 0x6c6479, 0x6c7372, 0x6e6f70, 0x6f7261, 0x6f7272, 0x706861,
    0x706870, 0x706878, 0x706879, 0x706c61, 0x706c70, 0x706c78, 0x706c79, 0x707368,
    0x70756c, 0x726f6c, 0x726f72, 0x727469, 0x727473, 0x736263, 0x736563, 0x736564,
    0x736569, 0x73686c, 0x736872, 0x737461, 0x737472, 0x737478, 0x737479, 0x73747a,
    0x746178, 0x746179, 0x747262, 0x747362, 0x747378, 0x747861, 0x747873, 0x747961
};

static const uint16_t disasm_c65c02[] = {
    525, 2357, 308, 308, 1363, 1333, 1282, 308, 312, 565, 2, 308, 1107, 1077,
    1026, 308, 4875, 3637, 3893, 308, 1362, 2101, 2050, 308, 272, 3125, 42, 308,
    1106, 1845, 1794, 308, 7470, 2305, 308, 308, 1287, 1281, 1345, 308, 316,
    513, 65, 308, 1031, 1025, 1089, 308, 4873, 3585, 3841, 308, 2055, 2049,
    2113, 308, 326, 3073, 26, 308, 1799, 1793, 1857, 308, 323, 2333, 308, 308,
    308, 1309, 1331, 308, 311, 541, 51, 308, 7469, 1053, 1075, 308, 4878, 3613,
    3869, 308, 308, 2077, 2099, 308, 274, 3101, 314, 308, 308, 1821, 1843, 308,
    324, 2304, 308, 308, 1359, 1280, 1346, 308, 315, 512, 66, 308, 6701, 1024,
    1090, 308, 4879, 3584, 3840, 308, 2127, 2048, 2114, 308, 328, 3072, 318,
    308, 6189, 1792, 1858, 308, 4876, 2379, 308, 308, 1358, 1355, 1357, 308,
    284, 519, 341, 308, 1102, 1099, 1101, 308, 4867, 3659, 3915, 308, 2126,
    2123, 3405, 308, 343, 3147, 342, 308, 1103, 1867, 1871, 308, 562, 2351, 561,
    308, 1330, 1327, 1329, 308, 337, 559, 336, 308, 1074, 1071, 1073, 308, 4868,
    3631, 3887, 308, 2098, 2095, 3377, 308, 276, 3119, 340, 308, 1842, 1839,
    3121, 308, 536, 2325, 308, 308, 1304, 1301, 1306, 308, 300, 533, 283, 308,
    1048, 1045, 1050, 308, 4874, 3605, 3861, 308, 308, 2069, 2074, 308, 273,
    3093, 313, 308, 308, 1813, 1818, 308, 535, 2373, 308, 308, 1303, 1349, 1322,
    308, 299, 581, 308, 308, 1047, 1093, 1066, 308, 4869, 3653, 3909, 308, 308,
    2117, 2090, 308, 327, 3141, 317, 308, 308, 1861, 1834, 308
};

static const uint8_t alias_c65c02[] = {
    0, 1, 2, 3, 4, 5, 4, 7, 3, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 79, 20,
    21, 21, 23, 24, 26, 26, 27, 28, 29, 30, 31, 32, 31, 30, 35, 36, 37, 38, 39,
    40, 42, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58,
    59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 2, 51, 75, 76, 77,
    78, 79, 80, 81, 82, 83, 84, 85, 86, 87
};

const struct cpu_s c65c02 = {
    "65c02",
    mnemonic_c65c02,
    opcode_c65c02,
    disasm_c65c02,
    alias_c65c02,
    0x3848001,
    88,
    0xffff,
    45, -1, 48, 47, 49, 50, -1, 76, 75, 77, 78, 79, 21, 22, 23, 24, -1, -1, -1,
    -1, 0, 69, 1, 54, 29, 7, 53, 83, 82, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1
};

static const uint16_t opcode_c6502i[] = {
    7424, 9731, 1831, 1832, 7425, 1833, 1834, 258, 1831, 1835, 515, 516, 517,
    516, 7174, 515, 519, 520, 521, 1035, 525, 526, 1536, 1537, 1539, 1540, 7440,
    7440, 2322, 2323, 8479, 8479, 2621, 1542, 1543, 7445, 2827, 2820, 2821,
    2820, 2827, 2823, 2824, 2825, 2828, 2830, 2622, 8481, 1546, 1548, 8481,
    8481, 1553, 7684, 7949, 9988, 9988, 10286, 8255, 4352, 9988, 4672, 4929,
    282, 1838, 10566, 7453, 4352, 1550, 1555, 1564, 1567, 6210, 6211, 8486, 292,
    293, 8487, 1574, 1576, 10792, 7456, 1835, 1577, 1578, 1579, 9731, 258, 282,
    9989, 9990, 11012, 8489, 8490, 8476, 4352, 6701, 6927, 9989, 1585, 1586,
    1589, 1590, 1591, 1593, 1833
};

static const uint32_t mnemonic_c6502i[] = {
    0x616463, 0x616878, 0x616c72, 0x616e63, 0x616e64, 0x616e65, 0x617272, 0x61736c,
    0x617372, 0x617873, 0x626363, 0x626373, 0x626571, 0x626765, 0x626974, 0x626c74,
    0x626d69, 0x626e65, 0x62706c, 0x62726b, 0x627663, 0x627673, 0x636c63, 0x636c64,
    0x636c69, 0x636c76, 0x636d70, 0x637061, 0x637078, 0x637079, 0x64636d, 0x646370,
    0x646563, 0x646578, 0x646579, 0x656f72, 0x676363, 0x676373, 0x676571, 0x676765,
    0x676c74, 0x676d69, 0x676e65, 0x67706c, 0x677663, 0x677673, 0x696e63, 0x696e73,
    0x696e78, 0x696e79, 0x697362, 0x697363, 0x6a616d, 0x6a6d70, 0x6a7372, 0x6c6165,
    0x6c6173, 0x6c6178, 0x6c6461, 0x6c6472, 0x6c6473, 0x6c6478, 0x6c6479, 0x6c7372,
    0x6c7861, 0x6e6f70, 0x6f7261, 0x6f7272, 0x706861, 0x706870, 0x706c61, 0x706c70,
    0x707368, 0x70756c, 0x726c61, 0x726f6c, 0x726f72, 0x727261, 0x727469, 0x727473,
    0x736178, 0x736263, 0x736278, 0x736563, 0x736564, 0x736569, 0x736861, 0x73686c,
    0x736872, 0x736873, 0x736878, 0x736879, 0x736c6f, 0x737265, 0x737461, 0x737472,
    0x737478, 0x737479, 0x746173, 0x746178, 0x746179, 0x747378, 0x747861, 0x747873,
    0x747961, 0x786161
};

static const uint16_t disasm_c6502i[] = {
    531, 2370, 308, 2396, 1345, 1346, 1287, 1372, 325, 578, 7, 515, 1089, 1090,
    1031, 1116, 4882, 3650, 321, 3676, 2113, 2114, 2055, 2140, 278, 3138, 321,
    3164, 1857, 1858, 1799, 1884, 7478, 2308, 321, 2378, 1294, 1284, 1355, 1354,
    327, 516, 75, 321, 1038, 1028, 1099, 1098, 4880, 3588, 321, 3658, 321, 2052,
    2123, 2122, 339, 3076, 321, 3146, 321, 1796, 1867, 1866, 334, 2339, 321,
    2397, 321, 1315, 1343, 1373, 324, 547, 63, 520, 7477, 1059, 1087, 1117,
    4884, 3619, 321, 3677, 321, 2083, 2111, 2141, 280, 3107, 321, 3165, 321,
    1827, 1855, 1885, 335, 2304, 321, 2381, 321, 1280, 1356, 1357, 326, 512, 76,
    518, 6709, 1024, 1100, 1101, 4885, 3584, 321, 3661, 321, 2048, 2124, 2125,
    341, 3072, 321, 3149, 321, 1792, 1868, 1869, 577, 2398, 321, 2384, 1377,
    1374, 1376, 1360, 290, 321, 358, 517, 1121, 1118, 1120, 1104, 4874, 3678,
    321, 3670, 2145, 2142, 3424, 3408, 360, 3166, 359, 3161, 1883, 1886, 3162,
    3158, 574, 2362, 573, 2361, 1342, 1338, 1341, 1337, 356, 570, 355, 569,
    1086, 1082, 1085, 1081, 4875, 3642, 321, 3641, 2110, 2106, 3389, 3385, 281,
    3130, 357, 3132, 1854, 1850, 3133, 3129, 541, 2330, 321, 2335, 1309, 1306,
    1312, 1311, 305, 538, 289, 594, 1053, 1050, 1056, 1055, 4881, 3610, 321,
    3615, 321, 2074, 2080, 2079, 279, 3098, 321, 3103, 321, 1818, 1824, 1823,
    540, 2385, 321, 2354, 1308, 1361, 1326, 1330, 304, 593, 321, 321, 1052,
    1105, 1070, 1074, 4876, 3665, 321, 3634, 321, 2129, 2094, 2098, 340, 3153,
    321, 3122, 321, 1873, 1838, 1842
};

static const uint8_t alias_c6502i[] = {
    0, 86, 8, 3, 4, 5, 6, 7, 8, 82, 10, 11, 12, 11, 14, 10, 16, 17, 18, 19, 20,
    21, 22, 23, 24, 25, 26, 26, 28, 29, 31, 31, 32, 33, 34, 35, 36, 37, 38, 37,
    36, 41, 42, 43, 44, 45, 46, 50, 48, 49, 50, 50, 52, 53, 54, 60, 60, 57, 58,
    59, 60, 61, 62, 63, 57, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
    78, 79, 80, 81, 82, 83, 84, 85, 86, 7, 63, 89, 90, 91, 92, 93, 94, 95, 96,
    97, 89, 99, 100, 101, 102, 103, 104, 5
};

const struct cpu_s c6502i = {
    "6502i",
    mnemonic_c6502i,
    opcode_c6502i,
    disasm_c6502i,
    alias_c6502i,
    0x1848001,
    106,
    0xffff,
    53, -1, 59, 58, 61, 62, -1, 95, 94, 96, 97, -1, 26, 27, 28, 29, -1, -1, -1,
    -1, 0, 81, 4, 67, 35, 14, 66, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1
};

static const uint16_t opcode_c65dtv02[] = {
    7424, 1831, 7425, 1833, 1834, 258, 1831, 515, 516, 517, 516, 7174, 515, 519,
    520, 521, 559, 1035, 525, 526, 1536, 1537, 1539, 1540, 7440, 7440, 2322,
    2323, 8479, 8479, 2621, 1542, 1543, 7445, 2827, 2820, 2821, 2820, 2827,
    2823, 2824, 2825, 2859, 2828, 2830, 2622, 8481, 1546, 1548, 8481, 8481,
    7684, 7949, 10286, 8255, 4352, 4672, 4929, 282, 1838, 1549, 7453, 4352,
    1550, 1555, 1564, 1567, 6210, 6211, 8486, 292, 293, 8487, 1574, 1576, 1840,
    10792, 7456, 1577, 1578, 1579, 258, 282, 1830, 8489, 8490, 8476, 4352, 6701,
    6927, 1585, 1586, 1589, 1590, 1591, 1593, 1833
};

static const uint32_t mnemonic_c65dtv02[] = {
    0x616463, 0x616c72, 0x616e64, 0x616e65, 0x617272, 0x61736c, 0x617372, 0x626363,
    0x626373, 0x626571, 0x626765, 0x626974, 0x626c74, 0x626d69, 0x626e65, 0x62706c,
    0x627261, 0x62726b, 0x627663, 0x627673, 0x636c63, 0x636c64, 0x636c69, 0x636c76,
    0x636d70, 0x637061, 0x637078, 0x637079, 0x64636d, 0x646370, 0x646563, 0x646578,
    0x646579, 0x656f72, 0x676363, 0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69,
    0x676e65, 0x67706c, 0x677261, 0x677663, 0x677673, 0x696e63, 0x696e73, 0x696e78,
    0x696e79, 0x697362, 0x697363, 0x6a6d70, 0x6a7372, 0x6c6178, 0x6c6461, 0x6c6472,
    0x6c6478, 0x6c6479, 0x6c7372, 0x6c7861, 0x6e6f70, 0x6f7261, 0x6f7272, 0x706861,
    0x706870, 0x706c61, 0x706c70, 0x707368, 0x70756c, 0x726c61, 0x726f6c, 0x726f72,
    0x727261, 0x727469, 0x727473, 0x736163, 0x736178, 0x736263, 0x736563, 0x736564,
    0x736569, 0x73686c, 0x736872, 0x736972, 0x736c6f, 0x737265, 0x737461, 0x737472,
    0x737478, 0x737479, 0x746178, 0x746179, 0x747378, 0x747861, 0x747873, 0x747961,
    0x786161
};

static const uint16_t disasm_c65dtv02[] = {
    529, 2365, 316, 2388, 316, 1341, 1285, 1364, 320, 573, 5, 316, 316, 1085,
    1029, 1108, 4879, 3645, 4880, 3668, 316, 2109, 2053, 2132, 276, 3133, 316,
    3156, 316, 1853, 1797, 1876, 7476, 2306, 316, 2373, 1291, 1282, 1350, 1349,
    322, 514, 70, 316, 1035, 1026, 1094, 1093, 4877, 3586, 587, 3653, 316, 2050,
    2118, 2117, 334, 3074, 316, 3141, 316, 1794, 1862, 1861, 329, 2337, 595,
    2389, 316, 1313, 1338, 1365, 319, 545, 58, 518, 7475, 1057, 1082, 1109,
    4882, 3617, 316, 3669, 316, 2081, 2106, 2133, 278, 3105, 316, 3157, 316,
    1825, 1850, 1877, 330, 2304, 316, 2376, 316, 1280, 1351, 1352, 321, 512, 71,
    516, 6707, 1024, 1095, 1096, 4883, 3584, 316, 3656, 316, 2048, 2119, 2120,
    336, 3072, 316, 3144, 316, 1792, 1863, 1864, 316, 2390, 316, 2380, 1369,
    1366, 1368, 1356, 288, 316, 349, 515, 1113, 1110, 1112, 1100, 4871, 3670,
    316, 316, 2137, 2134, 3416, 3404, 351, 3158, 350, 316, 316, 1878, 316, 316,
    569, 2358, 568, 2357, 1337, 1334, 1336, 1333, 347, 566, 346, 565, 1081,
    1078, 1080, 1077, 4872, 3638, 316, 3637, 2105, 2102, 3384, 3381, 279, 3126,
    348, 316, 1849, 1846, 3128, 3125, 539, 2328, 316, 2333, 1307, 1304, 1310,
    1309, 304, 536, 287, 316, 1051, 1048, 1054, 1053, 4878, 3608, 316, 3613,
    316, 2072, 2078, 2077, 277, 3096, 316, 3101, 316, 1816, 1822, 1821, 538,
    2381, 316, 2353, 1306, 1357, 1325, 1329, 303, 589, 316, 316, 1050, 1101,
    1069, 1073, 4873, 3661, 316, 3633, 316, 2125, 2093, 2097, 335, 3149, 316,
    3121, 316, 1869, 1837, 1841
};

static const uint8_t alias_c65dtv02[] = {
    0, 6, 2, 3, 4, 5, 6, 7, 8, 9, 8, 11, 7, 13, 14, 15, 16, 17, 18, 19, 20, 21,
    22, 23, 24, 24, 26, 27, 29, 29, 30, 31, 32, 33, 34, 35, 36, 35, 34, 39, 40,
    41, 42, 43, 44, 45, 49, 47, 48, 49, 49, 51, 52, 53, 54, 55, 56, 57, 58, 53,
    60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78,
    79, 80, 5, 58, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 3
};

const struct cpu_s c65dtv02 = {
    "65dtv02",
    mnemonic_c65dtv02,
    opcode_c65dtv02,
    disasm_c65dtv02,
    alias_c65dtv02,
    0x1848001,
    97,
    0xffff,
    51, -1, 55, 54, 56, 57, -1, 87, 86, 88, 89, -1, 24, 25, 26, 27, -1, -1, -1,
    -1, 0, 77, 2, 62, 33, 11, 61, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1
};

static const uint16_t opcode_c65el02[] = {
    11335, 11336, 258, 515, 516, 517, 516, 774, 515, 519, 520, 521, 522, 1035,
    525, 526, 1536, 1537, 1539, 1807, 2051, 1540, 11337, 11337, 2322, 2323,
    1541, 2580, 1542, 1543, 2095, 1610, 11339, 2827, 2820, 2821, 2820, 2827,
    2823, 2824, 2825, 2826, 2828, 2830, 1544, 1545, 2582, 1546, 1548, 8964,
    3853, 11596, 4352, 4685, 4889, 282, 1841, 2096, 1549, 1614, 1553, 11343,
    4352, 5406, 5637, 1311, 1550, 1616, 1555, 1557, 1563, 1564, 1617, 1567,
    1568, 1569, 5970, 6227, 5426, 5639, 1807, 11776, 1554, 1552, 1584, 1583,
    1575, 1566, 1588, 1587, 292, 293, 5972, 1574, 1576, 6229, 11350, 1623, 1577,
    1578, 1579, 1825, 258, 282, 12120, 1544, 4352, 6701, 6927, 2051, 1582, 1625,
    1585, 1586, 1626, 1627, 7185, 1565, 7195, 1589, 1590, 1628, 1551, 1591,
    1592, 1593, 1594, 1595, 1582, 1596, 1629
};

static const uint32_t mnemonic_c65el02[] = {
    0x616463, 0x616e64, 0x61736c, 0x626363, 0x626373, 0x626571, 0x626765, 0x626974,
    0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x627261, 0x62726b, 0x627663, 0x627673,
    0x636c63, 0x636c64, 0x636c69, 0x636c70, 0x636c72, 0x636c76, 0x636d70, 0x637061,
    0x637078, 0x637079, 0x646561, 0x646563, 0x646578, 0x646579, 0x646976, 0x656e74,
    0x656f72, 0x676363, 0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65,
    0x67706c, 0x677261, 0x677663, 0x677673, 0x686c74, 0x696e61, 0x696e63, 0x696e78,
    0x696e79, 0x6a6d70, 0x6a7372, 0x6c6461, 0x6c6472, 0x6c6478, 0x6c6479, 0x6c7372,
    0x6d6d75, 0x6d756c, 0x6e6f70, 0x6e7861, 0x6e7874, 0x6f7261, 0x6f7272, 0x706561,
    0x706569, 0x706572, 0x706861, 0x706864, 0x706870, 0x706878, 0x706879, 0x706c61,
    0x706c64, 0x706c70, 0x706c78, 0x706c79, 0x707368, 0x70756c, 0x726561, 0x726569,
    0x726570, 0x726572, 0x726861, 0x726869, 0x726878, 0x726879, 0x726c61, 0x726c69,
    0x726c78, 0x726c79, 0x726f6c, 0x726f72, 0x727368, 0x727469, 0x727473, 0x72756c,
    0x736263, 0x736561, 0x736563, 0x736564, 0x736569, 0x736570, 0x73686c, 0x736872,
    0x737461, 0x737470, 0x737472, 0x737478, 0x737479, 0x73747a, 0x737761, 0x746164,
    0x746178, 0x746179, 0x746461, 0x746978, 0x747262, 0x747278, 0x747362, 0x747378,
    0x747861, 0x747869, 0x747872, 0x747873, 0x747879, 0x747961, 0x747978, 0x776169,
    0x786261, 0x786365, 0x7a6561
};

static const uint16_t disasm_c65el02[] = {
    525, 2365, 316, 2621, 1398, 1341, 1282, 5949, 324, 573, 2, 339, 1142, 1085,
    1026, 1337, 4875, 3645, 3901, 2877, 1396, 2109, 2050, 5693, 272, 3133, 46,
    340, 1140, 1853, 1794, 2105, 7474, 2305, 287, 2561, 1287, 1281, 1370, 5889,
    329, 513, 90, 343, 1031, 1025, 1114, 1081, 4873, 3585, 3841, 2817, 2055,
    2049, 2138, 5633, 354, 3073, 27, 344, 1799, 1793, 1882, 1849, 349, 2336,
    315, 2592, 590, 1312, 1335, 5920, 322, 544, 55, 338, 7473, 1056, 1079, 1310,
    4878, 3616, 3872, 2848, 1359, 2080, 2103, 5664, 274, 3104, 326, 341, 377,
    1824, 1847, 2078, 350, 2304, 4673, 2560, 1389, 1280, 1371, 5888, 327, 512,
    91, 342, 6705, 1024, 1115, 1054, 4879, 3584, 3840, 2816, 2157, 2048, 2139,
    5632, 356, 3072, 331, 345, 6193, 1792, 1883, 1822, 4876, 2408, 4945, 2664,
    1388, 1384, 1387, 5992, 285, 519, 376, 378, 1132, 1128, 1131, 386, 4867,
    3688, 3944, 2920, 2156, 2152, 3435, 5736, 381, 3176, 379, 380, 1133, 1896,
    1901, 353, 566, 2355, 565, 2611, 1334, 1331, 1333, 5939, 369, 563, 368, 373,
    1078, 1075, 1077, 370, 4868, 3635, 3891, 2867, 2102, 2099, 3381, 5683, 277,
    3123, 375, 382, 1846, 1843, 3125, 367, 537, 2326, 592, 2582, 1305, 1302,
    1307, 5910, 304, 534, 284, 383, 1049, 1046, 1051, 328, 4874, 3606, 3862,
    2838, 1344, 2070, 2075, 5654, 273, 3094, 325, 361, 371, 1814, 1819, 323,
    536, 2400, 613, 2656, 1304, 1376, 1326, 5984, 303, 608, 314, 384, 1048,
    1120, 1070, 568, 4869, 3680, 3936, 2912, 575, 2144, 2094, 5728, 355, 3168,
    330, 385, 6194, 1888, 1838, 314
};

static const uint8_t alias_c65el02[] = {
    0, 1, 2, 3, 4, 5, 4, 7, 3, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 80, 109,
    21, 22, 22, 24, 25, 27, 27, 28, 29, 30, 31, 32, 33, 34, 35, 34, 33, 38, 39,
    40, 41, 42, 43, 105, 46, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58,
    59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
    78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96,
    97, 98, 99, 100, 101, 2, 55, 104, 105, 106, 107, 108, 109, 128, 111, 112,
    113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127,
    128, 129, 130
};

const struct cpu_s c65el02 = {
    "65el02",
    mnemonic_c65el02,
    opcode_c65el02,
    disasm_c65el02,
    alias_c65el02,
    0x3868109,
    131,
    0xffff,
    49, -1, 52, 51, 53, 54, -1, 106, 104, 107, 108, 109, 22, 23, 24, 25, -1,
    111, 121, 122, 0, 96, 1, 62, 32, 7, 61, 118, 116, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1
};

static const uint16_t opcode_r65c02[] = {
    8704, 8705, 258, 12288, 12289, 515, 516, 517, 516, 774, 515, 519, 520, 521,
    522, 1035, 525, 526, 1536, 1537, 1539, 2051, 1540, 8720, 8720, 2322, 2323,
    1541, 2580, 1542, 1543, 8725, 2827, 2820, 2821, 2820, 2827, 2823, 2824,
    2825, 2826, 2828, 2830, 1545, 2582, 1546, 1548, 8964, 7949, 9279, 4352,
    4672, 4929, 282, 12638, 8733, 4352, 1550, 1555, 1557, 1563, 1564, 1567,
    1568, 1569, 6212, 6213, 12804, 292, 293, 1574, 1576, 8736, 1577, 1578, 1579,
    258, 282, 12808, 9500, 4352, 6701, 6927, 2051, 1585, 1586, 7185, 7195, 1589,
    1590, 1591, 1593
};

static const uint32_t mnemonic_r65c02[] = {
    0x616463, 0x616e64, 0x61736c, 0x626272, 0x626273, 0x626363, 0x626373, 0x626571,
    0x626765, 0x626974, 0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x627261, 0x62726b,
    0x627663, 0x627673, 0x636c63, 0x636c64, 0x636c69, 0x636c72, 0x636c76, 0x636d70,
    0x637061, 0x637078, 0x637079, 0x646561, 0x646563, 0x646578, 0x646579, 0x656f72,
    0x676363, 0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65, 0x67706c,
    0x677261, 0x677663, 0x677673, 0x696e61, 0x696e63, 0x696e78, 0x696e79, 0x6a6d70,
    0x6a7372, 0x6c6461, 0x6c6472, 0x6c6478, 0x6c6479, 0x6c7372, 0x6e6f70, 0x6f7261,
    0x6f7272, 0x706861, 0x706870, 0x706878, 0x706879, 0x706c61, 0x706c70, 0x706c78,
    0x706c79, 0x707368, 0x70756c, 0x726d62, 0x726f6c, 0x726f72, 0x727469, 0x727473,
    0x736263, 0x736563, 0x736564, 0x736569, 0x73686c, 0x736872, 0x736d62, 0x737461,
    0x737472, 0x737478, 0x737479, 0x73747a, 0x746178, 0x746179, 0x747262, 0x747362,
    0x747378, 0x747861, 0x747873, 0x747961
};

static const uint16_t disasm_r65c02[] = {
    527, 2359, 310, 310, 1367, 1335, 1282, 5443, 314, 567, 2, 310, 1111, 1079,
    1026, 5123, 4877, 3639, 3895, 310, 1366, 2103, 2050, 5443, 274, 3127, 44,
    310, 1110, 1847, 1794, 5123, 7472, 2305, 310, 310, 1289, 1281, 1348, 5443,
    318, 513, 68, 310, 1033, 1025, 1092, 5123, 4875, 3585, 3841, 310, 2057,
    2049, 2116, 5443, 329, 3073, 28, 310, 1801, 1793, 1860, 5123, 326, 2335,
    310, 310, 1334, 1311, 1333, 5443, 313, 543, 53, 310, 7471, 1055, 1077, 5123,
    4880, 3615, 3871, 310, 2102, 2079, 2101, 5443, 276, 3103, 316, 310, 310,
    1823, 1845, 5123, 327, 2304, 310, 310, 1363, 1280, 1349, 5443, 317, 512, 69,
    310, 6703, 1024, 1093, 5123, 4881, 3584, 3840, 310, 2131, 2048, 2117, 5443,
    331, 3072, 320, 310, 6191, 1792, 1861, 5123, 4878, 2383, 566, 310, 1362,
    1359, 1361, 5454, 286, 521, 345, 310, 1106, 1103, 1105, 5124, 4869, 3663,
    3919, 310, 2130, 2127, 3409, 5454, 347, 3151, 346, 310, 1107, 1871, 1875,
    5124, 564, 2353, 563, 310, 1332, 1329, 1331, 5454, 341, 561, 340, 310, 1076,
    1073, 1075, 5124, 4870, 3633, 3889, 310, 2100, 2097, 3379, 5454, 278, 3121,
    344, 310, 1844, 1841, 3123, 5124, 538, 2327, 310, 310, 1306, 1303, 1308,
    5454, 302, 535, 285, 310, 1050, 1047, 1052, 5124, 4876, 3607, 3863, 310,
    310, 2071, 2076, 5454, 275, 3095, 315, 310, 1078, 1815, 1820, 5124, 537,
    2376, 310, 310, 1305, 1352, 1324, 5454, 301, 584, 310, 310, 1049, 1096,
    1068, 5124, 4871, 3656, 3912, 310, 310, 2120, 2092, 5454, 330, 3144, 319,
    310, 310, 1864, 1836, 5124
};

static const uint8_t alias_r65c02[] = {
    0, 1, 2, 3, 4, 5, 6, 7, 6, 9, 5, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 83,
    22, 23, 23, 25, 26, 28, 28, 29, 30, 31, 32, 33, 34, 33, 32, 37, 38, 39, 40,
    41, 42, 44, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59,
    60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 2, 53, 78,
    79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91
};

const struct cpu_s r65c02 = {
    "r65c02",
    mnemonic_r65c02,
    opcode_r65c02,
    disasm_r65c02,
    alias_r65c02,
    0x3848001,
    92,
    0xffff,
    47, -1, 50, 49, 51, 52, -1, 80, 79, 81, 82, 83, 23, 24, 25, 26, -1, -1, -1,
    -1, 0, 72, 1, 56, 31, 9, 55, 87, 86, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1
};

static const uint16_t opcode_w65c02[] = {
    8704, 8705, 258, 12288, 12289, 515, 516, 517, 516, 774, 515, 519, 520, 521,
    522, 1035, 525, 526, 1536, 1537, 1539, 2051, 1540, 8720, 8720, 2322, 2323,
    1541, 2580, 1542, 1543, 8725, 2827, 2820, 2821, 2820, 2827, 2823, 2824,
    2825, 2826, 2828, 2830, 1544, 1545, 2582, 1546, 1548, 8964, 7949, 9279,
    4352, 4672, 4929, 282, 12638, 8733, 4352, 1550, 1555, 1557, 1563, 1564,
    1567, 1568, 1569, 6212, 6213, 12804, 292, 293, 1574, 1576, 8736, 1577, 1578,
    1579, 258, 282, 12808, 9500, 1544, 4352, 6701, 6927, 2051, 1585, 1586, 7185,
    7195, 1589, 1590, 1591, 1593, 1595
};

static const uint32_t mnemonic_w65c02[] = {
    0x616463, 0x616e64, 0x61736c, 0x626272, 0x626273, 0x626363, 0x626373, 0x626571,
    0x626765, 0x626974, 0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x627261, 0x62726b,
    0x627663, 0x627673, 0x636c63, 0x636c64, 0x636c69, 0x636c72, 0x636c76, 0x636d70,
    0x637061, 0x637078, 0x637079, 0x646561, 0x646563, 0x646578, 0x646579, 0x656f72,
    0x676363, 0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65, 0x67706c,
    0x677261, 0x677663, 0x677673, 0x686c74, 0x696e61, 0x696e63, 0x696e78, 0x696e79,
    0x6a6d70, 0x6a7372, 0x6c6461, 0x6c6472, 0x6c6478, 0x6c6479, 0x6c7372, 0x6e6f70,
    0x6f7261, 0x6f7272, 0x706861, 0x706870, 0x706878, 0x706879, 0x706c61, 0x706c70,
    0x706c78, 0x706c79, 0x707368, 0x70756c, 0x726d62, 0x726f6c, 0x726f72, 0x727469,
    0x727473, 0x736263, 0x736563, 0x736564, 0x736569, 0x73686c, 0x736872, 0x736d62,
    0x737461, 0x737470, 0x737472, 0x737478, 0x737479, 0x73747a, 0x746178, 0x746179,
    0x747262, 0x747362, 0x747378, 0x747861, 0x747873, 0x747961, 0x776169
};

static const uint16_t disasm_w65c02[] = {
    527, 2360, 311, 311, 1369, 1336, 1282, 5444, 315, 568, 2, 311, 1113, 1080,
    1026, 5123, 4877, 3640, 3896, 311, 1368, 2104, 2050, 5444, 274, 3128, 45,
    311, 1112, 1848, 1794, 5123, 7473, 2305, 311, 311, 1289, 1281, 1349, 5444,
    319, 513, 69, 311, 1033, 1025, 1093, 5123, 4875, 3585, 3841, 311, 2057,
    2049, 2117, 5444, 330, 3073, 28, 311, 1801, 1793, 1861, 5123, 327, 2335,
    311, 311, 1335, 1311, 1334, 5444, 314, 543, 54, 311, 7472, 1055, 1078, 5123,
    4880, 3615, 3871, 311, 2103, 2079, 2102, 5444, 276, 3103, 317, 311, 311,
    1823, 1846, 5123, 328, 2304, 311, 311, 1365, 1280, 1350, 5444, 318, 512, 70,
    311, 6704, 1024, 1094, 5123, 4881, 3584, 3840, 311, 2133, 2048, 2118, 5444,
    332, 3072, 321, 311, 6192, 1792, 1862, 5123, 4878, 2384, 567, 311, 1364,
    1360, 1363, 5455, 286, 521, 347, 311, 1108, 1104, 1107, 5124, 4869, 3664,
    3920, 311, 2132, 2128, 3411, 5455, 349, 3152, 348, 311, 1109, 1872, 1877,
    5124, 565, 2354, 564, 311, 1333, 1330, 1332, 5455, 343, 562, 342, 311, 1077,
    1074, 1076, 5124, 4870, 3634, 3890, 311, 2101, 2098, 3380, 5455, 278, 3122,
    346, 311, 1845, 1842, 3124, 5124, 538, 2327, 311, 311, 1306, 1303, 1308,
    5455, 303, 535, 285, 350, 1050, 1047, 1052, 5124, 4876, 3607, 3863, 311,
    311, 2071, 2076, 5455, 275, 3095, 316, 337, 1079, 1815, 1820, 5124, 537,
    2377, 311, 311, 1305, 1353, 1325, 5455, 302, 585, 311, 311, 1049, 1097,
    1069, 5124, 4871, 3657, 3913, 311, 311, 2121, 2093, 5455, 331, 3145, 320,
    311, 311, 1865, 1837, 5124
};

static const uint8_t alias_w65c02[] = {
    0, 1, 2, 3, 4, 5, 6, 7, 6, 9, 5, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 85,
    22, 23, 23, 25, 26, 28, 28, 29, 30, 31, 32, 33, 34, 33, 32, 37, 38, 39, 40,
    41, 42, 81, 45, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59,
    60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 2, 54,
    79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94
};

const struct cpu_s w65c02 = {
    "w65c02",
    mnemonic_w65c02,
    opcode_w65c02,
    disasm_w65c02,
    alias_w65c02,
    0x3848001,
    95,
    0xffff,
    48, -1, 51, 50, 52, 53, -1, 82, 80, 83, 84, 85, 23, 24, 25, 26, -1, -1, -1,
    -1, 0, 73, 1, 57, 31, 9, 56, 89, 88, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1
};

static const uint16_t opcode_c65ce02[] = {
    13056, 13057, 258, 13407, 13617, 12288, 12289, 13827, 13828, 13829, 13828,
    774, 13827, 13831, 13832, 13833, 13834, 1035, 1331, 13837, 13838, 1536,
    1537, 1553, 1539, 1540, 13072, 13072, 2322, 2323, 2356, 1541, 2656, 14089,
    1542, 1543, 1588, 13077, 2827, 2820, 2821, 2820, 2827, 2823, 2824, 2825,
    2826, 2828, 2830, 1545, 2657, 14090, 1546, 1548, 1584, 8964, 9011, 14434,
    4352, 4672, 4963, 14692, 282, 14949, 1549, 13085, 4352, 1550, 1555, 5429,
    1557, 1563, 1544, 1564, 1567, 1568, 1569, 1596, 15206, 6247, 13622, 12804,
    292, 293, 13622, 1574, 1823, 1128, 13088, 1577, 1578, 1641, 1579, 258, 282,
    12808, 15466, 4352, 15661, 2667, 2051, 1583, 1583, 1585, 1586, 1554, 1587,
    1587, 7185, 7195, 1589, 1552, 1590, 1591, 1593, 1566, 1575
};

static const uint32_t mnemonic_c65ce02[] = {
    0x616463, 0x616e64, 0x61736c, 0x617372, 0x617377, 0x626272, 0x626273, 0x626363,
    0x626373, 0x626571, 0x626765, 0x626974, 0x626c74, 0x626d69, 0x626e65, 0x62706c,
    0x627261, 0x62726b, 0x627372, 0x627663, 0x627673, 0x636c63, 0x636c64, 0x636c65,
    0x636c69, 0x636c76, 0x636d70, 0x637061, 0x637078, 0x637079, 0x63707a, 0x646561,
    0x646563, 0x646577, 0x646578, 0x646579, 0x64657a, 0x656f72, 0x676363, 0x676373,
    0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65, 0x67706c, 0x677261, 0x677663,
    0x677673, 0x696e61, 0x696e63, 0x696e77, 0x696e78, 0x696e79, 0x696e7a, 0x6a6d70,
    0x6a7372, 0x6c6461, 0x6c6472, 0x6c6478, 0x6c6479, 0x6c647a, 0x6c7372, 0x6e6567,
    0x6e6f70, 0x6f7261, 0x6f7272, 0x706861, 0x706870, 0x706877, 0x706878, 0x706879,
    0x70687a, 0x706c61, 0x706c70, 0x706c78, 0x706c79, 0x706c7a, 0x707368, 0x70756c,
    0x726c77, 0x726d62, 0x726f6c, 0x726f72, 0x726f77, 0x727469, 0x72746e, 0x727473,
    0x736263, 0x736563, 0x736564, 0x736565, 0x736569, 0x73686c, 0x736872, 0x736d62,
    0x737461, 0x737472, 0x737478, 0x737479, 0x73747a, 0x746162, 0x746164, 0x746178,
    0x746179, 0x74617a, 0x746261, 0x746461, 0x747262, 0x747362, 0x747378, 0x747379,
    0x747861, 0x747873, 0x747961, 0x747973, 0x747a61
};

static const uint16_t disasm_c65ce02[] = {
    529, 2369, 279, 347, 1389, 1345, 1282, 5457, 324, 577, 2, 367, 1133, 1089,
    1026, 5125, 4879, 3649, 6977, 4623, 1388, 2113, 2050, 5457, 277, 3137, 50,
    310, 1132, 1857, 1794, 5125, 7480, 2305, 6712, 6200, 1291, 1281, 1362, 5457,
    330, 513, 82, 371, 1035, 1025, 1106, 5125, 4877, 3585, 6913, 4621, 2059,
    2049, 2130, 5457, 345, 3073, 32, 292, 1803, 1793, 1874, 5125, 341, 2341, 63,
    3, 1283, 1317, 1342, 5457, 323, 549, 62, 361, 7479, 1061, 1086, 5125, 4883,
    3621, 6949, 4627, 2051, 2085, 2110, 5457, 280, 3109, 327, 358, 320, 1829,
    1854, 5125, 343, 2304, 599, 4626, 1380, 1280, 1363, 5457, 329, 512, 83, 372,
    6711, 1024, 1107, 5125, 4884, 3584, 6912, 4628, 2148, 2048, 2131, 5457, 348,
    3072, 332, 363, 6199, 1792, 1875, 5125, 4880, 2400, 2912, 4624, 1379, 1376,
    1378, 5471, 291, 523, 368, 1891, 1123, 1120, 1122, 5126, 4871, 3680, 7008,
    4615, 2147, 2144, 3426, 5471, 370, 3168, 369, 3170, 1124, 1888, 1892, 5126,
    572, 2361, 571, 573, 1340, 1337, 1339, 5471, 360, 569, 359, 1085, 1084,
    1081, 1083, 5126, 4872, 3641, 6969, 4616, 2108, 2105, 3387, 5471, 281, 3129,
    366, 1853, 1852, 1849, 3131, 5126, 541, 2330, 542, 1313, 1309, 1306, 1312,
    5471, 309, 538, 290, 1028, 1053, 1050, 1056, 5126, 4878, 3610, 6938, 4622,
    1310, 2074, 2080, 5471, 278, 3098, 326, 328, 1054, 1818, 1824, 5126, 540,
    2392, 2873, 1331, 1308, 1368, 1330, 5471, 308, 600, 320, 1104, 1052, 1112,
    1074, 5126, 4873, 3672, 7000, 4617, 581, 2136, 2098, 5471, 346, 3160, 331,
    333, 1093, 1880, 1842, 5126
};

static const uint8_t alias_c65ce02[] = {
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 8, 11, 7, 13, 14, 15, 16, 17, 18, 19, 20, 21,
    22, 23, 24, 25, 26, 26, 28, 29, 30, 32, 32, 33, 34, 35, 36, 37, 38, 39, 40,
    39, 38, 43, 44, 45, 46, 47, 48, 50, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59,
    60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78,
    79, 80, 81, 82, 83, 80, 85, 87, 87, 88, 89, 90, 91, 92, 2, 62, 95, 96, 97,
    98, 99, 100, 102, 102, 103, 104, 105, 107, 107, 108, 109, 110, 111, 112,
    113, 114, 115, 116
};

const struct cpu_s c65ce02 = {
    "65ce02",
    mnemonic_c65ce02,
    opcode_c65ce02,
    disasm_c65ce02,
    alias_c65ce02,
    0x3848009,
    117,
    0xffff,
    55, -1, 58, 57, 59, 60, 61, 97, 96, 98, 99, 100, 26, 27, 28, 29, 30, 102,
    -1, -1, 0, 88, 1, 66, 37, 11, 65, 109, 108, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1
};

static const uint16_t opcode_c4510[] = {
    13056, 13057, 258, 13407, 13617, 12288, 12289, 13827, 13828, 13829, 13828,
    774, 13827, 13831, 13832, 13833, 13834, 1035, 1331, 13837, 13838, 1536,
    1537, 1553, 1539, 1540, 13072, 13072, 2322, 2323, 2356, 1541, 2656, 14089,
    1542, 1543, 1588, 1549, 13077, 2827, 2820, 2821, 2820, 2827, 2823, 2824,
    2825, 2826, 2828, 2830, 1545, 2657, 14090, 1546, 1548, 1584, 8964, 9011,
    14434, 4352, 4672, 4963, 14692, 282, 1628, 14949, 1549, 13085, 4352, 1550,
    1555, 5429, 1557, 1563, 1544, 1564, 1567, 1568, 1569, 1596, 15206, 6247,
    13622, 12804, 292, 293, 13622, 1574, 1823, 1128, 13088, 1577, 1578, 1641,
    1579, 258, 282, 12808, 15466, 4352, 15661, 2667, 2051, 1583, 1583, 1585,
    1586, 1554, 1587, 1587, 7185, 7195, 1589, 1552, 1590, 1591, 1593, 1566,
    1575
};

static const uint32_t mnemonic_c4510[] = {
    0x616463, 0x616e64, 0x61736c, 0x617372, 0x617377, 0x626272, 0x626273, 0x626363,
    0x626373, 0x626571, 0x626765, 0x626974, 0x626c74, 0x626d69, 0x626e65, 0x62706c,
    0x627261, 0x62726b, 0x627372, 0x627663, 0x627673, 0x636c63, 0x636c64, 0x636c65,
    0x636c69, 0x636c76, 0x636d70, 0x637061, 0x637078, 0x637079, 0x63707a, 0x646561,
    0x646563, 0x646577, 0x646578, 0x646579, 0x64657a, 0x656f6d, 0x656f72, 0x676363,
    0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65, 0x67706c, 0x677261,
    0x677663, 0x677673, 0x696e61, 0x696e63, 0x696e77, 0x696e78, 0x696e79, 0x696e7a,
    0x6a6d70, 0x6a7372, 0x6c6461, 0x6c6472, 0x6c6478, 0x6c6479, 0x6c647a, 0x6c7372,
    0x6d6170, 0x6e6567, 0x6e6f70, 0x6f7261, 0x6f7272, 0x706861, 0x706870, 0x706877,
    0x706878, 0x706879, 0x70687a, 0x706c61, 0x706c70, 0x706c78, 0x706c79, 0x706c7a,
    0x707368, 0x70756c, 0x726c77, 0x726d62, 0x726f6c, 0x726f72, 0x726f77, 0x727469,
    0x72746e, 0x727473, 0x736263, 0x736563, 0x736564, 0x736565, 0x736569, 0x73686c,
    0x736872, 0x736d62, 0x737461, 0x737472, 0x737478, 0x737479, 0x73747a, 0x746162,
    0x746164, 0x746178, 0x746179, 0x74617a, 0x746261, 0x746461, 0x747262, 0x747362,
    0x747378, 0x747379, 0x747861, 0x747873, 0x747961, 0x747973, 0x747a61
};

static const uint16_t disasm_c4510[] = {
    529, 2371, 279, 349, 1391, 1347, 1282, 5459, 326, 579, 2, 369, 1135, 1091,
    1026, 5125, 4879, 3651, 6979, 4623, 1390, 2115, 2050, 5459, 277, 3139, 51,
    311, 1134, 1859, 1794, 5125, 7481, 2305, 6713, 6201, 1291, 1281, 1364, 5459,
    332, 513, 84, 373, 1035, 1025, 1108, 5125, 4877, 3585, 6913, 4621, 2059,
    2049, 2132, 5459, 347, 3073, 32, 292, 1803, 1793, 1876, 5125, 343, 2342, 65,
    3, 1283, 1318, 1343, 5459, 325, 550, 63, 363, 7480, 1062, 1087, 5125, 4883,
    3622, 6950, 4627, 2051, 2086, 2111, 5459, 280, 3110, 329, 360, 320, 1830,
    1855, 5125, 345, 2304, 601, 4626, 1382, 1280, 1365, 5459, 331, 512, 85, 374,
    6712, 1024, 1109, 5125, 4884, 3584, 6912, 4628, 2150, 2048, 2133, 5459, 350,
    3072, 334, 365, 6200, 1792, 1877, 5125, 4880, 2402, 2914, 4624, 1381, 1378,
    1380, 5473, 291, 523, 370, 1893, 1125, 1122, 1124, 5126, 4871, 3682, 7010,
    4615, 2149, 2146, 3428, 5473, 372, 3170, 371, 3172, 1126, 1890, 1894, 5126,
    573, 2362, 572, 574, 1341, 1338, 1340, 5473, 362, 570, 361, 1086, 1085,
    1082, 1084, 5126, 4872, 3642, 6970, 4616, 2109, 2106, 3388, 5473, 281, 3130,
    368, 1854, 1853, 1850, 3132, 5126, 541, 2330, 542, 1313, 1309, 1306, 1312,
    5473, 310, 538, 290, 1028, 1053, 1050, 1056, 5126, 4878, 3610, 6938, 4622,
    1310, 2074, 2080, 5473, 278, 3098, 328, 330, 1054, 1818, 1824, 5126, 540,
    2394, 2874, 1332, 1308, 1370, 1331, 5473, 309, 602, 322, 1106, 1052, 1114,
    1075, 5126, 4873, 3674, 7002, 4617, 583, 2138, 2099, 5473, 348, 3162, 333,
    335, 1095, 1882, 1843, 5126
};

static const uint8_t alias_c4510[] = {
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 8, 11, 7, 13, 14, 15, 16, 17, 18, 19, 20, 21,
    22, 23, 24, 25, 26, 26, 28, 29, 30, 32, 32, 33, 34, 35, 36, 66, 38, 39, 40,
    41, 40, 39, 44, 45, 46, 47, 48, 49, 51, 51, 52, 53, 54, 55, 56, 57, 58, 59,
    60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78,
    79, 80, 81, 82, 83, 84, 85, 82, 87, 89, 89, 90, 91, 92, 93, 94, 2, 63, 97,
    98, 99, 100, 101, 102, 104, 104, 105, 106, 107, 109, 109, 110, 111, 112,
    113, 114, 115, 116, 117, 118
};

const struct cpu_s c4510 = {
    "4510",
    mnemonic_c4510,
    opcode_c4510,
    disasm_c4510,
    alias_c4510,
    0x3848009,
    119,
    0xffff,
    56, -1, 59, 58, 60, 61, 62, 99, 98, 100, 101, 102, 26, 27, 28, 29, 30, 104,
    -1, -1, 0, 90, 1, 68, 38, 11, 67, 111, 110, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1
};

static const uint16_t opcode_c45gs02[] = {
    15927, 16183, 15928, 16184, 16492, 2669, 366, 13423, 13617, 12288, 12289,
    13827, 13828, 13829, 13828, 774, 13827, 13831, 13832, 13833, 13834, 1035,
    1331, 7174, 13837, 13838, 1536, 1537, 1553, 1539, 1540, 15929, 15929, 16185,
    2322, 2323, 2356, 1541, 2672, 2673, 1541, 14089, 1542, 1543, 1588, 1549,
    16186, 15930, 2827, 2820, 2821, 2820, 2827, 2823, 2824, 2825, 2826, 2828,
    2830, 1545, 2674, 2675, 1545, 14090, 1546, 1548, 1584, 8964, 9011, 16738,
    16955, 4352, 4672, 4963, 14692, 2676, 373, 1628, 14949, 1549, 15932, 16188,
    4352, 1550, 1555, 5429, 1557, 1563, 1544, 1564, 1567, 1568, 1569, 1596,
    15206, 6247, 2678, 13622, 12804, 375, 376, 13622, 2681, 1574, 1823, 1128,
    15946, 16202, 1577, 1578, 1641, 1579, 366, 373, 12808, 17258, 16156, 4352,
    15661, 2667, 2051, 1583, 1583, 1585, 1586, 1554, 1587, 1587, 7185, 7195,
    1589, 1552, 1590, 1591, 1593, 1566, 1575
};

static const uint32_t mnemonic_c45gs02[] = {
    0x616463, 0x616471, 0x616e64, 0x616e71, 0x617264, 0x617364, 0x61736c, 0x617372,
    0x617377, 0x626272, 0x626273, 0x626363, 0x626373, 0x626571, 0x626765, 0x626974,
    0x626c74, 0x626d69, 0x626e65, 0x62706c, 0x627261, 0x62726b, 0x627372, 0x627471,
    0x627663, 0x627673, 0x636c63, 0x636c64, 0x636c65, 0x636c69, 0x636c76, 0x636d70,
    0x637061, 0x637071, 0x637078, 0x637079, 0x63707a, 0x646561, 0x646563, 0x646564,
    0x646571, 0x646577, 0x646578, 0x646579, 0x64657a, 0x656f6d, 0x656f71, 0x656f72,
    0x676363, 0x676373, 0x676571, 0x676765, 0x676c74, 0x676d69, 0x676e65, 0x67706c,
    0x677261, 0x677663, 0x677673, 0x696e61, 0x696e63, 0x696e64, 0x696e71, 0x696e77,
    0x696e78, 0x696e79, 0x696e7a, 0x6a6d70, 0x6a7372, 0x6c6461, 0x6c6471, 0x6c6472,
    0x6c6478, 0x6c6479, 0x6c647a, 0x6c7364, 0x6c7372, 0x6d6170, 0x6e6567, 0x6e6f70,
    0x6f7261, 0x6f7271, 0x6f7272, 0x706861, 0x706870, 0x706877, 0x706878, 0x706879,
    0x70687a, 0x706c61, 0x706c70, 0x706c78, 0x706c79, 0x706c7a, 0x707368, 0x70756c,
    0x726c64, 0x726c77, 0x726d62, 0x726f6c, 0x726f72, 0x726f77, 0x727264, 0x727469,
    0x72746e, 0x727473, 0x736263, 0x736271, 0x736563, 0x736564, 0x736565, 0x736569,
    0x73686c, 0x736872, 0x736d62, 0x737461, 0x737471, 0x737472, 0x737478, 0x737479,
    0x73747a, 0x746162, 0x746164, 0x746178, 0x746179, 0x74617a, 0x746261, 0x746461,
    0x747262, 0x747362, 0x747378, 0x747379, 0x747861, 0x747873, 0x747961, 0x747973,
    0x747a61
};

static const uint16_t disasm_c45gs02[] = {
    533, 2384, 284, 366, 1409, 1360, 1286, 5474, 340, 592, 6, 387, 1153, 1104,
    1030, 5129, 4883, 3664, 6992, 4627, 1408, 2128, 2054, 5474, 282, 3152, 60,
    322, 1152, 1872, 1798, 5129, 7492, 2306, 6724, 6212, 1295, 1282, 1379, 5474,
    346, 514, 99, 391, 1039, 1026, 1123, 5129, 4881, 3586, 6914, 4625, 2063,
    2050, 2147, 5474, 364, 3074, 38, 300, 1807, 1794, 1891, 5129, 359, 2351, 78,
    7, 1287, 1327, 1356, 5474, 339, 559, 76, 381, 7491, 1071, 1100, 5129, 4888,
    3631, 6959, 4632, 2055, 2095, 2124, 5474, 285, 3119, 343, 378, 333, 1839,
    1868, 5129, 361, 2304, 617, 4630, 1400, 1280, 1380, 5474, 345, 512, 100,
    392, 6723, 1024, 1124, 5129, 4889, 3584, 6912, 4633, 2168, 2048, 2148, 5474,
    367, 3072, 348, 383, 6211, 1792, 1892, 5129, 4884, 2419, 2931, 4628, 1399,
    1395, 1398, 5490, 299, 527, 388, 1911, 1143, 1139, 1142, 5130, 4875, 3699,
    7027, 4619, 2167, 2163, 3446, 5490, 390, 3187, 389, 3190, 1144, 1907, 1912,
    5130, 585, 2373, 584, 586, 1353, 1349, 1352, 5490, 380, 581, 379, 1098,
    1097, 1093, 1096, 5130, 4876, 3653, 6981, 4620, 2121, 2117, 3400, 5490, 286,
    3141, 386, 1866, 1865, 1861, 3144, 5130, 547, 2335, 548, 1321, 1315, 1311,
    1318, 5490, 321, 543, 298, 1032, 1059, 1055, 1062, 5130, 4882, 3615, 6943,
    4626, 1316, 2079, 2086, 5490, 283, 3103, 342, 344, 1060, 1823, 1830, 5130,
    546, 2410, 2885, 1343, 1314, 1386, 1340, 5490, 320, 618, 335, 1121, 1058,
    1130, 1084, 5130, 4877, 3690, 7018, 4621, 597, 2154, 2108, 5490, 365, 3178,
    347, 349, 1109, 1898, 1852, 5130
};

static const uint8_t alias_c45gs02[] = {
    0, 0, 2, 2, 7, 6, 6, 7, 8, 9, 10, 11, 12, 13, 12, 15, 11, 17, 18, 19, 20,
    21, 22, 15, 24, 25, 26, 27, 28, 29, 30, 31, 31, 31, 34, 35, 36, 38, 38, 38,
    38, 41, 42, 43, 44, 79, 47, 47, 48, 49, 50, 49, 48, 53, 54, 55, 56, 57, 58,
    60, 60, 60, 60, 63, 64, 65, 66, 67, 68, 69, 69, 71, 72, 73, 74, 76, 76, 77,
    78, 79, 80, 80, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 99,
    97, 98, 99, 100, 97, 100, 103, 105, 105, 106, 106, 108, 109, 110, 111, 6,
    76, 114, 115, 115, 117, 118, 119, 120, 122, 122, 123, 124, 125, 127, 127,
    128, 129, 130, 131, 132, 133, 134, 135, 136
};

const struct cpu_s c45gs02 = {
    "45gs02",
    mnemonic_c45gs02,
    opcode_c45gs02,
    disasm_c45gs02,
    alias_c45gs02,
    0x3858009,
    137,
    0xffff,
    67, -1, 71, 69, 72, 73, 74, 117, 115, 118, 119, 120, 31, 32, 34, 35, 36,
    122, -1, -1, 0, 106, 2, 82, 47, 15, 80, 129, 128, 1, 3, 4, 23, 33, 39, 46,
    61, 70, 81, 96, 102, 107, 5, 75, 116, 62, 40
};

const char *reg_names = "axysdrizbkpq";

const char *const addr_modes[ADR_LEN] = {
    "a", " ", "#", " ", " ", " ", " ,x", " ,x", " ,x", "(,x)", " ,s", "(,s),y",
    " ,y", " ,y", "(),y", "()", "[],y", "[]", " ", " ", " ", " ", "(,r),y",
    " ,r", "(,x)", "[]", "()", "(),z", "[],z", " ", "#"
};

const char *const addr_names[ADR_LEN] = {
    "", "", "", "long", "data bank", "direct page", "long x indexed",
    "data bank x indexed", "direct page x indexed", "", "", "",
    "data bank y indexed", "direct page y indexed", "", "", "", "", "", "", "",
    "", "", "", "", "", "", "", "", "", ""
};

const uint8_t regopcode_table[][REG_LEN] = {
    { ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ___ */
    { 0x0A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASL, SHL */
    { 0x3A, 0xCA, 0x88, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEC */
    { 0x1A, 0xE8, 0xC8, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INC */
    { ____, 0x8A, 0x98, 0x3B, 0x7B, ____, ____, ____, ____, ____, ____, ____ }, /* LDA */
    { 0xAA, ____, 0xBB, 0xBA, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDX */
    { 0xA8, 0x9B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDY */
    { 0x4A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LSR, SHR */
    { 0x48, 0xDA, 0x5A, ____, 0x0B, ____, ____, ____, 0x8B, 0x4B, 0x08, ____ }, /* PSH */
    { 0x68, 0xFA, 0x7A, ____, 0x2B, ____, ____, ____, 0xAB, ____, 0x28, ____ }, /* PUL */
    { 0x2A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ROL */
    { 0x6A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ROR */
    { ____, ____, ____, 0x1B, ____, ____, ____, ____, ____, ____, ____, ____ }, /* STA */
    { ____, ____, ____, 0x9A, ____, ____, ____, ____, ____, ____, ____, ____ }, /* STX */
    { ____, 0xCA, 0x88, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEC */
    { ____, 0xE8, 0xC8, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INC */
    { ____, 0x8A, 0x98, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDA */
    { 0xAA, ____, ____, 0xBA, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDX */
    { 0xA8, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDY */
    { 0x48, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x08, ____ }, /* PSH */
    { 0x68, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x28, ____ }, /* PUL */
    { 0x48, 0xDA, 0x5A, ____, ____, ____, ____, ____, ____, ____, 0x08, ____ }, /* PSH */
    { 0x68, 0xFA, 0x7A, ____, ____, ____, ____, ____, ____, ____, 0x28, ____ }, /* PUL */
    { ____, 0x8A, 0x98, ____, 0xAF, ____, ____, ____, ____, ____, ____, ____ }, /* LDA */
    { 0xAA, ____, 0xBB, 0xBA, ____, 0xAB, 0xDC, ____, ____, ____, ____, ____ }, /* LDX */
    { 0x48, 0xDA, 0x5A, ____, 0xDF, ____, ____, ____, ____, ____, 0x08, ____ }, /* PSH */
    { 0x68, 0xFA, 0x7A, ____, 0xCF, ____, ____, ____, ____, ____, 0x28, ____ }, /* PUL */
    { 0x4B, 0x1B, 0x5B, ____, ____, ____, 0x0B, ____, ____, ____, ____, ____ }, /* RSH */
    { 0x6B, 0x3B, 0x7B, ____, ____, ____, 0x2B, ____, ____, ____, ____, ____ }, /* RUL */
    { 0x43, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASR */
    { 0x3A, 0xCA, 0x88, ____, ____, ____, ____, 0x3B, ____, ____, ____, ____ }, /* DEC */
    { 0x1A, 0xE8, 0xC8, ____, ____, ____, ____, 0x1B, ____, ____, ____, ____ }, /* INC */
    { ____, 0x8A, 0x98, ____, 0x7B, ____, ____, 0x6B, ____, ____, ____, ____ }, /* LDA */
    { 0xA8, ____, ____, 0x0B, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDY */
    { 0x4B, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDZ */
    { 0x42, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* NEG */
    { 0x48, 0xDA, 0x5A, ____, ____, ____, ____, 0xDB, ____, ____, 0x08, ____ }, /* PSH */
    { 0x68, 0xFA, 0x7A, ____, ____, ____, ____, 0xFB, ____, ____, 0x28, ____ }, /* PUL */
    { ____, ____, ____, ____, 0x5B, ____, ____, ____, ____, ____, ____, ____ }, /* STA */
    { ____, ____, ____, 0x2B, ____, ____, ____, ____, ____, ____, ____, ____ }, /* STY */
    { ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x43 }, /* ARD */
    { ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x0A }, /* ASD */
    { 0x0A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x0A }, /* ASL, SHL */
    { 0x43, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x43 }, /* ASR */
    { 0x3A, 0xCA, 0x88, ____, ____, ____, ____, 0x3B, ____, ____, ____, 0x3A }, /* DEC */
    { ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x3A }, /* DED */
    { 0x1A, 0xE8, 0xC8, ____, ____, ____, ____, 0x1B, ____, ____, ____, 0x1A }, /* INC */
    { ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x1A }, /* IND */
    { ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x4A }, /* LSD */
    { 0x4A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x4A }, /* LSR, SHR */
    { ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x2A }, /* RLD */
    { 0x2A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x2A }, /* ROL */
    { 0x6A, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x6A }, /* ROR */
    { ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x6A }  /* RRD */
};

const uint32_t opcode_table_modes[] = {
    0x0003dffc, 0x000001b3, 0x00080004, 0x000001b4, 0x00000006, 0x00040004,
    0x00000002, 0x00000004, 0x000001b0, 0x00000034, 0x000001b1, 0x00080010,
    0x02000008, 0x27000000, 0x00000008, 0x21000000, 0x0003dffd, 0x00000000,
    0x00003035, 0x000001b5, 0x40000000, 0x00000014, 0x00008020, 0x00000025,
    0x00000001, 0x0003dff9, 0x00002031, 0x00000130, 0x00000030, 0x000053b4,
    0x24000000, 0x20000000, 0x000053b5, 0x000053b0, 0x0000d3b4, 0x25000000,
    0x0000d3b5, 0x0000d3b0, 0x00005000, 0x00001000, 0x00007234, 0x000001b6,
    0x00002230, 0x00000080, 0x00c0dfb4, 0x00c0dfb5, 0x00080000, 0x00c0dfb0,
    0x00100000, 0x00000136, 0x00200000, 0x080053b4, 0x00000123, 0x00000010,
    0x000c0004, 0x00000020, 0x08005bb5, 0x00000095, 0x00000003, 0x00000015,
    0x08005bb1, 0x00003031, 0x180053b4, 0x00028030, 0x00000121, 0x18005bb5,
    0x18000030, 0x18005bb1
};

const uint8_t opcode_table[][OPR_LEN] = {
    {  0, 0x18, 0x69, 0x6F, 0x6D, 0x65, 0x7F, 0x7D, 0x75, 0x61, 0x63, 0x73, 0x79, ____, 0x71, 0x72, 0x77, 0x67, ____, 0x82, 0x0F }, /* ADC, BBR, CLC, LDR, ORR, RER, STR */
    {  0, 0xD8, 0x29, 0x2F, 0x2D, 0x25, 0x3F, 0x3D, 0x35, 0x21, 0x23, 0x33, 0x39, ____, 0x31, 0x32, 0x37, 0x27, ____, ____, 0x8F }, /* AND, BBS, CLD */
    {  1, 0x0A, ____, ____, 0x0E, 0x06, ____, 0x1E, 0x16, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASL, SHL */
    {  0, 0x58, 0x90, 0x5C, 0x9C, 0x64, ____, 0x9E, 0x74, ____, ____, ____, 0x9F, ____, 0x93, ____, ____, 0xDC, 0x93, 0x90, ____ }, /* AHX, BCC, BLT, CLI, CLR, JML, SHA, STZ */
    {  0, 0xB8, 0xB0, 0x22, 0x4C, 0x07, ____, 0x9C, ____, 0x7C, ____, ____, 0xBB, ____, ____, 0x6C, ____, 0xDC, 0xB3, 0xB0, ____ }, /* BCS, BGE, CLV, GCS, GGE, JMP, JSL, LAE, LAS, LDS, RMB, SHY */
    {  0, 0x3A, 0xF0, ____, 0x4C, 0xD4, ____, ____, ____, ____, ____, ____, 0x9B, ____, ____, 0xD4, ____, ____, 0xF3, 0xF0, ____ }, /* BEQ, DEA, DEQ, GEQ, PEI, SHS, TAS */
    {  0, 0xCA, 0x89, ____, 0x2C, 0x24, ____, 0x3C, 0x34, ____, ____, ____, 0x9E, ____, ____, ____, ____, ____, ____, ____, ____ }, /* BIT, BTQ, DEX, SHX */
    {  0, 0x88, 0x30, ____, 0x4C, 0x54, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x54, ____, ____, 0x33, 0x30, ____ }, /* BMI, DEY, GMI, REI */
    {  0, 0xDB, 0xD0, ____, 0x4C, 0x87, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xD3, 0xD0, ____ }, /* BNE, GNE, HLT, PHZ, SMB, STP */
    {  0, 0x1A, 0x10, ____, 0x4C, 0xC3, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x13, 0x10, ____ }, /* BPL, DEW, GPL, INA, INQ */
    {  0, 0xE8, 0x80, ____, 0x4C, 0xE3, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x83, 0x80, ____ }, /* BRA, GRA, INW, INX */
    {  0, 0x00, 0x00, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x90, ____ }, /* BRK, GCC, GLT */
    {  0, 0xC8, 0x82, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x82, 0x50, ____ }, /* BRL, GVC, INY */
    {  0, 0xEA, 0x50, ____, 0x20, ____, ____, ____, ____, 0xFC, ____, ____, ____, ____, ____, ____, ____, ____, 0x53, 0x50, ____ }, /* BVC, EOM, JSR, NOP */
    {  0, 0x48, 0x70, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x73, 0x70, ____ }, /* BVS, GVS, PHA */
    {  0, 0x8B, 0xC2, ____, 0x8C, 0x84, ____, ____, 0x94, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CLP, PHB, REP, STY, TXR */
    {  0, 0x0B, 0xC9, 0xCF, 0xCD, 0xC5, 0xDF, 0xDD, 0xD5, 0xC1, 0xC3, 0xD3, 0xD9, ____, 0xD1, 0xD2, 0xD7, 0xC7, ____, ____, ____ }, /* CMP, CPA, PHD, RHI, TSY */
    {  0, 0x02, 0x02, ____, 0x1C, 0x14, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CLE, COP, CSP, JAM, NXT, TRB */
    {  0, 0x4B, 0xE0, ____, 0xEC, 0xE4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CPX, PHK, RHA, TAZ */
    {  0, 0x08, 0xC0, ____, 0xCC, 0xC4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CPY, PHP */
    {  2, ____, ____, ____, 0xCE, 0xC6, ____, 0xDE, 0xD6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEC */
    {  0, 0xDA, 0x49, 0x4F, 0x4D, 0x45, 0x5F, 0x5D, 0x55, 0x41, 0x43, 0x53, 0x59, ____, 0x51, 0x52, 0x57, 0x47, ____, ____, ____ }, /* EOR, PHX */
    {  3, ____, ____, ____, 0xEE, 0xE6, ____, 0xFE, 0xF6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INC */
    {  4, ____, 0xA9, 0xAF, 0xAD, 0xA5, 0xBF, 0xBD, 0xB5, 0xA1, 0xA3, 0xB3, 0xB9, ____, 0xB1, 0xB2, 0xB7, 0xA7, ____, ____, ____ }, /* LDA */
    {  5, ____, 0xA2, ____, 0xAE, 0xA6, ____, ____, ____, ____, ____, ____, 0xBE, 0xB6, ____, ____, ____, ____, ____, ____, ____ }, /* LDX */
    {  6, ____, 0xA0, ____, 0xAC, 0xA4, ____, 0xBC, 0xB4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDY */
    {  7, 0x4A, ____, ____, 0x4E, 0x46, ____, 0x5E, 0x56, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LSR, SHR */
    {  0, 0x5A, 0x54, ____, 0x0C, 0x04, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* MVN, PHY, TSB */
    {  0, 0x68, 0x44, ____, 0x8D, 0x85, ____, 0x9D, 0x95, 0x81, ____, ____, 0x99, ____, 0x91, 0x92, ____, 0x92, ____, ____, ____ }, /* MVP, PLA, STA, STQ */
    {  0, 0xAB, 0x09, 0x0F, 0x0D, 0x05, 0x1F, 0x1D, 0x15, 0x01, 0x03, 0x13, 0x19, ____, 0x11, 0x12, 0x17, 0x07, ____, ____, ____ }, /* ORA, PLB, TRX */
    {  0, 0x2B, 0xF4, ____, 0xF4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PEA, PLD, RLI, TYS */
    {  0, 0x28, 0x62, ____, 0xCF, 0xC7, ____, 0xDF, 0xD7, 0xC3, ____, ____, 0xDB, ____, 0xD3, ____, ____, ____, 0x62, ____, ____ }, /* DCM, DCP, PER, PLP, RTN */
    {  0, 0xFA, 0xE9, 0xEF, 0xED, 0xE5, 0xFF, 0xFD, 0xF5, 0xE1, 0xE3, 0xF3, 0xF9, ____, 0xF1, 0xF2, 0xF7, 0xE7, ____, ____, ____ }, /* PLX, SBC */
    {  0, 0x7A, 0xE2, ____, 0xEF, 0xE7, ____, 0xFF, 0xF7, 0xE3, ____, ____, 0xFB, ____, 0xF3, ____, ____, ____, ____, ____, ____ }, /* INS, ISB, ISC, PLY, SEP */
    {  8, ____, 0xF4, ____, ____, 0xD4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PSH */
    {  9, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PUL */
    { 10, 0x2A, ____, ____, 0x2E, 0x26, ____, 0x3E, 0x36, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ROL */
    { 11, 0x6A, ____, ____, 0x6E, 0x66, ____, 0x7E, 0x76, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ROR */
    {  0, 0x40, 0x42, ____, 0x2F, 0x27, ____, 0x3F, 0x37, 0x23, ____, ____, 0x3B, ____, 0x33, ____, ____, ____, ____, ____, ____ }, /* RLA, RTI, SIR, WDM */
    {  0, 0x6B, 0x4B, ____, 0x6F, 0x67, ____, 0x7F, 0x77, 0x63, ____, ____, 0x7B, ____, 0x73, ____, ____, ____, ____, ____, ____ }, /* ALR, ASR, RLA, RRA, RTL, TZA */
    {  0, 0x60, 0x0B, ____, 0x8F, 0x87, ____, ____, ____, 0x83, ____, ____, ____, 0x97, ____, ____, ____, ____, ____, ____, ____ }, /* ANC, RTS, SAX */
    {  0, 0x38, 0x8B, ____, 0x0F, 0x07, ____, 0x1F, 0x17, 0x03, ____, ____, 0x1B, ____, 0x13, ____, ____, ____, ____, ____, ____ }, /* ANE, SEC, SLO, XAA */
    {  0, 0xF8, 0x6B, ____, 0x4F, 0x47, ____, 0x5F, 0x57, 0x43, ____, ____, 0x5B, ____, 0x53, ____, ____, ____, ____, ____, ____ }, /* ARR, SED, SRE */
    {  0, 0x78, 0xCB, ____, 0x4C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x12, ____ }, /* AXS, GRA, SBX, SEI */
    { 12, ____, ____, 0x8F, 0x8D, 0x85, 0x9F, 0x9D, 0x95, 0x81, 0x83, 0x93, 0x99, ____, 0x91, 0x92, 0x97, 0x87, ____, ____, ____ }, /* STA */
    { 13, ____, ____, ____, 0x8E, 0x86, ____, ____, ____, ____, ____, ____, 0x9B, 0x96, ____, ____, ____, ____, ____, ____, ____ }, /* STX */
    {  0, 0xEB, 0xAB, ____, 0xAF, 0xA7, ____, ____, ____, 0xA3, ____, ____, 0xBF, 0xB7, 0xB3, ____, ____, ____, ____, ____, ____ }, /* LAX, LXA, SWA, XBA */
    {  0, 0x5B, 0x12, ____, 0x6F, 0x4F, ____, 0x7F, 0x5F, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0x12, ____ }, /* BRA, DIV, RHY, TAB, TAD, TCD */
    {  0, 0x1B, 0x32, ____, 0x2F, 0x0F, ____, 0x3F, 0x1F, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INZ, MUL, RHX, SAC, TAS, TCS */
    {  0, 0xAA, 0xEF, ____, 0xCB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASW, MMU, TAX */
    {  0, 0xA8, 0x44, ____, 0x44, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* REA, TAY */
    {  0, 0x7B, 0x63, ____, 0x20, ____, ____, ____, ____, 0x23, ____, ____, ____, ____, ____, 0x22, ____, ____, 0x63, ____, ____ }, /* BSR, JSR, RLY, TBA, TDA, TDC */
    {  0, 0x3B, 0xC2, ____, 0xDC, 0xD4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* CPZ, DEZ, RLX, TSA, TSC */
    {  0, 0xBA, 0xF4, ____, 0xFC, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PHW, TSX */
    {  0, 0x8A, ____, ____, 0xEB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RLW, ROW, TXA */
    {  0, 0x9A, 0x69, ____, 0x6D, 0x65, ____, 0x7D, 0x75, 0x61, ____, ____, 0x79, ____, 0x71, 0x72, 0x72, 0x72, ____, ____, ____ }, /* ADC, ADQ, TXS */
    {  0, 0x9B, 0x29, ____, 0x2D, 0x25, ____, 0x3D, 0x35, 0x21, ____, ____, 0x39, ____, 0x31, 0x32, 0x32, 0x32, ____, ____, ____ }, /* AND, ANQ, TXY */
    {  0, 0x98, 0xC9, ____, 0xCD, 0xC5, ____, 0xDD, 0xD5, 0xC1, ____, ____, 0xD9, ____, 0xD1, 0xD2, 0xD2, 0xD2, ____, ____, ____ }, /* CMP, CPA, CPQ, TYA */
    {  0, 0xBB, 0x49, ____, 0x4D, 0x45, ____, 0x5D, 0x55, 0x41, ____, ____, 0x59, ____, 0x51, 0x52, 0x52, 0x52, ____, ____, ____ }, /* EOQ, EOR, TYX */
    {  0, 0xCB, ____, ____, 0xAD, 0xA5, ____, ____, ____, ____, ____, ____, ____, ____, ____, 0xB2, 0xB2, ____, ____, ____, ____ }, /* LDQ, WAI */
    {  0, 0xFB, 0x09, ____, 0x0D, 0x05, ____, 0x1D, 0x15, 0x01, ____, ____, 0x19, ____, 0x11, 0x12, 0x12, 0x12, ____, ____, ____ }, /* ORA, ORQ, PLZ, XCE */
    { 14, ____, ____, ____, 0xCE, 0xC6, ____, 0xDE, 0xD6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEC */
    { 15, ____, ____, ____, 0xEE, 0xE6, ____, 0xFE, 0xF6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INC */
    { 16, ____, 0xA9, ____, 0xAD, 0xA5, ____, 0xBD, 0xB5, 0xA1, ____, ____, 0xB9, ____, 0xB1, 0xB2, ____, ____, ____, ____, ____ }, /* LDA */
    { 17, ____, 0xA2, ____, 0xAE, 0xA6, ____, ____, ____, ____, ____, ____, 0xBE, 0xB6, ____, ____, ____, ____, ____, ____, ____ }, /* LDX */
    { 18, ____, 0xA0, ____, 0xAC, 0xA4, ____, 0xBC, 0xB4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDY */
    { 19, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PSH */
    { 20, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PUL */
    { 21, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PSH */
    { 22, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PUL */
    {  0, 0xEA, 0x80, ____, 0x0C, 0x04, ____, 0x1C, 0x14, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* NOP */
    {  0, 0x67, 0x69, ____, 0x6D, 0x65, ____, 0x7D, 0x75, 0x61, 0x63, 0x73, 0x79, ____, 0x71, 0x72, 0x77, ____, ____, ____, ____ }, /* ADC */
    {  0, 0x27, 0x29, ____, 0x2D, 0x25, ____, 0x3D, 0x35, 0x21, 0x23, 0x33, 0x39, ____, 0x31, 0x32, 0x37, ____, ____, ____, ____ }, /* AND */
    {  0, 0xC7, 0xC9, ____, 0xCD, 0xC5, ____, 0xDD, 0xD5, 0xC1, 0xC3, 0xD3, 0xD9, ____, 0xD1, 0xD2, 0xD7, ____, ____, ____, ____ }, /* CMP, CPA */
    {  0, 0x22, 0xE9, ____, 0xED, 0xE5, ____, 0xFD, 0xF5, 0xE1, ____, ____, 0xF9, ____, 0xF1, 0xF2, 0xF2, 0xF2, ____, ____, ____ }, /* ENT, SBC, SBQ */
    {  0, 0x47, 0x49, ____, 0x4D, 0x45, ____, 0x5D, 0x55, 0x41, 0x43, 0x53, 0x59, ____, 0x51, 0x52, 0x57, ____, ____, ____, ____ }, /* EOR */
    { 23, 0xA7, 0xA9, ____, 0xAD, 0xA5, ____, 0xBD, 0xB5, 0xA1, 0xA3, 0xB3, 0xB9, ____, 0xB1, 0xB2, 0xB7, ____, ____, ____, ____ }, /* LDA */
    { 24, ____, 0xA2, ____, 0xAE, 0xA6, ____, ____, ____, ____, ____, ____, 0xBE, 0xB6, ____, ____, ____, ____, ____, ____, ____ }, /* LDX */
    {  0, 0x42, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* NXA */
    {  0, 0x07, 0x09, ____, 0x0D, 0x05, ____, 0x1D, 0x15, 0x01, 0x03, 0x13, 0x19, ____, 0x11, 0x12, 0x17, ____, ____, ____, ____ }, /* ORA */
    {  0, 0xDF, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PHD */
    {  0, 0xCF, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PLD */
    { 25, ____, 0xF4, ____, ____, 0xD4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PSH */
    { 26, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PUL */
    { 27, ____, 0x44, ____, ____, 0x54, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RSH */
    { 28, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RUL */
    {  0, 0xE7, 0xE9, ____, 0xED, 0xE5, ____, 0xFD, 0xF5, 0xE1, 0xE3, 0xF3, 0xF9, ____, 0xF1, 0xF2, 0xF7, ____, ____, ____, ____ }, /* SBC */
    {  0, 0x9F, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SEA */
    {  0, 0x87, ____, ____, 0x8D, 0x85, ____, 0x9D, 0x95, 0x81, 0x83, 0x93, 0x99, ____, 0x91, 0x92, 0x97, ____, ____, ____, ____ }, /* STA */
    {  0, 0xBF, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TAD */
    {  0, 0xAF, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TDA */
    {  0, 0xDC, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* TIX */
    {  0, 0x5C, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* MAP, TXI */
    {  0, 0x8F, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ZEA */
    {  0, 0xEA, 0x82, ____, 0xDC, 0x44, ____, ____, 0x54, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* NOP */
    { 29, 0x43, ____, ____, ____, 0x44, ____, ____, 0x54, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASR */
    { 30, ____, ____, ____, 0xCE, 0xC6, ____, 0xDE, 0xD6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEC */
    { 31, ____, ____, ____, 0xEE, 0xE6, ____, 0xFE, 0xF6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INC */
    { 32, ____, 0xA9, ____, 0xAD, 0xA5, ____, 0xBD, 0xB5, 0xA1, ____, 0xE2, 0xB9, ____, 0xB1, 0xB2, 0xB2, ____, ____, ____, ____ }, /* LDA */
    { 33, ____, 0xA0, ____, 0xAC, 0xA4, ____, 0xBC, 0xB4, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDY */
    { 34, ____, 0xA3, ____, 0xAB, ____, ____, 0xBB, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LDZ */
    { 35, 0x42, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* NEG */
    { 36, ____, 0xF4, ____, 0xFC, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PSH */
    { 37, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* PUL */
    {  0, 0x60, 0x62, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RTS */
    {  0, 0x03, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* SEE */
    { 38, ____, ____, ____, 0x8D, 0x85, ____, 0x9D, 0x95, 0x81, ____, 0x82, 0x99, ____, 0x91, 0x92, 0x92, ____, ____, ____, ____ }, /* STA */
    { 39, ____, ____, ____, 0x8C, 0x84, ____, 0x8B, 0x94, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* STY */
    { 40, ____, ____, ____, ____, 0x44, ____, ____, 0x54, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ARD */
    { 41, ____, ____, ____, 0x0E, 0x06, ____, 0x1E, 0x16, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASD */
    { 42, 0x0A, ____, ____, 0x0E, 0x06, ____, 0x1E, 0x16, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASL, SHL */
    { 43, 0x43, ____, ____, ____, 0x44, ____, ____, 0x54, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ASR */
    { 44, ____, ____, ____, 0xCE, 0xC6, ____, 0xDE, 0xD6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DEC */
    { 45, ____, ____, ____, 0xCE, 0xC6, ____, 0xDE, 0xD6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* DED */
    { 46, ____, ____, ____, 0xEE, 0xE6, ____, 0xFE, 0xF6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* INC */
    { 47, ____, ____, ____, 0xEE, 0xE6, ____, 0xFE, 0xF6, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* IND */
    { 48, ____, ____, ____, 0x4E, 0x46, ____, 0x5E, 0x56, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LSD */
    { 49, 0x4A, ____, ____, 0x4E, 0x46, ____, 0x5E, 0x56, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* LSR, SHR */
    { 50, ____, ____, ____, 0x2E, 0x26, ____, 0x3E, 0x36, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* RLD */
    { 51, 0x2A, ____, ____, 0x2E, 0x26, ____, 0x3E, 0x36, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ROL */
    { 52, 0x6A, ____, ____, 0x6E, 0x66, ____, 0x7E, 0x76, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }, /* ROR */
    { 53, ____, ____, ____, 0x6E, 0x66, ____, 0x7E, 0x76, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____, ____ }  /* RRD */
};
