/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Non-linear masking correction based on this paper:
 *   N. Moroney. Local Color Correction Using Non-Linear Masking.
 *   IS&T 8th Color Imaging Conference, 108-111 (2000).
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include "exrinput.h"
#include "exroutput.h"

/**
 * Luminance function for sRGB.
 */
static double luminance( double r, double g, double b )
{
    return ( 0.2126 * r ) + ( 0.7152 * g ) + ( 0.0722 * b );
}

int main( int argc, char **argv )
{
    exroutput_t *exrout;
    exrinput_t *exrin;
    exrinput_t *whitein;
    double *scanline;
    int width, height;
    int y;

    if( argc < 4 ) {
        fprintf( stderr, "Usage: exrnlm input.exr blur.exr output.exr\n" );
        return 1;
    }

    exrin = exrinput_new( argv[ 1 ] );
    if( !exrin ) {
        fprintf( stderr, "exrnlm: Cannot open %s.\n", argv[ 1 ] );
        return 1;
    }
    width = exrinput_get_width( exrin );
    height = exrinput_get_height( exrin );
    fprintf( stdout, "exrnlm: %s [%dx%d]\n", argv[ 1 ], width, height );
    scanline = malloc( width * 4 * sizeof( double ) );

    if( !scanline ) {
        fprintf( stderr, "exrnlm: Cannot allocate memory.\n" );
        exrinput_delete( exrin );
        return 1;
    }

    whitein = exrinput_new( argv[ 2 ] );
    if( !whitein ) {
        fprintf( stderr, "exrnlm: Cannot open %s.\n", argv[ 2 ] );
        exrinput_delete( exrin );
        free( scanline );
        return 1;
    }

    if( exrinput_get_width( whitein ) != width || exrinput_get_height( whitein ) != height ) {
        fprintf( stderr, "exrnlm: Whitepoint image (%dx%d) does not match input image (%dx%d).\n",
                 exrinput_get_width( whitein ), exrinput_get_height( whitein ), width, height );
        exrinput_delete( whitein );
        exrinput_delete( exrin );
        free( scanline );
        return 1;
    }

    exrout = exroutput_new( argv[ 3 ], width, height );
    if( !exrout ) {
        fprintf( stderr, "exrnlm: Cannot open %s.\n", argv[ 3 ] );
        exrinput_delete( whitein );
        exrinput_delete( exrin );
        free( scanline );
        return 1;
    }

    for( y = 0; y < height; y++ ) {
        double *input = exrinput_get_scanline( exrin, y );
        double *whiteinput = exrinput_get_scanline( whitein, y );
        int x;

        fprintf( stderr, "exrnlm: Correcting scanline %d\r", y );

        for( x = 0; x < width; x++ ) {
            double r, g, b;
            double wY;

            r = whiteinput[ (x * 4) + 0 ];
            g = whiteinput[ (x * 4) + 1 ];
            b = whiteinput[ (x * 4) + 2 ];
            wY = 1.0 - luminance( r, g, b );

            r = input[ (x * 4) + 0 ];
            g = input[ (x * 4) + 1 ];
            b = input[ (x * 4) + 2 ];

            r = pow( r, pow( 2, (0.5 - wY) / 0.5 ) );
            g = pow( g, pow( 2, (0.5 - wY) / 0.5 ) );
            b = pow( b, pow( 2, (0.5 - wY) / 0.5 ) );

            scanline[ (x * 4) + 0 ] = r;
            scanline[ (x * 4) + 1 ] = g;
            scanline[ (x * 4) + 2 ] = b;
            scanline[ (x * 4) + 3 ] = input[ (x * 4) + 3 ];
        }
        exroutput_scanline( exrout, scanline );
    }
    fprintf( stderr, "\nexrnlm: Done.\n" );

    exroutput_delete( exrout );
    exrinput_delete( exrin );
    free( scanline );
    return 0;
}

