/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file configfile.c
 * \brief Preferences parsing
 */

#include <ffgtk.h>

/**
 * \brief Get path basename
 * \param pnName name
 * \return path basename
 */
static gchar *getPathBaseName( const char *pnName ) {
	const char *pnChar;
	
	if ( ( pnChar = strrchr( pnName, '/' ) ) != NULL ) {
		return g_strdup( pnChar + 1 );
	}

	return g_strdup( pnName );
}

/**
 * \brief Get path dirname
 * \param pnName name
 * \return path dirname
 */
static char *getPathDirName( const char *pnName ) {
	char *pnChar, *pnString;

	pnString = g_strdup( pnName );

	if ( ( pnChar = strrchr( pnString, '/' ) ) != NULL ) {
		*pnChar = '\0';

		if ( *pnString == '\0' ) {
			g_free( pnString );
			pnString = g_strdup( "/" );
		}
	} else {
		g_free( pnString );
		pnString = g_strdup( "." );
	}

	return pnString;
}

/**
 * \brief Find preferences
 * \param psProfile profile structure
 * \param pnName preference name
 * \return preferences pointer or NULL on error
 */
static struct sPref *findPref( struct sProfile *psProfile, const char *pnName ) {
	g_return_val_if_fail( psProfile != NULL && pnName != NULL && pnName[ 0 ] == '/', NULL );

	if ( pnName[ 1 ] == '\0' ) {
		return &psProfile -> sPrefs;
	} else {
		if ( psProfile -> psPrefsHash ) {
			return g_hash_table_lookup( psProfile -> psPrefsHash, pnName );
		} else {
			return NULL;
		}
	}
}

/**
 * \brief Find preference parent
 * \param psProfile profile structure
 * \param pnName preference name
 * \return pointer to parent of preference or NULL on error
 */
static struct sPref *findPrefParent( struct sProfile *psProfile, const char *pnName ) {
	char *pnParentName = getPathDirName( pnName );
	struct sPref *psRet = &psProfile -> sPrefs;

	if ( strcmp( pnParentName, "/" ) ) {
		psRet = findPref( psProfile, pnParentName );
	}

	g_free( pnParentName );

	return psRet;
}

/**
 * \brief Add preference
 * \param psProfile profile structure
 * \param nType preference type
 * \param pnName preference name
 * \return new preference pointer or NULL on error
 */
static struct sPref *addPref( struct sProfile *psProfile, int nType, const char *pnName ) {
	struct sPref *psParent;
	struct sPref *psMe;
	struct sPref *psSibling;
	char *pnMyName;

	psParent = findPrefParent( psProfile, pnName );
	if ( !psParent ) {
		return NULL;
	}

	pnMyName = getPathBaseName( pnName );

	for ( psSibling = psParent -> psFirstChild; psSibling != NULL; psSibling = psSibling -> psSibling ) {
		if ( !strcmp( psSibling -> pnName, pnMyName ) ) {
			g_free( pnMyName );
			return NULL;
		}
	}

	psMe = g_new0( struct sPref, 1 );
	psMe -> nType = nType;
	psMe -> pnName = pnMyName;

	psMe -> psParent = psParent;
	if ( psParent -> psFirstChild ) {
		for ( psSibling = psParent -> psFirstChild; psSibling -> psSibling; psSibling = psSibling -> psSibling );
		psSibling -> psSibling = psMe;
	} else {
		psParent -> psFirstChild = psMe;
	}

	g_hash_table_insert( psProfile -> psPrefsHash, g_strdup( pnName ), ( gpointer ) psMe );

	return psMe;
}

/**
 * \brief Add none type to preferences
 * \param psProfile profile structure
 * \param pnName name
 */
void prefsAddNone( struct sProfile *psProfile, const char *pnName ) {
	addPref( psProfile, PREF_NONE, pnName );
}

/**
 * \brief Add string preference
 * \param psProfile profile structure
 * \param pnName preference name
 * \param pnValue preference value
 */
static void prefsAddString( struct sProfile *psProfile, const char *pnName, const char *pnValue ) {
	struct sPref *psPref;

	if ( pnValue != NULL && !g_utf8_validate( pnValue, -1, NULL ) ) {
		Debug( KERN_DEBUG, "Cannot store invalid UTF8 for string pref %s\n", pnName );
		return;
	}

	psPref = addPref( psProfile, PREF_STRING, pnName );
	if ( psPref != NULL ) {
		psPref -> uValue.pnString = g_strdup( pnValue );
	}
}

/**
 * \brief Add path preference
 * \param psProfile profile structure
 * \param pnName name
 * \param pnValue value
 */
static void prefsAddPath( struct sProfile *psProfile, const char *pnName, const char *pnValue ) {
	struct sPref *psPref = addPref( psProfile, PREF_PATH, pnName );

	if ( psPref != NULL ) {
		psPref -> uValue.pnString = g_strdup( pnValue );
	}
}

/**
 * \brief Add integer preference
 * \param psProfile profile structure
 * \param pnName name
 * \paam nValue value
 */
static void prefsAddInt( struct sProfile *psProfile, const char *pnName, int nValue ) {
	struct sPref *psPref = addPref( psProfile, PREF_INT, pnName );

	if ( psPref != NULL ) {
		psPref -> uValue.nInteger = nValue;
	}
}

/**
 * \brief Add bool preference
 * \param psProfile profile structure
 * \param pnName preference name
 * \param bValue boolean value
 */
static void prefsAddBool( struct sProfile *psProfile, const char *pnName, gboolean bValue ) {
	struct sPref *psPref = addPref( psProfile, PREF_BOOLEAN, pnName );

	if ( psPref != NULL ) {
		psPref -> uValue.bBoolean = bValue;
	}
}

/**
 * \brief Set string preference to new value or add new one
 * \param psProfile profile structure
 * \param pnName preference name
 * \param pnValue value
 */
void prefsSetString( struct sProfile *psProfile, const char *pnName, const char *pnValue ) {
	struct sPref *psPref = findPref( psProfile, pnName );

	if ( pnValue != NULL && !g_utf8_validate( pnValue, -1, NULL ) ) {
		Debug( KERN_WARNING, "Cannot store invalid UTF8 for string %s\n", pnName );
		return;
	}

	if ( psPref != NULL ) {
		if ( psPref -> nType != PREF_STRING && psPref -> nType != PREF_PATH ) {
			Debug( KERN_WARNING, "%s not a string pref\n", pnName );
			return;
		}

		if ( ( pnValue && !psPref -> uValue.pnString ) ||
			( !pnValue && psPref -> uValue.pnString ) ||
			( pnValue && psPref -> uValue.pnString &&
				strcmp( psPref -> uValue.pnString, pnValue ) ) ) {
			g_free( psPref -> uValue.pnString );
			psPref -> uValue.pnString = g_strdup( pnValue );
		}
	} else {
		prefsAddString( psProfile, pnName, pnValue );
	}
}

/**
 * \brief Set path preference or add new one
 * \param psProfile profile structure
 * \param pnName preference name
 * \param pnValue value
 */
void prefsSetPath( struct sProfile *psProfile, const char *pnName, const char *pnValue ) {
	struct sPref *psPref = findPref( psProfile, pnName );

	if ( psPref ) {
		if ( psPref -> nType != PREF_PATH ) {
			Debug( KERN_WARNING, "%s not a path pref\n", pnName );
			return;
		}

		if ( ( pnValue && !psPref -> uValue.pnString ) ||
			( !pnValue && psPref -> uValue.pnString ) ||
			( pnValue && psPref -> uValue.pnString &&
			strcmp( psPref -> uValue.pnString, pnValue ) ) ) {
			g_free( psPref -> uValue.pnString );
			psPref -> uValue.pnString = g_strdup( pnValue );
		}
	} else {
		prefsAddPath( psProfile, pnName, pnValue );
	}
}

/**
 * \brief Set new integer value or add new one
 * \param psProfile profile structure
 * \param pnName preference name
 * \param nValue value
 */
void prefsSetInt( struct sProfile *psProfile, const char *pnName, int nValue ) {
	struct sPref *psPref = findPref( psProfile, pnName );

	if ( psPref ) {
		if ( psPref -> nType != PREF_INT ) {
			Debug( KERN_WARNING, "%s not an integer pref\n", pnName );
			return;
		}

		if ( psPref -> uValue.nInteger != nValue ) {
			psPref -> uValue.nInteger = nValue;
		}
	} else {
		prefsAddInt( psProfile, pnName, nValue );
	}
}

/**
 * \brief Set new bool value or add new one
 * \param psProfile profile structure
 * \param pnName preference name
 * \param bValue boolean value
 */
void prefsSetBool( struct sProfile *psProfile, const char *pnName, bool bValue ) {
	struct sPref *psPref = findPref( psProfile, pnName );

	if ( psPref ) {
		if ( psPref -> nType != PREF_BOOLEAN ) {
			Debug( KERN_WARNING, "%s not an integer pref\n", pnName );
			return;
		}

		if ( psPref -> uValue.bBoolean != bValue ) {
			psPref -> uValue.bBoolean = bValue;
		}
	} else {
		prefsAddBool( psProfile, pnName, bValue );
	}
}

/**
 * \brief Get string preference
 * \param psProfile profile structure
 * \param pnName prefence name
 * \return preference string value
 */
const char *prefsGetString( struct sProfile *psProfile, const char *pnName ) {
	struct sPref *psPref = findPref( psProfile, pnName );

	if ( !psPref ) {
		if ( psProfile != NULL ) {
			Debug( KERN_WARNING, "Unknown pref %s in profile %s\n", pnName, psProfile -> pnName );
		} else {
			Debug( KERN_WARNING, "No profile\n" );
		}
		return NULL;
	} else if ( psPref -> nType != PREF_STRING ) {
		Debug( KERN_WARNING, "%s not a string pref\n", pnName );
		return NULL;
	}

	return psPref -> uValue.pnString;
}

/**
 * \brief Get path preference
 * \param psProfile profile structure
 * \param pnName prefence name
 * \return preference path value
 */
const char *prefsGetPath( struct sProfile *psProfile, const char *pnName ) {
	struct sPref *psPref = findPref( psProfile, pnName );

	if ( !psPref ) {
		if ( psProfile != NULL ) {
			Debug( KERN_WARNING, "Unknown pref %s in profile %s\n", pnName, psProfile -> pnName );
		} else {
			Debug( KERN_WARNING, "No profile\n" );
		}
		return NULL;
	} else if ( psPref -> nType != PREF_PATH ) {
		Debug( KERN_WARNING, "%s not a path pref\n", pnName );
		return NULL;
	}

	return psPref -> uValue.pnString;
}

/**
 * \brief Get integer preference
 * \param psProfile profile structure
 * \param pnName preference name
 * \return preference integer value
 */
int prefsGetInt( struct sProfile *psProfile, const char *pnName ) {
	struct sPref *psPref = findPref( psProfile, pnName );

	if ( !psPref ) {
		Debug( KERN_WARNING, "Unknown pref %s\n", pnName );
		return 0;
	} else if ( psPref -> nType != PREF_INT ) {
		Debug( KERN_WARNING, "%s not a integer pref\n", pnName );
		return 0;
	}

	return psPref -> uValue.nInteger;
}

/**
 * \brief Get boolean preference
 * \param psProfile profile structure
 * \param pnName preference name
 * \return boolean preference value
 */
gboolean prefsGetBool( struct sProfile *psProfile, const char *pnName ) {
	struct sPref *psPref = findPref( psProfile, pnName );

	if ( !psPref ) {
		Debug( KERN_WARNING, "Unknown pref %s\n", pnName );
		return 0;
	} else if ( psPref -> nType != PREF_BOOLEAN ) {
		Debug( KERN_WARNING, "%s not a integer pref\n", pnName );
		return 0;
	}

	return psPref -> uValue.bBoolean;
}

/**
 * \brief Start element handler
 * \param psContect markup parse context
 * \param pnElementName element name
 * \param ppnAttributeNames pointer to attribute names
 * \param ppnAttributeValues pointer to attribute values
 * \param pUserData pointer to profile structure
 * \param ppsError pointer to GError
 */
static void configStartElementHandler( GMarkupParseContext *psContext, const gchar *pnElementName, const gchar **ppnAttributeNames, const gchar **ppnAttributeValues, gpointer pUserData, GError **ppsError ) {
	int nIndex;
	int nPrefType = PREF_NONE;
	const char *pnPrefName = NULL, *pnPrefValue = NULL;
	GString *psPrefNameFull;
	GList *psTmp;
	struct sProfile *psProfile = pUserData;

	if ( strcmp( pnElementName, "pref" ) && strcmp( pnElementName, "item" ) ) {
		return;
	}

	for ( nIndex = 0; ppnAttributeNames[ nIndex ]; nIndex++ ) {
		if ( !strcmp( ppnAttributeNames[ nIndex ], "name" ) ) {
			pnPrefName = ppnAttributeValues[ nIndex ];
		} else if ( !strcmp( ppnAttributeNames[ nIndex ], "type" ) ) {
			if ( !strcmp( ppnAttributeValues[ nIndex ], "string" ) ) {
				nPrefType = PREF_STRING;
			} else if ( !strcmp( ppnAttributeValues[ nIndex ], "int" ) ) {
				nPrefType = PREF_INT;
			} else if ( !strcmp( ppnAttributeValues[ nIndex ], "bool" ) ) {
				nPrefType = PREF_BOOLEAN;
			} else if ( !strcmp( ppnAttributeValues[ nIndex ], "path" ) ) {
				nPrefType = PREF_PATH;
			} else {
				return;
			}
		} else if ( !strcmp( ppnAttributeNames[ nIndex ], "value" ) ) {
			pnPrefValue = ppnAttributeValues[ nIndex ];
		}
	}

	{
		char *pnDecoded;

		if ( !pnPrefName || !strcmp( pnPrefName, "/" ) ) {
			return;
		}

		psPrefNameFull = g_string_new( pnPrefName );

		for ( psTmp = psProfile -> psPrefsStack; psTmp != NULL; psTmp = psTmp -> next ) {
			psPrefNameFull = g_string_prepend_c( psPrefNameFull, '/' );
			psPrefNameFull = g_string_prepend( psPrefNameFull, psTmp -> data );
		}

		psPrefNameFull = g_string_prepend_c( psPrefNameFull, '/' );

		switch ( nPrefType ) {
			case PREF_NONE:
				prefsAddNone( psProfile, psPrefNameFull -> str );
				break;
			case PREF_STRING:
				prefsSetString( psProfile, psPrefNameFull -> str, pnPrefValue );
				break;
			case PREF_PATH:
				if ( pnPrefValue ) {
					pnDecoded = g_filename_from_utf8( pnPrefValue, -1, NULL, NULL, NULL );
					prefsSetPath( psProfile, psPrefNameFull -> str, pnDecoded );
					g_free( pnDecoded );
				} else {
					prefsSetPath( psProfile, psPrefNameFull -> str, NULL );
				}
				break;
			case PREF_INT:
				prefsSetInt( psProfile, psPrefNameFull -> str, atoi( pnPrefValue ) );
				break;
			case PREF_BOOLEAN:
				prefsSetBool( psProfile, psPrefNameFull -> str, atoi( pnPrefValue ) );
				break;
		}

		psProfile -> psPrefsStack = g_list_prepend( psProfile -> psPrefsStack, g_strdup( pnPrefName ) );
		g_string_free( psPrefNameFull, TRUE );
	}
}

/**
 * \brief end element handler
 * \param psContext markup parse context
 * \param pnElementName element name
 * \param pUserData user data pointer
 * \param GError pointer to GError
 */
static void configEndElementHandler( GMarkupParseContext *psContext, const gchar *pnElementName, gpointer pUserData, GError **ppnError ) {
	struct sProfile *psProfile = pUserData;

	if ( psProfile -> psPrefsStack && !strcmp( pnElementName, "pref" ) ) {
		g_free( psProfile -> psPrefsStack -> data );
		psProfile -> psPrefsStack = g_list_delete_link( psProfile -> psPrefsStack, psProfile -> psPrefsStack );
	}
}

/** Configuration markup parser */
static GMarkupParser sConfigParser = {
	configStartElementHandler,
	configEndElementHandler,
	NULL,
	NULL,
	NULL
};

/**
 * \brief Convert one preference to xmlnode
 * \param psParent parent xml node
 * \param psPref preference pointer
 */
static void pref_to_xmlnode( xmlnode *psParent, struct sPref *psPref ) {
	xmlnode *psNode = NULL;
	struct sPref *psChild;
	char anBuf[ 20 ];

	psNode = xmlnode_new_child( psParent, "pref" );
	xmlnode_set_attrib( psNode, "name", psPref -> pnName );

	if ( psPref -> nType == PREF_STRING ) {
		xmlnode_set_attrib( psNode, "type", "string" );
		xmlnode_set_attrib( psNode, "value", psPref -> uValue.pnString ? psPref -> uValue.pnString : "" );
	} else if ( psPref -> nType == PREF_INT ) {
		xmlnode_set_attrib( psNode, "type", "int" );
		snprintf( anBuf, sizeof( anBuf ), "%d", psPref -> uValue.nInteger );
		xmlnode_set_attrib( psNode, "value", anBuf );
	} else if ( psPref -> nType == PREF_BOOLEAN ) {
		xmlnode_set_attrib( psNode, "type", "bool" );
		snprintf( anBuf, sizeof( anBuf ), "%d", psPref -> uValue.bBoolean );
		xmlnode_set_attrib( psNode, "value", anBuf );
	} else if ( psPref -> nType == PREF_PATH ) {
		gchar *pnEncoded = g_filename_to_utf8( psPref -> uValue.pnString ? psPref -> uValue.pnString : "", -1, NULL, NULL, NULL );
		xmlnode_set_attrib( psNode, "type", "path" );
		xmlnode_set_attrib( psNode, "value", pnEncoded );
		g_free( pnEncoded );
	}

	for ( psChild = psPref -> psFirstChild; psChild != NULL; psChild = psChild -> psSibling ) {
		pref_to_xmlnode( psNode, psChild );
	}
}

/**
 * \brief Convert all preferences to xmlnodes
 * \param psProfile profile structure
 * \return xmlnode pointer or NULL on error
 */
static xmlnode *prefsToXmlnode( struct sProfile *psProfile ) {
	xmlnode *psNode;
	struct sPref *psPref, *psChild;

	psPref = &psProfile -> sPrefs;

	psNode = xmlnode_new( "pref" );
	xmlnode_set_attrib( psNode, "version", "1" );
	xmlnode_set_attrib( psNode, "name", "/" );

	for ( psChild = psPref -> psFirstChild; psChild != NULL; psChild = psChild -> psSibling ) {
		pref_to_xmlnode( psNode, psChild );
	}

	return psNode;
}

/**
 * \brief Set standard preferences
 * \param psProfile profile we want to save to
 * \return error code
 */
static int setPreferences( struct sProfile *psProfile ) {
	gint nIndex;

	if ( psProfile == NULL ) {
		Debug( KERN_WARNING, "WARNING: psProfile == NULL!!!\n" );
	}

	if ( psProfile -> psPrefsHash != NULL ) {
		g_hash_table_destroy( psProfile -> psPrefsHash );
		psProfile -> psPrefsHash = NULL;
	}

	psProfile -> psPrefsHash = g_hash_table_new_full( g_str_hash, g_str_equal, g_free, NULL );

	prefsAddNone( psProfile, "/ffgtk" );

	prefsAddNone( psProfile, "/ffgtk/router" );
	prefsSetString( psProfile, "/ffgtk/router/host", "" );
	prefsSetString( psProfile, "/ffgtk/router/password", "" );
	prefsSetInt( psProfile, "/ffgtk/router/type", 0 );
	prefsSetString( psProfile, "/ffgtk/router/lineaccesscode", "" );
	prefsSetString( psProfile, "/ffgtk/router/countrycode", "49" );
	prefsSetString( psProfile, "/ffgtk/router/areacode", "" );
	prefsSetString( psProfile, "/ffgtk/router/nationalprefix", "0" );
	prefsSetString( psProfile, "/ffgtk/router/internationalprefix", "00" );
	prefsSetString( psProfile, "/ffgtk/router/firmware", "" );
	prefsSetInt( psProfile, "/ffgtk/router/ports", PORT_ANALOG1 | PORT_ANALOG2 | PORT_ANALOG3 | PORT_ISDNALL |
				PORT_ISDN1 | PORT_ISDN2 | PORT_ISDN3 | PORT_ISDN4 |
				PORT_ISDN5 | PORT_ISDN6 | PORT_ISDN7 | PORT_ISDN8 |
				PORT_DECT1 | PORT_DECT2 | PORT_DECT3 | PORT_DECT4 | PORT_DECT5 | PORT_DECT6 );
	prefsSetString( psProfile, "/ffgtk/router/fon", "Fon 1" );
	routerSetStandardPorts( psProfile );

	prefsAddNone( psProfile, "/ffgtk/router/msn0" );
	prefsSetString( psProfile, "/ffgtk/router/msn0/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/msn0/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/msn1" );
	prefsSetString( psProfile, "/ffgtk/router/msn1/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/msn1/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/msn2" );
	prefsSetString( psProfile, "/ffgtk/router/msn2/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/msn2/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/msn3" );
	prefsSetString( psProfile, "/ffgtk/router/msn3/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/msn3/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/msn4" );
	prefsSetString( psProfile, "/ffgtk/router/msn4/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/msn4/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/msn5" );
	prefsSetString( psProfile, "/ffgtk/router/msn5/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/msn5/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/msn6" );
	prefsSetString( psProfile, "/ffgtk/router/msn6/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/msn6/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/msn7" );
	prefsSetString( psProfile, "/ffgtk/router/msn7/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/msn7/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/msn8" );
	prefsSetString( psProfile, "/ffgtk/router/msn8/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/msn8/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/msn9" );
	prefsSetString( psProfile, "/ffgtk/router/msn9/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/msn9/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/pots" );
	prefsSetString( psProfile, "/ffgtk/router/pots/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/pots/flags", 0x07 );
	prefsSetString( psProfile, "/ffgtk/router/mac", "" );

	/* IP-Phone */
	prefsAddNone( psProfile, "/ffgtk/router/ip0" );
	prefsSetString( psProfile, "/ffgtk/router/ip0/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/ip0/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/ip1" );
	prefsSetString( psProfile, "/ffgtk/router/ip1/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/ip1/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/ip2" );
	prefsSetString( psProfile, "/ffgtk/router/ip2/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/ip2/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/ip3" );
	prefsSetString( psProfile, "/ffgtk/router/ip3/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/ip3/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/ip4" );
	prefsSetString( psProfile, "/ffgtk/router/ip4/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/ip4/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/ip5" );
	prefsSetString( psProfile, "/ffgtk/router/ip5/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/ip5/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/ip6" );
	prefsSetString( psProfile, "/ffgtk/router/ip6/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/ip6/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/ip7" );
	prefsSetString( psProfile, "/ffgtk/router/ip7/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/ip7/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/ip8" );
	prefsSetString( psProfile, "/ffgtk/router/ip8/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/ip8/flags", 0x07 );
	prefsAddNone( psProfile, "/ffgtk/router/ip9" );
	prefsSetString( psProfile, "/ffgtk/router/ip9/number", "" );
	prefsSetInt( psProfile, "/ffgtk/router/ip9/flags", 0x07 );

	for ( nIndex = 0; nIndex < 18; nIndex++ ) {
		gchar *pnTmp = g_strdup_printf( "/ffgtk/router/sip%d", nIndex );
		prefsAddNone( psProfile, pnTmp );
		g_free( pnTmp );
		pnTmp = g_strdup_printf( "/ffgtk/router/sip%d/number", nIndex );
		prefsSetString( psProfile, pnTmp, "" );
		g_free( pnTmp );
		pnTmp = g_strdup_printf( "/ffgtk/router/sip%d/flags", nIndex );
		prefsSetInt( psProfile, pnTmp, 0x07 );
		g_free( pnTmp );
	}

	prefsAddNone( psProfile, "/ffgtk/callmonitor" );

	prefsSetBool( psProfile, "/ffgtk/callmonitor/type_visible", TRUE );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/datetime_visible", TRUE );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/name_visible", TRUE );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/company_visible", FALSE );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/number_visible", TRUE );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/localname_visible", TRUE );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/localnumber_visible", TRUE );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/duration_visible", TRUE );

	prefsSetInt( psProfile, "/ffgtk/callmonitor/displaytime", 10 );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/systemnotification", FALSE );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/ffgtknotification", TRUE );

	prefsSetBool( psProfile, "/ffgtk/callmonitor/in", TRUE );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/out", TRUE );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/playsound", TRUE );
	gchar *pnTmp = g_strdup_printf( "%s/call_in.wav", getDirectory( PKGDATADIR ) );
	prefsSetPath( psProfile, "/ffgtk/callmonitor/incoming_sound", pnTmp  );
	g_free( pnTmp );
	pnTmp = g_strdup_printf( "%s/call_out.wav", getDirectory( PKGDATADIR ) );
	prefsSetPath( psProfile, "/ffgtk/callmonitor/outgoing_sound", pnTmp  );
	g_free( pnTmp );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/mute", FALSE );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/lookup", TRUE );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/loadonstartup", FALSE );
	prefsSetInt( psProfile, "/ffgtk/callmonitor/width", 880 );
	prefsSetInt( psProfile, "/ffgtk/callmonitor/height", 450 );
	prefsSetInt( psProfile, "/ffgtk/callmonitor/paned_position", 180 );
	prefsSetInt( psProfile, "/ffgtk/callmonitor/popupposition", 2 );
	prefsSetBool( psProfile, "/ffgtk/callmonitor/replaceload", TRUE );

	prefsAddNone( psProfile, "/ffgtk/addressbook" );
	prefsSetInt( psProfile, "/ffgtk/addressbook/numberformat", 0);

	prefsAddNone( psProfile, "/ffgtk/voicebox" );
	prefsAddString( psProfile, "/ffgtk/voicebox/user", "ftpuser" );
	prefsAddString( psProfile, "/ffgtk/voicebox/password", "" );

	prefsAddNone( psProfile, "/ffgtk/softphone" );
	prefsAddString( psProfile, "/ffgtk/softphone/msn", "" );
	prefsSetInt( psProfile, "/ffgtk/softphone/controller", 0 );

	prefsAddNone( psProfile, "/ffgtk/dtmf" );
	prefsAddString( psProfile, "/ffgtk/dtmf/caller", "" );
	prefsAddString( psProfile, "/ffgtk/dtmf/called", "" );
	prefsAddString( psProfile, "/ffgtk/dtmf/code", "" );
	prefsAddString( psProfile, "/ffgtk/dtmf/execute", "" );

	prefsAddNone( psProfile, "/ffgtk/fax" );
	prefsSetString( psProfile, "/ffgtk/fax/sender", "ffgtk" );
	prefsSetString( psProfile, "/ffgtk/fax/faxid", "+00 0000 000000" );
	prefsSetString( psProfile, "/ffgtk/fax/msn", "0000000" );
	prefsSetInt( psProfile, "/ffgtk/fax/controller", 0 );
	prefsSetInt( psProfile, "/ffgtk/fax/bitrate", 2 );
	prefsSetBool( psProfile, "/ffgtk/fax/accept", FALSE );
	prefsSetBool( psProfile, "/ffgtk/fax/report", FALSE );
	prefsSetBool( psProfile, "/ffgtk/fax/ecm", TRUE );
	prefsSetPath( psProfile, "/ffgtk/fax/reportdir", "" );

	prefsAddNone( psProfile, "/ffgtk/actions" );

	prefsAddNone( psProfile, "/ffgtk/plugins" );
	prefsSetString( psProfile, "/ffgtk/plugins/lookup", "" );
	prefsSetString( psProfile, "/ffgtk/plugins/book", "" );
	prefsSetString( psProfile, "/ffgtk/plugins/fax", "" );
	prefsSetString( psProfile, "/ffgtk/plugins/password", "" );

	prefsAddNone( psProfile, "/ffgtk/misc" );
	prefsAddInt( psProfile, "/ffgtk/misc/toolbarstyle", 0 );
	prefsAddInt( psProfile, "/ffgtk/misc/password_type", 0 );

	prefsAddNone( psProfile, "/ffgtk/debug" );
	prefsAddInt( psProfile, "/ffgtk/debug/ffgtk", 3 );
	prefsAddInt( psProfile, "/ffgtk/debug/fax", 0 );

	prefsAddNone( psProfile, "/plugins" );

	return 0;
}

/**
 * \brief Load preferences
 * \param psProfile profile structure
 * \return error code
 */
int LoadPreferences( struct sProfile *psProfile ) {
	gchar *pnContents;
	gsize nLength;
	GError *psError = NULL;
	GMarkupParseContext *psContext;

	if ( psProfile == NULL ) {
		Debug( KERN_DEBUG, "Could not find profile\n" );
		return 0;
	}
	setPreferences( psProfile );

	if ( !g_file_get_contents( psProfile -> pnFile, &pnContents, &nLength, &psError ) ) {
		Debug( KERN_DEBUG, "Could not read prefs (maybe it does not exist yet, ignore it): %s\n", psError -> message );
		g_error_free( psError );
		return -1;
	}

	psContext = g_markup_parse_context_new( &sConfigParser, 0, psProfile, NULL );
	if ( !g_markup_parse_context_parse( psContext, pnContents, nLength, NULL ) ) {
		Debug( KERN_WARNING, "Error parsing context\n" );
		g_markup_parse_context_free( psContext );
		g_free( pnContents );
		return -2;
	}

	if ( !g_markup_parse_context_end_parse( psContext, NULL ) ) {
		Debug( KERN_WARNING, "Error parsing config\n" );
		g_markup_parse_context_free( psContext );
	
		g_free( pnContents );
		return -3;
	}

	g_markup_parse_context_free( psContext ),
	g_free( pnContents );

	return 0;
}

/**
 * \brief Save preferences to disk
 * \param psProfile profile structure
 */
void SavePreferences( struct sProfile *psProfile ) {
	xmlnode *psNode;
	char *pnData;

	psNode = prefsToXmlnode( psProfile );
	if ( psNode != NULL ) {
		pnData = xmlnode_to_formatted_str( psNode, NULL );
		if ( pnData != NULL ) {
			saveData( psProfile -> pnFile, pnData, -1 );
			g_free( pnData );
		}
		xmlnode_free( psNode );
	}
}
