/** @file
 * @brief HID report descriptor - stream source instance
 *
 * Copyright (C) 2010 Nikolai Kondrashov
 *
 * This file is part of hidrd.
 *
 * Hidrd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Hidrd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with hidrd; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @author Nikolai Kondrashov <spbnick@gmail.com>
 *
 * @(#) $Id: inst.c 437 2010-05-24 12:29:10Z spb_nick $
 */

#include <stdio.h>
#include "hidrd/cfg.h"
#include "hidrd/opt/spec_list.h"
#include "hidrd/strm/src/inst.h"


bool
hidrd_src_valid(const hidrd_src *src)
{
    return src != NULL &&
           hidrd_src_type_valid(src->type) &&
           (src->buf != NULL || src->size == 0) &&
           (src->type->valid == NULL || (*src->type->valid)(src));
}


bool
hidrd_src_error(const hidrd_src *src)
{
    assert(hidrd_src_valid(src));

    return src->error;
}


char *
hidrd_src_errmsg(const hidrd_src *src)
{
    assert(hidrd_src_valid(src));
    return (*src->type->errmsg)(src);
}


/**
 * Allocate (an uninitialized, but zeroed) source instance of specified
 * type and set the type field.
 *
 * @param type  Source type to allocate instance of.
 *
 * @return Uninitialized instance of the specified source type, or NULL if
 *         failed to allocate memory.
 */
static hidrd_src *
hidrd_src_alloc(const hidrd_src_type *type)
{
    hidrd_src *src;

    assert(hidrd_src_type_valid(type));

    src = calloc(1, type->size);
    if (src != NULL)
        src->type = type;

    return src;
}


/**
 * Initialize source instance (va_list version).
 *
 * @param src   Source instance to initialize.
 * @param perr  Location for a dynamically allocated error message pointer,
 *              in case the initialization failed, or for a dynamically
 *              allocated empty string otherwise; could be NULL.
 * @param buf   Source buffer pointer.
 * @param size  Source buffer size.
 * @param ap    Source type-specific arguments.
 *
 * @return True if initialization succeeded, false otherwise.
 */
static bool
hidrd_src_initv(hidrd_src      *src,
                char          **perr,
                const void     *buf,
                size_t          size,
                va_list         ap)
{
    assert(src != NULL);
    assert(hidrd_src_type_valid(src->type));
    assert(buf != NULL || size == 0);

    src->buf    = buf;
    src->size   = size;
    src->error  = false;

    if (src->type->initv != NULL)
    {
        if (!(*src->type->initv)(src, perr, ap))
            return false;
    }
    else if (perr != NULL)
        *perr = strdup("");

    assert(hidrd_src_valid(src));

    return true;
}


/**
 * Initialize source instance.
 *
 * @param src   Source instance to initialize.
 * @param perr  Location for a dynamically allocated error message pointer,
 *              in case the initialization failed, or for a dynamically
 *              allocated empty string otherwise; could be NULL.
 * @param buf   Source buffer pointer.
 * @param size  Source buffer size.
 * @param ...   Source type-specific arguments.
 *
 * @return True if initialization succeeded, false otherwise.
 */
static bool
hidrd_src_init(hidrd_src   *src,
               char       **perr,
               const void  *buf,
               size_t       size,
               ...)
{
    bool    result;
    va_list ap;

    va_start(ap, size);
    result = hidrd_src_initv(src, perr, buf, size, ap);
    va_end(ap);

    return result;
}


#ifdef HIDRD_WITH_OPT
/**
 * Initialize source instance with an option string, formatted using
 * sprintf.
 *
 * @param src       Source instance to initialize.
 * @param perr      Location for a dynamically allocated error message
 *                  pointer, in case the initialization failed, or for a
 *                  dynamically allocated empty string otherwise; could be
 *                  NULL.
 * @param buf       Source buffer pointer.
 * @param size      Source buffer size.
 * @param opts_fmt  Option format string: each option is a name/value pair
 *                  separated by equals sign, with surrounding space
 *                  removed; options are separated by comma.
 * @param ...       Option format arguments.
 *
 * @return True if initialization succeeded, false otherwise.
 */
static bool
hidrd_src_init_optsf(hidrd_src     *src,
                     char         **perr,
                     const void    *buf,
                     size_t         size,
                     const char    *opts_fmt,
                     ...)
                     __attribute__((format(printf, 5, 6)));
static bool
hidrd_src_init_optsf(hidrd_src     *src,
                     char         **perr,
                     const void    *buf,
                     size_t         size,
                     const char    *opts_fmt,
                     ...)
{
    static const hidrd_opt_spec empty_spec_list[] = {{.name = NULL}};

    va_list                 ap;
    bool                    result      = false;
    char                   *opts_buf    = NULL;
    const hidrd_opt_spec   *spec_list;
    hidrd_opt              *opt_list    = NULL;

    assert(src != NULL);
    assert(hidrd_src_type_valid(src->type));
    assert(buf != NULL || size == 0);
    assert(opts_fmt != NULL);

    va_start(ap, opts_fmt);

    /* Retrieve option specification list */
    spec_list = src->type->opts_spec;
    if (spec_list == NULL)
        spec_list = empty_spec_list;

    /* Format option string */
    if (vasprintf(&opts_buf, opts_fmt, ap) < 0)
    {
        if (perr != NULL)
            *perr = strdup("failed to format options string");
        goto cleanup;
    }

    /* Parse option list */
    opt_list = hidrd_opt_list_parse(spec_list, opts_buf);
    if (opt_list == NULL)
    {
        if (perr != NULL)
            *perr = strdup("failed to parse options string");
        goto cleanup;
    }

    /* If there is init_opts member */
    if (src->type->init_opts != NULL)
    {
        /* Initialize with option list */
        src->buf    = buf;
        src->size   = size;
        src->error  = false;

        if (!(*src->type->init_opts)(src, perr, opt_list))
            goto cleanup;
    }
    else
    {
        /* Do the regular initialization */
        if (!hidrd_src_init(src, perr, buf, size))
            goto cleanup;
    }

    assert(hidrd_src_valid(src));

    result = true;

cleanup:

    free(opt_list);
    free(opts_buf);
    va_end(ap);

    return result;
}


/**
 * Initialize source instance with an option string.
 *
 * @param src   Source instance to initialize.
 * @param perr  Location for a dynamically allocated error message pointer,
 *              in case the initialization failed, or for a dynamically
 *              allocated empty string otherwise; could be NULL.
 * @param buf   Source buffer pointer.
 * @param size  Source buffer size.
 * @param opts  Option string: each option is a name/value pair separated by
 *              equals sign, with surrounding space removed; options are
 *              separated by comma.
 *
 * @return True if initialization succeeded, false otherwise.
 */
static bool
hidrd_src_init_opts(hidrd_src      *src,
                    char          **perr,
                    const void     *buf,
                    size_t          size,
                    const char     *opts)
{
    return hidrd_src_init_optsf(src, perr, buf, size, "%s", opts);
}


hidrd_src *
hidrd_src_new_opts(const hidrd_src_type    *type,
                   char                   **perr,
                   const void              *buf,
                   size_t                   size,
                   const char              *opts)
{
    hidrd_src  *src;

    assert(opts != NULL);

    /* Allocate */
    src = hidrd_src_alloc(type);
    if (src == NULL)
    {
        if (perr != NULL)
            *perr = strdup("instance allocation failed");
        return NULL;
    }

    /* Initialize */
    if (!hidrd_src_init_opts(src, perr, buf, size, opts))
        return NULL;

    return src;
}
#endif /* HIDRD_WITH_OPT */


hidrd_src *
hidrd_src_new(const hidrd_src_type     *type,
              char                    **perr,
              const void               *buf,
              size_t                    size,
              ...)
{
    hidrd_src *src;
    bool        result;
    va_list     ap;

    /* Allocate */
    src = hidrd_src_alloc(type);
    if (src == NULL)
    {
        if (perr != NULL)
            *perr = strdup("instance allocation failed");
        return NULL;
    }

    /* Initialize */
    va_start(ap, size);
    result = hidrd_src_initv(src, perr, buf, size, ap);
    va_end(ap);
    if (!result)
        return NULL;

    return src;
}


size_t
hidrd_src_getpos(const hidrd_src *src)
{
    assert(hidrd_src_valid(src));

    return (src->type->getpos == NULL)
            ? 0
            : (*src->type->getpos)(src);
}


char *
hidrd_src_fmtpos(const hidrd_src *src, size_t pos)
{
    assert(hidrd_src_valid(src));

    return (src->type->fmtpos == NULL)
            ? strdup("UNKNOWN POSITION")
            : (*src->type->fmtpos)(src, pos);
}


const hidrd_item *
hidrd_src_get(hidrd_src *src)
{
    assert(hidrd_src_valid(src));

    return (*src->type->get)(src);
}


/**
 * Cleanup source instance - free any internal data, but don't free the
 * source itself.
 *
 * @param src  Source instance to cleanup.
 */
static void
hidrd_src_clnp(hidrd_src *src)
{
    assert(hidrd_src_valid(src));

    if (src->type->clnp != NULL)
        (*src->type->clnp)(src);
}


/**
 * Free source instance without freeing any internal data.
 *
 * @param src  Source instance to free.
 */
static void
hidrd_src_free(hidrd_src *src)
{
    if (src == NULL)
        return;

    assert(hidrd_src_type_valid(src->type));

    free(src);
}


void
hidrd_src_delete(hidrd_src *src)
{
    assert(src == NULL || hidrd_src_valid(src));
    if (src == NULL)
        return;
    hidrd_src_clnp(src);
    hidrd_src_free(src);
}


