//
//  ========================================================================
//  Copyright (c) 1995-2020 Mort Bay Consulting Pty Ltd and others.
//  ------------------------------------------------------------------------
//  All rights reserved. This program and the accompanying materials
//  are made available under the terms of the Eclipse Public License v1.0
//  and Apache License v2.0 which accompanies this distribution.
//
//      The Eclipse Public License is available at
//      http://www.eclipse.org/legal/epl-v10.html
//
//      The Apache License v2.0 is available at
//      http://www.opensource.org/licenses/apache2.0.php
//
//  You may elect to redistribute this code under either of these licenses.
//  ========================================================================
//

package org.eclipse.jetty.util.thread;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.eclipse.jetty.util.MultiException;
import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertTrue;

public class ThreadFactoryTest
{
    @Test
    public void testThreadFactory() throws Exception
    {
        ThreadGroup threadGroup = new ThreadGroup("my-group");
        MyThreadFactory threadFactory = new MyThreadFactory(threadGroup);

        QueuedThreadPool qtp = new QueuedThreadPool(200, 10, 2000, 0, null, threadGroup, threadFactory);
        try
        {
            qtp.start();

            int testThreads = 2000;
            CountDownLatch threadsLatch = new CountDownLatch(testThreads);
            MultiException mex = new MultiException();

            for (int i = 0; i < testThreads; i++)
            {
                qtp.execute(() ->
                {
                    try
                    {
                        TimeUnit.MILLISECONDS.sleep(ThreadLocalRandom.current().nextInt(20, 500));
                        Thread thread = Thread.currentThread();

                        if (!thread.getName().startsWith("My-"))
                        {
                            mex.add(new AssertionError("Thread " + thread.getName() + " does not start with 'My-'"));
                        }

                        if (!thread.getThreadGroup().getName().equalsIgnoreCase("my-group"))
                        {
                            mex.add(new AssertionError("Thread Group " + thread.getThreadGroup().getName() + " is not 'my-group'"));
                        }

                        threadsLatch.countDown();
                    }
                    catch (InterruptedException e)
                    {
                        e.printStackTrace();
                    }
                });
            }

            assertTrue(threadsLatch.await(5, TimeUnit.SECONDS), "Did not see all tasks finish");
            mex.ifExceptionThrow();
        }
        finally
        {
            qtp.stop();
        }
    }

    public static class MyThreadFactory implements ThreadFactory
    {
        private final ThreadGroup threadGroup;

        public MyThreadFactory(ThreadGroup threadGroup)
        {
            this.threadGroup = threadGroup;
        }

        @Override
        public Thread newThread(Runnable runnable)
        {
            Thread thread = new Thread(threadGroup, runnable);
            thread.setDaemon(false);
            thread.setPriority(Thread.MIN_PRIORITY);
            thread.setName("My-" + thread.getId());
            return thread;
        }
    }
}
