#    LyX::Client - a perl interface for communication with LyX
#    Copyright (C) 1999  Stefano Ghirlanda, stefano@zool.su.se
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

package LyX::Client;
use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

require Exporter;

@ISA= qw(Exporter);
@EXPORT=qw();
@EXPORT_OK = qw(lyxrc);
$VERSION = '0.01';

use LyX::Polite;
use Carp;

# Error "managing".
sub error {
    (my $self, $@) = @_;
    $@ = $self unless ref $self;
    return;
}


# Initialisation of the Lyx::Client object.
# It is passed the name of a program to communicate in behalf of
sub new {
    my $class = shift;
    my $self = {"program" => shift, # identifier for the client program
		"cleanup" => shift  # what to do when LyX exits
		    || sub {},      # default: do nothing
		"lyxin" => "",      # lyx input pipe
		"lyxout" => "",     # lyx output pipe
		"slave" => "",      # see DESTROY method
		};
    $self->{program} .= "-$$"; #unique identifier via pid
    
    bless $self, $class; #we're an object now!
    
    # get the pipe stem from the user's lyxrc file
    my $lyxpipe = $self->lyxrc('serverpipe');

    # check for existence of the lyxpipes
    return error $lyxpipe.".in not found" unless (-p $lyxpipe.".in");
    return error $lyxpipe.".out not found" unless (-p $lyxpipe.".out");

    # store pipe names for convenience
    $self->{lyxin} = $lyxpipe.".in";
    $self->{lyxout} = $lyxpipe.".out";

    # greet LyX via the server protocol
    my ($ok, $data) = $self->server("hello"); 
    return error "lyx does not answer" unless $ok;

    # everything ok
    return $self;
}


# Getting a definition from the user's lyxrc file.
# NOTE: the initial \ must *not* be given as a parameter
sub lyxrc {
    my ($self, $lookfor, $arg) = @_;
    ($arg = $lookfor, $lookfor = $self) unless ref $self;
    my $lyxrc = $ENV{HOME}."/.lyx/lyxrc";
    croak "$lyxrc does not exist!" unless -e $lyxrc; 
    open(LYXRC,"< $lyxrc") or croak "can't open $lyxrc: $!";

    my $line;
    my @found;
    while ($line=<LYXRC>) {
	chomp $line;
	push @found, $1 if $line =~ /^\\$lookfor\s+(.*)/;
    }
    close LYXRC or croak "can't close LYXRC: $!";
    return unless @found;
 
    unless (defined $arg) { # looking for: \command arg
	foreach my $item (@found) {
	    $item =~ s/^\"//;
	    $item =~ s/\"$//;
	}
    } else { # looking for: \command arg1 arg2
	my @candidates = @found;
	undef @found;
	foreach my $item (@candidates) {
	    if ($item =~ /\s*\"?$arg\"?\s*/) {
		$item = $`.$';
		$item =~ s/^\"//;
		$item =~ s/\"$//;
		push @found, $item;
	    }
	}
    }

    # client code might or might not expect 
    # that multiple values are found
    if (wantarray) {
	return @found;
    } elsif (@found > 1) {
	warn "found more than 1 $lookfor, returning first one only";
	return $found[0];
    } else {
	return $found[0];
    }
}


# Executing a lyx command and gathering its output.
sub command {
    my ($self, $command, $arg) = @_;
    croak "command() can only be called on onjects" unless ref $self;
    $arg = "" unless defined $arg;
    $command = "LYXCMD:".$self->{program}.":".$command.":".$arg."\n";
    return $self->communicate($command);
}


# Sending a lyx server message.
sub server {
    my ($self, $message) = @_;
    croak "server() can only be called on objects" unless ref $self;
    croak "server(): no message to send" unless defined $message;
    $message = "LYXSRV:".$self->{program}.":".$message."\n";
    return $self->communicate($message);
}


# Sends a line to lyx's input pipe and gathers lyx's output.
sub communicate {
    my ($self, $command) = @_;
    croak "communicate() can only be called on objects" unless ref $self;
    croak "communicate(): nothing to  send" unless defined $command;

    # preparing communication between 
    # parent and child processes
    pipe(READER, WRITER);
    my $old = select WRITER;
    $| = 1;
    select $old;

    if (my $pid = fork) {
	#parent process: sends a string to lyx,
	#then waits for child to give back lyx's output

	close WRITER or croak "communicate(): parent cannot close WRITER";

	# try to synchronize with the child:
	my $answer = <READER>;           # wait a "ready" signal
	select undef, undef, undef, .05; # wait some more...

	# hopefully the children is listening to LyX by now:
	# send command to LyX and wait for answer collected by children
	pipe_write $self->{lyxin}, $command;
	$answer = <READER>;

	# cleanup and return
	close READER or croak "parent: cannot close READER";
	waitpid($pid, 0);
	chomp $answer;
	my ($status, $data) = split /:/, $answer;
	return $self->server_message($answer) if $status eq "LYXSRV";
	return ($status, $data);

    } else {
	# child process: listens to LyX for a message intended
	# for $self->{program} and sends it to the parent 
 
	croak "cannot fork: $!" unless defined $pid;
	close READER or croak "child: cannot close READER";
	$self->{slave} = 1; # see DESTROY method
	
	my ($status, $client, $function, $data);
	my $message; # to send back to parent
	print WRITER "\n"; #send "ready" signal to parent
	
	while(1) { # start waiting for LyX messages

	    # get and pre-digest a line from LyX
	    my $answer = pipe_read $self->{lyxout};
	    chomp $answer;
	    ($status, $client, $function, $data) = split /:/, $answer;

	    # now manage various cases:

	    # server message, either general (*) or directed to us
	    if ($status eq "LYXSRV" && ($self->{program} =~ /$client/)) {
		$message = $answer;
		last;

	    # INFO or ERROR message directed to us
	    } elsif ($client eq $self->{program} ) {
		
		# paranoid check that LyX executed 
		# the function we asked for
		my (undef, undef, $request, undef) = split /:/, $command;
		if ($request ne $function) {
		    $message = "ERROR:different function";
		} else {
		    $message = "$status:";
		    $message .= $data if defined $data;
		}
		last;

	    # message for another client: put back in the pipe
	    # and wait until it is locked by someone else
	    } else {
		pipe_write $self->{lyxout}, $answer;
		wait_lock $self->{lyxout};
	    }
	}

	# report to parent and cleanup
	print WRITER "$message\n";
	close WRITER or croak "LyX::Client child cannot close WRITER";
	exit;
    }
}


# Wait for notify events from Lyx: pass a list of key sequences,
# the functions waits until one is NOTIFY'ed and returns it.
sub wait_event {
    my ($self, @events) = @_;
    croak "wait_event() can only be called on objects" unless ref $self;

    # check that the events are indeed \notified by Lyx
    my @bindings = $self->lyxrc("bind", "server-notify");
    foreach my $event (@events) {
	grep {$event eq $_} @bindings 
	    or return error "waiting for unnotified event: $event";
    }

    #now listening
    while (1) {
	my $line = pipe_read $self->{lyxout};
	foreach my $event (@events) {
	    if ($line =~ /^NOTIFY:$event/) {
		return $event;
	    } else {
		pipe_write $self->{lyxout}, $line;
		wait_lock $self->{lyxout}, .05; # give a chance to others
	    }
	}
    }
}


# Disconnetting from the lyx server.
sub DESTROY {
    my $self = shift;

    # say goodbye to lyx, unless we're an auxiliary process
    # created by communicate()
    unless ($self->{slave}) {
	
        # code to display a bye message in the lyx minibuffer
    
	pipe_write $self->{lyxin}, "LYXSRV:".$self->{program}."bye\n";
    }
}


# Deal with server messages.
sub server_message {
    my ($self, $message) = @_;
    croak "server_message() can only be called on objects" unless ref $self;
    
    # LYXSRV and $client have already been matched
    # in communicate() but we might need them in the future...
    my (undef, undef, $data) = split /:/, $message;
    
    # bye: time to die
    if ($data eq "bye") {
	$self->{slave} = 1; # so that DESTROY is not called
	$self->{cleanup}->();
	exit;

    # hello: just return the message
    } elsif ($data eq "hello") {
	return ("LYXSRV", $data);

    # no other messages!
    } else {
	croak "server_message(): unknown message: $data";
    }
    return;
}

1;
__END__

=head1 NAME

LyX::Client -  a perl class for communication with LyX

=head1 SYNOPSIS

C<use LyX::Client;>

C<$lyx->E<gt>C<new("clientname");>

C<$lyx->E<gt>C<command("self-insert", "this should appear in LyX");>

C<($status, $data) = $lyx->E<gt>C<command("server-get-xy");>

=head1 DESCRIPTION

LyX::Client implements an object interface for communication with the LyX
document processor (see L<lyx>). It is indended to help the developers of LyX clients letting them focus on what the client should do and not on implementing the communication part of the client. You need to understand the LyX server
mechanism in order to understand this documentation. See the LyX on-line help,
chapter 4 of the Customization document.

All complexity of communication is hidden inside the class. The developer
simply creates a LyX::Client object and uses the command() and wait_event()
methods to interact with LyX. 

Clients created via the LyX::Client class are "polite" to each other,
as defined in the LyX Customization document. This means that more clients can
coexist without stepping on each other's feet (e.g. reading each other's messages or try to write at the same time in LyX input pipe).

Note that all functions I<must> be called on objects, apart from lyxrc(). A run-time fatal error will occurr otherwise.

=head1 METHODS

=over 4

=item new($clientname [, $cleanup])

Creates a LyX::Client object that communicates using $clientname as a client 
identifier. Actually, the process id of the program creating the Lyx::Client
is automatically appended to $clientname, so that more copies of the same 
program can act simulatneously as LyX clients without conflict.

The optional argument $cleanup, if specified, must be a reference to a function. This function will be called automatically if the client receives a "bye" message, signifying that LyX is closing. The function should do whatever is necessary for the client program to exit cleanly, e.g. remove temporary files. The function should I<not> exit itself, that will be done from inside the LyX::Client object. The default is to do nothing.

=item command($function [, $arg])

Executes $function as a LyX command, optionally with $arg as argument. Returns
a list of two scalars. The first is INFO if the command succeeded, it is ERROR
in case of failure. The second argument is the $function's return value in case
of success (may be empty), the error message in case of failure. For example:

my ($status, $position) = $lyx->command("server-get-xy");

if ($status eq "INFO") { print "cursor at $position\n"; }
else { printf "error in communication: $position\n"; }

I<NOTE:> command() does not return simply a true or false value because other things might happen, for example a NOTIFY event or something not yet specified in the LyX protocol. While these events are still unsupported inside command() (see S<MISSING FEATURES> below), simply returning true or false may lead to compatibility problems in the future.

=item wait_event(@events)

Waits for the LyX user to type one of the key sequences in @events. The
return value is the key sequence captured. The key sequences must have been
bound by the user to the 'server-notify' function. wait_event() checks whether the appropriate line is in the user's F<lyxrc> file. If it is not found, undef is returned. For example:

my $event = $lyx->wait_events("C-a", "C-b");

if ($event eq "C-a") { ... }
elsif ($event eq "C-b") { ... }
else die "waiting for an unsupported event";

=item lyxrc($string [, $arg])

Looks in the user's F<lyxrc> file for a setting for $string. $string must not contain the initial backslash. For example, inside the new() method, the name of the LyX pipes is looked up by calling:

C<my $lyxpipe = lyxrc("serverpipe")>

Initial and ending double quotes are stripped from values, e.g. a line

C<\serverpipe ".lyxpipe">

will result in $lyxpipe containing .lyxpipe in the call above.

Lines in the F<lyxrc> file can be more complex than a "\name value" pair,
namely there are lines of the form "\name value1 value2". This happens for
example with the \bind function:

C<\bind "C-a" "server-notify">

Looking up such lines is supported by the two-argument version of the function.
For example:

C<my $binding = lyxrc("bind", "C-a")>

will give $binding="server-notify" given the F<lyxrc> line above. To ask for C<lyxrc("bind", "server-notify")> also works, yielding $binding="C-a".

In the case of \bind lines, it is clear that in some cases more than one match
can be found in a F<lyxrc> file, for example if server-notify is bound to more
than one key sequence. In this case, lyxrc() returns an array containing all
of the matches. But, if it was called in a scalar context, it returns only the
first match found and issues a warning to STDERR.

lyxrc() is the only function that can be called by itself as well as on an object. However, the function is not exported by default, so that "use Lyx::Client 'lyxrc';" should be used in order to call it without qualifying it as LyX::Client::lyxrc.

=item server($message)

Works like command(), but sends to LyX a message using the LyX server
protocol. Currently, only the two messages "hello" and "bye" are defined
by the protocol, signalling to LyX that a client is ready for communication or that it is exiting. LyX::Client automatically sends these two messages upon
creation and destruction of objects, so that the user should not explicitely
call the server() method.

=item communicate($string)

This is a I<private> method used internally to the class. Users are explicitely
discouraged from using it. This entry is only for the curious.

=back

=head1 FILES

F<$HOME/.lyx/lyxrc>

=head1 ERROR MANAGING

When a function fails in a way that is considered non-fatal, it sets the $@ variable to a description of the error message and returns undef. This allows for the construct 

function() or die "error: $@";

in client code.

In the future, it will be desirable that those errors that can be corrected by end-user intervention be directed to the LyX GUI, e.g. in the minibuffer.

=head1 MISSING FEATURES

Listening to NOTIFY events is supported only inside the wait_event() function. It is reasonable to assume that client programs should be in this waiting state for most of the time, but if the user requests the intervention of a client via a NOTIFY'ed key sequence, the program will not respond correctly.
Client documentation should instruct the user not to invoke again a client until a previous requests has been terminated.

Suggestions to overcome this shortcoming are warmly welcomed by the S<AUTHOR>. I can think of two solutions right now. One is to return the NOTIFY message to the caller, leaving to it all dirty business and cleaning up of potentially half-executed chains of action. The other is to let the user register events to listen as long as functions to be executed when they are NOTIFY'ed (that is, callbacks). I welcome discussion on this point. The basic problem is that a masochistic user can send thousands of keystrokes. More realistically, a user may mistype commands or forget to have issued one.

The simplest, and perhaps most cost-effective solution is just to ask the user to wait upon receiving a NOTIFY event outside wait_event(). This could be handled automatically by LyX::Client without much trouble once code to write to the LyX minibuffer is written.

=head1 BUGS

If colons are present in the data returned by LyX, the data field will be managed incorrectly.

The code has been tested on Linux only, while LyX and perl run on other platforms as well. Please report any bugs or portability suggestions to the author (see S<AUTHOR> below). According to its documentation, perl "tries hard" to have a system-call interface which works the same on the supported architectures, so situation might not be so bad.

Note however that the LyX server does not allow multiple clients to run on OS/2, and it is currently broken on Digital Unix.

=head1 SEE ALSO

See L<lyx> and its online documentation, particularly chapter 4 of the Customization document.

See L<LyX::Polite> if interested in the internals.

=head1 AUTHOR

Please send to Stefano Ghirlanda, stefano@zool.su.se, any bug reports
and suggestions.

=head1 COPYRIGHT

Copyright 1999 Stefano Ghirlanda.

Redistributable under the terms of the General Public License, version 2.0
or later, at your choice.


