// Quantifier expression -*- c++ -*-

#ifdef __GNUC__
# pragma implementation
#endif // __GNUC__
#include "Quantifier.h"
#include "Expression.h"
#include "VariableDefinition.h"
#include "LeafValue.h"

/** @file Quantifier.C
 * Quantifier variables
 */

/* Copyright  1998-2001 Marko Mkel (msmakela@tcs.hut.fi).

   This file is part of MARIA, a reachability analyzer and model checker
   for high-level Petri nets.

   MARIA is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   MARIA is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

Quantifier::Quantifier (char* name, const class Type& type,
			class Expression* condition) :
  myVariable (new class VariableDefinition (name, type, true)),
  myCondition (condition)
{
  assert (!condition || condition->getType ()->getKind () == Type::tBool);
}

Quantifier::Quantifier (class VariableDefinition& variable,
			class Expression* condition) :
  myVariable (&variable), myCondition (condition)
{
  assert (!condition || condition->getType ()->getKind () == Type::tBool);
}

Quantifier::~Quantifier ()
{
  myCondition->destroy ();
  delete myVariable;
}

enum Quantifier::Result
Quantifier::isAcceptable (const class Valuation& valuation) const
{
  if (myCondition) {
    if (class Value* v = myCondition->eval (valuation)) {
      assert (v->getType ().getKind () == Type::tBool);
      enum Result result =
	bool (static_cast<const class LeafValue&>(*v)) ? pass : fail;
      delete v;
      return result;
    }
    else
      return undefined;
  }

  return pass;
}

QuantifierList::QuantifierList () :
  myList ()
{
}

QuantifierList::~QuantifierList ()
{
  for (iterator i = begin (); i != end (); i++)
    delete *i;
}

enum Quantifier::Result
QuantifierList::augment (class Valuation& valuation) const
{
  bool skipFirst = false;

  for (const_iterator i = begin (); i != end (); i++) {
    valuation.setValue ((*i)->getVariable (),
			(*i)->getVariable ().getType ().getFirstValue ());
    switch ((*i)->isAcceptable (valuation)) {
    case Quantifier::pass:
      break;
    case Quantifier::fail:
      skipFirst = true;
      break;
    case Quantifier::undefined:
      return Quantifier::undefined;
    }
  }

  return skipFirst ? Quantifier::fail : Quantifier::pass;
}

enum Quantifier::Result
QuantifierList::advance (class Valuation& valuation) const
{
  next:
  for (const_iterator i = begin (); i != end (); i++) {
    bool noWrap = valuation.increment ((*i)->getVariable ());
    switch ((*i)->isAcceptable (valuation)) {
    case Quantifier::pass:
      if (!noWrap)
	continue;
      else {
	const_iterator j = i;
	while (++j != end ()) {
	  switch ((*j)->isAcceptable (valuation)) {
	  case Quantifier::fail:
	    goto next;
	  case Quantifier::pass:
	    continue;
	  case Quantifier::undefined:
	    return Quantifier::undefined;
	  }
	}

	return Quantifier::pass;
      }
    case Quantifier::fail:
      if (noWrap)
	goto next;
      return Quantifier::fail;
    case Quantifier::undefined:
      return Quantifier::undefined;
    }
  }

  return Quantifier::fail;
}

void
QuantifierList::clear (class Valuation& valuation) const
{
  for (const_iterator i = begin (); i != end (); i++)
    if (!valuation.undefine ((*i)->getVariable ()))
      break;
}
