/* tcl_packet.c

   This is a Tcl interface to libwireshark. The bulk of the
   implementation lives in the libpackets module (packets/ directory).
   The "packet" and "ws" Tcl commands and the "packet" Tcl object type
   are implemented here.

   Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012, 2013 Eloy Paris

   Some code here has been borrowed from the Wireshark source code, and
   is copyright Gerald Combs and others.

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

#include <assert.h>
#include <epan/prefs.h>
#include <epan/column-utils.h>
#include <wiretap/pcap-encap.h> /* For wtap_pcap_encap_to_wtap_encap() */

#include "proto_hier_stats.h"

#include "util-tcl.h"
#include "dumbhex.h"

static int print_framenum = 0;
static int print_ts = 0;

/*
 * Internal representation of a packet.
 */
struct packet {
    struct payload bytes; /* Actual packet data and length */
    int dl_type; /* Data link type. See man pcap */
    struct pcap_pkthdr pkthdr;
};

/********************************************************************
 *		     The "packet" Tcl object type                   *
 ********************************************************************/

/* Forward declarations */
static void update_string(Tcl_Obj *);
static void free_int_rep(Tcl_Obj *);
static int set_from_any(Tcl_Interp *interp, Tcl_Obj *objPtr);
static void dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr);

Tcl_ObjType tclPacketType = {
    .name = "packet",
    .freeIntRepProc = &free_int_rep,
    .dupIntRepProc = &dup_int_rep,
    .updateStringProc = &update_string,
    .setFromAnyProc = &set_from_any
};

static int
set_from_any(Tcl_Interp *interp, Tcl_Obj *objPtr _U_)
{
    if (interp)
	Tcl_SetResult(interp,
		      "packet object internal representation can't be created "
		      "from string", TCL_VOLATILE);

    return TCL_ERROR;
}

static void
dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr)
{
    struct packet *old, *new;

    old = srcPtr->internalRep.otherValuePtr;
    new = (struct packet *) ckalloc(sizeof(*new) );

    *new = *old;

    new->bytes.data = (uint8_t *) ckalloc(new->bytes.len);
    memcpy(new->bytes.data, old->bytes.data, new->bytes.len);

    dupPtr->typePtr = &tclPacketType;
    dupPtr->internalRep.otherValuePtr = new;
}

static char *
get_line_buf(size_t len)
{
    static char *line_bufp = NULL;
    static size_t line_buf_len = 256;
    size_t new_line_buf_len;

    for (new_line_buf_len = line_buf_len; len > new_line_buf_len;
	 new_line_buf_len *= 2)
	;

    if (line_bufp == NULL) {
	line_buf_len = new_line_buf_len;
	line_bufp = g_malloc(line_buf_len + 1);
    } else {
	if (new_line_buf_len > line_buf_len) {
	    line_buf_len = new_line_buf_len;
	    line_bufp = g_realloc(line_bufp, line_buf_len + 1);
	}
    }

    return line_bufp;
}

static void
fill_in_pkthdr(struct wtap_pkthdr *phdr, const struct pcap_pkthdr *h,
	       int ll_type)
{
    memset(phdr, 0, sizeof *phdr);

    phdr->len = h->len;
    phdr->caplen = h->caplen;
    phdr->pkt_encap = wtap_pcap_encap_to_wtap_encap(ll_type);
    assert(phdr->pkt_encap != WTAP_ENCAP_UNKNOWN);
    phdr->ts.secs = h->ts.tv_sec;
    phdr->ts.nsecs = h->ts.tv_usec*1000;
    phdr->presence_flags = WTAP_HAS_TS | WTAP_HAS_CAP_LEN;
}

static void
fill_in_framedata(frame_data *fdata, struct wtap_pkthdr *phdr)
{
    frame_data_init(fdata, 0 /* num */, phdr, 0 /* offset */, 0 /* cum_bytes */);

    nstime_set_unset(&fdata->rel_ts);
}

/* From tshark.c:print_columns() */
static
char *columns_to_string(column_info *cinfo)
{
    char *line_bufp;
    int i;
    size_t buf_offset;
    size_t column_len;

    line_bufp = get_line_buf(256);
    buf_offset = 0;
    *line_bufp = '\0';

    for (i = 0; i < cinfo->num_cols; i++) {
	switch (cinfo->col_fmt[i]) {
	case COL_NUMBER:
	    if (!print_framenum)
		continue;

	    column_len = strlen(cinfo->col_data[i]);
	    if (column_len < 3)
		column_len = 3;
	    line_bufp = get_line_buf(buf_offset + column_len);
	    sprintf(line_bufp + buf_offset, "%3s", cinfo->col_data[i]);
	    break;

	case COL_CLS_TIME:
	case COL_REL_TIME:
	case COL_ABS_TIME:
	case COL_ABS_DATE_TIME: /* XXX - wider */
	    if (!print_ts)
		continue;

	    column_len = strlen(cinfo->col_data[i]);
	    if (column_len < 10)
		column_len = 10;
	    line_bufp = get_line_buf(buf_offset + column_len);
	    sprintf(line_bufp + buf_offset, "%10s", cinfo->col_data[i]);
	    break;

	case COL_DEF_SRC:
	case COL_RES_SRC:
	case COL_UNRES_SRC:
	case COL_DEF_DL_SRC:
	case COL_RES_DL_SRC:
	case COL_UNRES_DL_SRC:
	case COL_DEF_NET_SRC:
	case COL_RES_NET_SRC:
	case COL_UNRES_NET_SRC:
	    column_len = strlen(cinfo->col_data[i]);
	    if (column_len < 12)
		column_len = 12;
	    line_bufp = get_line_buf(buf_offset + column_len);
	    sprintf(line_bufp + buf_offset, "%12s", cinfo->col_data[i]);
	    break;

	case COL_DEF_DST:
	case COL_RES_DST:
	case COL_UNRES_DST:
	case COL_DEF_DL_DST:
	case COL_RES_DL_DST:
	case COL_UNRES_DL_DST:
	case COL_DEF_NET_DST:
	case COL_RES_NET_DST:
	case COL_UNRES_NET_DST:
	    column_len = strlen(cinfo->col_data[i]);
	    if (column_len < 12)
		column_len = 12;
	    line_bufp = get_line_buf(buf_offset + column_len);
	    sprintf(line_bufp + buf_offset, "%-12s", cinfo->col_data[i]);
	    break;

	default:
	    column_len = strlen(cinfo->col_data[i]);
	    line_bufp = get_line_buf(buf_offset + column_len);
	    strcat(line_bufp + buf_offset, cinfo->col_data[i]);
	    break;
	}

	buf_offset += column_len;

	if (i != cinfo->num_cols - 1) {
	    /*
	     * This isn't the last column, so we need to print a
	     * separator between this column and the next.
	     *
	     * If we printed a network source and are printing a
	     * network destination of the same type next, separate
	     * them with "->"; if we printed a network destination
	     * and are printing a network source of the same type
	     * next, separate them with "<-"; otherwise separate them
	     * with a space.
	     *
	     * We add enough space to the buffer for " <- " or " -> ",
	     * even if we're only adding " ".
	     */

	    line_bufp = get_line_buf(buf_offset + 4);

	    switch (cinfo->col_fmt[i]) {
	    case COL_DEF_SRC:
	    case COL_RES_SRC:
	    case COL_UNRES_SRC:
		switch (cinfo->col_fmt[i + 1]) {
		case COL_DEF_DST:
		case COL_RES_DST:
		case COL_UNRES_DST:
		    strcat(line_bufp + buf_offset, " -> ");
		    buf_offset += 4;
		    break;
		default:
		    strcat(line_bufp + buf_offset, " ");
		    buf_offset += 1;
		}
		break;

	    case COL_DEF_DL_SRC:
	    case COL_RES_DL_SRC:
	    case COL_UNRES_DL_SRC:
		switch (cinfo->col_fmt[i + 1]) {
		case COL_DEF_DL_DST:
		case COL_RES_DL_DST:
		case COL_UNRES_DL_DST:
		    strcat(line_bufp + buf_offset, " -> ");
		    buf_offset += 4;
		    break;
		default:
		    strcat(line_bufp + buf_offset, " ");
		    buf_offset += 1;
		}
		break;

	    case COL_DEF_NET_SRC:
	    case COL_RES_NET_SRC:
	    case COL_UNRES_NET_SRC:
		switch (cinfo->col_fmt[i + 1]) {
		case COL_DEF_NET_DST:
		case COL_RES_NET_DST:
		case COL_UNRES_NET_DST:
		    strcat(line_bufp + buf_offset, " -> ");
		    buf_offset += 4;
		    break;
		default:
		    strcat(line_bufp + buf_offset, " ");
		    buf_offset += 1;
		}
		break;

	    case COL_DEF_DST:
	    case COL_RES_DST:
	    case COL_UNRES_DST:
		switch (cinfo->col_fmt[i + 1]) {
		case COL_DEF_SRC:
		case COL_RES_SRC:
		case COL_UNRES_SRC:
		    strcat(line_bufp + buf_offset, " <- ");
		    buf_offset += 4;
		    break;
		default:
		    strcat(line_bufp + buf_offset, " ");
		    buf_offset += 1;
		}
		break;

	    case COL_DEF_DL_DST:
	    case COL_RES_DL_DST:
	    case COL_UNRES_DL_DST:
		switch (cinfo->col_fmt[i + 1]) {
		case COL_DEF_DL_SRC:
		case COL_RES_DL_SRC:
		case COL_UNRES_DL_SRC:
		    strcat(line_bufp + buf_offset, " <- ");
		    buf_offset += 4;
		    break;
		default:
		    strcat(line_bufp + buf_offset, " ");
		    buf_offset += 1;
		}
		break;

	    case COL_DEF_NET_DST:
	    case COL_RES_NET_DST:
	    case COL_UNRES_NET_DST:
		switch (cinfo->col_fmt[i + 1]) {
		case COL_DEF_NET_SRC:
		case COL_RES_NET_SRC:
		case COL_UNRES_NET_SRC:
		    strcat(line_bufp + buf_offset, " <- ");
		    buf_offset += 4;
		    break;
		default:
		    strcat(line_bufp + buf_offset, " ");
		    buf_offset += 1;
		}
		break;

	    default:
		strcat(line_bufp + buf_offset, " ");
		buf_offset += 1;
	    }
	}
    }

    return line_bufp;
}

static void
update_string(Tcl_Obj *packet_obj)
{
    size_t len;
    struct packet *packet;
    epan_dissect_t *edt;
    char *string_rep;
    frame_data fdata;
    struct wtap_pkthdr phdr;

    packet = packet_obj->internalRep.otherValuePtr;

    fill_in_pkthdr(&phdr, &packet->pkthdr, packet->dl_type);

    fill_in_framedata(&fdata, &phdr);

    edt = epan_dissect_new(0 /* create_proto_tree */,
                           0 /* proto_tree_visible */);

    col_custom_prime_edt(edt, &cinfo);

    epan_dissect_run(edt, &phdr, packet->bytes.data, &fdata, &cinfo);

    epan_dissect_fill_in_columns(edt, FALSE, TRUE);

    string_rep = columns_to_string(&cinfo);

    len = strlen(string_rep);

    packet_obj->bytes = ckalloc(len + 1);
    if (!packet_obj->bytes)
	return;

    strlcpy(packet_obj->bytes, string_rep, len + 1);

    packet_obj->length = len;

    epan_dissect_free(edt);
    frame_data_destroy(&fdata);
}

static void
free_int_rep(Tcl_Obj *obj)
{
    struct packet *packet;

    packet = obj->internalRep.otherValuePtr;
    ckfree( (char *) packet->bytes.data);
    ckfree( (char *) packet);
    /* XXX - need to free string representation? */
}

Tcl_Obj *
Tcl_NewPacketObj(const uint8_t *data, struct pcap_pkthdr *pkthdr, int dl_type)
{
    Tcl_Obj *obj;
    struct packet *packet;

    if (pkthdr->caplen != pkthdr->len)
	return NULL;

    obj = Tcl_NewObj();

    packet = (struct packet *) ckalloc(sizeof(struct packet) );

    packet->bytes.len = pkthdr->len;
    packet->bytes.data = (uint8_t *) ckalloc(pkthdr->len);
    memcpy(packet->bytes.data, data, pkthdr->len);

    packet->pkthdr = *pkthdr;

    packet->dl_type = dl_type;

    obj->bytes = NULL; /* Mark as invalid the string representation */
    obj->typePtr = &tclPacketType;
    obj->internalRep.otherValuePtr = packet;

    return obj;
}

int
Tcl_GetPacketFromObj(Tcl_Obj *obj, struct wtap_pkthdr *phdr,
		     frame_data *fdata, uint8_t **pkt)
{
    struct packet *packet;

    if (obj->typePtr != &tclPacketType)
	return -1;

    packet = obj->internalRep.otherValuePtr;

    fill_in_pkthdr(phdr, &packet->pkthdr, packet->dl_type);

    fill_in_framedata(fdata, phdr);

    *pkt = packet->bytes.data;

    return 0;
}

/********************************************************************
 *				packet                              *
 ********************************************************************/

static int
tcl_packet_dissect(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    int i, index;
    struct packet *packet;
    Tcl_Obj *obj;
    frame_data fdata;
    const char *ns_name = NULL;
    static const char *options[] = {
	"-namespace", NULL
    };
    enum options {
	OPT_NAMESPACE
    };
    struct wtap_pkthdr phdr;

    /*
     * Parse command-line arguments.
     */
    for (i = 1; i < objc && *Tcl_GetString(objv[i]) == '-'; i++) {
	if (Tcl_GetIndexFromObj(interp, objv[i], options, "option", 0, &index)
	    != TCL_OK)
	    return TCL_ERROR;

	switch ( (enum options) index) {
	case OPT_NAMESPACE:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-namespace <ns name>");
		return TCL_ERROR;
	    }

	    ns_name = Tcl_GetString(objv[i]);
	}
    }

    if (objc != i + 1) {
	nexp_error(interp, "usage: packet dissect [-namespace <namespace>] <packet variable>");
	return TCL_ERROR;
    }

    obj = Tcl_ObjGetVar2(interp, objv[i], NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclPacketType) {
	nexp_error(interp, "\"%s\" is not a packet object",
		   Tcl_GetString(objv[i]));
	return TCL_ERROR;
    }

    packet = obj->internalRep.otherValuePtr;

    fill_in_pkthdr(&phdr, &packet->pkthdr, packet->dl_type);

    fill_in_framedata(&fdata, &phdr);

    if (!ns_name)
	/*
	 * There was no "-namespace <ns name>" CLI option. Try to use
	 * the namespace name stored in the global "ns_name" Tcl variable.
	 */
	ns_name = Tcl_GetVar(interp, DISSECTION_NS_VARNAME, TCL_GLOBAL_ONLY);

    pkt_dissect_tcl(interp, packet->bytes.data, &phdr, &fdata, NULL,
		    ns_name, 1);

    frame_data_destroy(&fdata);

    return TCL_OK;
}

static int
tcl_packet_hash(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    struct packet *packet;
    Tcl_Obj *packet_obj, *barray_obj;
    GByteArray *hash;
    frame_data fdata;
    struct wtap_pkthdr phdr;

    if (objc != 2) {
	nexp_error(interp, "usage: packet hash <packet variable>");
	return TCL_ERROR;
    }

    packet_obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!packet_obj)
	return TCL_ERROR;

    if (packet_obj->typePtr != &tclPacketType) {
	nexp_error(interp, "\"%s\" is not a packet object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    packet = packet_obj->internalRep.otherValuePtr;

    fill_in_pkthdr(&phdr, &packet->pkthdr, packet->dl_type);

    fill_in_framedata(&fdata, &phdr);

    hash = pkt_hash(interp, packet->bytes.data, &phdr, &fdata);

    barray_obj = Tcl_NewBArrayObj(hash->data, hash->len);

    g_byte_array_free(hash, TRUE);

    Tcl_SetObjResult(interp, barray_obj);

    return TCL_OK;
}

static int
tcl_packet_data(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    struct packet *packet;
    Tcl_Obj *packet_obj, *barray_obj;
    char *s;
    size_t start, end, slice_len;

    if (objc != 2 && objc != 3) {
	nexp_error(interp,
		   "usage: packet data <packet variable> [slice spec]");
	return TCL_ERROR;
    }

    packet_obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!packet_obj)
	return TCL_ERROR;

    if (packet_obj->typePtr != &tclPacketType) {
	nexp_error(interp, "\"%s\" is not a packet object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    packet = packet_obj->internalRep.otherValuePtr;

    if (objc == 3) {
	/*
	 * We have a slice specification. Calculate start and length of the slice.
	 */

	start = strtoul(Tcl_GetString(objv[2]), NULL, 0);

	s = strchr(Tcl_GetString(objv[2]), ':');
	if (!s) {
	    s = strchr(Tcl_GetString(objv[2]), 'l');
	    if (!s) {
		nexp_error(interp, "invalid slice specification");
		return TCL_ERROR;
	    }

	    slice_len = strtoul(s + 1, NULL, 0);

	    if (start + slice_len > packet->bytes.len) {
		nexp_error(interp, "invalid start or end of packet data");
		return TCL_ERROR;
	    }
	} else {
	    end = strtoul(s + 1, NULL, 0);
	    if (end == 0)
		end = packet->bytes.len - 1;

	    if (start > end || end > packet->bytes.len - 1) {
		nexp_error(interp, "invalid start or end of packet data");
		return TCL_ERROR;
	    }

	    slice_len = end - start + 1;
	}
    } else {
	/*
	 * There is no slice specification; assume full length of packet
	 */

	start = 0;
	slice_len = packet->bytes.len;
    }

    barray_obj = Tcl_NewBArrayObj(packet->bytes.data + start, slice_len);
    Tcl_SetObjResult(interp, barray_obj);

    return TCL_OK;
}

static int
tcl_packet_dump(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    struct packet *packet;
    Tcl_Obj *packet_obj;

    if (objc != 2) {
	nexp_error(interp, "usage: packet dump <packet variable>");
	return TCL_ERROR;
    }

    packet_obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!packet_obj)
	return TCL_ERROR;

    if (packet_obj->typePtr != &tclPacketType) {
	nexp_error(interp, "\"%s\" is not a packet object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    packet = packet_obj->internalRep.otherValuePtr;

    printf("Packet length: %zu\n", packet->bytes.len);
    printf("Packet data:\n");
    dump(packet->bytes.data, packet->bytes.len);

    return TCL_OK;
}

static int
tcl_packet_ts(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    struct packet *packet;
    Tcl_Obj *packet_obj, *tv_obj;

    if (objc != 2) {
	nexp_error(interp, "usage: packet ts <packet variable>");
	return TCL_ERROR;
    }

    packet_obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!packet_obj)
	return TCL_ERROR;

    if (packet_obj->typePtr != &tclPacketType) {
	nexp_error(interp, "\"%s\" is not a packet object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    packet = packet_obj->internalRep.otherValuePtr;

    tv_obj = Tcl_NewTimevalObj(&packet->pkthdr.ts);
    Tcl_SetObjResult(interp, tv_obj);

    return TCL_OK;
}

static int
tcl_packet_tdelta(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *packet_obj, *double_obj;
    struct packet *packet;
    struct timeval *tva, *tvb;
    int d_sec, d_usec;
    double delta;

    if (objc != 3) {
	nexp_error(interp, "usage: packet tdelta <packet var1> <packet var2>");
	return TCL_ERROR;
    }

    packet_obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!packet_obj)
	return TCL_ERROR;

    if (packet_obj->typePtr != &tclPacketType) {
	nexp_error(interp, "\"%s\" is not a packet object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    packet = packet_obj->internalRep.otherValuePtr;
    tva = &packet->pkthdr.ts;

    packet_obj = Tcl_ObjGetVar2(interp, objv[2], NULL, TCL_LEAVE_ERR_MSG);
    if (!packet_obj)
	return TCL_ERROR;

    if (packet_obj->typePtr != &tclPacketType) {
	nexp_error(interp, "\"%s\" is not a packet object",
		   Tcl_GetString(objv[2]) );
	return TCL_ERROR;
    }

    packet = packet_obj->internalRep.otherValuePtr;
    tvb = &packet->pkthdr.ts;

    d_sec = tva->tv_sec - tvb->tv_sec;
    d_usec = tva->tv_usec - tvb->tv_usec;

    while (d_usec < 0) {
	d_usec += 1000000;
	d_sec--;
    }

    delta = d_sec + d_usec/1000000.0;

    double_obj = Tcl_NewDoubleObj(delta);
    Tcl_SetObjResult(interp, double_obj);

    return TCL_OK;
}

static int
tcl_packet_show(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    struct packet *packet;
    Tcl_Obj *obj;
    frame_data fdata;
    struct wtap_pkthdr phdr;
    int i, index, verbose;
    static const char *options[] = {
	"-verbose", NULL
    };
    enum options {
	OPT_VERBOSE
    };

    verbose = 0;

    for (i = 1; i < objc && *Tcl_GetString(objv[i]) == '-'; i++) {
	if (Tcl_GetIndexFromObj(interp, objv[i], options, "flag", 0, &index)
	    != TCL_OK)
	    return TCL_ERROR;

	switch ( (enum options) index) {
	case OPT_VERBOSE:
	    verbose = 1;
	    break;
	}
    }

    if (i != objc - 1) {
	nexp_error(interp, "usage: packet show [-verbose] <packet variable>");
	return TCL_ERROR;
    }

    obj = Tcl_ObjGetVar2(interp, objv[i], NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclPacketType) {
	nexp_error(interp, "\"%s\" is not a packet object",
		   Tcl_GetString(objv[i]));
	return TCL_ERROR;
    }

    packet = obj->internalRep.otherValuePtr;

    fill_in_pkthdr(&phdr, &packet->pkthdr, packet->dl_type);

    fill_in_framedata(&fdata, &phdr);

    pkt_dissect_show(packet->bytes.data, &phdr, &fdata, verbose);

    frame_data_destroy(&fdata);

    return TCL_OK;
}

static void
print_tree_node(GNode *node, gpointer data)
{
    ph_stats_node_t *stats;
    ph_stats_t *ps;
    gchar *proto;

    stats = node->data;
    ps = data;

    proto = g_strdup_printf("%s%s",
			    make_prefix(g_node_depth(node) - 2),
			    stats->hfinfo->abbrev);

    printf("%-14s  %6.2f%% %12d %12d %12d %12d\n",
	   proto,
	   100.0*(float) stats->num_pkts_total/(float) ps->tot_packets,
	   stats->num_pkts_total, stats->num_bytes_total,
	   stats->num_pkts_last, stats->num_bytes_last);

    g_free(proto);

    g_node_children_foreach(node, G_TRAVERSE_ALL, print_tree_node, ps);
}

static int
tcl_packet_stats(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    ph_stats_t *ps;
    Tcl_Obj *obj;
    gchar *result;

    ps = ph_stats_tcl(interp, objc, objv);
    if (ps == NULL)
	return TCL_ERROR;

    if (g_node_n_nodes(ps->stats_tree, G_TRAVERSE_ALL) > 1) {
/*
1             15            29           42         53             68      */
	printf("\
Protocol      %% Packets      Packets        Bytes  End Packets    End Bytes\n\
===========================================================================\n\
");
	g_node_children_foreach(ps->stats_tree, G_TRAVERSE_ALL, print_tree_node,
				ps);
    }

    result = g_strdup_printf("%u %s, %u bytes",
			     ps->tot_packets,
			     ps->tot_packets == 1 ? "packet" : "packets",
			     ps->tot_bytes);
    obj = Tcl_NewStringObj(result, -1);
    Tcl_SetObjResult(interp, obj);

    g_free(result);
    ph_stats_free(ps);

    return TCL_OK;
}

static int
tcl_packet_heximport(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;
    GByteArray *array;
    int i, index;
    int prefixes_to_ignore = 0, ignore_after_col = 0, dl_type = DLT_RAW;
    struct pcap_pkthdr pkthdr;
    static const char *options[] = {
	"-y", "-ignore-prefixes", "-ignore-after-column", NULL
    };
    enum options {
	OPT_DATALINK, OPT_IGNPREFIXES, OPT_IGNCOLUMNS
    };

    for (i = 1; i < objc && *Tcl_GetString(objv[i]) == '-'; i++) {
	if (Tcl_GetIndexFromObj(interp, objv[i], options, "flag", 0, &index)
	    != TCL_OK)
	    return TCL_ERROR;

	switch ( (enum options) index) {
	case OPT_IGNPREFIXES:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, NULL);
		return TCL_ERROR;
	    }

	    Tcl_GetIntFromObj(interp, objv[i], &prefixes_to_ignore);
	    break;
	case OPT_IGNCOLUMNS:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, NULL);
		return TCL_ERROR;
	    }

	    Tcl_GetIntFromObj(interp, objv[i], &ignore_after_col);
	    break;
	case OPT_DATALINK:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, NULL);
		return TCL_ERROR;
	    }

	    dl_type = pcap_datalink_name_to_val(Tcl_GetString(objv[i]) );
	    if (dl_type == -1) {
		nexp_error(interp, "Invalid data link name \'%s\'",
			   Tcl_GetString(objv[i]) );
		return TCL_ERROR;
	    }
	    break;
	}
    }

    if (i != objc - 1) {
	nexp_error(interp,
		   "usage: packet hex-import [-ignore-prefixes <prefixes>] "
		   "[-ignore-after-column <col>] [-y <dltype>] <hex. string>");
	return TCL_ERROR;
    }

    array = dh_parsehex(Tcl_GetString(objv[i]), prefixes_to_ignore,
			ignore_after_col);

    if (array->len == 0) {
	nexp_error(interp, "Hexadecimal string produced no data");
	g_byte_array_free(array, TRUE);
	return TCL_ERROR;
    }

    pkthdr.caplen = pkthdr.len = array->len;
    gettimeofday(&pkthdr.ts, NULL);

    obj = Tcl_NewPacketObj(array->data, &pkthdr, dl_type);

    g_byte_array_free(array, TRUE);

    if (!obj)
	return TCL_ERROR;
    
    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

static int
tcl_packet_new(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    char *pdudef = NULL; 
    char errbuf[PDU_ERRBUF_SIZE];
    const GNode *pdu;
    Tcl_Obj *obj;
    struct pcap_pkthdr pkthdr;
    size_t packet_size;
    uint8_t *packet;
    int i, index;
    int dl_type = DLT_RAW;
    static const char *options[] = {
	"-y", NULL
    };
    enum options {
	OPT_DATALINK
    };

    for (i = 1; i < objc && *Tcl_GetString(objv[i]) == '-'; i++) {
	if (Tcl_GetIndexFromObj(interp, objv[i], options, "flag", 0, &index)
	    != TCL_OK)
	    return TCL_ERROR;

	switch ( (enum options) index) {
	case OPT_DATALINK:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, NULL);
		return TCL_ERROR;
	    }

	    dl_type = pcap_datalink_name_to_val(Tcl_GetString(objv[i]) );
	    if (dl_type == -1) {
		nexp_error(interp, "Invalid data link name \'%s\'",
			   Tcl_GetString(objv[i]) );
		return TCL_ERROR;
	    }
	    break;
	}
    }

    pdudef = copy_objv(objc - i, &objv[i]);
    if (!pdudef) {
	nexp_error(interp, "usage: packet new [-y <dltype>] <PDU def>");
	return TCL_ERROR;
    }

    if ( (pdu = pb_parsedef(pdudef, errbuf) ) == NULL) {
	nexp_error(interp, "%s", errbuf);
	free(pdudef);
	return TCL_ERROR;
    }

    free(pdudef);

    packet_size = pb_len(pdu);
    packet = xmalloc(packet_size);

    pb_build(pdu, packet, NULL);

    pkthdr.caplen = pkthdr.len = packet_size;
    gettimeofday(&pkthdr.ts, NULL);

    obj = Tcl_NewPacketObj(packet, &pkthdr, dl_type);
    Tcl_SetObjResult(interp, obj);

    free(packet);

    return TCL_OK;
}

static int
tcl_packet_len(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    struct packet *packet;
    Tcl_Obj *packet_obj, *obj;

    if (objc != 2) {
	nexp_error(interp, "usage: packet length <packet variable>");
	return TCL_ERROR;
    }

    packet_obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!packet_obj)
	return TCL_ERROR;

    if (packet_obj->typePtr != &tclPacketType) {
	nexp_error(interp, "\"%s\" is not a packet object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    packet = packet_obj->internalRep.otherValuePtr;

    obj = Tcl_NewIntObj(packet->bytes.len);
    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

static int
NExp_PacketObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int objc,
		  Tcl_Obj * const *objv)
{
    int index, retval;
    static const char *subcmds[] = {
	"exists", "dissect", "hash", "data", "dump", "ts", "tdelta", "show",
	"stats", "hex-import", "new", "length", NULL
    };
    enum subcmds {
	SUBCMD_EXISTS, SUBCMD_DISSECT, SUBCMD_HASH, SUBCMD_DATA, SUBCMD_DUMP,
	SUBCMD_TS, SUBCMD_TDELTA, SUBCMD_SHOW, SUBCMD_STATS, SUBCMD_HEXIMPORT,
	SUBCMD_NEW, SUBCMD_LENGTH
    };

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "subcmd packetName ?arg ...?");
	return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[1], subcmds, "subcmd", 0, &index)
	!= TCL_OK)
	return TCL_ERROR;

    switch ( (enum subcmds) index) {
    case SUBCMD_EXISTS: {
	Tcl_Obj *obj;

	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "packetName");
	    return TCL_ERROR;
	}

	obj = Tcl_ObjGetVar2(interp, objv[2], NULL, TCL_LEAVE_ERR_MSG);
	if (!obj || obj->typePtr != &tclPacketType) {
	    /* Variable does not exist or it's not of our type */
	    obj = Tcl_NewBooleanObj(0);
	} else {
	    obj = Tcl_NewBooleanObj(1);
	}

	Tcl_SetObjResult(interp, obj);
	return TCL_OK;
    }
    case SUBCMD_DISSECT:
	retval = tcl_packet_dissect(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_HASH:
	retval = tcl_packet_hash(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_DATA:
	retval = tcl_packet_data(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_DUMP:
	retval = tcl_packet_dump(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_TS:
	retval = tcl_packet_ts(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_TDELTA:
	retval = tcl_packet_tdelta(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_SHOW:
	retval = tcl_packet_show(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_STATS:
	retval = tcl_packet_stats(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_HEXIMPORT:
	retval = tcl_packet_heximport(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_NEW:
	retval = tcl_packet_new(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_LENGTH:
	retval = tcl_packet_len(interp, objc - 1, &objv[1]);
	break;
    }

    return retval;
}

/********************************************************************
 *			    ws setprefs                             *
 ********************************************************************/

static int
tcl_ws_setprefs(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    int i;
    char *pref;

    if (objc == 1) {
	nexp_error(interp, "usage: ws setprefs <wireshark pref> [...]");
	return TCL_ERROR;
    }

    for (i = 1; i < objc; i++) {
	pref = Tcl_GetString(objv[i]);

	switch (prefs_set_pref(pref) ) {
	case PREFS_SET_OK:
	    break;
	case PREFS_SET_SYNTAX_ERR:
	    nexp_error(interp, "Invalid ws preference \"%s\"", pref);
	    return TCL_ERROR;
	case PREFS_SET_NO_SUCH_PREF:
	case PREFS_SET_OBSOLETE:
	    nexp_error(interp, "Unknown preference \"%s\"", pref);
	    return TCL_ERROR;
	}
    }

    prefs_apply_all();

    return TCL_OK;
}

static int
NExp_WSObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int objc,
	      Tcl_Obj *CONST objv[])
{
    int index, retval;
    static const char *subcmds[] = {
	"setprefs", NULL
    };
    enum subcmds {
	SUBCMD_SETPREF
    };

    if (objc == 1) {
	/* XXX - need to print out usage information */
	nexp_error(interp, "wrong # args");
	return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[1], subcmds, "subcmd", 0, &index)
	!= TCL_OK)
	return TCL_ERROR;

    switch ( (enum subcmds) index) {
    case SUBCMD_SETPREF:
	retval = tcl_ws_setprefs(interp, objc - 1, &objv[1]);
	break;
    }

    return retval;
}

static struct nexp_cmd_data cmd_data[] = {
    {"packet", NExp_PacketObjCmd, NULL, 0, 0},
    {"ws", NExp_WSObjCmd, NULL, 0, 0},

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_packet_cmd(Tcl_Interp *interp)
{
    Tcl_RegisterObjType(&tclPacketType);

    nexp_create_commands(interp, cmd_data);
}
