/***************************************************************************
 *   Copyright (C) 2000-2008 by Johan Maes                                 *
 *   on4qz@telenet.be                                                      *
 *   http://users.telenet.be/on4qz                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "global.h"
#include "dispatcher.h"
#include "soundio.h"
#include <QDebug>
#include <QApplication>
#include "configparams.h"

soundIO *sndIO;;

soundIO::soundIO()
{
	smplrate=11025;
  status=SND_IDLE;
	abortRun=FALSE;
  output=FALSE;
  handle=0;
}

void soundIO:: setSamplingrate(int sr)
{
	smplrate=sr;
	waveIn.setSamplingrate(sr);
	waveOut.setSamplingrate(sr);
}

void soundIO::run()
{
  addToLog("soundIO: run()",DBSOUND);
	while(!abortRun)
		{
      switch(status)
				{
					case SND_IDLE:
						msleep(500);
					break;
					case SND_STARTPLAYBACK:
						{
							output=TRUE;
              if(!play()==0) status=SND_PLAYBACK;
						}
					case SND_PLAYBACK:
						if(play()==0) msleep(100);
            break;
					case SND_RECORD:
            if(record()==0)
              {
                msleep(100);
              }
					break;
					case SND_DELAYED_STOP:
            addToLog("soundIO: SND_DELAYED_STOP",DBSOUND);
            if(play()==0)
							{
							 	close();
                status=SND_IDLE;
							}
					break;
					case SND_STOP:
            addToLog("soundIO: SND_STOP",DBSOUND);
						waveIn.close();
						waveOut.close();
						close();
            status=SND_IDLE;
					break;
          case SND_FILE_STOP:
            if (rxBuffer.count()<RXSTRIPE) status=SND_STOP;
            msleep(1);
          break;



					case SND_CALIBRATE:
            msleep(1);
					break;
				}
		}
  addToLog("soundIO: abortrun",DBSOUND);
	abortRun=FALSE;
	waveIn.close();
	waveOut.close();
  status=SND_IDLE;
	if ( handle != 0 ) snd_pcm_close ( handle );
	handle = 0;
}

int soundIO::play()
{
	int numFrames,err;
	if((numFrames=txBuffer.count())>1024) numFrames=1024;
	if(numFrames==0) return 0;
	err =  snd_pcm_writei ( handle, txBuffer.readPointer(), numFrames );
	if ( err >= 0 )
		{
			if(soundRoutingOutput==SNDFILEOUT)
				{
					if(storedFrames<=(ulong)waveLen*1048576L)
						{
					 		waveOut.write(txBuffer.readPointer(),err);
							storedFrames+=err;
						}
					else
						{
              status=SND_STOP;
						}
				}
			txBuffer.skip(err);
		}
	else
		{
			if ( err ==  -EAGAIN ) return -1;
			else if ( err == -EPIPE )
				{
					/* underrun */
					printf ( "Underun recovery for %d\n", numFrames);
					err = snd_pcm_prepare (handle);
					if ( err < 0 )
						{
							printf ( "Can't recover from underrun, prepare failed: %s\n", snd_strerror ( err ) );
						}
					err=numFrames;
				}
			else
				{
					printf ( "Unhandled error in play %s\n", snd_strerror ( err ) );
					snd_pcm_drop ( handle );
				}
		}
	return err;
}

int soundIO::record()
{
	int count;
	if(rxBuffer.spaceLeft()<1024) return 0;
	if(soundRoutingInput==SNDFILEIN)
		{
			count=waveIn.read(rxBuffer.writePointer(),1024);
			//delay to give realtime feeling
      msleep((200*count)/smplrate);
			if(count<=0)
				{
					waveIn.close();
          status=SND_FILE_STOP;
				}
			rxBuffer.advance(count);
//			addToLog(QString("soundIO: count %1").arg(count),DBSOUND);
			return count;
		}
	count = snd_pcm_readi( handle, rxBuffer.writePointer(),1024);
	if ( count < 0 )
		{
			if ( count != -EAGAIN )
				{
					if ( count == -EPIPE )
						{
							// Overrun
							snd_pcm_prepare ( handle );
							snd_pcm_start ( handle );
							printf ( "Overrun\n" );
						}
					else
						{
							snd_pcm_drop ( handle );
							printf ( "Overrun , reason %d\nStopping device", count );
						}
				}
      addToLog("soundIO: sound eagain",DBSOUND);
			return 0;
		}
	else
		{
			if(soundRoutingInput==SNDINTOFILE)
				{
					if(storedFrames<=(ulong)waveLen*1048576L)
						{
					 		waveOut.write(rxBuffer.writePointer(),count);
              addToLog("storing data",DBSOUND);
							storedFrames+=count;
						}
					else
						{
              status=SND_STOP;
						}
				}
			rxBuffer.advance(count);
		}
	return count;
}

/*!
Only used for clock measurement
*/
int soundIO::clkRxCheck()
{
	int count=snd_pcm_readi( handle, rxBuffer.writePointer(),1024);
	return count;
}


int soundIO::clkTxCheck()
{
	int count =  snd_pcm_writei ( handle, txBuffer.readPointer(), 1024 );
	return count;
}


void soundIO::stop()
{
  addToLog("soundIO: stop called",DBSOUND);
	if(output)
		{
      while (status==SND_STARTPLAYBACK) msleep(100);
      status=SND_DELAYED_STOP;
      addToLog("soundIO: delayed stop",DBSOUND);
		}
  else status=SND_STOP;
}

void soundIO::close()
{
  snd_pcm_state_t pcmState;
  addToLog("soundIO: close called",DBSOUND);
	if (output)
		{
			addToLog("close waiting",DBSOUND);
			msleep ( (10000*1024)/11025 );
			//snd_pcm_drain(handle);
			addToLog("drained",DBSOUND);
			//while((pcmState=snd_pcm_state(handle))!=SND_PCM_STATE_XRUN );
		}
	if ( handle != 0 ) snd_pcm_close ( handle );
	handle = 0;
  status=SND_IDLE;
  output = false;
	soundcardIdleEvent *ce = new soundcardIdleEvent();
	QApplication::postEvent(dispatchPtr, ce );
  addToLog("soundIO: soundcardIdleEvent",DBSOUND);
}


bool soundIO::setPlayBack(QString *errorstring,bool calibrating )
{
	int err;
	storedFrames=0;
	snd_pcm_hw_params_t *hwparams;
	snd_pcm_sw_params_t *swparams;
	snd_pcm_uframes_t period_size = 1024;
	err = snd_pcm_open ( &handle,outputAudioDevice.toLatin1().data(), SND_PCM_STREAM_PLAYBACK, 0);
	if ( err != 0 )
	{
		*errorstring = QString ( "Unable to open " ) +outputAudioDevice +" "+ QString ( snd_strerror ( err ) );
		return FALSE;
	}
	snd_pcm_hw_params_malloc ( &hwparams );
	snd_pcm_sw_params_malloc ( &swparams );

	/* Choose all parameters */
	err = snd_pcm_hw_params_any ( handle, hwparams );
	if ( err < 0 )
	{
		*errorstring = QString ( "Broken configuration : no configurations available: " ) + QString ( snd_strerror ( err ) );
		return false;
	}
	/* Set the interleaved read/write format */
	err = snd_pcm_hw_params_set_access ( handle, hwparams, SND_PCM_ACCESS_RW_INTERLEAVED );

	if ( err < 0 )
	{
		*errorstring = QString ( "Access type not available : " ) + QString ( snd_strerror ( err ) );
		return false;
	}
	/* Set the sample format */
	err = snd_pcm_hw_params_set_format ( handle, hwparams, SND_PCM_FORMAT_S16_LE );
	if ( err < 0 )
	{
		*errorstring = QString ( "Sample format Float not available : " ) + QString ( snd_strerror ( err ) );
		return false;
	}
	/* Set the count of channels */
	err = snd_pcm_hw_params_set_channels ( handle, hwparams, 1 );
	if ( err < 0 )
	{
		*errorstring = QString ( "Channels count 1 not available, please modify your plugin: " ) + QString ( snd_strerror ( err ) );
		return false;
	}

	err = snd_pcm_hw_params_set_rate ( handle, hwparams, smplrate, 0 );
	if ( err < 0 )
	{
		*errorstring = QString ( "Samplerate %1 not available for playing: " ).arg ( smplrate ) + QString ( snd_strerror ( err ) );
		return false;
	}
	int kkk = 0;
	err = snd_pcm_hw_params_set_period_size_near ( handle, hwparams, &period_size, &kkk );
	if ( err < 0 )
	{
		*errorstring = QString ( "Unable to set period size %d for play: " ).arg ( period_size ) + QString ( snd_strerror ( err ) );
		return false;
	}
	snd_pcm_uframes_t buffer_size = 9192;
	err = snd_pcm_hw_params_set_buffer_size_near ( handle, hwparams, &buffer_size );
	if ( err < 0 )
	{
		*errorstring = QString ( "Unable to set buffer size %i for play: " ).arg ( buffer_size ) + QString ( snd_strerror ( err ) );
		return false;
	}
	err = snd_pcm_hw_params ( handle, hwparams );
	if ( err < 0 )
	{
		*errorstring = QString ( "Unable to set hw params for input: " ) + QString ( snd_strerror ( err ) );
		return false;
	}
	snd_pcm_hw_params_free ( hwparams );
	/* Get the current swparams */
	err = snd_pcm_sw_params_current ( handle, swparams );
	if ( err < 0 )
	{
		*errorstring = QString ( "Unable to determine current swparams for play: " ) + QString ( snd_strerror ( err ) );
		return false;
	}
	err = snd_pcm_sw_params_set_start_threshold ( handle, swparams, 2048 );
	if ( err < 0 )
	{
		printf ( "Unable to set start threshold mode : %s\n", snd_strerror ( err ) );
		return false;
	}
	/* Write the parameters to the record/playback device */
	err = snd_pcm_sw_params ( handle, swparams );
	if ( err < 0 )
	{
		printf ( "Unable to set sw params for output: %s\n", snd_strerror ( err ) );
		return false;
	}

	if(soundRoutingOutput==SNDFILEOUT)
		{
			if(!waveOut.openFileForWrite("",TRUE))
				{
					if ( handle != 0 ) snd_pcm_close ( handle );
				 	return FALSE;
				}
		}
	txBuffer.reset();
  if(calibrating) status=SND_CALIBRATE;
  else	status=SND_STARTPLAYBACK;
	return true;
}

bool soundIO::setRecord(QString *errorstring,bool calibrating)
{
	
	int err;
	storedFrames=0;
	snd_pcm_hw_params_t *hwparams;
	snd_pcm_sw_params_t *swparams;
	snd_pcm_uframes_t period_size = 1024;
	if(soundRoutingInput==SNDFILEIN)
		{
			if(!waveIn.openFileForRead("",TRUE))
				{
					*errorstring=QString("File not opened");
					return FALSE;
				}
		}
	else
		{
			err = snd_pcm_open ( &handle,inputAudioDevice.toLatin1().data(), SND_PCM_STREAM_CAPTURE, 0 );
			if ( err != 0 )
				{
					*errorstring = QString ( "Unable to open " ) +inputAudioDevice +"\n"+ QString ( snd_strerror ( err ) );
					return false;
				}
			snd_pcm_hw_params_malloc ( &hwparams );
			snd_pcm_sw_params_malloc ( &swparams );

			/* Choose all parameters */
			err = snd_pcm_hw_params_any ( handle, hwparams );
			if ( err < 0 )
				{
					*errorstring = QString ( "Broken configuration : no configurations available: " ) + QString ( snd_strerror ( err ) );
					return false;
				}
			/* Set the interleaved read/write format */
			err = snd_pcm_hw_params_set_access ( handle, hwparams, SND_PCM_ACCESS_RW_INTERLEAVED );
			if ( err < 0 )
				{
					*errorstring = QString ( "Access type not available : " ) + QString ( snd_strerror ( err ) );
					return false;
				}
			/* Set the sample format */
			err = snd_pcm_hw_params_set_format ( handle, hwparams, SND_PCM_FORMAT_S16_LE );
			if ( err < 0 )
				{
					*errorstring = QString ( "Sample format Float not available : " ) + QString ( snd_strerror ( err ) );
					return false;
				}
			/* Set the count of channels */
			err = snd_pcm_hw_params_set_channels ( handle, hwparams, 1 );
			if ( err < 0 )
				{
					*errorstring = QString ( "Channels count 1 not available, please modify your plugin: " ) + QString ( snd_strerror ( err ) );
					return false;
				}
			err = snd_pcm_hw_params_set_rate ( handle, hwparams, smplrate, 0 );
			if ( err < 0 )
				{
					*errorstring = QString ( "Samplerate %1 not available for capture: " ).arg ( smplrate ) + QString ( snd_strerror ( err ) );
					return false;
				}
			int kkk = 0;
			err = snd_pcm_hw_params_set_period_size_near ( handle, hwparams, &period_size, &kkk );
			if ( err < 0 )
				{
					*errorstring = QString ( "Unable to set period size %d for play: " ).arg ( period_size ) + QString ( snd_strerror ( err ) );
					return false;
				}
			snd_pcm_uframes_t buffer_size = 9192;
			err = snd_pcm_hw_params_set_buffer_size_near ( handle, hwparams, &buffer_size );
			if ( err < 0 )
				{
					*errorstring = QString ( "Unable to set buffer size %i for capture: " ).arg ( buffer_size ) + QString ( snd_strerror ( err ) );
					return false;
				}
			err = snd_pcm_hw_params ( handle, hwparams );
			if ( err < 0 )
				{
					*errorstring = QString ( "Unable to set hw params for input: " ) + QString ( snd_strerror ( err ) );
					return false;
				}
			snd_pcm_hw_params_free ( hwparams );
			/* Get the current swparams */
			err = snd_pcm_sw_params_current ( handle, swparams );
			if ( err < 0 )
				{
					*errorstring = QString ( "Unable to determine current swparams for capture: " ) + QString ( snd_strerror ( err ) );
					return false;
				}
			err = snd_pcm_sw_params_set_start_threshold ( handle, swparams, 1 );
			if ( err < 0 )
				{
				*errorstring = QString ( "Unable to set start threshold for capture: " )  + QString ( snd_strerror ( err ) );
				return false;
				}
			/* Write the parameters to the record device */
			err = snd_pcm_sw_params ( handle, swparams );
			if ( err < 0 )
				{
					*errorstring = QString ( "Unable to set sw params for input: " ) + QString ( snd_strerror ( err ) );
					return false;
				}
			snd_pcm_sw_params_free ( swparams );

			err = snd_pcm_start ( handle );
			if ( err != 0 ) 
				{
					*errorstring = QString ( "Unable to start soundcard for receiving: " ) + QString ( snd_strerror ( err ) );
					return FALSE;;
				}
		}
	if(soundRoutingInput==SNDINTOFILE)
		{
			if(!waveOut.openFileForWrite("",TRUE))
				{
					if ( handle != 0 ) snd_pcm_close ( handle );
					return FALSE;
				}
		}
  if(calibrating) status=SND_CALIBRATE;
  else status=SND_RECORD;
	output = false;
	rxBuffer.reset();
	return TRUE;
}

void soundIO::logStatus()
{
  QString stat= "soundIO:: ";
  switch(status)
    {
      case SND_IDLE: stat+="IDLE";  break;
      case SND_STARTPLAYBACK: stat+="Start Playback";  break;
      case SND_PLAYBACK: stat+="Playback";  break;
      case SND_RECORD: stat+="Record";  break;
      case SND_DELAYED_STOP: stat+="Delayed Stop";  break;
      case SND_STOP: stat+="Stop";  break;
      case SND_FILE_STOP: stat+="File Stop";  break;
      case SND_CALIBRATE: stat+="Calibrate";  break;
    }
  addToLog(stat,DBDISPAT);
}


