# Copyright (c) 2015 RIPE NCC
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import mock
import unittest
from ripe.atlas.cousteau import Probe as CProbe
from ripe.atlas.tools.helpers.rendering import SaganSet
from ripe.atlas.tools.commands.report import Command
from ripe.atlas.tools.renderers.ssl_consistency import Renderer
from ..base import capture_sys_output


class TestSSLConsistency(unittest.TestCase):

    @classmethod
    def setUpClass(cls):

        cls.results = [
            {u'rt': 737.834, u'msm_id': 1443369, u'from': u'210.6.135.152', u'dst_name': u'torproject.org', u'af': 4, u'timestamp': 1392727294, u'fw': 4570, u'cert': [u'-----BEGIN CERTIFICATE-----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\n-----END CERTIFICATE-----', u'-----BEGIN CERTIFICATE-----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\nIb4p1I5eFdZCSucyb6Sxa1GDWL4/bcf72gMhy2oWGU4K8K2Eyl2Us1p292E=\n-----END CERTIFICATE-----'], u'method': u'SSL', u'prb_id': 1003, u'dst_port': u'443', u'dst_addr': u'86.59.30.40', u'ttc': 355.078, u'src_addr': u'192.168.1.182', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'ver': u'3.0'},
            {u'rt': 496.488, u'msm_id': 1443369, u'from': u'218.254.74.122', u'dst_name': u'torproject.org', u'af': 4, u'timestamp': 1392727295, u'fw': 4570, u'cert': [u'-----BEGIN CERTIFICATE-----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\n-----END CERTIFICATE-----', u'-----BEGIN CERTIFICATE-----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\nIb4p1I5eFdZCSucyb6Sxa1GDWL4/bcf72gMhy2oWGU4K8K2Eyl2Us1p292E=\n-----END CERTIFICATE-----'], u'method': u'SSL', u'prb_id': 1004, u'dst_port': u'443', u'dst_addr': u'38.229.72.14', u'ttc': 239.37, u'src_addr': u'192.168.1.100', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'ver': u'3.0'},
            {u'from': u'183.62.4.188', u'dst_name': u'torproject.org', u'err': u'connect: timeout', u'fw': 4600, u'timestamp': 1392727294, u'af': 4, u'dst_addr': u'216.234.179.13', u'prb_id': 1008, u'dst_port': u'443', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'msm_id': 1443369},
            {u'rt': 658.671, u'msm_id': 1443369, u'from': u'113.255.77.244', u'dst_name': u'torproject.org', u'af': 4, u'timestamp': 1392727293, u'fw': 4570, u'cert': [u'-----BEGIN CERTIFICATE-----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\n-----END CERTIFICATE-----', u'-----BEGIN CERTIFICATE-----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\nIb4p1I5eFdZCSucyb6Sxa1GDWL4/bcf72gMhy2oWGU4K8K2Eyl2Us1p292E=\n-----END CERTIFICATE-----'], u'method': u'SSL', u'prb_id': 1033, u'dst_port': u'443', u'dst_addr': u'82.195.75.101', u'ttc': 324.6, u'src_addr': u'192.168.0.183', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'ver': u'3.0'},
            {u'rt': 518.082, u'msm_id': 1443369, u'from': u'203.218.168.183', u'dst_name': u'torproject.org', u'af': 4, u'timestamp': 1392727295, u'fw': 4600, u'cert': [u'-----BEGIN CERTIFICATE-----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\n-----END CERTIFICATE-----', u'-----BEGIN CERTIFICATE-----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\nIb4p1I5eFdZCSucyb6Sxa1GDWL4/bcf72gMhy2oWGU4K8K2Eyl2Us1p292E=\n-----END CERTIFICATE-----'], u'method': u'SSL', u'prb_id': 1038, u'dst_port': u'443', u'dst_addr': u'38.229.72.14', u'ttc': 244.924, u'src_addr': u'192.168.95.248', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'ver': u'3.0'},
            {u'rt': 615.64, u'msm_id': 1443369, u'from': u'103.244.157.214', u'dst_name': u'torproject.org', u'af': 4, u'timestamp': 1392727295, u'fw': 4600, u'cert': [u'-----BEGIN CERTIFICATE-----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\n-----END CERTIFICATE-----', u'-----BEGIN CERTIFICATE-----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\nIb4p1I5eFdZCSucyb6Sxa1GDWL4/bcf72gMhy2oWGU4K8K2Eyl2Us1p292E=\n-----END CERTIFICATE-----'], u'method': u'SSL', u'prb_id': 1047, u'dst_port': u'443', u'dst_addr': u'93.95.227.222', u'ttc': 302.403, u'src_addr': u'192.168.6.137', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'ver': u'3.0'},
            {u'from': u'116.237.191.131', u'dst_name': u'torproject.org', u'err': u'connect: timeout', u'fw': 4600, u'timestamp': 1392727292, u'af': 4, u'dst_addr': u'209.145.54.50', u'prb_id': 11572, u'dst_port': u'443', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'msm_id': 1443369},
            {u'from': u'116.247.110.182', u'dst_name': u'torproject.org', u'err': u'connect: timeout', u'fw': 4580, u'timestamp': 1392727294, u'af': 4, u'dst_addr': u'202.181.7.85', u'prb_id': 11700, u'dst_port': u'443', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'msm_id': 1443369},
            {u'rt': 673.665, u'msm_id': 1443369, u'from': u'103.6.131.59', u'dst_name': u'torproject.org', u'af': 4, u'timestamp': 1392727294, u'fw': 4580, u'cert': [u'-----BEGIN CERTIFICATE-----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\n-----END CERTIFICATE-----', u'-----BEGIN CERTIFICATE-----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\nIb4p1I5eFdZCSucyb6Sxa1GDWL4/bcf72gMhy2oWGU4K8K2Eyl2Us1p292E=\n-----END CERTIFICATE-----'], u'method': u'SSL', u'prb_id': 12203, u'dst_port': u'443', u'dst_addr': u'86.59.30.40', u'ttc': 326.088, u'src_addr': u'103.6.131.59', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'ver': u'3.0'},
            {u'rt': 768.923, u'msm_id': 1443369, u'from': u'103.15.180.250', u'dst_name': u'torproject.org', u'af': 4, u'timestamp': 1392727296, u'fw': 4580, u'cert': [u'-----BEGIN CERTIFICATE-----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\n-----END CERTIFICATE-----', u'-----BEGIN CERTIFICATE-----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\nIb4p1I5eFdZCSucyb6Sxa1GDWL4/bcf72gMhy2oWGU4K8K2Eyl2Us1p292E=\n-----END CERTIFICATE-----'], u'method': u'SSL', u'prb_id': 12208, u'dst_port': u'443', u'dst_addr': u'93.95.227.222', u'ttc': 381.856, u'src_addr': u'103.15.180.250', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'ver': u'3.0'},
            {u'dnserr': u'non-recoverable failure in name resolution (1)', u'from': u'103.6.129.113', u'dst_name': u'torproject.org', u'fw': 4580, u'timestamp': 1392727293, u'prb_id': 12214, u'dst_port': u'443', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'msm_id': 1443369},
            {u'rt': 514.766, u'msm_id': 1443369, u'from': u'158.132.255.55', u'dst_name': u'torproject.org', u'af': 4, u'timestamp': 1392727293, u'fw': 4580, u'cert': [u'-----BEGIN CERTIFICATE-----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\n-----END CERTIFICATE-----', u'-----BEGIN CERTIFICATE-----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\nIb4p1I5eFdZCSucyb6Sxa1GDWL4/bcf72gMhy2oWGU4K8K2Eyl2Us1p292E=\n-----END CERTIFICATE-----'], u'method': u'SSL', u'prb_id': 13026, u'dst_port': u'443', u'dst_addr': u'93.95.227.222', u'ttc': 252.836, u'src_addr': u'192.168.1.185', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'ver': u'3.0'},
            {u'rt': 868.668, u'msm_id': 1443369, u'from': u'116.231.230.67', u'dst_name': u'torproject.org', u'af': 4, u'timestamp': 1392727294, u'fw': 4580, u'cert': [u'-----BEGIN CERTIFICATE-----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\n-----END CERTIFICATE-----', u'-----BEGIN CERTIFICATE-----\nMIIGWDCCBUCgAwIBAgIQCl8RTQNbF5EX0u/UA4w/OzANBgkqhkiG9w0BAQUFADBs\nMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3\nd3cuZGlnaWNlcnQuY29tMSswKQYDVQQDEyJEaWdpQ2VydCBIaWdoIEFzc3VyYW5j\nZSBFViBSb290IENBMB4XDTA4MDQwMjEyMDAwMFoXDTIyMDQwMzAwMDAwMFowZjEL\nMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3\nLmRpZ2ljZXJ0LmNvbTElMCMGA1UEAxMcRGlnaUNlcnQgSGlnaCBBc3N1cmFuY2Ug\nQ0EtMzCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAL9hCikQH17+NDdR\nCPge+yLtYb4LDXBMUGMmdRW5QYiXtvCgFbsIYOBC6AUpEIc2iihlqO8xB3RtNpcv\nKEZmBMcqeSZ6mdWOw21PoF6tvD2Rwll7XjZswFPPAAgyPhBkWBATaccM7pxCUQD5\nBUTuJM56H+2MEb0SqPMV9Bx6MWkBG6fmXcCabH4JnudSREoQOiPkm7YDr6ictFuf\n1EutkozOtREqqjcYjbTCuNhcBoz4/yO9NV7UfD5+gw6RlgWYw7If48hl66l7XaAs\nzPw82W3tzPpLQ4zJ1LilYRyyQLYoEt+5+F/+07LJ7z20Hkt8HEyZNp496+ynaF4d\n32duXvsCAwEAAaOCAvowggL2MA4GA1UdDwEB/wQEAwIBhjCCAcYGA1UdIASCAb0w\nggG5MIIBtQYLYIZIAYb9bAEDAAIwggGkMDoGCCsGAQUFBwIBFi5odHRwOi8vd3d3\nLmRpZ2ljZXJ0LmNvbS9zc2wtY3BzLXJlcG9zaXRvcnkuaHRtMIIBZAYIKwYBBQUH\nAgIwggFWHoIBUgBBAG4AeQAgAHUAcwBlACAAbwBmACAAdABoAGkAcwAgAEMAZQBy\nAHQAaQBmAGkAYwBhAHQAZQAgAGMAbwBuAHMAdABpAHQAdQB0AGUAcwAgAGEAYwBj\nAGUAcAB0AGEAbgBjAGUAIABvAGYAIAB0AGgAZQAgAEQAaQBnAGkAQwBlAHIAdAAg\nAEMAUAAvAEMAUABTACAAYQBuAGQAIAB0AGgAZQAgAFIAZQBsAHkAaQBuAGcAIABQ\nAGEAcgB0AHkAIABBAGcAcgBlAGUAbQBlAG4AdAAgAHcAaABpAGMAaAAgAGwAaQBt\nAGkAdAAgAGwAaQBhAGIAaQBsAGkAdAB5ACAAYQBuAGQAIABhAHIAZQAgAGkAbgBj\nAG8AcgBwAG8AcgBhAHQAZQBkACAAaABlAHIAZQBpAG4AIABiAHkAIAByAGUAZgBl\nAHIAZQBuAGMAZQAuMBIGA1UdEwEB/wQIMAYBAf8CAQAwNAYIKwYBBQUHAQEEKDAm\nMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wgY8GA1UdHwSB\nhzCBhDBAoD6gPIY6aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0SGln\naEFzc3VyYW5jZUVWUm9vdENBLmNybDBAoD6gPIY6aHR0cDovL2NybDQuZGlnaWNl\ncnQuY29tL0RpZ2lDZXJ0SGlnaEFzc3VyYW5jZUVWUm9vdENBLmNybDAfBgNVHSME\nGDAWgBSxPsNpA/i/RwHUmCYaCALvY2QrwzAdBgNVHQ4EFgQUUOpzidsp+xCPnuUB\nINTeeZlIg/cwDQYJKoZIhvcNAQEFBQADggEBAB7ipUiebNtTOA/vphoqrOIDQ+2a\nvD6OdRvw/S4iWawTwGHi5/rpmc2HCXVUKL9GYNy+USyS8xuRfDEIcOI3ucFbqL2j\nCwD7GhX9A61YasXHJJlIR0YxHpLvtF9ONMeQvzHB+LGEhtCcAarfilYGzjrpDq6X\ndF3XcZpCdF/ejUN83ulV7WkAywXgemFhM9EZTfkI7qA5xSU1tyvED7Ld8aW3DiTE\nJiiNeXf1L/BXunwH1OH8zVowV36GEEfdMR/X/KLCvzB8XSSq6PmuX2p0ws5rs0bY\nIb4p1I5eFdZCSucyb6Sxa1GDWL4/bcf72gMhy2oWGU4K8K2Eyl2Us1p292E=\n-----END CERTIFICATE-----'], u'method': u'SSL', u'prb_id': 13377, u'dst_port': u'443', u'dst_addr': u'82.195.75.101', u'ttc': 431.422, u'src_addr': u'10.0.10.39', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'ver': u'3.0'},
            {u'rt': 471.564, u'msm_id': 1443369, u'from': u'103.247.138.60', u'dst_name': u'torproject.org', u'af': 4, u'timestamp': 1392727304, u'fw': 4600, u'cert': [u'-----BEGIN CERTIFICATE-----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\n-----END CERTIFICATE-----', u'-----BEGIN CERTIFICATE-----\nMIIGWDCCBUCgAwIBAgIQCl8RTQNbF5EX0u/UA4w/OzANBgkqhkiG9w0BAQUFADBs\nMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3\nd3cuZGlnaWNlcnQuY29tMSswKQYDVQQDEyJEaWdpQ2VydCBIaWdoIEFzc3VyYW5j\nZSBFViBSb290IENBMB4XDTA4MDQwMjEyMDAwMFoXDTIyMDQwMzAwMDAwMFowZjEL\nMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3\nLmRpZ2ljZXJ0LmNvbTElMCMGA1UEAxMcRGlnaUNlcnQgSGlnaCBBc3N1cmFuY2Ug\nQ0EtMzCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAL9hCikQH17+NDdR\nCPge+yLtYb4LDXBMUGMmdRW5QYiXtvCgFbsIYOBC6AUpEIc2iihlqO8xB3RtNpcv\nKEZmBMcqeSZ6mdWOw21PoF6tvD2Rwll7XjZswFPPAAgyPhBkWBATaccM7pxCUQD5\nBUTuJM56H+2MEb0SqPMV9Bx6MWkBG6fmXcCabH4JnudSREoQOiPkm7YDr6ictFuf\n1EutkozOtREqqjcYjbTCuNhcBoz4/yO9NV7UfD5+gw6RlgWYw7If48hl66l7XaAs\nzPw82W3tzPpLQ4zJ1LilYRyyQLYoEt+5+F/+07LJ7z20Hkt8HEyZNp496+ynaF4d\n32duXvsCAwEAAaOCAvowggL2MA4GA1UdDwEB/wQEAwIBhjCCAcYGA1UdIASCAb0w\nggG5MIIBtQYLYIZIAYb9bAEDAAIwggGkMDoGCCsGAQUFBwIBFi5odHRwOi8vd3d3\nLmRpZ2ljZXJ0LmNvbS9zc2wtY3BzLXJlcG9zaXRvcnkuaHRtMIIBZAYIKwYBBQUH\nAgIwggFWHoIBUgBBAG4AeQAgAHUAcwBlACAAbwBmACAAdABoAGkAcwAgAEMAZQBy\nAHQAaQBmAGkAYwBhAHQAZQAgAGMAbwBuAHMAdABpAHQAdQB0AGUAcwAgAGEAYwBj\nAGUAcAB0AGEAbgBjAGUAIABvAGYAIAB0AGgAZQAgAEQAaQBnAGkAQwBlAHIAdAAg\nAEMAUAAvAEMAUABTACAAYQBuAGQAIAB0AGgAZQAgAFIAZQBsAHkAaQBuAGcAIABQ\nAGEAcgB0AHkAIABBAGcAcgBlAGUAbQBlAG4AdAAgAHcAaABpAGMAaAAgAGwAaQBt\nAGkAdAAgAGwAaQBhAGIAaQBsAGkAdAB5ACAAYQBuAGQAIABhAHIAZQAgAGkAbgBj\nAG8AcgBwAG8AcgBhAHQAZQBkACAAaABlAHIAZQBpAG4AIABiAHkAIAByAGUAZgBl\nAHIAZQBuAGMAZQAuMBIGA1UdEwEB/wQIMAYBAf8CAQAwNAYIKwYBBQUHAQEEKDAm\nMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wgY8GA1UdHwSB\nhzCBhDBAoD6gPIY6aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0SGln\naEFzc3VyYW5jZUVWUm9vdENBLmNybDBAoD6gPIY6aHR0cDovL2NybDQuZGlnaWNl\ncnQuY29tL0RpZ2lDZXJ0SGlnaEFzc3VyYW5jZUVWUm9vdENBLmNybDAfBgNVHSME\nGDAWgBSxPsNpA/i/RwHUmCYaCALvY2QrwzAdBgNVHQ4EFgQUUOpzidsp+xCPnuUB\nINTeeZlIg/cwDQYJKoZIhvcNAQEFBQADggEBAB7ipUiebNtTOA/vphoqrOIDQ+2a\nvD6OdRvw/S4iWawTwGHi5/rpmc2HCXVUKL9GYNy+USyS8xuRfDEIcOI3ucFbqL2j\nCwD7GhX9A61YasXHJJlIR0YxHpLvtF9ONMeQvzHB+LGEhtCcAarfilYGzjrpDq6X\ndF3XcZpCdF/ejUN83ulV7WkAywXgemFhM9EZTfkI7qA5xSU1tyvED7Ld8aW3DiTE\nJiiNeXf1L/BXunwH1OH8zVowV36GEEfdMR/X/KLCvzB8XSSq6PmuX2p0ws5rs0bY\nIb4p1I5eFdZCSucyb6Sxa1GDWL4/bcf72gMhy2oWGU4K8K2Eyl2Us1p292E=\n-----END CERTIFICATE-----'], u'method': u'SSL', u'prb_id': 1386, u'dst_port': u'443', u'dst_addr': u'38.229.72.16', u'ttc': 228.561, u'src_addr': u'103.247.138.60', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'ver': u'3.0'},
            {u'rt': 1161.992, u'msm_id': 1443369, u'from': u'193.162.21.10', u'dst_name': u'torproject.org', u'af': 4, u'timestamp': 1392727296, u'fw': 4600, u'cert': [u'-----BEGIN CERTIFICATE-----\nMIID/TCCAuWgAwIBAgIkNhPSsip1AJR2DEGtGdtSpPBb3qgBcuJXh2Gtln9+2aoo\nacTlMA0GCSqGSIb3DQEBBQUAMGAxCzAJBgNVBAYTAkVVMQswCQYDVQQIEwJFVTEL\nMAkGA1UEBxMCRVUxDDAKBgNVBAoTA0RTVjEMMAoGA1UECxMDRFNWMRswGQYDVQQD\nExJhcGFjLnByb3h5LmRzdi5jb20wHhcNMTMwMTI5MDAwMDAwWhcNMTYwNTAzMTIw\nMDAwWjByMQswCQYDVQQGEwJVUzEWMBQGA1UECBMNTWFzc2FjaHVzZXR0czEQMA4G\nA1UEBxMHV2FscG9sZTEeMBwGA1UEChMVVGhlIFRvciBQcm9qZWN0LCBJbmMuMRkw\nFwYDVQQDDBAqLnRvcnByb2plY3Qub3JnMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8A\nMIIBCgKCAQEAoGZlXATizI+cktME+tx2rjLndU2i0ATLGmzuhlxqwQ6RiFasuMkp\nfJcQp64tETk7mXxRWFGWVHiUFjCoeSQtO9Y0gxUk4mLTHPuh5nE8V3BzaDQ6sO9R\nPBXhVlTnbUatAitvgKM5PZnAgT+hi3ZcByGERBys/eITyCI/Uh9HZFW7Q2mUBcjr\n3/u36xeQdKMSH8hyq06I7HpRRwZpmv3T67jBQC6vo62rBnfGQ3vTcSVOmDoUOjD3\nGF1P/Ko7ch4Ao2P9/+T8/tNooJOPBK68VL168PTYRgG9Vyeh4hU2ipP/dd9DqoR+\nt3UwNO6o8jQmNSxdB/rjMcUtS7enSy2P7wIDAQABo4GMMIGJMB0GA1UdDgQWBBSx\n87Iq0fmAeNURYjYpnSG8riduZjArBgNVHREEJDAighAqLnRvcnByb2plY3Qub3Jn\ngg50b3Jwcm9qZWN0Lm9yZzAOBgNVHQ8BAf8EBAMCBaAwHQYDVR0lBBYwFAYIKwYB\nBQUHAwEGCCsGAQUFBwMCMAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQEFBQADggEB\nAHEvFwq+/VeBSa56orsVzq9Af34ZuTTST7zAUjfpDSYr/VjmHB4n0Uq80/cRWlm9\nDhRafM36IEk7tYpCN5tK4MYyMVg3FUESDsx37ZNDTpUiVSQPYjxpWQ7I/1kz7Sua\n6f/WQElnghZqa8Fj0s7zL/z6OxhDQMBjZvUzxtlsPhWjryo14COmQhsxW9b2uQA9\n14fPo4AcoFcmze/NAz4oMGyhsGKXwbkXFS5S2fW3ncSTxZBKJV+h8uslvB3owLvm\nhyqTwMk2iY8Z0jWDzR24jL76hPM3MWdFSSZzSkRLASEAcc/M/B50BcLfIFUSizNq\n4UoEZ4u6z2ZSChVJ3nHHaI8=\n-----END CERTIFICATE-----', u'-----BEGIN CERTIFICATE-----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\nnzi9veZZknp8KIDEW6A=\n-----END CERTIFICATE-----'], u'method': u'SSL', u'prb_id': 2844, u'dst_port': u'443', u'dst_addr': u'38.229.72.16', u'ttc': 3.042, u'src_addr': u'145.218.88.254', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'ver': u'3.0'},
            {u'from': u'210.22.91.254', u'dst_name': u'torproject.org', u'err': u'connect: timeout', u'fw': 4570, u'timestamp': 1392727297, u'af': 4, u'dst_addr': u'4.36.66.178', u'prb_id': 3386, u'dst_port': u'443', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'msm_id': 1443369},
            {u'from': u'180.154.141.170', u'dst_name': u'torproject.org', u'err': u'connect: timeout', u'fw': 4600, u'timestamp': 1392727296, u'af': 4, u'dst_addr': u'209.145.54.50', u'prb_id': 3909, u'dst_port': u'443', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'msm_id': 1443369},
            {u'from': u'180.154.141.170', u'dst_name': u'torproject.org', u'err': u'connect: timeout', u'fw': 4600, u'timestamp': 1392727295, u'af': 4, u'dst_addr': u'216.234.179.13', u'prb_id': 3914, u'dst_port': u'443', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'msm_id': 1443369},
            {u'rt': 420.604, u'msm_id': 1443369, u'from': u'27.50.33.14', u'dst_name': u'torproject.org', u'af': 4, u'timestamp': 1392727295, u'fw': 4600, u'cert': [u'-----BEGIN CERTIFICATE-----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\n-----END CERTIFICATE-----', u'-----BEGIN CERTIFICATE-----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\nIb4p1I5eFdZCSucyb6Sxa1GDWL4/bcf72gMhy2oWGU4K8K2Eyl2Us1p292E=\n-----END CERTIFICATE-----'], u'method': u'SSL', u'prb_id': 400, u'dst_port': u'443', u'dst_addr': u'38.229.72.14', u'ttc': 203.67, u'src_addr': u'27.50.33.14', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'ver': u'3.0'},
            {u'dnserr': u'non-recoverable failure in name resolution (1)', u'from': u'211.68.70.39', u'dst_name': u'torproject.org', u'fw': 4600, u'timestamp': 1392727294, u'prb_id': 483, u'dst_port': u'443', u'group_id': 1443369, u'type': u'sslcert', u'msm_name': u'SSLCert', u'msm_id': 1443369}
        ]

        cls.probes = {
            1003: CProbe(id=1003, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            1004: CProbe(id=1004, meta_data={
                "country_code": "DE", "asn_v4": 3333, "asn_v6": "4444"}),
            1008: CProbe(id=1008, meta_data={
                "country_code": "DE", "asn_v4": 3332, "asn_v6": "4444"}),
            1033: CProbe(id=1033, meta_data={
                "country_code": "NL", "asn_v4": 3333, "asn_v6": "4444"}),
            1038: CProbe(id=1038, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            1047: CProbe(id=1047, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            11672: CProbe(id=11572, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            11700: CProbe(id=11700, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            12203: CProbe(id=12203, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            12208: CProbe(id=12208, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            12214: CProbe(id=12214, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            13026: CProbe(id=13026, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            13377: CProbe(id=13377, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            1386: CProbe(id=1386, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            2844: CProbe(id=2844, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            3386: CProbe(id=3386, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            3909: CProbe(id=3909, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            3914: CProbe(id=3914, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            400: CProbe(id=400, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
            483: CProbe(id=483, meta_data={
                "country_code": "GR", "asn_v4": 3333, "asn_v6": "4444"}),
        }

    def test_additional(self):
        """Tests whole functionality of additional unit."""
        expected_output = (
            "Certificate:\n"
            "  Issuer: C=US, O=DigiCert Inc, CN=DigiCert High Assurance CA-3\n"
            "  Subject: C=US, O=The Tor Project, Inc., CN=*.torproject.org\n"
            "  SHA256 Fingerprint=36:13:D2:B2:2A:75:00:94:76:0C:41:AD:19:DB:52:A4:F0:5B:DE:A8:01:72:E2:57:87:61:AD:96:7F:7E:D9:AA\n\n"
            "  Seen by 11 probes\n\n"
            "Certificate:\n"
            "  Issuer: C=US, O=DigiCert Inc, CN=DigiCert High Assurance EV Root CA\n"
            "  Subject: C=US, O=DigiCert Inc, CN=DigiCert High Assurance CA-3\n"
            "  SHA256 Fingerprint=21:EB:37:AB:4C:F6:EF:89:65:EC:17:66:40:9C:A7:6B:8B:2E:03:F2:D1:A3:88:DF:73:42:08:E8:6D:EE:E6:79\n\n"
            "  Seen by 11 probes\n\n"
            "Certificate:\n"
            "  Issuer: C=None, O=None, CN=DSV Enterprise CA 1\n"
            "  Subject: C=EU, O=DSV, CN=apac.proxy.dsv.com\n"
            "  SHA256 Fingerprint=1A:B8:9E:ED:1B:DD:A0:E2:EA:67:89:C1:C5:4B:20:1C:49:9D:74:27:B0:5D:11:F2:9A:5F:C1:0D:F9:18:48:DA\n\n"
            "  Seen by 1 probe\n\n"
            "  Below the threshold (80%)\n"
            "  Probes that saw it: \n"
            "    ID: 2844, country code: GR, ASN (v4/v6): 3333/4444\n"
            "Certificate:\n"
            "  Issuer: C=EU, O=DSV, CN=apac.proxy.dsv.com\n"
            "  Subject: C=US, O=The Tor Project, Inc., CN=*.torproject.org\n"
            "  SHA256 Fingerprint=07:52:BE:65:72:BF:02:D4:C9:E2:93:09:A8:E0:BE:3A:EA:D4:30:41:B8:49:FA:C5:F2:12:33:07:37:57:EE:C7\n\n"
            "  Seen by 1 probe\n\n"
            "  Below the threshold (80%)\n"
            "  Probes that saw it: \n"
            "    ID: 2844, country code: GR, ASN (v4/v6): 3333/4444\n"
        )

        path = 'ripe.atlas.tools.helpers.rendering.Probe.get_many'
        with mock.patch(path) as mock_get_many:
            mock_get_many.return_value = self.probes.values()
            self.assertEquals(
                set(Renderer().additional(SaganSet(self.results)).split("\n")),
                set(expected_output.split("\n"))
            )

    def test_gather_unique_certs(self):
        """Test gathering of the unique certs in sagans set"""
        expected_certs = {
            '1A:B8:9E:ED:1B:DD:A0:E2:EA:67:89:C1:C5:4B:20:1C:49:9D:74:27:B0:5D:11:F2:9A:5F:C1:0D:F9:18:48:DA': {'cnt': 1, 'probes': [self.probes[2844]]},
            '36:13:D2:B2:2A:75:00:94:76:0C:41:AD:19:DB:52:A4:F0:5B:DE:A8:01:72:E2:57:87:61:AD:96:7F:7E:D9:AA': {
                'cnt': 11, 'probes': [self.probes[1003], self.probes[1004], self.probes[1033], self.probes[1038], self.probes[1047], self.probes[12203], self.probes[12208], self.probes[13026], self.probes[13377], self.probes[1386], self.probes[400]]
            },
            '21:EB:37:AB:4C:F6:EF:89:65:EC:17:66:40:9C:A7:6B:8B:2E:03:F2:D1:A3:88:DF:73:42:08:E8:6D:EE:E6:79': {
                'cnt': 11, 'probes': [self.probes[1003], self.probes[1004], self.probes[1033], self.probes[1038], self.probes[1047], self.probes[12203], self.probes[12208], self.probes[13026], self.probes[13377], self.probes[1386], self.probes[400]]
            },
            '07:52:BE:65:72:BF:02:D4:C9:E2:93:09:A8:E0:BE:3A:EA:D4:30:41:B8:49:FA:C5:F2:12:33:07:37:57:EE:C7': {'cnt': 1, 'probes': [self.probes[2844]]}
        }

        path = 'ripe.atlas.tools.helpers.rendering.Probe.get_many'
        with mock.patch(path) as mock_get_many:
            mock_get_many.return_value = self.probes.values()
            sagans = SaganSet(self.results)
            renderer = Renderer()
            renderer.gather_unique_certs(sagans)
            # remove cert key to make easier to test
            for k in renderer.uniqcerts:
                renderer.uniqcerts[k].pop("cert")
            self.assertEquals(renderer.uniqcerts, expected_certs)

    def test_bucketize_result_cert(self):
        """Tests bucketize of a single sagan result"""
        expected_certs = {
            '36:13:D2:B2:2A:75:00:94:76:0C:41:AD:19:DB:52:A4:F0:5B:DE:A8:01:72:E2:57:87:61:AD:96:7F:7E:D9:AA': {'cnt': 1, 'probes': [self.probes[1003]]},
            '21:EB:37:AB:4C:F6:EF:89:65:EC:17:66:40:9C:A7:6B:8B:2E:03:F2:D1:A3:88:DF:73:42:08:E8:6D:EE:E6:79': {'cnt': 1, 'probes': [self.probes[1003]]}
        }
        path = 'ripe.atlas.tools.helpers.rendering.Probe.get_many'
        with mock.patch(path) as mock_get_many:
            mock_get_many.return_value = self.probes.values()
            sagans = SaganSet(self.results)
            renderer = Renderer()
            renderer.bucketize_result_cert(list(sagans)[0])
            # remove cert key to make easier to test
            for k in renderer.uniqcerts:
                renderer.uniqcerts[k].pop("cert")
            self.assertEquals(renderer.uniqcerts, expected_certs)

    def test_get_nprobes_ofpopular_cert(self):
        """Tests fetching the number of probes for the most popular cert"""

        path = 'ripe.atlas.tools.helpers.rendering.Probe.get_many'
        with mock.patch(path) as mock_get_many:
            mock_get_many.return_value = self.probes.values()
            sagans = SaganSet(self.results)
            renderer = Renderer()
            renderer.gather_unique_certs(sagans)
            self.assertEquals(renderer.get_nprobes_ofpopular_cert(), 11)

    def test_render_certificate(self):
        """Tests rendering of single certificate."""
        expected_output = (
            "Certificate:\n"
            "  Issuer: C=US, O=DigiCert Inc, CN=DigiCert High Assurance CA-3\n"
            "  Subject: C=US, O=The Tor Project, Inc., CN=*.torproject.org\n"
            "  SHA256 Fingerprint=36:13:D2:B2:2A:75:00:94:76:0C:41:AD:19:DB:52:A4:F0:5B:DE:A8:01:72:E2:57:87:61:AD:96:7F:7E:D9:AA\n\n"
            "  Seen by 11 probes\n"
        )
        path = 'ripe.atlas.tools.helpers.rendering.Probe.get_many'
        with mock.patch(path) as mock_get_many:
            mock_get_many.return_value = self.probes.values()
            sagans = SaganSet(self.results)
            renderer = Renderer()
            renderer.gather_unique_certs(sagans)
            self.assertEquals(
                renderer.render_certificate("36:13:D2:B2:2A:75:00:94:76:0C:41:AD:19:DB:52:A4:F0:5B:DE:A8:01:72:E2:57:87:61:AD:96:7F:7E:D9:AA"),
                expected_output
            )

    def test_render_below_thershold(self):
        """Tests rendering string for certs that are seen for below threshold probes."""
        expected_output = [
            "  Below the threshold (80%)",
            "  Probes that saw it: ",
            "    ID: 2844, country code: GR, ASN (v4/v6): 3333/4444"
        ]
        path = 'ripe.atlas.tools.helpers.rendering.Probe.get_many'
        with mock.patch(path) as mock_get_many:
            mock_get_many.return_value = self.probes.values()
            sagans = SaganSet(self.results)
            renderer = Renderer()
            renderer.gather_unique_certs(sagans)
            self.assertEquals(
                renderer.render_below_threshold("07:52:BE:65:72:BF:02:D4:C9:E2:93:09:A8:E0:BE:3A:EA:D4:30:41:B8:49:FA:C5:F2:12:33:07:37:57:EE:C7"),
                expected_output
            )

    def test_report_with_ssl_consistency_renderer(self):
        """Tests the report with the ssl renderer."""

        results = self.results[:2]

        expected_output = (
            "Certificate:\n"
            "  Issuer: C=US, O=DigiCert Inc, CN=DigiCert High Assurance CA-3\n"
            "  Subject: C=US, O=The Tor Project, Inc., CN=*.torproject.org\n"
            "  SHA256 Fingerprint=36:13:D2:B2:2A:75:00:94:76:0C:41:AD:19:DB:52:A4:F0:5B:DE:A8:01:72:E2:57:87:61:AD:96:7F:7E:D9:AA\n\n"
            "  Seen by 2 probes\n\n"
            "Certificate:\n"
            "  Issuer: C=US, O=DigiCert Inc, CN=DigiCert High Assurance EV Root CA\n"
            "  Subject: C=US, O=DigiCert Inc, CN=DigiCert High Assurance CA-3\n"
            "  SHA256 Fingerprint=21:EB:37:AB:4C:F6:EF:89:65:EC:17:66:40:9C:A7:6B:8B:2E:03:F2:D1:A3:88:DF:73:42:08:E8:6D:EE:E6:79\n\n"
            "  Seen by 2 probes\n\n"
        )

        with capture_sys_output() as (stdout, stderr):
            path = 'ripe.atlas.cousteau.AtlasRequest.get'
            with mock.patch(path) as mock_get:
                mock_get.side_effect = [
                    (True, {"creation_time": 1, "start_time": 1, "type": {"name": "sslcert"}, "description": ""}),
                    (True, results)
                ]
                mpath = 'ripe.atlas.tools.helpers.rendering.Probe.get_many'
                with mock.patch(mpath) as mock_get_many:
                    mock_get_many.return_value = [self.probes[1003], self.probes[1004]]
                    cmd = Command()
                    cmd.init_args(["1", "--renderer", "ssl_consistency"])
                    cmd.run()
                    expected_set = set(expected_output.split("\n"))
                    returned_set = set(stdout.getvalue().split("\n"))
                    self.assertEquals(returned_set, expected_set)
