#!/usr/bin/env python2
# -*- coding: utf-8 -*-

# SAT plugin for Jingle File Transfer (XEP-0231)
# Copyright (C) 2009-2018 Jérôme Poisson (goffi@goffi.org)

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.

# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from sat.core.i18n import _
from sat.core.constants import Const as C
from sat.core.log import getLogger
log = getLogger(__name__)
from sat.tools import xml_tools
from wokkel import disco, iwokkel
from zope.interface import implements
from twisted.words.protocols.jabber.xmlstream import XMPPHandler
import base64


PLUGIN_INFO = {
    C.PI_NAME: "Bits of Binary",
    C.PI_IMPORT_NAME: "XEP-0231",
    C.PI_TYPE: "XEP",
    C.PI_PROTOCOLS: ["XEP-0231"],
    C.PI_DEPENDENCIES: ["XEP-0071"],
    C.PI_MAIN: "XEP_0231",
    C.PI_HANDLER: "yes",
    C.PI_DESCRIPTION: _("""Implementation of bits of binary (used for small images/files)""")
}

NS_BOB = u'urn:xmpp:bob'


class XEP_0231(object):

    def __init__(self, host):
        log.info(_(u"plugin Bits of Binary initialization"))
        self.host = host
        host.trigger.add("xhtml_post_treat", self.XHTMLTrigger)

    def dumpData(self, client, data_elt, cid):
        """save file encoded in data_elt to cache

        @param data_elt(domish.Element): <data> as in XEP-0231
        @param cid(unicode): content-id
        @return(unicode): full path to dumped file
        """
        # FIXME: is it needed to use a separate thread?
        #        probably not with the little data expected with BoB
        mime_type = data_elt.getAttribute('type','')

        try:
            max_age = int(data_elt['max-age'])
        except (KeyError, ValueError):
            log.warning(u'invalid max-age found')
            max_age = None

        with client.cache.cacheData(
            PLUGIN_INFO['import_name'],
            cid,
            mime_type,
            max_age) as f:

            file_path = f.name
            f.write(base64.b64decode(str(data_elt)))

        return file_path

    def getHandler(self, client):
        return XEP_0231_handler()

    def _dataCb(self, iq_elt, client, img_elt, cid):
        for data_elt in iq_elt.elements(NS_BOB, u'data'):
            if data_elt.getAttribute('cid') == cid:
                file_path = self.dumpData(client, data_elt, cid)
                img_elt[u'src'] = u'file://{}'.format(file_path)
                break
        else:
            log.warning(u"invalid data stanza received, requested cid was not found:\n{iq_elt}\nrequested cid: {cid}".format(
                iq_elt = iq_elt,
                cid = cid
                ))

    def _dataEb(self, iq_elt):
        log.warning(u"Can't get requested data:\n{iq_elt}".format(iq_elt=iq_elt))

    def XHTMLTrigger(self, client, message_elt, body_elt, lang, treat_d):
        for img_elt in xml_tools.findAll(body_elt, C.NS_XHTML, u'img'):
            source = img_elt.getAttribute(u'src','')
            if source.startswith(u'cid:'):
                cid = source[4:]
                file_path = client.cache.getFilePath(cid)
                if file_path is not None:
                    # image is in cache, we change change the url
                    img_elt[u'src'] = u'file://{}'.format(file_path)
                    continue
                else:
                    # image is not in cache, is it given locally?
                    for data_elt in message_elt.elements(NS_BOB, u'data'):
                        if data_elt.getAttribute('cid') == cid:
                            file_path = self.dumpData(data_elt, cid)
                            img_elt[u'src'] = u'file://{}'.format(file_path)
                            break
                    else:
                        # cid not found locally, we need to request it
                        # so we use the deferred
                        iq_elt = client.IQ('get')
                        iq_elt['to'] = message_elt['from']
                        data_elt = iq_elt.addElement((NS_BOB, 'data'))
                        data_elt['cid'] = cid
                        d = iq_elt.send()
                        d.addCallback(self._dataCb, client, img_elt, cid)
                        d.addErrback(self._dataEb)
                        treat_d.addCallback(lambda dummy: d)


class XEP_0231_handler(XMPPHandler):
    implements(iwokkel.IDisco)

    def getDiscoInfo(self, requestor, target, nodeIdentifier=''):
        return [disco.DiscoFeature(NS_BOB)]

    def getDiscoItems(self, requestor, target, nodeIdentifier=''):
        return []
