# coding=UTF8
#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2009 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
"""
Unit tests for schooltool.intervention.generations.evolve7
"""

from datetime import datetime
import unittest
import doctest

from zope.annotation.interfaces import IAttributeAnnotatable
from zope.app.generations.utility import getRootFolder
from zope.app.testing import setup
from zope.component.hooks import getSite, setSite
from zope.interface import implements
from zope.site.folder import Folder

from schooltool.app.interfaces import ISchoolToolApplication
from schooltool.basicperson.person import BasicPerson
from schooltool.contact.contact import Contact
from schooltool.contact.interfaces import IContactable
from schooltool.generations.tests import ContextStub
from schooltool.person.interfaces import IPerson

from schooltool.intervention.generations.tests import provideAdapters
from schooltool.intervention.generations.evolve7 import evolve
from schooltool.intervention.intervention import InterventionMessage
from schooltool.intervention.intervention import InterventionGoal


class AppStub(Folder):
    implements(ISchoolToolApplication)


class ParentStub(object):
    implements(IAttributeAnnotatable)
    first_name = 'Parent'
    last_name = 'Name'


def contactName(contact):
    return '%s %s' % (contact.first_name, contact.last_name)


def doctest_evolve7():
    """Evolution to generation 7.

    We need an app.

        >>> context = ContextStub()
        >>> context.root_folder['app'] = app = AppStub()
        >>> manager = setup.createSiteManager(app)

    We need a teacher, a student and a parent.

        >>> persons = app[u'persons'] = {}
        >>> jteacher = persons['jteacher'] = BasicPerson('jteacher', 'John', 'Teacher')
        >>> jstudent = persons['jstudent'] = BasicPerson('jstudent', 'Jane', 'Student')
        >>> IContactable(jstudent).contacts.add(ParentStub())

    Finally we need an intervention root, schoolyear and student.

        >>> root = app[u'schooltool.interventions'] = {}
        >>> year = root['thisyear'] = {}
        >>> student = year['jstudent'] = {}
        >>> messages = student['messages'] = {}
        >>> goals = student['goals'] = {}

    Our first test will be with a message and goal with all good ids.

        >>> message1 = messages['1'] = InterventionMessage('', '', '')
        >>> message1.__dict__['sender'] = 'jteacher'
        >>> message1.__dict__['recipients'] = ['jstudent', 'jstudent:1']

        >>> goal1 = goals['1'] = InterventionGoal('', '', '', '', '', '', '', 
        ...                                       '')
        >>> goal1.__dict__['creator'] = 'jteacher'
        >>> goal1.__dict__['_persons_responsible'] = ['jstudent', 'jstudent:1']
        >>> goal1.__dict__['at_one_time_responsible'] = ['jstudent', 'jstudent:1']

    We'll run the evolve script, testing the effected values before and after:

        >>> message1.__dict__['sender']
        'jteacher'
        >>> message1.__dict__['recipients']
        ['jstudent', 'jstudent:1']
        >>> goal1.__dict__['creator']
        'jteacher'
        >>> goal1.__dict__['_persons_responsible']
        ['jstudent', 'jstudent:1']
        >>> goal1.__dict__['at_one_time_responsible']
        ['jstudent', 'jstudent:1']

        >>> evolve(context)

        >>> 'sender' in message1.__dict__
        False
        >>> [contactName(contact) for contact in message1.sender]
        ['John Teacher']
        >>> 'recipients' in message1.__dict__
        False
        >>> [contactName(contact) for contact in message1.recipients]
        ['Jane Student', 'Parent Name']
        >>> 'creator' in goal1.__dict__
        False
        >>> [contactName(contact) for contact in goal1.creator]
        ['John Teacher']
        >>> '_persons_responsible' in goal1.__dict__
        False
        >>> [contactName(contact) for contact in goal1._persons_responsible]
        ['Jane Student', 'Parent Name']
        >>> 'at_one_time_responsible' in goal1.__dict__
        False
        >>> [contactName(contact) for contact in goal1.at_one_time_responsible]
        ['Jane Student', 'Parent Name']

    Our second test will be with a message and goal with invalid ids.

        >>> message2 = messages['2'] = InterventionMessage('', '', '')
        >>> message2.__dict__['sender'] = 'invalid_id'
        >>> message2.__dict__['recipients'] = ['invalid_id', 'jstudent:2']

        >>> goal2 = goals['2'] = InterventionGoal('', '', '', '', '', '', '', 
        ...                                       '')
        >>> goal2.__dict__['creator'] = 'invalid_id'
        >>> goal2.__dict__['_persons_responsible'] = ['invalid_id', 'jstudent:2']
        >>> goal2.__dict__['at_one_time_responsible'] = ['invalid_id', 'invalid_id:1']

    We'll run the evolve script, testing the effected values before and after:

        >>> message2.__dict__['sender']
        'invalid_id'
        >>> message2.__dict__['recipients']
        ['invalid_id', 'jstudent:2']
        >>> goal2.__dict__['creator']
        'invalid_id'
        >>> goal2.__dict__['_persons_responsible']
        ['invalid_id', 'jstudent:2']
        >>> goal2.__dict__['at_one_time_responsible']
        ['invalid_id', 'invalid_id:1']

        >>> evolve(context)

        >>> 'sender' in message2.__dict__
        False
        >>> [contactName(contact) for contact in message2.sender]
        []
        >>> 'recipients' in message2.__dict__
        False
        >>> [contactName(contact) for contact in message2.recipients]
        []
        >>> 'creator' in goal2.__dict__
        False
        >>> [contactName(contact) for contact in goal2.creator]
        []
        >>> '_persons_responsible' in goal2.__dict__
        False
        >>> [contactName(contact) for contact in goal2._persons_responsible]
        []
        >>> 'at_one_time_responsible' in goal2.__dict__
        False
        >>> [contactName(contact) for contact in goal2.at_one_time_responsible]
        []

    What if the intervention container doesn't exist yet in the
    application:

        >>> context = ContextStub()
        >>> context.root_folder['app'] = app = AppStub()
        >>> manager = setup.createSiteManager(app)
        >>> evolve(context)

    """


def setUp(test):
    setup.placefulSetUp()
    provideAdapters()
    setSite()

def tearDown(test):
    setup.placefulTearDown()
    setSite()


def test_suite():
    return unittest.TestSuite([
        doctest.DocTestSuite(setUp=setUp, tearDown=tearDown,
                             optionflags=doctest.ELLIPSIS
                             | doctest.NORMALIZE_WHITESPACE
                             | doctest.REPORT_NDIFF
                             | doctest.REPORT_ONLY_FIRST_FAILURE),
        ])


if __name__ == '__main__':
    unittest.main(defaultTest='test_suite')

