#!/usr/bin/env python3
# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab

# target script

import sys
import os
import locale

def run(bk):
    # get current epub version
    epubversion = "2.0"
    if bk.launcher_version() >= 20160102:
        epubversion = bk.epub_version()

    errors = 0

    print(" ")
    # Verify Python sys.path is set correctly for your installation
    print("Verify sys.path settings manually")
    for apath in sys.path:
        print(" ", apath)

    # verify if all the proper external python packages are available
    # as documented in our Python Developer's Framework as being available
    print(" ")
    print("Verifying proper Python packages are available")

    print("  Checking GUI modules")
    if bk.launcher_version() >= 20170227:
        try:
            print("    Python Package: PyQt5", end=" ")
            from PyQt5 import QtWidgets
            print("(%s) Found"%QtWidgets.__file__)
        except ImportError:
            print("Missing")
            errors += 1

    try:
        print("    Python Package: tkinter", end=" ")
        import tkinter
        print("(%s) Found"%tkinter.__file__)
    except ImportError:
        print("Missing")
        errors += 1

    try:
        print("  Python Package: PIL", end=" ")
        import PIL
        print("(%s) Found"%PIL.__file__)
    except ImportError:
        print("Missing")
        errors += 1

    try:
        print("  Python Package: cssselect", end=" ")
        import cssselect
        print("(%s) Found"%cssselect.__file__)
    except ImportError:
        print("Missing")
        errors += 1

    try:
    	if bk.launcher_version() < 20190826:
            print("  Python Package: cssutils", end=" ")
            import cssutils
            print("(%s) Found"%cssutils.__file__)
    	else:
            print("  Python Package: css-parser", end=" ")
            import css_parser
            print("(%s) Found"%css_parser.__file__)
    except ImportError:
        print("Missing")
        errors += 1

    try:
        print("  Python Package: html5lib", end=" ")
        import html5lib
        print("(%s) Found"%html5lib.__file__)
    except ImportError:
        print("Missing")
        errors += 1

    try:
        print("  Python Package: lxml", end=" ")
        import lxml
        print("(%s) Found"%lxml.__file__)
    except ImportError:
        print("Missing")
        errors += 1

    try:
        print("  Python Package: regex", end=" ")
        import regex
        print("(%s) Found"%regex.__file__)
    except ImportError:
        print("Missing")
        errors += 1

    try:
        print("  Python Package: chardet", end=" ")
        import chardet
        print("(%s) Found"%chardet.__file__)
    except ImportError:
        print("Missing")
        errors += 1

    try:
        print("  Python Package: six", end=" ")
        import six
        print("(%s) Found"%six.__file__)
    except ImportError:
        print("Missing")
        errors += 1

    print(" ")
    print("Verifying Sigil Python Libraries can be found/loaded")
    try:
        print("  Sigil Python library: epub_utils", end=" ")
        import epub_utils
        print("Found")
    except ImportError:
        print("Missing")
        errors += 1

    try:
        print("  Sigil Python library: quickparser", end=" ")
        import quickparser
        print("Found")
    except ImportError:
        print("Missing")
        errors += 1

    try:
        print("  Sigil Python library: compatibility_utils", end=" ")
        import compatibility_utils
        print("Found")
    except ImportError:
        print("Missing")
        errors += 1

    try:
        print("  Sigil Python library: sigil_bs4", end=" ")
        import sigil_bs4
        print("Found")
    except ImportError:
        print("Missing")
        errors += 1

    print(" ")
    print("Verifying Hunspell Spell Checking")
    afffile = None
    dicfile = None
    dic_dirs = bk.get_dictionary_dirs();
    for adir in dic_dirs:
        afile = os.path.join(adir, "en_US.aff")
        dfile = os.path.join(adir, "en_US.dic")
        if os.path.exists(afile) and os.path.exists(dfile):
            afffile = afile
            dicfile = dfile
            break

    print("  Hunspell en_US affix file and dictionary", end=" ")
    if afffile is None or dicfile is None:
        print("Missing")
        errors += 1
    else:
        print("Found")

    print("  Hunspell shared library", end=" ")
    if bk.hspell is None or bk.hspell.hunspell is None:
        print("Missing")
        errors += 1
    else:
        print("Found")

    print("  Hunspell spellchecking works", end=" ")
    if bk.hspell is not None and bk.hspell.hunspell is not None and afffile is not None and dicfile is not None:
        try: 
            bk.hspell.loadDictionary(afffile, dicfile)
            checklist =  ["misteak","hello"]
            working = True
            for word in checklist:
                res = bk.hspell.check(word)
                if word == "misteak":
                    working = working and (res != 1)
                if word == "hello":
                    working = working and (res == 1)
            bk.hspell.cleanUp()
            print(working)
            if not working:
                errors += 1
        except:
            print("False")
            errors += 1
    else:
        print("False")
        errors += 1

    print("  ")
    print("Verifying Sigil Gumbo Library operation")
    working = False
    try:
        print("  Sigil Gumbo BS4 Adapter library", end=" ")
        import sigil_gumbo_bs4_adapter as gumbo_bs4
        print("Found")
        working = True
    except ImportError:
        print("Missing")
        errors += 1


    print("  Sigil BS4/Gumbo Parser works", end=" ")
    if working:
        try: 
            samp = """
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" 
  "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">
<html xmlns="http://www.w3.org/1999/xhtml/" xml:lang="en" lang="en-US">
<head><title>testing &amp; entities</title></head>
<body>
  <p class="first second">this is the <i><b>copyright</i></b> symbol "&copy;"</p>
</body>
</html>
"""
            soup = gumbo_bs4.parse(samp)
            for node in soup.find_all(attrs={'class':'second'}):
                res = repr(node)
                working = working and (res == '<p class="first second">this is the <i><b>copyright</b></i> symbol "©"</p>')
        except:
            working = False;
    else:
        working = False
    print(working)
    if not working:
        errors += 1

    print(" ")
    print("Verifying selected basic plugin interface elements")
    lastid = None
    # all xhtml/html files
    working = False
    print("  bk.text_iter() works", end=" ")
    for (id, href) in bk.text_iter():
        if epubversion.startswith("3"):
            working = (href == "Text/nav.xhtml")
        else:
            working = (href == "Text/Section0001.xhtml")
        lastid = id
    print(working)
    if not working:
        errors += 1

    # all files in the OPF manifest
    working = False
    print("  Listing manifested files...")
    for (id, href, mime) in bk.manifest_iter():
        print("    %s" % href)
        if epubversion.startswith("3"):
            working = href in ("Text/Section0001.xhtml", "toc.ncx", "Text/nav.xhtml", "Styles/sgc-nav.css")
        else:
            working = href in ("Text/Section0001.xhtml", "toc.ncx")
        if not working:
            break
    print("  bk.manifest_iter() works", end=" ")
    print(working)
    if not working:
        errors += 1

    # all files in the OPF spine in spine order
    working = False
    print("  bk.spine_iter() works", end=" ")
    for (id, linear, href) in bk.spine_iter():
        if epubversion.startswith("3"):
            working = href in ("Text/Section0001.xhtml", "Text/nav.xhtml")
        else:
            working = href in ("Text/Section0001.xhtml")
    print(working)
    if not working:
        errors += 1

    # all other ebook files not in the manifest
    print("  bk.other_iter() works", end=" ")
    for book_href in bk.other_iter():
        working in ("META-INF/container.xml", "OEBPS/content.opf")
        if not working:
            break
    print(working)
    if not working:
        errors += 1

    # Example of reading a non-manifest book file
    working = False
    print("  reading a non-manifest book file works", end=" ")
    stuff = bk.readotherfile('META-INF/container.xml')
    working = stuff is not None and len(stuff) > 0;
    print(working)
    if not working:
        errors += 1

    # Example of reading a file with a specific OPF manifest id
    working = False
    print("  reading a specific OPF manifest id works", end=" ")
    stuff = bk.readfile(lastid)
    working = stuff is not None and len(stuff) > 0
    print(working)
    if not working:
        errors += 1

    # Example of writing to a file specified by the manifest id
    working = False
    print("  writing to a specific OPF manifest id works", end=" ")
    bk.writefile(lastid, stuff)
    working = lastid in bk._w.modified.keys()
    print(working)
    if not working:
        errors += 1

    # Information about launcher version and epub version
    lv = bk.launcher_version()
    print("  Launcher Version", lv)
    print("  ePub Version", epubversion)

    # Example of adding your own file to the manifest
    working = False
    print("  adding your own file to the manifest works", end=" ")
    data = '<?xml version="1.0" encoding="utf-8"?>\n'
    if epubversion == "2.0":
        data += '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" '
        data += '"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">\n'
    else:
        data += '<!DOCTYPE html>\n'
    data += '<html xmlns="http://www.w3.org/1999/xhtml">\n'
    data += '<head>\n<title>Hello World</title>\n</head>\n'
    data += '<body>\n'
    data += '<p>Hello ... I was Created by a Plugin</p>\n'
    data += '</body>\n'
    data += '</html>\n'
    basename = "my_test.html"
    mt = "application/xhtml+xml"
    uid1 = 'add1'
    bk.addfile(uid1, basename, data, mt)
    working = uid1 in bk._w.id_to_mime.keys() and uid1 in bk._w.added
    print(working)
    if not working:
        errors += 1

    # Examples of Using the Convenience Functions
    working = False
    print("  basename to id mapping works", end=" ")
    bid = bk.basename_to_id(basename)
    working = bid == uid1
    print(working)
    if not working:
        errors += 1

    # Example of deleting our added file
    working = uid1 in bk._w.added
    print("  deleting our added file works", end=" ")
    bk.deletefile(uid1)
    working = working and uid1 not in bk._w.added
    print(working)
    if not working:
        errors += 1

    print(" ")
    print(" ")
    if errors > 0:
        print("Failure - ", errors, " Tests of Plugin Operations Failed")
    else:
        print("Success - All Tests of Plugin Operations Passed")


    # Setting the proper Return value is important.
    # 0 - means success
    # anything else means failure
    return 0

def main():
    print("I reached main when I should not have\n")
    return -1


if __name__ == "__main__":
    sys.exit(main())

