/* ------------------------------------------------------------------------ */
/*                                                                          */
/* [StyxParser.cs]   Styx Parser Implementation (C#)                        */
/*                                                                          */
/*                This file belongs to the Styx Scanner & parser generator. */
/*                                                                          */
/* Copyright (c) 2009 by Doelle, Manns                                      */
/* ------------------------------------------------------------------------ */


using System;
using System.Text;
using System.IO;
using System.Collections;
using System.Collections.Generic;


/** @file
    Abstract parse table interface and parser 
*/

namespace styx
{
  /** @addtogroup StyxLibraryCS
      C# Scanner & parser library
  */
  //@{

  /** @addtogroup StyxParser
      Abstract parse table interface and parser 
  */
  //@{

  /* Parse table Interface -------------------------------------------------- */

  /** @addtogroup AbstractParseTableAPI
      Abstract parse table interface 
  */
  //@{

  /** parser action (error) codes */
  public enum EStyxParseAction
  {
    PLR_SHIFT_ERROR = -1, //!< shift error
    PLR_SHIFT_NONE  = -2, //!< no shift (comment)
    PLR_GOTO_ERROR  = -1  //!< goto error
  }

  /** abstract parse table interface */
  public abstract class CStyxParseTableApi
  {

    //! The constructor creates an uninitialized parse table.
    public CStyxParseTableApi() {}

                             //! parse table is initialized ?
                             //! @return true for an initialized parse table
    public abstract  bool    initialized();

                             //! get language
                             //! @return language name or ""
    public abstract  string  getLanguage();

                             //! get number of tokens
                             //! @return number of tokens or -1 
    public abstract  long    getTokenCount();
                             //! get number of nonterminals
                             //! @return number of nonterminals or -1 
    public abstract  long    getNonterminalCount();
                             //! get number of symbols 
                             //! (tokens incl. keywords, nonterminals, start symbols)
                             //! @return number of symbols or -1 
    public abstract  long    getSymbolCount();
                             //! get number of start symbols
                             //! @return number of start symbols or -1 
    public abstract  long    getStartSymbolCount();
                             //! get number of terminals 
                             //! (tokens incl. keywords, accept tokens)
                             //! @return number of terminals or -1 
    public abstract  long    getTerminalCount();
                             //! get number of meta symbols 
                             //! (nonterminals, start symbols)
                             //! @return number of meta symbols or -1 
    public abstract  long    getMetaSymbolCount();

                             //! token index --> terminal index
                             //! @param TKIndex token index \>= 0 or EOF = -1 or error \< -1
                             //! @param SNIndex current start symbol index
                             //! @return terminal index \>= 0 or -1
    public abstract  long    TokenIndex2TerminalIndex(long TKIndex, long SNIndex);

                             //! accept terminal ?
                             //! @param TIndex terminal index \>= 0
                             //! @return true for an accept terminal
    public abstract  bool    isAcceptTerminal(long TIndex);
                             //! terminal index --> symbol
                             //! @param TIndex terminal index \>= 0
                             //! @return terminal symbol \>= 0 or -1 
    public abstract  long    TerminalIndex2Symbol(long TIndex);
                             //! terminal symbol --> index
                             //! @param TSymbol terminal symbol \>= 0
                             //! @return terminal index \>= 0 or -1 
    public abstract  long    TerminalSymbol2Index(long TSymbol);

                             //! start symbol index --> symbol
                             //! @param SNIndex start symbol index \>= 0
                             //! @return start symbol \>= 0 or -1 
    public abstract  long    StartIndex2Symbol(long SNIndex);
                             //! start symbol symbol --> index
                             //! @param SNSymbol start symbol \>= 0
                             //! @return start symbol index \>= 0 or -1 
    public abstract  long    StartSymbol2Index(long SNSymbol);
                             //! start symbol name --> index
                             //! @param StartSymbolName start symbol name or "" (first start symbol)
                             //! @return start symbol index \>= 0 or -1 
    public abstract  long    StartName2Index(string StartSymbolName/*=""*/);

                             //! nonterminal index --> symbol
                             //! @param NIndex nonterminal index \>= 0
                             //! @return nonterminal symbol \>= 0 or -1 
    public abstract  long    NonterminalIndex2Symbol(long NIndex);
                             //! nonterminal symbol --> index
                             //! @param NSymbol nonterminal symbol \>= 0
                             //! @return nonterminal index \>= 0 or -1 
    public abstract  long    NonterminalSymbol2Index(long NSymbol);
                             //! error (recovery) nonterminal ?
                             //! @param NIndex nonterminal index \>= 0
                             //! @return true for an error (recovery) nonterminal
    public abstract  bool    isErrorNonterminal(long NIndex);

                             //! nonterminal ?
                             //! @param Symbol CFG symbol \>= 0
                             //! @return true for a nonterminal
    public abstract  bool    isNonterminal(long Symbol);
                             //! regular token class ?
                             //! @param Symbol CFG symbol \>= 0
                             //! @return true for a regular token class
    public abstract  bool    isToken(long Symbol);
                             //! comment token ?
                             //! @param Symbol CFG symbol \>= 0
                             //! @return true for a comment token
    public abstract  bool    isComment(long Symbol);
                             //! keword ?
                             //! @param Symbol CFG symbol \>= 0
                             //! @return true for a keyword
    public abstract  bool    isKeyword(long Symbol);
                             //! get symbol name
                             //! @param Symbol CFG symbol \>= 0
                             //! @return symbol name or ""
    public abstract  string  getSymbolName(long Symbol);

                             //! get number of productions
                             //! @return number of productions or -1 
    public abstract  long    getProductionCount();
                             //! get production name
                             //! @param PIndex production index \>= 0
                             //! @return production name or ""
    public abstract  string  getProductionName(long PIndex);
                             //! get production nonterminal
                             //! @param PIndex production index \>= 0
                             //! @return nonterminal symbol \>= 0 or -1 
    public abstract  long    getProductionNonterminal(long PIndex);
                             //! get number of production member symbols
                             //! @param PIndex production index \>= 0
                             //! @return number of member symbols or -1 
    public abstract  long    getProductionSymbolCount(long PIndex);
                             //! get production member symbol
                             //! @param PIndex production index \>= 0
                             //! @param PMIndex production member index \>= 0
                             //! @return member symbol \>= 0 or -1 
    public abstract  long    getProductionSymbol(long PIndex, long PMIndex);

                             //! get number of states
                             //! @return number of states or -1 
    public abstract  long    getStateCount();
                             //! get error action
                             //! @return error action >= 0 or -1 
    public abstract  long    getErrorAction();
                             //! get accept action
                             //! @return accept action >= 0 or -1 
    public abstract  long    getAcceptAction();
                             //! get shift follow state for state and terminal index
                             //! @param STIndex current state index \>= 0
                             //! @param TIndex current terminal index \>= 0
                             //! @return follow state index \>= 0 or
                             //! PLR_SHIFT_NONE for a comment token or
                             //! PLR_SHIFT_ERROR
    public abstract  long    getShiftState(long STIndex, long TIndex); 
                             //! get goto follow state for state and nonterminal index
                             //! @param STIndex current state index \>= 0
                             //! @param NIndex current nonterminal index \>= 0
                             //! @return follow state index \>= 0 or PLR_GOTO_ERROR
    public abstract  long    getGoToState(long STIndex, long NIndex); 
                             //! get reduce action for state and terminal index
                             //! @param STIndex current state index \>= 0
                             //! @param TIndex current terminal index \>= 0
                             //! @return reduce action \>= 0
    public abstract  long    getReduceAction(long STIndex, long TIndex); 
                             //! try early action for state index
                             //! @param STIndex current state index \>= 0
                             //! @param Action [result] reduce action \>= 0
                             //! @return true if a default action exists
    public abstract  bool    tryEarlyAction(long STIndex, ref long Action); 
                             //! reduce action = error ?
                             //! @param Action reduce action \>= 0
                             //! @return true if action = error
    public abstract  bool    isErrorAction(long Action);
                             //! reduce action = accept ?
                             //! @param Action reduce action \>= 0
                             //! @return true if action = accept
    public abstract  bool    isAcceptAction(long Action);
                             //! reduce action --> production 
                             //! @param Action reduce action \>= 0
                             //! @param SNIndex current start symbol index \>= 0
                             //! @return production index \>= 0 or -1 
    public abstract  long    Action2Production(long Action, long SNIndex);
  }

  //@}

  /* Parser ----------------------------------------------------------------- */

  /** @addtogroup AbstractParser
      Abstract lalr(1) parser 
  */
  //@{

  /** type of scanned terminal */
  public class StyxToken<TV /* token value type */>
  {
    public long   Index;   //!< terminal index >= 0 or error terminal
    public TV     Value;   //!< token value
    public string File;    //!< token file 
    public long   Line;    //!< token line  
    public long   Column;  //!< token column

    public StyxToken()
    {
      Index = -1;
      Value = default(TV);
      File = "";
      Line = Column = 0;
    }
  }

  /** abstract lalr(1) parser template */
  public abstract class TStyxParser<TV /* token value type */, SV /* stack value type */> 
  {
    /** number of error messages to skip after the last reported */
    public static int PLR_SKIP_ERRMSG
    {
      get { return 3; }
    }

/*OBSOLET
  protected:
    typedef vector<long >    V_long;
    typedef vector<SV* >     V_value;
    typedef StyxToken<TV>    T_Token;
*/

    /** must be set appropiate in derived classes */
    protected bool initok;

    protected CStyxParseTableApi  ParseTable; //!< parse table
    protected long                StartIndex; //!< start symbol index
    protected List<long>          StateStack; //!< state stack
    protected List<SV>            ValueStack; //!< value stack 

    /** @name EarlyReductionAttributes 
        attributes related to early reduction 
    */
    //@{
    protected StyxToken<TV>    curToken;      //!< current token
    protected bool             firstToken;    //!< first token (respectively parse) ?
    protected bool             nextToken;     //!< (read) next token ? 
    //@}

    /** @name ErrorHandlingAttributes 
        attributes related to (panic mode) error handling
    */
    /* Skip: T --> { 1 .. StateStack } X NT  U  { undefined } */ 
    //@{
    protected List<long>       skipTokens;         //!< skip token array
    protected List<long>       skipStates;         //!< skip state array
    protected List<long>       skipNonterminals;   //!< skip nonterminal array
    protected long             SyntaxErrorCount;   //!< number of syntax errors
    //@}

                           //! init parse process 
                           //! @param StartSymbolName start symbol name
                           //! @param State [result] current state index \>= 0
                           //! @return true or false on error
    protected virtual bool initParse(string StartSymbolName, ref long State)
    {
      State = -1;
      SyntaxErrorCount = 0;
      if( initok && ParseTable != null )
      { long startidx = ParseTable.StartName2Index(StartSymbolName);
        if( StartIndex == -1 )
        {
          removeValues(true);
          StateStack.Clear();
          StateStack.Add(StartIndex=startidx);
        }
        else 
        {
          if( StartIndex != startidx || StateStack.Count == 0 )
            return false;
        }
        State = StateStack[StateStack.Count-1];
        return true;
      }
      return false;
    }

                           //! quit parse process
                           //! @param reset
                           //! = true --\> reset parse process
                           //! else   --\> keep last token (early reduction)
    protected virtual void quitParse(bool reset/*=false*/)
    {
      StartIndex = -1;
      if( reset )
      {
        curToken.Index = -1;
        firstToken     = nextToken = true;
      }
      removeValues(false);
      StateStack.Clear();
    }

                           //! reset parser
                           //! @return true or false on error
    protected virtual bool reset()
    {
      initok           = ParseTable != null && ParseTable.initialized();
      SyntaxErrorCount = 0;
      StartIndex       = -1;
      curToken         = new StyxToken<TV>();
      //curToken.Index   = -1;
      firstToken       = nextToken = true;
      removeValues(true);
      StateStack       = new List<long>();
      skipTokens       = new List<long>();
      skipStates       = new List<long>();
      skipNonterminals = new List<long>();
      return initok;
    }

                           //! remove (temporary) values 
                           //! @param reset
                           //! = true --\> release all values
    protected virtual void removeValues(bool reset)
    {
      ValueStack = new List<SV>();
    }

                           //! init skip tables 
                           //! @return true --\> try error recovery 
    protected virtual bool initSkipTables()
    {
      if( initok && ParseTable != null )
      { long tcnt = ParseTable.getTerminalCount(), 
             scnt = (long)StateStack.Count,
             ncnt = ParseTable.getNonterminalCount();
        for( long i=0; i < tcnt; ++i )
          skipTokens.Add(-1);
        for( long i=scnt-1; i >= 0; --i )
        {
          for( long j=0; j < ncnt; ++j ) 
          {
            if( ParseTable.isErrorNonterminal(j) )
            { long s = ParseTable.getGoToState(StateStack[(int)i],j);
            if (s != (long)EStyxParseAction.PLR_GOTO_ERROR)
              {
                for( long k=0; k < tcnt; ++k )
                { long p1 = ParseTable.getShiftState(s,k);
                if (p1 != (long)EStyxParseAction.PLR_SHIFT_ERROR &&
                      p1 != (long)EStyxParseAction.PLR_SHIFT_NONE &&
                      skipTokens[(int)k] == -1L )
                  {
                    skipNonterminals.Add(j);
                    skipStates.Add(i);
                    skipTokens[(int)k] = (long)skipNonterminals.Count-1;
                  }
                }
              }
            }
          }
        }
        if( skipStates.Count > 0 ) return true;
      }
      quitSkipTables();
      return false;
    }

                           //! release skip tables
    protected virtual void quitSkipTables()
    {
      skipTokens.Clear();
      skipStates.Clear();
      skipNonterminals.Clear();
    }

                           //! pop n states
                           //! @param n number of states
    protected virtual void popStates(long n) 
    {
      if( n >= StateStack.Count )
        StateStack.Clear();
      else
        for( long i=0; i < n; ++i )
          StateStack.RemoveAt(StateStack.Count-1);
    }

                           //! get & pop the top n values
                           //! @param n number of values
                           //! @param Values [result] values
                           //! @return true or false on error
    protected virtual bool topValues(long n, ref List<SV> Values)           
    {
      if( n < 0 || n > ValueStack.Count )
        return false;
      for( long cnt = ValueStack.Count, i=cnt-n; i < cnt; ++i )
      {
        Values.Add(ValueStack[(int)i]);
      }
      for( long i=0; i < n; ++i )
      {
        StateStack.RemoveAt(StateStack.Count-1);
      }
      return true;
    }

                           //! try early reduction 
                           //! @param State [changeable] current state index \>= 0
                           //! @param accept [result]
                           //! = true --\> finish parse process
                           //! @return true or false on error
    protected virtual bool earlyReduction(ref long State, ref bool accept) 
    {
      accept = false;
      if( initok && ParseTable != null )
      { long res = ParseTable.getErrorAction();
        while( ParseTable.tryEarlyAction(State,ref res) )
        { List<SV> Values = new List<SV>(); 
          SV val = default(SV);
          if( !ParseTable.isAcceptAction(res) )
          { long prod = ParseTable.Action2Production(res,StartIndex),
                 scnt = ParseTable.getProductionSymbolCount(prod);
            if( !topValues(scnt,ref Values) ) return false;
            if( !reduce(prod,ref Values,false,false,ref val) ) return false;
            popStates(scnt);
            long nidx = ParseTable.NonterminalSymbol2Index
                        (
                          ParseTable.getProductionNonterminal(prod)
                        );
            State = getGoToState(nidx);
            StateStack.Add(State); ValueStack.Add(val);
            if( State == (long)EStyxParseAction.PLR_GOTO_ERROR ) 
              return false;
          }
          else
          { long prod = ParseTable.Action2Production(res,StartIndex),
                 scnt = ParseTable.getProductionSymbolCount(prod);
            if( !topValues(scnt,ref Values) ) return false;
            if( !reduce(prod,ref Values,true,false,ref val) ) return false;
            accept = true; return true;
          }
        }
        return true;
      }
      return false;
    }

                           //! get goto follow state for top state and nonterminal index
                           //! @param NIndex nonterminal index \>= 0
                           //! @return follow state index \>= 0 or PLR_GOTO_ERROR
    protected virtual long getGoToState(long NIndex)
    {
      if( initok && ParseTable != null && StateStack.Count > 0 )
      { 
        return ParseTable.getGoToState(StateStack[StateStack.Count-1],NIndex); 
      }
      return (long)EStyxParseAction.PLR_GOTO_ERROR;
    }

                           //! (report) internal parser error
                           //! @param Text error message
    protected virtual void error(string Text)
    {
      Console.Error.WriteLine(Text);
    }

    /** @name AbstractParserMethods
        abstract methods, must be overloaded in derived classes
    */
    //@{
                            //! get next token
                            //! @param Token [result] next token
                            //! @return true or false on error
    protected abstract bool get(ref StyxToken<TV> Token);

                            //! shift token
                            //! @param Token current token
                            //! @param Value [result] new stack value
                            //! @return true or false on error
    protected abstract bool shift(ref StyxToken<TV> Token, ref SV Value);

                            //! reduce production
                            //! @param Index 
                            //! accept || !error --\> Index=production
                            //! else             --\> Index=nonterminal
                            //! @param Values reduction values
                            //! @param accept = true --\> reduce start production
                            //! @param error = true --\> syntax error
                            //! @param Value [result] new stack value
                            //! @return true or false on error
    protected abstract bool reduce
                            (
                              long Index, ref List<SV> Values, 
                              bool accept, bool error, ref SV Value
                            ); 

                            //! (report) syntax error
                            //! @param Token current token
    protected abstract void error(ref StyxToken<TV> Token);
    //@}

    //! The default constructor.
    public TStyxParser()
    {
      ParseTable = null;
      initok = reset();
    }

    //! The constructor creates and initializes the parser.
    //! @param PTable parse table
    public TStyxParser(ref CStyxParseTableApi PTable)
    {
      ParseTable = PTable;
      initok = reset();
    }

                        //! set parse table and
                        //! re-init (--> reset) parser
                        //! @param PTable parse table
                        //! @return true or false on error
    public virtual bool setParseTable(ref CStyxParseTableApi PTable)
    {
      ParseTable = PTable;
      initok = reset();
      if( !initok ) error("reset failed");
      return initok;
    }

                        //! get parse table
                        //! @return parse table
    public virtual void getParseTable(ref CStyxParseTableApi PTable) 
    {
      PTable = ParseTable;
    }

                        //! parse process 
                        //! @param StartSymbolName start symbol name or "" (first start symbol)
                        //! @param early try early reduction
                        //! @return true for a regular parse process (even in the case of syntax errors) 
                        //! or false on (internal) error
    public virtual bool parse(string StartSymbolName/*=""*/, bool early/*=false*/)
    {
      if( initok && ParseTable != null )
      { long cTok = -1, cState = -1, errcnt=0, tokcnt=0; 
        SyntaxErrorCount = 0;
        if( !early || nextToken )
        {
          if( !get(ref curToken) ) 
          { quitParse(true); return false; }
        }
        cTok = curToken.Index; ++tokcnt;
        if( early && !firstToken && ParseTable.isAcceptTerminal(curToken.Index) )
        {
          quitParse(true); return true;
        }
        if( !initParse(StartSymbolName,ref cState) )
        { quitParse(true); return false; }
        while( true )
        {
          long action = ParseTable.getShiftState(cState,cTok);
          if( action != (long)EStyxParseAction.PLR_SHIFT_ERROR )
          { SV val = default(SV);
            if( !shift(ref curToken,ref val) )
            { quitParse(true); return false; }
            if( action != (long)EStyxParseAction.PLR_SHIFT_NONE )
            {
              cState = action;
              StateStack.Add(cState); ValueStack.Add(val);
              if( early )
              { bool accept = false;
                if( !earlyReduction(ref cState,ref accept) )
                { quitParse(true); return false; }
                if( accept ) 
                {
                  firstToken = false; nextToken = true; 
                  quitParse(false); return true;
                }
              }
            }
            if( !get(ref curToken) ) 
            { quitParse(true); return false; }
            cTok = curToken.Index; ++tokcnt;
          }
          else
          { 
            action = ParseTable.getReduceAction(cState,cTok);
            long redsym = ParseTable.Action2Production(action,StartIndex);
            if( !ParseTable.isErrorAction(action) &&
                !ParseTable.isAcceptAction(action) )
            { long scnt = ParseTable.getProductionSymbolCount(redsym);
              List<SV> Values = new List<SV>(); 
              SV val = default(SV);
              if( !topValues(scnt,ref Values) )
              { quitParse(true); return false; }
              if( !reduce(redsym,ref Values,false,false,ref val) )
              { quitParse(true); return false; }
              popStates(scnt);
              long nidx = ParseTable.NonterminalSymbol2Index
                          (
                            ParseTable.getProductionNonterminal(redsym)
                          );
              cState = getGoToState(nidx);
              StateStack.Add(cState); ValueStack.Add(val);
              if( cState == (long)EStyxParseAction.PLR_GOTO_ERROR )
              { quitParse(true); return false; }
              if( early )
              { bool accept = false;
                if( !earlyReduction(ref cState,ref accept) )
                { quitParse(true); return false; }
                if( accept ) 
                {
                  firstToken = nextToken = false; 
                  quitParse(false); return true;
                }
              }
            }
            else
            {
              if( ParseTable.isAcceptAction(action) &&
                  ParseTable.isAcceptTerminal(cTok) )
              { long scnt = ParseTable.getProductionSymbolCount(redsym);
                List<SV> Values = new List<SV>(); 
                SV val = default(SV);
                if( !topValues(scnt,ref Values) )
                { quitParse(true); return false; }
                if( SyntaxErrorCount == 0 || scnt == ((long)ValueStack.Count) )
                { 
                  if( !reduce(redsym,ref Values,true,false,ref val) )
                  { quitParse(true); return false; }
                }
                else
                { 
                  if( !reduce(redsym,ref Values,true,true,ref val) )
                  { quitParse(true); return false; }
                }
                firstToken = nextToken = false;
                quitParse(false); return true;
              }
              else
      /*ALT
              { List<SV> Values = new List<SV>(); 
                SV val = default(SV);
                ++SyntaxErrorCount; error(ref curToken);
                if( !topValues((long)ValueStack.Count,ref Values) )
                { quitParse(true); return false; }
                if( !reduce(redsym,ref Values,true,true,ref val) )
                { quitParse(true); return false; }
                quitParse(true); return true;
              }
      */
      /*NEU*/
              { long p = -1, s = -1, n = -1;
                ++SyntaxErrorCount; 
                if( tokcnt >= errcnt )
                {
                  error(ref curToken); errcnt = tokcnt + PLR_SKIP_ERRMSG;
                }
                if( !early && initSkipTables() )
                {
                  while( !ParseTable.isAcceptTerminal(cTok) && 
                         ( cTok < 0 || skipTokens[(int)cTok] == -1 ) )
                  { SV val = default(SV);
                    if( !shift(ref curToken,ref val) )
                    { quitParse(true); return false; }
                    if( cTok < 0 ) 
                      ValueStack.Add(val);
                    else
                    { long tsym = ParseTable.TerminalIndex2Symbol(cTok);
                      if( ! ParseTable.isComment(tsym) )
                        ValueStack.Add(val);
                    }
                    if( !get(ref curToken) ) 
                    { quitParse(true); return false; }
                    cTok = curToken.Index; // cTok < 0 --> continue
                  }
                  if( cTok < 0 ) /* assertion: cTok >= 0 */
                  { quitParse(true); return false; }
                  if( !ParseTable.isAcceptTerminal(cTok) )
                  {
                    s = skipStates[(int)skipTokens[(int)cTok]];
                    n = skipNonterminals[(int)skipTokens[(int)cTok]];
                  }
                  quitSkipTables();
                }
                else // quit parsing on first error
                  cTok = ParseTable.TokenIndex2TerminalIndex(-1,StartIndex); 
                if( !ParseTable.isAcceptTerminal(cTok) )
                { long vcnt = ((long)ValueStack.Count)-(s+1),
                       scnt = ((long)StateStack.Count)-(s+1);
                  List<SV> Values = new List<SV>(); 
                  SV val = default(SV);
                  if( !topValues(vcnt,ref Values) )
                  { quitParse(true); return false; }
                  if( !reduce(n,ref Values,false,true,ref val) )
                  { quitParse(true); return false; }
                  popStates(scnt);
                  cState = getGoToState(n);
                  StateStack.Add(cState); ValueStack.Add(val);
                  if( cState == (long)EStyxParseAction.PLR_GOTO_ERROR || 
                      ValueStack.Count != StateStack.Count )
                  { quitParse(true); return false; }
                }
                else
                { 
                  List<SV> Values = new List<SV>(); 
                  SV val = default(SV);
                  if( !topValues((long)ValueStack.Count,ref Values) )
                  { quitParse(true); return false; }
                  p = ParseTable.Action2Production
                      (
                        ParseTable.getAcceptAction(),StartIndex  
                      );
                  if( !reduce(p,ref Values,true,true,ref val) )
                  { quitParse(true); return false; }
                  quitParse(true); return true;
                }
              }
      /**/
            }
          }
        }
      }
      error("parser not initialized");
      return false;
    }

                        //! parse process 
                        //! uses first start symbol, no early reduction
                        //! @return true for a regular parse process (even in the case of syntax errors) 
                        //! or false on (internal) error
    public virtual bool parse()
    {
      return parse("",false);
    }

                        //! get number of syntax errors
                        //! @return number of syntax errors
    public virtual long getSyntaxErrorCount()
    {
      return SyntaxErrorCount;
    }
  };

  //@}

  //@}

  //@}

}
